------------------------------------------------------------------
--  This is the example SQL that illustrates the way of 	--
--  creating DB objects according to fortified security rules   --
--  In our example we'll Create a table and some associated	--
--  objects in 'confdb' schema					--
------------------------------------------------------------------

--
-- Set current schema
-- Following objects will be assigned to it
--
-- Here goes 'private' section.
-- Tables, sequences, associated triggers should be placed
-- into 'confdb' or 'transdb' schemas
--
SET search_path TO confdb;

--
-- First create sequence for primary key field of our table
--
CREATE SEQUENCE seq_extable start 100;

-- Lookup table (contains foreign keys for out main table)
--
CREATE TABLE extype (
    typeid char(1) NOT NULL CONSTRAINT pk_extype PRIMARY KEY,
    name   varchar NOT NULL
);

-- Main table
--
CREATE TABLE extable (
    id      integer DEFAULT nextval('seq_extable') CONSTRAINT pk_extable PRIMARY KEY,
    udid    varchar(22) DEFAULT NULL UNIQUE,
    name    varchar(255) NOT NULL default '',
    type    char(1) NOT NULL REFERENCES extype(typeid) ON DELETE CASCADE,
    stime   timestamp without time zone default (now() at time zone 'UTC'),
    deleted integer default 0
);

-- Trigger
-- Resides in private namespace ('confdb' schema)
--
CREATE OR REPLACE FUNCTION upd_timestamp() RETURNS TRIGGER AS $$
BEGIN
    if OLD.stime = NEW.stime then
        NEW.stime := now() at time zone 'UTC';
    end if;
    return NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER trg_extable_upd_time BEFORE UPDATE
    ON extable FOR EACH ROW
    EXECUTE PROCEDURE upd_timestamp();


--
-- Here goes 'public' section
-- Following objects will be accessible to apl user and should be placed into 'apl' schema
-- 1. Views
--    1a. associated rules
-- 2. API functions 
--    2a. associated composite types
--    2b. associated aggregates
--
SET search_path TO apl;

--
-- Create views and rules for lookup table
-- Assume that it contains static data that cannot be changed after installation
-- Create rules for INSERT, UPDATE and DELETE that does nothing. Only SELECT is allowed to apl
--
CREATE OR REPLACE VIEW extype as select * from confdb.extype;
CREATE OR REPLACE RULE v_extype_ins AS ON INSERT TO extype
    DO INSTEAD NOTHING;
CREATE OR REPLACE RULE v_extype_upd AS ON UPDATE TO extype
    DO INSTEAD NOTHING;
CREATE OR REPLACE RULE v_extype_del AS ON DELETE TO extype
 	DO INSTEAD NOTHING;

--
-- Create views and rules for main table
-- INSERT/UPDATE/DLETE is allowed to apl
--
CREATE OR REPLACE VIEW extable as select * from confdb.extable;
CREATE OR REPLACE RULE v_extable_ins AS ON INSERT TO extable
    DO INSTEAD
	INSERT INTO confdb.extable VALUES (NEW.*);
CREATE OR REPLACE RULE v_extable_upd AS ON UPDATE TO extable
 	DO INSTEAD
 	UPDATE confdb.extable
 	SET
 	    udid=NEW.udid,
 	    name=NEW.name,
 	    deleted=NEW.deleted,
 	    stime=NEW.stime
 	WHERE id=OLD.id AND deleted=OLD.deleted;
CREATE OR REPLACE RULE v_extable_del AS ON DELETE TO extable
 	DO INSTEAD
	DELETE FROM confdb.extable 
	WHERE id=OLD.id AND deleted=OLD.deleted;
--
-- Assign default values to view cols as the original table has
--
ALTER VIEW extable ALTER COLUMN udid SET DEFAULT NULL;
ALTER VIEW extable ALTER COLUMN name SET DEFAULT '';
ALTER VIEW extable ALTER COLUMN stime SET DEFAULT (now() at time zone 'UTC');
ALTER VIEW extable ALTER COLUMN deleted SET DEFAULT 0;

--
-- Create replacement views for sequences
-- For now algorythm is as follows:
--
-- 1. Create view with a single field of integer type
-- 2. Create ON SELECT rule for that view which returns currval('your_private_sequence');
-- 3. Create ON UPDATE rule which calls nextval('your_private_sequence');
-- 4. Create public proxy functions that replaces calls to currval and nextval of original sequence
--    E.g. nextval_seq_extable and currval_seq_extable
CREATE OR REPLACE VIEW v_seq_extable (currval) AS SELECT 0::bigint;

CREATE OR REPLACE RULE "_RETURN" AS ON SELECT TO v_seq_extable DO INSTEAD
    SELECT currval('confdb.seq_extable');
CREATE OR REPLACE RULE v_seq_extable_nextval AS ON UPDATE TO v_seq_extable
DO INSTEAD
    SELECT nextval('confdb.seq_extable');

CREATE OR REPLACE FUNCTION nextval_seq_extable() RETURNS int AS $$
DECLARE
    val bigint;
BEGIN
    UPDATE v_seq_extable SET currval=DEFAULT;
    SELECT currval FROM v_seq_extable INTO val;
    RETURN val;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION currval_seq_extable() RETURNS int AS $$
DECLARE
    val bigint;
BEGIN
    SELECT currval FROM v_seq_extable INTO val;
    RETURN val;
END;
$$ LANGUAGE plpgsql;

-- Now we can set default value of the 'id' column of our view to proxy function return value
--
ALTER VIEW extable ALTER COLUMN id SET DEFAULT nextval_seq_extable();

--
-- Create public routines and associated composite types
--
CREATE TYPE ex_ctype AS (id integer, udid varchar, name varchar, typename varchar);

CREATE OR REPLACE FUNCTION getByType(typeid char(1)) RETURNS SETOF ex_ctype AS $$
DECLARE
   result ex_ctype;
BEGIN
    FOR result in
        SELECT e.id, e.udid, e.name, et.name
            FROM extable e INNER JOIN extype et ON e.type=et.typeid
            WHERE
        	et.typeid=typeid
            ORDER BY e.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- 
-- Finally, grant permissions on newly created views
-- DDL is forbidden for apl
-- Grant select only on lookup tables and select/insert/update/delete on others
--
GRANT select ON extype to apl;
GRANT select,insert,update,delete ON extable to apl;
--
-- Grant permissions on sequences
-- We must grant apl select and update on confdb original sequences, too
--
GRANT select,update ON v_seq_extable TO apl;
GRANT select,update ON confdb.seq_extable TO apl;
