
--DROP TABLE eventpolicy CASCADE;
--DROP TABLE workflow CASCADE;
--DROP TABLE eventsource CASCADE;
--DROP TABLE eventsourceworkflow CASCADE;
--DROP TABLE eventpolicycontent;
--DROP TABLE eventlifespan CASCADE;
--DROP TABLE eventstatetag;
--DROP TABLE eventtag;
--DROP TABLE eventstate CASCADE;
--DROP TABLE eventpolicystorage;
--DROP TABLE storagepolicy;
--DROP TABLE storagertntype;
--DROP TABLE eventtype;

--DROP TABLE event_emailer_tag;
--DROP TABLE event_emailer_snapshot;

--DROP FUNCTION el_get_workflow(integer, integer, integer);
--DROP FUNCTION el_get_workflow_context(integer);
--DROP FUNCTION el_get_object_info(integer);

--DROP TYPE EL_Workflow;
--DROP TYPE EL_Object_Info;
--DROP TYPE EL_Workflow_Context;



CREATE TABLE eventpropertydef (
	name			varchar		not null,
	description		varchar,
	CONSTRAINT pk_eventpropertydef PRIMARY KEY(name)
);



CREATE TABLE eventpolicy (
	eventpolicy		integer		not null,
	name			varchar		not null,
	description		varchar,
	protected		bool 		not null default false,
	CONSTRAINT pk_eventpolicy PRIMARY KEY(eventpolicy)
);



CREATE TABLE workflow (
	workflow		integer		not null,
	name			varchar		not null,
	description		varchar,
	context			varchar		not null,
	protected		bool		not null default false,
	deleted			bool		not null default false,
	CONSTRAINT pk_workflow PRIMARY KEY(workflow)
);



CREATE TABLE eventsource (
	eventsource		integer		not null,
	name			varchar		not null,
	description		varchar,
	CONSTRAINT pk_eventsource PRIMARY KEY(eventsource)
);



CREATE TABLE eventsourceworkflow (
	eventsource			integer,
	workflow			integer,
	CONSTRAINT pk_eventsourceworkflow PRIMARY KEY(eventsource, workflow),
	CONSTRAINT ref_eventsourceworkflow_src FOREIGN KEY (eventsource) REFERENCES eventsource,
	CONSTRAINT ref_eventsourceworkflow_wfl FOREIGN KEY (workflow) REFERENCES workflow ON DELETE CASCADE
);



CREATE TABLE eventpolicycontent (
	eventpolicy			integer,
	eventsource			integer		not null,
	workflow			integer		not null,
	infworkflow			integer		not null,
	parameters			varchar,
	CONSTRAINT pk_eventpolicycontent PRIMARY KEY(eventpolicy, eventsource),
	CONSTRAINT ref_eventpolicycontent_policy FOREIGN KEY (eventpolicy) REFERENCES eventpolicy,
	CONSTRAINT ref_eventpolicycontent_src_wlf FOREIGN KEY (eventsource, workflow) REFERENCES  eventsourceworkflow ON UPDATE CASCADE,
	CONSTRAINT ref_eventpolicycontent_src_infwlf FOREIGN KEY (eventsource, infworkflow) REFERENCES  eventsourceworkflow (eventsource, workflow) ON DELETE CASCADE ON UPDATE CASCADE
);



CREATE TABLE eventlifespan (
	eventlifespan	integer,
	name			varchar		not null,
	description		varchar,
	CONSTRAINT pk_eventlifespan PRIMARY KEY(eventlifespan)
);



CREATE TABLE eventpolicystorage (
	eventpolicy             integer,
	eventpriority           integer,
	preserverhours          integer not null,
	CONSTRAINT pk_eventpolicystorage PRIMARY KEY(eventpolicy, eventpriority),
	CONSTRAINT ref_eventpolicystorage_eventpolicy FOREIGN KEY (eventpolicy) REFERENCES eventpolicy ON DELETE CASCADE,
	CONSTRAINT ref_eventpolicystorage_eventpriority FOREIGN KEY (eventpriority) REFERENCES eventpriority,
	CONSTRAINT c_eventpolicystorage_preserverhours CHECK (preserverhours >= -2)
);



CREATE TABLE eventtype (
	eventtype		integer,
	name			varchar		not null,
	description		varchar,
	CONSTRAINT pk_eventtype PRIMARY KEY(eventtype)
);



CREATE TABLE eventstate (
	eventstate			integer,
	workflow			integer		not null,
	name				varchar		not null,
	description			varchar,
	actions				varchar,
	CONSTRAINT pk_eventstate PRIMARY KEY(eventstate),
	CONSTRAINT ref_eventstate_workflow FOREIGN KEY (workflow) REFERENCES workflow ON DELETE CASCADE
);



CREATE TABLE eventtag (
	eventtag			integer,
	name				varchar		not null,
	description			varchar,
	CONSTRAINT pk_eventtag PRIMARY KEY(eventtag)
);



CREATE TABLE eventstatetag (
	eventstate			integer		not null,
	eventtag			integer		not null,
	CONSTRAINT pk_eventstatetag PRIMARY KEY(eventstate, eventtag),
	CONSTRAINT ref_eventstatetag_eventstate FOREIGN KEY (eventstate) REFERENCES eventstate,
	CONSTRAINT ref_eventstatetag_eventtag FOREIGN KEY (eventtag) REFERENCES eventtag
);



CREATE TABLE storagertntype (
	storagertntype	integer,
	name			varchar		not null,
	description		varchar,
	CONSTRAINT pk_storagertntype PRIMARY KEY(storagertntype)
);



CREATE TABLE storagepolicy (
	storagepolicy			integer,
	name					varchar		not null,
	description				varchar,
	storagertntype			integer,
	preservedaysmin			integer		default 0,
	preservedaysmax			integer		default 0,
	frdtype					varchar,
	frdschedule				varchar,	
	protected				bool		not null default false,
	CONSTRAINT pk_storagepolicy PRIMARY KEY(storagepolicy),
	CONSTRAINT ref_storagepolicy_storagertntype FOREIGN KEY (storagertntype) REFERENCES storagertntype,
	CONSTRAINT c_storagepolicy_preservedays CHECK (preservedaysmin <= preservedaysmax)
);


-- AF: event emailer addition to EventLog objects

CREATE TABLE event_emailer_tag (			-- TABLE to trace which events were already processed for particular user and when
	userid 		integer,			-- User ID for whom tag is maintained
	last_eventid 	integer,			-- last eventif processed (not necessarily sent to user)
	last_send 	bigint,				-- last UNIX timestamp when email was sent
	--
	CONSTRAINT pk_event_emailer_tag PRIMARY KEY(userid)
);

CREATE SEQUENCE seq_event_emailer_snapshot;

CREATE TABLE event_emailer_snapshot (
 	snapshotid      integer not null, -- DEFAULT nextval('seq_evet_emailer_snapshot')
 	userid 		integer not null,
 	eventid 	integer not null,
 	ts_created 	timestamp without time zone default (now() at time zone 'UTC')
	--
	-- CONSTRAINT pk_event_emailer_snapshot PRIMARY KEY (snapshotid,userid,eventid)
);

CREATE INDEX event_emailer_snapshot_snapshot ON event_emailer_snapshot(snapshotid);

--
--
--

INSERT INTO eventlifespan (eventlifespan,name,description) VALUES (0 ,'Transient' ,'');
INSERT INTO eventlifespan (eventlifespan,name,description) VALUES (1 ,'Persisted' ,'');


INSERT INTO eventtype (eventtype, name, description) VALUES (0, 'Informational', '');
INSERT INTO eventtype (eventtype, name, description) VALUES (1, 'Alert', '');


INSERT INTO eventsource (eventsource,name,description) VALUES (1 ,'Motion detector' ,'');
INSERT INTO eventsource (eventsource,name,description) VALUES (2 ,'Camera Motion detector' ,'');
INSERT INTO eventsource (eventsource,name,description) VALUES (3 ,'Sensor' ,'');
INSERT INTO eventsource (eventsource,name,description) VALUES (4 ,'User' ,'');
INSERT INTO eventsource (eventsource,name,description) VALUES (5 ,'Access Control System' ,'');
INSERT INTO eventsource (eventsource,name,description) VALUES (6, 'Analytics', '');


INSERT INTO eventtag (eventtag, name, description) VALUES(1, 'Needs announcement confirmation', '');
INSERT INTO eventtag (eventtag, name, description) VALUES(2, 'Needs supervisor attention', '');
INSERT INTO eventtag (eventtag, name, description) VALUES(3, 'Requires Acknowledgment', '');
INSERT INTO eventtag (eventtag, name, description) VALUES(4, 'Acknowledged', '');
INSERT INTO eventtag (eventtag, name, description) VALUES(5, 'Updatable', '');
INSERT INTO eventtag (eventtag, name, description) VALUES(6, 'EADA enabled', '');
INSERT INTO eventtag (eventtag, name, description) VALUES(7, 'Clearable', '');
INSERT INTO eventtag (eventtag, name, description) VALUES(8, 'Cleared', '');
INSERT INTO eventtag (eventtag, name, description) VALUES(9, 'Final state', '');
INSERT INTO eventtag (eventtag, name, description) VALUES(10, 'GUI pop up', '');
INSERT INTO eventtag (eventtag, name, description) VALUES(11, 'Marked', 'Event displays with red background');


INSERT INTO eventpropertydef(name, description) VALUES ('MIGRATE', 'Event migration flag: 0-not migrate, 1-migrate, 2-migrated');
INSERT INTO eventpropertydef(name, description) VALUES ('ACS_EVENTID', 'Id of event assigned by Access Control System');
INSERT INTO eventpropertydef(name, description) VALUES ('USERID', 'Objid of user who created event');


INSERT INTO storagertntype (storagertntype,name,description) VALUES (1 ,'Fixed' ,'');
INSERT INTO storagertntype (storagertntype,name,description) VALUES (2 ,'Range' ,'');
INSERT INTO storagertntype (storagertntype,name,description) VALUES (3 ,'Space limited' ,'');
INSERT INTO storagertntype (storagertntype,name,description) VALUES (4 ,'Frame decimation' ,'');
-- CIRRUS_REMOVE_BEGIN
INSERT INTO storagertntype (storagertntype,name,description) VALUES (5 ,'Motions and Events' ,'');
INSERT INTO storagertntype (storagertntype,name,description) VALUES (6 ,'Events only' ,'');
-- CIRRUS_REMOVE_END

INSERT INTO storagepolicy (storagepolicy, name, description,
	storagertntype, preservedaysmin, preservedaysmax, frdtype, frdschedule,	protected)
	VALUES (1, 'STD Space limited', 'Storage policy with initially defined parameters', 3, 0, 1, null, null, true);
-- CIRRUS_REMOVE_BEGIN
INSERT INTO storagepolicy (storagepolicy, name, description,
	storagertntype, preservedaysmin, preservedaysmax, frdtype, frdschedule,	protected)
	VALUES (-5, 'STD Motions and Events','In this mode, Motion Detection is executed in background even if not specifically enabled in individual camera (events are not generated upon detection). Camera will store video only if motion is detected or if video is "tagged" by any Event (Motion Detection, Analytics, associated ACS event, etc..). If you enable Motion Detection on individual cameras, Events will be registered.', 5, 0, 1, null, null, true);

INSERT INTO storagepolicy (storagepolicy, name, description,
	storagertntype, preservedaysmin, preservedaysmax, frdtype, frdschedule,	protected)
	VALUES (-6, 'STD Events only', 'Camera will store video only if it is "tagged" by any Event (Motion Detection, Analytics, associated ACS event, etc..). Motion Detection is not automatically enabled in this mode. You have to specifically enable it in individual cameras', 6, 0, 1, null, null, true);
-- CIRRUS_REMOVE_END



--
-- Alerts workflow.
--

INSERT INTO workflow (workflow, protected, name,description,context) 
SELECT COALESCE(max(workflow),0) + 1, true, 'Simple Alert + merge' ,'Two state (new/cleared) workflow' ,
'<?xml version="1.0" encoding="us-ascii"?>
<scxml version="1.0" xmlns="http://www.w3.org/2005/SCXML" xmlns:skm="http://skm.videonext.com/elog/workflow"> 
  <state id="empty">
    <transition event="create">
      <target next="new"/>
    </transition>
  </state>

  <state id="new">
    <datamodel>
      <data name="statedesc">
        <skm:statedesc desc="Event is new" tags="3,5,10,11"/>
      </data>
    </datamodel>

    <onentry>
      <send target="elog_actions" targettype="x-runtime" event="db_append"/>
    </onentry>     
    <transition event="update">
      <target next="cleared"/>
    </transition>
    <transition event="ack">
      <target next="cleared"/>
    </transition>
  </state>

  <state id="cleared">
    <datamodel>
      <data name="statedesc">
        <skm:statedesc desc="Cleared" tags="4,5,8,9"/>
      </data>
    </datamodel>
    <onentry>
      <send target="elog_actions" targettype="x-runtime" event="db_update"/>
    </onentry>

    <transition event="update">
      <target next="cleared"/>
    </transition>
  </state>
</scxml>'
FROM workflow;

INSERT INTO eventsourceworkflow (workflow, eventsource)
SELECT w.workflowid, s.eventsource
  FROM (SELECT max(workflow) as workflowid FROM workflow) w,
       (SELECT eventsource FROM eventsource) s;


-- New state

INSERT INTO eventstate (eventstate, workflow, name, description, actions)
SELECT COALESCE(s.eventstateid, 0) + 1, w.workflowid, 'new', 'New', null
  FROM (SELECT max(eventstate) as eventstateid FROM eventstate) s,
       (SELECT max(workflow) as workflowid FROM workflow) w;

INSERT INTO eventstatetag (eventstate, eventtag)
SELECT s.eventstateid, t.eventtagid
  FROM (SELECT max(eventstate) as eventstateid FROM eventstate) s,
       (SELECT 3 as eventtagid
            UNION ALL
        SELECT 5
            UNION ALL
        SELECT 10
            UNION ALL
        SELECT 11) t;


-- Cleared state

INSERT INTO eventstate (eventstate, workflow, name, description, actions)
SELECT s.eventstateid + 1, w.workflowid, 'cleared', 'Cleared', null
  FROM (SELECT max(eventstate) as eventstateid FROM eventstate) s,
       (SELECT max(workflow) as workflowid FROM workflow) w;

INSERT INTO eventstatetag (eventstate, eventtag)
SELECT s.eventstateid, t.eventtagid
  FROM (SELECT max(eventstate) as eventstateid FROM eventstate) s,
       (SELECT 4 as eventtagid
            UNION ALL
        SELECT 5
            UNION ALL
        SELECT 8
            UNION ALL
        SELECT 9) t;



--
-- Simple informational (single state) workflow
--

INSERT INTO workflow (workflow, protected, name,description,context) 
SELECT COALESCE(max(workflow), 0) + 1, true, 'Simple Info / no merge' ,'Single state workflow' ,
'<?xml version="1.0" encoding="us-ascii"?>
<scxml version="1.0" xmlns="http://www.w3.org/2005/SCXML" xmlns:skm="http://skm.videonext.com/elog/workflow"> 
  <state id="empty">
    <transition event="create">
      <target next="new"/>
    </transition>
  </state>

  <state id="new">
    <datamodel>
      <data name="statedesc">
        <skm:statedesc desc="New" tags="4,5,8,9"/>
      </data>
    </datamodel>
    <onentry>
      <send target="elog_actions" targettype="x-runtime" event="db_update"/>
    </onentry>

    <transition event="update">
      <target next="new"/>
    </transition>
  </state>
</scxml>'
FROM workflow;


INSERT INTO eventsourceworkflow (workflow, eventsource)
SELECT w.workflowid, s.eventsource
  FROM (SELECT max(workflow) as workflowid FROM workflow) w,
       (SELECT eventsource FROM eventsource) s;


-- New state

INSERT INTO eventstate (eventstate, workflow, name, description, actions)
SELECT COALESCE(s.eventstateid,0) + 1, w.workflowid, 'new', 'New', null
  FROM (SELECT max(eventstate) as eventstateid FROM eventstate) s,
       (SELECT max(workflow) as workflowid FROM workflow) w;

INSERT INTO eventstatetag (eventstate, eventtag)
SELECT s.eventstateid, t.eventtagid
  FROM (SELECT max(eventstate) as eventstateid FROM eventstate) s,
       (SELECT 4 as eventtagid
            UNION
        SELECT 5
            UNION
        SELECT 8
            UNION
        SELECT 9) t;


--
--
--



CREATE TYPE EL_Workflow as 
(
    id                     integer, 
    eventpolicy_parameters character varying
);


CREATE TYPE EL_Workflow_Context as 
(
    context character varying
);


CREATE TYPE EL_Object_Info as
(
    objid        integer,
    name         character varying(255),
    otype        character(1), 
    subtype      character(1),
    timedst      character varying,
    tz           character varying,
    stat_vl_rate character varying
);


CREATE OR REPLACE FUNCTION el_get_workflow(integer, integer, integer) RETURNS setof EL_Workflow AS
$$
DECLARE 
   obj_id     ALIAS FOR $1;
   source_id  ALIAS FOR $2;
   event_type ALIAS FOR $3;
   result EL_Workflow;
BEGIN
  FOR result in
    SELECT
        workflow.workflow, eventpolicycontent.parameters  
        FROM _obj_attr, eventpolicycontent, workflow
        WHERE _obj_attr.obj=obj_id
          AND _obj_attr.attr='EVENT_POLICY'
          AND CAST(eventpolicycontent.eventpolicy AS TEXT)=_obj_attr.val 
          AND eventpolicycontent.eventsource=source_id
          AND
             CASE WHEN event_type = -1 THEN
                     CASE substring(eventpolicycontent.parameters from 'eventtype=([[:digit:]])')
                        WHEN '0' THEN workflow.workflow=eventpolicycontent.infworkflow
                        ELSE workflow.workflow=eventpolicycontent.workflow 
                     END
                  WHEN event_type = 0 THEN
                     workflow.workflow=eventpolicycontent.infworkflow
                  ELSE 
                     workflow.workflow=eventpolicycontent.workflow
             END
  LOOP
     RETURN NEXT result;
  END LOOP;
 
RETURN;

END;
$$ LANGUAGE 'plpgsql';


CREATE OR REPLACE FUNCTION el_get_workflow_context(integer) RETURNS setof EL_Workflow_Context AS
$$
 DECLARE
    result EL_Workflow_Context; 
 BEGIN
   FOR result in
       SELECT context FROM workflow WHERE workflow=$1
   LOOP
       RETURN NEXT result;
   END LOOP;
 
   RETURN;

END;
$$ LANGUAGE 'plpgsql';


CREATE OR REPLACE FUNCTION el_get_object_info(integer) RETURNS setof EL_Object_Info AS
$$
DECLARE 
   obj_id   ALIAS FOR $1;
   result   EL_Object_Info;
BEGIN
  FOR result in
      SELECT o.obj, o.name, o.otype, o.subtype, COALESCE(a1.val, 'UTC'), 0, COALESCE(a2.val, '0')
        FROM _objs o
        LEFT OUTER JOIN _obj_attr a1 ON a1.obj=o.obj AND a1.attr='TIME_ZONE'
        LEFT OUTER JOIN _obj_attr a2 ON a2.obj=o.obj AND a2.attr='STAT_VL_RATE_LAST'
       WHERE o.obj = obj_id
  LOOP
     RETURN NEXT result;
  END LOOP;
 
RETURN;

END;
$$ LANGUAGE 'plpgsql';

