-- $Id: credentials.sql 22408 2011-04-04 16:38:57Z teetov $
-------------------------------------------------------------------------------
-- This module creates tables, procedures and triggers for credentials management
-------------------------------------------------------------------------------
-- Author: Andrey Starostin
-- Edited by:
-- QA by:  Christopher C Gettings
-- Copyright: videoNEXT Network Solution Inc
-------------------------------------------------------------------------------

--DROP TRIGGER trigger_links_update_link_type ON _links;
--DROP FUNCTION insert_link_type();
--DROP TABLE _links;
--DROP TABLE permission_type;
--DROP AGGREGATE concatFields(text);
--DROP TABLE _link_types CASCADE;
--DROP TABLE _cred_types CASCADE;
--DROP TABLE _links CASCADE;
--DROP TYPE object_list_with_credentials CASCADE;
--DROP TYPE object_list_without_credentials CASCADE;
--DROP TYPE object_list CASCADE;



-- consist mapping permission to credentials
CREATE TABLE permission_type (
    permission char(1) PRIMARY KEY,
    credentials varchar(64) DEFAULT '',
    description varchar,
    UNIQUE(permission, credentials)
);

-- list of permissions
INSERT INTO permission_type(permission, credentials, description) VALUES ('V', 'LVv', 'View');
INSERT INTO permission_type(permission, credentials, description) VALUES ('C', 'LVvApaeDMrC', 'Control');
INSERT INTO permission_type(permission, credentials, description) VALUES ('M', 'LVvApaeDMrCSPsm', 'Manage');


CREATE TABLE _link_types (
    link_type       char(3)         CONSTRAINT pk__link_types PRIMARY KEY,
    otype_res       char(1)         NOT NULL REFERENCES _obj_otypes(otype) ON DELETE CASCADE,
    otype_cons      char(1)         NOT NULL REFERENCES _obj_otypes(otype) ON DELETE CASCADE,
    allow_creds     boolean,        -- allow credentials to be attached to such link
    use_types       boolean,        -- allow otype/subtype analyzis (in other words - show credentials relevant to object or all of them)
    name            varchar
);

INSERT INTO _link_types VALUES ('D2S', 'D', 'S', False, True,  'Device to Set');
INSERT INTO _link_types VALUES ('G2S', 'G', 'S', False, True,  'GUI to Set');
INSERT INTO _link_types VALUES ('S2R', 'S', 'R', True, False,  'Set to Role');
INSERT INTO _link_types VALUES ('U2R', 'U', 'R', False, False, 'User to Role');
INSERT INTO _link_types VALUES ('X2S', 'X', 'S', False, True,  'External Object to Set');
INSERT INTO _link_types VALUES ('M2S', 'M', 'S', False, True,  'Model to Set');


CREATE TABLE _cred_types (
    cred            char(1),
    otype           char(1)     NOT NULL REFERENCES _obj_otypes(otype) ON DELETE CASCADE,
    subtypes        varchar,    -- combination of subtype letter where credential makes sense
    name            varchar,
    description     varchar,
    CONSTRAINT pk_cred_types PRIMARY KEY (cred, otype)
);

INSERT INTO _cred_types VALUES ('S', 'D', 'GCARS',   'Setup',       'Device Setup Settings');
INSERT INTO _cred_types VALUES ('L', 'D', 'GCARS',   'Live',        'Live Access');
INSERT INTO _cred_types VALUES ('A', 'D', 'GCA',     'Archive',     'Archive Access');
INSERT INTO _cred_types VALUES ('C', 'D', 'GCAR',    'Control',     'Control Device');
INSERT INTO _cred_types VALUES ('P', 'D', 'GC',      'PTZ:Mng',     'Camera PTZ: Manage');
INSERT INTO _cred_types VALUES ('p', 'D', 'GC',      'PTZ:Use',     'Camera PTZ: Control');
INSERT INTO _cred_types VALUES ('v', 'D', 'GCARS',   'EL:View',     'Event Log: View/Query');
INSERT INTO _cred_types VALUES ('a', 'D', 'GCARS',   'EL:Add',      'Event Log: Add Record');
INSERT INTO _cred_types VALUES ('e', 'D', 'GCARS',   'EL:Edit',     'Event Log: Edit Record');
INSERT INTO _cred_types VALUES ('s', 'G', 'G*',      'Setup',       'GUI Setup Settings');
INSERT INTO _cred_types VALUES ('V', 'G', 'G*VW',    'View',        'Open GUI for View');
INSERT INTO _cred_types VALUES ('D', 'D', 'C',       'Download',    'Download AVI clip');
INSERT INTO _cred_types VALUES ('M', 'D', 'W',       'vMX:Mon',     'Manipulate vMX Monitors');
INSERT INTO _cred_types VALUES ('m', 'D', 'W',       'vMX:MngCfg',  'Manage saved vMX configs');
INSERT INTO _cred_types VALUES ('r', 'D', 'W',       'vMX:Recall',  'Recall saved vMX configs');
INSERT INTO _cred_types VALUES ('C', 'X', 'GSDPA',   'Control',     'Control Device');
INSERT INTO _cred_types VALUES ('v', 'X', 'SDPA',    'EL:View',     'Event Log: View/Query');
INSERT INTO _cred_types VALUES ('a', 'X', 'SDPA',    'EL:Add',      'Event Log: Add Record');
INSERT INTO _cred_types VALUES ('e', 'X', 'SDPA',    'EL:Edit',     'Event Log: Edit Record');
INSERT INTO _cred_types VALUES ('I', 'D', 'GCARS',   'Email',       'Email on Event');
INSERT INTO _cred_types VALUES ('I', 'X', 'GSDPA',   'Email',       'Email on Event');


CREATE TABLE _links (
    obj_res      integer REFERENCES _objs(obj) ON DELETE CASCADE,
    obj_cons     integer REFERENCES _objs(obj) ON DELETE CASCADE,
    link_type    char(3) REFERENCES _link_types(link_type) ON DELETE CASCADE,
    permission   char(1) REFERENCES permission_type(permission) ON DELETE CASCADE,
    special_credentials char(64) DEFAULT '', -- like I for email on event
    protected    integer DEFAULT 0,
    level        integer DEFAULT 0,    -- allow to resolve conflicts over device control like PTZ
    updated_at   timestamp without time zone default (now() at time zone 'UTC'),
    --
    PRIMARY KEY(obj_res, obj_cons, link_type)
);

CREATE TRIGGER trg_links_upd_time BEFORE UPDATE
    ON _links FOR EACH ROW
    EXECUTE PROCEDURE upd_timestamp();

-- write link type
CREATE OR REPLACE FUNCTION insert_link_type() RETURNS TRIGGER AS $$
BEGIN
    NEW.link_type := (SELECT otype FROM _objs WHERE obj = NEW.obj_res) || '2' || (SELECT otype FROM _objs WHERE obj = NEW.obj_cons);
    return NEW;
END;
$$ LANGUAGE plpgsql;

-- write link type when insert or update record in _links table
CREATE TRIGGER trigger_links_update_link_type BEFORE INSERT OR UPDATE
    ON _links FOR EACH ROW
    EXECUTE PROCEDURE insert_link_type();

-- add set with matrix gui to new role
CREATE OR REPLACE FUNCTION addMatrix() RETURNS TRIGGER AS $$
BEGIN
    -- do not write udid for nodes
    if (NEW.otype = 'R' AND NEW.subtype = '*' AND NEW.protected = 0) then
        INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (13, NEW.obj, 1, 'V'); -- matrix
    end if;
    return NEW;
END;
$$ LANGUAGE plpgsql;

-- add set with matrix gui to new role after role creating
CREATE TRIGGER trigger_objs_insert AFTER INSERT
    ON _objs FOR EACH ROW
    EXECUTE PROCEDURE addMatrix();

-- ===================== PREDEFINED LINKS ======================================
-- Set -> Role (with permission)

-- "ADMIN ROLE", which is the only one get permissions to all GUIs (Admin/Matrix) and "manage" permissions for "All xxxx" (cameras/relays/etc..) sets
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (1, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (2, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (3, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (4, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (5, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (6, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (7, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (8, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (9, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (10, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (11, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (12, 31, 1, 'M'); -- all guis
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (14, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (15, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (16, 31, 1, 'M');
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (99, 31, 0, 'M');

-- "VIEWER ROLE": receives permission to "Matrix" and "VIEW" super-attributes for "My set"
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (13, 32, 1, 'V'); -- matrix
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (99, 32, 0, 'V');

-- "OPERATOR ROLE": receives permission to "Matrix" and "CONTROL" super-attributes for "My set"
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (13, 33, 1, 'V'); -- matrix
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (99, 33, 0, 'C');

-- "MANAGER ROLE": receives permission to "Matrix" and "MANAGE" super-attributes for "My Set"
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (13, 34, 1, 'V'); -- matrix
INSERT INTO _links (obj_res, obj_cons, protected, permission) VALUES (99, 34, 0, 'M');

-- GUI -> Set
INSERT INTO _links (obj_res, obj_cons, protected) VALUES (51, 12, 1); -- control panel to all guis
INSERT INTO _links (obj_res, obj_cons, protected) VALUES (52, 12, 1); -- matrix to all guis
INSERT INTO _links (obj_res, obj_cons, protected) VALUES (52, 13, 1); -- matrix to matrix

-- User -> Role
INSERT INTO _links (obj_res, obj_cons, protected) VALUES (21, 31, 1); -- admin to Admin Role
INSERT INTO _links (obj_res, obj_cons, protected) VALUES (22, 33, 1); -- guard to Operator Role

-- concatenate many fields into one field
CREATE AGGREGATE concatFields(
    BASETYPE = text,
    SFUNC = textcat,
    STYPE = text,
    INITCOND = ''
);

-- object list type
CREATE TYPE object_list_with_credentials AS (obj integer, udid varchar, name varchar, description varchar, location varchar, otype char(1), subtype char(1), protected integer, permission varchar, credentials varchar);


-- select object credentials for user
CREATE OR REPLACE FUNCTION getObject(userobj integer, objectobj integer) RETURNS SETOF object_list_with_credentials AS $$
DECLARE
   result object_list_with_credentials;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected, concatFields(set2role.permission) as permission, concatFields(permission_type.credentials || set2role.special_credentials) as credentials
            FROM _objs, permission_type, _links as set2role, _links as user2role, _links as object2set
            WHERE
                set2role.link_type = 'S2R'
                AND user2role.link_type = 'U2R'
                AND (object2set.link_type = 'D2S' OR object2set.link_type = 'X2S' OR object2set.link_type = 'G2S')
                AND _objs.obj = object2set.obj_res
                AND set2role.obj_cons = user2role.obj_cons
                AND object2set.obj_cons = set2role.obj_res
                AND set2role.permission = permission_type.permission
                AND user2role.obj_res = userobj
                AND object2set.obj_res = objectobj
            GROUP BY _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select objects with credentials for user and role with specified otype and subtype (roleobj, objtype and objsubtype can be specified as NULL)
CREATE OR REPLACE FUNCTION getObjects(userobj integer, roleobj integer, objtype char(1), objsubtype char(1)) RETURNS SETOF object_list_with_credentials AS $$
DECLARE
   result object_list_with_credentials;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected, concatFields(set2role.permission) AS permission, concatFields(permission_type.credentials || set2role.special_credentials) as credentials
            FROM _objs, permission_type, _links as set2role, _links as user2role, _links as object2set
            WHERE
                (objtype IS NULL OR _objs.otype = objtype)
                AND (objsubtype IS NULL OR _objs.subtype = objsubtype)
                AND _objs.deleted = 0
                AND set2role.link_type = 'S2R'
                AND user2role.link_type = 'U2R'
                AND (object2set.link_type = 'D2S' OR object2set.link_type = 'X2S' OR object2set.link_type = 'G2S')
                AND _objs.obj = object2set.obj_res
                AND set2role.obj_cons = user2role.obj_cons
                AND object2set.obj_cons = set2role.obj_res
                AND set2role.permission = permission_type.permission
                AND user2role.obj_res = userobj
                AND (roleobj IS NULL OR user2role.obj_cons = roleobj)
            GROUP BY _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select objects from set with credentials for user and role with specified otype and subtype (roleobj, objtype and objsubtype can be specified as NULL)
CREATE OR REPLACE FUNCTION getObjects(userobj integer, roleobj integer, setobj integer, objtype char(1), objsubtype char(1)) RETURNS SETOF object_list_with_credentials AS $$
DECLARE
   result object_list_with_credentials;
BEGIN
    FOR result in
        SELECT object.obj, object.udid, object.name, object.description, object.location, object.otype, object.subtype, object.protected, object.permission, object.credentials
            FROM (
                SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected, concatFields(set2role.permission) AS permission, concatFields(permission_type.credentials || set2role.special_credentials) as credentials
                    FROM _objs, permission_type, _links as set2role, _links as user2role, _links as object2set
                    WHERE
                        (objtype IS NULL OR _objs.otype = objtype)
                        AND (objsubtype IS NULL OR _objs.subtype = objsubtype)
                        AND _objs.deleted = 0
                        AND set2role.link_type = 'S2R'
                        AND user2role.link_type = 'U2R'
                        AND (object2set.link_type = 'D2S' OR object2set.link_type = 'X2S' OR object2set.link_type = 'G2S')
                        AND _objs.obj = object2set.obj_res
                        AND set2role.obj_cons = user2role.obj_cons
                        AND object2set.obj_cons = set2role.obj_res
                        AND set2role.permission = permission_type.permission
                        AND user2role.obj_res = userobj
                        AND (roleobj IS NULL OR user2role.obj_cons = roleobj)
                    GROUP BY _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected
            ) AS object, _links as object2set
            WHERE
                (object2set.link_type = 'D2S' OR object2set.link_type = 'X2S' OR object2set.link_type = 'G2S')
                AND object.obj = object2set.obj_res
                AND object2set.obj_cons = setobj
            ORDER BY object.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select objects from set, without credentials with specified otype and subtype (otype and subtype can be specified as NULL)
CREATE OR REPLACE FUNCTION getObjectsFromSet(setobj integer, otype char(1), subtype char(1)) RETURNS SETOF object_list_with_credentials AS $$
DECLARE
   result object_list_with_credentials;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected
            FROM _objs, _links as object2set
            WHERE
                ($2 IS NULL OR _objs.otype = $2)
                AND ($3 IS NULL OR _objs.subtype = $3)
                AND _objs.deleted = 0
                AND (object2set.link_type = 'D2S' OR object2set.link_type = 'X2S' OR object2set.link_type = 'G2S')
                AND object2set.obj_res = _objs.obj
                AND object2set.obj_cons = $1
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select sets from role
CREATE OR REPLACE FUNCTION getSetsFromRole(roleobj integer) RETURNS SETOF object_list_with_credentials AS $$
DECLARE
   result object_list_with_credentials;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected, set2role.permission, permission_type.credentials || set2role.special_credentials
            FROM _objs, _links as set2role
            LEFT OUTER JOIN permission_type ON set2role.permission = permission_type.permission
            WHERE
                _objs.deleted = 0
                AND _objs.obj NOT IN (SELECT defaultsetobj FROM _obj_otype_subtype WHERE _obj_otype_subtype.disabled = 1)
                AND set2role.link_type = 'S2R'
                AND set2role.obj_res = _objs.obj
                AND set2role.obj_cons = $1
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';


-- object list without credentials type
CREATE TYPE object_list_without_credentials AS (setobj integer, obj integer, udid varchar, name varchar, description varchar, location varchar, otype char(1), subtype char(1), protected integer);

-- select objects from role with specified otype and subtype (otype and subtype can be specified as NULL)
CREATE OR REPLACE FUNCTION getObjectsFromRole(roleobj integer, otype char(1), subtype char(1)) RETURNS SETOF object_list_without_credentials AS $$
DECLARE
   result object_list_without_credentials;
BEGIN
    FOR result in
        SELECT set2role.obj_res as setobj, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected
            FROM _objs, _links as set2role, _links as object2set
            WHERE
                ($2 IS NULL OR _objs.otype = $2)
                AND ($3 IS NULL OR _objs.subtype = $3)
                AND _objs.deleted = 0
                AND set2role.link_type = 'S2R'
                AND (object2set.link_type = 'D2S' OR object2set.link_type = 'X2S' OR object2set.link_type = 'G2S')
                AND object2set.obj_res = _objs.obj
                AND set2role.obj_res = object2set.obj_cons
                AND set2role.obj_cons = $1
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- user list type
CREATE TYPE object_list AS (obj integer, udid varchar, name varchar, description varchar, location varchar, protected integer);

-- select all users
CREATE OR REPLACE FUNCTION getUsers() RETURNS SETOF object_list AS $$
DECLARE
   result object_list;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.protected
            FROM _objs
            WHERE
                _objs.deleted = 0
                AND _objs.otype = 'U'
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select users from role
CREATE OR REPLACE FUNCTION getUsers(roleobj integer) RETURNS SETOF object_list AS $$
DECLARE
   result object_list;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.protected
            FROM _objs, _links as user2role
            WHERE
                _objs.deleted = 0
                AND user2role.link_type = 'U2R'
                AND _objs.obj = user2role.obj_res
                AND _objs.otype = 'U'
                AND user2role.obj_cons = roleobj
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select all sets
CREATE OR REPLACE FUNCTION getSets() RETURNS SETOF object_list AS $$
DECLARE
   result object_list;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.protected
            FROM _objs
            WHERE
                _objs.deleted = 0
                AND _objs.obj NOT IN (SELECT defaultsetobj FROM _obj_otype_subtype WHERE _obj_otype_subtype.disabled = 1)
                AND _objs.otype = 'S'
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select all sets with object
CREATE OR REPLACE FUNCTION getSets(objectobj integer) RETURNS SETOF object_list AS $$
DECLARE
   result object_list;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.protected
            FROM _objs, _links as object2set
            WHERE
                _objs.deleted = 0
                AND _objs.obj NOT IN (SELECT defaultsetobj FROM _obj_otype_subtype WHERE _obj_otype_subtype.disabled = 1)
                AND _objs.otype = 'S'
                AND object2set.obj_cons = _objs.obj
                AND object2set.obj_res = objectobj
                AND (object2set.link_type = 'D2S' OR object2set.link_type = 'X2S' OR object2set.link_type = 'G2S')
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select sets for user, role and object (role and object can be null)
CREATE OR REPLACE FUNCTION getSets(userobj integer, roleobj integer, objectobj integer) RETURNS SETOF object_list AS $$
DECLARE
   result object_list;
BEGIN
    FOR result in
        SELECT DISTINCT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.protected
            FROM _objs, _links as user2role, _links as set2role, _links as object2set
            WHERE
                _objs.deleted = 0
                AND _objs.obj NOT IN (SELECT defaultsetobj FROM _obj_otype_subtype WHERE _obj_otype_subtype.disabled = 1)
                AND user2role.link_type = 'U2R'
                AND set2role.link_type = 'S2R'
                AND _objs.obj = set2role.obj_res
                AND _objs.otype = 'S'
                AND user2role.obj_res = userobj
                AND (roleobj IS NULL OR user2role.obj_cons = roleobj)
                AND set2role.obj_cons = user2role.obj_cons
                AND (
                    objectobj IS NULL OR (
                        object2set.obj_res = objectobj
                        AND object2set.obj_cons = set2role.obj_res
                        AND (object2set.link_type = 'D2S' OR object2set.link_type = 'X2S' OR object2set.link_type = 'G2S')
                    )
                )
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select all roles
CREATE OR REPLACE FUNCTION getRoles() RETURNS SETOF object_list AS $$
DECLARE
   result object_list;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.protected
            FROM _objs
            WHERE
                _objs.deleted = 0
                AND _objs.otype = 'R'
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select roles for user
CREATE OR REPLACE FUNCTION getRoles(userobj integer) RETURNS SETOF object_list AS $$
DECLARE
   result object_list;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.protected
            FROM _objs, _links as user2role
            WHERE
                _objs.deleted = 0
                AND user2role.link_type = 'U2R'
                AND _objs.obj = user2role.obj_cons
                AND _objs.otype = 'R'
                AND user2role.obj_res = $1
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';

-- select roles for set
CREATE OR REPLACE FUNCTION getRolesForSet(setobj integer) RETURNS SETOF object_list AS $$
DECLARE
   result object_list;
BEGIN
    FOR result in
        SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.protected
            FROM _objs, _links as set2role
            WHERE
                _objs.deleted = 0
                AND set2role.link_type = 'S2R'
                AND _objs.obj = set2role.obj_cons
                AND _objs.otype = 'R'
                AND set2role.obj_res = $1
            ORDER BY _objs.name
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$ LANGUAGE 'plpgsql';
