--#  $Id: db_obj.sql 22408 2011-04-04 16:38:57Z teetov $
--#-----------------------------------------------------------------------------
--#  This module creates OBJs
--#-----------------------------------------------------------------------------
--#  Author: Andrey Fomenko, Alexiy Titov, Serg Pososhenko
--#  Edited by: Andrey Starostin
--#  QA by:  Christopher C Gettings
--#  Copyright: videoNEXT Network Solution Inc
--#-----------------------------------------------------------------------------


-- drop section for dev & test only, comment out before commiting into cvs
--DROP TABLE _objs CASCADE;
--DROP TABLE _obj_otypes CASCADE;
--DROP TABLE _obj_otype_subtype CASCADE;
--DROP TABLE _obj_attr CASCADE;
--DROP SEQUENCE seq_obj;
--DROP TABLE eventRecyclePolicy CASCADE;
--DROP TABLE _obj_editor_tmpl;
--DROP TABLE _obj_editor_att_tmpl;


CREATE SEQUENCE seq_obj start 100;
GRANT select, update ON seq_obj to apache;

CREATE TABLE _obj_otypes (
    otype           char(1)         CONSTRAINT pk__obj_types PRIMARY KEY,
    name            varchar
);

INSERT INTO _obj_otypes VALUES ('U', 'User');
INSERT INTO _obj_otypes VALUES ('R', 'Role');
INSERT INTO _obj_otypes VALUES ('S', 'Set');
INSERT INTO _obj_otypes VALUES ('D', 'Device');
INSERT INTO _obj_otypes VALUES ('G', 'GUI');
INSERT INTO _obj_otypes VALUES ('X', 'External objects');
INSERT INTO _obj_otypes VALUES ('M', '3D model');

CREATE TABLE _obj_otype_subtype (
    otype           char(1),
    subtype         char(1),
    name            varchar,
    template        varchar,
    defaultsetobj   integer DEFAULT null, -- obj of predefined set in which we should place object after creating
    disabled        integer NOT NULL DEFAULT 0, -- mark, if we can use this otype/subtype pair or not
    CONSTRAINT pk__obj_otypes PRIMARY KEY (otype, subtype),
    CONSTRAINT fk__obj_0types_otype FOREIGN KEY (otype) REFERENCES _obj_otypes ON DELETE CASCADE
);

INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('D', 'C', 'Camera', 'camera', 14);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('D', 'R', 'Relay', 'relay', 1);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('D', 'S', 'Sensor', 'sensor_skm', 2);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('D', 'A', 'Audio Source', 'audio', 3);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('D', 'V', 'vMX Monitor', 'monitor', 4);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('D', 'W', 'vMX Wall', 'wall', 5);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('D', 'E', 'External device', 'external', 6);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('D', 'N', 'Node', 'node', null);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('G', '*', 'GUI', 'gui', 12);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('G', 'C', 'Matrix configuration', 'configuration', null);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('G', 'L', 'vMX layout', 'vmxlayout', null);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('M', '3', '3D model', 'model3d', 15);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('M', 'L', 'Layer model', 'modellayer', 16);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('S', '*', 'Set', 'set', null);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('R', '*', 'Role', 'role', null);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('U', '*', 'User', 'user', null);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('X', 'G', 'Gateway', 'gateway', 7);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('X', 'S', 'Access Control Sensor', 'sensor_acsg', 8);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('X', 'D', 'Door', 'door', 9);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('X', 'P', 'Perimeter', 'perimeter', 10);
INSERT INTO _obj_otype_subtype(otype, subtype, name, template, defaultsetobj) VALUES ('X', 'A', 'Area', 'area', 11);

CREATE TABLE _objs (
    obj             integer         DEFAULT nextval('seq_obj') CONSTRAINT pk__objs PRIMARY KEY,
    udid            varchar(22)     DEFAULT NULL UNIQUE,
    node_id         integer         null references _objs(obj),   -- we require node ID to be present and uniquie in the domain
    node_ip         varchar,        -- host IP from Dev Confg
    node_alt_ip     varchar,        -- host IP -alternative from Dev Confg
    otype           char(1)         NOT NULL REFERENCES _obj_otypes(otype) ON DELETE CASCADE,
    subtype         char(1)         NOT NULL,
    name            varchar(255)    NOT NULL default '',
    description     varchar(255)    default '', -- user-descriptive name
    location        varchar(255)    default '', -- location for devices
    updated_at      timestamp without time zone default (now() at time zone 'UTC'),
    deleted         integer         default 0,
    protected       integer         default 0,
    stime           timestamp without time zone default (now() at time zone 'UTC'),
    rtime           timestamp without time zone default NULL,
    owner           integer         default 21, -- default owner is admin
    --
    CONSTRAINT fk__objs_type FOREIGN KEY (otype, subtype) REFERENCES _obj_otype_subtype
);

-- write udid to _objs
CREATE OR REPLACE FUNCTION updateUDID() RETURNS TRIGGER AS $$
BEGIN
    -- do not write udid for nodes
    if NOT (NEW.otype = 'D' AND NEW.subtype = 'N') then
        NEW.udid = upper(NEW.otype || NEW.subtype || NEW.obj);
    end if;
    return NEW;
END;
$$ LANGUAGE plpgsql;

-- write udid to _objs when insert new record in _objs table
CREATE TRIGGER trigger_obj_attr_insert BEFORE INSERT
    ON _objs FOR EACH ROW
    EXECUTE PROCEDURE updateUDID();

CREATE TABLE _obj_attr (
    obj integer REFERENCES _objs(obj),
    attr varchar,
    val varchar,
    --
    PRIMARY KEY(obj, attr)
);

-- write attr from _obj_attr to _objs on insert
CREATE OR REPLACE FUNCTION insertAttributes() RETURNS TRIGGER AS $$
BEGIN
    if NEW.attr = 'OBJID' then
        NEW.val = NEW.obj; 
    end if;
    if NEW.attr = 'DEVID' then
        SELECT lower(subtype) || obj into NEW.val 
    	    FROM _objs 
    	    WHERE obj=NEW.obj AND otype='D' AND subtype NOT IN ('C','L','N');
    end if;
    if NEW.attr = 'NAME' then
        UPDATE _objs
            SET name = NEW.val
            WHERE obj = NEW.obj;
    end if;
    if NEW.attr = 'LOCATION' then
        UPDATE _objs
            SET location = NEW.val
            WHERE obj = NEW.obj;
    end if;
    if NEW.attr = 'DESCRIPTION' then
        UPDATE _objs
            SET description = NEW.val
            WHERE obj = NEW.obj;
    end if;
    if NEW.attr = 'UDID' then
        if NEW.val = '' then
            SELECT udid INTO NEW.val FROM _objs WHERE obj = NEW.obj;
        else
            NEW.val = upper(NEW.val);
            UPDATE _objs
                SET udid = NEW.val
                WHERE obj = NEW.obj;
        end if;
    end if;
    NEW.val = COALESCE(NEW.val,'');
    return NEW;
END;
$$ LANGUAGE plpgsql;

-- write field to _objs from _obj_attr when insert record in _obj_attr table
CREATE TRIGGER trigger_obj_attr_insert BEFORE INSERT
    ON _obj_attr FOR EACH ROW
    EXECUTE PROCEDURE insertAttributes();

-- write attr from _obj_attr to _objs on update
CREATE OR REPLACE FUNCTION updateAttributes() RETURNS TRIGGER AS $$
BEGIN
    if NEW.attr = 'NAME' then
        UPDATE _objs
            SET name = NEW.val
            WHERE obj = NEW.obj;
    end if;
    if NEW.attr = 'LOCATION' then
        UPDATE _objs
            SET location = NEW.val
            WHERE obj = NEW.obj;
    end if;
    if NEW.attr = 'DESCRIPTION' then
        UPDATE _objs
            SET description = NEW.val
            WHERE obj = NEW.obj;
    end if;
    if NEW.attr = 'UDID' then
        NEW.val = upper(NEW.val);
        UPDATE _objs
            SET udid = NEW.val
            WHERE obj = NEW.obj;
    end if;
    NEW.val = COALESCE(NEW.val,'');
    return NEW;
END;
$$ LANGUAGE plpgsql;

-- write field to _objs from _obj_attr when update record in _obj_attr table
CREATE TRIGGER trigger_obj_attr_update BEFORE UPDATE
    ON _obj_attr FOR EACH ROW
    EXECUTE PROCEDURE updateAttributes();

-- =============== insert or update object's attribute =========================
CREATE FUNCTION merge_obj_attr(objid INT, name VARCHAR, value VARCHAR) RETURNS VOID AS $$
BEGIN
    BEGIN
        INSERT INTO _obj_attr (obj, attr, val) VALUES (objid, name, value);
    EXCEPTION WHEN unique_violation THEN
        UPDATE _obj_attr SET val = value WHERE obj = objid AND attr = name;
    END;
END;
$$ LANGUAGE plpgsql;

-- ==================== replication-related stuff ==============================
CREATE OR REPLACE function upd_timestamp() RETURNS TRIGGER AS $$
BEGIN
    if OLD.updated_at = NEW.updated_at then
        NEW.updated_at := now() at time zone 'UTC';
    end if;
    return NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER trg_objs_upd_time BEFORE UPDATE
    ON _objs FOR EACH ROW
    EXECUTE PROCEDURE upd_timestamp();

-- Set Status (quick respons simulation)
CREATE OR REPLACE FUNCTION ATTR_ARCHSTATE_ON_OF() RETURNS TRIGGER AS $$
BEGIN
    if NEW.attr = 'ARCHSTATE' then
        if NEW.val = 'off' AND OLD.val = 'on' then
            UPDATE _obj_attr SET val = 'stopping' WHERE obj = NEW.obj AND attr = 'STATUS';
        end if;
        if NEW.val = 'on' and OLD.val = 'off' then
            UPDATE _obj_attr SET val = 'starting' WHERE obj = NEW.obj AND attr = 'STATUS';
        end if;
    end if;
    return NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER OBJ_ATTR_TBU BEFORE update
    ON _obj_attr FOR EACH ROW
    EXECUTE PROCEDURE ATTR_ARCHSTATE_ON_OF();

-- ===================== PREDEFINED OBJECTS ====================================
-- predefined SETS, setobj should be defined in _obj_otype_subtype
INSERT INTO _objs (obj, otype, subtype, protected) values (1, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (1, 'NAME', 'All Relays');
INSERT INTO _obj_attr (obj, attr, val) values (1, 'DESCRIPTION', 'All Relays');

INSERT INTO _objs (obj, otype, subtype, protected) values (2, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (2, 'NAME', 'Sensors');
INSERT INTO _obj_attr (obj, attr, val) values (2, 'DESCRIPTION', 'Sensors');

INSERT INTO _objs (obj, otype, subtype, protected) values (3, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (3, 'NAME', 'All Audio');
INSERT INTO _obj_attr (obj, attr, val) values (3, 'DESCRIPTION', 'All Audio');

INSERT INTO _objs (obj, otype, subtype, protected) values (4, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (4, 'NAME', 'All Monitors');
INSERT INTO _obj_attr (obj, attr, val) values (4, 'DESCRIPTION', 'All Monitors');

INSERT INTO _objs (obj, otype, subtype, protected) values (5, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (5, 'NAME', 'All Walls');
INSERT INTO _obj_attr (obj, attr, val) values (5, 'DESCRIPTION', 'All Walls');

INSERT INTO _objs (obj, otype, subtype, protected) values (6, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (6, 'NAME', 'All External devices');
INSERT INTO _obj_attr (obj, attr, val) values (6, 'DESCRIPTION', 'All External devices');

INSERT INTO _objs (obj, otype, subtype, protected) values (7, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (7, 'NAME', 'All Gateways');
INSERT INTO _obj_attr (obj, attr, val) values (7, 'DESCRIPTION', 'All Gateways');

INSERT INTO _objs (obj, otype, subtype, protected) values (8, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (8, 'NAME', 'Access Control Sensors');
INSERT INTO _obj_attr (obj, attr, val) values (8, 'DESCRIPTION', 'All Access Control Sensors');

INSERT INTO _objs (obj, otype, subtype, protected) values (9, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (9, 'NAME', 'All Doors');
INSERT INTO _obj_attr (obj, attr, val) values (9, 'DESCRIPTION', 'All Doors');

INSERT INTO _objs (obj, otype, subtype, protected) values (10, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (10, 'NAME', 'All Perimeters');
INSERT INTO _obj_attr (obj, attr, val) values (10, 'DESCRIPTION', 'All Perimeters');

INSERT INTO _objs (obj, otype, subtype, protected) values (11, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (11, 'NAME', 'All Areas');
INSERT INTO _obj_attr (obj, attr, val) values (11, 'DESCRIPTION', 'All Areas');

INSERT INTO _objs (obj, otype, subtype, protected) values (12, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (12, 'NAME', 'All GUIs');
INSERT INTO _obj_attr (obj, attr, val) values (12, 'DESCRIPTION', 'All GUIs');

INSERT INTO _objs (obj, otype, subtype, protected) values (13, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (13, 'NAME', 'Matrix');
INSERT INTO _obj_attr (obj, attr, val) values (13, 'DESCRIPTION', 'Matrix');

INSERT INTO _objs (obj, otype, subtype, protected) values (14, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (14, 'NAME', 'All Cameras');
INSERT INTO _obj_attr (obj, attr, val) values (14, 'DESCRIPTION', 'All Cameras');

INSERT INTO _objs (obj, otype, subtype, protected) values (15, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (15, 'NAME', 'All 3D models');
INSERT INTO _obj_attr (obj, attr, val) values (15, 'DESCRIPTION', 'All 3D models');

INSERT INTO _objs (obj, otype, subtype, protected) values (16, 'S', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (16, 'NAME', 'All layer models');
INSERT INTO _obj_attr (obj, attr, val) values (16, 'DESCRIPTION', 'All layer models');

-- predefined, but not protected default set
INSERT INTO _objs (obj, otype, subtype, protected) values (99, 'S', '*', 0);
INSERT INTO _obj_attr (obj, attr, val) values (99, 'NAME', 'my set');
INSERT INTO _obj_attr (obj, attr, val) values (99, 'DESCRIPTION', 'Default set');

-- predefined USERS
INSERT INTO _objs (obj, otype, subtype, protected) values (21, 'U', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (21, 'NAME', 'admin');
INSERT INTO _obj_attr (obj, attr, val) values (21, 'DESCRIPTION', 'Administrator');
INSERT INTO _obj_attr (obj, attr, val) values (21, 'PASSWORD', '9CDE3F9C7F7012D817A4A54F41DED3E151AF6B66');
INSERT INTO _obj_attr (obj, attr, val) values (21, 'DEFAULT_ROLE', '31');
INSERT INTO _obj_attr (obj, attr, val) values (21, 'LANGUAGE', 'en_US');
INSERT INTO _obj_attr (obj, attr, val) values (21, 'EMAIL', '');
INSERT INTO _obj_attr (obj, attr, val) values (21, 'EMAIL_INTERVAL', '1');
INSERT INTO _obj_attr (obj, attr, val) values (21, 'WARN_VIDEO_PLAYERS', '');
INSERT INTO _obj_attr (obj, attr, val) values (21, 'MAX_VIDEO_PLAYERS', '');

INSERT INTO _objs (obj, otype, subtype, protected) values (22, 'U', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (22, 'NAME', 'guard');
INSERT INTO _obj_attr (obj, attr, val) values (22, 'DESCRIPTION', 'Security Guard');
INSERT INTO _obj_attr (obj, attr, val) values (22, 'PASSWORD', '9CDE3F9C7F7012D817A4A54F41DED3E151AF6B66');
INSERT INTO _obj_attr (obj, attr, val) values (22, 'DEFAULT_ROLE', '33');
INSERT INTO _obj_attr (obj, attr, val) values (22, 'LANGUAGE', 'en_US');
INSERT INTO _obj_attr (obj, attr, val) values (22, 'EMAIL', '');
INSERT INTO _obj_attr (obj, attr, val) values (22, 'EMAIL_INTERVAL', '1');
INSERT INTO _obj_attr (obj, attr, val) values (22, 'WARN_VIDEO_PLAYERS', '8');
INSERT INTO _obj_attr (obj, attr, val) values (22, 'MAX_VIDEO_PLAYERS', '10');

-- predefined ROLES
INSERT INTO _objs (obj, otype, subtype, protected) values (31, 'R', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (31, 'NAME', 'Admin Role');
INSERT INTO _obj_attr (obj, attr, val) values (31, 'DESCRIPTION', 'Admin Role');
INSERT INTO _obj_attr (obj, attr, val) values (31, 'PTZ_PRIORITY', '9');

INSERT INTO _objs (obj, otype, subtype, protected) values (32, 'R', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (32, 'NAME', 'Viewer Role');
INSERT INTO _obj_attr (obj, attr, val) values (32, 'DESCRIPTION', 'Viewer Role');
INSERT INTO _obj_attr (obj, attr, val) values (32, 'PTZ_PRIORITY', '3');

INSERT INTO _objs (obj, otype, subtype, protected) values (33, 'R', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (33, 'NAME', 'Operator Role');
INSERT INTO _obj_attr (obj, attr, val) values (33, 'DESCRIPTION', 'Operator Role');
INSERT INTO _obj_attr (obj, attr, val) values (33, 'PTZ_PRIORITY', '5');

INSERT INTO _objs (obj, otype, subtype, protected) values (34, 'R', '*', 1);
INSERT INTO _obj_attr (obj, attr, val) values (34, 'NAME', 'Manager Role');
INSERT INTO _obj_attr (obj, attr, val) values (34, 'DESCRIPTION', 'Manager Role');
INSERT INTO _obj_attr (obj, attr, val) values (34, 'PTZ_PRIORITY', '7');

-- predefined GUIs
INSERT INTO _objs (obj, otype, subtype, name, description, protected) values (51, 'G', '*', 'Control Panel', 'System parameters', 1);
INSERT INTO _obj_attr (obj, attr, val) values (51, 'URL', '/sdi/admin/');

INSERT INTO _objs (obj, otype, subtype, name, description, protected) values (52, 'G', '*', 'Matrix', 'Operator screens', 1);
INSERT INTO _obj_attr (obj, attr, val) values (52, 'URL', '/TibcoGI/JSXAPPS/mx/mx.php');

INSERT INTO _objs (obj, otype, subtype, name, description, protected) values (53, 'G', '*', 'Identity', 'System attributes', 1);
INSERT INTO _obj_attr (obj, attr, val) values (53, 'URL', '/sdi/admin/#identity');

-- Capacity Charts are removed since untested
-- INSERT INTO _objs (obj, otype, subtype, name, description, url, protected) values (54, 'G', '*', 'Capacity Charts', 'Capacity Charts', '/TibcoGI/JSXAPPS/Charter/Charter.html', 1);

-- ================== NEW EVENTLOG-RELATED STUFF =================================

--DROP TABLE eventPriority CASCADE;

create table eventPriority (
    eventpriority   smallint,
    name            varchar not null,
    description     varchar,
    colorname       varchar,
    colorcode       varchar,
    --
    CONSTRAINT pk_eventpriority PRIMARY KEY(eventpriority)
);

insert into eventPriority
    select 4, 'Critical', 'Critical Priority', 'Red', '#FF0000'
     union all
    select 3, 'High', 'High priority', 'Orange', '#F88017'
     union all
    select 2, 'Normal', 'Normal Priority', 'Yellow', '#FFFF00'
     union all
    select 1, 'Low', 'Low Priority', 'Green', '#00FF00';

-- ================== DATA BLOCKS =================================

CREATE SEQUENCE seqBlockId;

CREATE TABLE obj_block (
    obj integer REFERENCES _objs(obj),
    blockid integer default nextval('seqBlockId'),
    name varchar default '',
    description varchar default '',
    mime varchar(255) default 'application/octet-stream',
    block bytea not null,
    PRIMARY KEY (obj, blockid)
);


-- ================== LOGGER =================================

create sequence log_common_seq;
create sequence log_sys_seq;

grant select,update on log_common_seq to apache;

create table app_log(
    ind         bigint default nextval('log_common_seq'),
    date        timestamp without time zone default (now() at time zone 'UTC'),
    category    int,
    loglevel    int,
    programm    varchar,
    line        varchar,
    message     varchar
);

grant insert on app_log to apache;

create table sys_log(
    ind         bigint default nextval('log_common_seq'),
    cnt_seq     bigint default nextval('log_sys_seq'),
    date        timestamp without time zone,
    category    int,
    loglevel    int,
    programm    varchar,
    line        varchar,
    message     varchar,
    CONSTRAINT  pk_sys_log PRIMARY KEY (cnt_seq)
    );

create index idx_app_log_category_date on app_log (category,date);
create index idx_sys_log_category_date on sys_log (category,date);

create table errlevel(
    code        int not null,
    log         varchar not null,
    CONSTRAINT pk_errlevel PRIMARY KEY (code)
    );

grant select on errlevel to apache;

create table category (
    code        int not null,
    Category    varchar not null,
    CONSTRAINT pk_category PRIMARY KEY (code)
    );

CREATE OR REPLACE FUNCTION logg(varchar) RETURNS integer AS 'select code from errlevel where log=$1;' LANGUAGE SQL;

INSERT INTO errlevel (code, log)
select 1, 'DEBUG'
 union all
select 2, 'INFO'
 union all
select 3, 'WARN'
 union all
select 4, 'ERROR'
 union all
select 5, 'FATAL';

INSERT INTO category (code, Category )
select 1, 'APACHE.AUDIT.DEV_MNG'
 union all
select 2, 'APACHE.AUDIT.USR_MNG'
 union all
select 3, '.SYSTEM';


-- ================== DynaData =================================

create sequence dyna_data_seq start 100;

create table dyna_types(
    typeid      varchar PRIMARY KEY,
    scope       varchar,
    requireid   bool,
    uniq      bool,
    restartcmd  varchar,
    backup  bool
    );

GRANT select,update,delete,insert ON dyna_types to apache;

create table dyna_data(
    ind         bigint default nextval('dyna_data_seq'),
    description varchar,
    typeid      varchar,
    contentid   varchar,
    data        varchar,
    FOREIGN KEY (typeid) REFERENCES dyna_types (typeid)
    );

GRANT select,update,delete,insert ON dyna_data to apache;


INSERT INTO dyna_types (typeid,scope,requireid,uniq,restartcmd,backup) VALUES ('Emailer', 'system', 'f', 't', '/opt/sarch/base/bin/emailer_sighup', 't');
INSERT INTO dyna_types (typeid,scope,requireid,uniq,restartcmd,backup) values('gui_matrix_settings', 'User', true, true, '', true);
INSERT INTO dyna_types (typeid,scope,requireid,uniq,restartcmd,backup) values('mx_gui_conf',         'User', true, false,'', true);
INSERT INTO dyna_types (typeid,scope,requireid,uniq,restartcmd,backup) values('mx_gui_conf_share',   'User', false, false,'', true);
INSERT INTO dyna_types (typeid,scope,requireid,uniq,restartcmd,backup) values('gui_matrix_layout',   'User', true, false,'', true);

INSERT INTO dyna_data (ind,description, typeid, data)
VALUES (1,'Emailer configuration', 'Emailer',
'<RESULT><POLICY LEVEL=\"INFO\" CODE=\"2\"><TIME VALUE=\"60\"/><EMAILS VALUE=\"\"/></POLICY>
<POLICY LEVEL=\"WARN\" CODE=\"3\"><TIME VALUE=\"60\"/><EMAILS VALUE=\"\"/></POLICY>
<POLICY LEVEL=\"ERROR\" CODE=\"4\"><TIME VALUE=\"60\"/><EMAILS VALUE=\"\"/></POLICY>
<POLICY LEVEL=\"FATAL\" CODE=\"5\"><TIME VALUE=\"60\"/><EMAILS VALUE=\"\"/></POLICY>
<CONFIG SERVER=\"\" USER=\"\" PASSWORD=\"\" STATUS=\"NO\"/></RESULT>');

INSERT INTO dyna_types (typeid,scope,requireid,uniq,restartcmd,backup)
VALUES ('maps_data', 'system', 'f', 't', '', 't');

INSERT INTO dyna_data (ind,description, typeid, contentid, data)
VALUES (2,'MapsDataBlock', 'maps_data', '0', '<node id="0" Src="map2d.jpg" Name="New Map"></node>');

-- this table will be depreciated in 3.3 but have to be created since exits in 2.7

create table processes(
  procname      varchar(30)     not null,
  startup       numeric         not null,
  check_time    numeric,
  description   varchar(100)    not null,
  start_script  varchar(400)    not null,
  stop_script   varchar(300),
  check_script  varchar(250),   -- DO NOT USE!! ignored in process manager
  check_mask    varchar(150),   -- mask to find process in "ps ax"
  CONSTRAINT pk_processes PRIMARY KEY (procname)
)
;

