#!/usr/bin/perl

# -----------------------------------------------------------------------------
#  $Id: db_create.pl 33301 2015-11-13 17:11:38Z atsybulnik $
#  It's assumed that postgres process is running
#  - create aplication user (as defined in db conf file) / if not exists
#  - create user $APL_HTTPD_USR / if not exists 
#  - create db / if not exists
#  - update db configuration
#
#  Note: script runs once by root on installation stage
# -----------------------------------------------------------------------------
#  Author: Andrey Baranetsky
#  Edited by: Andrey Fomenko
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2004-2010
# -----------------------------------------------------------------------------

use strict;
use warnings;
use Digest::MD5 qw(md5_hex md5_base64);


my $dbAdminUser = $ENV{APL_DB_USR};
my $dbHttpUser = $ENV{APL_HTTPD_USR};
my $pgsql_conf = "$ENV{APL_DB_DATA}/postgresql.conf";
my $pgsql_hba = "$ENV{APL_DB_DATA}/pg_hba.conf";
my $dbProperties;


sub getDBConf {
    my $dbConfName = shift;
	my $dbConfFile = "$ENV{APL}/etc/${dbConfName}.db.conf";
	
	open(DB_CONF_FILE, "<$dbConfFile")
    	or die "unable to open configuration file " || $dbConfFile;

	my %mcfg = map{/(^.+?)=(.*)/} grep {/^.+=.*/} <DB_CONF_FILE>;
	my $dbProperties = \%mcfg;
	close(DB_CONF_FILE);
	
	return $dbProperties;
}


sub createUser {
	my ($username, $canCreateDB, $isSuper, $canCreateRole) = @_;
	
	# check if user already exists 
#	my $dbUserExists = `su $dbAdminUser -lc \"psql -At  -c \\\"SELECT 1 FROM pg_catalog.pg_user WHERE usename='$username';\\\"\"`;
        my $cmd=qq{echo "SELECT usecreatedb,usesuper FROM pg_catalog.pg_user WHERE usename='$username'" | su $dbAdminUser -lc 'psql -At template1'};
        my $dbUserExists = `$cmd`; 
	if ($? != 0) {
		die "Failed to execute =$cmd=, returns $?\n";
	} elsif (chomp($dbUserExists) eq '1') {
		printf "User \"$username\" exists\n";
		# Alter user if necessary
		my ($createdb,$super) = split(/\|/, $dbUserExists);
		if ($createdb eq 't' or $super eq 't') {
		    $cmd=qq{
			echo "ALTER USER '$username' WITH NOCREATEDB NOSUPERUSER NOCREATEROLE" 
			| su $dbAdminUser -lc 'psql -At template1'
		    };
		    `$cmd`;
		    ($? == 0)  || die "Failed to alter user \"$username\"";
		}
	} else {
		# create user
		my $createDbOpt = ($canCreateDB) ? "d" : "D";
		my $isSuperOpt = ($isSuper) ? "s" : "S";
		my $createRoleOpt = ($canCreateRole) ? "r" : "R";
		`su $dbAdminUser -lc \"createuser -a${createDbOpt}${isSuperOpt}${createRoleOpt} $username\"`;
		($? == 0)  || die "Failed to create user \"$username\"";
		printf "User \"$username\" created\n";
	}
}


sub createDB
{
	#
	# Check if DB already exists
	#
	my $availableDbList = `su $dbAdminUser -lc \"psql -Atl\"`;
	if ($? != 0) {
	 	die "Failed to execute 'psql'";
	} elsif ($availableDbList =~ m/^$dbProperties->{'name'}\|\w+\|UTF8/i) {
		printf "Database \"$dbProperties->{'name'}\" exists\n";
	} else {
		my $dbLocale = ($ENV{TARGET_OS} eq 'Darwin') ? 'en_US.UTF-8' : 'en_US.utf8';

		if ($availableDbList =~ m/^$dbProperties->{'name'}\|\w+\|LATIN1/i) {
			#
			# convert DB to UTF8
			#
			printf "Convert \"$dbProperties->{'name'}\" FROM LATIN1 to UTF8\n";
			`su $dbAdminUser -lc \"pg_dump $dbProperties->{'name'} > $ENV{APL}/var/db/$dbProperties->{'name'}.sql\"`;
			`su $dbAdminUser -lc \"dropdb $dbProperties->{'name'}\"`;
			`su $dbAdminUser -lc \"createdb --encoding='UTF8' --template=template0 --lc-ctype=$dbLocale  --lc-collate=$dbLocale $dbProperties->{'name'}\"`;
			`su $dbAdminUser -lc \"psql $dbProperties->{'name'} < $ENV{APL}/var/db/$dbProperties->{'name'}.sql\"`;
			`su $dbAdminUser -lc \"rm -f $ENV{APL}/var/db/$dbProperties->{'name'}.sql\"`;
		} else {
			#
			# Create DB
			#
			`su $dbAdminUser -lc \"createdb --encoding='UTF8' --template=template0 --lc-ctype=$dbLocale  --lc-collate=$dbLocale $dbProperties->{'name'}\"`;
			if ($? == 0) {
				printf "Database \"$dbProperties->{'name'}\" created\n";
			} else {
				die "Failed to create database \"$dbProperties->{'name'}\"";
			}
		}	
	}
	
	#
	# Add plpgsql
	#
	my $plpgsqlInstalled = `su $dbProperties->{'user'} -lc \"psql -At  -c \\\"SELECT 1 FROM pg_catalog.pg_language WHERE lanname = 'plpgsql';\\\"\"`;
	if ($? != 0) {
		die "Failed to execute 'psql'";
	} elsif (chomp($plpgsqlInstalled) eq '1') {
		printf "plpgsql language is already installed\n";		
	} else {
		`su $dbAdminUser -lc \"createlang --dbname=$dbProperties->{'name'} plpgsql\"`;
		($? == 0)  || die "Failed to intall plpgsql language on \"$dbProperties->{'name'}\"";
		printf "plpgsql language is already installed\n";		
	}
	
	#
	# Give the default "postgres" user a random password
	#
	my $key = md5_base64(sprintf("%d=$$=%s",time,rand));
	my $cmd=qq {
		echo "ALTER USER postgres WITH ENCRYPTED PASSWORD '$key'" | su $dbAdminUser -lc 'psql -At template1'
	};
	`$cmd`;
	($? == 0)  || die "Failed to assign password to \"postgres\"";
	my $key_enc = md5_hex($key);
	`mkdir -p $ENV{APL}/var/db`;
	if (open FH, ">$ENV{APL_VAR}/db/skm_admin.db.conf") {
		print FH "type=PGSQL\nname=apl\nhost=/tmp\nuser=postgres\npassword=$key_enc\n";
		close(FH) || die "close() failed: $!";
		chmod(0600, "$ENV{APL_VAR}/db/skm_admin.db.conf") || die "chmod() failed: $!";
	}
	else {
		die "Failed to open admin key file for write: $!";
	}
}


sub updateDBConfiguration
{	
 #	
 # Update postgresql.conf
 #
 open(CONF,$pgsql_conf) ||die "Cant open: $pgsql_conf";
 my @conf=<CONF>;
 close CONF;
	
 foreach (@conf) {
#	$_= "tcpip_socket = true\n"	if /^#tcpip_socket/;		#depreciated
#	$_= "port = 5432\n"		if /^#port\s=\s5432/;
	$_= "max_connections = 320\n"	if /^#*\s*max_connections/;
	$_= "shared_buffers = 256MB\n"	if /^#*shared_buffers/; 	#initially 32MB
#	$_= "sort_mem = 2048\n"		if /^#*sort_mem/;		#depreciated
#	$_= "wal_buffers = 32kb\n"	if /^#*wal_buffers/;
#	$_= "commit_siblings = 5\n"	if /^#*commit_siblings/;	#initially 5
	$_= "checkpoint_segments = 4\n"	if /^#*checkpoint_segments/;
	$_= "listen_addresses = '*'\n"	if /^#*listen_addresses/;
	$_= "timezone = UTC\n"		if /^#*timezone/;		
	$_= "synchronous_commit = off\n"if /^#*\s*synchronous_commit/;  #improve insert performance
	$_= "maintenance_work_mem = 32MB\n" if /^#*\s*maintenance_work_mem/;
	$_= "password_encryption = on\n" if /^#password_encryption/;
 }
 
 open CONF, ">$pgsql_conf"	|| die "cannot open $pgsql_conf";
 print  CONF @conf;
 close CONF;

 printf "$pgsql_conf updated\n";
	
 #
 # Create initial pg_hba.conf
 #
 open(APLHBA,"$ENV{APL}/db/etc/pg_hba.init") || die "cannot open $ENV{APL}/db/etc/pg_hba.init";
 my @hba=<APLHBA>;
 close(APLHBA);
 open DBHBA, ">$pgsql_hba"; 
 print DBHBA @hba;
 close DBHBA;
 printf "$pgsql_hba updated\n";
}



$#ARGV >= 0  || die "database name is required";

# get DB configuration
$dbProperties = getDBConf($ARGV[0]);
my $dbType = $dbProperties->{'type'};
$dbType eq 'PGSQL' || die "Invalid/missing database 'type' ($dbType)";


createUser($dbProperties->{'user'}, 0, 0, 0);
createUser($dbHttpUser, 0, 0, 0);

createDB();
updateDBConfiguration();
