#!/usr/bin/perl

#
# Delete all objects owned by "dbConfigFile->user" (triggers, functions, views, types, tables, sequences) 
#

use strict;
use warnings;
use DBI;


my @dbObjTypes = ( 
	{	type => "trigger",
		lookupStmt => "SELECT t.tgname, c.relname FROM pg_catalog.pg_class c
 INNER JOIN pg_catalog.pg_trigger t ON c.oid = t.tgrelid
  LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
 WHERE pg_get_userbyid(c.relowner) = '#OWNER#' AND NOT t.tgisconstraint  AND n.nspname='public' #EXCLUDE#;",
 		excludeClause => "AND t.tgname NOT IN (#LIST#)",
		dropStmt => "DROP TRIGGER #OBJNAME# ON #TABLENAME# CASCADE;",
	},
	{	type => "function",
		lookupStmt => "SELECT proname || '(' || COALESCE(array_to_string(
 ARRAY(SELECT t.typname FROM pg_type t, generate_series(0,array_upper(p.proargtypes, 1)) as o WHERE t.oid=p.proargtypes[o]),','),'') || ')'
  FROM pg_catalog.pg_proc p 
 LEFT JOIN pg_namespace n ON n.oid=p.pronamespace
 WHERE NOT p.proisagg AND pg_get_userbyid(p.proowner)='#OWNER#' AND n.nspname='public' #EXCLUDE#;",
		excludeClause => "AND p.procname NOT IN (#LIST#)",
		dropStmt => "DROP FUNCTION #OBJNAME# CASCADE;",		
	},	
	{	type => "aggregate",
		lookupStmt => "SELECT proname || '(' || COALESCE(array_to_string(
 ARRAY(SELECT t.typname FROM pg_type t, generate_series(0,array_upper(p.proargtypes, 1)) as o WHERE t.oid=p.proargtypes[o]),','),'') || ')'
  FROM pg_catalog.pg_proc p
 LEFT JOIN pg_namespace n ON n.oid=p.pronamespace
 WHERE p.proisagg AND pg_get_userbyid(p.proowner)='#OWNER#' AND n.nspname='public' #EXCLUDE#;",
		excludeClause => "AND p.procname NOT IN (#LIST#)",
		dropStmt => "DROP AGGREGATE #OBJNAME# CASCADE;",		
	},	
	{	type => "view",
		lookupStmt => "SELECT c.relname FROM pg_class c
 LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
 WHERE c.relkind='v' AND pg_get_userbyid(c.relowner)='#OWNER#' AND n.nspname='public' #EXCLUDE#;",
		excludeClause => "AND c.relname NOT IN (#LIST#)",
		dropStmt => "DROP VIEW name #OBJNAME# CASCADE;",	
	},
	{	type => "type",
		lookupStmt => "SELECT c.relname FROM pg_class c
  LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
 WHERE c.relkind='c' AND pg_get_userbyid(c.relowner)='#OWNER#' AND n.nspname='public' #EXCLUDE#;",
		excludeClause => "AND c.relname NOT IN (#LIST#)",
		dropStmt => "DROP TYPE #OBJNAME# CASCADE;",	
	},
	{	type => "table",
		lookupStmt => "SELECT tablename FROM pg_catalog.pg_tables WHERE tableowner='#OWNER#' AND schemaname='public' #EXCLUDE#;",
		excludeClause => "AND tablename NOT IN (#LIST#)",
		dropStmt => "DROP TABLE #OBJNAME# CASCADE;",
		checkStmt => "SELECT 1 FROM pg_catalog.pg_tables WHERE upper(tablename)=upper('#OBJNAME#') AND tableowner='#OWNER#' AND schemaname='public';",
	},
	{	type => "sequence",
		lookupStmt => "SELECT c.relname FROM pg_class c
  LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
 WHERE c.relkind='S' AND pg_get_userbyid(c.relowner)='#OWNER#' AND n.nspname='public' #EXCLUDE#;",
		excludeClause => "AND c.relname NOT IN (#LIST#)",
		dropStmt => "DROP SEQUENCE #OBJNAME# CASCADE;",
		checkStmt => "SELECT 1 FROM pg_class c WHERE c.relkind='S' AND upper(c.relname)=upper('#OBJNAME#') AND pg_get_userbyid(c.relowner)='#OWNER#';",
	},
);

my $dbProperties;
my $dbh;
my $owner;



sub readPropertyFile {
	my ($fileName, $dieOnError) = @_;
	my $properties = {};
	
	if (open(PROP_FILE, "<$fileName")) {
		my %mcfg = map{/(^.+?)=(.*)/} grep {/^.+=.*/} <PROP_FILE>;
		$properties = \%mcfg;
		close(PROP_FILE);
	} elsif ($dieOnError) {
		die "unable to open configuration file " . $fileName;
	}

	return $properties;
}


sub getDBConf {
	my $dbConfName = shift;
	my $dbProperties = readPropertyFile("$ENV{APL}/etc/${dbConfName}.db.conf", 1);
	$dbProperties->{'dbi.connstr'} = "dbi:".$dbProperties->{'dbi.driver'}.":dbname=$dbProperties->{name};host=$dbProperties->{host}";
	return $dbProperties;
}


#
# Deletes non-transactional objects from DB
#
sub removeDBObjects {
	#
	# Get the list of all abjects to be dropped
	#
	my @dropStmts = ();
	foreach my $dbObjType (@dbObjTypes) {
		my $lookupStmt = $dbObjType->{'lookupStmt'};
		$lookupStmt =~ s/#OWNER#/$owner/g; 

		my $excludeClause = "";
		$lookupStmt =~ s/#EXCLUDE#/$excludeClause/g;
		$lookupStmt =~ s/\n//g;
		
		# execute lookup statement and assemble drop statements
		my $rows = $dbh->selectall_arrayref($lookupStmt);
		if ($rows) {
			my $dropStmt = $dbObjType->{'dropStmt'};
			
			foreach my $row (@$rows) {
				my $stmt = $dropStmt;
				$stmt =~ s/#OBJNAME#/$row->[0]/g;
				if ($dbObjType->{'type'} eq "trigger") { $stmt =~ s/#TABLENAME#/$row->[1]/g; }
				push @dropStmts, $stmt; 		
			}
		} elsif ($dbh->err != 0) {
			my $msg = $dbh->errstr;
			die "Failed to get list of $dbObjType->{'type'}s: $msg Statement=$lookupStmt\n"
		}
	}

	#
	# drop object from the list one by one
	#	
	foreach my $dropStmt (@dropStmts) {
		my $rv = $dbh->do($dropStmt);
		if (!defined $rv) {
  			die "FAILED to execute \"$dropStmt\": " . $dbh->errstr . "\n";
		} else {
			printf "$dropStmt - OK\n\n";
		}
	}	
}


#
# "main"
#
$#ARGV >= 0  || die "database name is required";
$dbProperties = getDBConf($ARGV[0]);
my $dbType = $dbProperties->{'type'};
$dbType eq 'PGSQL' || die "Invalid/missing database 'type' ($dbType)";

$owner = $dbProperties->{'user'};

# connect to db
$dbh = DBI->connect(
			$dbProperties->{'dbi.connstr'},
			$dbProperties->{'user'},
			$dbProperties->{'password'},
			{RaiseError => 0, PrintError=>0});
($dbh) or die "FAILED to connect to DB:" . $DBI::errstr . "\n"; 
 

removeDBObjects();


$dbh->disconnect;
