#!/usr/bin/perl
#  $Id: bck_events.pl 22408 2011-04-04 16:38:57Z teetov $
# -----------------------------------------------------------------------------
#  Backup/restove Event Log.
#  Supposed to be used for cron backups and manual restore if DB gets broken.
# -----------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: Alexey Tsibulnik
#  QA by:
#  Copyright: videoNEXT Netowk solutions Inc, 2006
# -----------------------------------------------------------------------------
use strict;
use warnings;
use SKM::DB;

# -----------------------------------------------------------------------------
# VARS

my $APL = $ENV{APL};
my $BACKUP_FILES_MAX = 3;              # number of backup files retained
my $EVENTS_ROOT = "$APL/var/backup/events";     # directory where backupos will be stored
my $EVENTS_DIR;	# backup directory name
my $DBEXEC = "$APL/db/bin/db_exec";
my @BCK_TABLES = ("event", "eventwitness", "eventproperty"); # tables to dump/restore
my $err = 0; # Error counter
my $dbm;
my %DB = %{ DBMasterConf() };

# MAIN

my $mode = shift @ARGV;
my $events_dir;
die Usage() unless $mode;

if(uc($mode) eq 'BACKUP') {
    $EVENTS_ROOT = shift @ARGV if @ARGV;
    doBackup();
}
elsif(uc($mode) eq 'RESTORE') {
    $events_dir = shift @ARGV;
    die Usage() unless $events_dir;
    ($EVENTS_ROOT, $events_dir) = ($1, $2) if $events_dir=~m|^(.+)/(.+)$|;
    doRestore();
}
else {
    die Usage();
}
Log("FINAL: ".($err?"ERROR":"SUCCESS"));
exit $err;

# -----------------------------------------------------------------------------
# ROUTINES

sub Log {
    my $msg = shift;
    $err++ if $msg=~/^ERROR/;
    print STDOUT "$msg\n";
}

sub Usage {
    return <<END;
USAGE: bck_events.pl backup [events_root]
       bck_events.pl restore events_dir
       
END
}

sub db_master
{
    for (my $i = 1; $i < 3; $i++) {
	eval {
    	    $dbm=DBMaster({'PrintError'=>0,'RaiseError'=>1});
	};
	if($@) {
    	    Log ("ERROR: Attempt $i (final). Cannot connect to master: $@"),last if $i>=2;
    	    Log("Attempt $i. Cannot connect to master: $@");
	    Log('Sleep '. $i*30 . ' before next attempt');
    	    sleep($i*30);
	} else {
	    last;
	}
    }
}

sub doBackup {
    # Generate dirname
    my ($tsec,$tmin,$thour,$tday,$tmon,$tyear)=(gmtime(time))[0..5];
    my $bckdir = sprintf("%02s%02s%02s_%02s%02s%02s",$tyear-100,$tmon+1,$tday,$thour,$tmin,$tsec);
    $events_dir = "$EVENTS_ROOT/$bckdir";
    Log("Backup identification sequence: [$bckdir]");
    Log("Backup area: $events_dir");
    # Dump tables
    unless (mkdir "$events_dir") {
	Log("ERROR: Cannot create $events_dir");
    }
    else {
	my $cnt = 0;
	db_master;
	eval {
	    my $r = $dbm->selectrow_arrayref("SELECT COUNT(1) from event");
	    $cnt = $r->[0] if $r;
	};
	if($@) {
	    Log("ERROR: Failed to count events in DB");
	}
	else {
	    Log("Total events: $cnt");
	    if(open(STAT, ">$events_dir/stat")) {
		print STAT "total_events=$cnt\n";
	        close STAT;
	    }
	}
	eval { $dbm->disconnect } if $dbm;
	
	foreach my $table (@BCK_TABLES) {
	    `pg_dump -h '$DB{host}' -U '$DB{user}' -F c -X disable-triggers -aOxt $table '$DB{name}'> $events_dir/${table}.tbck 2>/dev/null`;
	    Log("ERROR: Failed to back up $table"),next if $?;
	    Log("Backed up table: $table");
	}
    }
}

sub doRestore {
    my $bckdir = "$EVENTS_ROOT/$events_dir";
    Log("Event recovery started for: $bckdir");
    unless (-d $bckdir) {
	Log("ERROR: Bad event directory");
    }
    else {
	foreach my $table (@BCK_TABLES) {
	    Log("ERROR: file for table '$table' is missing"),next unless -f "$bckdir/${table}.tbck";
	    `$DBEXEC -c "DELETE FROM $table" skm_master 2>/dev/null`;
	    `pg_restore -d '$DB{name}' -h '$DB{host}' -U '$DB{user}' -F c -X disable-triggers -aOcxt $table $bckdir/${table}.tbck 2>/dev/null`;
	    Log("ERROR: Failed to restore $table"),next if $?;
	    Log("Restored DB table: $table");
	}
	# Connect to DB
	db_master;
	if($dbm) {
	    # Alter sequence
	    eval {
		my $arr = $dbm->selectall_arrayref("SELECT MAX(eventid) FROM event");
		$dbm->do("ALTER SEQUENCE seqeventid RESTART WITH $arr->[0][0]")
		    if defined $arr->[0][0];
	    };
	    Log($@?"ERROR: Failed to reset event sequence: $@":"Reset sequence: SUCCESS");
	    # Filter resulting tables
	    eval {
	        $dbm->do("DELETE FROM event WHERE objid NOT IN (SELECT obj FROM _objs) 
		        OR (workflow, state) NOT IN 
		        (SELECT workflow, eventstate FROM eventstate)"
		);
	        $dbm->do("DELETE FROM eventwitness 
		        WHERE eventid IN (SELECT eventid 
		        FROM eventwitness EXCEPT SELECT eventid FROM event)"
		);
		$dbm->do("DELETE FROM eventproperty 
		        WHERE eventid IN (SELECT eventid 
		        FROM eventproperty EXCEPT SELECT eventid FROM event)"
		);
	    };
	    Log($@?"Failed to filter events: $@":"Filter events: SUCCESS");
	    # Analyze tables
	    eval {
		$dbm->do("ANALYZE event");
		$dbm->do("ANALYZE eventproperty");
		$dbm->do("ANALYZE eventwitness");
	    };
	    Log($@?"Failed to analyze tables: $@":"Analyze tables: SUCCESS");
	    eval { $dbm->disconnect } if $dbm;
	}
	else {
	    Log("ERROR: Cannot connect to DB: $DBI::errstr");
	}
    }
}
