package Device::Conf;

use 5.008008;
use strict;
use warnings;
use HTTP::Request::Common;
use XML::Simple;
use Data::Dumper;
#use Carp;
use Cwd 'abs_path';

use SKM::Agent;

require Exporter;

our @ISA = qw(Exporter);

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration	use Device::Conf ':all';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.
our %EXPORT_TAGS = ( 'all' => [ qw(
 BrandList Cartridges DefaultCartridgeCfg ProbeInit  ProbeResult ProbeSamplePicture 
 ProbeErr ProbeRequest ProbeSession ProbeHttpHeader ProbeSnapshotList UA EscapeList
 ProbeWarn
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
 BrandList Cartridges DefaultCartridgeCfg ProbeInit  ProbeResult ProbeSamplePicture 
 ProbeErr ProbeRequest ProbeSession ProbeHttpHeader ProbeSnapshotList NormMediaFormat
 ProbeWarn
);

our $VERSION = '0.01';

# constants --------------
 my %DEFAULTS=(PROBE=>'NORMAL',HTTP_PORT=>80,DEVID=>0);
 my $CARTRIGE_HOME="$ENV{APL}/conf/etc/device";
 my $SAMPLE_PATH="$ENV{APL}/var/probe/image";

# vars      --------------
 my $conf;                       # point to the loaded conf
 my $args;                       # point to provided args
 my $ua;                         # LWP::UserAgent object

# Preloaded methods go here.
 
 sub BrandList{
    my $device=shift; # (expect camera | sensor | relay)
    my $dir="$CARTRIGE_HOME/$device";
    my @list;    
    if(opendir(DIR, $dir)){
      @list = grep { not /^(\.+|default)$/ and -d "$CARTRIGE_HOME/$device/$_"} readdir(DIR);
      closedir DIR;
    }
    return @list;
 }
 
 sub Cartridges {
   my $device=shift; # (expect camera | sensor | relay)
   my @list=BrandList($device);
   my @cartriges;
   foreach my $name (@list) {
      my %unit=(TYPE=>$device,NAME=>$name,
                PROBE=>"$CARTRIGE_HOME/$device/$name/bin/probe",
                DESCR=>"$CARTRIGE_HOME/$device/$name/etc/descr.xml",
                FIND=>"$CARTRIGE_HOME/$device/$name/bin/find");
      $unit{PROBE}='NONE' if ! -f $unit{PROBE};
      $unit{DESCR}='NONE' if ! -f $unit{DESCR};
      $unit{FIND} ='NONE' if ! -f $unit{FIND};
      if (open CFG,"$CARTRIGE_HOME/$device/$name/etc/cartridge.cfg") {
         my %cfg = map {/^(\w+)=(.+)$/} grep {/^\w+=.+/} <CFG>;
         close CFG;
         %unit = (%cfg,%unit);
      }
      push @cartriges,\%unit;
   }
   #--------------------------------------------------------
   #   we can sort catriges here if we have a priority list
   #--------------------------------------------------------
   @cartriges = sort {
      my $pa=$a->{PROBEORDER}; my $pb=$b->{PROBEORDER};
      $pa=undef if defined $pa and $pa!~/^\d+(?:\.\d+)?$/; 
      $pb=undef if defined $pb and $pb!~/^\d+(?:\.\d+)?$/;
      defined $pa ? (defined $pb ? $pa<=>$pb:-1) : (defined $pb ? 1:0)
   } @cartriges;
   return \@cartriges;
 }
 
 sub DefaultCartridgeCfg {
   my $device=shift; # (expect camera | sensor | relay)
   my %cfg;
   if (open CFG, "$CARTRIGE_HOME/$device/default/etc/cartridge.cfg") {
      %cfg = map {/^(\w+)=(.+)$/} grep {/^\w+=.+/} <CFG>;
      close CFG;
   }
   return \%cfg;
 }
 
 sub NormMediaFormat {
     my $list = lc($_[0]); # media format list
     $list =~ s/mjpeg/mjpg/;
     return $list;
 }

 sub ProbeInit {
   my $to = defined($_[0]) ? $_[0] : 3;
   my $file=abs_path((caller(0))[1]);
   my ($device,$brand,$dir)=(split(/\//,$file))[-4,-3,-2];
   die "$file should have a correct location ( device/<BRAND>/bin/probe) \n" if not $device=~/^(camera|audio|sensor|relay)$/;  
   die "$file should have a correct location ( device/<BRAND>/bin/probe) \n" if $dir ne 'bin';
   my %argv=map{/(^\w+)=(.*)/} grep {/^\w+=.*/} @ARGV; # script args
   $args={%DEFAULTS,%argv};              # static pointer. Merge defaults & args
   my %cfg=%$args;                       # put all ags to conf 
   delete $cfg{DEVID} if $cfg{DEVID} ne '0'; # but DEVID exclude 
   $cfg{_probe_brand}=$brand;            # special hash elements 
   $cfg{_probe_device}=$device;          #
   $conf=\%cfg;                          # static pointer
   initUA(timeout=>$to);				 # Init UserAgent
   my $cfg_name="$ENV{APL}/var/conf/$args->{DEVID}/conf";
   return $conf  if ! -f $cfg_name;
   return $conf  if -z $cfg_name;
   return $conf  if ! open(CONF,$cfg_name);
   %cfg=map{/(^\w+)=(.*)/} grep {/^\w+=.*/} <CONF>;
   close CONF;
   %cfg=(%DEFAULTS,%cfg,%argv,(_probe_brand=>$brand,_probe_device=>$device)); # overwrite conf with arguments
   return $conf;  # it is a "static" pointer to %cfg
 }
 
 sub ProbeResult {  # print result to conf.probe and stdout
   my $result=shift;
   my $dev=$conf->{DEVID};
   if($dev) {
     if(open CONF,">$ENV{APL}/var/conf/$dev/conf.probe") { 
       print(CONF "$_=$result->{$_}\n") foreach (sort keys %$result);
       close CONF;
     }
   }
   print("$_=$result->{$_}\n") foreach (sort keys %$result);
   exit 1 if $result->{STATUS}=~/^ERROR/;
   exit 0;
 }


 sub ProbeErr {     # use result to print the error
   my ($err,$text,$comments) =@_;  
   my $dev=$args->{DEVID};
   if(defined $comments and $comments) {
     $comments= "\n#". join "\n#",split(/\n/,$comments);
   }else{
     $comments='';
   }
   ProbeResult({STATUS=>"ERROR: $err [$dev] $text $comments"})
 }

 sub ProbeWarn {     # use result to print the warning
   my ($warn,$text,$comments) =@_;
   my $dev=$args->{DEVID};
   if(defined $comments and $comments) {
     $comments= "\n#". join "\n#",split(/\n/,$comments);
   }else{
     $comments='';
   }
   ProbeResult({STATUS=>"WARNING: $warn [$dev] $text $comments"})
 }


 sub ProbeSamplePicture {
   my $params=shift;
   my $descr_name="$CARTRIGE_HOME/camera/$conf->{_probe_brand}/etc/descr.xml";
   return "ERROR: $descr_name is missing" if ! -f $descr_name;
   my $snapshot='';
   eval {
     my $xml = new XML::Simple;
     my $data = $xml->XMLin($descr_name);
     #print Dumper($data);
     $snapshot=$data->{default}->{snapshot} if defined $data->{default}->{snapshot};
   };
   return "ERROR: broken XML ($descr_name)" if($@);
   return "ERROR: SNAPSHOT is not defined"  if not $snapshot;
   my ($usrname,$passwd,$addr,$port)=@$conf{qw(USRNAME PASSWD DEVIP HTTP_PORT)};
   $port=80 if ! $port;                           # use port 80 if HTTP_PORT is empty
   my $url="http://$addr:$port".$snapshot;
   $url=~s|{\$$_}|$params->{$_}| foreach(keys %$params);
   my $camera = defined($params->{CAMERA}) ? $params->{CAMERA} : $conf->{CAMERA} || '1';
   my $picture='ERROR';
   my $req = GET $url;
   initUA(usrname=>$usrname, passwd=>$passwd);
   my $rsp = $ua->request($req);
   if ($rsp->code eq "200") {         # successfully get respond
      mkdir $SAMPLE_PATH if not -d $SAMPLE_PATH;
      $picture="$SAMPLE_PATH/$$-$camera-".time.".jpg";
      if(! $rsp->content) {
        $picture="ERROR: empty picture";
      }elsif(open PIC, ">$picture") {
         print PIC $rsp->content;
         close PIC;
      }else {
         $picture="ERROR: cannot write $picture";
      }
   }else {
         $picture='ERROR: http respond '.$rsp->code;
   }
   return $picture;
 }
 
 sub ProbeSnapshotList {
   my @camlist=@_;
   if (not @camlist) {
     if (defined $conf->{CAMERA_LIST}) {
       @camlist=split(/,/,$conf->{CAMERA_LIST});
     } elsif (defined $conf->{CAMERA}) {
       @camlist = ($conf->{CAMERA});
     } else { @camlist=('1'); }
   }
   my @snaplist=();
   foreach my $camera (@camlist) {
     push(@snaplist,$camera,ProbeSamplePicture({CAMERA=>$camera}));
   }
   return EscapeList(@snaplist);
 }

#--------------------------------------------------------------------
# 
#--------------------------------------------------------------------
 sub probeGet {
   my $part=shift;                             # header | text
   my ($path,$method,$hdrs,$content) = @_;
   my ($usrname,$passwd,$addr,$port)=@$conf{qw(USRNAME PASSWD DEVIP HTTP_PORT)};
   $port=80 if ! $port;                        # use port 80 if HTTP_PORT is empty
   $method='GET' if ! $method;
   $hdrs=[] if ref($hdrs) ne 'ARRAY';	       # hdrs should be a plain array reference of key/value pairs
   my $url="http://$addr:$port".$path;
   #print "$url\n";
   $url =~ tr/ //d;
   my $req;
   for ($method) {
     /^POST$/ and do { $req=POST $url,@$hdrs,Content=>$content; last };
     /^PUT$/  and do { $req=PUT  $url,@$hdrs,Content=>$content; last };
     /^GET$/  and do { $req=GET  $url,@$hdrs; last };
   }
   initUA(usrname=>$usrname, passwd=>$passwd);
   my $rsp=$ua->request($req);
   return "HTTP ERROR [".$rsp->code."]\n$url" if ! $rsp->is_success;
   return $rsp->header('server') if ($part eq 'header');
   return $rsp->header('SessionID') if ($part eq 'session'); 
   return $rsp->content;
 }

 sub ProbeRequest {
   return probeGet('text',@_);
 }

 sub ProbeHttpHeader {
   return probeGet('header',@_);
 }
 
 sub ProbeSession {
   return probeGet('session',@_);
 }


 sub initUA {
     my %args = @_;
     my $to = defined $args{timeout} ? $args{timeout} : 3;
     if (! $ua) {
	 $ua=new SKM::Agent;
	 $ua->timeout($to);
     }
     if (exists $args{usrname} and exists $args{passwd}) {
         $ua->set_creds($args{usrname}, $args{passwd});
     }
     $ua;
 }
 
 sub UA {
   my ($new,$old) = (shift,$ua || initUA);
   $ua=$new if $new and ref($new) and $new->isa('LWP::UserAgent');
   return $old;
 }
 
 sub EscapeList {
   my %arg=@_;
   my $str="";
   foreach my $key (keys %arg) {
     s/:/\\:/g for ($key,$arg{$key}); # Escape commas
     s/,/\\,/g for ($key,$arg{$key}); # Escape colon
     $str .= "$key:$arg{$key},";
   }
   chop $str;
   return $str;
 }
 
1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

Device::Conf - Perl extension for device probbing

=head1 SYNOPSIS

  use Device::Conf;

=head1 DESCRIPTION

 Device::Conf is intended to offer easy probe operations

=head2 EXPORT

None by default.

=head1 SEE ALSO

Mention other useful documentation such as the documentation of
related modules or operating system documentation (such as man pages
in UNIX), or any relevant external documentation such as RFCs or
standards.

If you have a mailing list set up for your module, mention it here.

If you have a web site set up for your module, mention it here.

=head1 AUTHOR

A.Teetov, E<lt>teetov@videonext.com<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2008 by videoNEXT

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.8 or,
at your option, any later version of Perl 5 you may have available.


=cut
