/**
 * @version $Id: validate.js 33561 2016-02-01 07:03:22Z sabaev $
 * ------------------------------------------------------------------------------
 * Validation functions
 * Used for validate camera attributes. Function names defined in the template.
 * ------------------------------------------------------------------------------
 * @author Serg Tsybulsky
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2012
 * ------------------------------------------------------------------------------
 */

(function(window){
	"use strict";

window.Validate = window.Validate || {};
Validate.DEFAULT = Validate.DEFAULT || {};

/**
 * return through callback function camera model information
 getCameraModel({
 	ip: "192.168.0.1",
 	port: "80",
 	user: "root",
 	password: "pass",
 	model: "Axis",
 	ticket: "192.168.0.123_80_1269269272",
 	callback: fn
 });
 * @param {number} obj
 * @param {object} parameters
 * @returns {boolean}
 */
Validate.getCameraModel = function(parameters)
{
	if (!arguments[0])
	{
		return false;
	}

	var nodeid      = parameters.nodeid || "",
		ticket      = parameters.ticket || "",
		callback    = parameters.callback || null,
		stop_exists = parameters.stop_exists,
		attributes  = parameters.attributes || {};

	if (stop_exists && stop_exists.stop) {
		stop_exists.stop = false;
		return false;
	}

	if (typeof callback !== "function")
	{
		return false;
	}

	var api = new API();
	if (ticket === "")
	{
		var getTicketParameters = {
			attributes: JSON.stringify(attributes),
			nodeid: nodeid
		};

		if (parameters.obj)
		{
			getTicketParameters.obj = parameters.obj;
		}
		api.getTicketForCameraModel(getTicketParameters)
			.fail(function(code, message){
				Log.error(message);
				callback({error: message});
			})
			.done(function(response){
				Validate.getCameraModel({
					ticket: response.ticket,
					callback: callback,
					stop_exists: stop_exists
				});
			});
	} else {
		api.getCameraModelByTicket({
			ticket: ticket
		})
		.fail(function(code, message){
			Log.error(message);
			callback({error: message});
		})
		.done(function(response){
			if (stop_exists && stop_exists.stop)
			{
				stop_exists.stop = false;
				return;
			}
			var checkAgain = true;
			if (response){
				if (response.error === "")
				{
					//data = {"error": error, "cameramodel" : cameramodel, "modelid" : modelid, "firmware" : firmware, "picture" : picture, "exists": exists};
					var data = {"error": ""};
					if (response.parameters)
					{
						if (response.parameters.state === "OK")
						{
							checkAgain = false;
							data.attributes = $.extend({}, response.attributes);
							data.parameters = $.extend({}, response.parameters);
							data.picture = response.parameters.SNAPSHOT;
							data.exists = response.exists;
							data.existList = response.existList;
						} else
						if (response.parameters.state === "ERROR")
						{
							checkAgain = false;
							if (response.parameters.error === "PDE-0401")
							{
								data.error = __("Wrong user name or password.");
							} else
							if (
								response.parameters.error === "PDE-0001"
								|| response.parameters.error === "PDE-0003")
							{
								data.error = __("Cannot connect to specified address.");
							} else
							if (response.parameters.error === "PDE-0002")
							{
								data.error = __("Unknown camera type.");
							} else
							if (response.parameters.error === "PDE-0404")
							{
								data.error = __("Cannot retrieve camera configuration.");
							}
						}
						// call callback function only if state equal "OK" or "ERROR"
						if (stop_exists && stop_exists.stop) {
							stop_exists.stop = false;
							return;
						}
						if (!checkAgain)
						{
							callback(data);
						}
					}
				}
			}

			if (checkAgain)
			{
				// check for camera model until we do not have it
				setTimeout(
					function()
					{
						Validate.getCameraModel({
							ticket: ticket,
							callback: callback,
							stop_exists:stop_exists
						});
					},
					1000
				);
			}
		});
	}

	return true;
};

Validate.checkCacheWarning = function(params){
	var AVATARID = params[0];
	var EVENT_POLICY = params[1];
	var STORAGE_POLICY = params[2];
	
	if(AVATARID == "" && (STORAGE_POLICY == -5 || STORAGE_POLICY == -6)){
		var api = new API();
		$.when(
				api.getSMWriterConf(),
				api.getEventSourceParametersList({"id": EVENT_POLICY})
			)
			.done(function(getSMConfResponse, getPolicyResponse){
				var warn = false;
				var MAX_CACHE_SIZE = getSMConfResponse[0].list['MAX_CACHE_SIZE'];
				
				for (var i = 0; i < getPolicyResponse[0].eventsourceparameters.length; i++){
					if(getPolicyResponse[0].eventsourceparameters[i].parameters['preevent'] > MAX_CACHE_SIZE){
						warn = true;
						break;
					}
				}

				if(warn){
					Log.warning(__("Event policy you selected has pre-event time exceeding ") + MAX_CACHE_SIZE + __(" seconds, which is your system maximum pre-event time. It may result in less video stored that policy requests"), 10*1000);
				}
			});
	}
};

Validate.DEFAULT.EVENT_POLICY = function()
{
	Validate.checkCacheWarning([
		$(Validate.getObj("AVATARID", this)).val(),
		$(Validate.getObj("EVENT_POLICY", this)).val(),
		$(Validate.getObj("STORAGE_POLICY", this)).val()
	]);
};

Validate.DEFAULT.STORAGE_POLICY = function()
{
	$(Validate.getObj("VAE_ACTIVE", this)).trigger("change");
};

Validate.DEFAULT.PROBE = function()
{
	var obj = $(this).data("obj");

	var UDID = $(Validate.getObj("UDID", this)).val();
	var ip = $(Validate.getObj("DEVIP", this)).val();
	var port = $(Validate.getObj("HTTP_PORT", this)).val();
	var user = $(Validate.getObj("USRNAME", this)).val();
	var pass = $(Validate.getObj("PASSWD", this)).val();
	var name = $(Validate.getObj("NAME", this)).val();
	var location = $(Validate.getObj("LOCATION", this)).val();
	var associate = $(Validate.getObj("ASSOCIATE", this)).val();
	var audio_devid = $(Validate.getObj("AUDIO_DEVID", this)).val();
	var modelId = $(Validate.getObj("MODELID", this)).val();
	var cameraModel = $(Validate.getObj("CAMERAMODEL", this)).val();
	var imageSizeList = $(Validate.getObj("IMAGESIZE_LIST", this)).val();
	var mediaFormatList = $(Validate.getObj("MEDIA_FORMAT_LIST", this)).val();

	var NODEID = $(Validate.getObj("NODEID", this)).val();
	var AVATARID = $(Validate.getObj("AVATARID", this)).val();
	var searchid = NODEID;
	if (AVATARID != "")
	{
		searchid = AVATARID;
	}

	$(Validate.getObj("PROBE", this)).mask();

	var thisObject = this;
	Validate.getCameraModel({
		attributes: {
			DEVIP: ip,
			HTTP_PORT: port,
			USRNAME: user,
			PASSWD: pass
		},
		nodeid: searchid,
		"callback": function(model)
		{
			if (model["error"] !== "")
			{
				alert(model["error"]);
			} else {
				if (
					model["attributes"]['MODELID'].toLowerCase() !== modelId.toLowerCase()
					|| model["attributes"]['CAMERAMODEL'].toLowerCase() !== cameraModel.toLowerCase()
					)
				{
					if (confirm(__(
							"CAMERA MANUFACTURER/MODEL UPDATE:\n" +
							"Your camera has incorrect settings.\n" +
							"The real camera model is " + model["attributes"]['CAMERAMODEL'] + " " + model["attributes"]['MODELID'] + ", while in current configuration is for " + cameraModel + " " + modelId + ".\n" +
							"All the camera settings will have to be reset to default values.\n" +
							"Do you want to proceed and save updated configuration?"
						)))
					{
						var changeModelAttributes = $.extend({}, model["attributes"]);

						changeModelAttributes["UDID"] = UDID;
						changeModelAttributes["OBJID"] = obj;
						changeModelAttributes["AVATARID"] = AVATARID;
						changeModelAttributes["NODEID"] = NODEID;
						changeModelAttributes["DEVIP"] = ip;
						changeModelAttributes["HTTP_PORT"] = port;
						changeModelAttributes["USRNAME"] = user;
						changeModelAttributes["PASSWD"] = pass;
						changeModelAttributes["NAME"] = name;
						changeModelAttributes["LOCATION"] = location;
						changeModelAttributes["ASSOCIATE"] = associate;
						changeModelAttributes["AUDIO_DEVID"] = audio_devid;

						// save all VAE attributes
						$("#tab_content_VAE").find("select[data-attr][data-obj=" + obj + "], input[data-attr][data-obj=" + obj + "]").each(function(){
							var attribute = $(this).data("attr");
							var value = $(this).val();

							changeModelAttributes[attribute] = value;
						});

						var api = new API();
						api.setAttributes({
							obj: obj,
							attributes: JSON.stringify(changeModelAttributes)
						})
							.fail(function(code, message){
								Log.error(message);
							});

						Log.warning("Press 'Reload' button after new model is confirmed");
					}
				}

				var isChanged = false;

				if (
					typeof imageSizeList !== "undefined"
					&&
					model["attributes"]['IMAGESIZE_LIST']
					&&
					model["attributes"]['IMAGESIZE_LIST'].toLowerCase() !== imageSizeList.toLowerCase()
				)
				{
					isChanged = true;

					$(Validate.getObj("IMAGESIZE_LIST", thisObject)).val(model["attributes"]['IMAGESIZE_LIST']);
					$(Validate.getObj("IMAGESIZE", thisObject)).trigger("change");
					alert(__("Image size list of you camera was changed, please save changes."));
				}
				if (
					typeof mediaFormatList !== "undefined"
					&&
					model["attributes"]['MEDIA_FORMAT_LIST']
					&&
					model["attributes"]['MEDIA_FORMAT_LIST'].toLowerCase() !== mediaFormatList.toLowerCase()
				)
				{
					isChanged = true;

					$(Validate.getObj("MEDIA_FORMAT_LIST", thisObject)).val(model["attributes"]['MEDIA_FORMAT_LIST']);
					$(Validate.getObj("MEDIA_FORMAT", thisObject)).trigger("change");
					alert(__("Media format list of you camera was changed, please save changes."));
				}

				if (!isChanged)
				{
					$(Validate.getObj("PROBE", thisObject))
						.prop("disabled", true)
						.val(__("Confirmed"));
				}
			}
			$(Validate.getObj("PROBE", thisObject)).unmask();
		}
	});
};

Validate.DEFAULT.RC_MODE = function()
{
	var value = $(Validate.getObj("RC_MODE", this)).val();
	if (value === 'cbr'){
		Validate.showAttribute("RC_TARGETBITRATE", this);
		Validate.showAttribute("RC_MAXBITRATE", this);
	} else
	if (value === 'vbr'){
		Validate.hideAttribute("RC_TARGETBITRATE", this);
		Validate.showAttribute("RC_MAXBITRATE", this);
	}
};

Validate.DEFAULT.MEDIAFORMAT = function()
{
	var value = $(Validate.getObj("MEDIA_FORMAT", this)).val();
	if (value === 'mjpg'){
		Validate.hideAttribute("RC_MODE", this);
		Validate.hideAttribute("RC_TARGETBITRATE", this);
		Validate.hideAttribute("RC_MAXBITRATE", this);

		Validate.showAttribute("FRAMERATE", this);
		Validate.showAttribute("ARCFRAMERATE", this);
		Validate.showAttribute("CAMCOMPRESSION", this);

		$(Validate.getObj("PROTO", this)).val('HTTP');

		Validate.showAttribute("HTTP_PORT", this);
		Validate.hideAttribute("RTSP_PORT", this);

		Validate.hideAttribute("RTP_OVER_TCP", this);
		Validate.hideAttribute("RTP_TIME", this);
		Validate.hideAttribute("RTP_FORCE_MULTICAST", this);
	} else
	if (value === 'mpeg4' || value === 'h264' || value === 'mpeg2'){
		Validate.showAttribute("RC_MODE", this);
		$(Validate.getObj("RC_MODE", this)).trigger('change');

		Validate.hideAttribute("FRAMERATE", this);
		Validate.hideAttribute("ARCFRAMERATE", this);
		Validate.hideAttribute("CAMCOMPRESSION", this);

		$(Validate.getObj("PROTO", this)).val('RTSP');

		Validate.hideAttribute("HTTP_PORT", this);
		Validate.showAttribute("RTSP_PORT", this);

		Validate.showAttribute("RTP_OVER_TCP", this);
		Validate.showAttribute("RTP_TIME", this);
		Validate.showAttribute("RTP_FORCE_MULTICAST", this);

	}
};

Validate.DEFAULT.PROTO = function()
{
	var value = $(Validate.getObj("PROTO", this)).val();
	if (value === 'RTSP'){
		Validate.hideAttribute("HTTP_PORT", this);
		Validate.showAttribute("RTSP_PORT", this);
		Validate.showAttribute("RTP_OVER_TCP", this);
		Validate.showAttribute("RTP_TIME", this);
		Validate.showAttribute("RTP_FORCE_MULTICAST", this);
	} else
	if (value === 'HTTP'){
		Validate.showAttribute("HTTP_PORT", this);
		Validate.hideAttribute("RTSP_PORT", this);
		Validate.hideAttribute("RTP_OVER_TCP", this);
		Validate.hideAttribute("RTP_TIME", this);
		Validate.hideAttribute("RTP_FORCE_MULTICAST", this);
	}
};

Validate.DEFAULT.POSITIONCTL = function ()
{
	Validate.hideAttribute([
		"PTZGROUP",
		"PTZ_TCP_IP",
		"PTZ_CAMERATYPE_ICD001"
	],
	this);

	Validate.showAttribute([
		"PTZID",
		"PTZ_CAMERATYPE",
		"PTZ_TCP_PORT",
		"PTZ_TRANSPORT",
		"PTZSPEED",
		"PTZ_PRESET1TIMEOUT",
		"PTZ",
		"PTZ_HFOV_MIN",
		"PTZ_HFOV_MAX"
	],
	this);

	var value = $(Validate.getObj("POSITIONCTL", this)).val();
	switch (value)
	{
		case 'none':
			// hide all ptz tab attributes
			Validate.hideAttribute([
				"PTZID",
				"PTZ_CAMERATYPE",
				"PTZ_TCP_PORT",
				"PTZ_TRANSPORT",
				"PTZGROUP",
				"PTZSPEED",
				"PTZ_PRESET1TIMEOUT",
				"PTZ",
				"PTZZOOMSPEED",
				"PTZ_HFOV",
				"PTZ_HFOV_MIN",
				"PTZ_HFOV_MAX"
			],
			this);
			break;
		case 'Axis':
			//$(Validate.getObj("PTZ_CAMERATYPE", this)).val('Axis');
			break;
		case 'ptcr20bae':
			$(Validate.getObj("PTZ_CAMERATYPE", this)).val('ptcr20bae');
			break;
		case 'ptcr20flir':
			$(Validate.getObj("PTZ_CAMERATYPE", this)).val('ptcr20flir');
			break;
		case 'flir_35x_140':
			$(Validate.getObj("PTZ_CAMERATYPE", this)).val('flir');
			break;
		case 'icd001':
			Validate.showAttribute([
				"PTZGROUP",
				"PTZ_CAMERATYPE_ICD001"
			],
			this);

			Validate.hideAttribute("PTZ_CAMERATYPE", this);

			$(Validate.getObj("PTZ_CAMERATYPE_ICD001", this)).trigger("change");
			break;
		case 'Pelco':
			Validate.showAttribute("PTZ_TCP_IP", this);
			break;
		case 'UDP':
			Validate.showAttribute("PTZZOOMSPEED", this);
			break;
	}
};

Validate.DEFAULT.PTZ = function()
{
	var url = $(this).data("url");

	window.open(url, "PTZwindow", "width=434,height=210,toolbar=no,menubar=no,location=no,scrollbars=no,resizable=yes");
};

Validate.DEFAULT.PTZ_HFOV = function()
{
	var value = $(Validate.getObj("PTZ_HFOV", this)).val();

	var attributeList = [
		"PTZ_HFOV_MIN",
		"PTZ_HFOV_MAX",
		"PTZ_ZOOM_OPTICAL_MAX",
		"PTZ_ZOOM_DIGITAL_MAX",
		"PTZ_FOCAL_LENGTH_MIN",
		"PTZ_FOCAL_LENGTH_MAX"
	];

	if (value == 1)
	{
		Validate.showAttribute(attributeList, this);
	} else {
		Validate.hideAttribute(attributeList, this);
	}
};

Validate.DEFAULT.PTZ_ASSOCIATED_CAMERAS = function()
{
	var value = $(Validate.getObj("PTZ_ASSOCIATED_CAMERAS", this)).val();
	var list = $.parseJSON(value);

	var attributeList = [
		"PTZ_CALIBRATOR_BUTTON"
	];

	if (list.length > 0)
	{
		Validate.showAttribute(attributeList, this);
	} else {
		Validate.hideAttribute(attributeList, this);
	}
};

Validate.DEFAULT.AUDIOHASTWOWAY = function(){
	var value = $(Validate.getObj("AUDIO_HAS_TWO_WAY", this)).val();
	if (value === 'off'){
		Validate.showAttribute("AUDIO_TWO_WAY", this);
		$(Validate.getObj("AUDIO", this)).val('off');
		$(Validate.getObj("AUDIO_TWO_WAY", this)).val('off');
	} else
	if (value === 'on'){
		$(Validate.getObj("AUDIO", this)).val('on');
		$(Validate.getObj("AUDIO_TWO_WAY", this)).val('off');
	} else
	if (value === 'on_2way'){
		$(Validate.getObj("AUDIO", this)).val('on');
		$(Validate.getObj("AUDIO_TWO_WAY", this)).val('on');
	}
};

Validate.DEFAULT.AUDIO_AUDIO_TWO_WAY = function(){
	var value = $(Validate.getObj("AUDIO", this)).val();
	if (value === 'on'){
		Validate.showAttribute("AUDIO_TWO_WAY", this);
	} else
	if (value === 'off'){
		Validate.hideAttribute("AUDIO_TWO_WAY", this);
	}
};

Validate.DEFAULT.SET_LIST = function()
{
	var element = $(Validate.getObj("SET_LIST", this));
	var obj = element.data("obj");

	// check if curreent element is loaded
	var isLoaded = element.data("isLoaded");
	if (isLoaded)
	{
		var list = element.data("list");

		var save = function(params, element)
		{
			if (!params.obj)
				return;

			params.addList = params.addList || [];
			params.removeList = params.removeList || [];

			// save changes maded by user
			// add or remove from set
			element.mask();

			var api = new API();
			api.changeObjectList({
				type: "set",
				parentobj: params.obj,
				addList: JSON.stringify(params.addList),
				removeList: JSON.stringify(params.removeList)
			})
				.fail(function(code, message){
					Log.error(message);
				})
				.done(function(json){
					Log.info(__("Save complete"));
				})
				.always(function(){
					element.unmask();
				});
		};

		// create list
		element
			.find("input[type='checkbox']")
			.each(function(){
				var setObj = $(this).val();

				if (!this.checked && list[setObj])
				{
					delete list[setObj];
					save(
						{
							obj: setObj,
							removeList: [obj]
						},
						$(this).parent()
					);
				} else
				if (this.checked && !list[setObj])
				{
					list[setObj] = obj;
					save(
						{
							obj: setObj,
							addList: [obj]
						},
						$(this).parent()
					);
				}
			});
	} else {
		var isLoading = element.data("isLoading");
		if (isLoading)
			return;

		element.data("isLoading", true);

		// get sets in which camera is presented

		var api = new API();
		api.getObjectList({
			type: "set",
			isAll: true,
			childObj: obj
		})
			.fail(function(code, message){
				Log.error(message);
			})
			.done(function(response){
				response.list = response.list || [];

				var list = {};
				for (var i = 0; i < response.list.length; i++)
				{
					var listItem = response.list[i];

					if (listItem["protected"] == "0")
						list[listItem.obj] = listItem.name;
				}

				element.mask();
				element
					.find("input[type='checkbox']")
					.each(function(){
						if (list[$(this).val()])
							this.checked = true;
					});
				element.unmask();

				element.data("isLoading", false);
				element.data("isLoaded", true);
				element.data("list", list);
			});
	}
};

Validate.getObjectNameList = function(params, callback)
{
	var id = params.id || null;
	var objList = params.objList || null;
	var type = params.type || null;
	var data = {};
	if (objList){
		data.objList = JSON.stringify(objList);
	}
	if (type){
		data.type = type;
	}
	if (!id)
		return;

	id.mask();

	var iAdmin = require("iadmin");

	var api = new API();
	api.getObjectList(data)
		.fail(function(code, message){
			Log.error(message);
		})
		.done(function(response){
			var objectList = response.list;
			var objParamList = [];
			if (objectList)
			{
				for (var num in objectList)
				{
					if (!objectList.hasOwnProperty(num))
						continue;

					var obj = objectList[num]["obj"];
					objParamList[obj] = [];
					objParamList[obj]['name'] = objectList[num]["name"];
					objParamList[obj]['udid'] = objectList[num]["udid"];

					var otype = objectList[num]["otype"];
					var subtype = objectList[num]["subtype"];
					objParamList[obj]['type'] = iAdmin.objectTypes[otype][subtype]["name"];
					objParamList[obj]['template'] = iAdmin.objectTypes[otype][subtype]["template"];
					objParamList[obj]['location'] = objectList[num]["location"];
					objParamList[obj]['fullname'] = objParamList[obj]['type'] + " [" + objParamList[obj]["udid"] + "] "  + objParamList[obj]["name"];
					if (objParamList[obj]['location']) objParamList[obj]['fullname'] += " / " + objParamList[obj]['location'];
				}
			}
			callback(objParamList);
		})
		.always(function(){
			id.unmask();
		});
};

Validate.DEFAULT.AE_LIST = function()
{
	var value = $(Validate.getObj("AE_LIST", this)).val();
	if (value === ''){
		Validate.showAttribute("AE_ACTIVATED", this);
		Validate.hideAttribute("AE_LIST", this);
	} else{
		Validate.hideAttribute("AE_ACTIVATED", this);
		Validate.showAttribute("AE_LIST", this);

		/*Correctly show attribute
		it must be links to necessary devices
		AE_LIST has list of OBJ_ID with delimer ","
		Parse list, create link?
		Create correct Label Name
		 */

		if (value !== ""){
			var element = $(Validate.getObj("AE_LIST", this));
			var options = value.split(',');
			Validate.getObjectNameList({id: element, objList: options},
				function(objParamList){
					element.empty();
					var content = [];
					var hash = window.location.hash;
					for (var j = 0; j < options.length; j++)
					{
						var objID = options[j].replace(/"/g, '&quot;');
						if (objParamList[objID]){
							var objName = "[" + objParamList[objID]['udid'] + "] " + objParamList[objID]["name"];
							content.push('<div><a href="' + hash + '/' + objParamList[objID]["template"] + '_' + objID + '">' + objName +'</a></div>');
						}
					}
					element.html(content.join(''));
				}
			);
		}
	}
};

Validate.DEFAULT.AE_ACTIVATED = function()
{
	var obj = $(this).data("obj");
	$(Validate.getObj("AE_LIST", this)).trigger("change");

	var ip = $(Validate.getObj("DEVIP", this)).val();
	var port = $(Validate.getObj("HTTP_PORT", this)).val();
	var user = $(Validate.getObj("USRNAME", this)).val();
	var pass = $(Validate.getObj("PASSWD", this)).val();
	var name = $(Validate.getObj("NAME", this)).val();
	var location = $(Validate.getObj("LOCATION", this)).val();
	var cameraModel = $(Validate.getObj("CAMERAMODEL", this)).val();
	var timezone = $(Validate.getObj("TIME_ZONE", this)).val();

	var value = $(Validate.getObj("AE_LIST", this)).val();
	if (value === ''){
		$(Validate.getObj("AE_ACTIVATED", this)).mask();

		var thisObject = this;
		var attributes = {};
		attributes.DEVIP = ip;
		attributes.HTTP_PORT = port;
		attributes.USRNAME = user;
		attributes.PASSWD = pass;
		attributes.CAMERAMODEL = cameraModel;
		attributes.NAME = name;
		attributes.LOCATION = location;
		attributes.TIME_ZONE = timezone;

		var api = new API();
		api.createCameraAssociatedDevices({
			cameraid: obj,
			attributes: JSON.stringify(attributes)
		})
			.fail(function(code, message){
				Log.error(message);
			})
			.done(function(response){
				Validate.hideAttribute("AE_ACTIVATED", this);
				Log.info(__("Automatic associated objects for you camera was created, please save changes or reload editor to view created objects."));
				window.location.reload();
			})
			.always(function(){
				$(Validate.getObj("AE_ACTIVATED", thisObject)).unmask();
			});
	} else {
		Validate.hideAttribute("AE_ACTIVATED", this);
		Validate.showAttribute("AE_LIST", this);
	}
};

Validate.DEFAULT.SET_MANAGER = function()
{
	var obj = $(this).data("obj");

	if (obj)
	{
		window.location.href = $(this).data("url");
	}
};

Validate.DEFAULT.ENCODER_SETTING_OVERRIDE = function()
{
	Validate.hideAttribute("STAT_IMAGESIZE", this);
	Validate.hideAttribute("STAT_BITRATE", this);
	Validate.hideAttribute("STAT_MEDIA_FORMAT", this);

	if ($(Validate.getObj("ENCODER_SETTING_OVERRIDE", this)).val() == "no")
	{
		Validate.showAttribute("STAT_IMAGESIZE", this);
		Validate.showAttribute("STAT_BITRATE", this);
		Validate.showAttribute("STAT_MEDIA_FORMAT", this);
	}
};

Validate.DEFAULT.AUDIT_INFO = function()
{
	if (Validate.isMassObjectEditor)
	{
		return;
	}

	var api = new API();
	api.getIdentityAttributes()
		.fail(function(code, message){
			Log.error("[" + code + "] " + message);
		})
		.done(function(response){
			var list = response.list;

			var dateFormat;
			for(var name in list) if (list.hasOwnProperty(name)) {
				if(name == "NLS_LONG_DATE_FORMAT"){
					dateFormat = list[name];
					break;
				}
			}

			auditObj.create($("#tab_content_Audit"), dateFormat, 10, true)
				.done(function(){
					auditObj.objid = $("#tabs_content").data("obj");
					auditObj.from = null;
					auditObj.to = null;
					auditObj.showUserActivity(true, false);
					auditObj.callGetAuditRecords();
				});
		});
};

Validate.DEFAULT.AV_DELIVERY = function ()
{
	var LIVE_LR = (1 << 0);
	var LIVE_NR = (1 << 1);
	var ARCHIVE_LR = (1 << 2);
	var ARCHIVE_NR = (1 << 3);
	var ARCHIVE_EVENTS_LR = (1 << 4);
	var ARCHIVE_EVENTS_NR = (1 << 5);
	var METADATA_EVENTS = (1 << 6);

	if ($(Validate.getObj("AVATARID", this)).val() == "")
	{
		Validate.hideAttribute("AV_DELIVERY_HR", this);
		Validate.hideAttribute("AV_STORAGE_POLICY", this);
	} else {
		// Convert to digit
		var d = eval($(Validate.getObj("AV_DELIVERY_HR", this)).val());
		$(Validate.getObj("AV_DELIVERY", this)).val(d);

		// Disable 'AV_STORAGE_POLICY'
		$(Validate.getObj("AV_STORAGE_POLICY", this)).val('1');
		//$(Validate.getObj("AV_STORAGE_POLICY", this)).prop("disabled", true);
	}
};

Validate.DEFAULT.SCHEDULE_LINK = function()
{
	var value = $(Validate.getObj("SCHEDULEID", this)).val();
	if (value === ''){
		Validate.hideAttribute("SCHEDULE_LINK", this);
	} else{
		Validate.showAttribute("SCHEDULE_LINK", this);

		/*Correctly show attribute
		it must be links to necessary devices
		AE_LIST has list of OBJ_ID with delimer ","
		Parse list, create link?
		Create correct Label Name
		 */

		if (value !== ""){
			var element = $(Validate.getObj("SCHEDULE_LINK", this));
			var options = value.split(',');
			Validate.getObjectNameList({id: element, objList: options, type: 'schedule'},
				function(objParamList){
					element.empty();
					var content = [];
					var hash = window.location.hash;
					for (var j = 0; j < options.length; j++)
					{
						var objID = options[j].replace(/"/g, '&quot;');
						if (objParamList[objID]){
							var objName = "[" + objParamList[objID]['udid'] + "] " + objParamList[objID]["name"];
							content.push('<div><a href="' + hash + '/' + 'schedule_manager_' + objID + '">' + objName +'</a></div>');
						}
					}
					element.html(content.join(''));
				}
			);
		}
	}
};
Validate.DEFAULT.SM_GRAPH = function()
{
	var camera_id = $(this).data("obj");
	var url = $(this).data("url");
	var error_msg = $(this).data("error");

	var api = new API();
	api.getAttributes({
		obj: camera_id
	})
		.fail(function(code, message){
			Log.error("[" + code + "] " + message);
		})
		.done(function(result){
			$("#playerVideoImage").css('visibility', 'hidden');

			var attributes = result.list;

			window.scrollTo(0,0);
			$('body').css({
				'position': 'relative',
				'height': '100%'
			});
			if ($(window).width() >= 1225) {
				$('body').css('overflow', 'hidden');
			}

			$('body').append(
				'<div id="overflower" style="width: 100%; height: 100%; position: absolute; top: 0; left: 0; opacity: 0.5; filter: alpha(opacity=50); background-color: white;"></div>' +
				'<iframe id="sm_graph_content" frameBorder="0" allowTransparency="true" src="/sdi/sm/index.php?objid=' + camera_id + '"></iframe>'
			);

			var left = ($(window).width() - 1250) / 2;

			if (left < 20) {
				left = -10;
			}

			$('#sm_graph_content').css({
				'background-color': 'transparent',
				left: left,
				top: '50px',
				width: '1250px',
				height: $(window).height() - 50,
				position: 'absolute',
				'z-index': 10000,
				align: 'center',
				border: 'none'
			});
		});
};

Validate.DEFAULT.SM_GRAPH_CLOSE = function() {
	$("#playerVideoImage").css('visibility', 'visible');
	$('body').css({'position': 'static', 'overflow': 'visible'});
	$('#overflower, #sm_graph_content').remove();
};

Validate.DEFAULT.RTP_FORCE_MULTICAST = function()
{
	var value = $(Validate.getObj("RTP_FORCE_MULTICAST", this)).val();
	if (value === 'on'){
		$(Validate.getObj("RTP_OVER_TCP", this)).val('off');
	}
};

Validate.DEFAULT.RTP_OVER_TCP = function()
{
	var value = $(Validate.getObj("RTP_OVER_TCP", this)).val();
	if (value === 'on'){
		$(Validate.getObj("RTP_FORCE_MULTICAST", this)).val('off');
	}
};

Validate.VAE = Validate.VAE || {};
Validate.VAE.VAE_ACTIVE = function()
{
	var obj = $(this).data("obj");
	var VAE_ACTIVE = $(this).val();
	// #6873 Implement logic:
	// If camera has PTZ and PTZ_PRESET1TIMEOUT is not set, do not allow to activate VAE
	var ptz = $(Validate.getObj("POSITIONCTL", this)).val();
	var preset1_tmt = $(Validate.getObj("PTZ_PRESET1TIMEOUT", this)).val();
	if (VAE_ACTIVE == "yes" && ptz && ptz != 'none' && (preset1_tmt == '' || preset1_tmt <= 0)) {
		Log.warning(__("Cannot enable VAE on PTZ camera until 'Timeout Before Go To Preset 1' is set!"));
		$(this).val("no");
		$(Validate.getObj("VAE_ACTIVE", this)).trigger("change");
	}
	else {
		if (VAE_ACTIVE == "yes")
		{
			Validate.showAttribute([
				"VAE_VISUALIZATION"
			], this);

			$("#tab_content_VAE").find("select[data-attr!=VAE_ACTIVE][data-obj=" + obj + "], input[data-attr][data-obj=" + obj + "]").each(function(){
				var attribute = $(this).data("attr");
				if (attribute.match(/^VAE_(.+)_ACTIVE$/))
				{
					Validate.showAttribute(attribute, this);
					$(this).trigger("change");
				}
			});
		} else
		if (VAE_ACTIVE == "no")
		{
			$("#tab_content_VAE").find("select[data-attr!=VAE_ACTIVE][data-obj=" + obj + "], input[data-attr][data-obj=" + obj + "]").each(function(){
				var attribute = $(this).data("attr");
				Validate.hideAttribute(attribute, this);
			});

			$(Validate.getObj("VAE_MOTION_ACTIVE", this)).trigger("change");
		}
	}
};

/**
 * @returns {boolean}
 */
Validate.DEFAULT.METADATA_PORT = function()
{
	// to use JS code as validator for text field validate javascript must return res.data("error") on errors
	var value = $(Validate.getObj("METADATA_PORT", this)).val();
	var minvalue = 1023;
	var maxvalue = 65535;

	var regexp = new RegExp(/^\d*$/);
	if (!regexp.test(value) || (value != '' && (value < minvalue || value > maxvalue)))
	{
		var error_str = Gettext.strargs(__("Metadata port must be integer from %1 to %2"), [minvalue, maxvalue]);
		$(this).data("error", error_str);
		return false;
	}
	$(this).removeData("error");
	return true;
};

Validate.DEFAULT.TIME_ZONE = function() {
	var value = $(Validate.getObj("SCHEDULEID", this)).val();
	if (value === ''){
		$(Validate.getObj("TIME_ZONE", this)).prop("disabled", false);
		//Validate.hideAttribute("TIME_ZONE", this);
	} else{
		$(Validate.getObj("TIME_ZONE", this)).prop("disabled", true);
		//Validate.showAttribute("TIME_ZONE", this);
	}
};

Validate.DEFAULT.NODEID = function(event, parameters) {
	if (typeof parameters === "undefined" || !parameters.firstRun) {
		Log.warning(__("NODE CHANGE: After node change complete camera arhive will be lost!"));
	}
	if ($(Validate.getObj("AVATARID", this)).val()) {
		$(Validate.getObj("NODEID", this)).prop("disabled", true);
	} else {
		$(Validate.getObj("NODEID", this)).prop("disabled", false);
	}
};

/*
 * #6873 Implement logic:
 * If camera has PTZ and VAE enabled, setting PTZ_PRESET1TIMEOUT to 'Never' will switch VAE_ACTIVE to 'no'
 */
Validate.DEFAULT.PRESET1TIMEOUT = function() {
	var tmt = $(this).val();
	var ptz = $(Validate.getObj("POSITIONCTL", this)).val();
	var vae = $(Validate.getObj("VAE_ACTIVE", this)).val();
	if ( vae == "yes" && ptz && ptz != "none" && (!tmt || tmt <= 0) ) {
		Log.warning(__("Analytics was turned off for camera! Timeout to go to preset 1 should be set if you want analytics to be enabled"));
		$(Validate.getObj("VAE_ACTIVE", this)).val("no");
		$(Validate.getObj("VAE_ACTIVE", this)).trigger("change");
	}
};

})(window);
