#!/usr/bin/perl

# -----------------------------------------------------------------------------
#  Synchronize directory structure inside $APL_CONF/$dev/vae with 
#  currently installed VAE cartriges
# -----------------------------------------------------------------------------
#  Author: Alex Tsybulnik
#  QA by:
#  Copyright: videoNEXT LLC
# -----------------------------------------------------------------------------

use strict;
use warnings;

use NextCAM::Conf "CameraList";

# CONS
my $APL = $ENV{APL};
my $CONF = $ENV{APL_CONF};
my $VAE_LIST = "$APL/conf/bin/cartridge-ctl type=vae query=list";
my $VAR = "$ENV{APL_VAR}/vae";

# VARS
my $log = undef;
my $obj = $ARGV[0];
my $err = 0;

# SUB
sub Log {
    if ($log) {
	$log->info(@_);
    } else {
	print @_, "\n";
    }
}

sub Err {
    $err++;
    if ($log) {
	$log->error(@_);
    } else {
	warn @_, "\n";
    }
}

# MAIN
if (defined $obj) {
    die "Argument should be a valid camera ObjId\n" unless $obj =~ /^\d+$/;
}
else { # Log to a file only if script is called by rpm trigger
    require Log::Log4perl;
    Log::Log4perl->import("get_logger");
    require "$ENV{APL}/common/bin/logger.patrol";
    $log = get_logger('NEXTCAM::CONF::CONF_SYNC_VAE_DIRS');
}
my %vae_list = map {/^(\w+)=(.*)$/} grep {/^\w+=/} `$VAE_LIST`;
Err("Error fetching vae list") if $?;
if (%vae_list and not -d $VAR) {
	mkdir $VAR;
}
# Create license dir for OV6 if not exist
foreach my $vae (keys %vae_list) {
	if ($vae eq 'ov6') {
		mkdir "$VAR/vae-ov6" unless -d "$VAR/vae-ov6";
	}
}
my @cams = ($obj) || CameraList;
Log("Sync vae dirs. Engines installed: ".join(',',keys %vae_list));
Log("Obj=$obj") if $obj;
foreach my $cam (@cams) {
    unless (-d "$CONF/$cam") {
	Err("Camera dir missing for $cam");
	next;
    }
    my $vae_dir = "$CONF/$cam/vae";
    # Create dirs if not exist
    foreach my $vae (keys %vae_list) {
	next if $vae eq 'sample'; # Skip 'sample' engine
	unless (-d "$vae_dir/$vae/conf") {
	    Log("Creating '$vae' vae structure for $cam");
	    my $out = `/bin/mkdir -p -m 0750 $vae_dir/$vae/{conf,log} 2>&1`;
	    Err("Error creating '$vae' vae dir for $cam: $out") if $?;
	}
    }
    # Remove dirs for uninstalled VAE
    if (opendir DH, $vae_dir) {
	my @vae = grep { not /^\./ } readdir DH;
	closedir DH;
	foreach my $vae (@vae) {
	    next if exists $vae_list{$vae};
	    # Completely remove engine files
	    Log("Removing '$vae' vae structure for $cam");
	    my $out = `rm -rf $vae_dir/$vae 2>&1`;
	    Err("Error removing '$vae' vae dir for $cam: $out") if $?;
	}
    }
    else {
	Err("Cannot open dir $vae_dir: $!") if -f $vae_dir;
    }
}

END {
    $? = $err;
}
