#!/usr/bin/perl -w
#  $Id:$
# -----------------------------------------------------------------------------
#  Assemble CAMERA LOG based on multiple system log files
#
#  The primary purpose is to show a short fragment of log relevant to 
#  particular camera in Admin GUI (to facilitate troubleshooting)
# -----------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Modified by:
#  QA by:
#  Copyright: 2009, videoNEXT Network solutions, Inc.
# -----------------------------------------------------------------------------

use strict;
use warnings;
use Data::Dumper;
use Net::STOMP::Client;

use constant LOG_ENTRIES  => 50;    # lines from each log to extract
use constant LOG_SCAN_MAX => 3;     # maximum depth in log history to scan

my $APL=$ENV{APL};
my $APL_VAR = $ENV{APL_VAR};
my $APL_CONF = "$APL_VAR/conf";
my $STOMP_PORT = 61613;
my $TIMEOUT = 3;
my $Stomp;
my %log;


sub GetMretrLog { # ---------------------- read entries from system_mretr.log ----
	my $objid = shift;
	
	my @tail;
	my $re_rec = qr{^\w{3}\s+\d+\s+\d{2}:\d{2}:\d{2}};
	my $re_match = qr{^\w{3}\s+\d+\s+\d{2}:\d{2}:\d{2}.*\[$objid\]};
	
	for (my $i = 0; $i < LOG_SCAN_MAX; $i++) {
	
	    my $name = "$APL/var/log/system_mretr.log".($i==0?'':".$i");
	    next unless -f $name;
	    
	    open (FH, $name);
	    my $found = 0;
	    while (<FH>) {
	        if (/$re_rec/) {
	            if (/$re_match/) {
			$found = 1;
	    		push (@tail, $_);
		    }
		    else {
		        $found = 0;
		    }
		}
	        else {
	    	    $tail[$#tail] .= $_ if $found;
		}
		shift @tail if @tail > LOG_ENTRIES;
	    }
	    
	    close FH;
	}
	
	foreach (@tail) {
		chomp;
		# Aug 27 18:44:45 localhost.localdomain : mretr: ERROR - [327] StreamState.cpp 123 sendto(): error code 111 : Connection refused 
		next if not /(\w{3}\s\s?\d\d?\s\d{2}:\d{2}:\d{2})\s[\w._-]+\s.?\s?\w+:\s(\w+)\s-?\s?(\[\d+\])\s(.*)/s;
		my $ts = FormatDate($1);
		push @{$log{$ts}}, "${2} ${3} ${4}";
		
	}
} # sub GetMretrLog

sub GetSysLog { # ----------------------------- read entries from system.log -----
	my $objid = shift;
	
	my $i = 0;
	my @tail;
	do {
		my $name = "$APL/var/log/system.log".($i==0?'':".$i");
		if (-f $name)
		{	
			my $cmd = "egrep '(DEVID=$objid|ID:$objid)' $name | tail -n ".LOG_ENTRIES;
			if(open(SYSLOG,"$cmd|")) {
				push @tail, <SYSLOG>;
				close SYSLOG;
			}
		}
	} until ++$i > LOG_SCAN_MAX || scalar @tail >= LOG_ENTRIES;

	foreach (@tail) {
		# 2009/05/10 03:17:22 /opt/sarch/cam/bin/cam_patrol 417 INFO - CPT-0003 Retriever /opt/sarch/mgears/bin/cam_mretr for DEVID=2 has been started
		next if not /(\w{3}\s\s?\d\d?\s\d{2}:\d{2}:\d{2})\s\S+\s\S+\s\S+\s\S+\s(.*)\s-\s((.*)DEVID=(\d+)(.*))/;
		my $ts = FormatDate($1);
		push @{$log{$ts}}, "$2 - [$5] $3";
	}
} # sub GetSysLog

sub FormatDate {
	my ($month, $day, $time) = split ' ', shift;
	my %months = qw(jan 01 feb 02 mar 03 apr 04 may 05 jun 06 jul 07 aug 08 sep 09 oct 10 nov 11 dec 12);

	$day = "0$day" if $day < 10;

	$month = $months{lc $month};
	my $year = ($month > (localtime)[4]+1) ? (localtime)[5]+1899 : (localtime)[5]+1900;
	return "$year/$month/$day $time";
}

sub CollectLogs {
	my $objid = shift;

	%log = (); # Cleanup
	GetMretrLog($objid);
        GetSysLog($objid);
}

sub SendLogs {
	my $objid = shift;
	my $queue_id = shift;
	
	my $msg = "";
	foreach my $tm (sort keys %log) {
		foreach (@{$log{$tm}}) {
			$msg .= "$tm $_\n";
		}
	}
	eval {
		$Stomp->send(
			destination => "/queue/cameralog/$queue_id", 
			body => $msg,
			'content-length' => ""
		);
	};
	die "Send logs for $objid failed: $@\n" if $@;
}

sub HandleMessage {
	my($self, $frame) = @_;
	
    	#$self->ack(frame => $frame);
    	#printf("received: %s\n", $frame->body());
    	my $body= $frame->body();
    	if ($body =~ /^(\d+);(\w+)$/) {
    		my $objid = $1;
    		my $queue_id = $2;
    		if (-d "$APL_CONF/$objid") {
    			CollectLogs($objid);
    			SendLogs($objid, $queue_id);
    		}
    	}
    	return ($self);
}

sub Listen {
	eval {
		$Stomp->subscribe(
    			destination => "/topic/cameralog_req",
            	        id          => "cameralog"          # required in STOMP 1.1
            		#ack         => "client",           # client side acknowledgment
    		);
		$Stomp->wait_for_frames(callback => sub { return (0) });
	};
	die "Error in Listen: $@\n" if $@;
}

sub Init {
	eval {
		$Stomp = Net::STOMP::Client->new(
	    		host => "127.0.0.1", 
	    		port => $STOMP_PORT,
	    		timeout => $TIMEOUT
	    	);
		$Stomp->connect;
		$Stomp->message_callback(\&HandleMessage);
	};
	if ($@) {
		die "Stomp client init failed: $@\n";
	}
}

sub Main {
	Init;
	Listen;
}

Main;
