package SKM::Session;

use 5.008008;
use strict;
use warnings;

use JSON;
use XML::Simple;
use FileHandle;
use IPC::Open2;
use Digest;
use Data::Dumper;
use LWP::UserAgent;

require Exporter;

our @ISA = qw(Exporter);

our @EXPORT = qw(OpenSession CloseSession LastCode);

our $VERSION = '0.01';

our $SESSION_COOKIE_NAME = "PHPSESSID";
our $TOKEN_COOKIE_NAME = "token";

# Vars ---------------------------------------------------------------
our $SID = '';
our $Token = '';
our $Host = '';
our $Err = '';
our $LastRsp = '';
our $Legacy = 0;

# Init and tune UA
our $UA = LWP::UserAgent->new();
$UA->cookie_jar({});
$UA->timeout(10);
$UA->agent("skm-perl");

# Preloaded methods go here.

sub _sha512hex {
    my $src = shift;
    
    local $SIG{PIPE} = 'IGNORE';
    my $pid = open2(*Reader, *Writer, "/usr/bin/sha512sum --text");
    print Writer $src;
    close Writer;
    my $digest = <Reader>;
    chomp $digest if defined $digest;
    close Reader;
    if (defined $digest) {
	$digest = (split(/\s+/, $digest))[0];
    }
    return $digest;
}

sub _sha1hex {
    my $src = shift;
    
    my $sha1 = Digest->new("SHA-1");
    $sha1->add($src);
    return $sha1->hexdigest;
}

sub _getCookies {
    my %cookies;
    $UA->cookie_jar()->scan( sub {
	$cookies{$_[1]} = $_[2];
    } );
    return \%cookies;
}

sub _getCookie {
    my $name = shift;
    my $cookies = _getCookies;
    return $cookies->{$name};
}

sub OpenSession {

    $Host = shift;
    my ($username, $password) = @_;
    $Err = '';
    $SID = '';
    $Legacy = 0;

    if (! $Host) {
	$Err = "Remote address must be specified!";
	return;
    }

    eval {
	# Detect if this is legacy or 3.7 Stratus	
	my $encKey;
	my $resp = $UA->get("http://$Host/usr/api/login.php?return=logininfo");
	if ($resp->is_success) { # Got legacy system
		$Legacy = 1;
	        my $xml = eval { XMLin($resp->content,KeyAttr=>'NAME') };
    		die("Bad XML returned\n#".$resp->content) if $@;
    		die("Request failed\n#".$resp->content) if $xml->{STATUS}{VALUE} ne 'OK';
    		$encKey = $xml->{LOGIN}{PARAM}{ENCRYPTIONKEY}{VALUE};
    		die("No encryption key in response\n#".$resp->content) unless $encKey;
	}
	else {
	    $resp = $UA->get("http://$Host/api/call.php?function=getLoginInfo");
	    my $json = _parseJSON($resp);
	    $encKey = $json->{loginInfo}{encryptionKey};
	    my $cookies = _getCookies;
	    $Token = $cookies->{$TOKEN_COOKIE_NAME};
	}
	
        # Compute SHA-512 (or SHA-1 for legacy) digest
        my $fhash = $Legacy ? \&_sha1hex : \&_sha512hex;
        my $tmp = $encKey.($Legacy ? uc($fhash->($password)) : $fhash->($password)).$encKey;
        my $digest = $fhash->($tmp);

        # logging in..
        if ($Legacy) {
    	    $resp = $UA->get(
        	"http://$Host/usr/api/login.php?return=login&username=$username&credentials=".
    	        $digest."&hashcash=123"
    	    );
    	    die("HTTP ERROR [".$resp->code."]") if $resp->is_error;
    	    die("Authrorization error\n#",$resp->content) if $resp->content !~m{<STATUS VALUE="OK"/>}s;
        }
        else {
    	    $resp = $UA->post(
        	"http://$Host/api/call.php",
        	{
        	    credentials => $digest,
        	    function    => "login",
        	    name        => $username,
        	    token       => $Token
        	}
    	    );
    	    my $json = _parseJSON($resp);
    	    die("Authrorization error\n#",$resp->content) if $json->{error};
        }
    };
    if ($@) {
	chomp $@;
	$Err = $@;
	return;
    }
    # Success
    my $cookies = _getCookies;
    $SID = $cookies->{$SESSION_COOKIE_NAME};
    $Token = $cookies->{$TOKEN_COOKIE_NAME} if not $Legacy;
    # Clear cookies
    #$ua->cookie_jar()->clear;
    
    if (! $SID) {
	$Err = "No SessionId in response";
	return;
    }

    return wantarray ? ($SID, $Token, $Legacy) : $SID;
}

sub _parseJSON {
    my $httpResp = shift;
    die("HTTP ERROR [".$httpResp->code."]") if $httpResp->is_error;

    my $json = eval { decode_json($httpResp->content) };
    $LastRsp = $json;
    die("Bad JSON returned\n#".$httpResp->content) if $@;
    die("Request failed\n#".$httpResp->content) if $json->{error};
    
    return $json;
}

sub _setCookie {
    my ($name, $val) = @_;
    return if not defined $name or not defined $val;
    $UA->default_headers(HTTP::Headers->new('Cookie'=>$name."=".$val));
}

sub _setSessionCookie {
    return unless $_[0];
    _setCookie($SESSION_COOKIE_NAME, $_[0]);
}

sub CloseSession {
    my $sid = shift;

    if (! $Host) {
	$Err = "Host should be specified!";
	return;
    }
    _setSessionCookie($sid || $SID);
    my $url = "http://$Host/api/call.php?function=logout";
    $url .= "&token=$Token" if $Token;
    my $resp = $UA->get($url);
    eval {
	my $json = _parseJSON($resp);
    };
    if ($@) {
	chomp $@;
	$Err = $@;
	return;
    }
    $SID = '' if not $sid or ($sid and $sid eq $SID);
    return 1;
}

sub LastCode {
    return '' if not $LastRsp or ref $LastRsp ne 'HASH';
    return $LastRsp->{code};
}

1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

SKM::Session - Simple API for remote session management

=head1 SYNOPSIS

  use SKM::Session;
  blah blah blah

=head1 DESCRIPTION

SKM::Session - Perl module for internal use
Provides simple API for open/check/prolong/close of remote PHP session

=head1 SEE ALSO

Mention other useful documentation such as the documentation of
related modules or operating system documentation (such as man pages
in UNIX), or any relevant external documentation such as RFCs or
standards.

If you have a mailing list set up for your module, mention it here.

If you have a web site set up for your module, mention it here.

=head1 AUTHOR

A. Tsibulnik, E<lt>crash@videonext.netE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2010 by videoNEXT

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.8 or,
at your option, any later version of Perl 5 you may have available.


=cut
