package SKM::SF;

use 5.008008;
use strict;
use warnings;
use XML::Simple;
use File::Basename;
use File::Copy;
use Tie::IxHash;
use SKM::Common ":all";
use SKM::DB;
use NextCAM::Conf "GetCfgs";

require Exporter;

our @ISA = qw(Exporter);

our %EXPORT_TAGS = ( 'all' => [ qw(
    SFInit SFCollect SFPackage SFComplete SFID SFCollectDir
    SFExitOK SFExitNospace SFExitFail SFExitIncomplete SFExitSignal SFExitStatus
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
    SFInit SFCollect SFPackage SFComplete SFID SFCollectDir
    SFExitOK SFExitNospace SFExitFail SFExitIncomplete SFExitSignal SFExitStatus
);

# Globals
our $VERSION = '0.01';
our $DULimit;

# Constants ----------------------------------------------------------
my $APL = $ENV{APL} || '/opt/sarch';
my $SF_CONF   = "$APL/conf/etc/sf_collect.xml";
my $VCTL = "$APL/vpatch/bin/vctl";
my $TMPDIR = "$APL/var/sf/tmp/node";
my $SFDIR = "$APL/var/sf/node";
my $LOGDIR = "/var/sarch/log";
my $MINSPACE = 300; # Minimum free space on device to start collecting
my $MAXSPACE = 700; # Maximum temporary dir size
my $STATUS_OK = 1;
my $STATUS_FAIL = 0;
my $SMINS = "$APL/sm/bin/sm_wheels ins";
my $MASTER = -f "$APL/var/conf/master/s_master";
my $PSQL = "/usr/bin/psql";
my $GZIP = "/usr/bin/gzip";

# Targets
my $TARGET_ALL = 'all';
my $TARGET_MASTER = 'master';
my $TARGET_SLAVE = 'slave';

# Exit codes
my $EXIT_OK = 0;
my $EXIT_FAIL = 1;
my $EXIT_INCOMPLETE = 2;
my $EXIT_SIGNAL = 3;
my $EXIT_NOSPACE = 4;

# Vars ---------------------------------------------------------------
my $RootMode = 0;
my $Log;
my $Caller = 'user';
my $CollectDir;
my $Password = 'X3ins1de';
my $ConfigFile = "$APL/conf/etc/sf_collect.xml";
my $dbm;
my $ExitStatus = $EXIT_OK;
my $HTML = "";
my $SF;
my $SFID;
my %Actions = ();
my %Dest;
my %DU;         # Disk space used by currently collected files
my %Vinfo;
my %Summary;

# Preloaded methods go here.

# Subs for updatign exit status
sub SFExitOK { $ExitStatus = $EXIT_OK }
sub SFExitNospace { $ExitStatus = $EXIT_NOSPACE }
sub SFExitFail { $ExitStatus = $EXIT_FAIL if $ExitStatus != $EXIT_NOSPACE }
sub SFExitIncomplete { $ExitStatus = $EXIT_INCOMPLETE }
sub SFExitSignal { $ExitStatus = $EXIT_SIGNAL }
sub SFExitStatus { $ExitStatus }

# Get version info
#
sub VerInfo {
    return \%Vinfo if %Vinfo;
    open(VCTL, "$VCTL info|") or die "Failed to run $VCTL: $!\n";
    %Vinfo = %{{map {/(\w+)=(.*)/} grep {/^\w+=.*$/} <VCTL>}};
    close(VCTL);
    return \%Vinfo;
}

# Check command exit code, log error on failure
# 
sub ExecCmd {
    my $cmd = shift;
    
    system($cmd);
    return if $? == 0;
    my $err;
    if ($? == -1) {
        $err = "failed to execute cmd '$cmd': $!";
    }
    elsif ($? & 127) {
        $err = sprintf "cmd '$cmd' died with signal %d, %s coredump",
                    ($? & 127),  ($? & 128) ? 'with' : 'without';
    }
    else {
        $err = sprintf "cmd '$cmd' exited with value %d", $? >> 8;
    }
    $Log->error($err);
}

# Build unique support file name 
#
sub SFID {
    return $SFID if $SFID;
    VerInfo;
    $Vinfo{sysid}=$Vinfo{uni};
    
    my $short_verid = substr($Vinfo{verid}, 0, index($Vinfo{verid}, $Vinfo{tag}));
    my ($tmin,$thour,$tday,$tmon,$tyear)=(gmtime(time))[1..5];
    my $uid = "${Vinfo{sysid}}_${short_verid}" . ($tyear+1900) . sprintf("%02s", $tmon+1) .
              sprintf("%02s", $tday) . '_' . sprintf("%02s", $thour) . sprintf("%02s", $tmin) .
              ($Caller eq 'auto'?'a':'u');
    return $uid;
}

sub SFCollectDir {
    if (@_) {
	my $prev = $CollectDir;
	$CollectDir = $_[0];
	return $prev;
    }
    return $CollectDir;
}

# Establishes connection to database
#
sub DB_Connect {
    my $num_of_tries = shift || 2;
    my $timeout = shift || 5;
    
    for(my $i=0; $i < $num_of_tries; ++$i) {
	eval {
	    $dbm = DBMaster({RaiseError => 1});
	};
	if($@) { 
	    $Log->error("Failed to connect to DB: $@");
	    last if $i == $num_of_tries - 1;
	    sleep $timeout;
	}
	else { last; }
    }
}

# Initialize all variables
# Perform necessary checks
#
sub SFInit {
    my %args = @_;
    $Log = $args{Log};
    $CollectDir = $args{CollectDir};
    $SFID = $args{SFID} || SFID;
    $Password = $args{Password} if defined $args{Password};
    $ConfigFile = $args{Config} if defined $args{Config};
    $Caller = $args{Caller} if defined $args{Caller};
    $DULimit = $args{DULimit} if defined $args{DULimit};
    $CollectDir = "$TMPDIR/$SFID" unless $CollectDir;
    
    $SF = "$SFDIR/$SFID.INPR";
    
    die "'Log' should be a valid Log4perl logger!\n"
	unless UNIVERSAL::isa($Log, 'Log::Log4perl::Logger');
    
    if ($> == 0) { # Fix for running from root
	$RootMode = 1;
    }
    
    my $r = eval { XMLin($SF_CONF, ForceArray=>1) };
    die "Error parsing sf_collect.xml: $@" if $@;
    
    %Actions = %{$r->{'SET'}};
    tie(%Summary, 'Tie::IxHash');
    
    unless ($RootMode) {
	# Find out if SF with the same ID already present
        opendir(DH, $SFDIR) or die "Failed to open $SFDIR: $!";
        while(my $sf = readdir(DH)) {
    	    next unless $sf =~ /$SFID/;
	    if ($sf =~ /\.TERM$/) {
		system("rm -f $SFDIR/sf");
	        last;
	    }
	    else {
		closedir(DH);
	        die "SF with current timestamp already exist!";
	    }
	}
	closedir(DH);

	# Check available space on devices
        DiskSpaceCheck($TMPDIR);
    }
    
    # Establish DB connection if needed
    my $dbconnect = 0;
    foreach my $rset (values %Actions) {
	last if $dbconnect;
	foreach my $action (values %{ $rset->{ACTION} }) {
	    if ($action->{type} eq 'query') {
		$dbconnect = 1;
		last;
	    }
	}
    }
    DB_Connect if $dbconnect;
}

# Check if SF temporary files fo not exceed overal space usage limit
# If yes, do partial packaging and remove files
# 
sub CheckDU {
    my ($file, $pre) = @_;
    
    $DU{$file} = -s $file;
    my $total = 0;
    foreach my $f (keys %DU) {
        next if not $DU{$f};
        $total += int($DU{$f});
    }
    # Convert bytes to MB
    $total = int($total / 1024 / 1024);
    if ($total >= $DULimit) {
        PackageAndClean();
        %DU = ();
    }
    elsif ($pre) { # Pre-check - remove file from cache after check
        delete $DU{$file};
    }
}

# Prints formatted content of a snapshot to a file
#
sub DumpRowset {
    my $action = shift;
    my $fh = shift;    
    my $rowset;
    
    # If compressed option was specified, do not use own DB connection, use psql+gzip instead
    if ($action->{compress}) {
        system("$PSQL -c '$action->{sql}' | $GZIP >> $action->{dst}");
        die "Error status returned when storing compressed query\n" if $?;
    }
    else {
        # One more attempt to connect to DB
        DB_Connect(1, 1) unless $dbm;
    
        print $fh "QUERY: $action->{sql}\n\n";
        # Fetch rows
        my $sth = $dbm->prepare($action->{sql});
        $sth->execute();
        $rowset = $sth->fetchall_arrayref([]);
    
        # Print to filehandle
        ArrayTable($sth->{NAME}, $rowset, $fh);
    }
}

# Dumps command output to a file
#
sub DumpCmd {
    my $action = shift;
    
    my $cmd_line;
    if($action->{type} =~ /copy/i) {
	$cmd_line = "cp $action->{options} $action->{src} $action->{dst} 2>/dev/null";
    } else {
	$cmd_line = "$action->{cmd} >> \"$action->{dst}\" 2>/dev/null";
    }
    ExecCmd($cmd_line);
    CheckDU($action->{dst});
}

# Copy group of files matching specified mask
#
sub GroupCopy {
    my $action = shift;
    
    $action->{toc_href_list} = [];
    system("mkdir -p $action->{dst} 2>/dev/null") unless -e $action->{dst};
    # List files and compute their sizes
    eval {
        open(LS, "ls -A1 $action->{src} 2>/dev/null |") or die "Cannot list files: $!";
        my @files = <LS>;
        close LS;
        foreach my $file (@files) {
            chomp $file;
            my $base = basename($file);
            push @{ $action->{toc_href_list} }, $base;
            my $dst = "$action->{dst}/$base";
            unlink $dst if -f $dst;
            # Pre-check space and do packaging+cleanup before copying (useful for big files)
            CheckDU($file, 1);
            die "Failed to copy '$file' to $action->{dst}: $!" 
                if not copy($file, $dst);
            CheckDU($dst) if -f $dst;
        }
    };
    die $@ if $@;
}

# Executes one action from the config file
#
sub DoAction {
    my $action = shift;
    # Do nothing if destination file already exists and hasn't been created by us
    return if -f $action->{dst} && -s $action->{dst} && !$Dest{$action->{dst}};
    
    $Dest{$action->{dst}}++;
    if($action->{type} =~ /^query$/i) {
	open(FD, ">>$action->{dst}") or die "Failed to open file: $!";
	print FD "$action->{description}\n" if defined($action->{description});
	if (not $action->{compress}) {
	    DumpRowset($action, \*FD);
	    close(FD);
	}
	else {
	    close(FD);
	    DumpRowset($action);
	}
	CheckDU($action->{dst});
    } 
    elsif($action->{type} =~ /^cmd$/i) {
	open(FD, ">>$action->{dst}") or die "Failed to open file: $!";
	print FD "$action->{description}\n" if defined($action->{description});
	close(FD);
	DumpCmd($action);
    }
    elsif($action->{type} =~ /^copy$/i) {
	DumpCmd($action);
    }
    elsif($action->{type} =~ /^groupcopy$/i) {
	GroupCopy($action);
    }
}

# Executes all actions from the set
#
sub ExecSet {
    my $rset = shift;
    my $dir = "$CollectDir/$rset->{dir}";
    my $set_target = $rset->{target} || $TARGET_ALL;
    mkdir($dir) or die "Unable to create directory: $!" unless -d $dir;
    
    my %actions = %{$rset->{ACTION}};
    while( my ($name, $action) = each(%actions) ){
	my $action_target = $action->{target} || $set_target;
	my $action_su = defined($action->{superuser}) && $action->{superuser} eq 'yes';
	if ( ( $MASTER && $action_target eq $TARGET_SLAVE) ||
	     (!$MASTER && $action_target eq $TARGET_MASTER)||
	     ( $action_su != $RootMode) )
	{
	    $action->{skip} = 1;
	    next;
	}
	$action->{dst} = '' unless defined $action->{dst}; # can be empty for groupcopy entries
	$action->{toc_href} = "$rset->{dir}/$action->{dst}";
	$action->{dst} = "$dir/$action->{dst}";
	eval {
	    DoAction($action);
	};
	if($@) {
	    $action->{status} = $STATUS_FAIL;
	    $action->{error} = $@;
	    chomp $action->{error};
	    SFExitIncomplete;
	    $Log->error("Action $name failed: $@");
	    die $@ if $@=~/^TIMEOUT_SET/;
	} else {
	    $action->{status} = $STATUS_OK;
	}
	print "Action: $name. Status: " . ($action->{status} eq $STATUS_OK?'OK':"FAIL ($action->{error})") . "\n";
    }
}

# Collect system summary
#
sub Summary {
    # 1. date
    $Summary{date} = gmtime();
    print "\nSummary:\n";
    print "date: $Summary{date}\n";
    # 2. sysid and verid
    VerInfo;
    @Summary{'sysid', 'verid', 'rev'} = @Vinfo{'sysid', 'verid', 'rev'};         
    print "sysid: $Vinfo{sysid}\nverid: $Vinfo{verid}\nrev: $Vinfo{rev}\n";
    # 3. system uptime
    open(UPTIME, "/usr/bin/uptime|") or warn "Failed to get uptime";
    $Summary{uptime} = $1 if <UPTIME>=~/\sup\s+\b(.+?),\s/;
    close(UPTIME);
    print "uptime: $Summary{uptime}\n";
    # 4. Config
    my %cfgs = GetCfgs();
    @Summary{'cameras', 'audio', 'sensors', 'relays', 'monitors', 'walls'} = (0, 0, 0, 0, 0, 0);
    foreach my $dev (keys %cfgs) {
	$Summary{cameras}++  if $cfgs{$dev}{DEVICETYPE} =~ /CAMERA/i;
	$Summary{audio}++    if $cfgs{$dev}{DEVICETYPE} =~ /AUDIO/i;
	$Summary{sensors}++  if defined $cfgs{$dev}{SUBTYPE} and $cfgs{$dev}{SUBTYPE} eq 'S';
	$Summary{relays}++   if defined $cfgs{$dev}{SUBTYPE} and $cfgs{$dev}{SUBTYPE} eq 'R';
	$Summary{monitors}++ if $cfgs{$dev}{DEVICETYPE} =~ /MONITOR/i;
	$Summary{walls}++    if $cfgs{$dev}{DEVICETYPE} =~ /WALL/i;
    }
    print "cameras: $Summary{cameras}\n";
    print "audio: $Summary{audio}\n";
    print "sensors: $Summary{sensors}\n";
    print "monitors: $Summary{monitors}\n";
    print "walls: $Summary{walls}\n";
    
    # 5. Storage summary
    if(open(SMINS, "$SMINS |")) {
	my ($size, $free, $used);
	while (my $line = <SMINS>) {
	    next unless $line =~ /^Total:\s+(\d+)\s+(\d+)/;
	    ($size, $free) = ($1, $2);
	    $used = $size - $free;
	    last;
	}
	close SMINS;
	$Summary{storage} = "${free}GB free, ${used}GB used"
	    if defined($free)&&defined($used);
	print "storage: $Summary{storage}\n";
    }
}

# Write statistics to the table of contents file
#
sub AppendTOCAction {
    my $set_name = shift;
    my $name = shift;
    my $action = shift;
    
    my $pos_start = index($HTML, "<H1>$set_name</H1>");
    if($pos_start != -1) {
	my $pos_end = index($HTML, "<H1>", $pos_start+1);
	my $status = $action->{status} eq $STATUS_OK?'<FONT COLOR="#00FF00">OK</FONT>':'<FONT COLOR="#FF0000">FAIL</FONT>';
	my $insert = "";
	$action->{toc_href} =~ s|/$|| unless $action->{toc_href} eq '/';
	
	if($action->{toc_href_list}) {
	    my @href_list = sort {$a cmp $b} @{$action->{toc_href_list}};    
	    unless(@href_list) {
		$insert .= $name;
	    }
	    elsif(@href_list == 1) {
		$insert .= "<A HREF='$action->{toc_href}/$href_list[0]'>$name</A>"
	    }
	    else {
		$insert .= "<A HREF='$action->{toc_href}/$href_list[0]'>$name</A>&nbsp;&nbsp;[&nbsp;";
		for (my $i = 1; $i < @href_list; ++$i) {
		    $insert .= "<A HREF='$action->{toc_href}/$href_list[$i]'>$i</A>" .
		    ($i<$#href_list?',':'') . "&nbsp;";
		}
		$insert .= ']';
	    }
	    
	} 
	elsif(defined $action->{toc_href}) {
	    $insert .= "<A HREF='$action->{toc_href}'>$name</A>";
	} 
	else {
	    $insert .= $name;
	}
	$insert .=  '&nbsp;' x 6 . $status . "<BR>\n";
	
	# Check if action already is included in TOC
	my $is_in = index($HTML, $insert, $pos_start);
	return if ($is_in != -1 || ($is_in >= $pos_end && $pos_end != -1));
	
	# Insert in the middle of  HTML
	if($pos_end != -1) {
	    eval {
		substr($HTML, $pos_end, 0) = $insert;
	    };
	} else {
	    # Add to the end of HTML
	    $HTML .= $insert;
	}
    }    
}

sub AppendTOCSet {
    my $set_name = shift;
    my $insert = "<H1>$set_name</H1>";
    return if index($HTML, $insert) != -1;
    $HTML .= "$insert\n";
}

# Create Table of Contents for SF
#
sub GenTOC {
    my $toc_file = "$CollectDir/toc.html";
    my $header = '<HTML><TITLE>Table of Contents</TITLE><BODY>';
    my $footer = '</BODY></HTML>';
    my $summary = '<H1>Summary</H1>';
    
    # Header
    $HTML .= "$header\n";
    
    # Read file contents to $HTML if file already exists and has some data
    if(-s $toc_file) {
	open(TOC, "$toc_file") or die "Failed to open $toc_file for reading: $!";
	while(<TOC>) {
	    $HTML .= $_ unless (/$header/ || /$footer/);
	}
	close(TOC);
    }    
    
    # Write summary
    if(index($HTML, $summary) == -1) {
	$HTML .= "$summary\n";
	while( my ($key, $val) = each(%Summary) ) {
	    $HTML .= "$key: $val<BR>\n";
	}
    }
    
    # Write other sets
    while( my ($set_name, $rset) = each(%Actions) ) {
	AppendTOCSet($rset->{title});
	my %actions = %{$rset->{ACTION}};
	while( my ($name, $action) = each(%actions) ){
	    AppendTOCAction($rset->{title}, $name, $action) unless $action->{skip};
	}
    }
    
    # Footer
    $HTML .= $footer;
    
    open(TOC, ">$toc_file")  or die "Failed to open $toc_file for writing: $!";
    print TOC $HTML;
    close(TOC);		
}

# Collecting files and error handling
#
sub SFCollect {
    my $cleanup = shift; # Do package and clean after collecting
    
    $Log->info("SF collecting starts: RootMode=$RootMode DULimit=$DULimit CollectDir=$CollectDir");
    
    # Create support file (in-progress state)
    $SF = "$SFDIR/$SFID.INPR";
    open(SF, ">$SF") or die "Failed to create support file: $!";
    close(SF);
    # Create collect dir
    mkdir($CollectDir) or die "Failed to create collect dir $CollectDir: $!" unless -d $CollectDir;

    while( my ($set_name, $action_set) = each(%Actions) ) {
	eval {
	    # 5 min timeout per set by default
	    local $SIG{ALRM} = sub { die "TIMEOUT_SET" }; alarm($action_set->{timeout} || 300);
	    ExecSet($action_set);
	};
	if($@) {
	    $Log->error("Set $set_name failed: $@");
	}
    }    
    # Turn off timer
    alarm 0;
        
    unless ($RootMode) {
	# Now collect summary
	eval {
	    # 60 sec timeout for collecting summary
	    local $SIG{ALRM} = sub { die "TIMEOUT_SUMMARY" }; alarm 60;
	    Summary;
	};
        if($@) {
    	    $Log->error("Summary failed: $@");
	}
        alarm 0;
    }
    
    # Add records to the TOC file
    eval {
	GenTOC;
    };
    $Log->error("Generate TOC failed: $@") if $@;
    
    # Do cleanup if specified
    PackageAndClean() if $cleanup;
}

sub DiskSpaceCheck {
    my $path = shift;
    my $free_space;
    my $min_space = $MINSPACE;
    open(DF, "/bin/df $path 2>/dev/null|") || $Log->error("Cannot do df $path");
    my @rows=<DF>;
    close DF;
    foreach(@rows) {
      if(/\s\d+\s+\d+\s+(\d+)\s+\d+%\s+(.+)/) {
        my ($free_space, $dev) = (int($1/1024), $2);
	if($free_space < $min_space) {
	    SFExitNospace;
	    die "Not enough free space to start collecting! ${free_space}MB found on $dev but need at least $min_space";
	}
	if ($free_space >= 1000) {
	    $DULimit = $free_space - 400;
	}
	elsif ($free_space >= 600) {
	    $DULimit = $free_space - 250;
	}
	else {
            $DULimit = $free_space - 200;
        }
        $DULimit = $MAXSPACE if $DULimit > $MAXSPACE;
      }
    }
}

sub PackageAndClean {
    $Log->info("Disk usage limit ($DULimit) is reached. Do interim packaging");
    print "Package and clean\n";
    my $basename = basename($CollectDir);
    # Do zipping
    ExecCmd("cd $TMPDIR && /usr/bin/zip -ry $TMPDIR/$SFID.tmp $basename 1>/dev/null");
    # Remove all files from tmp dir, but leave dirs
    ExecCmd("find $CollectDir -type f -not -name 'toc.html' -delete");
    # Chown interim archive
    system("chown apl:apache $TMPDIR/$SFID.tmp &>/dev/null");
}

sub SFPackage {
    print "Final packaging\n";
    my $basename = basename($CollectDir);
    # First chmod files
    system("chmod -R +r $CollectDir");
    # Do zipping
    ExecCmd("cd $TMPDIR && /usr/bin/zip -ry $TMPDIR/$SFID.tmp $basename 1>/dev/null");
    rename("$TMPDIR/$SFID.tmp", $SF) or die "Failed to move tmp zip file to $SFDIR: $!";
    system("chgrp apache $SF &>/dev/null");
}

sub SFComplete {
    unless ($RootMode) {
	# Set appropriate extension for SF
        my $ext = $1 if $SF=~m|/$SFID\.(.+)$|;
        return unless $ext;
        for($ExitStatus) {
	    $_==$EXIT_OK	 && do { $ext = 'nspt'    };
	    $_==$EXIT_FAIL       && do { $ext = 'FAIL'    };
	    $_==$EXIT_INCOMPLETE && do { $ext = 'nspt'    };
	    $_==$EXIT_SIGNAL     && do { $ext = 'TERM'    };
	    $_==$EXIT_NOSPACE    && do { $ext = 'NOSPACE' };
	}	
        rename($SF, "$SFDIR/$SFID.$ext");
    
        # Cleanup
        system("rm -rf $CollectDir");
    }
    eval { $dbm->disconnect } if $dbm;
}

1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

SKM::DB - Perl extension for blah blah blah

=head1 SYNOPSIS

  use SKM::DB;
  blah blah blah

=head1 DESCRIPTION

Stub documentation for SKM::DB, created by h2xs. It looks like the
author of the extension was negligent enough to leave the stub
unedited.

Blah blah blah.


=head1 SEE ALSO

Mention other useful documentation such as the documentation of
related modules or operating system documentation (such as man pages
in UNIX), or any relevant external documentation such as RFCs or
standards.

If you have a mailing list set up for your module, mention it here.

If you have a web site set up for your module, mention it here.

=head1 AUTHOR

A. Tsibulnik, E<lt>crash@videonext.netE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2010 by videoNEXT

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.8 or,
at your option, any later version of Perl 5 you may have available.


=cut
