package SKM::Common;

use 5.008008;
use strict;
use warnings;
use Time::Local;

require Exporter;

our @ISA = qw(Exporter);

our %EXPORT_TAGS = ( 'all' => [ qw(
 CheckPid WritePid ReadPid RemovePid UpdatePid ProcName ArrayTable UNIX2ReadableTime ReadableTime2UNIX
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
 ReadPid CheckPid WritePid RemovePid UpdatePid ProcName
);

our $VERSION = '0.01';

my $APL      = $ENV{APL};
my $NODECONF = "$APL/var/conf/node/conf";
my $PIDPATH  = "$APL/var/pids";


#--------------------------------
# Pid management		|
#--------------------------------

sub ProcName { # -------------------------------------------------------------
    (split(/\//,$0))[-1];
} # sub ProcName

sub CheckPid { # -------------------------------------------------------------
    my $procname = ProcName;
    my $pid_name = shift || $procname;
    
    my $old_pid = ReadPid($pid_name);
    if ($old_pid > 1) {
	my $format = $^O =~ /darwin/i ? "command=" : "comm=";
	open PS, "ps -p $old_pid -o $format |"; # get the name of proc with old_pid
	my $proc = <PS>;
	close PS;
	if (defined $proc) {
	    chomp $proc;
	    return $old_pid if $^O =~ /darwin/i and $proc =~ m|^/usr/bin/perl.+$procname|;
	    return $old_pid if $proc eq $procname;
	    
	}
    }
    return 0;
} # sub CheckPid

sub ReadPid { # --------------------------------------------------------------
    my $pid_name = shift || ProcName;
    
    my $pid = 0;
    if (open PID,"$PIDPATH/${pid_name}.pid") {
	$pid = <PID>;
	close PID;
	chomp $pid;
    } 
    return $pid;
} # sub ReadPid

sub WritePid { # -------------------------------------------------------------
    my $pid_name = shift || ProcName;
    
    if (open PID,">$PIDPATH/${pid_name}.pid") {
	print PID "$$\n";
	close PID;
	return 1;
    } else {
	return 0;
    }
} # sub WritePid

sub RemovePid { # ------------------------------------------------------------
    my $pid_name = shift || ProcName;
    
    if (ReadPid($pid_name) == $$) {
	unlink "$PIDPATH/${pid_name}.pid" or return 0;
    }
    return 1;
} # sub RemovePid

sub UpdatePid { # ------------------------------------------------------------
    my $pid_name = shift || ProcName;
    
    utime undef, undef, "$PIDPATH/${pid_name}.pid" or return 0;
    return 1;
} # sub UpdatePid

#--------------------------------
# Miscellaneous			|
#--------------------------------

# Prints pretty-formatted view of 2-dimensional array
sub ArrayTable {
    return if not $_[0] or not $_[1];
    my @names = @{$_[0]};
    my @table = @{$_[1]};
    my $fh = $_[2] || \*STDOUT;
    return if not @names or not @table;
    no warnings 'uninitialized';
    
    my $num_of_fields = @names;
    my $num_of_rows = @table;
    
    # Get maximum length required for each field
    my @maxlen;
    $maxlen[$_] = length($names[$_]) foreach (0..$#names);
    my $r;
    for(my $i = 0; $i < $num_of_rows; ++$i) {
	for (my $j = 0; $j < $num_of_fields; $j++) {
	    my $len = length($table[$i][$j]);
	    $maxlen[$j] = $len if $len > $maxlen[$j];
	}
    }
    
    my $pre = 2;
    my $post = 2;
    my $total_len = 1;
    $total_len += ($pre+$post+$_+1) foreach (@maxlen);
    
    my $line = '-' x $total_len;
    my $post_spaces = 0;
    my $header = "|";
    
    # Print header
    for(my $i = 0; $i < $num_of_fields; ++$i) {
	$post_spaces = $maxlen[$i] + $post - length($names[$i]);
	$header .= (' ' x $pre) . $names[$i] . (' ' x $post_spaces) . '|';
    }
    print $fh "$line\n$header\n$line\n";
    
    # Print rows
    for(my $i = 0; $i < $num_of_rows; ++$i) {
	my $row = '|';
	for(my $j = 0; $j < $num_of_fields; ++$j) {
	    my $max_len = $maxlen[$j];
	    $post_spaces = $max_len + $post - length($table[$i][$j]);
	    $row .=  ((' ' x $pre) . $table[$i][$j] . (' ' x $post_spaces) . '|');
	}
	print $fh "$row\n";	
    }
    
    print $fh "$line\n";
} # Sub ArrayTable

# Converts UNIX timestamp to human readable time string
sub UNIX2ReadableTime {
    my ($time, $format_code) = @_;
    $format_code = 0 if not $format_code;
    my $format = "%02s%02s%02s%02s%02s%02s";
    if ($format_code == 1) {
        $format = "%02s%02s%02s_%02s%02s%02s";
    } elsif ($format_code == 2) {
        $format = "%02s-%02s-%02s %02s:%02s:%02s";
    }
    my ($tsec,$tmin,$thour,$tday,$tmon,$tyear)=(gmtime($time))[0..5];
    return sprintf($format,$tyear-100,$tmon+1,$tday,$thour,$tmin,$tsec);
}

# Converts radable time string to UNIX timestamp
sub ReadableTime2UNIX {
    my $ts = shift;
    if ($ts =~ /^(\d{2})\D*(\d{2})\D*(\d{2})\D*(\d{2})\D*(\d{2})\D*(\d{2})$/) {
        my ($year,$mon,$day,$hour,$min,$sec) = ($1,$2,$3,$4,$5,$6);
        return timegm($sec,$min,$hour,$day,$mon-1,$year+100);
    }
    return -1;
}

1;
__END__
