package NextCAM::Messenger;

use IO::Socket;
use IO::Select;

use Data::Dumper;
use NextCAM::Messenger::Shared;

use threads;
use threads::shared;

use 5.008;

use strict;
use warnings;

use vars qw($VERSION);
$VERSION = 2.22;

our $mutex : shared;

use constant CHUNK_HEADER_LENGTH => 8;
use constant CRC_LENGTH          => 4;
use constant AUTH_NONE           => 0;
use constant AUTH_SESSION        => 1;

################
# Construction #
################

sub new {

	my($class, %opts) = @_;
	
	my $host = delete $opts{host};
	my $port = delete $opts{port};
	my $callback = delete $opts{callback};
	my $clbk_disconnect = delete $opts{clbk_disconnect};
	my $conn_timeout = delete $opts{conn_timeout} || 1000;
	my $autoconnect = delete $opts{autoconnect};
	my $helo = delete $opts{helo};
	my $ext_sid = delete $opts{ext_sid};
	
	$autoconnect = 1 unless defined $autoconnect;
	$helo = "$0 (NextCAM::Messenger)" unless defined $helo;
	
	
	if( %opts && $^W ) {
	    Carp::carp("Unrecognized NextCAM::Messenger options: @{[sort keys %opts]}");
	}
	
	my $self  = bless {
			    host            => $host,
			    port            => $port,
			    callback        => $callback,
			    clbk_disconnect => $clbk_disconnect,
			    conn_timeout    => $conn_timeout,
			    autoconnect     => $autoconnect,
			    helo            => $helo,
			    ext_sid         => $ext_sid
			}, $class;
	
	$self->{subscriptions} = {};
			
	if(  $autoconnect >= 1 )
	{
	    my $i = 0;
	    while( $i < $autoconnect && !$self->connect )
	    {
		$i++;
	    }
	    
	    return undef if $i == $autoconnect;
	}
	
	$self->{chunk_queue} = ();
	
	return $self;	
}



sub host { shift->{host}; }
sub port { shift->{port}; }
sub conn_timeout { shift->_elem("conn_timeout", @_); }
sub sock { shift->{sock}; }
sub sid { shift->{sid}; }

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# 'Private' functions		                           +
#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

##################
# Chunk Builders #
##################

sub build_text {
    
    my %chunk;
    my $len = length $_[1];
    $chunk{'sign'} = 'TEXT';
    $chunk{'length'} = $len + CRC_LENGTH + 4;
    $chunk{'text_length'} = $len;
    $chunk{'text'} = $_[1];
    return \%chunk;
}


sub build_spec {

    my $self = shift;
    my %chunk;
    my $device_IP = $self->build_text($_[0]);
    my $device_ID = $self->build_text($_[1]);
    $chunk{'sign'} = 'SPEC';
    $chunk{'length'} = $device_IP->{length} + $device_ID->{length} + 
		      2 * CHUNK_HEADER_LENGTH + CRC_LENGTH + 16;
    $chunk{device_IP} = $device_IP;
    $chunk{device_ID} = $device_ID;
    $chunk{ctxType} = $_[2];
    $chunk{evtType} = $_[3];
    $chunk{procLvl} = $_[4];
    $chunk{message_ID} = $_[5];
    return \%chunk;
}


#############################
# Chunk readers and writers #
#############################


sub read_text {
    
    lock $mutex;
    
    my ($self, $chunk, $timeout) = @_;
    my $sock = $self->{sock};
    my ($text, $text_length) = ("", 0);
    	
    my ($bytes_read, $bytes_total) = (0, 0);
    my $select = $self->{select};
    
    return 0 unless $select->can_read($timeout);

    $text_length = $self->read_uint32($timeout);
    $text_length = unpack('L', $text_length);
    
    while($bytes_total != $text_length)
    {
       if($select->can_read($timeout))
       {		
       	  $bytes_read = $sock->sysread($text, $text_length - $bytes_total, $bytes_total);
       	  return undef unless $bytes_read;
       	  $bytes_total += $bytes_read;
       }
    } 
   
    
    $chunk->{'text_length'} = $text_length;	
    $chunk->{'text'} = $text;
    
    return 1;
}


sub send_text {
    
    my ($self, $chunk) = @_;
    my $sock = $self->{sock};
    
    my $text_length = $chunk->{text_length};
    $sock->syswrite(pack('L', $text_length), 4);
    $sock->syswrite($chunk->{text}, $text_length);
}

sub read_ackw {
    
    lock $mutex;
    
    my ($self, $chunk, $timeout) = @_;
    my $sock = $self->{sock};
    my ($request, $result);
    
    my $select = $self->{select};
  
    return 0 unless $select->can_read($timeout);
    
    $sock->sysread($request, 4);
    $sock->sysread($result, 4);
    if ($chunk->{length} > 16)  { # Protocol version sent
	my $ver;
	$sock->sysread($ver, 4);
	$chunk->{'protocol_version'} = unpack('L', $ver);
    }
    
    $chunk->{'result'} = unpack('L', $result);
    $chunk->{'request'} = $request;
	
    if( $request eq 'CONN' )
    {
	my $session_ID;

	return 0 unless $select->can_read($timeout);
	
	$sock->sysread($session_ID, 4);	
	$chunk->{'session_ID'} = unpack('L', $session_ID);
    }
    elsif( $request eq 'SUBS' || $request eq 'UNSB' )
    {
	my $subscription_ID;
	
	return 0 unless $select->can_read($timeout);
	$sock->sysread($subscription_ID, 4);  
	$chunk->{'subscription_ID'} = unpack('L', $subscription_ID);
    }
    elsif( $request eq 'LIST' )
    {
	my $slst;
	
	$slst = $self->read_chunk($timeout) or
		    return $slst;
	
	$chunk->{'subscriptions'} = $slst;
    }
    
    return 1;
}


sub read_slst {

    lock $mutex;

    my ($self, $chunk, $timeout) = @_;
    my $sock = $self->{sock};
    
    my $count;
    my $select = $self->{select};
    return 0 unless $select->can_read($timeout);
    
    $sock->sysread($count, 4);
    $count = unpack('L', $count);
    
    $chunk->{'count'} = $count;
	
    my @item = ();
    my @subscriber = ();

    for( my $i = 0; $i < $count; ++$i )
    {
	$item[$i] =  $self->read_chunk($timeout)
		    or return $item[$i];
	$subscriber[$i] = $self->read_chunk($timeout) 
		    or return $subscriber[$i];
    }

    $chunk->{'item'} = \@item;
    $chunk->{'subscriber'} = \@subscriber;
    
    return 1;
}

sub read_spec {

    lock $mutex;

    my ($self, $chunk, $timeout) = @_;
    my $sock = $self->{sock};
    my $select = $self->{select};
    
    my (
	$device_IP, $device_ID, $ctxType, 
	$evtType, $procLvl, $message_ID
	 );
	
    $device_IP = $self->read_chunk($timeout)
		    or return $device_IP;
    $device_ID = $self->read_chunk($timeout)
		    or return $device_ID;

    $chunk->{'device_IP'} = $device_IP;
    $chunk->{'device_ID'} = $device_ID;
    
    return 0 unless $select->can_read($timeout);
    
    $sock->sysread($ctxType, 4);
    $sock->sysread($evtType, 4);
    $sock->sysread($procLvl, 4);
    $sock->sysread($message_ID, 4);
    
    $chunk->{'ctxType'} = unpack('L', $ctxType);
    $chunk->{'evtType'} = unpack('L', $evtType);
    $chunk->{'procLvl'} = unpack('L', $procLvl);
    $chunk->{'message_ID'} = unpack('L', $message_ID);
    
    return 1;
}


sub send_spec {
    
    my ($self, $chunk) = @_;
    my $sock = $self->{sock};
    
    $self->send_chunk($chunk->{'device_IP'});
    $self->send_chunk($chunk->{'device_ID'});
    
    $sock->syswrite(pack('L', $chunk->{'ctxType'}), 4);
    $sock->syswrite(pack('L', $chunk->{'evtType'}), 4);
    $sock->syswrite(pack('L', $chunk->{'procLvl'}), 4);
    $sock->syswrite(pack('L', $chunk->{'message_ID'}), 4);
}


sub send_addl {
    
    my ($self, $chunk) = @_;
    my $sock = $self->{sock};
    
    my $count = $chunk->{'attachments_count'};
    $sock->syswrite(pack('L', $count), 4);
    
    for(my $i = 0; $i < $count; ++$i)
    {
	$self->send_chunk($chunk->{description}[$i]);
	$self->send_chunk($chunk->{data}[$i]);
    }
    
}

sub read_addl {

    lock $mutex;

    my ($self, $chunk, $timeout) = @_;
    my $sock = $self->{sock};
    
    my $att_count;
    my $select = $self->{select};
    return 0 unless $select->can_read($timeout);
    
    $sock->sysread($att_count, 4);
    $att_count = unpack('L', $att_count);
    
    $chunk->{'attachments_count'} = $att_count;
	
    my @description = ();
    my @data = ();

    for( my $i = 0; $i < $att_count; ++$i )
    {
	$description[$i] =  $self->read_chunk($timeout)
		    or return $description[$i];
	$data[$i] = $self->read_chunk($timeout) 
		    or return $data[$i];
    }

    $chunk->{'description'} = \@description;
    $chunk->{'data'} = \@data;
    
    return 1;
}

sub read_data {
    
    lock $mutex;
    
    my ($self, $chunk, $timeout) = @_;
    my $sock = $self->{sock};
    my ( $spec, $data, $attachments);
    
    # TP1403: first read userid (4-bytes integer)
    my $userid = 0;
    $sock->sysread($userid, 4);
    $chunk->{'userid'} = unpack('L', $userid);
    
    $spec = $self->read_chunk($timeout) 
	    or return $spec;
    $data = $self->read_chunk($timeout)
	    or return $data;
	    
    if($chunk->{'length'} > ( $spec->{'length'} + $data->{'length'} + 24 ))
    {
	$attachments = $self->read_chunk($timeout)
		or return $attachments;
    }

    $chunk->{'spec'} = $spec;
    $chunk->{'data'} = $data;
    $chunk->{'attachments'} = $attachments;
    
    return 1;
}

sub send_data {
    
    my ($self, $chunk) = @_;
    my $sock = $self->{sock};
    
    # TP1403: Add userid into DATA chunk
    $sock->syswrite(pack('L', $chunk->{userid} || 0), 4);
    
    $self->send_chunk($chunk->{'spec'});
    $self->send_chunk($chunk->{'data'});
    $self->send_chunk($chunk->{'attachments'}) if defined $chunk->{'attachments'};
}

sub send_conn {
    
    my ($self, $chunk) = @_;
    my $sock = $self->{sock};
    
    $sock->syswrite(pack('L', $chunk->{auth_type}), 4);
    if($chunk->{auth_type} == AUTH_SESSION)
    {
	$self->send_chunk($chunk->{ext_sid});
    }
    if($chunk->{helo}) {
	$self->send_chunk($chunk->{helo});
    }
}


sub send_subs {

    my ($self, $chunk) = @_;
    my $sock = $self->{sock};
    
    $sock->syswrite(pack('L', $chunk->{'session_ID'}), 4);
    $self->send_chunk($chunk->{'spec'});
}

sub send_unsb {

    my ($self, $chunk) = @_;
    my $sock = $self->{sock};
    
    $sock->syswrite(pack('L', $chunk->{'session_ID'}), 4);
    $sock->syswrite(pack('L', $chunk->{'subscription_ID'}), 4);
}


sub send_list {

    my ($self, $chunk) = @_;
    my $sock = $self->{sock};
    
    $sock->syswrite(pack('L', $chunk->{'session_ID'}), 4);
}


sub read_uint32 {
    
    my ($self, $timeout) = @_;
    my $sock = $self->{sock};
    my $select= $self->{select};
    
    my ($bytes_total, $bytes_read) = (0, 0);
    my $var;
    
    while($bytes_total != 4)
    {
       if($select->can_read($timeout))
       {		
       	  $bytes_read = $sock->sysread($var, 4 - $bytes_total, $bytes_total);
       	  return undef unless $bytes_read;
       	  $bytes_total += $bytes_read;
       }
    }
    
    return $var;
    
}

sub read_chunk {
    
    lock $mutex;    

    my $self = shift;
    my $timeout = shift || 0;
    $timeout *= 0.001;
	
    my $header = $self->read_header($timeout);
    
    return $header unless $header;
        
    my %chunk;
    my $sign = $chunk{'sign'} = $header->{'sign'}; 
    $chunk{'length'} = $header->{'length'};
    $chunk{'crc'} = $header->{'crc'};
    
    if($chunk{'crc'} != compute_crc(unpack('L', $chunk{'sign'}), $chunk{'length'}) )
    {
        #warn("BAD CHUNK: $chunk{$sign}\n"); #DEBUG
	return 0;
    }
     
    my $rez;
    
    if   ($sign eq 'ACKW') { $rez = $self->read_ackw(\%chunk, $timeout); }
		
    elsif($sign eq 'TEXT') { $rez = $self->read_text(\%chunk, $timeout); }
	
    elsif($sign eq 'SPEC') { $rez = $self->read_spec(\%chunk, $timeout); }
	
    elsif($sign eq 'DATA') { $rez = $self->read_data(\%chunk, $timeout); }
	
    elsif($sign eq 'SLST') { $rez = $self->read_slst(\%chunk, $timeout); }
	
    elsif($sign eq 'ADDL') { $rez = $self->read_addl(\%chunk, $timeout); }
	
    else { $rez = $self->skip_chunk(\%chunk, $timeout); }
   
    return $rez unless $rez;
    
    return \%chunk;
}


sub get_chunk {
    
    lock $mutex;
    	
    my ($self, $sign, $timeout) = @_;

    my $chunk;
    my $size =  $#{$self->{chunk_queue}};
    for(my $i = 0; $i < $size; ++$i)
    {
	$chunk = $self->{chunk_queue}[$i];
	if( $chunk->{sign} eq $sign )
	{ 
	    splice @{$self->{chunk_queue}}, $i, 1;
	    return $chunk;
	}
    }
    
    
    $chunk = $self->read_chunk($timeout);
    
    return $chunk unless $chunk;
    return $chunk if $chunk->{sign} eq $sign;
    
    while(1) 
    {
	$chunk = $self->read_chunk($timeout);
	
	return undef unless defined $chunk;
	
	unless($chunk)
	{
	    next if defined $chunk;
	    
	    return undef unless scalar( @{$self->{chunk_queue}} );
	    
	    return 0;
	}

	if($chunk->{sign} ne $sign)
	{
	    push @{$self->{chunk_queue}}, $chunk;
	}
	else
	{
		return $chunk;
	}
    }

    return 0;

}


sub send_chunk {
    
    my ($self, $chunk) = @_;
    my $sock = $self->{sock};
    
    eval {
    
	my $packed_length = pack('L', $chunk->{length});
	my $sign = $chunk->{sign};
	
	$sock->syswrite($sign, 4);
	$sock->syswrite($packed_length, 4);
	$sock->syswrite($packed_length ^ $sign, 4);
	
	if  ($sign eq 'TEXT') { $self->send_text($chunk); }
	
	elsif($sign eq 'SPEC') { $self->send_spec($chunk); }
	
	elsif($sign eq 'DATA') { $self->send_data($chunk); }
	
	elsif($sign eq 'LIST') { $self->send_list($chunk); }
	    
	elsif($sign eq 'CONN') { $self->send_conn($chunk); }
	    
	elsif($sign eq 'SUBS') { $self->send_subs($chunk); }
	    
	elsif($sign eq 'UNSB') { $self->send_unsb($chunk); }
	
	elsif($sign eq 'ADDL') { $self->send_addl($chunk); }
	    
        else { return 0; }
	 
    };
    if($@  && $@ =~ /sigpipe/ )
    {
	return undef;
    }
    
    return 1;
    
}




sub skip_chunk {
    
   
    my ($self, $chunk) = @_;
    my $sock = $self->sock;
    
    my $raw_data;

    my $len = $sock->sysread($raw_data, $chunk->{length});	
    my $diff = $chunk->{length} - $len;
        
    if($diff != 0)
    {
	my $select = $self->{select};
	if($select->can_read(0.001))
	{
	    my $len2 = $sock->sysread($raw_data, $diff, $len);
	    return 0 if $len2 != $diff;
	    
	}
	else
	{
	    #extremely bad chunk :(
	    return 0;
	}
    }

    $chunk->{'raw'} = $raw_data;
    
    return 1;
}


sub read_header {

    lock $mutex;
    
    my ($self, $timeout) = @_;
    my $sock = $self->sock;
    
    my ( %header, $sign, $length, $crc );
    
    my $select = $self->{select};
    
    if($select->can_read($timeout))
    {
	$sign = $self->read_uint32($timeout) || return undef;
    }
    else
    {
	return 0;
    }
    
    #Now we should certainly read length and crc
    
    $length = $self->read_uint32($timeout) 
	    || return undef; #probably impossible, i hope...
	    
    $crc = $self->read_uint32($timeout)
	    || return undef;	
    
    
    
    $header{sign} = $sign;
    $header{length} = unpack('L', $length);
    $header{crc} = unpack('L', $crc);
    
    return \%header;
}


sub disconnect {

    my $self = shift;
    my $sock = $self->{sock};
    if( $sock )
    {
	eval
	{
	    $sock->shutdown(2);
	};
	return 0 if $@;
	
    }
    
    undef $self->{sock};
    return 1; 
}

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# 'Public' Interface		                           +
#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

sub connect {

    my ($self, $ext_sid, $helo) = @_;
    $helo = $self->{helo} unless defined $helo;
    $ext_sid = $self->{ext_sid} unless defined $ext_sid;
    
    my $sock = $self->sock;
    if($sock) 
    {
	$self->disconnect;
    }
    
    if( defined($self->host) && defined($self->port) ) 
    {			
	$sock = IO::Socket::INET->new(
		PeerAddr => $self->host,
		PeerPort => $self->port,
		Proto    => 'tcp',
		Timeout  => $self->conn_timeout	
	    );
	
	return undef unless $sock;
	
	$sock->autoflush(1);
	
	$self->{sock} = $sock;
	
	my $select = IO::Select->new($sock);
	$self->{select} = $select;
	
	my %conn_chunk;
	$conn_chunk{sign} = 'CONN';
	if($ext_sid) {
	    $conn_chunk{auth_type} = AUTH_SESSION;
	    $conn_chunk{ext_sid} = $self->build_text($ext_sid);
	    $conn_chunk{length} = CRC_LENGTH + 4 + $conn_chunk{ext_sid}->{length} + CHUNK_HEADER_LENGTH;
	} else {
	    $conn_chunk{auth_type} = AUTH_NONE;
	    $conn_chunk{length} = CRC_LENGTH + 4;
	}
	# Add HELO message
	if (length $helo) {
	    $conn_chunk{helo} = $self->build_text("[$$] ".$helo);
	    $conn_chunk{length} += $conn_chunk{helo}->{length} + CHUNK_HEADER_LENGTH;
	}
	

	$self->send_chunk(\%conn_chunk) || 
		return undef;
	my $ackw = $self->get_chunk('ACKW', $self->{conn_timeout}) || 
		return undef;
	
	return 0 if $ackw->{'result'} != 0;
	my $sid = $ackw->{'session_ID'};
	$self->{sid} = $sid;
	$self->{protocol_version} = $ackw->{protocol_version};
	
	if(defined $self->{callback})
	{
	    $self->{thread} = threads->create("thread_rcv", $self);
	    $self->{thread}->detach();
	}
	
	return $sid;
    }
    else
    {
	return undef;
    }
    
}

sub subscribe {

	lock($mutex);
    
	my ($self, $msg) = @_;
	
	my $devip = $msg->{'device_IP'} || "0.0.0.0";
	my $devid = $msg->{'device_ID'} || "0";
	my $message_ID;
	
	if(defined $msg->{message_ID})
	{
	    $message_ID = $msg->{message_ID};
	}
	else
	{
	    $message_ID = 0;
	}
	
	my (%subs, $spec);
	
	$spec = $self->build_spec($devip, $devid, $msg->{ctxType}, $msg->{evtType},
				    $msg->{procLvl}, $message_ID);
	
	#prepare SUBS shunk
	
	$subs{'sign'} = 'SUBS';
	$subs{'length'} = $spec->{'length'} + CHUNK_HEADER_LENGTH + CRC_LENGTH + 4;
	$subs{'session_ID'} = $self->{sid};
	$subs{'spec'} = $spec;
	
	$self->send_chunk(\%subs) || 
		return undef;
	my $ackw;
	$ackw = $self->get_chunk('ACKW', $self->{conn_timeout}) or 
		return $ackw;
	
	return 0 if $ackw->{'result'} != 0;
	
	my $subid = $ackw->{'subscription_ID'};
	$self->{subscriptions}->{$subid} = $msg;
	
	return $subid;
}


sub unsubscribe {
	
    lock($mutex);
	
    my $self = shift;
    my $subscription_ID;
	
    if( ref($_[0]) eq 'HASH' )
    {
    	my $msg = shift;
        while( my ($key, $val) = each(%{$self->{subscriptions}}) )
        {
	    if( subscmp($val, $msg) == 0)
    	    {
    		$subscription_ID = $key;
		delete $self->{subscriptions}->{$key};
	    }
	}
	
    }
    else
    {	
    	$subscription_ID = shift;
    } 

					
    my %unsb;
    
    $unsb{'sign'} = 'UNSB';
    $unsb{'length'} = CRC_LENGTH + 8;
    $unsb{'session_ID'} = $self->{sid};
    $unsb{'subscription_ID'} = $subscription_ID;
	
    $self->send_chunk(\%unsb) ||
	    return undef;
	
    my $ackw = $self->get_chunk('ACKW', $self->{conn_timeout}) ||
	    return undef;
	
    return 0 if $ackw->{'result'} != 0;
    return $ackw->{'subscription_ID'};
}


sub send {

	my ($self, $msg) = @_;
	
    	local $SIG{PIPE} = sub { die "sigpipe"; };

	my $devip = $msg->{device_IP} || "0.0.0.0";
	my $devid = $msg->{device_ID} || "0";
	my $tdata = $msg->{data};
        
	my $message_ID;
	if(defined $msg->{message_ID})
	{
	    $message_ID = $msg->{message_ID};
	}
	else
	{
	    $message_ID = 0;
	}
	
	my  (
		$deviplen, $devidlen, $datalen
	    ) =  
	    (
		length($devip), length($devid), length($tdata)
	    );
       
	my (%chdata, $spec, $device_IP, $device_ID, $data, %attachments);
	
	my ($chdata_length, $spec_length, $devip_length, 
	    $devid_length, $data_length);
	    
	my $att_count;
	
	if(defined($msg->{'att_description'}) && defined($msg->{'att_data'}))
	{
	    $att_count = ($#{$msg->{att_data}} < $#{$msg->{att_description}})?
			    ($#{$msg->{att_data}}+1):($#{$msg->{att_description}}+1);
	}
	
	if($att_count) #prepare attachments
	{
	    my (@ch_description, @ch_data);
	    my $att_length = 8;
	
	    #$#ch_description = $#ch_data = $att_count;
			    
	    for(my $i = 0; $i < $att_count; ++$i)
	    {
		my ($description, $data);
		my ($descr_len, $data_len);
		$descr_len = length($msg->{att_description}[$i]);
		$data_len = length($msg->{att_data}[$i]);
		
		$description = $self->build_text($msg->{att_description}[$i]);
		$data = $self->build_text($msg->{att_data}[$i]);
		$att_length += ($description->{'length'} + $data->{'length'} + 
				2 * CHUNK_HEADER_LENGTH);
		$ch_description[$i] = $description;
		$ch_data[$i] = $data;
	    }
	
	    $attachments{'sign'} = 'ADDL';
	    $attachments{'length'} = $att_length;
	    $attachments{'attachments_count'} = $att_count;
	    $attachments{'description'} = \@ch_description;
	    $attachments{'data'} = \@ch_data;
	}
	
    
	$data = $self->build_text($tdata);
	
	$spec = $self->build_spec($devip, $devid, $msg->{ctxType}, $msg->{evtType},
				    $msg->{procLvl}, $message_ID);
	
	#prepare DATA chunk
	$chdata{'sign'} = 'DATA';
	$chdata{'userid'} = 0;
	$chdata{'length'} = $spec->{'length'} + $data->{'length'} + 
			    2 * CHUNK_HEADER_LENGTH + 8; # TP1403: add 4-bytes field for 'userid'
	
	$chdata{'length'} += $attachments{'length'} + CHUNK_HEADER_LENGTH
	     if $att_count;
	
	$chdata{'spec'} = $spec;
	$chdata{'data'} = $data;
	$chdata{'attachments'} = \%attachments if $att_count;
	
	return $self->send_chunk( \%chdata );
}


sub quick_send {

    my ($self, $msg) = @_;
    my $sock = $self->{sock};

    my $devip = $msg->{device_IP} || "0.0.0.0";
    my $devid = $msg->{device_ID} || "0";
    my $data  = $msg->{data};
    
    my $message_ID;
    if(defined $msg->{message_ID})
    {
        $message_ID = $msg->{message_ID};
    }
    else
    {
        $message_ID = 0;
    }
    
    my  ($deviplen, $devidlen, $datalen) = 
       ( length($devip), length($devid), length($data) );
    
    
    my $crc;
    my ($chdata_length, $spec_length, $devip_length, 
	$devid_length, $data_length);
	
    my ($has_att, $att_count, $att_length) = (0, 0, 0);
    
    $devip_length = $deviplen + CRC_LENGTH + 4;
    $devid_length = $devidlen + CRC_LENGTH + 4;
    $data_length = $datalen + CRC_LENGTH + 4;
    
    
    if(defined($msg->{att_description}) && defined($msg->{att_data}))
    {
	$att_count = ($#{$msg->{att_data}} < $#{$msg->{att_description}})?
			    ($#{$msg->{att_data}}+1):($#{$msg->{att_description}}+1);
    }
    
    if($att_count)
    {
	$has_att = 1;
	$att_length = 8;
	
	for(my $i = 0; $i < $att_count; ++$i)
	{
	    my ($description, $data);
	    my ($descr_len, $data_len);
	    $descr_len = length($msg->{att_description}[$i]);
	    $data_len = length($msg->{att_data}[$i]);
		
	    $att_length += $descr_len + $data_len + 2 * CRC_LENGTH + 8 +  
	    				2 * CHUNK_HEADER_LENGTH;
	}
    }
    
    
    $spec_length = $devip_length + $devid_length + 
		      2 * CHUNK_HEADER_LENGTH + CRC_LENGTH + 16;
    $chdata_length = $spec_length + $data_length + 2 * CHUNK_HEADER_LENGTH + 4 +
			$has_att * ($att_length + CHUNK_HEADER_LENGTH);
			    
    $sock->syswrite('DATA', 4);
    $chdata_length = pack('L', $chdata_length);
    $sock->syswrite($chdata_length, 4);
    $sock->syswrite(('DATA' ^ $chdata_length), 4);
    
    # TP1403: Add 'userid' to chunk. Set 0 in 'send'
    $sock->syswrite(pack('L', 0), 4);
    
    $sock->syswrite('SPEC', 4);
    $spec_length = pack('L', $spec_length);
    $sock->syswrite($spec_length, 4);
    $sock->syswrite( ('SPEC' ^ $spec_length), 4 );
    
    $sock->syswrite('TEXT', 4);
    $devip_length = pack('L', $devip_length);
    $sock->syswrite($devip_length, 4);
    $sock->syswrite(('TEXT' ^ $devip_length), 4);
    
    $sock->syswrite(pack('L', $deviplen), 4);
    $sock->syswrite($devip, $deviplen);
    
    $sock->syswrite('TEXT', 4);
    $devid_length = pack('L', $devid_length);
    $sock->syswrite($devid_length, 4);
    $sock->syswrite(('TEXT' ^ $devid_length), 4);
    
    $sock->syswrite(pack('L', $devidlen), 4);
    $sock->syswrite($devid, $devidlen);
    
    $sock->syswrite(pack('L', $msg->{ctxType}), 4);
    $sock->syswrite(pack('L', $msg->{evtType}), 4);
    $sock->syswrite(pack('L', $msg->{procLvl}), 4);
    $sock->syswrite(pack('L', $message_ID), 4);
    
    
    $sock->syswrite('TEXT', 4);
    $data_length = pack('L', $data_length);
    $sock->syswrite($data_length, 4);
    $sock->syswrite(('TEXT' ^ $data_length), 4);
    
    $sock->syswrite(pack('L', $datalen), 4);
    $sock->syswrite($data, $datalen);
    
    #send attachments

    if($att_count)
    {
	
	$sock->syswrite('ADDL', 4);
	$att_length = pack('L', $att_length);
	$sock->syswrite($att_length, 4);
	$sock->syswrite('ADDL' ^ $att_length, 4);
	$sock->syswrite(pack('L', $att_count), 4);
	
	for(my $i = 0; $i < $att_count; ++$i)
	{
	    $sock->syswrite('TEXT');
	    my $descr_len = length($msg->{att_description}[$i]);
	    my $packed_len = pack('L', $descr_len + CRC_LENGTH + 4);
	    $sock->syswrite($packed_len, 4);
	    $sock->syswrite('TEXT' ^ $packed_len, 4);
	    $sock->syswrite(pack('L', $descr_len), 4);
	    $sock->syswrite($msg->{att_description}[$i], $descr_len);
	    
	    $sock->syswrite('TEXT');
	    my $data_len = length($msg->{att_data}[$i]);
	    $packed_len = pack('L', $data_len + CRC_LENGTH + 4);
	    $sock->syswrite($packed_len, 4);
	    $sock->syswrite('TEXT' ^ $packed_len, 4);
	    $sock->syswrite(pack('L', $data_len), 4);
	    $sock->syswrite($msg->{att_data}[$i], $data_len);
	}
	
    }
    
}

sub recv
{
	lock $mutex;
    
	my ($self, $timeout) = @_;
	my $data = $self->get_chunk('DATA',$timeout);
	
	unless($data)
	{
	    return $data;
	}	
	else
	{
	    if($data->{sign} ne 'DATA')
	    {
		#Carp::carp("Not a DATA chunk got in recv!\n");
		return undef;
	    }
	
	    my %msg = ();
	    
	    
	    $msg{device_IP} = $data->{'spec'}->{'device_IP'}->{'text'};
	    $msg{device_ID} = $data->{'spec'}->{'device_ID'}->{'text'};
	    $msg{ctxType} = $data->{'spec'}->{'ctxType'};
	    $msg{evtType} = $data->{'spec'}->{'evtType'};
	    $msg{procLvl} = $data->{'spec'}->{'procLvl'};
	    $msg{message_ID} = $data->{'spec'}->{'message_ID'};
	    $msg{data} = $data->{'data'}->{'text'};
	    $msg{userid} = $data->{userid};
	    if(defined($data->{'attachments'})) 
	    {
		my $att_count = $data->{'attachments'}->{'attachments_count'};
		my @ch_description = @{$data->{'attachments'}->{'description'}};
		my @ch_data = @{$data->{'attachments'}->{'data'}};
		my (@att_description, @att_data);
		
		for(my $i = 0; $i < $att_count; ++$i)
		{
		    $att_description[$i] = $ch_description[$i]->{'text'};
		    $att_data[$i] = $ch_data[$i]->{'text'};
		}
		
		$msg{'att_description'} = \@att_description;
		$msg{'att_data'} = \@att_data;
		
	    }
	    return \%msg;
	}

}

sub quick_recv {
    
    lock $mutex;
    my ($self, $timeout) = @_;
    
    my $sock = $self->{sock};
    my $select = IO::Select->new($sock);
    
    my $msg = {};
    
    return 0 unless $select->can_read($timeout);
    
    my ($sign, $length, $crc, $text_length);
    my ($chdata_length, $spec_length, $data_length);
    my $userid;
    
    $sock->sysread($sign, 4);   #DATA header
    $sock->sysread($chdata_length, 4);
    $sock->sysread($crc, 4);
    
    # TP1403: read userid
    $sock->sysread($userid, 4);
    $msg->{userid} = $userid;
    
    $sock->sysread($sign, 4);   #SPEC header
    $sock->sysread($spec_length, 4);
    $sock->sysread($crc, 4);
    
    $sock->sysread($sign, 4);   #device_IP header
    $sock->sysread($length, 4);
    $sock->sysread($crc, 4);
    
    $sock->sysread($text_length, 4); #device_IP
    $sock->sysread($msg->{device_IP}, unpack('L', $text_length));
    
    $sock->sysread($sign, 4);   #device_ID header
    $sock->sysread($length, 4);
    $sock->sysread($crc, 4);
    
    $sock->sysread($text_length, 4); #device_ID
    $sock->sysread($msg->{device_ID}, unpack('L', $text_length));
    
    
    $sock->sysread($crc, 4); #Context class
    $msg->{ctxType} = unpack('L', $crc);
    
    $sock->sysread($crc, 4); #Event class
    $msg->{evtType} = unpack('L', $crc);
    
    $sock->sysread($crc, 4); #Data Processing level
    $msg->{procLvl} = unpack('L', $crc);
    
    $sock->sysread($crc, 4); #Message ID
    $msg->{message_ID} = unpack('L', $crc);
    
    
    $sock->sysread($sign, 4);   #data header
    $sock->sysread($data_length, 4);
    $sock->sysread($crc, 4);
    
    $sock->sysread($text_length, 4); #data
    $sock->sysread($msg->{data}, unpack('L', $text_length));
    
    if(
	unpack('L', $chdata_length) > 
	(unpack('L', $spec_length) + unpack('L', $data_length) + 24)
      )
    {
	my $att_count = 0;
	
	$sock->sysread($sign, 4);   #ADDL header
	$sock->sysread($length, 4);
	$sock->sysread($crc, 4);
	
	$sock->sysread($att_count, 4);
	$att_count = unpack('L', $att_count);
    
	my @description = ();
	my @data = ();

	for( my $i = 0; $i < $att_count; ++$i )
	{
	    
	    $sock->sysread($sign, 4);   #attachment description header
	    $sock->sysread($length, 4);
	    $sock->sysread($crc, 4);
	    $sock->sysread($text_length, 4); #attachment description
	    $sock->sysread($description[$i], unpack('L', $text_length));
	    
	    $sock->sysread($sign, 4);   #attachment data header
	    $sock->sysread($length, 4);
	    $sock->sysread($crc, 4);
	    $sock->sysread($text_length, 4); #attachment data
	    $sock->sysread($data[$i], unpack('L', $text_length));
	
	}

	$msg->{description} = \@description;
	$msg->{data} = \@data;
    }
    
    return $msg;
}
                                                               


sub subscriptionList {

    lock($mutex);
    
    my $self = shift;
    my $session_ID = shift || $self->{sid};
    
    my %list;
    $list{'sign'} = 'LIST';
    $list{'length'} = CRC_LENGTH + 4;
    $list{'session_ID'} = $session_ID;
    
    $self->send_chunk(\%list) || 
		return ();
    
    my $ackw = $self->get_chunk('ACKW', $self->{conn_timeout}) ||
	return ();
	
    return () if $ackw->{'result'} != 0;
        my $count = $ackw->{'subscriptions'}->{'count'};

    my ($items, $subs) = ( $ackw->{'subscriptions'}->{'item'}, 
			   $ackw->{'subscriptions'}->{'subscriber'} );
     
    if( !defined($items) || !defined($subs) )
    {
	return ();
    }   
    my @subs = @$items;
    my @subscribers = @$subs; ;
    my @subscriptions = ();
    
    for( my $i = 0; $i < $count; ++$i )
    {
	my $spec = $subs[$i];
	
	$subscriptions[$i]{device_IP}  = $spec->{'device_IP'}->{'text'};
	$subscriptions[$i]{device_ID}  = $spec->{'device_ID'}->{'text'};
	$subscriptions[$i]{ctxType}    = $spec->{'ctxType'};
	$subscriptions[$i]{evtType}    = $spec->{'evtType'};
	$subscriptions[$i]{procLvl}    = $spec->{'procLvl'};
	$subscriptions[$i]{message_ID} = $spec->{'message_ID'};
	$subscriptions[$i]{subscriber} = $subscribers[$i]->{'text'};
    }
        
    return @subscriptions;
    
}


sub thread_rcv {
    
    my $self = shift;
    
    while(1)
    {
	my $msg = $self->recv(10);
	unless(defined($msg))
	{
	    if(defined $self->{clbk_disconnect})
	    {
		$self->{clbk_disconnect}->($self);
	    }
	    else
	    {
		$self->disconnect;
	    }
	    
	    return 1;
	}
	if($msg)
	{
	    $self->{callback}->($msg);
	}
    }
    
    return 0;
    
} 


1;

__END__

=head1 NAME

NextCAM::Messenger - Perl client for NextCAM message server

=head1 SYNOPSIS

  use NextCAM::Messenger;
  
  $messenger = NextCAM::Messenger->new( %options );
  This class method constructs a new C<NextCAM::Messenger> object and
  returns a reference to it.

  Key/value pair arguments may be provided to set up the initial state
  of the messenger.

   KEY                     DEFAULT
   -----------             --------------------
   host                    undef
   port                    undef
   callback                undef
   clbk_disconnect         undef
   conn_timeout            1000
   autoconnect             1

  # so now you can subscribe/unsubscribe, send/recv messages
  
  my %msg = (
	device_IP => 1,
	device_ID => 2,
	ctxType   => 3,
	evtType   => 4,
	procLvl   => 5
  );
  
  # subscribe
  my $subscription_ID = $messenger->subscribe(\%msg);
  if (!$subscription_ID)
  {
	print "Subscribe failed\n";
  }
  
  # unsubscribe
  if (!$messenger->unsubscribe(\%msg))
  {
	print "Unsubscribe failed\n";
  }
  
  # or you may use subscription id to unsubscribe:
  $messenger->unsubscribe($subscription_ID);
  
  # get list of subscriptions
  my @list = $messenger->subscriptionList;
  
  if( @list )
  {
    for(my $i = 0; $i <= $#list; ++$i)
    {
	print "Subscription: " .  $list[$i]{device_IP} . "\n";
    }
  }
  
  # send messages (with or w/o attachments)
  
  #prepare attachments if necessary
  my @att_description = ( "att_description0", "att_description1" );
  my @att_data = ( "att_data0", "att_data1" );
  
  my $msg = {
    ctxType         => Context_Class_NODE,
    evtType         => Event_Class_CMD,
    procLvl         => DATA_Processing_Level_LVL1,
    message_ID      => 1,
    data            => "message data"
    att_description => \@att_description,
    att_data        => \@att_data
    }
    
  $messenger->send($msg);
  
  # receive messages
  # Example of blocking mode
  while (1)
  {
	my $msg = $messenger->recv(1000);
	
	if ($msg)
	{
		print $msg->{device_IP} . "." .
			  $msg->{device_ID} . "." .
			  $msg->{ctxType} . "." .
			  $msg->{evtType} . " " .
			  $msg->{procLvl} . " " .
			  $msg->{message_ID} . "\n";
		print "Body: " . $msg->{"data"} . "\n\n";
	}
	
   }

=head1 ABSTRACT

  

=head1 DESCRIPTION




=head2 EXPORT

None by default.


=head1 AUTHOR

Alexey Tsibulnik <lt>crash@code-it-now.com<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006 by VideoNEXT


=cut
