#!/usr/bin/perl
#  $Id: Init.pm 7993 2007-05-31 19:33:22Z teetov $
# -----------------------------------------------------------------------------
#  Perl module provides method for conf access
#  from other NEXTcam engines
# -----------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: Teetov
#  QA by:  Christopher C Gettings
#  Copyright: videoNEXT LLC
# -----------------------------------------------------------------------------

package NextCAM::Conf;

use 5.008;
use strict;
use warnings;
#use Data::Dumper;
use Master::Conf;

require Exporter;
our @ISA = qw(Exporter);

# This allows declaration       use Node::Conf ':all';
our %EXPORT_TAGS = ( 'all' => [ qw( GetCfgs GetDefaultCfg WriteCfg DeviceList CameraList GetIdentity) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw( GetCfgs );

our $VERSION = '1.01';


# Preloaded methods go here.

my $APL=$ENV{APL};					# directory structure root
my $APL_CONF=$ENV{APL_CONF} || "$APL/var/conf";		# conf-files are here - will be depreciated!!!

# ---------------------------------------------- get templates into memory -----
# print STDERR "parsing templates\n";
my %templ;
my @flds = ( 'PAGENAME', 'PARAMETER', 'DESCRIPTION', 'ITEMTYPE', 'DEFAULT', 'VALUELIST', 'REGEXP', 'ERRMSG' );
for my $template ( 'CAMERA', 'MACRO', 'NEXTCAM', 'RELAY', 'SENSOR', 'SWITCH', 'TERM', 'JOYSTICK' ) {
    open F, "$APL/conf/etc/${template}.cfg" or next;
    #print STDERR "TEMPLATE: $APL/conf/etc/${template}.cfg \n";
    while($_=<F>) {
        chomp;
        next if /^#/;
        my @a = split(/:::/);
        next if $#a < 3;
        my $s = '';
        for(my $i=0; $i <= $#a; $i++ ) {
            next if not $a[$i];
            $templ{$template}{$a[1]}{$flds[$i]} = $a[$i];
            #print STDERR "$template : $a[1]}{$flds[$i] => $a[$i]\n";
            last if $i > 7;
        }
    }
    close F;
}

# ---------------------------------------------------------------- GetDefaultCfg
#  return default configuration for device
#  usage:  
#    my $newcam=GetDefaultCfg('CAMERA');
#    $newcam->{DEVIP}='123.123.123.123';
#  return undef if cannot read template

sub GetDefaultCfg {
	my ($devtype,$model,$modelid)=@_;
	return undef unless $devtype;
	my %conf;
	my $delimiter = '~';
	if ($devtype eq 'identity') {
        my $template = "$APL/conf/etc/identity/etc/edit.cfg";
        open(TEMPL, $template) || return undef;
        %conf = map {(split(/$delimiter/))[1,4]} grep{!/^\s*#/} <TEMPL>;
        close TEMPL;
	}
	elsif ($devtype =~ /^(schedule|posture|timecard)$/) {
        my $template = "$APL/conf/etc/schedule/$devtype/default/etc/edit.cfg";
        open(TEMPL, $template) || return undef;
        %conf = map {(split(/$delimiter/))[1,4]} grep{!/^\s*#/} <TEMPL>;
        close TEMPL;

	}
	elsif (-f "$APL/conf/etc/$devtype.cfg") {
		my $template = "$APL/conf/etc/$devtype.cfg";
		open(TEMPL, $template) || return undef;
		%conf = map {(split(/$delimiter/))[1,4]} grep{!/^\s*#/} <TEMPL>;
		close TEMPL;
	}
	elsif (-d "$APL/conf/etc/device/$devtype" or $devtype eq 'CAMERA') {
		my $APL_MOD=$ENV{APL_MOD} || 'SKM';
		$devtype='camera' if $devtype eq 'CAMERA';
		$model = 'default' unless $model;
		my $root="$APL/conf/etc/device/$devtype";
		my @tmpls=("default/etc/edit.cfg","default/etc/edit.$APL_MOD.cfg");
		push @tmpls,"$model/etc/edit.cfg","$model/etc/edit.$APL_MOD.cfg" if $model ne 'default';
		foreach my $tmpl (@tmpls) {
			open(TEMPL,"$root/$tmpl") or next;
			if(defined $model and $tmpl=~/$model/) {
				my ($general_section,$our_section)=(1,0);
				while(my $line = <TEMPL>) {
					chomp $line;
					$line=~s/\x{0D}$//; # Remove carriage return if any
					next unless $line;
					next if $line=~/^\s*#/;
					if($line=~/^MODELID:\s*(.+)$/) {
						last unless $modelid; # Nothing to do
						my $model_list=$1;
						$general_section=0;
						$our_section=0;
						foreach (split /,/,$model_list) {
							s/^\s+//; # trim
							s/\s+$//;
							$our_section=1,last if $_=~/^$modelid$/i;
						}
						next;
					}
					if($general_section or $our_section) {
						my ($attr,$val)=(split(/$delimiter/,$line))[1,4];
						next unless defined $attr;
						$conf{$attr}=$val;
					}
				}
			}
			else {
				%conf=(%conf,map {(split(/$delimiter/))[1,4]} grep{!/^\s*#/} <TEMPL>);
			}
			close TEMPL;
		}
	}
	else {
	    return undef;
	}
	return \%conf;
}

sub WriteCfg {
#---==========-------------------------------------------------
# write conf to $APL/var/conf/conf/devid.conf
# attention! die inside. should be used with eval{block}
#--------------------------------------------------------------
 my $ref=shift;
 my $dev=$ref->{DEVID};
 my $CONFDIR="$APL_CONF/conf";
 die ("WriteCfg: unexpected device id=$dev\n") if $dev!~/^[anjvwsr]?\d+/;
 unlink "$CONFDIR/$dev.conf.tmp" if -f  "$CONFDIR/$dev.conf.tmp";
 open(CF,">$CONFDIR/$dev.conf.tmp") || die "WriteCfg: cannot create $CONFDIR/$dev.conf.tmp\n";
 print CF "$_=$ref->{$_}\n" foreach( sort keys %$ref);
 close CF;
 unlink "$CONFDIR/$dev.conf" if -f  "$CONFDIR/$dev.conf";
 rename "$CONFDIR/$dev.conf.tmp",   "$CONFDIR/$dev.conf";
}

# ---------------------------------------------------------------- GetCfgs -----
#  Return hash with device' configs.
#  On default, return all devices, but you can specify filter criteria, like:
#  %my_cfgs = GetCfgs( ('CAMERAMODEL'=>'Axis','DEVIP'=>'\d+\.\d+\.\d+\.\d+') );
#
sub GetCfgs {
    my (%filter)=@_;
    opendir(CFGDIR,$APL_CONF);   
    opendir(CFGTMP,"$APL_CONF/conf");
    opendir(CFGAV,"$APL_CONF/av");
    my (%devs,$dev,$cfg,$ignore);
    while (defined($dev = readdir(CFGDIR)) || defined($dev = readdir(CFGTMP)) || defined($dev = readdir(CFGAV))) {
        next if $dev=~/^\./;
        next if $dev=~/^asrv/;
        next if $dev=~/^n\d+/; # AT: substituted by NodeList
        next if $dev=~/^(node|master)$/;
        my $file="$APL_CONF/conf/$dev";       
        if(open(CFG,$file)) {
            $dev =~ /^(\w?\d+).conf/; 
            $dev = $1;
            $cfg=\%{{map {/(\w+)=(.*)/} grep {/^\w+=.*$/} <CFG>}};
            close(CFG);
        } else {
            $file = "$APL_CONF/$dev/conf";
 #           $log->error("Unable open for reading: [$APL_CONF/conf/$dev]");
            if(open(CFG,$file)) {
              $cfg=\%{{map {/(\w+)=(.*)/} grep {/^\w+=.*$/} <CFG>}};
              close(CFG);              
            }
            else
            {
              $file = "$APL_CONF/av/$dev/conf";
              if (open(CFG,$file)) {
                $cfg=\%{{map {/(\w+)=(.*)/} grep {/^\w+=.*$/} <CFG>}};
                close(CFG);
              }
              else 
              {
                next;
              }
            }
        } # if(open(CFG,$file))
        
        $ignore = 0;
        foreach(keys %filter) {
            $ignore = 1, last if not defined($cfg->{$_}) or not defined($filter{$_});
            $ignore = 1, last if not ( $cfg->{$_} eq $filter{$_} );
        }
        next if $ignore;
        foreach (keys %{$cfg}) { $devs{$dev}{$_}=$cfg->{$_}; }
        # here we check if all the fields are populated and add from templates as necessary
        next if $ENV{APL_MOD} eq 'CIRRUS';             # AT: do not load defaults from templates for CIRRUS
        next if not defined $devs{$dev}{DEVICETYPE};   # device MUST have DEVICETYPE attribute, but if it does not...
        foreach my $tplkey ( keys %{$templ{ $devs{$dev}{DEVICETYPE} }} ) {
            next if exists $devs{$dev}{$tplkey};
            $devs{$dev}{$tplkey} = $templ{$devs{$dev}{DEVICETYPE}}{$tplkey}{DEFAULT} || '';
            # print STDERR "$tplkey => $devs{$dev}{$tplkey}\n";
        }
    } # while (defined($file = readdir(CFGDIR)))
    #---------------------------------------------- substitution for UNI nodes
    my $nlist=NodeList;
    foreach my $uni(keys %$nlist) {
       my $name=$nlist->{$uni}->{FQDN};
       $name='MASTER' if $name eq 's_master';   # hardcoded MASTER is required by AdminGUI
       $name=$1 if $name ne $nlist->{$uni}->{IP} and $name=~/^(\w+)/;
       my %fake=(DEVICETYPE=>'NEXTCAM',LOCATION=>'none',WAN_HTTP_PORT=>80,USER_EDITABLE=>'no',
                 NAME=>$name,LAN_IP=>$nlist->{$uni}->{IP},FQDN=>$nlist->{$uni}->{FQDN},
                 WAN_IP=>$nlist->{$uni}->{IP},DEVID=>$uni,WAN_RTSP_PORT=>$nlist->{$uni}->{RTSP_PORT});
       my $ignore = 0;
       foreach(keys %filter) {
           $ignore = 1, last if not defined($fake{$_}) or not defined($filter{$_});
           $ignore = 1, last if not ( $fake{$_} eq $filter{$_} );
       }
       next if $ignore; 
       $devs{$uni}=\%fake;
    }
    #---------------------------------------------- end of UNI nodes
    closedir(CFGDIR);
    return %devs;
} # sub GetCfgs

# ---------------------------------------------------------------- DeviceList -----
#  List all available device IDs
#
sub DeviceList {
    my $chartype = shift;
    $chartype = '\w?' unless defined $chartype;
    my @devs;
    if (opendir DH, $APL_CONF) {
	@devs = grep {/^$chartype(\d+)$/} readdir DH;
	closedir DH;
    }
    return wantarray ? @devs : [@devs];
}

sub CameraList {
    return DeviceList '';
}

# ------------------------------------------------------------ GetIdentity --------
# Read Identity attributes from flat file
#
sub GetIdentity {
        return undef unless open FH, "$APL_CONF/identity.conf";
        my %identity = map {/^(\w+)=(.*)$/} grep {/^\w+=/} <FH>;
        close FH;
        return \%identity;
}

1;
__END__

