#!/usr/bin/perl
#  $Id: PTZ_messenger.pm 29829 2013-11-18 22:13:23Z teetov $
# -----------------------------------------------------------------------------
#  Proxy from PTZ client to PTZ server.
# -----------------------------------------------------------------------------
#  Author: Starostin Andrey
#  Edited by:
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT LLC, 2005
# -----------------------------------------------------------------------------
package NextCAM::Apache::PTZ_messenger;

use strict;
use NextCAM::Messenger;
use NextCAM::WEBSession;
use SKM::DB;
use Data::Dumper;
use URI::Escape;
use JSON;
#use Apache2::Const -compile => qw(:http);

my $MBUS_PORT=10000;
my $MBUS_CONN_TIMEOUT=1000;

# Lock manage
my $LOCKDIR = "/opt/sarch/var/ptz/locks";
my $SESSDIR = $^O=~/darwin/i ? '/tmp' : '/var/lib/php/session';
my $LOCK_TIMEOUT = 30;
my $CFGREAD_TIMEOUT = 30;
my %PRI_MAP = (
	low    => 0,
	normal => 5,
	high   => 9
);

my $messenger = undef;
my $dbm;	# DB handle
my %dbs;	# prepared statements
my $last_cfgread_time = 0;

# Actions
my $ACT_LOCK 	 = "lock";
my $ACT_OVERRIDE = "override";
my $ACT_RELEASE  = "release";
my $ACT_PASS	 = "pass";

# Audit
my $AUDIT_CATEGORY = 9;

sub handler : method
{
	my ($class, $r) = @_;
	my $retval = Apache2::Const::OK;
	my %resp = ();
	# Read params from DB
	readcfg();

	if ($ENV{REQUEST_METHOD} eq "GET")
	{
		my %args;
		my $query_str = $ENV{'QUERY_STRING'};
		if ($query_str ne '')
		{
			foreach(split /&/,$query_str){ $args{uc($1)} =
				uri_unescape($2) if /(\S+?)=(.*)$/; }

			# Check client session/IP
			#my ($sid, $userId, $userName, $isAdmin, $trustedIP) =
			my $sinfo = WEBSessionAttr($args{SID},1,1);
			if(not $sinfo)
			{
				#$retval = Apache2::Const::HTTP_FORBIDDEN;
				$resp{error} = "Unauthorized access";
				$resp{code}  = 401;
			}
			elsif (defined $args{DATA} and $args{DATA}=~/(dev|target)=/i)
			{
				my $sid =  $sinfo->{sid};
				my $userId = $sinfo->{userid};
				my $userName = $sinfo->{username};

				# Process PTZ locks
				my ($ok,$err,$lockinfo,$action) = (1,"",undef,undef);
				my $prevLock;

				my ($priority) = $args{DATA}=~/priority=(\w+)/i;
				$priority = 1 unless $priority;
				# Handle legacy values for priority
				$priority=$PRI_MAP{lc($priority)}
				if $priority=~/^(low|normal|high)$/i;

				($err,$lockinfo,$action,$prevLock) = ptz_lock_manage(
					$args{DATA},
					$sid,
					$userId,
					$userName,
					$priority
				);

				if (defined $lockinfo) {
					if ($lockinfo->{SID} == $sid) {
						$lockinfo->{allow_override} = 'sameuser';
					}
					else {
						$lockinfo->{allow_override} =
						$priority >= $lockinfo->{PRIORITY} ? 'yes' : 'no';
					}

					# Remove all unnecessary data from response
					delete $lockinfo->{$_}
						foreach grep {$_ ne 'allow_override'} keys %$lockinfo;

					$resp{lockinfo} = $lockinfo;
				}

				if ($err) {
					$resp{error} = $err;
					$resp{code} = 400;
				}
				elsif ($action) {
			
					# TP702. GEO-PTZ commands exposed to HTTP PTZ API
					# Ex: do.ptz?target=target_123&mode=abs&lat=...&long=...&elev=...&slew=objid1,objid2,...
					if ($args{DATA}=~/\?target=/i)
					{
						my ($query) = $args{DATA} =~ /\?(target=.+)<\/PTZ_Command>/i;
						my %params;

						foreach (split /&/,$query) {
							$params{uc($1)} = $2
							if /(\S+?)=(.*)$/;
						}
						my ($target,$lat,$long,$elev,$fov,$slew) =
						@params{'TARGET','LAT','LONG','ELEV','FOV','SLEW'};
						# Check for required params
						if ( !defined($target) ||
							!defined($lat) ||
							!defined($long) ||
							!defined($elev))
						{
		    					$resp{error} = "Bad request";
							$resp{code}  = 400;
						}
						else
						{
							my $tmpl =
								'<PTZ CMD="AutoSlew" CASE="DirectGeoCoordinates">
								<TARGET ID="%s" LAT="%s" LONG="%s" ELEV="%s"/>
								<AUTO_SLEW METHOD="ALL"%s>
									%s
								</AUTO_SLEW>
								</PTZ>';
		
							my $xmlcams = "";
							my $request = "";
							my $xmlfov  = "";
		
							$xmlcams .= "<CAM OBJID=\"$_\"/>" foreach split(/,/,$slew);
							$xmlfov   = " TARGET_FOV=\"${fov}%\"" if defined $fov;
							$request  = sprintf(
								$tmpl,
								$target,$lat,$long,$elev,
								$xmlfov,$xmlcams
							);
		
							# First, send request to GWM daemon
							my $hdr = {
								device_ID => "0",
								devIP     => "0.0.0.0",
								ctxType   => 16,
								evtType   => 1,
								procLvl   => 2,
							};
							my ($ok,$err) = sendToMBus($request,undef,$hdr);
							# Next, send command to repositionClient if any camera objids provided
							if ($slew) {
								$hdr->{procLvl} = 1;
								my $attach = [
									"sid=$sid",
									"userid=$userId",
									"username=$userName",
									"action=$action"
								];
								($ok,$err) = sendToMBus($request,$attach,$hdr);
							}
							if (not $ok) {
								$resp{error} = "Internal error: $err";
								$resp{code}  = 500;
							}
						}
					}
					else # if dev=
					{
						# Finally forward command to PTZ_server
						# Pass additional information inside attachment
						my $attach = [
							"sid=$sid",
						    	"userid=$userId",
							"username=$userName",
							"action=$action"
						];
						($ok,$err) = sendToMBus($args{DATA},$attach);
						if (not $ok) {
							$resp{error} = "Internal error: $err";
							$resp{code}  = 500;
						}

						# Do audit
						my ($objid) = $args{DATA}=~/dev=(.*?)&/;
						insertAuditData(
							$sid,
							$userId,
							$objid,
							$action,
							{},
							$lockinfo,
							$prevLock
						);
					} # end else
				}
			}
			else
			{
				#$retval = Apache2::Const::HTTP_BAD_REQUEST;
				$resp{error} = "Bad request";
				$resp{code}  = 400;
			}
		}
	}
	else
	{
		#$retval = Apache2::Const::HTTP_METHOD_NOT_ALLOWED;
		$resp{error} = "Method not allowed";
		$resp{code}  = 405;
	}

	$resp{code}  = 200 if not $resp{code};
	$resp{error} = "" if not exists $resp{error};
	$resp{lockinfo} = {} unless defined $resp{lockinfo};

	$r->no_cache(1);
	$r->content_type('application/json');
	$r->print(to_json(\%resp));

	return $retval;
}


sub sendToMBus
{
	my ($data,$attach,$hdr) = @_;
	my ($ok,$err) = (1,"");
	my $dev = $1 if $data=~/dev=(.*?)&/;
	$dev = $hdr->{device_ID} if !$dev && $hdr;
	if (defined $dev)
	{
		if (!defined($messenger))
		{
			($ok,$err) = mbus_connect();
			return (0,"Cannot send to MBus ($err)") if not $ok;
		}
		my $msg = $hdr || {
			device_ID => $dev,
			devIP     => "0.0.0.0",
			ctxType   => 16,
			evtType   => 1,
			procLvl   => 1,
		};
		if ($attach and @$attach) {
			$msg->{att_data} = $attach;
			my @desc = ("") x @$attach;
			$msg->{att_description} = \@desc;
		}

		$msg->{data} = $data; # "<PTZ_Command>do.ptz?dev=123&mode=speed&pt=0,0&</PTZ_Command>";
		my $ret = $messenger->send($msg);
		if (!defined($ret))
		{
			($ok,$err) = mbus_connect();
			return (0,"Cannot send to MBus ($err)") if not $ok;
			$ret = $messenger->send($msg);
			return (0,"Cannot send to MBus") if not defined $ret;
		}
	} else {
		return (0,'devID not set');
	}

	return ($ok,$err);
}

sub mbus_connect {
	eval {
		$messenger = NextCAM::Messenger->new(
			host => 's_master',
		port => $MBUS_PORT,
		conn_timeout => $MBUS_CONN_TIMEOUT,
		autoconnect => 0
	);
	die "Failed to create Messenger instance\n" unless $messenger;
	my $sid = $messenger->connect;
	unless ($sid) {
		die "Connection failed\n" unless $messenger->{sock};
		die "Failed to read response\n" unless defined $sid;
		die "bad ACKW result\n" if $sid == 0;
	}

	};
	if ($@) {
	my $err=$@;
	chomp $err;
	$messenger->disconnect;
	$messenger = undef;
	return (0,$err);
	}
	return (1,"");
}

sub ptz_lock_manage {
	my ($data, $sid, $userid, $username, $priority) = @_;

	# manage locks
	my %locks;
	if (not -d $LOCKDIR) {
		mkdir $LOCKDIR, 0770;
	}
	elsif (opendir DH, $LOCKDIR) {
		foreach my $objid (grep {/^\d+$/} readdir DH) {
			my $lck = "$LOCKDIR/$objid";
			if (open FH, $lck) {
				my %props = map {/^(\w+)=(.+)$/} grep {/^\w+=.+/} <FH>;
				close FH;

				$locks{$objid}         = \%props;
				$locks{$objid}{_objid} = $objid;
				$locks{$objid}{_path}  = $lck;
				$locks{$objid}{_mtime} = (stat $lck)[9];
				$locks{$objid}{UPDATED}=0 unless defined $locks{$objid}{UPDATED};
			}
			else {
				unlink $lck; # Remove lockfile if cannot be read
			}
		}
		close DH;
	}

	# remove expired locks
	foreach my $lock (values %locks) {
		if (time - $lock->{UPDATED} > $LOCK_TIMEOUT || ! -f "$SESSDIR/sess_$lock->{SID}") {
			delete $locks{$lock->{_objid}};
			unlink $lock->{_path};
		}
	}

	# parse request data
	my (@devs)  = $data=~/\Wdev=(\d+)/i;
	my ($mode) = $data=~/\Wmode=(\w+)/i;
	my ($cmd)  = $data=~/\Wcmd=(\w+)/i;
	my ($slew)  = $data=~/\Wslew=([\d,]+)/i;

	my ($err, $linfo, $action, $curLock) = ("", undef, undef, undef);
	$mode = 'target' if $data=~/\Wtarget=/i;
	if ($mode eq 'target') {
		@devs = split(/,/, $slew);
	}
	
	if (not @devs) 
	{
		$err = "Device ID is missing";
	}
	else 
	{
		foreach my $dev (@devs)
		{
			$curLock = $linfo = $locks{$dev};

			my $newLock = {
				PRIORITY => $priority,
				SID      => $sid,
				USERID   => $userid,
				USERNAME => $username,
				UPDATED  => time,
				_objid   => $dev,
				_path    => "$LOCKDIR/$dev"
			};

			if (lc $mode eq 'lock') { # Lock management commands
				for ($cmd) {
					/^info$/ and do {
						# Nothing to do
						last;
					};
					/^release$/ and do { # Release current lock
						if (not defined $curLock) {
							# Nothing to do
						}
						elsif ($curLock->{SID} eq $sid) {
							release_lock($curLock);
							delete $locks{$dev};
							$linfo = undef;
							$action = $ACT_RELEASE;
						}
						else {
							$linfo = undef;
							$err = "You are not a lock owner";
						}
						last;
					};
					/^lock$/ and do { # Obtain lock if none exists
						if (not defined $curLock or $curLock->{SID} eq $sid) {
							write_lock($linfo=$newLock);
							$action = $ACT_LOCK;
						}
						else {
							$err = "Device is locked";
						}
						last;
					};
					/^override$/ and do {
						if (not defined $curLock or $curLock->{SID} eq $sid) {
							write_lock($linfo=$newLock);
							$action = $ACT_LOCK;
						}
						elsif ($curLock->{PRIORITY} <= $priority )
						{
							write_lock($linfo=$newLock);
							$action = $ACT_OVERRIDE;
						}
						else {
							$err = "Existing lock has higher priority";
						}
						last;
					};
					/^keepalive$/ and do {
						if (not defined $curLock) {
							$err = "Lock does not exist";
						}
						elsif ($curLock->{SID} eq $sid) {
							#utime undef, undef, $curLock->{_path};
							write_lock($linfo=$newLock);
						}
						else {
							$err = "You are not a lock owner";
						}
						last;
					};
					# Unknown command
					do {
		    				$err = "command not supported";
					};
				}
			}
			elsif ($mode=~/setptztour|stopptztour/i) { # Omit any checks for PTZ Tour commands
				$action = $ACT_PASS;
			}
			else { # Other PTZ commands. Deny if lock exists and belongs to other user
				if ($curLock and $curLock->{SID} ne $sid and $curLock->{SID} ne '-1') 
				{
					$err = "Device is locked";
				}
				elsif ($curLock and $curLock->{SID} eq $sid and
					$priority == 0 and
					$curLock->{PRIORITY} > 0)
				{
					# Special logic to solve conflicts between videotours+presets and operator actions
					# withing same session
					# Skip tour commands (priority=1) if current lock has higher priority
					$err = "Device is locked within same session";
				}
				else {
					if ($curLock->{SID} eq '-1') {
						$action = $ACT_PASS;
					}
					else {
						write_lock($linfo=$newLock);
						$action = $ACT_LOCK;
					}
				}
			}
		
			last if $err;
		}
	} # if (not @devs)

	return ($err, $linfo, $action, $curLock);

}

sub write_lock {
	my $lock = shift;
	$lock->{UPDATED} = time unless $lock->{UPDATED};
	if (open (FH, '>', $lock->{_path})) {
		print FH "$_=$lock->{$_}\n" foreach grep {/^[^_]/} keys %$lock;
		close FH;
		return 1;
	}
	return 0;
}

sub release_lock {
	unlink $_[0]->{_path} if -f $_[0]->{_path};
}

sub readcfg {
	return if time - $last_cfgread_time < $CFGREAD_TIMEOUT;
	db_master() if not $dbm;
	eval {
		my $arr = $dbm->selectrow_arrayref(
			"SELECT val FROM _obj_attr WHERE obj=53 AND ATTR='PTZ_LOCK_TIMEOUT'"
		);
		$LOCK_TIMEOUT = $arr->[0] if defined $arr->[0];
		$last_cfgread_time = time;
	};
	if ($@) {
		db_close();
	}
}

sub insertAuditData {
	my ($sid, $userid, $objid, $action, $attr, $curLock, $prevLock) = @_;

	db_master() unless $dbm;

	# Nothing to audit if lock owner was not changed
	if ($prevLock and $curLock and $prevLock->{SID} eq $curLock->{SID}) {
		return;
	}

	# Depends on action
	for ($action) {
		/$ACT_LOCK/ 	and do { $action = "Locked"; last };
		/$ACT_OVERRIDE/ and do { $action = "Overriden"; last };
		/$ACT_RELEASE/  and do { $action = "Released"; last };
	}

	if ($sid and $sid ne '1' and $ENV{REMOTE_ADDR}) {
		$attr->{IP} = $ENV{REMOTE_ADDR};
	}

	my $params = [[$action],[]];
	foreach my $key (keys %$attr) {
		push ( @{ $params->[1] }, ['@AV', $key, $attr->{$key}] );
	}

	eval {
		$dbm->do("INSERT INTO audit (category,userid,objid,parameters)
			VALUES ($AUDIT_CATEGORY, ?, ?, ?)", undef,
			$userid, $objid, to_json($params)
		);
	};
	db_close() if $@;
}

sub db_close {
	eval { $dbm->disconnect() if $dbm; $dbm=''; }; # disconnect if defined
}

sub db_master {
	db_close;
	eval {
		$dbm=DBMaster({PrintError=>0,RaiseError => 1});
		$dbm->{FetchHashKeyName} = 'NAME_uc';
	};
	if($@) {
		db_close;
	}
}
1;
