#!/usr/bin/perl

use strict;
use SKM::DB;
use Data::Dumper;
use IO::Socket::INET;
use NextCAM::Conf "GetCfgs";
use Master::Conf;
use Node::Conf;

# CONST
my $APL = $ENV{APL};
my $APL_CONF = $ENV{APL_CONF};
my $PROC_CTL = "$APL/conf/bin/node_proc_ctl";


sub do_cmd {
	my $node = $_[0];
	my $cmd = $_[1];
	my $result;
    
	if ($node->{TYPE} eq 'master') {
		$result = `$cmd 2>&1`;
	} else {
		$result = `ssh apl\@$node->{IP} '. ~/.bash_profile; $cmd' 2>&1`;
	}
	chomp $result;
	return $result;
}# do_cmd

sub check_cache {
	my $node = $_[0];
	my ($status, $comment) = ('UNDEF', '');
	
	my $cache_status = do_cmd($node, "$APL/cache/bin/cachectl status");
	if ($cache_status =~ /^(\w+) (\d+) (\d+)/) {
	    if ($1 eq 'on') {
		if ($3==0) {
		    $status = 'ERROR';
		} elsif (int($2/$3)>10) {
		    $status = 'WARN'; # Cache is 90% full
		} else {
		    $status = 'OK';
		}
		$comment = "total=${2}M, free=${3}M";
	    } elsif ($1 eq 'off') {
		$status = 'OK';
		$comment = "cache is OFF";
	    }
	} else {
	    $status = 'ERROR';
	    $comment= "Error getting cache status: $cache_status";
	}
	return ($status,$comment);
}# check_cache

sub check_memory {
	my $node = $_[0];
	my ($status, $comment) = ('UNDEF', '');

	if ($node->{OS} eq 'Linux') {
	    my $mem_status = do_cmd($node, 'free -m | grep Mem');
	    if ($mem_status =~ /^Mem:\s+(\d+)\s+(\d+)\s+(\d+)/) {
		$comment = "total=${1}M used=${2}M free=${3}M";
		$status =  int($1/$3) >10 ? 'WARN' : 'OK';
	    } else {
		$status = 'ERROR';
		$comment = 'no memory status grabed';
	    }
	} elsif ($node->{OS} eq 'Darwin') {
	    my $mem_status = do_cmd($node, 'top -l 1 | grep PhysMem:');
	    if ($mem_status =~ /(\d+)M used, (\d+)M free\.$/) {
		my $total = $1+$2;
		$comment = "total=${total}M used=${1}M free=${2}M";
		$status =  int($total/$2) >10 ? 'WARN' : 'OK';
	    } else {
		$status = 'ERROR';
		$comment = 'no memory status grabed';
	    }
	}
	return ($status,$comment);
}# check_memory

sub check_samba {
    my ($node,$dbh) = @_;
    my ($status, $comment) = ('UNDEF', '');
    my $conns;

    if ($node->{TYPE} eq 'slave') {
	$conns = do_cmd($node, 'smbstatus -S 2>/dev/null | grep -c -P "^vasm"');
	if ($conns==0) {
	    $status = 'ERROR';
	    $comment = 'no SMB connections from master at all';
	} elsif ($conns==1) {
	    $status = 'OK';
	    $comment = 'one SMB connection from master';
	} else {
	    $status = 'WARN';
	    $comment = 'more than one connection anomaly';
	}
    } else {
	$conns = do_cmd($node, 'df | grep -c -P "/vanfs/"');
	if ($conns==$node->{NN}) {
	    $status = 'OK';
	} else {
	    $status = 'ERROR';
	}
	$comment = "$conns SMB connections, $node->{NN} nodes number";

    }

    return ($status, $comment)
}# check_samba

sub check_mretr {
    my ($node,$dbh) = @_;
    my ($status, $comment) = ('UNDEF', '');
    my $query = "select  count(o.obj) from _objs o inner join _obj_attr oa on o.obj=oa.obj \
    		where o.node_ip='$node->{UNI}' and o.otype='D' and (o.subtype='C' or o.subtype='A') \
		    and o.deleted=0 and oa.attr='STATUS'  and oa.val='ON'";
    my $row = $dbh->selectrow_arrayref( $query,	{ Slice => [] });
    my $cam_on = $row->[0];
    my $cam_retr;
    
    $cam_retr = do_cmd($node, 'ps -e -o command | grep  -c -P "cam_mretr a?\d+$"');
    $status = $cam_retr eq $cam_on ? "OK" : "WARN";
    $comment = "cameras: $cam_on, retriewers: $cam_retr";
    return ($status, $comment)
}# check_mretr

sub check_wheel {
    my ($node,$dbh) = @_;
    my ($status, $comment) = ('UNDEF', '');

    my $wheels = do_cmd($node, '/opt/sarch/sm/bin/sm_wheels ins | grep -c -P "Degraded|Full"');

    $status = $wheels==0  ? "OK" : "WARN";
    $comment = "Degraded or Full $wheels wheels";
    return ($status, $comment)
}# check_wheel


sub main {
    die 'Must be run on master!'  unless -f "$APL_CONF/master/s_master";
    chomp(my $master_ip = `cat $APL_CONF/master/s_master`);
    die 'Can not define master ip address' if ($master_ip eq '');

    my $nlist = NodeList;
    my %plist = ( 'cache' => [\&check_cache, 'on'], 
		'memory' => [\&check_memory, 'on'],
		'samba' => [\&check_samba, 'on'],
		'mretr' => [\&check_mretr, 'on'],
		'wheel' => [\&check_wheel, 'on'],
	    );

    my $dbh = DBMaster({ PrintError=>0, RaiseError=>1 });
    $dbh->{FetchHashKeyName} = 'NAME_uc';
    
    
    foreach my $id (keys %$nlist) {
	my $node = $nlist->{$id};
	$node->{TYPE} = $master_ip eq $node->{IP} ? 'master' : 'slave';
	$node->{NN} = scalar (keys %$nlist)-1 if $node->{TYPE} eq 'master';
	$node->{STATUS} = (system "ping -n -c 1 $node->{IP} 2>&1 >/dev/null") ? 'dead' : 'alive';
	if ($node->{STATUS} eq 'dead') {
	    $node->{OS} = 'unknown';
	    print "STATUS[$node->{STATUS}] TYPE[$node->{TYPE}] IP[$node->{IP}] OS[$node->{OS}] VERID[$node->{VERID}] UNI[$node->{UNI}]\n";
	    next;
	} else {
	    $node->{OS} = do_cmd($node,'uname');
	    print "STATUS[$node->{STATUS}] TYPE[$node->{TYPE}] IP[$node->{IP}] OS[$node->{OS}] VERID[$node->{VERID}] UNI[$node->{UNI}]\n";
	}

	foreach my $param (keys %plist) {
	    if ($plist{$param}[1] eq 'on') {
		my @status =  $plist{$param}[0]->($node,$dbh);
		print "  param[$param] status[${status[0]}] comment[${status[1]}]\n";
	    }
	}
    }
	   
}# main
main;
