#!/usr/bin/perl
#  $Id: msg_server.pl 6458 2006-11-20 00:55:13Z teetov $
# -----------------------------------------------------------------------------
#  Messaging server.
#  Controls message unicasts/broadcasts inside NEXTcam.
# -----------------------------------------------------------------------------
#  Author: Andrey Fomenko
#  Edited by: 
#  QA by:  Christopher C Gettings
#  Copyright: (c) videoNEXT Network Solutions LLC, 2005
# -----------------------------------------------------------------------------

use strict;
use POSIX;
use IO::Socket;
use IO::Select;
#use Socket;
use Fcntl;
use Tie::RefHash;
use CGI qw/escape unescape/;

use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.engine";
my $log=get_logger('NEXTCAM::COMMON::MSG_SERVER');

# -----------------------------------------------------------------------------
my $TCP_PORT = 8500; # TCP port where server communicates
# -----------------------------------------------------------------------------
use constant DEFAULT_POLL => 10;
use constant DEFAULT_TIMEOUT => 1;
# -----------------------------------------------------------------------------

# Close filehandles
# close(STDIN); close(STDOUT); close(STDERR);

$/ = "\0"; # Set the input terminator to a zero byte string, pursuant to the protocol in the Flash documentation.

# =========================== PTZ client socket ======================================
my $server = new IO::Socket::INET(
    LocalPort => $TCP_PORT, 
    Listen => 5, 
    Reuse => 1, 
    Proto => 'tcp' 
    ) or $log->logdie("Couldn't open port $TCP_PORT: $!");

my %inbuffer  = ();
my %outbuffer = ();
my %ready     = ();
my %clients   = ();

my $nextClientId = time;

nonblock($server);
my $select = new IO::Select( $server );
tie %ready, 'Tie::RefHash';

$log->warn("Messaging server is ready. Waiting for connections on port $TCP_PORT ...");

while(1) {
    my $client;
    my $rv;
    my $data;
    my $cnt = 0;
    
    foreach $client ($select->can_read(0.1)) {
        if($client == $server) { # ACCEPT NEW CONNECTION
            $cnt++;
            $log->debug('Accepted new client: ',$client);
            $client = $server->accept();
            $select->add($client);
            nonblock($client);
        }
        else { # READ DATA
            $cnt++;
            # read data
            $data = '';
            $rv   = $client->recv($data, POSIX::BUFSIZ, 0);

            unless (defined($rv) && length $data) {
                # This would be the end of file, so close the client
                delete $inbuffer{$client};
                delete $outbuffer{$client};
                delete $ready{$client};
                delete $clients{$client};
                $select->remove($client);
                close $client;
                next;
            }

            $inbuffer{$client} .= $data;

            # test whether the data in the buffer or the data we
            # just read means there is a complete request waiting
            # to be fulfilled.  If there is, set $ready{$client}
            # to the requests waiting to be fulfilled.
            while ($inbuffer{$client} =~ s/(.+)[\n,\0,\r]//m) {
                push( @{$ready{$client}}, $1 );
            }
        }
    } # foreach CLIENT

    # Any complete requests to process?
    foreach $client (keys %ready) {
        handle($client);
    }

    # Buffers to flush?
    foreach $client ($select->can_write(0.01)) {
        # Skip this client if we have nothing to say
        next unless exists $outbuffer{$client};

        $rv = $client->send($outbuffer{$client}, 0);
        unless (defined $rv) {
            # Whine, but move on.
            $log->warn("I was told I could write, but I can't.");
            next;
        }
        if ($rv == length $outbuffer{$client} ||
            $! == POSIX::EWOULDBLOCK) {
            substr($outbuffer{$client}, 0, $rv) = '';
            delete $outbuffer{$client} unless length $outbuffer{$client};
        } else {
            # Couldn't write all the data, and it wasn't because
            # it would have blocked.  Shutdown and move on.
            delete $inbuffer{$client};
            delete $outbuffer{$client};
            delete $ready{$client};

            $select->remove($client);
            close($client);
            next;
        }
    }

    # Out of band data?
    foreach $client ($select->has_exception(0)) {  # arg is timeout
        # Deal with out-of-band data here, if you want to.
    }

    # if it was nothing from clients - let server idle a little bit
    select(undef,undef,undef,.05) if not $cnt; 

} # while(1)



# ---------------------------------------------------------------- handle -----
#   handle($socket) deals with all pending requests for $client
sub handle {
    # requests are in $ready{$client}
    # send output to $outbuffer{$client}
    my $client = shift;
    my ($request,$env,$body,$scope);

    foreach $request (@{$ready{$client}}) {
        # $request is the text of the request
        # put text of reply into $outbuffer{$client}
        $request=~s/[\n\0\r]//gms;
        next if not $request;
        #$log->debug("[$request]");
        my @nodes = split(/</,$request);
        shift @nodes; # strange, but we get empty first element, get rid of it
        goto err_handler if not shift(@nodes) =~ /^msg\s+(.+)>/;
        $env = $1;
        goto err_handler if not pop(@nodes)=~/\/msg>/;
        $body='';
        foreach my $node (@nodes) {
            $body .= "<$node"
        }
        #$log->debug("ENVELOPE:[$env] BODY:[$body]");
        goto err_handler if not $env=~/scope="(\S+)"/;
        $scope=$1;
        #$log->debug("SCOPE:[$scope]");
        
        
        if($scope=~/server:/) {
            $log->debug('Handle message for server');
            handleServerMessages($env,$body,$client,$scope)
        }
        else {
            next if not defined($clients{$client}); # first message MUST be authentication before we go any further
            if($scope eq '*') {
                handleMulticast($env,$body,$client)
            }
            elsif($scope=~/client:(.+)/) {
                handleUnicast($env,$body,$client,$1)
            }
        }
        next;
        err_handler:
        $outbuffer{$client} = '<msg scope="C2S><asString>Wrong envelope syntax</asString><result status="err" error="wrong envelope syntax"/></msg>';
    }
    delete $ready{$client};

}


# -------------------------------------------------- handleServerMessages -----
sub handleServerMessages {
    my $env    = shift; # ENVELOPE
    my $body   = shift; # MESSAGE BODY
    my $client = shift; # CLIENT ID
    my $scope  = shift; # server:XXXX
    
    $log->debug('Processing scope=[',$scope,'] envelope=[',$env,'] body=[',$body,']');
    
    if(not defined($clients{$client})) { 
        return if not $scope eq 'server:connect'; # until we authenticate - ignore any message
    }
    
    if($scope eq 'server:connect') { # ========================================
        # Here is message structure:
        # <msg scope=server:connect [poll=10 timeout=1]>
        #	<authenticate login=name passwd=password/>
        # </msg>
        #
        $log->debug('processing "server:connect"');
        my $poll = ($env=~/poll="(\d+)"/)? $1 : DEFAULT_POLL;
        my $timeout = ($env=~/timeout="(\d+)"/)? $1 : DEFAULT_TIMEOUT;
        $body =~ /<authenticate\s*login="(.+)"\s*passwd="(.+)"\s*\/>/;
        my ($login, $passw) = ($1,$2);
        $log->debug("Register new client: $login");
        # here we should authenticate - do not do for now !!!!!!!!!!!!!!
        my $authResultAbbr = 'ok';          # ok / err
        my $authResultStr = 'SUCCESS';      # SUCCESS / FAILURE
        $clients{$client}{LOGIN} = $login;
        $clients{$client}{POLL} = $poll;
        $clients{$client}{TIMEOUT} = $timeout;
        $outbuffer{$client} = 
            '<msg scope="server:connect">'.
            '<asString>Authentication result: '.$authResultStr.'</asString>'.
            '<result status="'.$authResultAbbr.'" error="'.$authResultStr.'"/>'.
            '<clientId value="'.$client.'"/>'.
            '</msg>';
    } # server:connect
    elsif($scope eq 'server:subscription') { # ================================
        # Here is message structure:
        # <msg scope=server:subscription>
        #	<subscribe class=class1 [devid=devid1]/>
        #	<subscribe class=class2 [devid=devid2]/>
        #	
        #	<unsubscribe class=classN [devid=devidN]/>
        # </msg>
        #
        # Here is memory representation after parsing:
        # $clients{$client}{SUBS}{<class1>}{SCOPE} = 'ALL' / 'LIST'
        # $clients{$client}{SUBS}{<class1>}{LIST}{<devid1>} - if "LIST"
        my ($class,$devid,$node);
        my @nodes = split(/</ms,$body);
        foreach $node (@nodes) {
            # chunks will be like 'subscribe class="class1" [devid="devid1"]/>'
            next if not $node=~/class="(.+)"/;
            $class = $1;
            $devid = ($node=~/devid="(.+)"/)? $1: '*';
            if($node=~/^subscribe/) { ###############################
                if( not defined($clients{$client}{SUBS}{$class})) {
                    if($devid == '*') {
                        $clients{$client}{SUBS}{$class}{SCOPE} = 'ALL';
                    }
                    else {
                        $clients{$client}{SUBS}{$class}{SCOPE} = 'LIST';
                        $clients{$client}{SUBS}{$class}{LIST}{$devid} = 1;
                    }
                }
                elsif($clients{$client}{SUBS}{$class}{SCOPE} == 'ALL') {
                    next; # just ignore - already subscribed to entire class
                }
                else { # subscriptions present, but not for "all", so we deal with DEVIDs
                    $clients{$client}{SUBS}{$class}{LIST}{$devid} = 1;
                }
            }
            elsif($node=~/^unsubscribe/) { ##########################
                if( not defined($clients{$client}{SUBS}{$class})) {
                    next; # just ignore
                }
                elsif($clients{$client}{SUBS}{$class}{SCOPE} == 'ALL') {
                    next if $devid == '*'; # ignore request if previous subscription for "ALL"
                    delete $clients{$client}{SUBS}{$class};
                }
                else { # subscriptions present, but not for "all", so we deal with DEVIDs
                    if($devid == '*') { # unsubscribe from entire class
                        delete $clients{$client}{SUBS}{$class};
                    }
                    else {
                        if(defined($clients{$client}{SUBS}{$class}{LIST}{$devid})) {
                            delete $clients{$client}{SUBS}{$class}{LIST}{$devid};
                        }
                    }
                }
            } ### subscribe / unsubscribed ###
        }
        # now, let's report back to client the current state
        $outbuffer{$client} = '<msg scope="server:subscription">'."\n".
            '<asString>Subscriptions processed</asString>'."\n".
            '<result status="ok" error="success"/>';
        foreach $class ( keys %{$clients{$client}{SUBS}} ) {
            if($clients{$client}{SUBS}{$class}{SCOPE} == 'ALL') {
                $outbuffer{$client} .= "\n<subscribe class=\"${class}\"/>"
            }
            elsif($clients{$client}{SUBS}{$class}{SCOPE} == 'LIST') {
                foreach $devid ( sort keys %{$clients{$client}{SUBS}{$class}{LIST}} ) {
                    $outbuffer{$client} .= "\n<subscribe class=\"${class}\" devid=\"$devid\"/>"
                }
            }
        }
        $outbuffer{$client} .= "\n</msg>";
    } # server:subscription
    elsif($scope eq 'server:presence') { # ====================================
        # Here is message structure:
        # <msg scope=server:presence>
        #     <check what="login|clientId" value="xxx"/>
        # </msg>
        # Message may have either "login" or "clientId" request. Only one will be honored.
        $outbuffer{$client} = '<msg scope="server:presence">'."\n";
        my ( $what, $value ) = ('','');
        if($body=~/check/) {
            if($body=~/what="(.+)"/) {
                if(($1=='login') || ($1=='clientId')) {
                    $what = $1;
                }
            }
            if($body=~/value="(.+)"/) {
                $value = $1;
            }
        }
        if($what && $value) {
            my @foundClietns;
            my $cln;
            foreach $cln ($select->handles) {
                next if not defined $clients{$cln}; # ignore not authorised yet
                if($what=='login') {
                    next if not $clients{$cln}{LOGIN} == $value;
                }
                else {
                    next if not $cln == $value;
                }
                # here we have something what meets query criteria
                push @foundClietns, $cln;
            }
            $outbuffer{$client} .= 
                "<asString>Presence query response: FOUND ".($#foundClietns+1)." client(s) connected\n".
                '<result status="ok" error="success"/>'."\n";
            foreach $cln (@foundClietns) {
                $outbuffer{$client} .= "<client ID=\"$cln\">\n";
            }
            $outbuffer{$client} .= '</msg>';
        }
        else {do not get question
            $outbuffer{$client} .= 
                '<asString>Error while checking for presence</asString>'."\n".
                '<result status="err" error="malformed request"/>'."\n";
        }
        $outbuffer{$client} = '</msg>';
    } # server:presence
    elsif($scope eq 'server:poll') { # ========================================
    } # server:poll
} # sub handleServerMessages

# --------------------------------------------------------- handleUnicast -----
sub handleUnicast {
    my $env        = shift; # ENVELOPE
    my $body       = shift; # MESSAGE BODY
    my $client     = shift; # CLIENT ID (Sender)
    my $targetClId = shift; # CLIENT ID (Target)
    
    $outbuffer{$client} = '<msg><asString>NOT IMPLEMENTED YET</asString></msg>'; 
    
} # sub handleUnicast
# ------------------------------------------------------- handleMulticast -----
sub handleMulticast {
    my $env    = shift; # ENVELOPE
    my $body   = shift; # MESSAGE BODY
    my $client = shift; # CLIENT ID
    
    my $class = $env=~/class="(.+)"/? $1 : '';
    my $devid = $env=~/devid="(.+)"/? $1 : '';
    
    $log->debug('MULTICAST CLASS:$class DEVID:$devid');
    
    if(not $class) {
        $outbuffer{$client} = 
                "<msg $env>\n".
                "<asString>Error while sending message</asString>\n".
                "<result status=\"err\" error=\"class was not provided\"/>\n".
                '</msg>';
        return;
    }
    my $cnt=0;
    foreach my $cln ($select->can_write(0.001)) {
        next if not defined $clients{$cln};
        next if not defined $clients{$cln}{SUBS}{$class};
        if($clients{$cln}{SUBS}{$class}{SCOPE} eq 'LIST') {
            next if not defined $clients{$cln}{SUBS}{$class}{LIST}{$devid};
        }
        next if $cln eq $client; # do not echo back to sender
        $cnt++;
        $outbuffer{$cln} = "<msg $env>$body</msg>";
    }
    $outbuffer{$client} = 
            "<msg $env>\n".
            "<asString>Message multicasted to $cnt subscribed clients</asString>\n".
            "<result status=\"ok\" error=\"success\"/>\n".
            '</msg>';
} # sub handleMulticast

# --------------------------------------------------------- encryptString -----
sub encryptString {
    return shift;
}
# --------------------------------------------------------- decryptString -----
sub decryptString {
    return shift;
}

# -------------------------------------------------------------- nonblock -----
#   nonblock($socket) puts socket into nonblocking mode
sub nonblock {
    my $socket = shift;
    my $flags;
    
    $flags = fcntl($socket, F_GETFL, 0)
            or die "Can't get flags for socket: $!\n";
    fcntl($socket, F_SETFL, $flags | O_NONBLOCK)
            or die "Can't make socket nonblocking: $!\n";
}

