#!/usr/bin/perl -w
# -----------------------------------------------------------------------------
#  The part of 'IP cameras retrival' project
#  Provisioning of AXIS Motion Detector out of SKM
#  enable/disable and configure AXIS internal motion detector out of Admin GUI rather then by configuring individual encoders
#  require admin username and password in camera parameters
#  based on AXIS API v2
# -----------------------------------------------------------------------------
#  Author: ryabovol,
#  Edited by:
#  QA by:
#  Copyright: videoNEXT LLC
# -----------------------------------------------------------------------------
#  usage:
#     md_provision_axis.pl devid=1           - show MD provision status
#     md_provision_axis.pl objid=123         - show MD provision status
#     md_provision_axis.pl devid=1 md=on     - turn MD notifications on
#     md_provision_axis.pl objid=123 md=on   - turn MD notifications on
#     md_provision_axis.pl devid=1 md=off    - turn MD notifications off
#     md_provision_axis.pl objid=123 md=off  - turn MD notifications off

use strict;
use Socket;
use NextCAM::Init;

# VARS -------------------------------------------------------------------------
my ($pname) = reverse(split(/\//, $0)); #name of the program
my $DEBUG = 2;	# use 1 for errors output only and 2 for both debug and errors
my $err;
my $APL = "$ENV{APL}";
my $cam_bin = "$APL/cam/bin/";

# PROC -------------------------------------------------------------------------
sub usage {
    my $err=shift;
    print STDERR "\n$err\n" if $err;
    print STDERR "\nProvisioning of AXIS Motion Detector out of SKM\nUsage:\n"
	."$pname devid=1          - show MD provision status\n"
	."$pname objid=123        - show MD provision status\n"
	."$pname devid=1 md=on    - turn MD notifications on\n"
	."$pname objid=123 md=on  - turn MD notifications on\n"
	."$pname devid=1 md=off   - turn MD notifications off\n"
	."$pname objid=123 md=off - turn MD notifications off\n\n";
}

sub isErrorAnswer {
    my ($str) = @_;
    if(($str =~ /Error:(.+)/) or
	 ($str =~ /\b(Bad\s+Request)/) or	# correct!
	 ($str =~ /(^HTTP ERROR.+)/) or
	 ($str =~ /(Request failed:.+)/)) {
		return $1;
    }
    return '';
}

sub check_md {
    my ($cfg) = shift;
    # -------------- Check some properties -------------------------------------
    check_properties($cfg);
    # -------------- Check list of Events on camera ----------------------------
    debug(2, "Check list of Events on camera [devid=$cfg->{DEVID}] at $cfg->{DEVIP} ...");

    my $answ = `$cam_bin/cam_axis_ctl.pl devid=$cfg->{DEVID} Event`;
    my $err = isErrorAnswer($answ);
    error(1, $err), exit 1	if $err;
    
    debug(2, "List received");

    my $md = 0;    
    while($answ =~ /\w+\.Event\.(E\d+)\.Type=T\s*\w+\.Event\.\1\.Enabled=(.+)\s+(.+\s+){4}\w+\.Event\.\1\.SWInput=M(\d+):\//g) {
	if($2 eq 'yes') {
	    $md = 1;
	}
    }
	
    return ($md) ? "Motion Detection on camera [devid=$cfg->{DEVID}] at $cfg->{DEVIP} is currently turned on\n" :
		   "Motion Detection on camera [devid=$cfg->{DEVID}] at $cfg->{DEVIP} is currently turned off\n";
}

sub check_properties {
    my ($cfg) = shift;
    # ------------------ check out camera HTTP API version ---------------------
    debug(2, "Check HTTP API version for devid=$cfg->{DEVID} ...");

    my $ver = `$cam_bin/cam_axis_ctl.pl devid=$cfg->{DEVID} Properties.API.HTTP.Version`;
    $err = isErrorAnswer($ver);
    error(1, $err), exit 1	if $err;

    debug(2, $ver);

    $ver = ($ver =~ /[\w\.]+=(\d+)/) ? $1 : undef;
    error(1, "HTTP API version is undefined"), exit if not defined $ver;
    error(1, "HTTP API version is $ver, smaller then 2"), exit if($ver < 2);

    # ------------------ check out camera Motion property status ---------------
    debug(2, "Check camera Motion property status ...");

    my $motion_status = `$cam_bin/cam_axis_ctl.pl devid=$cfg->{DEVID} Properties.Motion.Motion`;
    $err = isErrorAnswer($motion_status);
    error(1, $err), exit 1	if $err;

    error(1, "$motion_status"), exit if not $motion_status =~ /=yes\s*$/;
    debug(2, "$motion_status");
}

sub debug : locked { print STDERR "DEBUG @_\n" if $DEBUG>=shift }
sub error : locked { print STDERR "ERROR @_\n" if $DEBUG>=shift }
							 

# MAIN -------------------------------------------------------------------------
my ($device, $cfg) = (shift @ARGV);
usage(), exit if not $device;
if($device =~ /devid=(a*\d+)/) {    #devid parameter
    my %conf = GetCfgs('DEVID' => $1);
    print(STDERR "CFG ERROR: Device $device is not defined in system\n"), exit 1 if not %conf;
    $cfg = $conf{$1};
}elsif($device =~ /objid=(\d+)/) {  #objid parameter
    my %conf = GetCfgs('OBJID' => $1);
    print(STDERR "CFG ERROR: Device $device is not defined in system\n"), exit 1 if not %conf;
    my $devid = (keys %conf)[0];
    $cfg = $conf{$devid};
}
else {
    usage("CFG ERROR: Wrong Device '$device' specification");
    exit 1;
}
print(STDERR "CFG ERROR: This is not AXIS camera! CAMERAMODEL=$cfg->{CAMERAMODEL}\n"),exit 1    if $cfg->{CAMERAMODEL} ne 'axis';

my $md_state = shift @ARGV;
print(check_md($cfg)), exit if not $md_state;
if($md_state =~ /md=on/) {
    $md_state = 1;
}
elsif($md_state =~ /md=off/) {
    $md_state = 0;
}
else {
    print(check_md($cfg));
    usage();
    exit;
}

# ----------------- now read all configs into hash -----------------------------
#my %conf =  GetCfgs( ('DEVICETYPE'=>'CAMERA') );
#foreach my $dev (keys %conf) {
#    delete $conf{$dev} if $conf{$dev}{CAMERAMODEL} ne 'axis';
#}

# -------------- Check some properties -----------------------------------------
check_properties($cfg);

if($md_state) {
    # ---------------- Getting out own hostname and IP address -----------------
    my ($hostname, $addr) = `hostname`;
    chomp($hostname);
    $addr = join ".", unpack('C4',inet_aton($hostname));
    debug(2, "HOST: $hostname, IP: $addr");

    # -------------- Check list of the HTTP Event Servers for camera -----------
    debug(2, "Check list of the Event Servers for camera [devid=$cfg->{DEVID}] at $cfg->{DEVIP} ...");

    my ($http_serv, $event, $answ, @p) = ('', '');
    $answ = `$cam_bin/cam_axis_ctl.pl devid=$cfg->{DEVID} EventServers.HTTP`;
    $err = isErrorAnswer($answ);
    error(1, $err), exit 1	if $err;
    
    if(($answ =~ /EventServers\.HTTP\.(H\d+)\.Address=http:\/\/(.+)\/sarch\/cgi-bin\/cam\/cam_alarm.pl\?objid=$cfg->{OBJID}&message=debug/) and
	(($2 eq $addr) or ($2 eq $hostname))) {
	    $http_serv = $1;
	    debug(2, "HTTP EventServer $http_serv found.");
    }
    else {
	debug (2, "The current \"$addr\" server is not found in the list of the Event Servers at $cfg->{DEVIP}. We add one.");
	my $add = "group:=EventServers.HTTP "
		 ."template:=http_config "
		 ."EventServers.HTTP.H.Name:=$hostname "
		 ."EventServers.HTTP.H.Address:=http://$addr/sarch/cgi-bin/cam/cam_alarm.pl?objid=$cfg->{OBJID}%26message=debug";
	$add = `$cam_bin/cam_axis_ctl.pl devid=$cfg->{DEVID} $add`;
	$err = isErrorAnswer($add);
	error(1, $err), exit 1	if $err;
	debug(2, "$add");
	if($add =~ /(H\d+) OK/) {
	    $http_serv = $1;
	} else { exit 1; }
    }

    # -------------- Check list of Events on camera ----------------------------
    debug(2, "Check list of Events on camera [devid=$cfg->{DEVID}] at $cfg->{DEVIP} ...");

    $answ = `$cam_bin/cam_axis_ctl.pl devid=$cfg->{DEVID} Event`;
    $err = isErrorAnswer($answ);
    error(1, $err), exit 1	if $err;
    
    debug(2, "List received");

    my $count = 0;
    while($answ =~ /\w+\.Event\.(E\d+)\.Type=T\s*\w+\.Event\.\1\.Enabled=(.+)\s+(.+\s+){4}\w+\.Event\.\1\.SWInput=M(\d+):\//g) {
	$event = $1;
	if($2 eq 'no') {
	    my $answer = `$cam_bin/cam_axis_ctl.pl devid=$cfg->{DEVID} Event.$1.Enabled=yes`;
	    $err = isErrorAnswer($answer);
	    error(1, $err), exit 1	if $err;

	    debug(2, "Motion Detection Event $1 turned on");
	}
	else { debug(2, "Motion Detection Event $1 enabled"); }
	
	if($answ =~ /\w+\.Event\.$event\.Actions\.(A\d+)\.Type=N\s+\w+\.Event\.$event\.Actions\.\1\.Protocol=HTTP\s+.+\s+\w+\.Event\.$event\.Actions\.\1\.Server=$http_serv/) {
	    debug (2, "We already have Motion Detection HTTP notifications Action $1 for Event $event on camera [devid=$cfg->{DEVID}] at $cfg->{DEVIP}");
	}
	else {
	    my $add = "group:=Event.$event.Actions "
		  ."template:=httpaction "
		  ."Event.$event.Actions.A.Type:=N "
		  ."Event.$event.Actions.A.Protocol:=HTTP "
		  ."Event.$event.Actions.A.Server:=$http_serv";
	    $add = `$cam_bin/cam_axis_ctl.pl devid=$cfg->{DEVID} $add`;
	    $err = isErrorAnswer($add);
	    error(1, $err), exit 1	if $err;

	    debug(2, "Added Motion Detection Action for Event $event: $add");
	}
	
	$count++;
    }

    if(!$count) {    
	  debug(2, "Create an Event, triggered when Motion Detection starts");
	  
	  # ------- Create an Event, trigger when motion detection starts ------
	  my $add = "group:=Event "
		     ."template:=event "
		     ."Event.E.Name:=MD_starts "
		     ."Event.E.Type:=T "
		     ."Event.E.SWInput:=M0:/ "	# We will assume, that the window with key=0 is already created and adjusted
		     ."Event.E.Duration:=24:00";
	  $add = `$cam_bin/cam_axis_ctl.pl devid=$cfg->{DEVID} $add`;
	  $err = isErrorAnswer($add);
	  error(1, $err), exit 1	if $err;

	  $event = $1 if $add =~ /(E\d+) OK/;
	  debug(2, "$add");

	  # - Create an Action to the last Event for HTTP notification sending -

	  debug (2, "Create an Action to $event at $cfg->{DEVIP} for [devid=$cfg->{DEVID}]");
  
	  $add = "group:=Event.$event.Actions "
		  ."template:=httpaction "
		  ."Event.$event.Actions.A.Type:=N "
		  ."Event.$event.Actions.A.Protocol:=HTTP "
		  ."Event.$event.Actions.A.Server:=$http_serv";
	  $add = `$cam_bin/cam_axis_ctl.pl devid=$cfg->{DEVID} $add`;
	  $err = isErrorAnswer($add);
	  error(1, $err), exit 1	if $err;

	  debug(2, "$add");
    }
}
else {
    # -------------- Check list of Events on camera ----------------------------
    debug(2, "Check list of Events on camera [devid=$cfg->{DEVID}] at $cfg->{DEVIP} ...");

    my $answ = `$cam_bin/cam_axis_ctl.pl devid=$cfg->{DEVID} Event`;
    $err = isErrorAnswer($answ);
    error(1, $err), exit 1	if $err;
    
    debug(2, "List received");

    my $count = 0;    
    while($answ =~ /\w+\.Event\.(E\d+)\.Type=T\s*\w+\.Event\.\1\.Enabled=(.+)\s+(.+\s+){4}\w+\.Event\.\1\.SWInput=M(\d+):\//g) {
	my $event = $1;
	if($2 eq 'yes') {
	    $answ = `$cam_bin/cam_axis_ctl.pl devid=$cfg->{DEVID} Event.$1.Enabled=no`;
	    $err = isErrorAnswer($answ);
	    error(1, $err), exit 1	if $err;

	    debug(2, "Motion Detection Event $event turned off");
	}
	else { debug(2, "Motion Detection Event $event disabled"); }
	
	$count++;
    }
    
    debug(2, "Camera [devid=$cfg->{DEVID}] at $cfg->{DEVIP} don't have Motion Detection Events") if(!$count);
}