#!/usr/bin/perl -w
#  $Id: cam_axis_ctl.pl 32385 2015-05-06 07:32:20Z atsybulnik $
# -----------------------------------------------------------------------------
#  The part of 'IP cameras retrival' project
#  get/set AXIS parameters
#  require admin username and password in camera parameters
#  based on AXIS API v2
# -----------------------------------------------------------------------------
#  Author: teetov, 
#  Edited by:
#  QA by:
#  Copyright: videoNEXT LLC
# -----------------------------------------------------------------------------
#  usage:
#         cam_axis_ctl.pl devid=1                                        - show axis model and version
#         cam_axis_ctl.pl objid=123                                      - show axis model and version
#         cam_axis_ctl.pl devid=1   Image.I0.RateControl.MaxBitrate      - show axis parameters
#         cam_axis_ctl.pl objid=123 Image.I0.RateControl.MaxBitrate      - show axis parameter
#         cam_axis_ctl.pl devid=1   Image.I0.RateControl.MaxBitrate=1000 - set parameter
#         cam_axis_ctl.pl objid=123 Image.I0.RateControl.MaxBitrate=1000 - set parameter
#         cam_axis_ctl.pl devid=1   group:=Event template:=event         - add an Event with all parameters set to their default value
#         cam_axis_ctl.pl objid=123 group:=Event template:=event         - add an Event with all parameters set to their default value
#         cam_axis_ctl.pl devid=1   -Event.E0.Actions.A0                 - remove an action A0 from event E0
#         cam_axis_ctl.pl objid=123 -Event.E0.Actions.A0                 - remove an action A0 from event E0
#         cam_axis_ctl.pl devid=1   1                                    - check current state of axis input 1
#         cam_axis_ctl.pl objid=123 1                                    - check current state of axis input 1
#         cam_axis_ctl.pl devid=1   1:/1000\                             - turn axis output 1 on for 1 second and after that turn it off
#         cam_axis_ctl.pl objid=123 1:/1000\                             - turn axis output 1 on for 1 second and after that turn it off


use strict;
use HTTP::Request::Common;
use SKM::Agent; # for digest auth

# VARS -------------------------------------------------------------------------
my ($P) = (split(/\//, $0))[-1]; #name of the program
my $device = shift @ARGV;
my $http_timeout=5;

# PROC -------------------------------------------------------------------------
sub usage {
   my $err=shift;
   print STDERR "\n$err\n" if $err;
   print STDERR "\nAXIS provisioning helper\nUsage:\n"
	."  $P devid=1                                        - show model and version\n"
	."  $P objid=123                                      - show model and version\n"
	."  $P devid=1   Image.I0.RateControl.MaxBitrate      - show parameter\n"
        ."  $P objid=123 Image.I0.RateControl.MaxBitrate      - show parameter\n"
        ."  $P devid=1   Image.I0.RateControl.MaxBitrate=1000 - set parameter\n"
        ."  $P objid=123 Image.I0.RateControl.MaxBitrate=1000 - set parameter\n"
        ."  $P devid=1   group:=Event template:=event         - add an Event with all parameters set to their default value\n"
        ."  $P objid=123 group:=Event template:=event         - add an Event with all parameters set to their default value\n"
        ."  $P devid=1   -Event.E0.Actions.A0                 - remove an action A0 from event E0\n"
        ."  $P objid=123 -Event.E0.Actions.A0                 - remove an action A0 from event E0\n"
        ."  $P devid=1   1                                    - check current state of axis input 1\n"
        ."  $P objid=123 1                                    - check current state of axis input 1\n"
        ."  $P devid=1   1:/1000\\                             - turn axis output 1 on for 1 second and after that turn it off\n"
        ."  $P objid=123 1:/1000\\                             - turn axis output 1 on for 1 second and after that turn it off\n\n";
}

sub send_request {
 my ($url, $username, $password) = @_;
 my $ua = new SKM::Agent;
 $ua->timeout($http_timeout);
 $url =~ tr/ //d;
 my $req = GET $url;
 $ua->set_creds($username, $password);
 my $rsp=$ua->request($req);
 return ($rsp->is_success) ? $rsp->content."\n" : "HTTP ERROR [".$rsp->code."]";
}

sub get_axis_param {
 my $cf = shift;
 my $key = join ',',@_;
 my $prev_tmt=$http_timeout;
 $http_timeout=3;
 my $res = send_request("http://$cf->{DEVIP}:$cf->{HTTP_PORT}/axis-cgi/admin/param.cgi?action=list&group=$key",
				$cf->{USRNAME}, $cf->{PASSWD});
 $http_timeout=$prev_tmt;			
 if ($res =~ /^HTTP ERROR \[(\d+)\]/) {
    die "Device does not respond\n" if $1=~/^5\d\d/;
    die "Authorization error\n"     if $1==401;
    return $res.": Cannot get $key\n";
 }
 return $res;
}

sub add_axis_param {
 my $cf = shift;
 my $val = join '&',@_;
 my $res = send_request("http://$cf->{DEVIP}:$cf->{HTTP_PORT}/axis-cgi/admin/param.cgi?action=add&$val",
				$cf->{USRNAME}, $cf->{PASSWD});
 return ($res =~ /HTTP ERROR/) ? $res.": Cannot add $val\n" : $res;
}

sub del_axis_param {
 my $cf = shift;
 my $key = join ',',@_;
 my $res = send_request("http://$cf->{DEVIP}:$cf->{HTTP_PORT}/axis-cgi/admin/param.cgi?action=remove&group=$key",
				$cf->{USRNAME}, $cf->{PASSWD});
 return ($res =~ /HTTP ERROR/) ? $res.": Cannot get $key\n" : $res;
} 

sub set_axis_param {
 my $cf = shift;
 my $val = join '&',@_;
 my $res = send_request("http://$cf->{DEVIP}:$cf->{HTTP_PORT}/axis-cgi/admin/param.cgi?action=update&$val",
				$cf->{USRNAME}, $cf->{PASSWD});
 return ($res =~ /HTTP ERROR/) ? $res.": Cannot set $val\n" : $res;
}

sub set_and_check {
 my $cf = shift;
 my $val = join "\n",@_;
 my $key = join ',', map {/^(.+)=.+/} @_;
 my $old = get_axis_param($cf, $key);
 my $set = set_axis_param($cf, @_);
 my $new = get_axis_param($cf, $key);
 return " Old value:\n$old Set value:\n$val\n $set New value:\n$new";
}

sub check_axis_inputs {
 my $cf = shift;
 my $key = join ',',@_;
 my $res = send_request("http://$cf->{DEVIP}:$cf->{HTTP_PORT}/axis-cgi/io/input.cgi?check=$key",
				$cf->{USRNAME}, $cf->{PASSWD});
 return ($res =~ /HTTP ERROR/) ? $res.": Cannot get $key\n" : $res;
}

sub setup_axis_outputs {
 my $cf = shift;
 my $key = join ',',@_;
 my $res = send_request("http://$cf->{DEVIP}:$cf->{HTTP_PORT}/axis-cgi/io/output.cgi?action=$key",
				$cf->{USRNAME}, $cf->{PASSWD});
 return ($res =~ /HTTP ERROR/) ? $res.": Cannot get $key\n" : $res;
}


# Local implementation of GetCfgs for breaking dependancy from NextCAM::Init
sub get_cfgs {
    my (%filter)=@_;
    my $apl_conf = $ENV{APL_CONF} || "/opt/sarch/var/conf";
    opendir(CFGDIR,$apl_conf);   
    opendir(CFGTMP,"$apl_conf/conf");
    my (%devs,$dev,$cfg,$ignore);
    while (defined($dev = readdir(CFGDIR)) || defined($dev = readdir(CFGTMP))) {
        next if $dev=~/^\./;
        next if $dev=~/^asrv/;
        next if $dev=~/^n\d+/; # AT: substituted by NodeList
        next if $dev=~/^(node|master)$/;
        my $file="$apl_conf/conf/$dev";       
        if(open(CFG,$file)) {
            $dev =~ /^(\w?\d+).conf/; 
            $dev = $1;
            $cfg=\%{{map {/(\w+)=(.*)/} grep {/^\w+=.*$/} <CFG>}};
            close(CFG);
        } else {
            $file = "$apl_conf/$dev/conf";
 #           $log->error("Unable open for reading: [$apl_conf/conf/$dev]");
            if(open(CFG,$file)) {
              $cfg=\%{{map {/(\w+)=(.*)/} grep {/^\w+=.*$/} <CFG>}};
              close(CFG);              
            }
            else
            {
              next;
            }
        } # if(open(CFG,$file))
        
        $ignore = 0;
        foreach(keys %filter) {
            $ignore = 1, last if not defined($cfg->{$_}) or not defined($filter{$_});
            $ignore = 1, last if not ( $cfg->{$_} eq $filter{$_} );
        }
        next if $ignore;
        foreach (keys %{$cfg}) { $devs{$dev}{$_}=$cfg->{$_}; }
        
    } # while (defined($file = readdir(CFGDIR)))
    closedir(CFGDIR);
    return %devs;
} # sub get_cfg

# MAIN -------------------------------------------------------------------------
my $cfg;
usage(), exit if not $device;
if($device =~ /devid=(a*\d+)/) {	#devid parameter
   my %conf = get_cfgs('DEVID' => $1);
   print(STDERR "CFG ERROR: Device $device is not defined in system\n"), exit 1	if not %conf;
   $cfg = $conf{$1};
}elsif($device =~ /objid=(\d+)/) {	#objid parameter
   my %conf = get_cfgs('OBJID' => $1);
   print(STDERR "CFG ERROR: Device $device is not defined in system\n"), exit 1	if not %conf;
   my $devid = (keys %conf)[0];
   $cfg = $conf{$devid};
}else {
  usage("CFG ERROR: Wrong Device '$device' specification");
  exit 1;
}
print(STDERR "CFG ERROR: This is not AXIS camera! CAMERAMODEL=$cfg->{CAMERAMODEL}\n"),exit 1    if $cfg->{CAMERAMODEL} ne 'Axis';


if(not @ARGV) {	# only show model & version
  print get_axis_param($cfg, 'Properties.Firmware.Version', 'brand.ProdShortName');
  exit 0;
}

my (@get, @set, @add, @del, @chk_inputs, @set_outputs);
foreach(@ARGV) {
 push(@get, $1)     if /^(\w[\w\.]+)$/;
 push(@set, $1)     if /^(\w[\w\.]+=.+)$/;
 push(@add, $1.$2)  if /^(\w[\w\.]+):(=.+)$/;
 push(@del, $1)     if /^-(\w[\w\.]+)$/;

 if($_ =~ /^(\d),?(\d)?,?(\d)?,?(\d)?[,\d]*$/) {
    push(@chk_inputs, $1);
    push(@chk_inputs, $2) if defined $2;
    push(@chk_inputs, $3) if defined $3;
    push(@chk_inputs, $4) if defined $4;
 }
 
 push(@set_outputs,$1) if /^(\d:[\/\\0-9]+)$/;
}

print set_and_check($cfg, @set)  if @set;
print get_axis_param($cfg, @get) if @get;
print add_axis_param($cfg, @add) if @add;
print del_axis_param($cfg, @del) if @del;
print check_axis_inputs($cfg, @chk_inputs) if @chk_inputs;
print setup_axis_outputs($cfg,@set_outputs) if @set_outputs;
