<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: VideoHelper.php 33581 2016-02-08 18:25:00Z astarostin $
 * ------------------------------------------------------------------------------
 * video helper
 * ------------------------------------------------------------------------------
 * @author Alex Teetovfunctions
 * @author Andrey Baranetsky
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2012
 * ------------------------------------------------------------------------------
 */

class VideoHelper
{
	public function __construct()
	{
	}

	// Assumption:
	// $_SESSION['MEDIA_TRANSPORT'] = http,  https, rtsp/tcp, rtsp/udp
	// $_SESSION['NAT'] = 0, 1

	/**
	 * params:
	 * in:  OBJID, PLAYER, STREAMTYPE
	 * out: MEDIA_FORMAT, MEDIA_TRANSPORT, NODE_ADDR, NODE_PORT_MEDIA, NODE_PORT_HTTP, TIME_ZONE, DEVICE_TYPE
	 *      DEVICE_ID (node scope device id), FRAMERATE
	 *
	 * @param array $params
	 * @throws Exception
	 */
	public function getDeviceInfo(array &$params)
	{
		$nodeId = null;

		// Get device information
		//
		$query = "SELECT obj, otype, subtype, node_id, deleted FROM _objs WHERE obj = ?";
		$list = DB::select($query, array($params['OBJID']));
		if (count($list) == 0)
		{
			throw new Exception('UNKNOWN CAMERA (MCA-0003)'); // "Device Id ${params['OBJID']} is not found in database";
		}

		if ($list[0]["deleted"] != '0')
		{
			throw new Exception('DELETED (MCA-0012)'); // "Object is deleted";
		}

		if (!($list[0]["otype"] == 'D' && ($list[0]["subtype"] == 'C' || $list[0]["subtype"] == 'A')))
		{
			throw new Exception('Device configuration error (MCA-0002)'); // "Device Id ${params['OBJID']} is not a camera or audio source";
		}

		$params['DEVICE_ID'] = $list[0]["obj"];
		$params['DEVICE_TYPE'] = $list[0]["subtype"];
		$nodeId = $list[0]["node_id"];

		// Get device attributes (TIME_ZONE, MEDIA_FORMAT)
		//
		$params['MEDIA_FORMAT'] = 'mjpg';

		$query = "select attr, val from _obj_attr where obj = ?  and attr in ('MEDIA_FORMAT','TIME_ZONE','FRAMERATE','ARCFRAMERATE','PIXEL_ASPECT_RATIO','POSITIONCTL','STATUS','MULTICAST', 'AVATARID', 'AV_DELIVERY')";
		$list = DB::select($query, array($params['OBJID']));
		foreach ($list as $row)
		{
			if ($row['attr'] == 'AVATARID' && !empty($row['val']) && isset($params['STREAMLOCAL']))
			{
				$nodeId = $row['val'];
				$params['AVATARID'] = $row['val'];
			} else
			if ($row["attr"] == 'FRAMERATE' && $params['STREAMTYPE'] == 'archive')
			{
				$params['FRAMERATE'] = $row["val"];
			} else
			if ($row["attr"] == 'ARCFRAMERATE' && $params['STREAMTYPE'] == 'archive')
			{
				$params['FRAMERATE'] = $row["val"];
			} else {
				$params[$row["attr"]] = $row["val"];
			}
		}

		// handle supplementary AUDIOOBJID -> SUPL_AUDIO_ID
		if (isset($params['AUDIOOBJID']))
		{
			$query = "SELECT obj, node_id, otype, subtype, deleted FROM _objs WHERE obj = ?;";
			$list = DB::select($query, array($params['AUDIOOBJID']));
			if (count($list) == 0)
			{
				throw new Exception('UNKNOWN AUDIO SOURCE (MCA-0003)'); // "Device Id ${params['OBJID']} is not found in database";
			}

			if ($list[0]["deleted"] != '0')
			{
				throw new Exception('DELETED (MCA-0012)'); // "Object is deleted";
			}

			if ($list[0]["node_id"] != $nodeId && !isset($params['STREAMLOCAL']))
			{
				throw new Exception('Supplimentary device is hosted on different node (MCA-0013)'); // Audio and video ara located on different nodes
			}

			if ($list[0]["otype"] != 'D' || $list[0]["subtype"] != 'A')
			{
				throw new Exception('Supplimentary device is not audio (MCA-0002)'); // "Device Id ${params['AUDIOOBJID']} is not audio"
			}

			$params['SUPL_AUDIO_ID'] = $list[0]["obj"];
		}

		$this->_getNodeNetworkParams($params, $nodeId);

		// use multicasting in live mode if media server and user application both supports multicast
		//
		$multicastSupported = true;
		if (isset($_SESSION[SESSION_MCASTSUPPORTED]) && $_SESSION[SESSION_MCASTSUPPORTED] == 'FALSE')
		{
			$multicastSupported = false;
		} else
		if (isset($params['MCASTSUPPORTED']) && strtoupper($params['MCASTSUPPORTED']) == 'FALSE')
		{
			$_SESSION[SESSION_MCASTSUPPORTED] = 'FALSE';
			$multicastSupported = false;
		}
		if ($multicastSupported && $params['STREAMTYPE'] == 'live' &&
			isset($params['MULTICAST']) && strtoupper($params['MULTICAST']) == 'ON'
		)
		{
			$params['MULTICAST_STREAM'] = 'TRUE';
		}

		// Get media transport
		//
		// Apply default MEDIA_TRANSPORT if it's set
		if (isset($_SESSION[SESSION_MEDIA_TRANSPORT]))
		{
			$userMediaTransport = $_SESSION[SESSION_MEDIA_TRANSPORT];
		} else {
			$CLIENT_TRANSPORT_STREAM = Identity::getAttribute("CLIENT_TRANSPORT_STREAM");
			$userMediaTransport = ($params['PLAYER'] == 'MPLAYER') ? "rtsp/" . strtolower($CLIENT_TRANSPORT_STREAM) : 'http';
		}

		if (strpos($userMediaTransport, 'rtsp') === false)
		{
			$params['MEDIA_TRANSPORT'] = $userMediaTransport;
			$params['NODE_PORT_MEDIA'] = 80;
		} else {
			if ($params['PLAYER'] == 'MPLAYER')
			{
				if (isset($params['MULTICAST_STREAM']) && $params['MULTICAST_STREAM'] == 'TRUE')
				{
					$params['MEDIA_TRANSPORT'] = 'udp';
				} else {
					$params['MEDIA_TRANSPORT'] = (strpos($userMediaTransport, 'udp') === false) ? 'tcp' : 'udp';
				}
			} else {
				$params['MEDIA_TRANSPORT'] = 'http';
			}
		}

		if ($params['STREAMTYPE'] == 'archive')
		{
			// convert archive interval dates into seconds
			if (!isset($params['TIME_START']))
			{
				throw new Exception('Device configuration error (MCA-0006)'); // "start date of archive interval is missing";
			}

			$intTimestamp = $this->parseTimestamp($params['TIME_START']);
			if ($intTimestamp == -1)
			{
				throw new Exception('(MCA-0009)');
			}

			$params['TIME_START_SEC'] = strval($intTimestamp);

			if (isset($params['TIME_END']))
			{
				$intTimestamp = $this->parseTimestamp($params['TIME_END']);
				if ($intTimestamp == -1)
				{
					throw new Exception('(MCA-0009)');
				}

				$params['TIME_END_SEC'] = strval($intTimestamp);
			}
		}
	}

	/**
	 * Retrieve node-specific network parameters
	 * in: OBJID, PLAYER, $nodeId
	 * out: NODE_ADDR, NODE_PORT_MEDIA, NODE_PORT_HTTP, MEDIA_TRANSPORT, NODE_PORT_MEDIA
	 *
	 * @param  array  $params
	 * @param  int    $nodeId
	 */
	private function _getNodeNetworkParams(&$params, $nodeId)
	{
		// Get NODE info
		//
		$node = new Node();
		$nodeAttributes = $node->getAttributes($nodeId);

		$NODE_ADDR = $nodeAttributes["IP"];
		if (isset($nodeAttributes["URI"]) && $nodeAttributes["URI"] != '' && $nodeAttributes["URI"] != 'localhost.localdomain')
		{
			$NODE_ADDR = $nodeAttributes["URI"];
		}

		$params["NODE_ADDR"] = $NODE_ADDR;
		$params["NODE_ID"] = $node->getName($nodeId);
		$params["NODE_PORT_MEDIA"] = $nodeAttributes["RTSP_PORT"];
		$params["MAX_GUI_FRAMERATE"] = Identity::getAttribute("MAX_GUI_FRAMERATE");
	}

	/**
	 * Check if user is allow to access media stream
	 *
	 * @param array $params
	 * @return bool
	 */
	public function checkMediaCredentials(array &$params)
	{
		$result = false;

		$auditlog = true;
		if (isset($params['NOAUDIT']))
		{
			$naVal = strtoupper($params['NOAUDIT']);
			$auditlog = !($naVal == '1' || $naVal == 'TRUE' || $naVal == 'YES');
		}

		if ($params['STREAMTYPE'] == 'archive')
		{
			if ($auditlog)
			{
				Audit::addRecordVArg(12, $params['OBJID'], null, 'Archive',
					Audit::attrValueEntity('From', $params['TIME_START_SEC']),
					Audit::attrValueEntity('To', $params['TIME_END_SEC']));
			}

			$result = $this->_checkCred($params['OBJID'], 'A');

			if ($result && !empty($params['AUDIOOBJID']))
			{
				if ($auditlog)
				{
					Audit::addRecordVArg(12, $params['AUDIOOBJID'], null, 'Archive',
						Audit::attrValueEntity('From', $params['TIME_START_SEC']),
						Audit::attrValueEntity('To', $params['TIME_END_SEC']));
				}
				$result = $this->_checkCred($params['AUDIOOBJID'], 'A');
			}
		} else {
			if ($auditlog)
			{
				Audit::addRecordVArg(12, $params['OBJID'], null, 'Live');
			}

			$result = $this->_checkCred($params['OBJID'], 'L');

			if ($result && !empty($params['AUDIOOBJID']))
			{
				if ($auditlog)
				{
					Audit::addRecordVArg(12, $params['AUDIOOBJID'], null, 'Live');
				}
				$result = $this->_checkCred($params['AUDIOOBJID'], 'L');
			}
		}

		return $result;
	}

	/**
	 * input: MEDIA_FORMAT, NODE_ADDR, NODE_PORT_MEDIA,
	 *        PLAYER, STREAMTYPE, TIME_START, TIME_END
	 * output: URL, MEDIA_TRANSPORT
	 *
	 * @param array $params
	 * @throws Exception
	 */
	public function getMediaUrl(array &$params)
	{
		if ($params['PLAYER'] != 'MPLAYER')
		{
			$url = $params['MEDIA_TRANSPORT'] . '://' . $params['NODE_ADDR'] . ':' . $params['NODE_PORT_MEDIA'] .
					'/sarch/cgi-bin/cam/nph-mjpg?dev=' . $params['DEVICE_ID'];
		} else {
			if ($params['STREAMTYPE'] == 'snapshot')
			{
				// Still image
				//
				$port = ($_SERVER['SERVER_PORT'] == 80) ? '' : ':' . $_SERVER['SERVER_PORT'];
				$protocol = isset($_SERVER['HTTPS']) ? "https" : "http";
				if ($params['DEVICE_TYPE'] == 'A')
				{
					// TODO: find trumpet image
					$url = $protocol . '://' . $_SERVER['SERVER_NAME'] . $port . '/common/image/trumpet.jpg';
				} else {
					$url = $protocol . '://' . $_SERVER['SERVER_NAME'] . $port . '/storage/snapshot?objid=' . $params['DEVICE_ID'];
				}
			} else {
				$mediaFormat = 'xmedia'; // by default

				if (!empty($params['TRANSCODE']))
				{
					$mediaFormat = $params['TRANSCODE'] . 'media';
				}

				if ($params['DEVICE_TYPE'] == 'A')
				{
					$params['DEVICE_ID'] = "a" . $params['DEVICE_ID'];
				}

				$url = 'rtsp://' . $params['NODE_ADDR'] . ':' . $params['NODE_PORT_MEDIA'] .
						'/' . $mediaFormat . '?dev=' . $params['DEVICE_ID'] . '&objid=' . $params['OBJID'];

				if ($params['STREAMTYPE'] == 'archive')
				{
					// Attach dates for archive interval &start= &end=
					//
					if (!empty($params['TIME_START_SEC']))
					{
						$url .= '&start=' . $params['TIME_START_SEC'];
					}
					if (!empty($params['TIME_END_SEC']))
					{
						$url .= '&end=' . $params['TIME_END_SEC'];
					}
					if (!empty($params['ANALYTICS']))
					{
						$url .= '&analytics=' . $params['ANALYTICS'];
					}
					if (!empty($params['EVENTID']))
					{
						$url .= '&eventid=' . $params['EVENTID'];
					}
				} else
				if ($params['STREAMTYPE'] == 'live')
				{
					if (!empty($params['ANALYTICS']))
					{
						$url .= '&analytics=' . $params['ANALYTICS'];
					}
					if (isset($params['STATUS']) && strtoupper($params['STATUS']) == 'OFF')
					{
						throw new Exception('OFFLINE (MCA-0011)');
					}
				}

				if (!empty($params['TRANSCODE']))
				{
					$url .= '&transcode';
				}

				if (!empty($params['DIMENSIONS']))
				{
					$url .= '&dimensions=' . $params['DIMENSIONS'];
				} else
				if (!empty($params['RES']))
				{
					$url .= '&dimensions=' . $params['RES'];
				}

				if (isset($params['DOWNSCALE']))
				{
					$url .= '&downscale';
				}

				if (!empty($params['FMT']))
				{
					$url .= '&fmt=' . $params['FMT'];
				}

				if (!empty($params['METADATA']))
				{
					$url .= '&metadata=' . $params['METADATA'];
				}

				if (!empty($params['VAE']))
				{
					$url .= '&vae=' . $params['VAE'];
				}

				if (!empty($params['SUPL_AUDIO_ID']))
				{
					$url .= '&audio=a' . $params['SUPL_AUDIO_ID'];
				}

				if (isset($params['MULTICAST_STREAM']) && $params['MULTICAST_STREAM'] == 'TRUE')
				{
					$url .= '&mcast=1';
				}

				if (isset($params['IFRAMESONLY']) && $params['IFRAMESONLY'] == "1")
				{
					$url .= '&iframesonly=1';
					if (isset($params['IFRAMES_MAXBITRATE']))
					{
						$url .= '&iframes_maxbitrate=' . $params['IFRAMES_MAXBITRATE'];
					}
				}

				if (isset($params['STREAMNUM']) && is_numeric($params['STREAMNUM']))
				{
					$url .= '&streamnum=' . $params['STREAMNUM'];
				}

				if (!empty($params['GRAN']))
				{
					$url .= '&gran=' . $params['GRAN'];
				}

				if (!empty($params['MULT']))
				{
					$url .= '&mult=' . $params['MULT'];
				}

				if (isset($params['SKIP_GAPS']))
				{
					$url .= '&skip_gaps=' . $params['SKIP_GAPS'];
				}

				if (isset($params['STATUS']))
				{
					$url .= '&status=1';
				}
			}
		}

		$params['URL'] = $url;
	}

	/*
	 * Convert string timestamp in format yyyy-mm-dd_hh-mm-ss into integer timestamp
	 * Returns timestamp as int or -1 (if any error happens)
	 *
	 * @param string $strTimestamp
	 * @return int
	 */
	public function parseTimestamp($strTimestamp)
	{
		$intTimestamp = -1;

		if (is_numeric($strTimestamp))
		{
			$intTimestamp = $strTimestamp;
		} else {
			$parts = preg_split("/(-|\s|_)/", $strTimestamp, 6);

			if (count($parts) == 6)
			{
				$intTimestamp = gmmktime($parts[3], $parts[4], $parts[5], $parts[1], $parts[2], $parts[0]);
			}
		}

		if ($intTimestamp == -1)
		{
			throw new Exception("Invalid date: $strTimestamp - expected format 'yyyy-mm-dd_hh-mm-ss' or unix timestamp");
		}

		return $intTimestamp;
	}

	/**
	 * Request authorization ticket
	 *
	 * @param array $params
	 * @throws Exception
	 */
	public function getAuthorizationId(array &$params)
	{
		// Still image does not require authorization at this time
		//
		if ($params['STREAMTYPE'] == 'snapshot')
		{
			return;
		}

		// Get Master-Host Ip address (can be different from where web app is running)
		//
		$params['MASTER_HOST_IP'] = 's_master';

		if (isset($params['STREAMLOCAL']))
		{
			$avatar = new Avatar();
			$attrs = $avatar->getAttributes($params['AVATARID']);

			list($usec, $sec) = explode(" ", microtime());
			$sec += 10;
			$valid_until = "$sec" . sprintf("%06d", $usec * 1000000);

			$params['AUTHID'] = md5($attrs['TICKET_SALT'] . md5($valid_until)) . '.' . $valid_until;
		} else {
			// Submit authorization ticket request
			//
			$strResponse = '';

			if (!($socket = socket_create(AF_INET, SOCK_STREAM, SOL_TCP)))
			{
				throw new Exception('Media authorization error (MCA-0106)'); // 'Error creating connection to TServer: ' . socket_strerror(socket_last_error());
			}

			if (!socket_connect($socket, $params['MASTER_HOST_IP'], 10001))
			{
				throw new Exception('Media authorization error (MCA-0105)'); // 'Connection to TServer failed: ' . socket_strerror(socket_last_error($socket));
			}

			$url = str_replace('&', '&#38;', $params['URL']);
			$strRequest = "<TICKETREQ><RESOURCE XLINK=\"${url}\"/>" .
				"<PRINCIPAL IP=\"${_SERVER['REMOTE_ADDR']}\"/></TICKETREQ>" . chr(26);

			$bytesSent = socket_write($socket, $strRequest, strlen($strRequest));
			if ($bytesSent == false)
			{
				throw new Exception('Media authorization error (MCA-0104)'); //'Data transmission to TServer failed: ' . socket_strerror(socket_last_error($socket));
			}
			if ($bytesSent == strlen($strRequest))
			{
				// Get response
				while (true)
				{
					$strChunk = socket_read($socket, 512);
					if ($strChunk == false)
					{
						throw new Exception('Media authorization error (MCA-0103)'); // 'Data receiving from TServer failed: ' . socket_strerror(socket_last_error($socket));
					}

					// check if complete response is received
					if ($termCharPos = strpos($strChunk, chr(26)))
					{
						$strResponse .= substr($strChunk, 0, $termCharPos);
						break;
					} else {
						$strResponse .= $strChunk;
					}
				}
			} else {
				throw new Exception('');
			}

			if ($socket)
			{
				socket_close($socket);
			}

			$parser = xml_parser_create();
			if (xml_parse_into_struct($parser, $strResponse, $values, $tags) == 0)
			{
				throw new Exception('Media authorization error (MCA-0108)'); // 'Error XML parsing TServer response: ' . $strResponse;
			}
			xml_parser_free($parser);

			foreach ($values as $el)
			{
				if ($el['level'] == 1 && $el['tag'] == 'TICKETRSP' && isset($el['attributes']['TICKET']))
				{
					$params['AUTHID'] = $el['attributes']['TICKET'];
				} else
				if ($el['level'] == 2 && $el['tag'] == 'ERROR')
				{
					if (isset($el['attributes']['MSG']))
					{
						throw new Exception('Media authorization error (MCA-0107)'); // 'TServer error: ' . $el['attributes']['MSG'];
					}
				}
			}

			if (!isset($params['AUTHID']) || empty($params['AUTHID']))
			{
				throw new Exception('Media authorization error (MCA-0109)'); //'Error retriving authorization ticket. TServer response: ' . $strResponse;
			}
		}
	}

	/**
	 * Check if "$userId" has permission "$credential" on object "$objId"
	 *
	 * @param int    $objId object Id
	 * @param string $credential Credential symbol. See table _cred_types for list of credentials;
	 * @param int    $userId (optional) user Id. Default value is $_SESSION[SESSION_USERID]
	 * @throws Exception
	 * @return bool true if permission is granted, false - otherwise.
	 */
	private function _checkCred($objId, $credential, $userId = null)
	{
		// TODO: if request made from domain host - skip verification
		// return true;

		if (empty($userId))
		{
			if (empty($_SESSION[SESSION_USERID]))
			{
				throw new Exception('Error getting user Id');
			} else {
				$userId = $_SESSION[SESSION_USERID];
			}
		}

		$user = new User();
		return $user->checkObjectCredentials($userId, $objId, $credential);
	}
}
