<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: User.php 33215 2015-10-28 00:39:03Z astarostin $
 * ------------------------------------------------------------------------------
 * This class represents logic for User
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions LLC 2006
 * ------------------------------------------------------------------------------
 */

class User extends ObjectTemplate
{
	/**
	 * predefined user obj id's from DB
	 */
	const ADMIN = 21;
	const GUARD = 22;

	public function __construct()
	{
		$this->otype = 'U';
		$this->subtype = '*';
		parent::__construct("user");
	}

	/**
	 * @throws InvalidArgumentException
	 * @param  array $attributes
	 * @param bool $isLDAPUser
	 * @return int
	 */
	public function create(array $attributes, $isLDAPUser = false)
	{
		// check for NAME uniqueness
		if (isset($attributes["NAME"]))
		{
			$value = $attributes["NAME"];
			$checkObj = $this->checkUniqueness(null, "NAME", $value);
			if (isset($checkObj))
			{
				throw new InvalidArgumentException(sprintf(__("user with obj=%d and name=%s already exists"), $checkObj, $value));
			}
		}

		if (!isset($attributes["WARN_VIDEO_PLAYERS"]))
		{
			$attributes["WARN_VIDEO_PLAYERS"] = Identity::getAttribute("WARN_VIDEO_PLAYERS");
		}

		if (!isset($attributes["MAX_VIDEO_PLAYERS"]))
		{
			$attributes["MAX_VIDEO_PLAYERS"] = Identity::getAttribute("MAX_VIDEO_PLAYERS");
		}

		if (!$isLDAPUser)
		{
			// password should be placed in password_history table and not consist in PASSWORD attribute
			$password = null;
			if (!isset($attributes["PASSWORD"]) && !isset($attributes["CONFIRM_PASSWORD"]))
			{
				throw new InvalidArgumentException(__("'Password' and 'Confirm password' attributes should be not empty"));
			}

			$password = $attributes["PASSWORD"];
			$confirmPassword = $attributes["CONFIRM_PASSWORD"];
			if ($password != $confirmPassword)
			{
				throw new InvalidArgumentException("'Password' and 'Confirm password' fields should be equals");
			}

			unset($attributes["PASSWORD"]);
			unset($attributes["CONFIRM_PASSWORD"]);
		}

		$obj = parent::create($attributes);

		if (!$isLDAPUser)
		{
			$this->setPassword($obj, $password);
		}

		// add new created User to default role
		if (isset($attributes["DEFAULT_ROLE"]) && isset($obj))
		{
			$role = new Role();
			$role->add($attributes["DEFAULT_ROLE"], $obj);
		}

		// audit user creation
		$attributes = $this->getAttributes($obj);
		Audit::addRecordVArg(34, $obj, $_SESSION[SESSION_USERID], $attributes["NAME"], "created", Audit::attrValueEntity('EMAIL', $attributes["EMAIL"]));

		return $obj;
	}

	/**
	 * set user attributes
	 *
	 * @param  int   $obj
	 * @param  array $attributes
	 * @throws InvalidArgumentException
	 */
	public function setAttributes($obj, array $attributes)
	{
		// protect some attributes of protected user
		if ((isset($attributes["NAME"]) || isset($attributes["DESCRIPTION"])) && $this->isProtected($obj))
			throw new InvalidArgumentException(sprintf("attribute 'NAME' and 'DESCRIPTION' for user with obj=%s is protected", $obj));

		// check for NAME uniqueness
		if (isset($attributes["NAME"]))
		{
			$checkObj = $this->checkUniqueness($obj, "NAME", $attributes["NAME"]);
			if (isset($checkObj) > 0)
			{
				throw new InvalidArgumentException(sprintf("user with obj=%s and name=%s already exists", $checkObj, $attributes["NAME"]));
			}
		}

		$currentAttributes = $this->getAttributes($obj);
		$auditValues = array();

		parent::setAttributes($obj, $attributes);

		foreach ($attributes as $key => $value)
		{
			array_push($auditValues, Audit::attrChangeEntity($key, $currentAttributes[$key], $value));
		}

		array_push($auditValues, $this->getName($obj));
		array_push($auditValues, "edited");
		// audit user edit
		Audit::addRecord(34, $auditValues, $obj, $_SESSION[SESSION_USERID]);
	}

	/**
	 * delete user
	 *
	 * @param  int  $obj
	 * @param  bool $onlyMark - set deleted = 1 (by default) or delete from DB
	 * @return bool
	 */
	public function delete($obj, $onlyMark = true)
	{
		$username = $this->getName($obj);
		$result = parent::delete($obj, $onlyMark);
		if ($result)
		{
			// audit user delete
			Audit::addRecordVArg(34, $obj, $_SESSION[SESSION_USERID], $username, "deleted");
		}
		return $result;
	}

	/**
	 * @param  string $password
	 * @param  string $hashFunction
	 * @return string
	 */
	private function hashPassword($password, $hashFunction = "sha512")
	{
		return hash($hashFunction, $password);
	}

	/**
	 * select sets which are belongs to user and consists specified object
	 *
	 * @param  int      $userid
	 * @param  int|null $roleid
	 * @param  int|null $obj
	 * @param  bool     $withAttributes return with attributes
	 * @param  bool $withNumber
	 * @param  bool $withCredentials
	 * @throws InvalidArgumentException
	 * @return array list of sets
	 */
	public function getSets($userid, $roleid = null, $obj = null, $withAttributes = false, $withNumber = false, $withCredentials = false)
	{
		if (!isset($userid))
		{
			throw new InvalidArgumentException("Some parameters are missing.");
		}

		// TODO: use more correct query, move to stored procedure
		if ($withCredentials)
		{
			$query =
				"SELECT _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.protected, concatFields(set2role.permission) AS permission, concatFields(permission_type.credentials || set2role.special_credentials) as credentials
				    FROM _objs, permission_type, _links as set2role, _links as user2role, getsets(?, ?, ?) as sets
				    WHERE
				    	sets.obj = _objs.obj
				        AND _objs.deleted = 0
				        AND _objs.otype = 'S'
				        AND set2role.link_type = 'S2R'
				        AND user2role.link_type = 'U2R'
				        AND set2role.obj_cons = user2role.obj_cons
				        AND set2role.permission = permission_type.permission
				        AND user2role.obj_res = ?
				    GROUP BY _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected
				    ORDER BY _objs.name;";

			$list = DB::select($query, array($userid, $roleid, $obj, $userid));
		} else {
			$query = "SELECT * FROM getSets(?, ?, ?);";
			if ($withNumber)
			{
				$query =
					"SELECT
						*, (SELECT count(obj_res) FROM _links WHERE obj_cons = obj) as number
						FROM getSets(?, ?, ?);";
			}

			$list = DB::select($query, array($userid, $roleid, $obj));
		}

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$row["attributes"] = $this->getAttributes($row["obj"]);
			}
		}

		return $list;
	}

	/**
	 * select roles in which user is presented
	 *
	 * @param  int  $userid
	 * @param  bool $withAttributes return with attributes
	 * @throws InvalidArgumentException
	 * @return array list of roles
	 */
	public function getRoles($userid, $withAttributes = false)
	{
		if (!isset($userid))
		{
			throw new InvalidArgumentException("Some parameters are missing.");
		}

		$list = DB::select("SELECT * FROM getRoles(?);", array($userid));

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$row["attributes"] = $this->getAttributes($row["obj"]);
			}
		}

		return $list;
	}

	/**
	 * select object permissions and credentials for user
	 *
	 * @param  int $userid
	 * @param  int $objectid
	 * @throws InvalidArgumentException
	 * @return array
	 */
	public function getObject($userid, $objectid)
	{
		if (!isset($userid) || !isset($objectid))
		{
			throw new InvalidArgumentException("Some parameters are missing.");
		}

		$result = null;

		$list = DB::select("SELECT * FROM getObject(?, ?);", array($userid, $objectid));
		if (count($list) > 0)
			$result = $list[0];

		return $result;
	}

	/**
	 * select object credentials for user
	 *
	 * @param  int $userid
	 * @param  int $objectid
	 * @throws InvalidArgumentException
	 * @return string which contains list of credentials
	 */
	public function getObjectCredentials($userid, $objectid)
	{
		if (!isset($userid) || !isset($objectid))
		{
			throw new InvalidArgumentException("Some parameters are missing.");
		}

		$credentials = '';

		$object = $this->getObject($userid, $objectid);
		if (isset($object))
		{
			$credentials = $object["credentials"];
		}

		return $credentials;
	}

	/**
	 * check if user has credential for specified object
	 * @throws InvalidArgumentException
	 * @param  int $userid
	 * @param  int $objectid
	 * @param  string $credential
	 * @return bool
	 */
	public function checkObjectCredentials($userid, $objectid, $credential = "")
	{
		if (!isset($userid) || !isset($objectid))
		{
			throw new InvalidArgumentException("Some parameters are missing.");
		}

		$result = false;

		$credentials = $this->getObjectCredentials($userid, $objectid);
		if (strpos($credentials, $credential) > -1)
		{
			$result = true;
		}

		return $result;
	}

	/**
	 * get usr obj by name
	 * @param  string $name
	 * @return int|null
	 */
	public function getObj($name)
	{
		$obj = null;

		$list = DB::select("
			SELECT _objs.obj
				FROM _objs
				WHERE
					deleted = 0
					AND _objs.name = ?
					AND _objs.otype = ?
					AND _objs.subtype = ?;",
			array($name, $this->otype, $this->subtype)
		);
		if (count($list) > 0)
		{
			$obj = $list[0]["obj"];
		}

		return $obj;
	}

	/**
	 * select objects with credentials for user with specified otype and subtype
	 *
	 * @param int         $userid
	 * @param int|null    $roleid
	 * @param bool        $withAttributes
	 * @param bool        $withBlocks
	 * @param string|null $otype
	 * @param string|null $subtype
	 * @param string|null $filterAttribute
	 * @param string|null $filterMask
	 * @throws InvalidArgumentException
	 * @return array
	 */
	public function getObjects($userid, $roleid, $withAttributes = false, $withBlocks = false, $otype = null, $subtype = null, $filterAttribute = null, $filterMask = null)
	{
		if (!isset($userid))
		{
			throw new InvalidArgumentException("Some parameters are missing.");
		}

		// TODO: we must modify getting objects list for new type M
		if (isset($otype) && $otype == "M")
		{
			$list = DB::select("
				SELECT
					_objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected, 'M' AS permission, 'LVvApaeDMrCSPsm' AS credentials
					FROM _objs
					WHERE
						otype = ?
						AND subtype = ?
						AND deleted = 0;",
				array($otype, $subtype));
		}
		else
		if (isset($filterAttribute) && isset($filterMask))
		{
			$list = DB::select("
				SELECT objects.obj, name, description, location, otype, subtype, permission, credentials, _obj_attr.val as match
					FROM getObjects(?, ?, ?, ?) as objects, _obj_attr
					WHERE
						objects.obj = _obj_attr.obj
						AND _obj_attr.attr = ?
						AND _obj_attr.val iLIKE ?;",
				array($userid, $roleid, $otype, $subtype, $filterAttribute, "%$filterMask%")
			);
		} else {
			$list = DB::select("SELECT * FROM getObjects(?, ?, ?, ?);", array($userid, $roleid, $otype, $subtype));
		}

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				// add list of linked attributes
				$attributes = $this->getAttributes($row["obj"]);
				$row["attributes"] = $attributes;
			}
		}

		if ($withBlocks)
		{
			foreach ($list as &$row)
			{
				// add list of linked blocks
				$blocks = $this->getBlocksInfo($row["obj"]);
				$row["blocks"] = $blocks;
			}
		}

		return $list;
	}

	/**
	 * select objects from set with credentials for user with specified otype and subtype
	 *
	 * @param  int         $userid
	 * @param  int         $roleid
	 * @param  int         $setid
	 * @param  bool        $withAttributes
	 * @param  bool        $withBlocks
	 * @param  string|null $otype
	 * @param  string|null $subtype
	 * @param  string|null $filterAttribute
	 * @param  string|null $filterMask
	 * @throws InvalidArgumentException
	 * @return array
	 */
	public function getObjectsFromSet($userid, $roleid, $setid, $withAttributes = false, $withBlocks = false, $otype = null, $subtype = null, $filterAttribute = null, $filterMask = null)
	{
		if (!isset($userid) || !isset($setid))
		{
			throw new InvalidArgumentException("Some parameters are missing.");
		}

		// TODO: we must modify gettting objects list for new type M
		if (isset($otype) && $otype == "M")
		{
			$list = DB::select("
				SELECT
					_objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected, 'M' AS permission, 'LVvApaeDMrCSPsm' AS credentials
					FROM _objs
					WHERE
						otype = ?
						AND subtype = ?
						AND deleted = 0;",
				array($otype, $subtype));
		}
		else
		if (isset($filterAttribute) && isset($filterMask))
		{
			$list = DB::select("
				SELECT objects.obj, name, description, location, otype, subtype, permission, credentials, _obj_attr.val as match
					FROM getObjects(?, ?, ?, ?, ?) as objects, _obj_attr
					WHERE
						objects.obj = _obj_attr.obj
						AND _obj_attr.attr = ?
						AND _obj_attr.val iLIKE ?;",
				array($userid, $roleid, $setid, $otype, $subtype, $filterAttribute, "%$filterMask%")
			);
		} else {
			$list = DB::select("SELECT * FROM getObjects(?, ?, ?, ?, ?);", array($userid, $roleid, $setid, $otype, $subtype));
		}

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				// add list of linked attributes
				$attributes = $this->getAttributes($row["obj"]);
				$row["attributes"] = $attributes;
			}
		}

		if ($withBlocks)
		{
			foreach ($list as &$row)
			{
				// add list of linked blocks
				$blocks = $this->getBlocksInfo($row["obj"]);
				$row["blocks"] = $blocks;
			}
		}

		return $list;
	}

	/**
	 * return list of guis for user
	 * @param  int $userid
	 * @param  int|null $roleid
	 * @param  int|null $roleid
	 * @return array list of guis
	 */
	public function getGuis($userid, $roleid = null)
	{
		$list = $this->getObjects($userid, $roleid, true, false, 'G');

		return $list;
	}

	/**
	 * @param  int|null $obj
	 * @param  array $attributes
	 * @return array
	 */
	public function getTemplateAttributes($obj = null, array $attributes = array())
	{
		if (!isset($obj))
		{
			if (!isset($attributes["WARN_VIDEO_PLAYERS"]))
			{
				$attributes["WARN_VIDEO_PLAYERS"] = Identity::getAttribute("WARN_VIDEO_PLAYERS");
			}
			if (!isset($attributes["MAX_VIDEO_PLAYERS"]))
			{
				$attributes["MAX_VIDEO_PLAYERS"] = Identity::getAttribute("MAX_VIDEO_PLAYERS");
			}
		}

		return parent::getTemplateAttributes($obj, $attributes);
	}

	/**
	 * @return array
	 */
	public function getObjectCountOnSystemSets(){
		$list = DB::select("
			SELECT count(o.obj) as cnt, s.obj as set, s.name
			FROM _objs o, _obj_otype_subtype oos, _objs s
			WHERE o.otype=oos.otype and o.subtype=oos.subtype and oos.defaultsetobj=s.obj and o.deleted=0 and (o.otype = 'D' or o.otype = 'X')
			GROUP BY  s.obj, s.name;
			");
		return $list;
	}

	/**
	 * check password strength with cracklib
	 *
	 * @param  string $password
	 * @throws InvalidArgumentException
	 * @throws Exception
	 * @return bool return true if password is strength enough
	 */
	private function _checkPasswordWithCracklib($password)
	{
		// Clean up password
		$password = str_replace("\r", "", $password);
		$password = str_replace("\n", "", $password);

		// Run password through cracklib-check
		exec("echo " . escapeshellarg($password) . " | cracklib-check 2>/dev/null", $output, $return_var);

		// Check it ran properly
		if ($return_var !== 0)
		{
			// Some sort of execution error
			throw new Exception("Failed to run cracklib-check.");
		}

		if (!preg_match("/^.*\: ([^:]+)$/", $output[0], $matches))
		{
			// Badly formatted response from cracklib-check.
			throw new Exception("Didn't understand cracklib-check response.");
		}

		// Check response
		if (strtoupper($matches[1]) !== "OK")
		{
			$message = $matches[1];
			throw new InvalidArgumentException($message);
		}

		return true;
	}

	/**
	 * check password strength
	 *
	 * @param  int    $userid
	 * @param  string $newPassword
	 * @param  string $currentPassword
	 * @throws InvalidArgumentException
	 * @return bool return true if password is strength enough
	 */
	public function checkPasswordStrength($userid, $newPassword, $currentPassword = null)
	{
		// Clean up password
		$newPassword = trim($newPassword);

		$parameters = $this->getPasswordParameters();

		// Minimum of N characters
		$parameters["MIN_LENGTH"] = intval($parameters["MIN_LENGTH"], 10);
		if (strlen($newPassword) < $parameters["MIN_LENGTH"])
		{
			throw new InvalidArgumentException(sprintf(__("Password must be at least %d characters"), $parameters["MIN_LENGTH"]));
		}

		$lowercaseNum = 0;
		$uppercaseNum = 0;
		$numberNum = 0;
		$symbolNum = 0;
		for ($i = 0;$i < strlen($newPassword); $i++)
		{
			$char = $newPassword[$i];

			if (ctype_digit($char))
			{
				$numberNum++;
			}
			elseif (ctype_lower($char))
			{
				$lowercaseNum++;
			}
			elseif (ctype_upper($char))
			{
				$uppercaseNum++;
			}
			elseif (ctype_punct($char))
			{
				$symbolNum++;
			}
		}

		// At least N lower case letter
		$parameters["LOWERCASE_NUM"] = intval($parameters["LOWERCASE_NUM"], 10);
		if ($lowercaseNum < $parameters["LOWERCASE_NUM"] && $parameters["LOWERCASE_NUM"] > 0)
		{
			throw new InvalidArgumentException(sprintf(__("Password must include at least %d lower case letter(s)"), $parameters["LOWERCASE_NUM"]));
		}

		// At least N upper case letter
		$parameters["UPPERCASE_NUM"] = intval($parameters["UPPERCASE_NUM"], 10);
		if ($uppercaseNum < $parameters["UPPERCASE_NUM"] && $parameters["UPPERCASE_NUM"] > 0)
		{
			throw new InvalidArgumentException(sprintf(__("Password must include at least %d upper case letter(s)"), $parameters["UPPERCASE_NUM"]));
		}

		// At least N number
		$parameters["NUMBER_NUM"] = intval($parameters["NUMBER_NUM"], 10);
		if ($numberNum < $parameters["NUMBER_NUM"] && $parameters["NUMBER_NUM"] > 0)
		{
			throw new InvalidArgumentException(sprintf(__("Password must include at least %d number(s)"), $parameters["NUMBER_NUM"]));
		}

		// At least N symbol
		$parameters["SYMBOL_NUM"] = intval($parameters["SYMBOL_NUM"], 10);
		if ($symbolNum < $parameters["SYMBOL_NUM"] && $parameters["SYMBOL_NUM"] > 0)
		{
			throw new InvalidArgumentException(sprintf(__("Password must include at least %d symbol(s)"), $parameters["SYMBOL_NUM"]));
		}

        // Dictionary a.k.a cracklib check (dictionary words, names, etc)
		if (Template::boolVal($parameters["CRACKLIB_CHECK"]))
		{
			$result = $this->_checkPasswordWithCracklib($newPassword);
		}

		// Differs from previous password by at least N characters
		if (isset($currentPassword))
		{
			$parameters["DIFF_FROM_PREVIOUS_PASSWORD_NUM"] = intval($parameters["DIFF_FROM_PREVIOUS_PASSWORD_NUM"], 10);
			$diffNum = 0;
			$currentPasswordCharMap = array();
			for ($i = 0;$i < strlen($currentPassword); $i++)
			{
				$char = $currentPassword[$i];
				$currentPasswordCharMap[$char] = true;
			}
			for ($i = 0;$i < strlen($newPassword); $i++)
			{
				$char = $newPassword[$i];
				if (!isset($currentPasswordCharMap[$char]))
				{
					$diffNum++;
				}
			}
			if ($diffNum < $parameters["DIFF_FROM_PREVIOUS_PASSWORD_NUM"])
			{
				throw new InvalidArgumentException(sprintf(__("New password must differ from previous password by at least %d characters"), $parameters["DIFF_FROM_PREVIOUS_PASSWORD_NUM"]));
			}
		}

		// Not one of their last N passwords
		$parameters["DIFF_FROM_PREVIOUS_PASSWORDS"] = intval($parameters["DIFF_FROM_PREVIOUS_PASSWORDS"], 10);
		$newPasswordHash = $this->hashPassword($newPassword);
		$passwordHistory = $this->_getPasswordHistory($userid, $parameters["DIFF_FROM_PREVIOUS_PASSWORDS"]);

		$node = new Node();
		$hashFunction = $node->isLegacySupport() ? "sha1" : "sha512";

		foreach ($passwordHistory as $row)
		{
			$password = json_decode($row["password"], true);

			if ($password[$hashFunction] == $newPasswordHash)
			{
				throw new InvalidArgumentException(sprintf(__("New password must be different than previous %d passwords"), $parameters["DIFF_FROM_PREVIOUS_PASSWORDS"]));
			}
		}

		return true;
	}

	/**
	 * get password parameters
	 *
	 * @return array
	 */
	public function getPasswordParameters()
	{
		$parameters = array();

		$passwordConfigPath = APL_PATH . "/conf/etc/user/etc/password.cfg";
		$passwordConfigLegacyPath = APL_PATH . "/conf/etc/user/etc/password-legacy.cfg";
		if (is_file($passwordConfigLegacyPath))
		{
			$passwordConfigPath = $passwordConfigLegacyPath;
		}
		// read parameters from file
		$rows = file($passwordConfigPath, FILE_IGNORE_NEW_LINES|FILE_SKIP_EMPTY_LINES);
		foreach ($rows as $row)
		{
			// skip comments
			if ($row[0] == "#")
			{
				continue;
			}

			list($parameter, $value) = preg_split("/=/", $row, 2);
			$parameters[$parameter] = $value;
		}

		return $parameters;
	}

	/**
	 * get password history for user
	 *
	 * @param int $userid
	 * @param int $limit
	 * @return array
	 */
	private function _getPasswordHistory($userid, $limit = 1)
	{
		$list = DB::select("SELECT userid, password, extract(epoch from changed) as timestamp, expired FROM password_history WHERE userid = ? ORDER BY changed DESC LIMIT ?;", array($userid, $limit));
		return $list;
	}

	/**
	 * @param int $userid
	 * @return array
	 */
	private function _getLastPasswordHistoryItem($userid)
	{
		$passwordHistory = $this->_getPasswordHistory($userid);
		$last = null;
		if (count($passwordHistory) > 0)
			$last = $passwordHistory[0];
		return $last;
	}

	/**
	 * get current user password hash
	 *
	 * @param int $userid
	 * @param string $hashFunction
	 * @return string|null
	 */
	public function getPasswordHash($userid, $hashFunction = "sha512")
	{
		$passwordHistoryItem = $this->_getLastPasswordHistoryItem($userid);
		$password = json_decode($passwordHistoryItem["password"], true);
		return isset($password[$hashFunction]) ? $password[$hashFunction] : "";
	}

	/**
	 * set new password
	 *
	 * @param int    $userid
	 * @param string $password
	 * @param bool   $isExpired
	 * @throws InvalidArgumentException
	 * @return bool
	 */
	public function setPassword($userid, $password, $isExpired = false)
	{
		$result = false;

		if (!$this->checkPasswordStrength($userid, $password))
		{
			throw new InvalidArgumentException(__("New password not strong enough"));
		}

		$passwordInfo = array(
			"sha512" => $this->hashPassword($password)
		);
		$node = new Node();
		if ($node->isLegacySupport())
		{
			$passwordInfo["sha1"] = $this->hashPassword($password, "sha1");
		}

		$expired = $isExpired ? 1 : 0;
		$number = DB::query("INSERT INTO password_history(userid, password, expired) VALUES(?, ?, ?);", array($userid, json_encode($passwordInfo), $expired));
		if ($number > 0)
		{
			$result = true;
		}

		$this->updateSTime($userid);

		return $result;
	}

	/**
	 * set new password with checking the current one
	 *
	 * @param int    $userid
	 * @param string $currentPassword
	 * @param string $newPassword
	 * @throws InvalidArgumentException
	 * @return bool
	 */
	public function changePassword($userid, $currentPassword, $newPassword)
	{
		if ($newPassword == "")
		{
			throw new InvalidArgumentException(__("New password should not be empty"));
		}

		$currentPasswordHash = $this->hashPassword($currentPassword);
		$currentPasswordHashOfUser = $this->getPasswordHash($userid);

		if ($currentPasswordHashOfUser != $currentPasswordHash)
		{
			throw new InvalidArgumentException(__("Incorrect current password"));
		}

		if (!$this->checkPasswordStrength($userid, $newPassword, $currentPassword))
		{
			throw new InvalidArgumentException(__("New password not strong enough"));
		}

		return $this->setPassword($userid, $newPassword);
	}

	/**
	 * user can change account password once every N hours
	 *
	 * @param int $userid
	 * @param int $changePasswordInterval
	 * @return bool
	 */
	public function isAbleToChangePassword($userid, &$changePasswordInterval)
	{
		$parameters = $this->getPasswordParameters();
		$changePasswordInterval = intval($parameters["CHANGE_PASSWORD_INTERVAL"], 10);

		$passwordInfo = $this->_getLastPasswordHistoryItem($userid);
		$lastChangedTimestamp = intval($passwordInfo["timestamp"], 10);
		$isExpired = $passwordInfo["expired"] == "1";

		return $isExpired || (time() - $lastChangedTimestamp >= $changePasswordInterval * 60 * 60);
	}

	public function isPasswordExpired($userid)
	{
		$LDAP_USER = $this->getAttribute($userid, "LDAP_USER");
		if (!empty($LDAP_USER))
		{
			return false;
		}

		$PASSWORD_EXPIRATION = intval(Identity::getAttribute("PASSWORD_EXPIRATION"), 10); // days

		$passwordInfo = $this->_getLastPasswordHistoryItem($userid);
		$lastChangedTimestamp = 0;
		$isExpired = false;
		if (isset($passwordInfo))
		{
			$lastChangedTimestamp = intval($passwordInfo["timestamp"], 10);
			$isExpired = intval($passwordInfo["expired"], 10) == 1;
		}

		return $isExpired || time() - $lastChangedTimestamp >= $PASSWORD_EXPIRATION * 24 * 60 * 60;
	}

	/**
	 * check if user is in Admin role
	 *
	 * @param int $userid
	 * @return bool
	 */
	public function isAdmin($userid)
	{
		$roles = $this->getRoles($userid);
		foreach ($roles as $role)
		{
			if ($role["obj"] == Role::ADMIN)
			{
				return true;
			}
		}

		return false;
	}

	/**
	 * TODO: check is this method needed
	 * check if user has special credentials
	 *
	 * @param int $userid
	 *
	 * @return bool
	 */
	public function hasSpecialCredentials($userid)
	{
		// has special credentials
		if($this->checkObjectCredentials($userid, Obj::CONTROL_PANEL, "u")
		   || $this->checkObjectCredentials($userid, Obj::CONTROL_PANEL, "U")
		   || $this->checkObjectCredentials($userid, Obj::CONTROL_PANEL, "f")
		   || $this->checkObjectCredentials($userid, Obj::CONTROL_PANEL, "F")
		   || $this->checkObjectCredentials($userid, Obj::CONTROL_PANEL, "R")
		   || $this->checkObjectCredentials($userid, Obj::CONTROL_PANEL, "E"))
		{
			return true;
		}

		return false;
	}

	/**
	 * @param string $userName
	 * @param string|null $password
	 * @throws Exception
	 */
	public function syncRolesWithLDAP($userName, $password = null)
	{
		$identity = Identity::getAttributes();
		$LDAP_AUTHENTICATION = Template::boolVal($identity["LDAP_AUTHENTICATION"]);
		if ($LDAP_AUTHENTICATION)
		{
			$ldapUserName = $userName;
			$ldapPassword = $password;
			if (!isset($password))
			{
				$ldapUserName = $identity["LDAP_USER"];
				$ldapPassword = $identity["LDAP_PASSWORD"];
			}
			$ldap = new LDAP();
			$ldap->auth($ldapUserName, $ldapPassword);
			$groupList = $ldap->getGroups($userName);

			$user = new User();
			$userObj = $user->getObj($userName);
			$role = new Role();

			// remove all roles from use
			// TODO: remove only not used roles
			$roleList = $user->getRoles($userObj);
			foreach ($roleList as $userRole)
			{
				$role->remove($userRole["obj"], $userObj);
			}

			// add user to roles
			foreach ($groupList as $group)
			{
				$groupName = $group["cn"];

				$result = DB::select("SELECT obj FROM _obj_attr WHERE attr = 'LDAP_GROUP' AND val = ?;", array($groupName));
				if (count($result) > 0)
				{
					foreach ($result as $row)
					{
						$roleObj = $row["obj"];
						$role->add($roleObj, $userObj);
					}
				}
			}
		}
	}

	/**
	 * check is user accepted "License Agreement"
	 *
	 * @param int $userid
	 * @return bool
	 */
	public function isLicenseAccepted($userid)
	{
		$LICENSE_HASH = $this->getAttribute($userid, "LICENSE_HASH");

		if (empty($LICENSE_HASH))
		{
			return false;
		}

		$node = new Node();
		$licenseHash = $node->getLicenseHash();

		return ($licenseHash == $LICENSE_HASH);
	}

	/**
	 * accept "License Agreement" for specified user
	 *
	 * @param int $userid
	 * @return bool
	 */
	public function acceptLicense($userid)
	{
		$node = new Node();
		$licenseHash = $node->getLicenseHash();

		$this->setAttributes($userid, array('LICENSE_HASH' => $licenseHash));
	}
}
