<?php defined('APL_PATH') or die('No direct script access.');
/*
# $Id: Template.php 33519 2016-01-18 19:50:01Z atsybulnik $
#------------------------------------------------------------------------------
# This class represents logic for Templates
#------------------------------------------------------------------------------
# Author: Sergey Tsybulsky
# Edited by: Andrey Starostin
# QA by:
# Copyright: videoNEXT Network Solutions LLC 2006
#-----------------------------------------------------------------------------
*/

/*
 * example of using
 * $tmpl = new Template('camera');
 * $tmpl->getDefaultAttributes('axis','215');
 *
 * */

require_once APL_PATH . '/api/lib/php/gettext.php';

class Template
{
	private $SKM_VERSION = "";
	private $type;
	private static $formatDelimiter = '/(?<!\\\)~/';
	private static $commaDelimiter = '/(?<!\\\),/';
	private static $colonDelimiter = '/(?<!\\\):/';
	private static $modelIdPrefix = 'MODELID:';

	private $_templatesDirectory = "";
	private $_defaultTemplatesDirectory = "";

	/**
	 * @param string $type camera, relay, sensor_skm, audio, monitor, wall, external, node, gui, configuration, vmxlayout, model3d, modellayer, set, role, user, gateway, sensor_acsg, door, perimeter, area, avatar, posture, timecard, schedule
	 */
	public function __construct($type)
	{
		$this->SKM_VERSION = Node::getBrand();
		$this->type = $type;

		$this->_defaultTemplatesDirectory = APL_PATH . "/conf/etc";
		$this->_templatesDirectory = $this->_defaultTemplatesDirectory;
	}

	/**
	 * set templates directory path
	 *
	 * @param $path
	 */
	public function setTemplatesDirectory($path)
	{
		$this->_templatesDirectory = $path;
	}

	/**
	 * get template path
	 *
	 * @param string $type
	 * @param string $objectModel                                                                           `
	 * @param bool   $useSKM_VERSION
	 * @param bool   $isDefaultDir
	 * @return string
	 */
	private function _getTemplatePath($type, $objectModel = "default", $useSKM_VERSION = false, $isDefaultDir = false)
	{
		$templatePath = $this->_templatesDirectory . "/" . $type;

		if ($type == 'camera'
		    || $type == 'door'
		    || $type == 'gateway'
		    || $type == 'relay'
		    || $type == 'sensor_acsg'
		    || $type == 'sensor_skm'
		)
		{
			$templatePath = $this->_templatesDirectory;
			if (strpos($this->_templatesDirectory, APL_PATH, 0) === 0)
			{
				$templatePath .= "/device";
			}
			if (!$isDefaultDir)
			{
				$templatePath .= "/" . $type . "/" . $objectModel . "/etc/edit";
			} else {
				$templatePath .= "/default/etc/edit";
			}
		} else
		if ($type == 'schedule' || $type == 'posture' || $type == 'timecard')
		{
			$templatePath = $this->_templatesDirectory . "/schedule/" . $type . "/" . $objectModel . "/etc/edit";
		} else
		if ($type == 'monitor' || $type == 'wall' || $type == 'vmxlayout')
		{
			$templatePath = $this->_templatesDirectory . "/vmx/" . $type . "/" . $objectModel . "/etc/edit";
		} else
		if ($type == 'user')
		{
			$templatePath = $this->_templatesDirectory . "/" . $type . "/etc/edit";
		} else
		if ($type == 'identity')
		{
			$templatePath = $this->_templatesDirectory . "/identity/etc/edit";
		}

		if ($useSKM_VERSION)
		{
			$templatePath .= "." . $this->SKM_VERSION . ".cfg";
		} else {
			$templatePath .= ".cfg";
		}

		return $templatePath;
	}

	/**
	 * Return default attributes of camera from cartridge template
	 *
	 * @param  string|null $objectModel axis, sony etc.
	 * @param  string|null $modelid 211, 215, 207 etc.
	 * @return array array(attribute => value)
	 */
	public function getDefaultAttributes($objectModel = null, $modelid = null)
	{
		$attributes = $this->_getDefaultAttributes(array(), $objectModel, $modelid);

		if ($this->type == "camera")
		{
			$VAEList = Node::getVAEList();
			foreach($VAEList as $vae => $path)
			{
				$this->setTemplatesDirectory("$path/etc/");

				// use edit.cfg from VAE cartridge
				$attributes = $this->_getDefaultAttributes($attributes, $objectModel, $modelid);

				$this->setTemplatesDirectory($this->_defaultTemplatesDirectory);
			}
		}

		return $attributes;
	}

	/**
	 * Return default attributes of camera from cartridge template
	 *
	 * @param array $templateArray
	 * @param  string|null $objectModel axis, sony etc.
	 * @param  string|null $modelid 211, 215, 207 etc.
	 * @return array array(attribute => value)
	 */
	private function _getDefaultAttributes(array $templateArray = array(), $objectModel = null, $modelid = null)
	{
		// use /default/etc/edit.cfg template
		$path = $this->_getTemplatePath($this->type, "default", false, true);
		if (is_file($path))
		{
			$templateArray = $this->_getKeyValueFromFile($path, $templateArray);
		}

		// use /etc/edit.cfg template
		$path = $this->_getTemplatePath($this->type, "default", false);
		$file = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);

		$is_model_in_cfg = 0;
		$was_modelid = 0;
		foreach ($file as $str)
		{
			$str = trim($str);
			if ($str == '')
				continue;
			if ($str[0] == '#')
				continue;

			if (strpos($str, self::$modelIdPrefix) === 0)
			{
				$was_modelid = 1;
				$str = substr($str, strlen(self::$modelIdPrefix));
				$modelid_arr = preg_split(self::$commaDelimiter, $str);
				$is_model_in_cfg = 0;
				foreach ($modelid_arr as $mid)
				{
					$mid = trim($mid);
					$mid = stripslashes($mid);
					if ($modelid == $mid)
					{
						$is_model_in_cfg = 1;
						break;
					}
				}
				continue;
			}

			if (!$was_modelid || ($was_modelid && $is_model_in_cfg))
			{
				list($ATTR, $DEFAULT_VALUE) = $this->_getKeyValue($str);
				if (!isset($ATTR) || !isset($DEFAULT_VALUE))
				{
					continue;
				}

				$templateArray[$ATTR] = $DEFAULT_VALUE;
			}
		}

		// use /etc/edit.SKM.cfg template
		$templateSKMPath = $this->_getTemplatePath($this->type, "default", true);
		if (is_file($templateSKMPath))
		{
			$templateArray = $this->_getKeyValueFromFile($templateSKMPath, $templateArray);
		}

		// use /$objectModel/etc/edit.cfg template
		$templateModelPath = $this->_getTemplatePath($this->type, $objectModel, false);
		if (isset($objectModel) && is_file($templateModelPath))
		{
			$templateArray = $this->_useObjectModel($templateModelPath, $templateArray, $modelid);
		}

		// use /$objectModel/etc/edit.SKM.cfg template
		$templateSKMModelPath = $this->_getTemplatePath($this->type, $objectModel, true);
		if (isset($objectModel) && is_file($templateSKMModelPath))
		{
			$templateArray = $this->_useObjectModel($templateSKMModelPath, $templateArray, $modelid);
		}

		return $templateArray;
	}

	/**
	 * @param $str
	 *
	 * @return array [key, value]
	 */
	private function _getKeyValue($str)
	{
		list($SECTION, $ATTR, $TYPE, $LABEL, $DEFAULT_VALUE, $LIST, $VALIDATE_FUNCTION, $ERROR_MESSAGE) = preg_split(self::$formatDelimiter, $str);
		if (!isset($DEFAULT_VALUE))
		{
			$DEFAULT_VALUE = "";
		}
		if (preg_match("/^textfield/", $TYPE))
		{
			$DEFAULT_VALUE = __($DEFAULT_VALUE, "template");
		}
		if ($TYPE === "stat"
			|| $TYPE === "system")
		{
			$DEFAULT_VALUE = "";
		}
		if ($TYPE === "urlbutton"
			|| $TYPE === "checkboxlist"
			|| $TYPE === "refview")
		{
			$ATTR = null;
			$DEFAULT_VALUE = null;
		}

		return array($ATTR, $DEFAULT_VALUE);
	}

	private function _getKeyValueFromFile($path, array $templateArray = array())
	{
		$templateArray = array();
		if (is_file($path))
		{
			$file = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
			foreach ($file as $str)
			{
				$str = trim($str);
				if ($str == '')
					continue;
				if ($str[0] == '#')
					continue;

				list($ATTR, $DEFAULT_VALUE) = $this->_getKeyValue($str);
				if (!isset($ATTR) || !isset($DEFAULT_VALUE))
				{
					continue;
				}

				$templateArray[$ATTR] = $DEFAULT_VALUE;
			}
		}
		return $templateArray;
	}

	/**
	 * Return default values for all camera attributes
	 *
	 * @param int|null $devid
	 * @param null $nodeid
	 * @param array $templateArray
	 * @param array $presetAttributes
	 * @param bool $isDefaultDir
	 *
	 * @throws Exception
	 * @return array
	 */
	private function _getAttributes($devid = null, $nodeid = null, array $templateArray = array(), array $presetAttributes = array(), $isDefaultDir = false)
	{
		$templatePath = $this->_getTemplatePath($this->type, "default", false, $isDefaultDir);

		if (is_file($templatePath))
		{
			$file = file($templatePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
			$oldSection = '';
			$attr_num = 0;
			$section_num = 0;

			foreach ($file as $str)
			{
				$str = trim($str);
				// skip blank lines and comments
				if ($str == '' || $str[0] == '#')
					continue;
				if (strpos($str, self::$modelIdPrefix) === 0)
					break;

				list($SECTION, $ATTR, $TYPE, $LABEL, $DEFAULT_VALUE, $LIST, $VALIDATE_FUNCTION, $ERROR_MESSAGE) = preg_split(self::$formatDelimiter, $str);
				if (!isset($ERROR_MESSAGE))
				{
					throw new Exception(print_r($str ,true));
				}

				if (!isset($DEFAULT_VALUE))
					$DEFAULT_VALUE = '';

				if (!isset($LIST))
					$LIST = '';

				$list_str = 'LIST';
				$listValue = $this->_getListValue($LIST, $devid, $nodeid, $presetAttributes);
				$LIST = $listValue[0];
				$list_str = $listValue[1];

				if (!isset($VALIDATE_FUNCTION))
					$VALIDATE_FUNCTION = '';
				if (!isset($ERROR_MESSAGE))
					$ERROR_MESSAGE = '';

				$valid_js = 'VALIDATE';
				$validateValue = $this->_getValidateValue($VALIDATE_FUNCTION);
				$VALIDATE_FUNCTION = $validateValue[0];
				$valid_js = $validateValue[1];

				if ($SECTION == $oldSection)
					$attr_num++;
				else
					$attr_num = 0;

				if ($TYPE == "refview")
				{
					$userid = $_SESSION[SESSION_USERID];
					$DEFAULT_VALUE = str_ireplace(
						array('$DEVID$', '$NODEID$', '$USERID$'),
						array($devid, $nodeid, $userid),
						$DEFAULT_VALUE
					);
				}
				$attr_val = array(
					'ORDER' => $attr_num,
					'VALUE' => $DEFAULT_VALUE,
					'TYPE' => $TYPE,
					'LABEL' => __($LABEL, "template"),
					$list_str => $LIST,
					$valid_js => $VALIDATE_FUNCTION,
					'ERROR' => __($ERROR_MESSAGE, "template")
				);
				if ($attr_val['TYPE'] == "urlbutton")
				{
					$attr_val['VALUE'] = __($attr_val['VALUE'], "template");
				}

				if ($SECTION != $oldSection)
				{
					$oldSection = $SECTION;
					$section_num++;

					if (!isset($templateArray[$SECTION]))
					{
						$templateArray[$SECTION] = array(
							'ORDER' => $section_num,
							'ATTR' => array()
						);
					}
				}
				$templateArray[$SECTION]['ATTR'][$ATTR] = $attr_val;
			}
		}
		return $templateArray;
	}

	private function _isModelInCFG($file, $modelid)
	{
		$isModelInCfg = false;
		foreach ($file as $str)
		{
			$str = trim($str);
			if (strpos($str, self::$modelIdPrefix) === 0)
			{
				$str = substr($str, strlen(self::$modelIdPrefix));
				$modelid_arr = preg_split(self::$commaDelimiter, $str);
				foreach ($modelid_arr as $mid)
				{
					$mid = trim($mid);
					$mid = stripslashes($mid);
					if ($modelid == $mid)
					{
						$isModelInCfg = true;
						break;
					}
				}
			}
			if ($isModelInCfg)
				break;
		}
		return $isModelInCfg;
	}

	/**
	 * @param string $devid
	 * @param $nodeid
	 * @param array $allAttributes
	 * @param bool $useSKM_VERSION
	 * @param string $objectModel
	 * @param string $modelid
	 * @param array $presetAttributes
	 * @return mixed
	 */
	private function _getModelAttributes($devid, $nodeid, array $allAttributes, $useSKM_VERSION = false, $objectModel = "default", $modelid, array $presetAttributes = array())
	{
		$templatePath = $this->_getTemplatePath($this->type, $objectModel, $useSKM_VERSION);

		if (is_file($templatePath))
		{
			if (($file = file($templatePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES)) !== FALSE)
			{
				// Check: is modelID in cfg file.
				$isModelInCFG = $this->_isModelInCFG($file, $modelid);

				// Parse CFG file
				$inputflag = 0;
				foreach ($file as $str)
				{
					$str = trim($str);
					if ($str == '' || $str[0] == '#')
						continue;

					if (strpos($str, self::$modelIdPrefix) === 0)
					{
						$str = substr($str, strlen(self::$modelIdPrefix));
						$modelid_arr = preg_split(self::$commaDelimiter, $str);
						$inputflag = 2;

						foreach ($modelid_arr as $mid)
						{
							$mid = trim($mid);
							$mid = stripslashes($mid);

							if ($isModelInCFG && $modelid == $mid
							    || stripos($modelid, $mid, 0) === 0)
							{
								$inputflag = 1;
								break;
							}
						}
						continue;
					} else
					if ($inputflag == 0 || $inputflag == 1)
					{
						list($SECTION, $ATTR, $TYPE, $LABEL, $DEFAULT_VALUE, $LIST, $VALIDATE_FUNCTION, $ERROR_MESSAGE) = preg_split(self::$formatDelimiter, $str);

						if (!isset($DEFAULT_VALUE))
							$DEFAULT_VALUE = '';
						if (!isset($LIST))
							$LIST = '';

						$list_str = 'LIST';
						$listValue = $this->_getListValue($LIST, $devid, $nodeid, $presetAttributes);
						$LIST = $listValue[0];
						$list_str = $listValue[1];

						if (!isset($VALIDATE_FUNCTION))
							$VALIDATE_FUNCTION = '';
						if (!isset($ERROR_MESSAGE))
							$ERROR_MESSAGE = '';

						if ($TYPE == "refview")
						{
							$userid = $_SESSION[SESSION_USERID];
							$DEFAULT_VALUE = str_ireplace(
								array('$DEVID$', '$NODEID$', '$USERID$'),
								array($devid, $nodeid, $userid),
								$DEFAULT_VALUE
							);
						}

						$valid_js = 'VALIDATE';
						$validateValue = $this->_getValidateValue($VALIDATE_FUNCTION);
						$VALIDATE_FUNCTION = $validateValue[0];
						$valid_js = $validateValue[1];

						if (isset($allAttributes[$SECTION]['ATTR'][$ATTR]))
						{
							$curnum = $allAttributes[$SECTION]['ATTR'][$ATTR]['ORDER'];
						} else {
							if (!isset($allAttributes[$SECTION]['ATTR']))
							{
								$curnum = 0;
							} else {
								$curnum = count($allAttributes[$SECTION]['ATTR']);
							}
						}

						$attr_val = array(
							'ORDER' => $curnum,
							'VALUE' => $DEFAULT_VALUE,
							'TYPE' => $TYPE,
							'LABEL' => __($LABEL, "template"),
							$list_str => $LIST,
							$valid_js => $VALIDATE_FUNCTION,
							'ERROR' => __($ERROR_MESSAGE, "template")
						);
						if ($attr_val['TYPE'] == "urlbutton")
						{
							$attr_val['VALUE'] = __($attr_val['VALUE'], "template");
						}

						$allAttributes[$SECTION]['ATTR'][$ATTR] = $attr_val;
					}
				}
			}
		}
		return $allAttributes;
	}

	/**
	 * @param array $allAttributes
	 * @param string $cameramodel
	 * @param string $modelid
	 * @param string $devid
	 * @param $nodeid
	 * @param array $presetAttributes
	 * @return array
	 */
	private function _getTemplateAttributes(array $allAttributes, $cameramodel = "default", $modelid = '', $devid = null, $nodeid = null, array $presetAttributes = array())
	{
		// use /default/edit.cfg template
		$allAttributes = $this->_getAttributes($devid, $nodeid, $allAttributes, $presetAttributes, true);

		// use /edit.cfg template
		$allAttributes = $this->_getAttributes($devid, $nodeid, $allAttributes, $presetAttributes);

		// use /edit.cfg template with modelid
		$allAttributes = $this->_getModelAttributes($devid, $nodeid, $allAttributes, false, $cameramodel, $modelid, $presetAttributes);

		// use /edit.SKM.cfg template with modelid
		$allAttributes = $this->_getModelAttributes($devid, $nodeid, $allAttributes, true, $cameramodel, $modelid, $presetAttributes);

		return $allAttributes;
	}

	/**
	 * @param string $cameraModel
	 * @param string $modelid
	 * @param string $devid
	 * @param $nodeid
	 * @param array $presetAttributes
	 * @return array
	 */
	public function getTemplateAttributes($cameraModel = "default", $modelid = '', $devid = null, $nodeid = null, array $presetAttributes = array())
	{
		$allAttributes = $this->_getTemplateAttributes(array(), $cameraModel, $modelid, $devid, $nodeid, $presetAttributes);

		if ($this->type == "camera")
		{
			$VAEList = Node::getVAEList();
			foreach($VAEList as $vae => $path)
			{
				$this->setTemplatesDirectory("$path/etc/");

				// use edit.cfg from VAE cartridge
				$allAttributes = $this->_getTemplateAttributes($allAttributes, $cameraModel, $modelid, $devid, $nodeid, $presetAttributes);

				$this->setTemplatesDirectory($this->_defaultTemplatesDirectory);
			}
		}

		return $allAttributes;
	}

	/**
	 * @param  string $listStr
	 * @param  int $devid
	 * @param  $nodeid
	 * @param  array $presetAttributes
	 * @throws Exception
	 * @return array
	 */
	private function _getListValue($listStr, $devid, $nodeid, array $presetAttributes = array())
	{
		$list_str = 'LIST';
		$param = array();

		$userid = null;
		if (!isset($_SESSION[SESSION_USERID]))
		{
			throw new Exception("session userid not defined");
		}

		if (($listStr != ''))
		{
			$matches = array();
			if (preg_match("/^ls!(.*)$/", $listStr, $matches))
			{
				$list_str = 'LIST';
				$ls_str = $matches[1];
				$ls_arr = preg_split(self::$commaDelimiter, $ls_str);

				$i = 0;
				$tls_arr = array();
				foreach ($ls_arr as $ls)
				{
					$ls = stripslashes($ls);
					$valueLabelList = preg_split(self::$colonDelimiter, $ls);
					if (count($valueLabelList) == 2)
						list($value, $label) = $valueLabelList;
					else
						list($value, $label) = array($valueLabelList[0], $valueLabelList[0]);

					$tls_arr[$i]['value'] = $value;
					$tls_arr[$i]['label'] = $label;
					$i++;
				}
				$param = $tls_arr;
			} else
			if (preg_match("/^sql!(.*)$/", $listStr, $matches))
			{
				$list_str = 'LIST';

				$queryString = $matches[1];

				$queryString = str_ireplace(
					array('$DEVID$', '$NODEID$', '$USERID$'),
					array($devid, $nodeid, $userid),
					$queryString
				);

				$list = DB::select($queryString);
				$param = $list;
			} else
			// ref!IMAGESIZE_LIST
			// ref!IMAGESIZE_LIST(CAMERA)
			if (preg_match("/^ref!(\w+)(?:(?:\((.+)\)$)?|$)/", $listStr, $matches))
			{
				$list_str = 'LIST';

				$refAttribute = $matches[1];

				$refAttributeParam = null;
				if (isset($matches[2])) $refAttributeParam = $matches[2];

				$values = array();
				$object = new Obj();
				if($devid){
					$refAttributeValue = $object->getAttribute($devid, $refAttribute);
				} else {
					if(isset($presetAttributes[$refAttribute]))$refAttributeValue = $presetAttributes[$refAttribute];
				}

				if (isset($refAttributeValue) && $refAttributeValue != "")
				{
					$values = self::parseList($refAttributeValue);
					if (isset($refAttributeParam))
					{
						$refAttributeParamValue = $object->getAttribute($devid, $refAttributeParam);
						$values = self::parseList($values[$refAttributeParamValue]);
					}
				}

				foreach ($values as $key => $value)
				{
					//$valueLabelList = preg_split(self::$colonDelimiter, $value);
					if (isset($key) && isset($value))
						$option = array("value" => $key, "label" => $value);
					else
						$option = array("value" => $value, "label" => $value);
					$param[] = $option;
				}
			} else
			if (preg_match("/^url!(.*)$/", $listStr, $matches))
			{
				$list_str = 'URL';
				$ls_str = $matches[1];

				$ls_str = str_ireplace(
					array('$DEVID$', '$NODEID$', '$USERID$'),
					array($devid, $nodeid, $userid),
					$ls_str
				);

				$param = $ls_str;
			} else
			if (preg_match("/^func!(.*)$/", $listStr, $matches))
			{
				$list_str = 'LIST';
				$ls_str = $matches[1];

				if (method_exists($this, $ls_str))
				{
					$param = call_user_func(array("Template", $ls_str));
				} else {
					$param = array();
				}
			}
		}

		return array($param, $list_str);
	}

	/**
	 * @param string $param
	 * @return array
	 */
	private function _getValidateValue($param)
	{
		$valid_js = 'VALIDATE';
		if (($param != '') && strlen($param) > 3)
		{
			if (strpos($param, 're!') === 0)
			{
				$param = substr($param, strlen('re!'));
				$valid_js = 'VALIDATE';
			} else
			if (strpos($param, 'js!') === 0)
			{
				$param = substr($param, strlen('js!'));
				$valid_js = 'JS';
			}
		}
		return array($param, $valid_js);
	}

	/**
	 * @param string $templatePath
	 * @param array $templateArray
	 * @param string|null $modelid
	 * @return array
	 */
	private function _useObjectModel($templatePath, $templateArray, $modelid = NULL)
	{
		$file = file($templatePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
		// Check: is modelID in cfg file.
		$isModelInCFG = $this->_isModelInCFG($file, $modelid);

		// Parse CFG file
		$inputflag = 0;
		foreach ($file as $str)
		{
			$str = trim($str);
			if ($str == '' || ($str[0] == '#'))
				continue;

			if (strpos($str, self::$modelIdPrefix) === 0)
			{
				$str = substr($str, strlen(self::$modelIdPrefix));
				$modelid_arr = preg_split(self::$commaDelimiter, $str);
				$inputflag = 2;

				foreach ($modelid_arr as $mid)
				{
					$mid = trim($mid);
					$mid = stripslashes($mid);

					if ($isModelInCFG && $modelid == $mid
					    || stripos($modelid, $mid, 0) === 0)
					{
						$inputflag = 1;
						break;
					}
				}
				continue;
			} else
			if ($inputflag == 0 || $inputflag == 1)
			{
				list($SECTION, $ATTR, $TYPE, $LABEL, $DEFAULT_VALUE, $LIST, $VALIDATE_FUNCTION, $ERROR_MESSAGE) = preg_split(self::$formatDelimiter, $str);
				if (!isset($DEFAULT_VALUE))
					$DEFAULT_VALUE = '';

				$templateArray[$ATTR] = $DEFAULT_VALUE;
			}
		}
		return $templateArray;
	}

	/**
	 * check boolean value of variable
	 * @static
	 * @param  mixed $var
	 * @return bool
	 */
	public static function boolVal($var)
	{
		$result = false;

		if (in_array($var, array('true', 'True', 'TRUE', 'yes', 'Yes', 'y', 'Y', '1', 'on', 'On', 'ON', true, 1), true))
		{
			$result = true;
		}

		return $result;
	}

	/**
	 * @static
	 * @param  string $string key:val1\,val2,key1:value1
	 * @return array array("key" => "value", "key1" => "value1")
	 */
	public static function parseList($string)
	{
		$list = array();

		$elements = preg_split(self::$commaDelimiter, $string, -1, PREG_SPLIT_NO_EMPTY);
		foreach ($elements as $element)
		{
		    $result = preg_split(self::$colonDelimiter, $element, -1, PREG_SPLIT_NO_EMPTY);

			$key = $result[0];
			if (count($result) == 1)
				$value = $key;
			else
				$value = preg_replace(array('/\\\,/', '/\\\:/'), array(',', ':'), $result[1]);

			$list[$key] = $value;
		}

		return $list;
	}

	/**
	 * get timezone list
	 * for use only in templates
	 *
	 * @static
	 * @return array list of timezones
	 */
	private static function getTimeZoneList()
	{
		$timezone_identifiers = DateTimeZone::listIdentifiers();

		// TS I don't know does we need UTC time zone
		$timeZone_city[] = array(
			'value' => 'UTC',
			'label' => 'UTC'
		);

		foreach ($timezone_identifiers as $value)
		{
			if (preg_match('/^(Africa|America|Antarctica|Arctic|Asia|Australia|Atlantic|Europe|Indian|Pacific)\//', $value))
			{
				$timeZone_city[] = array('value' => $value, 'label' => $value);
			}
		}
		return $timeZone_city;
	}

	/**
	 * get list of supported Cloud Storage providers
	 * for use only in templates
	 *
	 * @static
	 * @return array list of cloud providers
	 */
	private static function getCloudProviders()
	{
		$list[] = array(
			'value' => 'none',
			'label' => 'none'
		);

		$info = va_admin_info();
		$cls = $info["cls"]["limit"];
		if ($cls && $cls !== "1NaN" && $cls !== "DISABLED")
		{
			$list[0] = array(
				'value' => $cls,
				'label' => $cls
			);
		}

		return $list;
	}

	/**
	 * get qtCam audio list in system
	 * for use only in templates
	 *
	 * @static
	 * @return array list of audio qtCam devices
	 */
	private static function getqtCamAudioList()
	{
		$node = new Node();
		$parameters = $node->getqtCamParametersList("audio");

		$list = array();
		$list[] = array(
			"value" => "",
			"label" => ""
		);
		foreach ($parameters as $parameter)
		{
			$list[] = array(
				"value" => $parameter["UNIQUEID"],
				"label" => $parameter["MODELID"]
			);
		}

		return $list;
	}

	/**
	 * get LDAP group list
	 * for use only in templates
	 *
	 * @static
	 * @return array list of LDAP groups
	 */
	private static function getLDAPGroupList()
	{
		$list = array();
		$list[] = array(
			"value" => "",
			"label" => ""
		);

		$identity = Identity::getAttributes();
		$LDAP_AUTHENTICATION = Template::boolVal($identity["LDAP_AUTHENTICATION"]);
		if ($LDAP_AUTHENTICATION)
		{
			$ldap = new LDAP();
			$ldap->auth($identity["LDAP_USER"], $identity["LDAP_PASSWORD"]);
			$groupList = $ldap->getGroups();

			foreach ($groupList as $group)
			{
				$list[] = array(
					"value" => $group["cn"],
					"label" => $group["cn"]
				);
			}
		}

		return $list;
	}

	/**
	 * convert string date format (from Identity for example) to PHP format
	 *
	 * @param string $dateFormat
	 * @return string converted date format
	 */
	public static function convertStrDateFormatToPHPDateFormat($dateFormat = "")
	{
		$dateFormat = trim($dateFormat);
		if (!isset($dateFormat) || $dateFormat == "") $dateFormat = "y/m/d";
		$dateFormat = str_replace("DD", "d", $dateFormat);
		$dateFormat = str_replace("MM", "m", $dateFormat);
		$dateFormat = str_replace("YYYY", "Y", $dateFormat);
		$dateFormat = str_replace("D", "j", $dateFormat);
		$dateFormat = str_replace("M", "n", $dateFormat);
		$dateFormat = str_replace("YY", "y", $dateFormat);
		return $dateFormat;
	}

	/**
	 * convert string time format (from Identity for example) to PHP format
	 *
	 * @param string $timeFormat
	 * @return string converted time format
	 */
	public static function convertStrTimeFormatToPHPTimeFormat($timeFormat = "")
	{
		$dateFormat = trim($timeFormat);
		if (!isset($timeFormat) || $timeFormat == "") $timeFormat = "y/m/d";
		$timeFormat = str_replace("hh", "h", $timeFormat);
		$timeFormat = str_replace("mm", "i", $timeFormat);
		$timeFormat = str_replace("SS", "s", $timeFormat);
		$timeFormat = str_replace("ss", "u", $timeFormat);
		return $timeFormat;
	}
	
	/*
	 * get list of avatar external commands listed in configuration file
	 * for use in templates only 
	 *
	 * @static
	 * @return array list of avatar external commands
	 */
	public static function getAvatarExternalCommands()
	{
		$list = array();
		$avatar = new Avatar();
		$cmds = $avatar->getExternalCommands();
		foreach ($cmds as $cmdname => $cmd) {
			$list[] = array(
				"value" => $cmdname,
				"label" => $cmdname
			);
		}
		
		if (count($list) == 0) {
			$list[] = array(
				"value" => "",
				"label" => "none"
			);
		}
		
		return $list;
	}
}
