<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: TargetWatcher.php 32398 2015-05-13 10:34:05Z astarostin $
 * ------------------------------------------------------------------------------
 * target zone ptz
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 * ------------------------------------------------------------------------------
 */

class TargetWatcher
{
	/**
	 * @var array
	 */
	private $_targetList = array();
	private $_targetList2 = array();

	/**
	 * @var array
	 */
	public $_ptzLoopList = array();

	/**
	 * @var int
	 */
	private $_ptzLoopTime = 0;

	/**
	 * @var int
	 */
	private $_ptzLoopInterval = 2;

	/**
	 * @var int
	 */
	private $_ptzTargetLifeInterval = 15;

	public function waitForTagets()
	{
		$AMQConf = Node::getAMQConf();
		try
		{
			$stomp = new Stomp($AMQConf["stomp.url"]);
			$stomp->connect();
			$stomp->subscribe("/topic/amq.gis.topic");

			$dom = new DOMDocument('1.0', 'utf-8');

			$stomp->setReadTimeout(0, 100000);

			while (true)
			{
				$frame = $stomp->readFrame();
				if ($frame != NULL)
				{
					if ($dom->loadXML($frame->body))
					{
						$attributes = $dom->childNodes->item(0)->attributes;
						$type = $attributes->getNamedItem("type")->nodeValue;

						if ($type == "targetRegister" || $type == "targetPTZTest")
						{
							$obj = intval($attributes->getNamedItem("objId")->nodeValue);
							$targetId = $attributes->getNamedItem("targetId")->nodeValue;
							$lat = floatval($attributes->getNamedItem("targetLatitude")->nodeValue);
							$lng = floatval($attributes->getNamedItem("targetLongitude")->nodeValue);
							$alt = 0; // TODO: correct incorrect altitude $attributes->getNamedItem("targetAltitude")->nodeValue;

							//$cameraFovX = $attributes->getNamedItem("cameraFovX")->nodeValue;
							//$cameraFovY = $attributes->getNamedItem("cameraFovY")->nodeValue;
							//$cameraTiltAngel = $attributes->getNamedItem("cameraTiltAngel")->nodeValue;
							//$cameraAzimuthAngle = $attributes->getNamedItem("cameraAzimuthAngle")->nodeValue;
							//$cameraLatitude = $attributes->getNamedItem("cameraLatitude")->nodeValue;
							//$cameraLongitude = $attributes->getNamedItem("cameraLongitude")->nodeValue;
							//$cameraHeight = $attributes->getNamedItem("cameraHeight")->nodeValue;
							//$cameraResolutionWidth = $attributes->getNamedItem("cameraResolutionWidth")->nodeValue;
							//$cameraResolutionHeight = $attributes->getNamedItem("cameraResolutionHeight")->nodeValue;
							//$imageX = $attributes->getNamedItem("imageX")->nodeValue;
							//$imageY = $attributes->getNamedItem("imageY")->nodeValue;

							$targetDistanceItem = $attributes->getNamedItem("targetDistance");
							$targetDistance = isset($targetDistanceItem) ? floatval($attributes->getNamedItem("targetDistance")->nodeValue) : 0;

							$this->addTarget($obj, $targetId, $lat, $lng, $alt, $targetDistance);

							$this->addTarget2($obj, $targetId, $lat, $lng, $alt);
						}
					}

					$stomp->ack($frame);
				}

				$this->cleanTargets();

				$this->ptzLoopTargets();

				usleep(100000);
			}
		}
		catch (StompException $e)
		{
			die('Connection failed: ' . $e->getMessage() . "\n");
		}
	}

	private function getIndex()
	{
		$element = current($this->_ptzLoopList);
		if ($element === FALSE)
		{
			$element = reset($this->_ptzLoopList);
		}

		return $element === FALSE ? -1 : key($this->_ptzLoopList);
	}

	private function getNextIndex()
	{
		$element = next($this->_ptzLoopList);
		if ($element === FALSE)
		{
			$element = reset($this->_ptzLoopList);
		}

		return $element === FALSE ? -1 : key($this->_ptzLoopList);
	}

	/**
	 * @param string $targetId
	 * @return array
	 */
	public function getTargetRelatedZone($targetId)
	{
		$result = array("cameraList" => array(), "position" => array());

		$zonesList= $this->getPTZZonesList();

		$targetPosition = $this->getTargetPosition($targetId);
		if (count($targetPosition) == 0)
		{
			return $result;
		}

		$lat = $targetPosition["lat"];
		$lng = $targetPosition["lng"];

		$result["position"] = $targetPosition;

		foreach ($zonesList as $obj => $polygons)
		{
			if ($this->inPolygons($lat, $lng, $polygons))
			{
				$result["cameraList"][] = $obj;
			}
		}

		return $result;
	}

	/**
	 * @param string $targetId
	 * @return array
	 */
	private function getTargetPosition($targetId)
	{
		$list = DB::select("
			SELECT
				targetlatitude as lat, targetlongitude as lng, targetaltitude as alt
			FROM
				gismo_data
			WHERE
				deleted = 0
				AND targetid = ?;",
			array($targetId)
		);

		$result = array();
		if (count($list) > 0)
		{
			$result = array_map("floatval", $list[0]);
		}

		return $result;
	}

	/**
	 * @param float $lat1
	 * @param float $lon1
	 * @param float $lat2
	 * @param float $lon2
	 * @return float
	 */
	private function distanceBetween2Points($lat1, $lon1, $lat2, $lon2)
	{
		$distance = 2 * asin(sqrt(
			sin(($lat1 - $lat2) / 2) * sin(($lat1 - $lat2) / 2) +
			cos($lat1) * cos($lat2) * (sin(($lon1 - $lon2) / 2) * sin(($lon1 - $lon2) / 2))
		));
		return 6371000 * $distance;
	}

	/**
	 * @param string $targetId
	 * @param float $lat
	 * @param float $lng
	 * @param float $alt
	 * @return bool
	 */
	private function ptz($targetId, $lat, $lng, $alt)
	{
		$isMoved = false;

		$zonesList= $this->getPTZZonesList();
		foreach ($zonesList as $obj => $polygons)
		{
			$camera = new Camera();
			$attributes = $camera->getAttributes($obj);
			$cameraLat = intval($attributes["CAM_GEO_LAT"]);
			$cameraLng = intval($attributes["CAM_GEO_LONG"]);
			$cameraAlt = intval($attributes["CAM_GEO_ALT"]);
			$targetDistance = $this->distanceBetween2Points(deg2rad($cameraLat), deg2rad($cameraLng), deg2rad($lat), deg2rad($lng));

			$cameraTiltAngel = atan2(deg2rad($cameraAlt), $targetDistance); // $attributes->getNamedItem("cameraTiltAngel")->nodeValue;

			$D = $targetDistance / cos($cameraTiltAngel);
			$W = 20;

			$hfov = rad2deg(2 * atan($W / (2 * $D)));
			print "tilt=$cameraTiltAngel distance=$targetDistance fov=$hfov\n";

			// some ptz drivers not correctrly processes float hfov values
			$hfov = round($hfov, 1);

			if ($this->inPolygons($lat, $lng, $polygons))
			{
				$isMoved = true;
				// move
				$data = urlencode("<PTZ_Command>do.ptz?target=$targetId&mode=abs&lat=$lat&long=$lng&elev=$alt&slew=$obj</PTZ_Command>");
				$url = "http://s_master/ptz/cgi-bin/send_message.pl?data=$data";

				$result = file_get_contents($url);
				print "MOVE target=$targetId&mode=abs&lat=$lat&long=$lng&elev=$alt&slew=$obj $result\n";

				usleep(200000);

				// zoom
				$data = urlencode("<PTZ_Command>do.ptz?dev=$obj&mode=abs&hfov=$hfov</PTZ_Command>");
				$url = "http://s_master/ptz/cgi-bin/send_message.pl?data=$data";

				$result = file_get_contents($url);
				print "FOV dev=$obj&mode=abs&hfov=$hfov $result\n";

				usleep(200000);
			}
		}

		return $isMoved;
	}

	/**
	 * @param float $lat
	 * @param float $lng
	 * @param array $polygon
	 * @return bool
	 */
	private function inPolygon($lat, $lng, array $polygon)
	{
	    // ray-casting algorithm based on
	    // http://www.ecse.rpi.edu/Homepages/wrf/Research/Short_Notes/pnpoly.html

		$coord = $this->proj($lng, $lat);
	    $x = $coord["x"];
		$y = $coord["y"];

	    $inside = false;
	    for ($i = 0, $j = count($polygon) - 1; $i < count($polygon); $j = $i++)
	    {
		    $coord = $this->proj($polygon[$i][0], $polygon[$i][1]);
		    $xi = $coord["x"];
		    $yi = $coord["y"];
		    $coord = $this->proj($polygon[$j][0], $polygon[$j][1]);
		    $xj = $coord["x"];
		    $yj = $coord["y"];

	        $intersect = (($yi > $y) != ($yj > $y))
		        && ($x < ($xj - $xi) * ($y - $yi) / ($yj - $yi) + $xi);
	        if ($intersect) $inside = !$inside;
	    }

	    return $inside;
	}

	/**
	 * equirectangular projection
	 *
	 * @param float $lat
	 * @param float $lng
	 * @return array
	 */
	private function proj($lat, $lng)
	{
		return array(
			"x" => deg2rad($lng) * cos(deg2rad($lat)),
			"y" => deg2rad($lat)
		);
	}

	/**
	 * @param float $lat
	 * @param float $lng
	 * @param array $polygons
	 * @return bool
	 */
	private function inPolygons($lat, $lng, array $polygons)
	{
		$inZone = false;

		foreach ($polygons as $index => $polygon)
		{
			if ($this->inPolygon($lat, $lng, $polygon))
			{
				$inZone = true;
				break;
			}
		}

		return $inZone;
	}

	/**
	 * @param string $zoneType
	 * @return array
	 */
	private function getPTZZonesList($zoneType = "include")
	{
		$list = DB::select(
			"SELECT zone.obj, zone.val FROM
				_obj_attr as state,
				_obj_attr as calibration,
				_obj_attr as monitoring,
				_obj_attr as ptz,
				_obj_attr as zone
			WHERE
				state.attr = 'ARCHSTATE' AND state.val = 'on' AND state.obj = zone.obj
				AND calibration.attr = 'CAM_GEO_CALIBRATION' AND calibration.val != 'no' AND calibration.obj = zone.obj
				AND monitoring.attr = 'TARGET_ZONE_MONITORING' AND monitoring.val = 'yes' AND monitoring.obj = zone.obj
				AND ptz.attr = 'POSITIONCTL' AND ptz.val != 'none' AND ptz.obj = zone.obj
				AND zone.attr = 'TARGET_ZONE';");

		return $this->geoJSONtoArray($list, $zoneType);
	}

	/**
	 * @param int $obj
	 * @param string $zoneType
	 * @return array
	 */
	private function getZonesList($obj, $zoneType = "include")
	{
		$list = DB::select(
			"SELECT zone.obj, zone.val FROM
				_obj_attr as zone
			WHERE
				zone.obj = ? AND zone.attr = 'TARGET_ZONE';", array($obj));

		return $this->geoJSONtoArray($list, $zoneType);
	}

	/**
	 * @param array $geoJSONList
	 * @param string $zoneType
	 * @return array
	 */
	private function geoJSONtoArray($geoJSONList, $zoneType = "include")
	{
		$result = array();

		/*
		{
			"type":     "FeatureCollection",
			"features": [{
				"type":       "Feature",
				"properties": {"zoneType":"exclude"},
				"geometry":   {
					"type":        "Polygon",
					"coordinates": [[[-77.47077941894531, 38.91325055644748], [-77.47125148773193, 38.91191485030754], [-77.46745347976685, 38.912607751005346], [-77.46904134750365, 38.9134509102004], [-77.47077941894531, 38.91325055644748]]]
				}
			}]
		};
		*/
		foreach ($geoJSONList as $row)
		{
			$zones = json_decode($row["val"], true);
			foreach ($zones["features"] as $index => $feature)
			{
				if (!isset($result[$row["obj"]]))
				{
					$result[$row["obj"]] = array();
				}
				$featureZoneType = isset($feature["properties"]["zoneType"]) ? $feature["properties"]["zoneType"] : "include";
				if ($featureZoneType == $zoneType)
				{
					$result[$row["obj"]][] = $feature["geometry"]["coordinates"][0];
				}
			}
		}

		return $result;
	}

	/**
	 * @param int $obj
	 * @param string $targetId
	 * @param float $lat
	 * @param float $lng
	 * @param float $alt
	 */
	private function addTarget2($obj, $targetId, $lat, $lng, $alt)
	{
		if (!isset($this->_targetList2[$obj]))
		{
			$this->_targetList2[$obj] = array();
		}

		$time = microtime(true);
		if (!isset($this->_targetList2[$obj][$targetId]))
		{
			$this->_targetList2[$obj][$targetId] = array(
				"time" => $time,
				"lat" => $lat,
				"lng" => $lng,
				"alt" => $alt
			);
		} else {
			$target = $this->_targetList2[$obj][$targetId];
			$target["lat"] = $lat;
			$target["lng"] = $lng;
			$target["alt"] = $alt;
		}

		// exclude target
		$zonesList = $this->getZonesList($obj, "exclude");
		foreach ($zonesList as $objId => $polygons)
		{
			if ($this->inPolygons($lat, $lng, $polygons))
			{
				return;
			}
		}

		$globalZonesList = $this->getZonesList(Identity::getObj());
		foreach ($globalZonesList as $objId => $polygons)
		{
			if ($this->inPolygons($lat, $lng, $polygons))
			{
				$this->createEvent($obj, "IN");
			} else {
				$this->createEvent($obj, "OUT");
			}

			break;
		}
	}

	/**
	 * @param int $obj
	 * @param string $targetId
	 * @param float $lat
	 * @param float $lng
	 * @param float $alt
	 * @param float $targetDistance
	 */
	private function addTarget($obj, $targetId, $lat, $lng, $alt, $targetDistance)
	{
		if ($targetDistance < 0.01)
		{
			print "skip too close target\n";
		} else {

			if (!isset($this->_targetList[$obj]))
			{
				$this->_targetList[$obj] = array();
			}

			$index = "";
			$time = microtime(true);
			if (!isset($this->_targetList[$obj][$targetId]))
			{
				$index = uniqid();
				$this->_targetList[$obj][$targetId] = array(
					"time" => $time,
					"lat" => $lat,
					"lng" => $lng,
					"alt" => $alt,
					"index" => $index
				);

				$this->_ptzLoopList[$index] = array("obj" => $obj, "targetId" => $targetId);
			} else {
				$target = $this->_targetList[$obj][$targetId];
				$index = $target["index"];
				$target["lat"] = $lat;
				$target["lng"] = $lng;
				$target["alt"] = $alt;
			}

			if ($index == $this->getIndex())
			{
				print "$time new move to $obj $targetId $index\n";

				if ($this->ptz($targetId, $lat, $lng, $alt))
				{
					$this->_ptzLoopTime = microtime(true);
				}
			}
		}
	}

	/**
	 * remove old targets from list
	 */
	private function cleanTargets()
	{
		foreach ($this->_targetList as $obj => $targets)
		{
			foreach ($targets as $targetId => $target)
			{
				$time = microtime(true);
				if ($time - $target["time"] > $this->_ptzTargetLifeInterval)
				{
					$index = $target["index"];

					print "$time remove $obj $targetId $index from " . count($this->_ptzLoopList) . "\n";
					unset($this->_ptzLoopList[$index]);
					unset($this->_targetList[$obj][$targetId]);
				}
			}
		}
	}

	/**
	 * send ptz command to next camera
	 */
	private function ptzLoopTargets()
	{
		$time = microtime(true);
		if ($time - $this->_ptzLoopTime > $this->_ptzLoopInterval)
		{
			$index = $this->getNextIndex();
			print "$time time to loop $index from " . count($this->_ptzLoopList) . "\n";
			if ($index !== -1)
			{
				$element = $this->_ptzLoopList[$index];
				$obj = $element["obj"];
				$targetId = $element["targetId"];

				$target = $this->_targetList[$obj][$targetId];
				$lat = $target["lat"];
				$lng = $target["lng"];
				$alt = $target["alt"];

				print "$time loop move to $obj $targetId $index from " . count($this->_ptzLoopList) . "\n";

				if ($this->ptz($targetId, $lat, $lng, $alt))
				{
					$this->_ptzLoopTime = microtime(true);
				}
			}
		}
	}

	public function createEvent($obj, $message)
	{
		$result = false;

		try {
			$node = new Node();
			$node->submitAction(
				"create",
				array(
					"objid" => $obj,
					"source" => 6, // Analytics
					"eventtype" => 1, // Alert
					"msg" => $message
				),
				array()
			);

			$result = true;
		}
		catch (Exception $e)
		{
			print $e->getMessage();
		}

		return $result;
	}
}
