<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: SessionManager.php 33575 2016-02-03 19:07:53Z astarostin $
 * ------------------------------------------------------------------------------
 * description
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @author Andrey Baranetsky
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2012
 * ------------------------------------------------------------------------------
 */

class SessionManager
{
	const SESSION_STATUS_PASSWORD_EXPIRED = "PasswordExpired";
	const SESSION_STATUS_LOGGED_IN = "LoggedIn";
	const SESSION_STATUS_LOG_IN_PROGRESS = "LoginInProgress";

	/**
	 * session list
	 *
	 * @var array
	 */
	private $_sessions = array();

	/**
	 * @var string|null
	 */
	private $_sessionSavePath = null;

	/**
	 * @var int
	 */
	private $_loginTTL = 5;

	public function __construct()
	{
		$this->update();
	}

	/**
	 * update session list
	 */
	public function update()
	{
		$this->_parseSessionSavePath();
		$this->_parseSessions();
	}

	/**
	 * read session save path from config
	 */
	private function _parseSessionSavePath()
	{
		//list($level, $mode, $sessionSavePath) = explode(";", session_save_path());
		$sessionSavePath = "/var/lib/php/session";
		$this->_sessionSavePath = realpath($sessionSavePath);
	}

	/**
	 * fill list of session
	 */
	private function _parseSessions()
	{
		clearstatcache();

		$this->_sessions = array();
		$files = array_diff(scandir($this->_sessionSavePath), array('.', '..'));

		foreach ($files as $file)
		{
			$match = array();
			// "sess_7480686aac30b0a15f5bcb78df2a3918"
			if (preg_match('/^sess_(.+)$/', $file, $match) == 0)
			{
				continue;
			}

			$session = $match[1];

			$path = $this->_sessionSavePath . DIRECTORY_SEPARATOR . $file;

			$this->_sessions[$session] = array(
				"age" => time() - filectime($path),
				"creation" => filectime($path),
				"modification" => filemtime($path),
				"content" => $this->_unserializeSession(file_get_contents($path))
			);
		}
	}

	/**
	 * unserialize session
	 *
	 * @param string $data
	 * @return array
	 */
	private function _unserializeSession($data)
	{
		$result = array();

		$vars = preg_split('/([a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff^|]*)\|/', $data, -1, PREG_SPLIT_NO_EMPTY | PREG_SPLIT_DELIM_CAPTURE);

		for ($i = 0; $i < count($vars); $i += 2)
		{
			$result[$vars[$i]] = unserialize($vars[$i + 1]);
		}

		return $result ;
	}

	/**
	 * get list of current logged in sessions
	 *
	 * @param  int|null $userid get sessions only for specified user
	 * @param  string|null $status status of session
	 * @return array
	 */
	public function getList($userid = null, $status = SessionManager::SESSION_STATUS_LOGGED_IN)
	{
		$list = array();

		if (!isset($userid) && !isset($status))
		{
			$list = $this->_sessions;
		} else {
			foreach ($this->_sessions as $sessionId => $session)
			{
				if (
					(!isset($userid) || isset($session["content"]["userid"]) && $session["content"]["userid"] == $userid)
					&&
					(!isset($status) || isset($session["content"]["status"]) && $session["content"]["status"] == $status)
				)
				{
					$list[$sessionId] = $session;
				}
			}
		}

		return $list;
	}

	/**
	 * get session content
	 *
	 * @param $sessionId
	 * @return array
	 */
	public function getSessionContent($sessionId)
	{
		return $this->_sessions[$sessionId]["content"];
	}

	/**
	 * remove session and notify user about it
	 *
	 * @param string $sessionId
	 * @param string $reason
	 * @return bool
	 */
	public function stop($sessionId, $reason)
	{
		// print("stop session $sessionId, $reason\n");
		$result = unlink($this->_sessionSavePath . DIRECTORY_SEPARATOR . "sess_" . $sessionId);

		$status = $this->_sessions[$sessionId]["content"]["status"];
		if ($result && $status != SessionManager::SESSION_STATUS_LOG_IN_PROGRESS)
		{
			// get session username
			$userid = $this->_sessions[$sessionId]["content"]["userid"];
			$user = new User();
			$username = $user->getName($userid);

			if ($reason == "SESSION_EXPIRED")
			{
				Audit::addRecordVArg(4, null, null, $userid, $username, Audit::attrValueEntity('SessionId', $sessionId));
			} else
			if ($reason == "SESSION_TERMINATED")
			{
				$currentUserName = $user->getName($_SESSION[SESSION_USERID]);
				Audit::addRecordVArg(5, null, null, $username, $currentUserName, Audit::attrValueEntity('SessionId', $sessionId));
			}

			$AMQConf = Node::getAMQConf();
			$con = new Stomp($AMQConf["stomp.url"]);
			$con->connect();
			$message = array(
				"error" => "",
				"reason" => $reason
			);
			$destination = "/topic/sessionClose_" . $sessionId;
			$result = $con->send($destination, json_encode($message), array('persistent'=>'true', 'slow_consumer_policy'=>'queue'));
		}

		return $result;
	}

	/**
	 * remove all sessions of user
	 *
	 * @param  int $userid
	 * @return bool
	 */
	public function stopForUser($userid)
	{
		$result = true;

		foreach ($this->_sessions as $sessionId => $session)
		{
			if (isset($session["content"]["userid"]) && $session["content"]["userid"] == $userid)
			{
				$result = $result && $this->stop($sessionId, "SESSION_TERMINATED");
			}
		}

		return $result;
	}

	/**
	 * remove oldest sessions that reached session limit
	 *
	 * @param int $userid
	 * @param int $limit
	 */
	public function stopReachedLimit($userid, $limit)
	{
		$userSessions = $this->getList($userid);
		if ($limit > 0 && count($userSessions) > $limit)
		{
			uasort($userSessions, function($a, $b){
				return $a["age"] < $b["age"];
			});

			$numberToRemove = count($userSessions) - $limit;
			foreach ($userSessions as $sessionId => $session)
			{
				if ($numberToRemove == 0)
					break;
				$this->stop($sessionId, "SESSION_TERMINATED");
				$numberToRemove--;
			}
		}
	}

	/**
	 * run garbage collector
	 *
	 * @param int $lifeTime session lifetime (sec)
	 */
	public function gc($lifeTime = 600)
	{
		foreach ($this->_sessions as $sessionId => $session)
		{
			if ($session["age"] > $lifeTime)
			{
				$this->stop($sessionId, "SESSION_EXPIRED");
			}
		}
	}

	/**
	 * Start new user session
	 * Note: this function must be called before session initialization session_start()
	 *       and before any part of header/body gets submitted to client.
	 *
	 * @static
	 * @param string $userName
	 * @param int $userid
	 * @throws AuthException
	 */
	public function login($userName, $userid)
	{
		if (session_id())
			throw new AuthException('ERROR: start_new_session() failed: session has already started [DEV0572]');

		if (headers_sent())
			throw new AuthException('ERROR: start_new_session() failed: HTTP headers are already sent [DEV0573]');

		// New session id
		session_id(self::_getNewSessionId());

		// Start Session
		session_start();

		// Add indicator
		$sessionStatus = SessionManager::SESSION_STATUS_LOGGED_IN;
		$user = new User();
		if ($user->isPasswordExpired($userid))
		{
			$sessionStatus = SessionManager::SESSION_STATUS_PASSWORD_EXPIRED;
		}
		$_SESSION[SESSION_STATUS] = $sessionStatus;
		$_SESSION[SESSION_USERNAME] = $userName;
		$_SESSION[SESSION_USERID] = $userid;
		$_SESSION[SESSION_IP] = $_SERVER["REMOTE_ADDR"];
		$_SESSION[SESSION_USER_AGENT] = $_SERVER["HTTP_USER_AGENT"];
		$_SESSION[SESSION_CSRF_TOKEN_SALT] = hash("sha256", uniqid());

		$this->_createCSRFTokenCookie();

		// flush session to disk
		session_write_close();
		session_start();

		// reread session list from disk
		$this->update();

		// close oldest sessions for preventing reaching session limit
		$SESSION_MAX_NUMBER = intval(Identity::getAttribute("SESSION_MAX_NUMBER"), 10);
		$this->stopReachedLimit($userid, $SESSION_MAX_NUMBER);
	}

	private function _createCSRFTokenCookie()
	{
		$CSRFtoken = self::_getCSRFToken();
		$_SESSION[SESSION_CSRF_TOKEN] = $CSRFtoken;
		$secure = isset($_SERVER["HTTPS"]);
		setcookie("token", $CSRFtoken, 0, "/", "", $secure);
	}

	public static function getCSRFToken()
	{
		return self::_getCSRFToken();
	}

	/**
	 * @static
	 * @param string $userName
	 * @param string $credentials
	 * @param string|null $password
	 * @param bool $isLegacy
	 * @throws AuthException
	 * @throws Exception
	 */
	public function startSession($userName = "", $credentials = "", $password = null, $isLegacy = false)
	{
		if (empty($userName) ||
			empty($credentials) && empty($password))
		{
			throw new AuthException(__('Missing user name or password.'));
		}

		// Check if we need to use temporary login session
		if (strlen(session_id()) == 0)
		{
			session_cache_expire($this->_loginTTL);
			session_start();
		}
		if (!isset($_SESSION[SESSION_STATUS]))
		{
			// it's a new session
			throw new Exception(__('Login page has expired. Please update page and try to log in again.'));
		}

		if ($_SESSION[SESSION_STATUS] != SessionManager::SESSION_STATUS_LOG_IN_PROGRESS)
		{
			// previous session is not closed,
			// close it now and ask to log in again
			SessionManager::closeSession();

			session_id(hash("sha256", uniqid(openssl_random_pseudo_bytes(32))));
			session_start();

			throw new AuthException(__('Previous session closed. Please try to login again to start new session.'));
		}

		$encryptionKey = $_SESSION[SESSION_ENCRYPTIONKEY];

		$user = new User();
		$userid = $user->getObj($userName);

		$LDAP_AUTHENTICATION = Template::boolVal(Identity::getAttribute("LDAP_AUTHENTICATION"));

		if (!isset($userid))
		{
			$isNewLDAPUser = false;
			if ($_SESSION[SESSION_NTLM_AUTH])
			{
				$isNewLDAPUser = true;
			} else
			if ($LDAP_AUTHENTICATION)
			{
				try
				{
					$this->_checkUserPasswordLDAP($userName, $password);
					$isNewLDAPUser = true;
				}
				catch (AuthException $e)
				{
				}
			}

			if (!$isNewLDAPUser)
			{
				// throw new AuthException(__("User does not registered"));
				throw new AuthException(__("Invalid username/password"));
			}

			// create user associated with LDAP user
			$userid = $user->create(array(
				"NAME" => $userName,
				"LDAP_USER" => $userName
			), true);
		}

		$parameters = $user->getPasswordParameters();
		$userAttributes = $user->getAttributes($userid);

		$LAST_LOGON_ATTEMPT_TIME = intval($userAttributes["LAST_LOGON_ATTEMPT_TIME"], 10);
		$LOGON_FAILURE_ATTEMPTS = intval($userAttributes["LOGON_FAILURE_ATTEMPTS"], 10);

		$USER_LOGON_ATTEMPTS_TIME = intval($parameters["USER_LOGON_ATTEMPTS_TIME"], 10);
		$MAX_LOGON_FAILURE_ATTEMPTS = intval($parameters["MAX_LOGON_FAILURE_ATTEMPTS"], 10);

		if ($MAX_LOGON_FAILURE_ATTEMPTS > 0)
		{
			if ($LAST_LOGON_ATTEMPT_TIME == 0
				|| $LAST_LOGON_ATTEMPT_TIME < (time() - $USER_LOGON_ATTEMPTS_TIME * 3600))
			{
				$LAST_LOGON_ATTEMPT_TIME = time();
				$LOGON_FAILURE_ATTEMPTS = 0;
			} else
			if ($LOGON_FAILURE_ATTEMPTS > $MAX_LOGON_FAILURE_ATTEMPTS)
			{
				throw new AuthException(__("User account is blocked"));
			}

			$user->setAttributes($userid, array(
				"LAST_LOGON_ATTEMPT_TIME" => $LAST_LOGON_ATTEMPT_TIME,
				"LOGON_FAILURE_ATTEMPTS" => $LOGON_FAILURE_ATTEMPTS
			));
		}

		// check user login data
		try
		{
			if ($_SESSION[SESSION_NTLM_AUTH])
			{
				$user->syncRolesWithLDAP($userName);
			} else {
				try
				{
					$this->_checkUserPassword($userid, $credentials, $encryptionKey, $isLegacy);
				}
				catch (AuthException $e)
				{
					if ($LDAP_AUTHENTICATION)
					{
						$this->_checkUserPasswordLDAP($userName, $password);
						$user->syncRolesWithLDAP($userName, $password);
						/*
						// update user2role links
						$parent = new Role();

						// remove old links
						$userRoleList = $user->getRoles($userid);
						foreach ($userRoleList as $userRole)
						{
							$parent->remove($userRole["obj"], $userid);
						}

						// add new links
						$ldap = new LDAP();
						$ldap->auth($userName, $password);
						$groupList = $ldap->getGroups();

						$node = new Node();
						$roleList = $node->getRoles(true);

						foreach ($groupList as $group)
						{
							$groupName = $group["cn"];

							foreach ($roleList as $role)
							{
								$roleid = $role["obj"];
								$roleAttributes = $role["attributes"];

								$LDAP_GROUP = $roleAttributes["LDAP_GROUP"];
								if (!empty($LDAP_GROUP))
								{
									if ($groupName["cn"] == $LDAP_GROUP)
									{
										$parent->add($roleid, $userid);
									}
								}
							}
						}
						*/
					} else {
						throw $e;
					}
				}
			}
		}
		catch (AuthException $e)
		{
			if ($MAX_LOGON_FAILURE_ATTEMPTS > 0)
			{
				if ($LOGON_FAILURE_ATTEMPTS <= $MAX_LOGON_FAILURE_ATTEMPTS)
				{
					$LOGON_FAILURE_ATTEMPTS++;
				}

				$user->setAttributes($userid, array(
					"LAST_LOGON_ATTEMPT_TIME" => $LAST_LOGON_ATTEMPT_TIME,
					"LOGON_FAILURE_ATTEMPTS" => $LOGON_FAILURE_ATTEMPTS
				));

				if ($LOGON_FAILURE_ATTEMPTS > $MAX_LOGON_FAILURE_ATTEMPTS)
				{
					Audit::addRecord(45, array($userName, intval($USER_LOGON_ATTEMPTS_TIME * 60, 10)), null, $userid);
					throw new AuthException(__("User account is blocked"));
				}
			}

			Audit::addRecord(44, array($userName), null, $userid);

			throw $e;
		}

		$this->_startSession($userName, $userid);
	}

	/**
	 * Verify that user submit correct username/password combination
	 *
	 * @param int $userid
	 * @param string $credentials
	 * @param string $encryptionKey
	 * @param bool $isLegacy
	 * @throws AuthException
	 * @return bool
	 */
	private function _checkUserPassword($userid, $credentials, $encryptionKey, $isLegacy)
	{
		//allow certificate login without password
		if (isset($_SERVER['SSL_CLIENT_VERIFY']) && $_SERVER['SSL_CLIENT_VERIFY'] == "SUCCESS")
		{
			$PKI_DN_users = $this->getUsersWithPKIDN($_SERVER['SSL_CLIENT_S_DN']);
			for ($i = 0; $i < count($PKI_DN_users); $i++)
			{
				if ($PKI_DN_users[$i] == $userid)
				{
					return true;
				}
			}
		}

		$hashFunction = "";
		$user = new User();
		if (!$isLegacy)
		{
			$hashFunction = "sha512";
			$currentPasswordHash = $user->getPasswordHash($userid, $hashFunction);
		} else {
			$hashFunction = "sha1";
			$currentPasswordHash = strtoupper($user->getPasswordHash($userid, $hashFunction));
		}

		if (!isset($userid) || !isset($currentPasswordHash))
		{
			throw new AuthException(__('Unable to verify username/password. Please try again'));
		}

		// Compare passwords
		$passwordDigest = hash($hashFunction, $encryptionKey . $currentPasswordHash . $encryptionKey);
		if (strtolower($credentials) != $passwordDigest)
		{
			throw new AuthException(__('Invalid username/password'));
		}

		return true;
	}

	/**
	 * Verify with LDAP that user submit correct username/password combination
	 *
	 * @param $userName
	 * @param $password
	 * @return bool
	 * @throws AuthException
	 * @throws Exception
	 */
	private function _checkUserPasswordLDAP($userName, $password)
	{
		if (!isset($userName) || !isset($password))
		{
			throw new AuthException(__('Unable to verify username/password. Please try again'));
		}

		try
		{
			$ldap = new LDAP();
			$ldap->auth($userName, $password);
		}
		catch (Exception $e)
		{
			throw new AuthException(__($e->getMessage()));
			// throw new AuthException(__('Invalid username/password'));
		}

		return true;
	}

	/**
	 * @param string $PKI_DN
	 * @return array
	 */
	public function getUsersWithPKIDN($PKI_DN)
	{
		$PKI_DN = strtoupper($PKI_DN);
		$PKI_DN = str_replace("= ", "=", $PKI_DN);
		$PKI_DN = str_replace(" =", "=", $PKI_DN);
		$PKI_DN = str_replace(" ,", ",", $PKI_DN);
		$PKI_DN = str_replace(", ", ",", $PKI_DN);

		$pki_dn_users = array();

		$users = DB::select("SELECT _objs.obj, _obj_attr.val as pki_dn FROM _objs INNER JOIN _obj_attr ON (_objs.obj = _obj_attr.obj) WHERE _objs.otype = 'U' AND _obj_attr.attr = 'PKI_DN';");
		for ($i = 0; $i < count($users); $i++)
		{
			$users[$i]['pki_dn'] = strtoupper($users[$i]['pki_dn']);
			$users[$i]['pki_dn'] = str_replace("= ", "=", $users[$i]['pki_dn']);
			$users[$i]['pki_dn'] = str_replace(" =", "=", $users[$i]['pki_dn']);
			$users[$i]['pki_dn'] = str_replace(" ,", ",", $users[$i]['pki_dn']);
			$users[$i]['pki_dn'] = str_replace(", ", ",", $users[$i]['pki_dn']);

			if (strpos($PKI_DN, $users[$i]['pki_dn']) !== FALSE)
			{
				array_push($pki_dn_users, $users[$i]['obj']);
			}
		}

		return $pki_dn_users;
	}

	/**
	 * @return string
	 */
	private function _genEncryptionKey()
	{
		return hash("sha256", uniqid());
	}

	public function getLoginInfo()
	{
		return array(
			"encryptionKey" => $_SESSION[SESSION_ENCRYPTIONKEY],
			"loginTTL" => $this->_loginTTL,
			"LDAP_AUTHENTICATION" => Template::boolVal(Identity::getAttribute("LDAP_AUTHENTICATION"))
		);
	}

	public function startTempSession()
	{
		if (strlen(session_id()) == 0)
		{
			session_start();
		}

		// Check if we need to use temporary login session
		if (!isset($_SESSION[SESSION_STATUS]) || $_SESSION[SESSION_STATUS] != SessionManager::SESSION_STATUS_LOG_IN_PROGRESS)
		{
			$_SESSION[SESSION_STATUS] = SessionManager::SESSION_STATUS_LOG_IN_PROGRESS;
			$_SESSION[SESSION_CSRF_TOKEN_SALT] = hash("sha256", uniqid());

			$this->_createCSRFTokenCookie();
		}

		if (!isset($_SESSION[SESSION_ENCRYPTIONKEY]))
		{
			$_SESSION[SESSION_ENCRYPTIONKEY] = $this->_genEncryptionKey();
		}
	}

	private function _startSession($userName, $userid)
	{
		// Destroy current session if it exists
		if (strlen(session_id()) > 0)
		{
			session_destroy();
		}

		// Create new session
		$this->login($userName, $userid);

		Audit::addRecordVArg(2, null, null,
			Audit::attrValueEntity('IP', $_SERVER['REMOTE_ADDR']),
			Audit::attrValueEntity('SessionId', session_id()));
	}

	/**
	 * Close current session
	 * Note: this function must be called before any part of header/body
	 *       gets submitted to client.
	 *
	 * @static
	 */
	public static function closeSession()
	{
		// initialize session in case if it exists, but session_start() was not called yet
		if (strlen(session_id()) == 0)
		{
			session_start();
		}

		// Unset all of the session variables.
		$_SESSION = array();

		// If it's desired to kill the session, also delete the session cookie.
		// Note: This will destroy the session, and not just the session data!
		$sessionName = session_name();
		if (isset($_COOKIE[$sessionName]))
		{
			setcookie($sessionName, null, 0, "/");
		}

		// remove token cookie
		if (isset($_COOKIE["token"]))
		{
			setcookie("token", null, 0, "/");
		}

		// remove lang_direction cookie
		if (isset($_COOKIE["lang_direction"]))
		{
			setcookie("lang_direction", null, 0, "/");
		}

		session_destroy();

		// close JSESSIONID cookie created by AMQ servlet
		setcookie("JSESSIONID", null, 0, "/api");
	}

	/**
	 * Verify if user has logged in.
	 *
	 * @static
	 * @param string $sessionStatus
	 * @return bool
	 */
	public static function isUserLoggedIn($sessionStatus = SessionManager::SESSION_STATUS_LOGGED_IN)
	{
		$result = false;

		if (strlen(session_id()) == 0)
		{
			session_start();
		}

		if (isset($_SESSION[SESSION_STATUS])
			&& $_SESSION[SESSION_STATUS] == $sessionStatus)
		{
			$result = true;
		}

		return $result;
	}

	/**
	 * check if token is related to current session
	 *
	 * @static
	 * @param  string $token
	 * @return bool
	 */
	public static function checkCSRFToken($token)
	{
		return ($token == self::_getCSRFToken());
	}

	/**
	 * get CSRF token for current session
	 *
	 * @static
	 * @return string
	 */
	private static function _getCSRFToken()
	{
		return hash("sha256", $_SESSION[SESSION_CSRF_TOKEN_SALT]);
	}

	/**
	 * Force browser to load $page
	 *
	 * @static
	 * @param string $redirectPage
	 */
	public static function redirect($redirectPage = null)
	{
		if (!isset($redirectPage))
		{
			$redirectPage = "/sdi/login/index.php";
		}

		header("Location: $redirectPage");
		exit;
	}

	/**
	 * Generate session id
	 *
	 * @static
	 * @return string
	 */
	private static function _getNewSessionId()
	{
		return hash("sha256", uniqid(openssl_random_pseudo_bytes(32), true));
	}
}
