<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: Schedule.php 33243 2015-11-01 23:59:59Z astarostin $
 * ------------------------------------------------------------------------------
 * This class represents logic for Schedule
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions LLC 2006
 * ------------------------------------------------------------------------------
 */

/**
 * @var array $scheduleInfo structure
"scheduleinfo":{
	"103":{  //scheduleid
		"obj":"103",
		"name":"111",
		"attributes":{
			"NAME":"111",
			"DESCRIPTION":""
		},
		"devices":[105,110,115],
		"calendar":[
			{
				"id":101,//timecardID
				"name":"111111",//posturename
				"timecardid":"110",
				"postureid":"104",
				"posture":{
					"attributes":{
						"NAME":"111",
						"DESCRIPTION":""
					}
				},
				"timecard":{
					"attributes":{
						"UDID":"TS103",
						"NAME":"111",
						"DESCRIPTION":"",
						"ICAL_DATA":{
							"startTS":234536252,	//date in timestamp
							"endTS":234536252,		//date in timestamp
							"start":{"date":"2011-07-13 05:00:00","timezone_type":3,"timezone":"UTC"},
							"end":{	"date":"2011-07-13 08:30:00","timezone_type":3,"timezone":"UTC"},
							"summary":false,
							"allday":false,
							"timezone":tc_timezone,
							"rrule":[]
							///////////////////////////
							// {
							'interval':interval,  	//1,2,3...
							'frequency':frequency, 	//freq = "SECONDLY" / "MINUTELY" / "HOURLY" / "DAILY" / "WEEKLY" / "MONTHLY"
							'byday':byday, 			//["SU","MO","TU","WE","TH","FR","SA"]
							'count':count, 			//1,2,3...
							'until':until 			//date in timestamp
							};
							///////////////////////
						},
					}
				}
			},
			{posture-timecard-info2...},
			{posture-timecard-info3...},
		],
	}
}
*/

class Schedule extends ScheduleObject
{
	public function __construct()
	{
		$this->otype = 'T';
		$this->subtype = 'S';
		parent::__construct("schedule");
	}

	/**
	 * @param array $attributes
	 * @return int
	 */
	public function create(array $attributes)
	{
		$scheduleattributes = array();
		$postureattributes = array();

		foreach($attributes as $name=>$value){
			if (strpos($name, "PROPERTY_")===0 && trim($value)!=''){
				$postureattributes[$name] = $value;
			}else{
				$scheduleattributes[$name] = $value;
			}
			if ($name == "TIME_ZONE"){
				$postureattributes["PROPERTY_TIME_ZONE"] = $value;
			}
		}
		if (!isset($postureattributes["PROPERTY_TIME_ZONE"])) $postureattributes["PROPERTY_TIME_ZONE"] = Identity::getAttribute("TIME_ZONE");
		$obj = parent::create($scheduleattributes);
		$name = isset($attributes["NAME"]) ? 'DEFAULT for ' . $attributes["NAME"] : 'Default posture for ['.$obj.']';
		$postureattributes['NAME'] = $name;
		$postureattributes['DESCRIPTION'] = 'Default posture for ['.$obj.']';
		$postureattributes['DEFAULTFOR'] = $obj;
		$postureattributes['PRIORITY'] = 1;
		$defaultposture = new Posture();
		$postureid = $defaultposture->create($postureattributes, $obj);
		$this->setAttributes($obj, array("DEFAULT_POSTUREID" => $postureid));
		Audit::addRecordVArg(40, $obj, $_SESSION[SESSION_USERID], "created", Audit::attrValueEntity('DEFAULT_POSTUREID', $postureid));

		return $obj;
	}

	public function getTemplateAttributes($obj=null, $attributes = array())
	{
		$attrs = parent::getTemplateAttributes($obj, $attributes);
		//get defaultposture attributes
		$posture = new Posture();
		if (!is_null($obj)){
			// get default posture
			$postureid = $this->getAttribute($obj, "DEFAULT_POSTUREID");
		}else{
			$postureid = null;
		}
		$postureprop = $posture->getTemplateProperty($postureid);
		foreach ($postureprop['attributes'] as $name=>$attribute){
			$attrs['attributes'][$name] = $attribute;
			$attrs['categories'][0]['attributes'][] = $name;

		}
		return $attrs;
	}

	/**
	 * set object attributes
	 * @param  $obj
	 * @param  array $attributes
	 * @return void
	 */
	public function setAttributes($obj, array $attributes){
		$scheduleattributes = array();
		$postureattributes = array();

		foreach($attributes as $name=>$value){
			if (strpos($name, "PROPERTY_")===0){
				$postureattributes[$name] = $value;
			}else{
				$scheduleattributes[$name] = $value;
			}
			if ($name == "TIME_ZONE"){
				$postureattributes["PROPERTY_TIME_ZONE"] = $value;
			}
		}
		$timezoneBefore = $this->getAttribute($obj, "TIME_ZONE");

		parent::setAttributes($obj,$scheduleattributes);
		$postureid = $this->getAttribute($obj, "DEFAULT_POSTUREID");
		if (isset($postureid) && !is_null($postureid) && $postureid>0){
			$posture = new Posture();
			$posture->setAttributes($postureid,$postureattributes);

		}
		$timezoneAfter = $this->getAttribute($obj, "TIME_ZONE");
		if ($timezoneBefore != $timezoneAfter){
			// TODO recalc ICAL_DATA on timecards
			$timecards = $this->getTimecards($obj, true);
			$timecardObject = new Timecard();
			foreach ($timecards as $timecard){
				$obj = $timecard['obj'];
				$ical = $timecard['attributes']['ICAL_DATA'];
				$rrule = $timecardObject->icaltoarray($ical, $timezoneBefore);
				$ical = $timecardObject->ruleArrToICal($rrule, $timezoneAfter);
				$timecardObject->setAttribute($obj,'ICAL_DATA',$ical);
			}
		}
	}

	/**
	 * set object attributes
	 *
	 * @param int $obj
	 * @return array
	 */
	public function getAttributes($obj){
		$attributes = parent::getAttributes($obj);
		if (isset($attributes['DEFAULT_POSTUREID']))
		{
			$postureid = $attributes['DEFAULT_POSTUREID'];
			$posture = new Posture();
			$postureattributes = $posture->getAttributes($postureid);
			foreach ($postureattributes as $name => $value)
			{
				if (strpos($name, "PROPERTY_") === 0)
				{
					$attributes[$name] = $value;
				}
			}
		}

		return $attributes;
	}

	public function delete($obj, $onlyMark = true)
	{
		if (!isset($obj))
		{
			throw new InvalidArgumentException("Some parameters are missing.");
		}

		//get attached devices
		$deviceList = $this->getObjectOnScheduleList($obj);
		if (count($deviceList)>0){
			$devObjList = array();
			foreach ($deviceList as $obj){
				$devObjList[] = '[' . $obj['udid'] . '] ' .$obj['name'];
			}
			$devObjList = implode(', ', $devObjList);
			throw new Exception("Can't delete. Next objects attached to schedule: " . $devObjList ."!");
		}

		// get postures and timecards
		$postureList = $this->getPostures($obj);
		$timecardList = $this->getTimecards($obj);
		$posture = new Posture();
		foreach ($postureList as $oneposture){
			$posture->delete($oneposture['obj'], $onlyMark);
		}
		$timecard = new Timecard();
		foreach ($timecardList as $onetimecard){
			$timecard->delete($onetimecard['obj'], $onlyMark);
		}
		$number = DB::query("delete from schedulelink where scheduleid = ?;",array($obj));
		$result = parent::delete($obj, $onlyMark);

		Audit::addRecordVArg(40, $obj, $_SESSION[SESSION_USERID], "deleted");
		return $result;
	}

	public function getPostures($obj, $withAttributes = false)
	{
		$query = "
			SELECT o.obj, o.udid, o.name, description, location, node_id, otype, subtype, protected
			FROM _objs o, _obj_attr oa
			WHERE oa.obj=o.obj
   			AND o.otype='T' AND o.subtype='P'
   			AND oa.attr='SCHEDULE' AND oa.val = ?
   			AND o.deleted = 0;";
		$list = DB::select($query, array($obj));
		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$attributes = $this->getAttributes($row["obj"]);
				$row["attributes"] = $attributes;
			}
		}
		return $list;
	}

	public function getTimecards($obj, $withAttributes = false)
	{
		$query = "
			SELECT o.obj, o.udid, o.name, description, location, node_id, otype, subtype, protected
			FROM _objs o, schedulelink as sl
			WHERE o.obj=sl.timecardid
			AND sl.scheduleid = ?
   			AND o.deleted = 0;";
		$list = DB::select($query, array($obj));
		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$attributes = $this->getAttributes($row["obj"]);
				$row["attributes"] = $attributes;
			}
		}
		return $list;
	}

	function getObjectOnScheduleList($scheduleid=null){
		if ($scheduleid){
			$list = DB::select("SELECT oa.obj, COALESCE(o.udid, '') as UDID, o.name, o.otype, o.subtype, o.protected, oa.val scheduleid FROM _obj_attr as oa, _objs as o where o.obj=oa.obj and o.deleted=0 and oa.attr='SCHEDULEID' and  oa.val = ?;", array($scheduleid));
		}else{
			$list = DB::select("select o.obj, COALESCE(o.udid, '') as UDID, o.name, o.otype, o.subtype, o.protected, oa.val scheduleid from _objs o, _obj_attr oa where o.obj=oa.obj and o.deleted=0 and otype<>'G' and otype<>'T' and o.obj>100 AND oa.attr='SCHEDULEID' order by oa.val;");
		}
		return $list;
	}

	function getScheduleInfo($scheduleid=null){
		$rez = array();
		if ($scheduleid){
			//TODO replace to use
			//SELECT * FROM getObjects(21, 31, 18, null, null);
			$list = DB::select("SELECT obj, name, otype, subtype, protected FROM _objs where otype='T' and subtype='S' and deleted=0 and obj=?;", array($scheduleid));
			$calendarlist = DB::select("SELECT o.obj, o.name, o.otype, o.subtype, o.protected, sl.scheduleid, sl.timecardid, sl.postureid  FROM schedulelink as sl, _objs as o where o.obj = ? and o.obj=sl.scheduleid and o.deleted=0;", array($scheduleid));
			$calendarposturelist = DB::select("SELECT distinct sl.postureid  FROM schedulelink as sl, _objs as o where o.obj = ? and o.obj=sl.scheduleid and o.deleted=0;", array($scheduleid));
			$calendartimecardlist = DB::select("SELECT distinct sl.timecardid FROM schedulelink as sl, _objs as o where o.obj = ? and o.obj=sl.scheduleid and o.deleted=0;", array($scheduleid));
		}else{
			//TODO replace to use
			//SELECT * FROM getObjects(21, 31, 18, null, null);
			$list = DB::select("SELECT obj, name, otype, subtype, protected FROM _objs where otype='T' and subtype='S' and deleted=0;");
			$calendarlist = DB::select("SELECT o.obj, o.name, o.otype, o.subtype, o.protected, sl.scheduleid, sl.timecardid, sl.postureid  FROM schedulelink as sl, _objs as o where o.obj=sl.scheduleid and o.deleted=0;");
			$calendarposturelist = DB::select("SELECT distinct sl.postureid  FROM schedulelink as sl, _objs as o where o.obj=sl.scheduleid and o.deleted=0;");
			$calendartimecardlist = DB::select("SELECT distinct sl.timecardid FROM schedulelink as sl, _objs as o where o.obj=sl.scheduleid and o.deleted=0;");
		}
		if (!(count($list) > 0)) return $rez;
		$list_chr = array ();
		$list_nchr = array();
		foreach ($list as $obj){
			$list_nchr[] = $obj["obj"];
			$list_chr[] = "'".$obj["obj"]."'";
		}
		$schedulelist = implode(", ", $list_nchr);
		$schedulelist_chr = implode (", ", $list_chr);

		$posture_chr_arr = array ();
		$timecard_chr_arr = array ();
		$posture_nchr_arr = array ();
		$timecard_nchr_arr = array ();
		foreach ($calendarposturelist as $obj){
			$posture_chr_arr[] = "'".$obj["postureid"]."'";
			$posture_nchr_arr[] = $obj["postureid"];
		}
		foreach ($calendartimecardlist as $obj){
			$timecard_chr_arr[] = "'".$obj["timecardid"]."'";
			$timecard_nchr_arr[] = $obj["timecardid"];
		}
		$posturelist = implode (", ", $posture_nchr_arr);
		$posturelist_chr = implode (", ", $posture_chr_arr);
		$timecardlist_chr = implode (", ", $timecard_chr_arr);

		$posturelist = $posturelist==''?"0":$posturelist;
		$posturelist_chr = $posturelist_chr==''?"0":$posturelist_chr;
		$timecardlist_chr = $timecardlist_chr==''?"0":$timecardlist_chr;

		$attrlist = DB::select("SELECT * FROM _obj_attr where obj in (".$schedulelist.");");
		$devlist = DB::select("SELECT oa.obj, oa.obj as UDID, o.name, o.otype, o.subtype, o.protected, oa.val as scheduleid FROM _obj_attr as oa, _objs as o where o.obj=oa.obj and o.deleted=0 and oa.attr='SCHEDULEID' and  oa.val in ($schedulelist_chr);");

		$postureattrinfolist = DB::select("SELECT * FROM _obj_attr where obj in ($posturelist_chr);");
		$postureinfolist = DB::select("SELECT * FROM _obj_attr where obj in($posturelist);");
		$timecardinfolist = DB::select("SELECT * FROM _obj_attr where obj in ($timecardlist_chr);");

		//create ScheduleInfo array
		foreach ($list as $obj){
			$sched_info = $obj;
			$sched_info_attr = array();
			foreach ($attrlist as $attrrow){
				if ($attrrow['obj']==$obj['obj']){
					$sched_info_attr[$attrrow['attr']]=$attrrow['val'];
				}
			}
			$sched_info['attributes'] = $sched_info_attr;
			$timezone = 'UTC';
			if (isset($sched_info_attr['TIME_ZONE']) && $sched_info_attr['TIME_ZONE']!==''){
				$timezone = $sched_info_attr['TIME_ZONE'];
			}

			$sched_info_dev = array();
			foreach ($devlist as $devrow){
				if ($devrow['scheduleid']==$obj['obj']){
					$sched_info_dev[$devrow['obj']]=$devrow;
				}
			}
			$sched_info['devices'] = $sched_info_dev;

			$posture = new Posture();
			$sched_info_post = array();
			$postureOnScheduleList = $posture->getUnusedPosturesList($obj['obj']);
			foreach ($postureOnScheduleList as $onePosture){
				$sched_info_post[]=$onePosture['obj'];
			}
			$sched_info['postures'] = $sched_info_post;

			$sched_info_calendar = array();
			foreach ($calendarlist as $calendarrow){
				if ($calendarrow['scheduleid']==$obj['obj']){

					$onecalend = $calendarrow;
					$onecalend['posture'] = array();
					foreach ($postureattrinfolist as $posturerow){
						if ($posturerow['obj']==$calendarrow['postureid']){
							$onecalend['posture']['attributes'][$posturerow['attr']] = $posturerow['val'];
							if ($posturerow['attr'] == 'NAME') {
								$onecalend['title'] = $posturerow['val'];
							}
						}
					}
					foreach ($postureinfolist as $posturerow){
						if ($posturerow['obj']==$calendarrow['postureid'] ){
							if (!is_array($onecalend['posture'])){
								$onecalend['posture'] = array();
							}
							$onecalend['posture']['scheduleattr'][] = $posturerow;
						}
					}

					$timecard = new Timecard();
					$onecalend['timecard'] = array();
					foreach ($timecardinfolist as $timecardrow){
						if ($timecardrow['obj']==$calendarrow['timecardid']){
							$onecalend['timecard']['attributes'][$timecardrow['attr']] = $timecardrow['val'];
							if ($timecardrow['attr'] == 'ICAL_DATA'){
								$rrule = $timecard->icaltoarray($timecardrow['val'], $timezone);
								$onecalend['timecard']['attributes']['ICAL_DATA'] = $rrule;

							}
						}
					}
					$sched_info_calendar[] = $onecalend;

				}
			}
			$sched_info['calendar'] = $sched_info_calendar;
			$rez[$obj['obj']] = $sched_info;
		}
		return $rez;
	}

	function setScheduleInfo($scheduleInfo){
		$timecard = new Timecard();

		$scheduleid = null;
		foreach ($scheduleInfo as $key=>$row){ // parse for scheduleID
			// $key --> scheduleID
			$scheduleid = $key;
			if (isset($row) && !is_null($row)){
				$timezone = 'UTC';
				if (isset($row["attributes"]['TIME_ZONE']) && $row["attributes"]['TIME_ZONE'] !== ''){
					$timezone = $row["attributes"]['TIME_ZONE'];
				}

				// Update posture priority
				$i=2;
				$posture = new Posture();
				foreach($row["postures"] as $postureid){
					if($postureid != $row["attributes"]['DEFAULT_POSTUREID']){
						$posture->setAttributes($postureid, array('PRIORITY' => $i));
						$i++;
					}else{
						$posture->setAttributes($postureid, array('PRIORITY' => 1));
					}
				}

				// get "calendar" array, parse it
				if (isset($row["calendar"])){
					$calendar = $row["calendar"];
					$linkparam = array();
					$onelinkparam = array();

					foreach ($calendar as &$calendarrow){ // parse for calendar
						$onelinkparam['postureid'] = $calendarrow['postureid'];
						if (isset($calendarrow['timecardid']) && $calendarrow['timecardid']!=0 && $calendarrow['timecardid']>100 && isset($calendarrow['timecard']['attributes'])){
							// update timecards
							foreach ($calendarrow['timecard']['attributes'] as $attr=>$val){ // parse for calendar
								if ($attr=="ICAL_DATA"){

									$calendarrow['timecard']['attributes']["ICAL_DATA"] = $timecard->ruleArrToICal($val, $timezone);
								}
							}
							$timecard->setAttributes($calendarrow['timecardid'], $calendarrow['timecard']['attributes']);
							$onelinkparam['timecardid'] = $calendarrow['timecardid'];
						}else if (isset($calendarrow['timecard']['attributes'])){
							// create timecard
							// add attr with time
							foreach ($calendarrow['timecard']['attributes'] as $attr=>$val){ // parse for calendar
								if ($attr=="ICAL_DATA"){
									$calendarrow['timecard']['attributes']["ICAL_DATA"] = $timecard->ruleArrToICal($val, $timezone);
								}
							}
							$onelinkparam['timecardid'] = $timecard->create($calendarrow['timecard']['attributes'], $scheduleid, $onelinkparam['postureid']);
						}
						$linkparam[] = $onelinkparam;
					}

					if (count($linkparam)>0){
						DB::query("DELETE FROM schedulelink where scheduleid = ?;", array($scheduleid));
						foreach ($linkparam as $lrow){
							if (isset($lrow['postureid']) && $lrow['postureid']!=0 && isset($lrow['timecardid']) && $lrow['timecardid']!=0){
								DB::query("INSERT INTO schedulelink(scheduleid, timecardid, postureid) VALUES(?, ?, ?);", array($scheduleid, $lrow['timecardid'], $lrow['postureid']));
								$this->updateSTime($lrow['postureid']);
								$this->updateSTime($lrow['timecardid']);
							}
						}
					}
					$this->updateSTime($scheduleid);

				}
				if (isset($row["deleted"])){
					foreach ($row["deleted"] as $rowfordelete){
						$this->releasePosture($scheduleid, $rowfordelete['postureid'], $rowfordelete['timecardid']);
						$this->updateSTime($rowfordelete['postureid']);
						$this->updateSTime($rowfordelete['timecardid']);
					}
					$this->updateSTime($scheduleid);
				}

			}
		}
		return $this->getScheduleInfo($scheduleid);
	}

	function saveObjectsOnSchedule($scheduleid, array $oldobjlist, array $objlist){//$params -- array objlist;
		//TODO if  $objlist not clear
		$str_oldobjlist = implode(",", $oldobjlist);
		$str_objlist = implode(",", $objlist);
		$list = array();

		if ($str_oldobjlist != ''){
			//get old scheduleid list
			//updateSTime for old scheduleid list
			$sql = "SELECT distinct oa.val  FROM _objs as o, _obj_attr as oa WHERE o.obj=oa.obj and o.deleted=0 and oa.attr='SCHEDULEID' and o.obj in ($str_oldobjlist); ";
			$schedulelist = DB::select($sql);
			foreach($schedulelist as $row){
				if ($row['val'] != '')
					$this->updateSTime($row['val']);
			}

			$list = DB::query("UPDATE _obj_attr as oa SET val='' FROM _objs as o where o.obj=oa.obj and o.deleted=0 and oa.attr='SCHEDULEID' and  o.obj in ($str_oldobjlist);");
		}

		if ($str_objlist != ''){
			$sql = "SELECT distinct oa.val  FROM _objs as o, _obj_attr as oa WHERE o.obj=oa.obj and o.deleted=0 and oa.attr='SCHEDULEID' and o.obj in ($str_objlist)";
			$schedulelist = DB::select($sql);
			foreach($schedulelist as $row){
				if ($row['val'] != '')
					$this->updateSTime($row['val']);
			}

			$list = DB::query("UPDATE _obj_attr as oa SET val='$scheduleid' FROM _objs as o where o.obj=oa.obj and o.deleted=0 and oa.attr='SCHEDULEID' and  o.obj in ($str_objlist);");
		}
		$this->auditObjectsOnSchedule($scheduleid);
		$this->updateSTime($scheduleid);
		$list = $this->getObjectOnScheduleList($scheduleid);
		return $list;
	}

	function auditObjectsOnSchedule($scheduleid){
		$oldSUBJECTSLIST = $this->getAttribute($scheduleid, 'SUBJECTS_LIST');
		$newList = $this->getObjectOnScheduleList($scheduleid);
		$objs = array();
		foreach($newList as $row){
			$objs[] = $row['obj'];
		}
		$newSUBJECTSLIST = implode(",", $objs);
		$this->setAttributes($scheduleid, array('SUBJECTS_LIST' => $newSUBJECTSLIST));
		Audit::addRecordVArg(40, $scheduleid, $_SESSION[SESSION_USERID], "updated", Audit::attrChangeEntity('SUBJECTS_LIST', $oldSUBJECTSLIST, $newSUBJECTSLIST));
	}

	function releasePosture($scheduleid, $postureid, $timecardid){
		if (isset($scheduleid) && isset($postureid) && isset($timecardid)) {
			$timecard = new Timecard();
			$timecard->delete($timecardid, true, $scheduleid, $postureid);
			DB::query("DELETE FROM schedulelink where scheduleid = ? and postureid =? and timecardid = ?;", array($scheduleid, $postureid, $timecardid));
		}

	}

	function convertScheduleInfoToEventSource($scheduleid, $scheduleInfo, $datefrom, $dateto){
		// convert scheduleInfo to iCal
		// get parsed DATA from iCal
		// put data to EventSource array

		//$myscheduleInfo = (array) $scheduleInfo;
		$timecard = new Timecard();
		//$scheduleInfo structure
		$myscheduleInfo = Utils::object_to_array($scheduleInfo);
		$timezone = 'UTC';
		if (isset($myscheduleInfo[$scheduleid]["attributes"]["TIME_ZONE"]) && $myscheduleInfo[$scheduleid]["attributes"]["TIME_ZONE"]!=''){
			$timezone = $myscheduleInfo[$scheduleid]["attributes"]["TIME_ZONE"];
		}

		$rez = array();
		if (isset($myscheduleInfo[$scheduleid]) && isset($myscheduleInfo[$scheduleid]["calendar"]) && !is_null($myscheduleInfo[$scheduleid]["calendar"])){
			$calendar = $myscheduleInfo[$scheduleid]["calendar"];
			foreach ($calendar as &$calendarrow){ // parse for calendar
				if (isset($calendarrow['timecard']['attributes'])){
					// update timecards
					foreach ($calendarrow['timecard']['attributes'] as $attr=>$val){ // parse for calendar
						if ($attr=="ICAL_DATA"){
							$calendarrow['timecard']['attributes']["ICAL_DATA"] = $timecard->ruleArrToICal($val);
						}
					}
				}
			}
			//$calendar is prepared
			$eventSource = array();
			foreach ($calendar as $calrow){
				$event = array();
				if (isset($calrow['posture']['attributes']['NAME'])){
					$event['title'] = $calrow['posture']['attributes']['NAME'];
				}else{
					$event['title'] = $calrow['title'];
				}
				$event['postureid'] = $calrow['postureid'];
				$event['id'] = $calrow['timecardid'];
				$event['timecardid'] = $calrow['timecardid'];

				$event['NAME'] = $calrow['timecard']['attributes']['NAME'];
				$event['DESCRIPTION'] = $calrow['timecard']['attributes']['DESCRIPTION'];
				$event['UDID'] = $calrow['timecard']['attributes']['UDID'];

				$timecard = new Timecard();

				$rrule = $timecard->icaltoarray1($calrow['timecard']['attributes']["ICAL_DATA"], $timezone, $datefrom, $dateto);
				foreach ($rrule as $onetime){
					$event['startTS'] =$onetime['startTS'];
					$event['endTS'] =$onetime['endTS'];
					$event['eventstartTS'] =$onetime['eventstartTS'];
					$event['eventendTS'] =$onetime['eventendTS'];
					$event['startSTR'] =$onetime['startSTR'];
					$event['endSTR'] =$onetime['endSTR'];
					$event['startArr'] =$onetime['startArr'];
					$event['endArr'] =$onetime['endArr'];
					if (isset($onetime['first'])){
						$event['first'] = $onetime['first'];
					}else{
						$event['first'] = false;
					}
					$eventSource[] = $event;
				}
			}
			$rez = $eventSource;

		}
		return $rez;
	}

	function getEventSource($scheduleid, $scheduleInfo, $datefrom, $dateto){

		// get timecards from DB
		// convert RRULE to parsed DATA from iCal
		// put data to EventSource array

		// array[
		//	{
		//		id: timecardid
		//		title: posturename,
		//		start: '2011-04-04'
		//		end:
		//		postureid:postureid
		//	},
		//	{
		//		...
		//	}
		// ]

		//TODO replace to use
		//SELECT * FROM getObjects(21, 31, 18, null, null);
		$calendarlist = DB::select("SELECT o.obj, o.name, o.otype, o.subtype, o.protected, sl.scheduleid, sl.timecardid, sl.postureid  FROM schedulelink as sl, _objs as o where o.obj = ? and o.obj=sl.scheduleid and o.deleted=0;", array($scheduleid));
		$calendarposturelist = DB::select("SELECT distinct sl.postureid  FROM schedulelink as sl, _objs as o where o.obj = ? and o.obj=sl.scheduleid and o.deleted=0;", array($scheduleid));
		$calendartimecardlist = DB::select("SELECT distinct sl.timecardid FROM schedulelink as sl, _objs as o where o.obj = ? and o.obj=sl.scheduleid and o.deleted=0;", array($scheduleid));

		$posture_chr_arr = array ();
		$timecard_chr_arr = array ();
		$posture_nchr_arr = array ();
		$timecard_nchr_arr = array ();
		foreach ($calendarposturelist as $obj){
			$posture_chr_arr[] = "'".$obj["postureid"]."'";
			$posture_nchr_arr[] = $obj["postureid"];
		}
		foreach ($calendartimecardlist as $obj){
			$timecard_chr_arr[] = "'".$obj["timecardid"]."'";
			$timecard_nchr_arr[] = $obj["timecardid"];
		}
		$posturelist = implode (", ", $posture_nchr_arr);
		$posturelist_chr = implode (", ", $posture_chr_arr);
		$timecardlist = implode (", ", $timecard_nchr_arr);
		$timecardlist_chr = implode (", ", $timecard_chr_arr);

		$posturelist_chr = $posturelist_chr==''?"0":$posturelist_chr;
		$timecardlist_chr = $timecardlist_chr==''?"0":$timecardlist_chr;

		$postureattrinfolist = DB::select("SELECT * FROM _obj_attr where obj in ($posturelist_chr) and attr='NAME';");
		$timecardinfolist = DB::select("SELECT * FROM _obj_attr where obj in ($timecardlist_chr) and (attr='ICAL_DATA' );");
		$scheduleattrlist = DB::select("SELECT * FROM _obj_attr where obj=?;", array($scheduleid));
		$timecardattributes = DB::select("SELECT * FROM _obj_attr where obj in ($timecardlist_chr);");

		$timezone='UTC';
		foreach ($scheduleattrlist as $row){
			if($row['attr'] == 'TIME_ZONE' && $row['val'] !== ''){
				$timezone = $row['val'];
			}
		}

		//create ScheduleInfo array
		$eventSource = array();

		foreach ($calendarlist as $calendarrow){
			$event = array();
			foreach ($postureattrinfolist as $posturerow){
				if ($posturerow['obj']==$calendarrow['postureid'] && $posturerow['attr'] = 'NAME'){
					$event['title'] = $posturerow['val'];
					$event['postureid'] = $posturerow['obj'];
				}
			}
			foreach ($timecardattributes as $timecardrow){
				if ($timecardrow['obj']==$calendarrow['timecardid'] && ($timecardrow['attr'] == 'NAME' || $timecardrow['attr'] == 'UDID' || $timecardrow['attr'] == 'DESCRIPTION' || $timecardrow['attr'] == 'SUMMARY')) {
					$event[$timecardrow['attr']] = $timecardrow['val'];
				};
			}

			$timecard = new Timecard();
			foreach ($timecardinfolist as $timecardrow){
				if ($timecardrow['obj']==$calendarrow['timecardid'] && $timecardrow['attr'] == 'ICAL_DATA'){
					$event['id'] =$timecardrow['obj'];
					$event['timecardid'] =$timecardrow['obj'];
					$rrule = $timecard->icaltoarray1($timecardrow['val'], $timezone, $datefrom, $dateto);
					foreach ($rrule as $onetime){
						$event['startTS'] =$onetime['startTS'];
						$event['endTS'] =$onetime['endTS'];
						$event['eventstartTS'] =$onetime['eventstartTS'];
						$event['eventendTS'] =$onetime['eventendTS'];
						$event['startSTR'] =$onetime['startSTR'];
						$event['endSTR'] =$onetime['endSTR'];
					$event['startArr'] =$onetime['startArr'];
					$event['endArr'] =$onetime['endArr'];

						if (isset($onetime['first'])){
							$event['first'] = $onetime['first'];
						}else{
							$event['first'] = false;
						}
						$eventSource[] = $event;
					}
				}
			}
		}
		$rez = $eventSource;
		return $rez;
	}

	/**
	 * get default posture id
	 *
	 * @param $scheduleid
	 * @return null|int
	 */
	public function getDefaultPostureId($scheduleid)
	{
		return $this->getAttribute($scheduleid, "DEFAULT_POSTUREID");
	}

	/**
	 * get describe-info about postures and timecards on schedule
	 *
	 * @param $scheduleid
	 * @param bool $useDefault
	 * @return array
	 */
	public function getPosturesInfo($scheduleid, $useDefault=false, $allPostures=false, $actualOnly = false)
	{
		if($useDefault) {
			$default_posture = $this->getDefaultPostureId($scheduleid);
		}

		if($allPostures) {

			$posturelist = DB::select("SELECT distinct oa.obj as postureid, oa.val FROM _objs as o, _obj_attr as oa where oa.obj in (select obj from _obj_attr where attr='SCHEDULE' and val=?) and o.deleted=0 and o.obj=oa.obj and oa.attr='PRIORITY' order by oa.val;", array($scheduleid));
		}else{
			$posturelist = DB::select("SELECT distinct sl.postureid, oa.val FROM schedulelink as sl, _objs as o, _obj_attr as oa where sl.scheduleid = ? and o.obj=sl.postureid and o.deleted=0 and o.obj=oa.obj and oa.attr='PRIORITY' order by oa.val;", array($scheduleid));
		}

		array_unshift($posturelist, array('postureid' => $default_posture));
		$posture = new Posture();
		$ret = array();
		foreach ($posturelist as $postureid){
			$postureinfo = $posture->getPostureInfo($postureid['postureid'], $actualOnly);
			$ret[$postureid['postureid']] = $postureinfo;
		}
		return $ret;
	}
}
