<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: Role.php 26714 2012-09-05 14:52:33Z teetov $
 * ------------------------------------------------------------------------------
 * This class represents logic for Role
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions LLC 2006
 * ------------------------------------------------------------------------------
 */

class Role extends CredentialContainer
{
	/**
	 * predefined role obj id's from DB
	 */
	const ADMIN = 31;
	const MANAGER = 34;
	const OPERATOR = 33;
	const VIEWER = 32;

	public function __construct()
	{
		$this->otype = 'R';
		$this->subtype = '*';
		parent::__construct("role");
	}

	public function add($parentobj, $childobj)
	{
		parent::add($parentobj, $childobj, "V");
	}

	/**
	 * @throws InvalidArgumentException
	 * @param array $attributes
	 * @return int
	 */
	public function create(array $attributes)
	{
		// check for NAME uniqueness
		if (isset($attributes["NAME"]))
		{
			$value = $attributes["NAME"];
			$checkObj = $this->checkUniqueness(null, "NAME", $value);
			if (isset($checkObj))
			{
				throw new InvalidArgumentException(sprintf("role with obj=%s and name=%s already exists", $checkObj, $value));
			}
		}

		return parent::create($attributes);
	}

	/**
	 * set role attributes
	 *
	 * @param  int   $obj
	 * @param  array $attributes
	 * @throws InvalidArgumentException
	 */
	public function setAttributes($obj, array $attributes)
	{
		if ($this->isProtected($obj))
			throw new InvalidArgumentException(sprintf("role with obj=%s is protected", $obj));

		// check for NAME uniqueness
		if (isset($attributes["NAME"]))
		{
			$checkObj = $this->checkUniqueness($obj, "NAME", $attributes["NAME"]);
			if (isset($checkObj) > 0)
			{
				throw new InvalidArgumentException(sprintf("role with obj=%s and name=%s already exists", $checkObj, $attributes["NAME"]));
			}
		}

		parent::setAttributes($obj, $attributes);
	}

	/**
	 * select users from role
	 *
	 * @param  int  $roleid
	 * @param  bool $withAttributes return with attributes
	 * @throws InvalidArgumentException
	 * @return array
	 */
	public function getUsers($roleid, $withAttributes = false)
	{
		if (!isset($roleid))
		{
			throw new InvalidArgumentException("Some parameters are missing.");
		}

		$list = DB::select("SELECT * FROM getUsers(?);", array($roleid));

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$row["attributes"] = $this->getAttributes($row["obj"]);
			}
		}

		return $list;
	}

	/**
	 * select sets from role
	 *
	 * @param  int  $roleid
	 * @param  bool $withAttributes return with attributes
	 * @throws InvalidArgumentException
	 * @return array
	 */
	public function getSets($roleid, $withAttributes = false)
	{
		if (!isset($roleid))
		{
			throw new InvalidArgumentException("Some parameters are missing.");
		}

		$list = DB::select("SELECT * FROM getSetsFromRole(?);", array($roleid));

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$row["attributes"] = $this->getAttributes($row["obj"]);
			}
		}

		return $list;
	}

	/**
	 * select devices from role with specified otype and subtype
	 *
	 * @param int         $roleid
	 * @param string|null $otype
	 * @param string|null $subtype
	 * @throws InvalidArgumentException
	 * @return array
	 */
	public function getSetsWithDevices($roleid, $otype = null, $subtype = null)
	{
		if (!isset($roleid))
		{
			throw new InvalidArgumentException("Some parameters are missing.");
		}

		$list = DB::select("SELECT * FROM getObjectsFromRole(?, ?, ?);", array($roleid, $otype, $subtype));
		return $list;
	}

	public function setPermission($roleid, $setid, $permission)
	{
		if ($this->isLinkProtected($setid, $roleid))
		{
			throw new Exception("Could not set Special Credentials. Link protected for changes");
		}

		DB::query(
			"UPDATE _links SET permission = ? WHERE obj_res = ? AND obj_cons = ? AND protected = ?;",
			array($permission, $setid, $roleid, 0)
		);
	}

	public function setSpecialCredentials($roleid, $setid, $credentials)
	{
		if ($this->isLinkProtected($setid, $roleid))
		{
			throw new Exception("Could not set Special Credentials. Link protected for changes");
		}

		DB::query(
			"UPDATE _links SET special_credentials = ? WHERE obj_res = ? AND obj_cons = ? AND protected = ?;",
			array($credentials, $setid, $roleid, 0)
		);
	}

}
