<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: Proxy.php 33229 2015-10-29 15:21:32Z astarostin $
 * ------------------------------------------------------------------------------
 * This class represents Proxy class for calling methods
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions LLC 2006
 * ------------------------------------------------------------------------------
 */

/**
 * Proxy is a class that has the knowledge of every method available, their parameters
 * and default values.
 * Proxy receives all the API calls requests via call() and forwards them to the right
 * object, with the parameters in the right order.
 *
 * It will also log the performance of API calls (time spent, parameter values, etc.) if logger available
 */
class Proxy
{
	/**
	 * @var bool
	 */
	private static $_isDebug = SESSION_DEBUG;

	/**
	 * Will execute $className->$methodName($parametersValues)
	 * If any error is detected (wrong number of parameters, method not found, class not found, etc.)
	 * it will throw an exception
	 *
	 * It also logs the API calls, with the parameters values, the returned value, the performance, etc.
	 * You can enable logging by setting $isLog = true
	 *
	 * @param array  $classNameList the class name list
	 * @param string $methodName the method name
	 * @param array  $args the parameters pairs (name=>value)
	 * @throws InvalidArgumentException
	 * @throws BadMethodCallException
	 * @return array|mixed
	 */
	public static function call(array $classNameList, $methodName, array $args = array())
	{
		$response = array();
		$error = false;
		$exception = null;

		try
		{
			/**
			 * @var ReflectionClass
			 */
			$reflectionClass = null;
			/**
			 * @var ReflectionMethod
			 */
			$reflectionMethod = null;

			$methodExists = false;
			foreach ($classNameList as $className)
			{
				$reflectionClass = new ReflectionClass($className);
				try
				{
					$reflectionMethod = $reflectionClass->getMethod($methodName);
					$methodExists = true;
					break;
				}
				catch (ReflectionException $e)
				{}
			}
			if (!$methodExists)
			{
				throw new BadMethodCallException(sprintf("Method %s does not exist", $methodName));
			}
			$parameters = $reflectionMethod->getParameters();

			$genAPI = new GenAPI();
			$reflectionClassArgs = array();
			foreach ($parameters as $parameter)
			{
				/* @var ReflectionParameter $parameter */
				$parameterName = $parameter->getName();

				if (isset($args[$parameterName]))
				{
					$parameter2type = $genAPI->parameter2type($reflectionMethod->getDocComment());
					if (isset($parameter2type[$parameterName])
						&& $parameter2type[$parameterName]["type"] == "array"
						&& strrpos($parameter2type[$parameterName]["description"], "JSONobject") === 0)
					{
						$arg = json_decode($args[$parameterName]);

						if (!is_object($arg))
							throw new InvalidArgumentException(sprintf("Parameter '%s' must be an object", $parameterName));

						$arg = json_decode($args[$parameterName], true);

						$reflectionClassArgs[] = $arg;
					} else
					if ($parameter->isArray())
					{
						$arg = json_decode($args[$parameterName]);

						if (!is_array($arg))
							throw new InvalidArgumentException(sprintf("Parameter '%s' must be an array", $parameterName));

						$arg = json_decode($args[$parameterName], true);

						$reflectionClassArgs[] = $arg;
					} else {
						$reflectionClassArgs[] = $args[$parameterName];
					}
				} else {
					if (!$parameter->isOptional())
						throw new InvalidArgumentException(sprintf("Parameter '%s' is not optional", $parameterName));

					$reflectionClassArgs[] = $parameter->getDefaultValue();
				}
			}

			$time = microtime(true);

			$response = $reflectionMethod->invokeArgs($reflectionClass, $reflectionClassArgs);

			$time = microtime(true) - $time;

			if (self::$_isDebug)
			{
				$response["debug"] = array();
				$response["debug"]["time"] = $time * 1000; // ms
			}

			$response["error"] = "";
			$response["code"] = 200;
		}
		catch (BadMethodCallException $e)
		{
			$error = true;
			$exception = $e;
			$response["error"] = $e->getMessage();
			$response["code"] = 400;
		}
		catch (InvalidArgumentException $e)
		{
			$error = true;
			$exception = $e;
			$response["error"] = $e->getMessage();
			$response["code"] = 400;
		}
		catch (AuthException $e)
		{
			$error = true;
			$exception = $e;
			$response["error"] = $e->getMessage();
			$response["code"] = 401;
		}
		catch (DBException $e)
		{
			$error = true;
			$exception = $e;
			$response["error"] = self::$_isDebug ? $e->getMessage() : "Incorrect request";
			$response["code"] = 500;
		}
		catch (Exception $e)
		{
			$error = true;
			$exception = $e;
			$response["error"] = $e->getMessage();
			$response["code"] = 500;
		}

		if (self::$_isDebug)
		{
			if ($error)
			{
				$response["debug"] = array();
				$response["debug"]["callstack"] = $exception->getTraceAsString();
			}

			$response["debug"]["memory"] = memory_get_usage(true) / 1024 / 1024; // Mb
			$response["debug"]["memoryPeak"] = memory_get_peak_usage(true) / 1024 / 1024; // Mb
			$response["debug"]["DB"] = array();
			$response["debug"]["DB"]["time"] = DB::getTime(); // ms
			$response["debug"]["DB"]["number"] = DB::getNumber();
			$response["debug"]["DB"]["log"] = DB::getLog();
		}

		return $response;
	}
}
