<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: Obj.php 33243 2015-11-01 23:59:59Z astarostin $
 * ------------------------------------------------------------------------------
 * Basic class for all abject in the system
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2010
 * ------------------------------------------------------------------------------
 */

class Obj
{
	/**
	 * @var string
	 */
	protected $otype;

	/**
	 * @var string
	 */
	protected $subtype;

	/**
	 * @var string
	 */
	protected $type;

	/**
	 * @var array
	 */
	protected $name = array();

	/**
	 * @var array
	 */
	protected $isProtected = array();

	/**
	 * @var array
	 */
	protected $modifiedObjects = array("edit" => array(), "delete" => array());

	/**
	 * predefined obj id's from DB
	 */
	const CONTROL_PANEL = 51;

	/**
	 * @param  string $otype
	 * @param  string $subtype
	 * @return \Obj
	 */
	public function __construct($otype = null, $subtype = null)
	{
		$this->otype = $otype;
		$this->subtype = $subtype;
	}

	/**
	 * create new object in system
	 * @param  array $attributes
	 * @param  null|int $parentObj not used
	 * @return int|null objid
	 */
	public function create(array $attributes, $parentObj = null)
	{
		$obj = null;

		if (isset($this->otype) && isset($this->subtype))
		{
			$number = DB::query("INSERT INTO _objs(name, otype, subtype) VALUES (?, ?, ?);", array("", $this->otype, $this->subtype));
			if ($number > 0)
			{
				$list = DB::select("SELECT lastval from lastval();");
				if (count($list) > 0)
				{
					$obj = $list[0]["lastval"];
					$this->createAttributes($obj, $attributes);

					// add object to default set and protect it
					$set = new Set();
					$types = Obj::getTypes();
					$defaultsetobj = $types[$this->otype][$this->subtype]["defaultsetobj"];
					if (isset($defaultsetobj))
					{
						$set->add($defaultsetobj, $obj);
					}

					$this->updateSTime($obj);
				}
			}
		}

		return $obj;
	}

	/**
	 * create block attributes
	 * @param  $obj
	 * @param  array $attributes list of attributes
	 * @return void
	 */
	protected function createAttributes($obj, array $attributes)
	{
		foreach ($attributes as $key => $value)
		{
			$this->createAttribute($obj, $key, $value);
		}
	}

	/**
	 * create attribute
	 * @param  int $obj
	 * @param  string $attribute
	 * @param  string $value
	 * @return int|resource
	 */
	protected function createAttribute($obj, $attribute, $value)
	{
		if (is_object($value) || is_array($value))
			$value = json_encode($value);

		if ($value === null)
		{
			$value = "";
		}

		$result = DB::query(
			"INSERT INTO _obj_attr(obj, attr, val) VALUES (?, ?, ?)",
			array($obj, $attribute, $value)
		);

		ObjectManager::setAttribute($obj, $attribute, $value, $this->otype, $this->subtype);

		return $result;
	}

	/**
	 * delete attribute
	 *
	 * @param  int $obj
	 * @param  string $attribute
	 * @return int|resource
	 */
	protected function deleteAttribute($obj, $attribute)
	{
		$result = DB::query(
			"DELETE FROM _obj_attr WHERE obj = ? AND attr = ?;",
			array($obj, $attribute)
		);

		return $result;
	}

	/**
	 * delete all attributes from object
	 *
	 * @param  $obj
	 * @return int|resource
	 */
	protected function deleteAllAttributes($obj)
	{
		$result = DB::query(
			"DELETE FROM _obj_attr WHERE obj = ?;",
			array($obj)
		);

		return $result;
	}

	/**
	 * sync attributes with template
	 * @param  int $obj
	 * @param  array $newAttributesList
	 * @return array do
	 */
	public function syncAttributesWithTemplate($obj, array $newAttributesList)
	{
		// we need only DB attributes
		$object = new Obj();
		$attributes = $object->getAttributes($obj);

		$do = array(
			"delete" => array(),
			"create" => array()
		);

		// add attributes from template to DB
		foreach ($newAttributesList as $attribute => $value)
		{
			if (!isset($attributes[$attribute]))
			{
				$number = $this->createAttribute($obj, $attribute, $value);
				if ($number > 0)
				{
					if (!isset($do["create"][$obj]))
						$do["create"][$obj] = array();

					$do["create"][$obj][$attribute] = $value;
				}
			}
		}

		// remove attributes from DB
		foreach ($attributes as $attribute => $value)
		{
			if (!isset($newAttributesList[$attribute]))
			{
				$number = $this->deleteAttribute($obj, $attribute);
				if ($number > 0)
				{
					if (!isset($do["delete"][$obj]))
						$do["delete"][$obj] = array();

					$do["delete"][$obj][$attribute] = null;
				}
			}
		}

		return $do;
	}

	/**
	 * create block
	 * @param  $obj
	 * @param  string $value
	 * @param  string $mime
	 * @param  string $name
	 * @param  string $description
	 * @return string block
	 */
	protected function createBlock($obj, $value, $mime, $name, $description = "")
	{
		$value = str_replace(array("\\\\", "''"), array("\\", "'"), pg_escape_bytea(DB::getConnection(), $value));

		$queryString = "INSERT INTO obj_block(obj, name, description, mime, block) VALUES (?, ?, ?, ?, ?)";
		$result = DB::query($queryString, array($obj, $name, $description, $mime, $value));

		$blockId = null;

		if ($result > 0)
		{
			$list = DB::select("SELECT currval_seqblockid() as currval;");
			if (count($list) > 0)
				$blockId = $list[0]['currval'];
		}

		return $blockId;
	}

	/**
	 * check for block existense
	 * @param  $obj
	 * @param  $blockId
	 * @return bool
	 */
	protected function checkBlockExistence($obj, $blockId)
	{
		if (!isset($blockId))
			return false;

		$isExist = false;

		$list = DB::select("SELECT blockid FROM obj_block WHERE obj = ? AND blockid = ?;", array($obj, $blockId));
		if (count($list) > 0)
		    $isExist = true;

		return $isExist;
	}

	/**
	 * set object attributes
	 *
	 * @param  int $obj
	 * @param  array $attributes
	 */
	public function setAttributes($obj, array $attributes)
	{
		if (isset($attributes["NAME"]))
		{
			unset($this->name[$obj]);
		}

		foreach ($attributes as $key => $value)
		{
			$this->setAttribute($obj, $key, $value);
		}

		$this->updateSTime($obj);
	}

	/**
	 * set object attribute
	 *
	 * @param  int    $obj
	 * @param  mixed  $attribute
	 * @param  string $value
	 * @return bool
	 */
	protected function setAttribute($obj, $attribute, $value)
	{
		$result = false;

		if (is_object($value) || is_array($value))
			$value = json_encode($value);

		if ($value === null)
		{
			$value = "";
		}
		$value = trim($value);

		$queryString =
			"UPDATE _obj_attr
				SET val = ?
				FROM _objs
				WHERE
					_objs.obj = _obj_attr.obj
					AND _objs.deleted = 0
					AND _objs.obj = ?
					AND attr = ?;";
		$number = DB::query($queryString, array($value, $obj, $attribute));
		if ($number > 0)
		{
			$result = true;
			ObjectManager::setAttribute($obj, $attribute, $value, $this->otype, $this->subtype);
		}

		return $result;
	}

	/**
	 * create or update block value
	 * @param  $obj
	 * @param  $blockId
	 * @param  string $value
	 * @param  string $mime
	 * @param  string $name
	 * @param  string $description
	 * @return string
	 */
	public function setBlock($obj, $blockId, $value, $mime, $name, $description)
	{
		if (!isset($blockId) || !$this->checkBlockExistence($obj, $blockId))
		{
			$blockId = $this->createBlock($obj, $value, $mime, $name, $description);
		} else {
			$value = str_replace(array("\\\\", "''"), array("\\", "'"), pg_escape_bytea(DB::getConnection(), $value));

			$number = DB::query(
				"UPDATE obj_block
					SET block = ?, mime = ?, name = ?, description = ?
					FROM _objs
					WHERE
						_objs.obj = obj_block.obj
						AND _objs.deleted = 0
						AND _objs.obj = ?
						AND obj_block.blockid = ?;",
				array($value, $mime, $name, $description, $obj, $blockId));
			if ($number == 0)
			{
				$blockId = null;
			}
		}

		return $blockId;
	}

	/**
	 * delete block
	 * @param  $obj
	 * @param  $blockId
	 * @return int|resource
	 */
	public function deleteBlock($obj, $blockId)
	{
		$result = DB::query("DELETE FROM obj_block WHERE obj = ? AND blockid = ?;", array($obj, $blockId));

		return $result;
	}

	/**
	 * get list of object attributes
	 *
	 * @param int $obj
	 * @return array list of attributes
	 */
	public function getAttributes($obj)
	{
		$queryString =
			"SELECT _obj_attr.attr, _obj_attr.val
				FROM _objs, _obj_attr
				WHERE
					_objs.obj = _obj_attr.obj
					AND _objs.deleted = 0
					AND _objs.obj = ?;";
		$list = DB::select($queryString, array($obj));

		$attributes = array();
		foreach ($list as $row)
		{
			$attributes[$row["attr"]] = $row["val"];
		}
		return $attributes;
	}

	/**
	 * get object attribute value
	 * @param  $obj
	 * @param  string $attribute
	 * @return string|null
	 */
	public function getAttribute($obj, $attribute)
	{
		$value = null;

		$queryString =
			"SELECT _obj_attr.val
				FROM _objs, _obj_attr
				WHERE
					_objs.obj = _obj_attr.obj
					AND _objs.deleted = 0
					AND _objs.obj = ?
					AND attr = ?;";
		$list = DB::select($queryString, array($obj, $attribute));
		if (count($list) > 0)
			$value = $list[0]['val'];

		return $value;
	}

	/**
	 * get block value
	 * @param  $obj
	 * @param  $blockId
	 * @return string
	 */
	public function getBlock($obj, $blockId)
	{
		$value = null;
		$mime = null;
		$name = null;
		$description = null;

		$queryString =
			"SELECT obj_block.name, obj_block.description, mime, block
				FROM _objs, obj_block
				WHERE
					_objs.obj = obj_block.obj
					AND _objs.deleted = 0
					AND _objs.obj = ?
					AND blockid = ?;";
		$list = DB::select($queryString, array($obj, $blockId));
		if (count($list) > 0)
		{
			$name = $list[0]['name'];
			$description = $list[0]['description'];
			$mime = $list[0]['mime'];
			$value = pg_unescape_bytea($list[0]['block']);
		}

		return array(
			"name" => $name,
			"description" => $description,
			"mime" => $mime,
			"value" => $value
		);
	}

	/**
	 * get list of url's for downloading block value
	 * @param  $obj
	 * @return array list of blocks with url's
	 */
	public function getBlocksInfo($obj)
	{
		$blocks = array();

		$queryString =
			"SELECT obj_block.blockid, obj_block.description, obj_block.mime
				FROM _objs, obj_block
				WHERE
					_objs.obj = obj_block.obj
					AND _objs.deleted = 0
					AND _objs.obj = ?;";
		$list = DB::select($queryString, array($obj));

		foreach ($list as $row)
		{
			$blocks[$row["blockid"]] = array(
				"url" => $this->getBlockURL($obj, $row["blockid"]),
				"description" => $row["description"],
				"mime" => $row["mime"],
			);
		}

		return $blocks;
	}

	/**
	 * get url for downloading block attribute
	 * @param  $obj
	 * @param  $blockId
	 * @return string
	 */
	public function getBlockURL($obj, $blockId)
	{
		$name = null;

		$list = DB::select(
			"SELECT obj_block.name
				FROM _objs, obj_block
				WHERE
					_objs.obj = obj_block.obj
					AND _objs.deleted = 0
					AND obj_block.obj = ?
					AND obj_block.blockid = ?;",
			array($obj, $blockId));
		if (count($list) > 0)
			$name = $list[0]["name"];

		$url = null;
		if (isset($name))
			$url = "/api/$obj/$blockId/$name";

		return $url;
	}

	/**
	 * delete object
	 * @throws InvalidArgumentException
	 * @param  $obj
	 * @param  bool $onlyMark - set deleted = 1 (by default) or delete from DB
	 * @return bool
	 */
	public function delete($obj, $onlyMark = true)
	{
		if (!isset($obj))
		{
			throw new InvalidArgumentException("Some parameters are missing.");
		}

		$result = false;

		if ($onlyMark)
		{
			$number = DB::query("
				UPDATE _objs
					SET
						deleted = 1,
						stime = now() at time zone 'UTC'
					WHERE
						obj = ?
						AND protected = 0;",
				array($obj)
			);
		} else {
			$number = DB::query("
				DELETE FROM _objs
					WHERE
						obj = ?
						AND protected = 0;",
				array($obj)
			);
		}

		if ($number > 0)
		{
			$result = true;

			ObjectManager::delete($obj, $this->otype, $this->subtype);
		}

		return $result;
	}

	/**
	 * get object name
	 * @param  $obj
	 * @return string
	 */
	public function getName($obj)
	{
		if (!isset($this->name[$obj]))
		{
			$queryString = "SELECT name FROM _objs WHERE obj = ? AND deleted = 0;";
			$list = DB::select($queryString, array($obj));
			if (count($list) > 0)
				$this->name[$obj] = $list[0]['name'];
		}
		return $this->name[$obj];
	}

	/**
	 * get object template name
	 * @param  $obj
	 * @return string
	 */
	public function getTemplateName($obj)
	{
		$templateName = null;

		$list = DB::select(
			"SELECT _obj_otype_subtype.template
				FROM _obj_otype_subtype, _objs
				WHERE
					_objs.deleted = 0
					AND _obj_otype_subtype.disabled = 0
					AND _objs.otype = _obj_otype_subtype.otype
					AND _objs.subtype = _obj_otype_subtype.subtype
					AND _objs.obj = ?;",
			array($obj)
		);
		if (count($list) > 0)
		{
			$templateName = $list[0]["template"];
		}

		return $templateName;
	}

	/**
	 * get object UNI
	 * @static
	 * @param  $obj
	 * @return string object UNI
	 */
	public static function getUNI($obj)
	{
		$uni = null;

		$query = "SELECT obj, node_ip as uni, name as node_uni, otype, subtype FROM _objs WHERE obj = ?;";
		$list = DB::select($query, array($obj));

		if (count($list) > 0)
		{
			$row = $list[0];

			$uni = $row["uni"];
			if ($row["otype"] == "D" && $row["subtype"] == "N")
			{
				$uni = $row["node_uni"];
			}
		}

		return $uni;
	}

	/**
	 * get object type
	 * @param  $obj
	 * @return string object type
	 */
	public function getType($obj)
	{
		$type = null;

		if (isset($obj) && $obj != "" && is_numeric($obj))
		{
			if ($obj == 53)
			{
				$type = "identity";
			} else {
				$list = DB::select(
					"SELECT template
						FROM _obj_otype_subtype, _objs
						WHERE
							_objs.obj = ?
							AND _obj_otype_subtype.disabled = 0
							AND _objs.otype = _obj_otype_subtype.otype
							AND _objs.subtype = _obj_otype_subtype.subtype;",
					array($obj)
				);
				if (count($list) > 0)
				{
					$type = $list[0]["template"];
				}
			}
		}

		return $type;
	}

	/**
	 * get list of object types in system
	 * @static
	 * @param  bool $byTemplate
	 * @return array list of relations type to otype/subtype
	 */
	public static function getTypes($byTemplate = false)
	{
		$result = array();

		$list = DB::select(
			"SELECT *
				FROM _obj_otype_subtype
				WHERE _obj_otype_subtype.disabled = 0;",
			array()
		);

		if (!$byTemplate)
		{
			foreach ($list as $row)
			{
				$otype = $row["otype"];
				$subtype = $row["subtype"];
				$name = $row["name"];
				$template = $row["template"];
				$defaultsetobj = $row["defaultsetobj"];

				if (!isset($result[$otype]))
				{
					$result[$otype] = array();
				}

				$result[$otype][$subtype] = array(
					"name" => $name,
					"template" => $template,
					"defaultsetobj" => $defaultsetobj
				);
			}
		} else {
			foreach ($list as $row)
			{
				$otype = $row["otype"];
				$subtype = $row["subtype"];
				$name = $row["name"];
				$template = $row["template"];

				if (!isset($result[$template]))
				{
					$result[$template] = array();
				}

				$result[$template]["otype"] = $otype;
				$result[$template]["subtype"] = $subtype;
				$result[$template]["name"] = $name;
			}
		}

		return $result;
	}

	/**
	 * check if object is exists
	 * @param  $obj
	 * @return bool
	 */
	public function isExists($obj)
	{
		$list = DB::select("SELECT obj, name FROM _objs WHERE obj = ?;", array($obj));

		$isExists = false;
		if (count($list) > 0 )
		{
			$isExists = true;
		}

		return $isExists;
	}

	/**
	 * check if object is protected
	 * return null if object not exists
	 * @param  $obj
	 * @return bool|null
	 */
	public function isProtected($obj)
	{
		if (!isset($this->isProtected[$obj]))
		{
			$list = DB::select("SELECT protected FROM _objs WHERE obj = ?;", array($obj));

			if (count($list) > 0)
			{
				$this->isProtected[$obj] = false;
				if ($list[0]["protected"] == "1")
				{
					$this->isProtected[$obj] = true;
				}
			}
		}

		return $this->isProtected[$obj];
	}

	/**
	 * check for object uniqueness with attribute=>value pair
	 * @param  $obj
	 * @param  string $attribute
	 * @param  string $value
	 * @param  bool $isDeleted
	 * @return int obj of already existens object with specified attribute
	 */
	public function checkUniqueness($obj, $attribute, $value, $isDeleted = false)
	{
		$checkObj = null;
		$deletedstr = !$isDeleted ? " AND _objs.deleted = 0 " : "";
		$list = DB::select(
			"SELECT _objs.obj
				FROM _objs, _obj_attr
				WHERE
					_objs.obj = _obj_attr.obj " . $deletedstr . "
					AND (COALESCE(?, '') = '' OR _objs.otype = ?)
					AND (COALESCE(?, '') = '' OR _objs.subtype = ?)
					AND _objs.obj != COALESCE(?, 0)
					AND attr = ?
					AND val = ?;",
			array($this->otype, $this->otype, $this->subtype, $this->subtype, $obj, $attribute, $value)
		);

		if (count($list) > 0)
		{
			$checkObj = $list[0]["obj"];
		}

		return $checkObj;
	}

	/**
	 * return list of child objects
	 *
	 * @param  int $obj
	 * @param  string|null $otype
	 * @param  string|null $subtype
	 * @param  bool|null $withAttributes
	 * @param  bool|null $withBlocks
	 * @return array list of child objects
	 */
	public function getObjects($obj, $otype = null, $subtype = null, $withAttributes = false, $withBlocks = false)
	{
		return array();
	}

	/**
	 * update object change time
	 * @param  $obj
	 * @return void
	 */
	protected function updateSTime($obj)
	{
		DB::query(
			"UPDATE _objs
				SET
					stime = now() at time zone 'UTC'
				WHERE
					obj = ?
					AND deleted = 0;",
			array($obj)
		);
	}
}
