<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: Node.php 33538 2016-01-27 00:39:59Z astarostin $
 * ------------------------------------------------------------------------------
 * This class contains logic for Node
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions LLC 2006
 * ------------------------------------------------------------------------------
 */

class Node extends ObjectTemplate
{
	private static $systemStatus = null;

	public function __construct($nodeid = null)
	{
		if (isset($nodeid))
		{
			$this->nodeid = $this->getMasterNodeId();
		}
		parent::__construct("node");
	}

	/**
	 * get master node id
	 * @static
	 * @return int|null
	 */
	public static function getMasterNodeId()
	{
		$masterUNI = Node::getUNI();
		$queryString = "SELECT obj FROM _objs WHERE otype = 'D' AND subtype = 'N' AND name = ? and deleted = 0;";
		$list = DB::select($queryString, array($masterUNI));

		$nodeid = null;
		if (count($list) > 0)
			$nodeid = $list[0]['obj'];

		return $nodeid;
	}

	/**
	 *
	 * @static
	 * @param $uni
	 * @return int|null
	 */
	public static function getNodeId($uni)
	{
		$queryString = "SELECT obj FROM _objs WHERE otype = 'D' AND subtype = 'N' AND name = ? and deleted = 0;";
		$list = DB::select($queryString, array($uni));

		$nodeid = null;
		if (count($list) > 0)
			$nodeid = $list[0]['obj'];

		return $nodeid;
	}

	/**
	 *
	 * @static
	 * @param $uni
	 * @throws InvalidArgumentException
	 * @return array
	 */
	public static function getStatistics($uni)
	{
		$nodeid = Node::getNodeId($uni);
		if (!$nodeid) {
			throw new InvalidArgumentException("Node '$uni' not found");
		}

		$queryString = "SELECT attr,val FROM _obj_attr WHERE obj=? AND attr LIKE 'STAT_%'";
		$list = DB::select($queryString, array($nodeid), "attr");

		return $list;
	}

	/**
	 * add object to node
	 *
	 * @param $type
	 * @param array $attributes
	 * @param null $parentObj
	 * @throws InvalidArgumentException
	 * @return int|null|void
	 */
	public function add($type, array $attributes, $parentObj = null)
	{
		$object = Factory::getByType($type);

		$type = $object->getType($parentObj);
		if ($type != "node")
			throw new InvalidArgumentException("type of parentObj should be 'node'");

		$obj = null;
		if (isset($object))
		{
			$obj = $object->create($attributes, $parentObj);
		}
		return $obj;
	}

	/**
	 * get Axis camera list
	 *
	 * @throws Exception
	 * @return array
	 */
	public function getAxisCameraList()
	{
		$command = "VERSIONER_PERL_VERSION=" . $_SERVER["VERSIONER_PERL_VERSION"] . " APL=" . $_SERVER["APL"] . " PERL5LIB=" . $_SERVER["PERL5LIB"] . " " . APL_PATH . "/conf/bin/axis_list.pl";

		$output = array();
		exec(escapeshellcmd($command), $output, $return_var);
		if ($return_var != 0)
		{
			$error = sprintf(__("Return value: %s"), $return_var);
			throw new Exception($error);
		}

		$cameras = array();
		foreach ($output as $str)
		{
			if (preg_match("/(.*)\s+\((\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}):(\d+)\)/i", $str, $regs))
			{
				$element = array();
				$element["name"] = $regs[1];
				$element["ip"] = $regs[2];
				$element["port"] = $regs[3];
				array_push($cameras, $element);
			}
		}
		return $cameras;
	}

	/**
	 * get parameters of qtCam cameras on node
	 *
	 * @param string $type
	 * @throws Exception
	 * @throws InvalidArgumentException
	 * @return array
	 */
	public function getqtCamParametersList($type)
	{
		$command = "";
		$parameters = "";
		switch ($type)
		{
			case "camera":
				$command = APL_PATH . "/conf/etc/device/camera/qtCam/bin/list";
			break;
			case "audio":
				$command = APL_PATH . "/conf/etc/device/camera/qtCam/bin/list";
				$parameters = "-a";
			break;
			default:
				throw new InvalidArgumentException("invalid argument");
		}

		$output = array();
		if (is_file($command))
		{
			exec(escapeshellcmd("sudo $command $parameters"), $output, $return_var);
		}
		else
		{
			$error = sprintf(__('Error reading file: %s'), $command);
			throw new Exception($error);
		}

		$cameras = array();
		foreach ($output as $row)
		{
			$element = Template::parseList($row);
			array_push($cameras, $element);
		}

		return $cameras;
	}

	/**
	 * get path to qtCam camera picture
	 *
	 * @param  string $id
	 * @throws Exception
	 * @return string
	 */
	public function getqtCamCameraPicture($id)
	{
		$command = APL_PATH . "/conf/etc/device/camera/qtCam/bin/snapshot";

		$output = array();
		if (is_file($command))
		{
			exec(escapeshellcmd("sudo $command $id"), $output, $return_var);
		}
		else
		{
			$error = sprintf(__('Error reading file: %s'), $command);
			throw new Exception($error);
		}

		$path = basename($output[0]);

		return $path;
	}

	/**
	 * get version
	 *
	 * @return string
	 */
	public function getVersion()
	{
		return exec(escapeshellcmd(APL_PATH . "/vpatch/bin/vctl ver"));
	}

	/**
	 * get brand
	 *
	 * @static
	 * @return string
	 */
	public static function getBrand()
	{
		return exec(escapeshellcmd(APL_PATH . "/vpatch/bin/vctl brand"));
	}

	/**
	 * get lavel
	 * example: va-stratus-3.6.0-26248_20120706.el5
	 *
	 * @static
	 * @return string
	 */
	public static function getLabel()
	{
		return exec(escapeshellcmd(APL_PATH . "/vpatch/bin/vctl label"));
	}

	/**
	 * is node a master
	 * @static
	 * @return string
	 */
	public static function isMaster()
	{
		$stype = exec(escapeshellcmd(APL_PATH . "/vpatch/bin/vctl stype"));
		return $stype === "master";
	}

	/**
	 * get info
	 *
	 * @static
	 * @return array
	 */
	public static function getInfo()
	{
		$result = array();

		$output = array();
		exec(escapeshellcmd(APL_PATH . "/vpatch/bin/vctl info"), $output);
		foreach ($output as $row)
		{
			list($key, $value) = explode('=', $row);
			$result[$key] = $value;
		}

		return $result;
	}

	/**
	 * get storage manager status
	 *
	 * @static
	 * @return string|null
	 */
	public static function getStorageManagerStatus()
	{
		$status = null;

		$fileName = APL_VAR_PATH . "/sm/status";
		if (is_file($fileName))
		{
			$status = file_get_contents($fileName);
		}

		return $status;
	}

	/**
	 * @param int $obj
	 * @param int $startTime
	 * @param int $endTime
	 * @param string $granularity min|hour|day|mon
	 * @param int $streamNumber
	 * @param int $multiplier
	 * @throws InvalidArgumentException
	 * @return array
	 */
	public static function getStorageCoverage($obj, $startTime, $endTime, $granularity, $streamNumber = 1, $multiplier = 1)
	{
		$coverage = array();

		// ./storage_coverage -s 1360758402 -e 1360762002 -o 110 -g min -n 1 -m 1
		$command = sprintf(
			APL_PATH . "/mgears/bin/storage_coverage -o %s -s %s -e %s -g %s -n %s -m %s 2>&1",
			escapeshellarg($obj),
			escapeshellarg($startTime),
			escapeshellarg($endTime),
			escapeshellarg($granularity),
			escapeshellarg($streamNumber),
			escapeshellarg($multiplier)
		);
		$result = exec($command, $output, $return_var);

		if ($return_var != 0)
		{
			throw new InvalidArgumentException(implode("\n", $output));
		}

		foreach ($output as $row)
		{
			if (preg_match("/^(\d+)=(\d+)$/", $row, $matches))
			{
				$time = $matches[1];
				$percentage = $matches[2];
				$coverage[$time] = $percentage;
			}
		}

		return $coverage;
	}

	/**
	 * get system status
	 *
	 * @static
	 * @param  bool $cached use cached result or not
	 * @return array("status" => string, "description" => string, "reasons" => array())
	 */
	public static function getSystemStatus($cached = false)
	{
		$statusDescription = array(
			"" => "",
			"ONLINE" => "",
			"OFFLINE" => __("OFFLINE: The application is not running (OFFLINE state). Please contact your System Administrator."),
			"STARTING" => __("STARTING: System is Starting. Please wait until the system is fully functional."),
			"STOPPING" => __("STOPPING: System is Stopping and will not be functional till restart."),
			"BROKEN" => __("BROKEN: The application is not running (BROKEN state). Please urgently contact your System Administrator."),
			"CRITICAL" => __("CRITICAL: System in a CRITICAL state.  See details for more information."),
			"ALERT" => __("ALERT: System is functional but requires attention. See ALERT details for more information.")
		);

		$systemStatus = array();
		if ($cached && isset(self::$systemStatus))
		{
			$systemStatus = self::$systemStatus;
			$systemStatus["description"] = $statusDescription[$systemStatus["status"]];
		} else {
			$status = "";
			$reasons = array();

			$fileName = APL_VAR_PATH . "/sdi/system_status.json";
			if (is_file($fileName))
			{
				$jsonString = file_get_contents($fileName);
				$json = json_decode($jsonString, true);
				$status = $json["status"];
				$reasons = $json["reasons"];
			}

			$systemStatus = array(
				"status" => $status,
				"description" => $statusDescription[$status],
				"reasons" => $reasons
			);

			self::$systemStatus = $systemStatus;
		}

		return $systemStatus;
	}

	/**
	 * get status warning formatted message
	 *
	 * @static
	 * @return string
	 */
	public static function getWarningFormattedMessage()
	{
		$warningMessage = "";
		$systemStatusMessage = "";
		$systemStatus = Node::getSystemStatus();

		if ($systemStatus["status"] != "" && $systemStatus["status"] != "ONLINE")
		{
			$systemStatusReasons = "";
			foreach ($systemStatus["reasons"] as $reason)
			{
				if ($reason != "")
				{
					$systemStatusReasons .= "<li>$reason</li>";
				}
			}

			$systemStatusMessage = "{$systemStatus["description"]}<ul>$systemStatusReasons</ul>";
		}

		$message = "";
		if ($warningMessage != "")
		{
			$message .= "<li>$warningMessage</li>";
		}
		if ($systemStatusMessage != "")
		{
			$message .= "<li>$systemStatusMessage</li>";
		}
		if ($message != "")
		{
			$message = "<ul>$message</ul>";
		}

		return $message;
	}

	/**
	 * get IP by UNI
	 * @static
	 * @param  $uni
	 * @return string|null
	 */
	public static function getIpByUNI($uni)
	{
		$queryString = "SELECT val FROM _objs, _obj_attr WHERE _objs.obj = _obj_attr.obj AND otype = 'D' AND subtype = 'N' AND name = ? AND deleted = '0' AND attr = 'IP';";
		$list = DB::select($queryString, array($uni));

		$ip = null;

		if (isset($list[0]['val']))
			$ip = $list[0]['val'];

		return $ip;
	}

	/**
	 * get URI by UNI
	 * @static
	 * @param  $uni
	 * @return string|null
	 */
	public static function getUriByUNI($uni)
	{
		$queryString = "SELECT val FROM _objs, _obj_attr WHERE _objs.obj = _obj_attr.obj AND otype = 'D' AND subtype = 'N' AND name = ? AND deleted = '0' AND attr = 'URI';";
		$list = DB::select($queryString, array($uni));

		$uri = null;

		if (isset($list[0]['val']))
		{
			$uri = $list[0]['val'];
		} else {
			$uri = self::getIpByUNI($uni); // fallback if Uri is not set
		}

		return $uri;
	}

	/**
	 * get UNI
	 *
	 * @static
	 * @param $nodeid null - then return uni of master node
	 * @return string|null
	 */
	public static function getUNI($nodeid = null)
	{
		$nodeUNI = null;

		if (!isset($nodeid))
		{
			$nodeUNI = exec(escapeshellcmd(APL_PATH . "/vpatch/bin/vctl uni"));
		} else {
			$list = DB::select("
				SELECT name
					FROM _objs as node
					WHERE
					 	deleted = 0
						AND node.otype = 'D'
						AND node.subtype = 'N'
						AND node.obj = ?;",
				array($nodeid)
			);
			if (count($list) > 0)
			{
				$nodeUNI = $list[0]["name"];
			}
		}

		return $nodeUNI;
	}

	/**
	 * get list of nodes
	 *
	 * @param bool $withAttributes
	 * @param string|null $filterAttribute
	 * @param string|null $filterMask
	 * @return array
	 */
	public function getList($withAttributes = false, $filterAttribute = null, $filterMask = null)
	{
		if (isset($filterAttribute) && isset($filterMask))
		{
			$list = DB::select(
				"SELECT _objs.obj, _obj_attr.val as match
					FROM _objs, _obj_attr
					WHERE
						_objs.obj = _obj_attr.obj
						AND otype = 'D'
						AND subtype = 'N'
						AND _obj_attr.attr = ?
						AND _obj_attr.val ilike ?
						AND deleted = '0';",
				array($filterAttribute, "%$filterMask%")
			);
		} else {
			$list = DB::select(
				"SELECT _objs.obj
					FROM _objs
					WHERE
						otype = 'D'
						AND subtype = 'N'
						AND deleted = '0';"
			);
		}

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$row["attributes"] = $this->getAttributes($row["obj"]);
				// only for nodes, list should UNI (see docs)
				$row["attributes"]["UNI"] = $this->getUNI($row["obj"]);
			}
			unset($row);
		}

		return $list;
	}

	/**
	 * get node list with camera number on it
	 *
	 * @static
	 * @param  bool $withAttributes
	 * @return array
	 */
	public function getListWithNumberOfCameras($withAttributes = false)
	{
		$list = DB::select("SELECT * FROM getNodes();");

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$row["attributes"] = $this->getAttributes($row["obj"]);
				// only for nodes, list should UNI (see docs)
				$row["attributes"]["UNI"] = $this->getUNI($row["obj"]);
			}
			unset($row);
		}

		foreach ($list as &$row)
		{
			$avatarList = $this->getObjects($row["obj"], "V", "*", false);
			$row["number_of_avatars"] = count($avatarList);
		}
		unset($row);

		return $list;
	}

	/**
	 * get list of users
	 * @param  bool $withAttributes return with attributes
	 * @return array
	 */
	public function getUsers($withAttributes = false)
	{
		$list = DB::select("SELECT * FROM getUsers();");
		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$row["attributes"] = $this->getAttributes($row["obj"]);
			}
			unset($row);
		}
		return $list;
	}

	/**
	 * get list of roles
	 * @param  bool $withAttributes return with attributes
	 * @return array
	 */
	public function getRoles($withAttributes = false)
	{
		$list = DB::select("SELECT * FROM getRoles();");

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$row["attributes"] = $this->getAttributes($row["obj"]);
			}
			unset($row);
		}

		return $list;
	}

	/**
	 * get list of sets
	 * @param  $childObj
	 * @param  bool $withAttributes return with attributes
	 * @return array
	 */
	public function getSets($childObj = null, $withAttributes = false)
	{
		$list = array();
		if (isset($childObj))
		{
			$list = DB::select("SELECT * FROM getSets(?);", array($childObj));
		} else {
			$list = DB::select("SELECT * FROM getSets();");
		}

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$row["attributes"] = $this->getAttributes($row["obj"]);
			}
			unset($row);
		}

		return $list;
	}

	/**
	 * get list of objects from node
	 *
	 * @param int|null $obj if equal null then return objects from all nodes
	 * @param string|null $otype
	 * @param string|null $subtype
	 * @param bool $withAttributes
	 * @param bool $withBlocks not implemented
	 * @return array
	 */
	public function getObjects($obj = null, $otype = null, $subtype = null, $withAttributes = false, $withBlocks = false)
	{
		$list = DB::select(
            "SELECT
                obj, udid, name, description, location, otype, subtype, protected, '' as permission, '' as credentials
            FROM
                _objs
            WHERE
                deleted = 0
                AND (cast(? as char) IS NULL OR otype = ?)
                AND (cast(? as char) IS NULL OR subtype = ?)
				AND (cast(? as char) IS NULL OR node_id = ?);",
			array($otype, $otype, $subtype, $subtype, $obj, $obj)
		);

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$row["attributes"] = $this->getAttributes($row["obj"]);
			}
			unset($row);
		}

		return $list;
	}

	/**
	 * get list of all objects from system
	 * @param string|null $otype
	 * @param string|null $subtype
	 * @param bool $withAttributes
	 * @return array
	 */
	public function getAllObjects($otype = null, $subtype = null, $withAttributes = false)
	{
		$list = DB::select(
			"SELECT
				obj, udid, name, description, location, otype, subtype, protected, '' as permission, '' as credentials
			FROM
				_objs
			WHERE
				deleted = 0
				AND (cast(? as char) IS NULL OR otype = ?)
				AND (cast(? as char) IS NULL OR subtype = ?);",
			array($otype, $otype, $subtype, $subtype)
		);

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$row["attributes"] = $this->getAttributes($row["obj"]);
			}
			unset($row);
		}

		return $list;
	}

	/**
	 * @return string
	 */
	public function getTrustedHostsPath()
	{
		return APL_VAR_PATH . "/conf/trusted_hosts";
	}

	/**
	 * get list of trusted hosts
	 *
	 * @return array
	 */
	public function getTrustedHosts()
	{
		$trustedHostsPath = $this->getTrustedHostsPath();

		$hosts = array();
		$hostsFile = file($trustedHostsPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);

		if ($hostsFile !== FALSE)
		{
			// Validate entries
			foreach ($hostsFile as $row)
			{
				if ($row[0] == '#') continue;
				$hosts[] = $row;
			}
		}

		return $hosts;
	}

	/**
	 * @param string $host
	 */
	public function addToTrustedHosts($host)
	{
		$hosts = $this->getTrustedHosts();

		if (!in_array($host, $hosts))
		{
			$hosts[] = $host;
			$trustedHostsPath = $this->getTrustedHostsPath();
			file_put_contents($trustedHostsPath, $hosts);
		}
	}

	/**
	 * @param string $host
	 */
	public function removeFromTrustedHosts($host)
	{
		$hosts = $this->getTrustedHosts();

		$pos = array_search($host, $hosts);
		if ($pos !== FALSE)
		{
			unset($hosts[$pos]);

			$trustedHostsPath = $this->getTrustedHostsPath();
			file_put_contents($trustedHostsPath, $hosts);
		}
	}

	/**
	 * get list of interfaces on current node
	 *
	 * @static
	 * @return array
	 */
	public static function getInterfaces()
	{
		$interfacesFile = APL_VAR_PATH . "/conf/interfaces";

		if (($interfaces = file($interfacesFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES)) == FALSE)
		{
			$interfaces = array();
		}

		return $interfaces;
	}

	/**
	 * get host name of current node
	 *
	 * @static
	 * @return string
	 */
	public static function getHostName()
	{
		return exec(escapeshellcmd("/bin/hostname"));
	}

	/**
	 * get lis of license keys
	 *
	 * @static
	 * @return array
	 */
	public static function getLicenseKeys()
	{
		$GETKEY = "/opt/sarch/common/bin/helper";
		$KEYS = array(
			"sensor_skm" => "SENSORS",
			"relay" => "RELAYS",
			"monitor" => "VMX_MONITORS",
			"wall" => "VMX_MONITORS",
			"gateway" => "VAC_GATEWAYS"
		);

		$licenseKey = array();

		foreach($KEYS as $key => $val)
		{
			$count = trim(exec("$GETKEY $val"));
			if ($count == "" && $count == "0")
			{
				$licenseKey[$key] = false;
			}
		}

		return $licenseKey;
	}

	/**
	 * get list of parameters for AMQ client
	 *
	 * @static
	 * @return array
	 */
	public static function getAMQConf()
	{
		$parameters = array();

		// read parameters from file
		$rows = file(APL_PATH . "/smix/etc/skm_local.amq.conf", FILE_IGNORE_NEW_LINES|FILE_SKIP_EMPTY_LINES);
		foreach ($rows as $row)
		{
			list($parameter, $value) = preg_split("/=/", $row, 2);
			$parameters[$parameter] = $value;
		}

		return $parameters;
	}

	/**
	 * This function is used to get array with device configurationsm where on the
	 * first level will be device ID, on the second - all reported parameters.
	 * Third parameter is a filtering condition for DEVICETYPE. If it is '', then
	 * all devices will be queried and returned, otherwize - only of requested type.
	 * Last parameter ($query) tells what will be returned: 'conf' means configuration
	 * attributes, 'geometry' - camera geometry, 'runstate' - status on if device is
	 * functioning propertly or not; these parameters can be combined with comma.
	 *
	 * @param string $host
	 * @param int $port
	 * @param string $devicetype
	 * @param string $query
	 * @return array
	 */
	public static function confGetDevices($host = '', $port = null, $devicetype = '', $query = 'conf')
	{
		if (!$host)
		{
			$host = $_SERVER['SERVER_ADDR'];
		}
		if (!$port)
		{
			$port = ($_SERVER['SERVER_PORT'] != 80 && $_SERVER['SERVER_PORT'] != 443) ? $_SERVER['SERVER_PORT'] : 80;
		}

		$configXML = File_Get_Contents("http://" . $host . ":" . $port . "/api/cgi-bin/devinfo.cgi?return=" . $query);

		// ($_COOKIE['PHPSESSID'])?($sid="&sid=".$_COOKIE['PHPSESSID']):($sid='');
		// $configXML = File_Get_Contents("http://".$host.":".$port."/api/cgi-bin/devinfo.cgi?return=".$query."&SID=".$sid);
		$parser = xml_parser_create();
		xml_parser_set_option($parser, XML_OPTION_CASE_FOLDING, 0);
		xml_parser_set_option($parser, XML_OPTION_SKIP_WHITE, 1);
		xml_parse_into_struct($parser, $configXML, $values, $tags);
		xml_parser_free($parser);
		unset($tags);
		unset($configXML);

		$result = Array();
		$devid = 0;
		foreach ($values as $el)
		{
			if ($el['tag'] == 'DEVICE' && $el['type'] == 'open' && $el['level'] == 2 && isset($el['attributes']['ID']))
				if ($devicetype and strtoupper($devicetype) != $el['attributes']['DEVICETYPE'])
				{
					$devid = 0;
				} else
				{
					$devid = $el['attributes']['ID'];
				}
			elseif ($el['tag'] == 'DEVICE' && $el['type'] == 'close' && $el['level'] == 2)
			{
				$devid = 0;
			}
			elseif ($devid && $el['level'] == 3 && isset($el['attributes']))
			{
				$result[$devid][$el['tag']] = $el['attributes']['VALUE'];
			}
		}
		return $result;
	}

	/**
	 * @return array
	 */
	public static function getVAEList()
	{
		$command = APL_PATH . "/conf/bin/cartridge-ctl type=vae query=list";

		$output = array();
		exec(escapeshellcmd($command), $output, $return_var);

		$vaeList = array();
		foreach ($output as $row)
		{
			if (preg_match("/(.*)=(.*)/", $row, $regs))
			{
				$vaeList[$regs[1]] = $regs[2];
			}
		}

		return $vaeList;
	}

	/**
	 * @param string $name
	 * @param array $parameters
	 * @param array $properties
	 * @param array $actions
	 * @throws Exception
	 */
	public function submitAction($name, array $parameters, array $properties, array $actions = array())
	{
		$nameList = array("create", "announce", "ack", "clear", "update", "eada");

		if (!in_array($name, $nameList))
		{
			throw new InvalidArgumentException("Incorrect 'name' parameter");
		}

		$propertiesString = "";
		foreach ($properties as $propertieName => $propertieValue)
		{
			$propertiesString .= "$propertieName=$propertieValue,";
		}

		$args = array(
			// name
			"n" => $name,
			// eventsource
			"s" => isset($parameters["source"]) ? $parameters["source"] : null,
			// eventtype
			"e" => isset($parameters["eventtype"]) ? $parameters["eventtype"] : null,
			// objid
			"o" =>  isset($parameters["objid"]) ? $parameters["objid"] : null,
			// msg
			"m" =>  isset($parameters["msg"]) ? $parameters["msg"] : null,
			// note
			"N" => isset($parameters["note"]) ? $parameters["note"] : null,
			// utc_from
			"f" =>  isset($parameters["from"]) ? $parameters["from"] : null,
			// utc_to
			"t" =>  isset($parameters["to"]) ? $parameters["to"] : null,
			// utc_when
			"w" => null,
			// properties
			"p" => $propertiesString,
			// eventid
			"i" => isset($parameters["eventid"]) ? $parameters["eventid"] : null,
			// witnesses
			"W" => isset($parameters["witnesses"]) ? $parameters["witnesses"] : null,
			// priority
			"P" => isset($parameters["priority"]) ? $parameters["priority"] : null
		);

		$command = APL_PATH . "/elog/bin/submitAction";

		foreach ($args as $arg => $value)
		{
			if (!isset($value) || empty($value))
			{
				continue;
			}

			$command .= " -$arg " . escapeshellarg($value);
		}

		exec("$command 2>&1", $output, $return_var);
		if ($return_var != 0)
		{
			throw new Exception(implode($output, "\n"));
		}
	}

	/**
	 * @return bool
	 */
	public function isLegacySupport()
	{
		$LEGACY_SUPPORT = Identity::getAttribute("LEGACY_SUPPORT");
		return ($LEGACY_SUPPORT == "yes");
	}

	/**
	 * 1. parse pcap file
	 * 2. create new camera
	 *
	 * @param $pcapFile
	 * @param bool $publishProgress
	 * @throws Exception
	 * @internal param $ {string} $pcapFile $pcapFile
	 * @return array
	 */
	public function parsePCAPfile($pcapFile, $publishProgress = false)
	{
		/*
		>Error: <message>\n
		>Warning: <message>\n
		>Progress: 11%\n
		>NewStream: 192.168.160.102:554=>192.168.160.11:53405 TCP JPEG 1393602229\n
		<Ok: 125\n | Error: <message>\n
		*/

		$pcapReader = "sudo -u apl APL=/opt/sarch APL_CONF=/opt/sarch/var/conf APL_VAR=/var/sarch PERL5LIB=/opt/sarch/common/lib:/opt/sarch/imp/lib/perl5/site_perl /opt/sarch/mgears/bin/pcap_reader";

		$log = array();
		$cameras = array();

		$descriptorspec = array(
		   0 => array("pipe", "r"),
		   1 => array("pipe", "w")
		);

		$command = "$pcapReader $pcapFile 2>&1";
		$process = proc_open($command, $descriptorspec, $pipes);
		if (is_resource($process))
		{
			while (!feof($pipes[1]))
			{
				$row = fgets($pipes[1]);
				if ($row == "")
				{
					continue;
				}

				list($key, $value) = explode(': ', $row);
				$value = trim($value);
				$log[] = array($key, $value);

				if ($key == "Warning"
					|| $key == "Error")
				{
					// print "$value\n";
				}

				if ($key == "Progress" && $publishProgress)
				{
					$AMQConf = Node::getAMQConf();
					$con = new Stomp($AMQConf["stomp.url"]);
					$con->connect();
					$message = array(
						"error" => "",
						"progress" => $value
					);
					$destination = "/topic/uploadProgress_" . session_id();
					$result = $con->send($destination, json_encode($message), array('persistent'=>'true', 'slow_consumer_policy'=>'queue'));
				}

				// NewStream: 192.168.160.102:554=>192.168.160.11:53405 TCP JPEG 1393602229\n
				if ($key == "NewStream")
				{
					$dividerPosition = strrpos($value, " ");
					$name = substr($value, 0, $dividerPosition);
					$timestamp = substr($value, $dividerPosition + 1);

					// create camera
					$camera = new Camera();
					$cameraId = $camera->create(array(
						"CAMERAMODEL" => "PCAP",
						"NAME" => $name,
						"ARCHIVE_START_TIME" => $timestamp
					));

					$cameras[$cameraId] = array(
						"name" => $name,
						"timestamp" => $timestamp
					);

					$time = time();
					$confDir = "/vasm/store/va-3.3.0/$cameraId";
					while (!is_dir($confDir))
					{
						if (time() - $time > 60)
						{
							throw new Exception("Something wrong");
						}
						sleep(1);
					}

					fwrite($pipes[0], "Ok: $cameraId\n");
				}
			}

			fclose($pipes[0]);
			fclose($pipes[1]);

			proc_close($process);
		}

		return array(
			"timestamp" => $cameras,
			"log" => $log
		);
	}

	/**
	 * @param array $source
	 * @param array $destination
	 *
	 * @return mixed
	 * @throws Exception
	 */
	public function findHomography(array $source, array $destination)
	{
		$sourceJSON = json_encode($source);
		$destinationJSON = json_encode($destination);

		$output = array();
		exec(escapeshellcmd(APL_PATH . "/trgt/bin/find_homography $sourceJSON $destinationJSON"), $output, $return_var);
		if ($return_var != 0)
		{
			$error = sprintf(__("Return value: %s"), $return_var);
			throw new Exception($error);
		}

		$homography = json_decode($output[0]);
		return $homography;
	}

	/**
	 * get AdminUI config
	 *
	 * @static
	 * @return array
	 */
	public static function getAdminUIConfig()
	{
		$config = array();

		$fileName = APL_PATH . "/sdi/etc/AdminUI.cfg";
		if (is_file($fileName))
		{
			$tmpConfig = file($fileName, FILE_IGNORE_NEW_LINES|FILE_SKIP_EMPTY_LINES);

			for($i = 0; $i < count($tmpConfig); $i++){
				$tmpLine = explode("=", $tmpConfig[$i]);

				// allow format name=value, skip comments - #
				if(count($tmpLine) == 2 && $tmpLine[0][0] != "#"){
					$config[trim($tmpLine[0])] = trim($tmpLine[1]);
				}
			}
		}

		return $config;
	}

	/**
	 * @param $obj
	 * @param bool|true $onlyMark
	 *
	 * @return bool
	 * @throws Exception
	 */
	public function delete($obj, $onlyMark = true)
	{
		$uni = $this->getUNI($obj);

		$isDead = $this->isDead($uni);
		$isMaster = $this->getUNI() == $uni;

		if ($isMaster)
		{
			throw new Exception(__("Cannot remove master node"));
		}

		if (!$isDead)
		{
			throw new Exception(__("Cannot remove online node"));
		}

		$exit_code = 0;

		$env_vars = "APL=" . getenv('APL') . " APL_CONF=" . getenv('APL_CONF') . " PERL5LIB=" . getenv('PERL5LIB') . " VERSIONER_PERL_VERSION=" . getenv("VERSIONER_PERL_VERSION");
		$out = system(escapeshellcmd($env_vars . " sudo -u apl /opt/sarch/conf/bin/master_rmnode $uni ") . "1>/dev/null 2>&1", $exit_code);

		if ($exit_code != 0)
		{
			throw new Exception(__("Process exited with code $exit_code"));
		}
		return true;
	}

	/**
	 * @param string $uni
	 *
	 * @return bool
	 */
	public function isDead($uni)
	{
		// Node is considered to be dead if UNI file isn't updated for a long time
		$DEADAFTER = 60;
		clearstatcache();
		$stat = stat($_SERVER['APL']."/var/conf/master/nodes/$uni");
		$mtime = $stat[9];
		$isDead = time() - $mtime > $DEADAFTER ? true : false;

		return $isDead;
	}

	/**
	 * @param int $obj node Object ID
	 * @param array $attributes new attributes
	 *
	 * @throws DBException
	 */
	public function setAttributes($obj, array $attributes)
	{
		if (isset($attributes["NODENAME"]))
		{
			$name = $attributes["NODENAME"];
			$uni = $this->getUNI($obj);
			DB::query("UPDATE sm_nodes SET name = ? WHERE nodeid = ?;", array($name, $uni));
		}

		parent::setAttributes($obj, $attributes);
	}

	/**
	 * calculate hash of "License Agreement"
	 *
	 * @return string
	 */
	public function getLicenseHash()
	{
		return md5_file($_SERVER['APL'] . "/base/www/common/sla.htm");
	}

	/**
	 * @return array
	 */
	public static function getLowCredentialAttributes()
	{
		return array(
			"UDID",
			"DEVICETYPE",
			"DEVID",
			"OBJID",
			"CAMERAMODEL",
			"STATUS",
			"NAME",
			"LOCATION",
			"AUDIO",
			"ASSOCIATE",
			"NOTE",
			"METADATATAG_INFO",
			"EVENT_POLICY",
			"STORAGE_POLICY",
			"CLOUD_SEPARATOR_TS",
			"CAM_GEO_CALIBRATION",
			"CAM_GEO_LAT",
			"CAM_GEO_LONG",
			"CAM_GEO_ALT",
			"CAM_GEO_RANGE",
			"CAM_GEO_TILT",
			"CAM_GEO_AZIMUTH",
			"CAM_GEO_HFOV",
			"GEO_CALIBRATOR_CONFIG",
			"CAMERA",
			"TIME_ZONE",
			"AVATARID",
			"AV_DELIVERY",
			"FRAMERATE",
			"PIXEL_ASPECT_RATIO",
			"POSITIONCTL",
			"PTZ_TOURS",
			"PTZ_FEEDBACK",
			"VAE_ACTIVE",
			"VAE_VISUALIZATION",
			"ANALYTICS",
			"VAE_MOTION_ACTIVE",
			"VAE_MOTION_CELLS_H",
			"VAE_MOTION_CELLS_V",
			"VAE_MOTION_ZONEMAP",
			"TYPE",
			"SUBTYPE",
			"NODE_ID",
			"AUDIO_TWO_WAY",
			"TCP_IP",
			"STATE",
			"WALLW",
			"WALLH",
			"XML_CONF",
			"IP",
			"HISTORY",
			"UNI",
			"OWNER",
			"SHARED",
			"DATA",
			"ROLEID",
			"ROLENAME",
			"USERID",
			"PROTECTED",
			"WALL",
			"NUMBER",
			"DESCRIPTION",
			"PTZ_PRIORITY",
			"WARN_VIDEO_PLAYERS",
			"MAX_VIDEO_PLAYERS",
			"EXT_ID",
			"DESC",
			"ENABLED",
			"PRIORITY",
			"LOC_DATA",
			"LOC_TYPE",
			"LOC_ROLEID",
			"VIEW_DATA",
			"NLS_LONG_DATE_FORMAT",
			"SYSTEM_CLASSIFICATION",
			"EVENTLOG_MAX_EVENT_LENGTH",
			"PTZ_EXTENDED_PRESETS",
			"PTZ_LOCK_TIMEOUT",
			"URL_HOME_PAGE",
			"MAX_GUI_FRAMERATE",
			"MEDIA_EXPORT_FORMAT",
			"PLAYER_BROKEN_STREAM_INDICATION",
			"PLAYER_BROKEN_STREAM_INDICATION_DELAY",
			"AVATAR_LOCAL_STREAMING"
		);
	}

	public static function isLowCredentialAttribute($attribute)
	{
		return strpos($attribute, "STAT_") === 0 || in_array($attribute, self::getLowCredentialAttributes());
	}

	public function getActivatedParameters()
	{
		$lines = file(APL_VAR_PATH . "/vctl/activated", FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);

		$parameters = array();
		foreach ($lines as $line)
		{
			list($key, $value) = preg_split('/=/', trim($line));
			$parameters[$key] = $value;
		}

		return $parameters;
	}

	public function isOpenVPNActivated()
	{
		$parameters = $this->getActivatedParameters();
		return strpos($parameters["ACTIVATED"], "v") !== false;
	}
}
