<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id:$
 * ------------------------------------------------------------------------------
 * LDAP
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2014
 * ------------------------------------------------------------------------------
 */

class LDAP
{
	private $_link;

	private $LDAP_USER;
	private $LDAP_GROUP_DN;
	private $LDAP_GROUP_FILTER;
	private $LDAP_USER_DN;
	private $LDAP_GROUP_USER_FILTER_TEMPLATE;

	private $userNamePlaceholder = "(0)";
	private $userDNPlaceholder = "(1)";

	public function __destruct()
	{
		$this->close();
	}

	/**
	 * @param string $userName
	 * @param string $password
	 * @throws Exception
	 */
	public function auth($userName, $password)
	{
		defined("LDAP_OPT_DIAGNOSTIC_MESSAGE") || define("LDAP_OPT_DIAGNOSTIC_MESSAGE", 0x0032);

		$identityAttributes = Identity::getAttributes();

		$LDAP_DOMAIN_CONTROLLER = $identityAttributes["LDAP_DOMAIN_CONTROLLER"]; //"ldap.dev.videonext.net";
		$LDAP_DOMAIN_CONTROLLER_PORT = $identityAttributes["LDAP_DOMAIN_CONTROLLER_PORT"]; //389
		$LDAP_USER_TEMPLATE = $identityAttributes["LDAP_USER_TEMPLATE"]; //"uid=(0),ou=People,dc=videonext,dc=net";
		$this->LDAP_USER = $userName;
		$LDAP_PASSWORD = $password;
		$this->LDAP_GROUP_DN = $identityAttributes["LDAP_GROUP_DN"]; //"ou=Group,dc=videonext,dc=net";
		$this->LDAP_GROUP_FILTER = $identityAttributes["LDAP_GROUP_FILTER"];

		if (strpos($LDAP_USER_TEMPLATE, $this->userNamePlaceholder) !== FALSE)
		{
			$this->LDAP_USER_DN = substr($LDAP_USER_TEMPLATE, strpos($LDAP_USER_TEMPLATE, $this->userNamePlaceholder . ",") + strlen($this->userNamePlaceholder . ","));
		}
		$rdn = str_replace($this->userNamePlaceholder, $this->LDAP_USER, $LDAP_USER_TEMPLATE);

		$this->LDAP_GROUP_USER_FILTER_TEMPLATE = $identityAttributes["LDAP_GROUP_USER_FILTER_TEMPLATE"];

		if (($this->_link = ldap_connect($LDAP_DOMAIN_CONTROLLER, $LDAP_DOMAIN_CONTROLLER_PORT)) === FALSE)
		{
			throw new Exception("Could not connect to LDAP");
		}

		if (!ldap_set_option($this->_link, LDAP_OPT_PROTOCOL_VERSION, 3))
		{
			throw new Exception('Unable to set LDAP protocol version');
		}
		ldap_set_option($this->_link, LDAP_OPT_REFERRALS, 0);

		$isBind = @ldap_bind($this->_link, $rdn, $LDAP_PASSWORD);
		if (!$isBind)
		{
			if (ldap_get_option($this->_link, LDAP_OPT_DIAGNOSTIC_MESSAGE, $extended_error))
			{
				throw new Exception("Error Binding to LDAP: $extended_error");
			} else {
				throw new Exception("Error Binding to LDAP: No additional information is available.");
			}
		}
	}

	public function close()
	{
		ldap_close($this->_link);
	}

	/**
	 * @param string $host
	 * @param int $port
	 * @param int $timeout
	 * @return bool
	 */
	public function ping($host, $port = 389, $timeout = 1)
	{
		$op = fsockopen($host, $port, $errno, $errstr, $timeout);
		if (!$op)
		{
			return false; //DC is N/A
		} else {
			fclose($op);
			return true; //DC is up & running, we can safely connect with ldap_connect
		}
	}

	/**
	 * @param string|null $userName
	 *
	 * @return array
	 * @throws Exception
	 */
	public function getGroups($userName = null)
	{
		$filter = $this->LDAP_GROUP_FILTER;
		if (isset($userName))
		{
			$filter = $this->LDAP_GROUP_USER_FILTER_TEMPLATE;
			if (strpos($filter, $this->userNamePlaceholder) !== FALSE)
			{
				$filter = str_replace($this->userNamePlaceholder, $userName, $filter);
			}
			if (strpos($filter, $this->userDNPlaceholder) !== FALSE)
			{
				$userDN = $this->getUserInfo($userName)["dn"];
				$filter = str_replace($this->userDNPlaceholder, $userDN, $filter);
			}
		}

		$results = @ldap_search($this->_link, $this->LDAP_GROUP_DN, $filter);
		if ($results === FALSE)
		{
			throw new Exception(ldap_error($this->_link));
		}
		$entries = @ldap_get_entries($this->_link, $results);
		if ($entries === FALSE)
		{
			throw new Exception(ldap_error($this->_link));
		}

		$groupList = array();

		if (!empty($entries) && $entries["count"] != 0)
		{
			foreach ($entries as $entryKey => $entry)
			{
				if ($entryKey === "count")
				{
					continue;
				}

				$groupList[] = array(
					"cn" => $entry["cn"][0],
					"dn" => $entry["dn"]
				);
			}
		}

		return $groupList;
	}

	/**
	 * @param string|null $userName
	 *
	 * @return array
	 * @throws Exception
	 */
	public function getUserList($userName = null)
	{
		$userName = isset($userName) ? $userName : "*";
		// search for users
		$filter = "(&(objectClass=user)(cn=$userName))";
		$results = @ldap_search($this->_link, $this->LDAP_USER_DN, $filter);
		if ($results === FALSE)
		{
			throw new Exception(ldap_error($this->_link));
		}
		$entries = @ldap_get_entries($this->_link, $results);
		if ($entries === FALSE)
		{
			throw new Exception(ldap_error($this->_link));
		}

		$userList = array();

		if (!empty($entries) && $entries["count"] != 0)
		{
			foreach ($entries as $entryKey => $entry)
			{
				if ($entryKey === "count")
				{
					continue;
				}

				$userList[] = array(
					"cn" => $entry["cn"][0],
					"dn" => $entry["dn"]
				);
			}
		}

		return $userList;
	}

	/**
	 * @param string $userName
	 *
	 * @return array|null
	 */
	public function getUserInfo($userName)
	{
		$userInfo = $this->getUserList($userName);
		return count($userInfo) > 0 ? $userInfo[0] : null;
	}

	/**
	 * @param string|null $userName
	 *
	 * @return array
	 * @throws Exception
	 */
	public function getGroupsWithoutFilter($userName = null)
	{
		// search for users
		// $filter = "(&(objectClass=group)(member=CN=user,OU=stuff,DC=place)";
		// $filter = "(member:1.2.840.113556.1.4.1941:=cn=user,CN=Users,DC=win2008,DC=dev,DC=videonext,DC=net)";
		$filter = "(&(objectClass=group)(member=*))";
		$results = @ldap_search($this->_link, $this->LDAP_GROUP_DN, $filter);
		if ($results === FALSE)
		{
			throw new Exception(ldap_error($this->_link));
		}
		$entries = @ldap_get_entries($this->_link, $results);
		if ($entries === FALSE)
		{
			throw new Exception(ldap_error($this->_link));
		}

		$groupList = array();

		if (!empty($entries) && $entries["count"] != 0)
		{
			foreach ($entries as $entryKey => $entry)
			{
				if ($entryKey === "count")
				{
					continue;
				}

				$group = array(
					"cn" => $entry["cn"][0],
					"dn" => $entry["dn"],
					"member" => array()
				);

				$isFound = false;

				foreach ($entry["member"] as $memberKey => $member)
				{
					if ($memberKey === "count")
					{
						continue;
					}

					if (!isset($userName)
					    || stripos($member, "CN=$userName") === 0)
					{
						$isFound = true;
					}

					$group["member"][] = $member;
				}

				if (isset($userName) && !$isFound)
				{
					continue;
				}

				$groupList[] = $group;
			}
		}

		return $groupList;
	}
}
