<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id:$
 * ------------------------------------------------------------------------------
 * This class for creating JS lib for PHP API
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions LLC 2013
 * ------------------------------------------------------------------------------
 */

require_once(APL_PATH . "/api/html/test/Markdown.php");

class GenAPI
{

	private $docs = array();

	/**
	 * @param array  $classNameList the class name list
	 * @throws InvalidArgumentException
	 * @throws BadMethodCallException
	 * @return array|mixed
	 */
	public function parse(array $classNameList)
	{
		$response = array();

		try
		{
			foreach ($classNameList as $className)
			{
				/**
				 * @var ReflectionClass
				 */
				$reflectionClass = new ReflectionClass($className);
				$methodLIst = $reflectionClass->getMethods();
				foreach ($methodLIst as $index => $method)
				{
					$methodName = $method->getName();
					$response[$methodName] = array();

					//preg_match_all('/@([a-z0-9_-]+)\s+(\$[a-z0-9_-]+)\s+([^\n]+)/is', $method->getDocComment(), $arr);
					$PHPDoc = $method->getDocComment();
					$parameter2type = $this->parameter2type($PHPDoc);

					$response[$methodName]["description"] = $this->parseComments($PHPDoc);

					$parameterList = $method->getParameters();
					foreach ($parameterList as $index => $parameter)
					{
						$parameterName = $parameter->getName();
						$defaultValue = null;
						$type = null;
						$description = null;
						if (isset($parameter2type[$parameterName]))
						{
							$type = $parameter2type[$parameterName]["type"];
							$description = $parameter2type[$parameterName]["description"];
						}
						if ($parameter->isOptional())
						{
							$defaultValue = $parameter->getDefaultValue();
							if ($defaultValue === null) $defaultValue = "null";
							if (is_bool($defaultValue)) $defaultValue = $defaultValue ? "true" : "false";
						}

						$response[$methodName]["parameterList"][$parameterName] = array(
							"name" => $parameterName,
							"type" => $type,
							"defaultValue" => $defaultValue,
							"description" => $description
						);
					}
				}
			}
		}
		catch (Exception $e)
		{
			print($e->getMessage());
		}

		return $response;
	}

	public function parameter2type($PHPDoc)
	{
		$parameter2type = array();
		$PHPDocLines = preg_split('/\n/is', $PHPDoc);
		foreach ($PHPDocLines as $index => $line)
		{
			$result = preg_match('/@[a-z0-9_-]+\s+([a-z0-9_-|]+)\s+\$([^\n\s]+)\s+(.*)/i', $line, $matches);
			if ($result != false && $result > 0)
			{
				$parameter2type[$matches[2]] = array(
					"type" => $matches[1],
					"description" => $matches[3]
				);
			}
		}

		return $parameter2type;
	}

	private function parseComments($PHPdoc)
	{
		//Get the comment
		if(preg_match('#^/\*\*(.*)\*/#s', $PHPdoc, $comment) === false)
			die("Error");

		if (!isset($comment[1]))
			return;

		$comment = trim($comment[1]);

		//Get all the lines and strip the * from the first character
		if(preg_match_all('#^\s*\*(.*)#m', $comment, $lines) === false)
			die('Error');

		$params = array(
			"description" => ""
		);
		$isDescription = true;
		$param = "";
		foreach ($lines[1] as $line)
		{
			$line = ltrim($line);
			$matches = array();
			if (preg_match("/^@(\w+)(\s+(.*))?$/", $line, $matches))
			{
				$isDescription = false;
				$param = $matches[1];
				$value = "";
				if (isset($matches[3]))
				{
					$value = $matches[3];
				}
				$params[$param] = $value;
			} else
			if ($isDescription)
			{
				$params["description"] .= "$line\n";
			} else {
				$params[$param] .= "$line  \n";
			}
		}

		$result = null;

		if (isset($params["docBlock"]))
		{
			$result = $params["docBlock"];
		}

		return $result;
	}

	public function createDoc(array $classNameList)
	{
		$docs = array();

		$methodList = $this->parse($classNameList);
		foreach ($methodList as $methodName => $methodInformation)
		{
			if (!isset($methodInformation["description"]))
			{
				continue;
			}

			$description = $methodInformation["description"];
			// find paragraph
			$lines = explode("\n", $description);
			$paragraph = "";
			foreach ($lines as $number => $line)
			{
				if (preg_match("/^#([^#].*)$/", $line, $matches))
				{
					$paragraph = $matches[1];
					unset($lines[$number]);
					break;
				}
			}

			$parameters = array();
			if (isset($methodInformation["parameterList"]))
			{
				$php2jsTypes = array(
					"int" => "Integer",
					"string" => "String",
					"array" => "Array JSON",
					"bool" => "Boolean"
				);

				array_push($lines, "####Parameters:\n");
				$parameterList = $methodInformation["parameterList"];
				foreach ($parameterList as $parameterName => $parameterInformation)
				{
					$parameters[$parameterName] = $parameterInformation;
					$required = isset($parameterInformation["defaultValue"]) ? "Optional" : "Required";
					$type = isset($parameterInformation["type"]) ? (isset($php2jsTypes[$parameterInformation["type"]]) ? $php2jsTypes[$parameterInformation["type"]] : $parameterInformation["type"]) : "";
					$description = isset($parameterInformation["description"]) ? $parameterInformation["description"] : "";
					array_push($lines, ">**$parameterName** $required, $type, $description\n");
				}
			}

			$result = implode("\n", $lines);
			$html = \Michelf\Markdown::defaultTransform($result);

			if (!isset($docs[$paragraph]))
			{
				$docs[$paragraph] = array();
			}

			array_push($docs[$paragraph], $html);
		}

		$result = "";
		foreach ($docs as $paragraph => $content)
		{
			$result .=  "<h1>" . $paragraph . "</h1>";
			foreach ($content as $doc)
			{
				$result .= $doc;
			}
		}

		return $result;
	}

	public function getDocs()
	{
		return $this->docs;
	}

	public function convertToJS(array $classNameListGET = array(), array $classNameListPOST = array())
	{
		$result = <<<EOT
/**
 * AUTO-GENERATED FILE - DO NOT EDIT!!
 */
(function(window){
	"use strict";

	window.API = API;

	/**
	 * @param {Boolean} [async] use only if you CANNOT use async
	 * @constructor
	 */
	function API(async)
	{
		this._async = typeof async != "undefined" ? !!async : true;
	}

	/**
	 * @param {String} funcName
	 * @param {String} funcMethod
	 * @param {Object} parameters
	 * @param {Array} requiredParameters
	 * @returns {Deferred}
	 */
	API.prototype._send = function(funcName, funcMethod, parameters, requiredParameters)
	{
		var deferred = $.Deferred();

		var parameter;
		for (var i = 0; i < requiredParameters.length; i++)
		{
			parameter = requiredParameters[i];
			if (typeof parameters[parameter] == "undefined")
			{
				deferred.reject(400, "parameter '" + parameter + "' of function '" + funcName + "' should be specified", parameters);
				return deferred.promise();
			}
		}

		for (parameter in parameters)
		{
			if (parameters[parameter] == null)
			{
				deferred.reject(400, "parameter '" + parameter + "' of function '" + funcName + "' could not be null", parameters);
				return deferred.promise();
			}
		}

		var url = "/api/call/" + funcName;

		if (!this._async)
		{
			var e = new Error();
			if (!e.stack)
			{
				try { throw e; }
				catch (e) {}
			}
			var where = e.stack ? e.stack.toString().split(/\\r\\n|\\n/)[2] : "";
			window.console && console.warn("do not use sync http call: " + url + "\\n" + where);
		}

		$.ajax({
			url: url,
			type: funcMethod,
			dataType: "json",
			cache: false,
			data: parameters,
			async: this._async,
			error: function(jqXHR){
				var code = jqXHR.status;
				// use jqXHR.responseJSON.error for Safari
				var message = (jqXHR.responseJSON && jqXHR.responseJSON.error) ? jqXHR.responseJSON.error : jqXHR.statusText;
				deferred.reject(code, message, parameters);
			},
			success: function(result){
				if (result.code == 200)
				{
					deferred.resolve(result, parameters);
				} else {
					deferred.reject(result.code, result.error, parameters);
				}
			}
		});

		return deferred.promise();
	};


EOT;

		$php2jsType = array(
			"int" => "Number",
			"string" => "String",
			"array" => "Array",
			"null" => "Null",
			"bool" => "Boolean",
		);

		$methodsGET = $this->parse($classNameListGET);
		$methodsPOST = $this->parse($classNameListPOST);

		$methods = array_merge($methodsGET, $methodsPOST);
		foreach ($methods as $method => $parameters)
		{
			/*
			$result .= "\t/**\n";
			foreach ($parameters as $parameter)
			{
				$name = $parameter["name"];
				$type = isset($parameter["type"]) ? "{" . $php2jsType[$parameter["type"]] . "}" : "";
				$result .= "\t *@param $type $name\n";
			}*/
			//$result .= "\t */\n";

			/**
			 * @type {{pan: null|Number, tilt: null|Number, hfov: null|Number, step: Number}}
			 * @private
			 */
			/*$result .= "\t/**\n";
			$result .= "\t *@type \n";
			foreach ($parameters as $parameter)
			{
				$name = $parameter["name"];
				$type = isset($parameter["type"]) ? "{" . $php2jsType[$parameter["type"]] . "}" : "";
				$result .= "\t *@param $type $name\n";
			}*/
			//$result .= "\t */\n";


			$requiredParametersList = array();
			if (isset($parameters["parameterList"]))
			{
				foreach ($parameters["parameterList"] as $parameterName => $parameterInformation)
				{
					//$type = isset($parameterInformation["type"]) ? "{" . $php2jsType[$parameterInformation["type"]] . "}" : null;
					$defaultValue = isset($parameterInformation["defaultValue"]) ? $parameterInformation["defaultValue"] : null;

					if (!isset($defaultValue))
					{
						array_push($requiredParametersList, $parameterName);
					}
				}

				if (count($requiredParametersList) > 0)
				{
					$result .= "\t/**\n";
					$result .= "\t * @param {Object} parameters\n";
					$result .= "\t * @returns {Deferred}\n";
					$result .= "\t */\n";
					$result .= "\tAPI.prototype.$method = function(parameters){\n";
				} else {
					$result .= "\t/**\n";
					$result .= "\t * @param {Object} [parameters]\n";
					$result .= "\t * @returns {Deferred}\n";
					$result .= "\t */\n";
					$result .= "\tAPI.prototype.$method = function(parameters){\n";
				}
			} else {
				$result .= "\t/**\n";
				$result .= "\t * @returns {Deferred}\n";
				$result .= "\t */\n";
				$result .= "\tAPI.prototype.$method = function(){\n";
				$result .= "\t\tvar parameters = {};\n";
			}

			$funcMethod = in_array($method, array_keys($methodsGET)) ? "GET" : "POST";

			$result .= "\t\tvar funcName = \"$method\";\n";
			$result .= "\t\tvar funcMethod = \"$funcMethod\";\n";
			$result .= "\t\tvar requiredParameters = [";
			$length = count($requiredParametersList);
			for ($i = 0; $i < $length; $i++)
			{
				$name = $requiredParametersList[$i];
				$result .= "\n\t\t\t\"$name\"";
				if ($i != $length - 1)
				{
					$result .= ",";
				}
			}
			$result .= "\n\t\t];\n";

			$result .= "\t\treturn this._send(funcName, funcMethod, parameters, requiredParameters);\n";

			$result .= "\t};\n\n";
		}

		$result .= "})(window);\n";

		return $result;
	}
}
