<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: DB.php 28266 2013-04-01 21:02:12Z astarostin $
 *
 * Class for working with DB
 *
 * Select rows from DB:
 * <code>
 * $list = DB::select("SELECT obj, name FROM _objs WHERE name = ?;", array($name));
 *
 * $list = DB::select("SELECT * FROM _objs WHERE name = ?;", array($name), "name");
 * </code>
 *
 * Do any query to DB except SELECT:
 * <code>
 * $result = DB::query("UPDATE _objs SET name = ?;", array($name));
 * </code>
 *
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions LLC 2006
 */

class DB
{
	private static $connection = null;
	private static $connectionString = null;

	private static $configuration = array(
		"master" => array(
			"path" => "/db/etc/skm_master.db.conf",
			"parameters" => null
		),
		"local" => array(
			"path" => "/db/etc/skm_local.db.conf",
			"parameters" => null
		)
	);
	private static $currentConfiguration = "master";

	private static $isDebug = SESSION_DEBUG;

	private static $numberOfQueries = 0;
	private static $timeForQueries = 0;

	private static $lastQuery = "";
	private static $lastQueryParams = array();

	private static $queryList = array(); // array(array("query" => "", "queryParams" => ""))

	/**
	 * connect to DB
	 *
	 * @static
	 * @throws DBException
	 * @return void
	 */
	private static function connect()
	{
		if (!isset(self::$connection))
		{
			if ((self::$connection = pg_pconnect(self::getConnectionString())) === FALSE)
			{
				self::$connection = null;
				throw new DBException(pg_last_error());
			}
		}
	}

	/**
	 * expand placeholders
	 *
	 * SELECT obj, name FROM _objs WHERE obj = ? AND name = ?;
	 * will be expanded to
	 * SELECT obj, name FROM _objs WHERE obj = $1 AND name = $2;
	 *
	 * @static
	 * @param string $query
	 * @return mixed
	 */
	private static function expandPlaceholders($query)
	{
		$placeholderNumber = 0;

		$query = preg_replace_callback(
			"/\?/",
			function($match) use (&$placeholderNumber)
			{
				return "\$" . ++$placeholderNumber;
			},
			$query
		);

		return $query;
	}

	/**
	 * select rows from DB
	 *
	 * Example:
	 * <code>
	 * $list = DB::select("SELECT obj, name FROM _objs WHERE name = ?;", array($name));
	 * // $list is like array = [["obj"=>"1", "name"=>"name1"],["obj"=>"1", "name"=>"name2"]]
	 *
	 * $list = DB::select("SELECT obj, name FROM _objs WHERE name = ?;", array($name), "name");
	 * // $list is like array = ["name" => ["obj"=>"1", "name"=>"name"]]
	 * </code>
	 *
	 * @static
	 * @param  string $query
	 * @param  array $vars
	 * @param  string $key
	 * @return array
	 */
	public static function select($query = "", array $vars = array(), $key = null)
	{
		$list = array();

		if ($query !== "")
		{
			$time = microtime(true);

			$result = self::query($query, $vars);

			if (pg_num_fields($result))
			{
				if (isset($key))
				{
					while ($row = pg_fetch_array($result, NULL, PGSQL_ASSOC))
					{
						$list[$row[$key]] = $row;
					}
				} else {
					while ($row = pg_fetch_array($result, NULL, PGSQL_ASSOC))
					{
						$list[] = $row;
					}
				}
			}

			self::$timeForQueries += microtime(true) - $time;
		}

		return $list;
	}

	/**
	 * do any query to DB except SELECT
	 *
	 * Example:
	 * <code>
	 * $result = DB::query("UPDATE _objs SET name = ?;", array($name));
	 * </code>
	 *
	 * @static
	 * @throws DBException
	 * @param  string $query
	 * @param  array $vars
	 * @return int|resource number of affected rows in case of non-select query, or resource in case of select query
	 */
	public static function query($query = "", array $vars = array())
	{
		$result = 0;

		if ($query !== "")
		{
			self::$numberOfQueries++;

			self::connect();

			self::log($query, $vars);

			self::$lastQuery = $query;
			self::$lastQueryParams = $vars;

			if (($result = pg_query_params(self::$connection, self::expandPlaceholders($query), $vars)) === FALSE)
			{
				throw new DBException(pg_last_error(self::$connection) . "\n" . self::expandPlaceholders($query) . "\n" . print_r($vars, true));
			}

			if (!pg_num_fields($result))
			{
				// Non-SELECT queries return number of affected rows, SELECT - resource.
				return pg_affected_rows($result);
			}
		}

		return $result;
	}

	/**
	 * @deprecated 2.6.3
	 * @static
	 * @throws DBException
	 * @param string $query
	 * @return resource|string
	 */
	public static function doQuery($query = "")
	{
		trigger_error('Function ' . __METHOD__ . ' is deprecated');

		$result = "";

		if ($query !== "")
		{
			self::connect();

			self::log($query);

			self::$lastQuery = $query;
			self::$lastQueryParams = "";

			if (($result = pg_query(self::$connection, $query)) === FALSE)
			{
				throw new DBException(pg_last_error());
			}
		}

		return $result;
	}

	/**
	 * write to log
	 *
	 * @static
	 * @param string $query
	 * @param array $vars
	 * @return void
	 */
	private static function log($query = "", array $vars = array())
	{
		if (self::$isDebug)
		{
			if ($query !== "")
			{
				// TODO: extend code for logging

				self::$queryList[] = array(
					"query" => $query,
					"params" => $vars
				);
			}
		}
	}

	/**
	 * get all queries with params
	 *
	 * @static
	 * @return array
	 */
	public static function getLog()
	{
		return self::$queryList;
	}

	/**
	 * get last query string
	 *
	 * @static
	 * @return string
	 */
	public static function getLastQuery()
	{
		return self::$lastQuery;
	}

	/**
	 * get last query params
	 *
	 * @static
	 * @return array
	 */
	public static function getLastQueryParams()
	{
		return self::$lastQueryParams;
	}

	/**
	 * return time of all queries in ms
	 *
	 * @static
	 * @return float time in ms
	 */
	public static function getTime()
	{
		return self::$timeForQueries * 1000;
	}

	public static function getNumber()
	{
		return self::$numberOfQueries;
	}

	/**
	 * creates DB connection string
	 *
	 * @static
	 * @return string connection string
	 */
	private static function getConnectionString()
	{
		if (!isset(self::$connectionString))
		{
			if (!isset(self::$configuration[self::$currentConfiguration]["parameters"]))
			{
				$parameters = array();

				// read parameters from file
				$rows = file(APL_PATH . self::$configuration[self::$currentConfiguration]["path"], FILE_IGNORE_NEW_LINES|FILE_SKIP_EMPTY_LINES);
				foreach ($rows as $row)
				{
					list($parameter, $value) = preg_split("/=/", $row, 2);
					$parameters[$parameter] = $value;
				}

				self::$configuration[self::$currentConfiguration]["parameters"] = $parameters;
			}

			$parameters = self::$configuration[self::$currentConfiguration]["parameters"];

			// create DB connection string
			self::$connectionString = "host={$parameters["host"]} dbname={$parameters["name"]} user={$parameters["user"]} password={$parameters["password"]}";
		}

		return self::$connectionString;
	}

	/**
	 * switch to DB local conf
	 * @static
	 * @return void
	 */
	public static function switchToLocal()
	{
		self::$currentConfiguration = "local";
		self::connect();
	}

	/**
	 * switch to DB master conf
	 * @static
	 * @return void
	 */
	public static function switchToMaster()
	{
		self::$currentConfiguration = "master";
		self::connect();
	}

	/**
	 * Return connection to DB.
	 * Implemented for old code, please use methods of this class for working with DB.
	 *
	 * @static
	 * @return resource
	 */
	public static function getConnection()
	{
		self::connect();

		return self::$connection;
	}

	/**
	 * check connection to DB
	 *
	 * @static
	 * @return bool
	 */
	public static function checkConnection()
	{
		$result = true;

		try
		{
			self::connect();
			self::select("SELECT * FROM now();");
		}
		catch (DBException $ex)
		{
			$result = false;
		}

        return $result;
    }
}
