<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: Configuration.php 33126 2015-10-01 19:01:51Z astarostin $
 * ------------------------------------------------------------------------------
 * This class represents HTTP API for Configuration objects
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2010
 * ------------------------------------------------------------------------------
 */

class Configuration extends ObjectTemplate
{
	public function __construct()
	{
		$this->otype = "G";
		$this->subtype = "C";
		parent::__construct("configuration");
	}

	/**
	 * @param array $attributes
	 * @return int
	 */
	public function create($attributes)
	{
		$obj = parent::create($attributes);

		$userid = $attributes['OWNER'];
		if(isset($attributes['USERID']))
		{
			$userid = $attributes['USERID'];
		}
		Audit::addRecordVArg(8, $userid, null, 'added', Audit::attrValueEntity('NAME', $attributes['NAME']), Audit::attrValueEntity('SHARED', $attributes['SHARED'] ? "Yes" : "No"));

		return $obj;
	}

	/**
	 * delete configuration
	 *
	 * @throws InvalidArgumentException
	 * @param  $obj
	 * @param  bool $onlyMark - set deleted = 1 (by default) or delete from DB
	 * @return bool
	 */
	public function delete($obj, $onlyMark = true)
	{
		parent::delete($obj, $onlyMark);

		Audit::addRecordVArg(8, $this->getAttribute($obj, 'OWNER'), null, 'deleted', Audit::attrValueEntity('NAME', $this->getAttribute($obj, 'NAME')), Audit::attrValueEntity('SHARED', $this->getAttribute($obj, 'SHARED') ? "Yes" : "No"));
	}

	/**
	 * get configuration attribute value
	 * @param  $obj
	 * @param  string $attribute
	 * @return string|null
	 */
	public function getAttribute($obj, $attribute)
	{
		$value = parent::getAttribute($obj, $attribute);

		if ($attribute == "DATA")
		{
			Audit::addRecordVArg(8, $this->getAttribute($obj, 'OWNER'), null, "loaded", Audit::attrValueEntity('NAME', $this->getAttribute($obj, 'NAME')), Audit::attrValueEntity('SHARED', $this->getAttribute($obj, 'SHARED') ? "Yes" : "No"));
		}

		return $value;
	}

	public function setAttributes($obj, $attributes)
	{
		parent::setAttributes($obj, $attributes);

		Audit::addRecordVArg(8, $attributes['OWNER'], null, $attributes['SHARED'] ? "shared" : "privatized", Audit::attrValueEntity('NAME', $this->getAttribute($obj, 'NAME')));
	}

	public function getList()
	{
		$userId = $_SESSION[SESSION_USERID];

		// Select only configurations that belong to current user
		// and shared configurations of other users
		// Configurations must belong to one of user roles
		$attr = DB::select("
			SELECT *
				FROM _obj_attr
				WHERE
					obj IN (
						SELECT _objs.obj
							FROM _objs, _obj_attr
							WHERE
								_objs.otype = 'G'
								AND _objs.subtype = 'C'
								AND _objs.deleted != 1
								AND _objs.obj = _obj_attr.obj
								AND (
									(_obj_attr.attr = 'OWNER' AND _obj_attr.val = ?)
									OR (_obj_attr.attr = 'SHARED' AND _obj_attr.val = '1')
								)
					)
					AND attr != 'DATA'",
			array($userId)
		);

		$obj = new Obj();
		$list = array();
		foreach ($attr as $row) {
			$list[$row['obj']]['obj'] = $row['obj'];
			$list[$row['obj']]['attributes'][$row['attr']] = $row['val'];
			if ($row['attr'] == 'OWNER') {
				// Get user name for object
				if ($row['val'] == $userId) {
					$list[$row['obj']]['attributes']['OWNER_NAME'] = $_SESSION[SESSION_USERNAME];
				} else {
					$list[$row['obj']]['attributes']['OWNER_NAME'] = $obj->getName($row['val']);
				}
			}
		}

		// Select all roles avaliable for current user
		$user = new User();
		$guis = $user->getGuis($userId);

		foreach ($guis as $gui) {
			// for admin role we return all available confs regardless to their roles
			if ($gui['obj'] == Obj::CONTROL_PANEL)
				return array_values($list);
		}

		$roles = $user->getRoles($userId);
		$rolesId = array();
		foreach ($roles as $role) {
			$rolesId[] = $role['obj'];
		}

		foreach ($list as $row) {
			if (!in_array($row['attributes']['ROLEID'], $rolesId)) {
				unset($list[$row['obj']]);
			}
		}

		return array_values($list);
	}

	public function getListByType($roleid, $subtype = 'configuration') {
		if ($subtype == 'configuration') {
			return $this->getList();
		}

		if (!isset($roleid))
		{
			throw new InvalidArgumentException("roleid should be specified");
		}

		if (!in_array($subtype, array('tab', 'ge'))) {
			throw new InvalidArgumentException("Incorrect value of subtype argument");
		}

		// Get all configurations of subtype
		$attr = DB::select("
			SELECT *
				FROM _obj_attr
				WHERE
					obj IN (
						SELECT _objs.obj
							FROM _objs, _obj_attr
							WHERE
								_objs.otype = 'G'
								AND _objs.subtype = 'C'
								AND _objs.deleted != 1
								AND _objs.obj = _obj_attr.obj
								AND _obj_attr.attr = 'SUBTYPE'
								AND _obj_attr.val = ?
					)",
			array($subtype)
		);

		$list = array();
		foreach ($attr as $row) {
			$list[$row['obj']]['obj'] = $row['obj'];
			$list[$row['obj']][strtolower($row['attr'])] = $row['val'];
		}

		// Filter all views by role id
		foreach ($list as $row) {
			if ($row['roleid'] != $roleid) {
				unset($list[$row['obj']]);
			}
		}

		$obj = new Obj();
		foreach ($list as $row) {
			$list[$row['obj']]['username'] = $obj->getName($row['userid']);
			$list[$row['obj']]['rolename'] = $obj->getName($row['roleid']);
		}

		return $list;
	}

}
