<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: Camera.php 34479 2016-04-27 15:44:15Z astarostin $
 * ------------------------------------------------------------------------------
 * This class represents logic for Camera type devices
 * ------------------------------------------------------------------------------
 * @author Sergey Tsybulsky, Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2010
 * ------------------------------------------------------------------------------
 */

class Camera extends Device
{
	public function __construct()
	{
		parent::__construct("camera", "D", "C");
	}

	/**
	 * Add new camera. Some attributes added from $attr, other attributes input from necessary cfg files.
	 *
	 * @param array $attributes attributes
	 * @param string $nodeUNI
	 *
	 * @throws Exception
	 * @return int objid
	 */
	public function create(array $attributes = array(), $nodeUNI = NULL)
	{
		if (!isset($nodeUNI))
		{
			$nodeUNI = Node::getUNI();
		}
		if ($nodeUNI !== "")
		{
			$list = DB::select(
				"SELECT obj FROM _objs WHERE otype = 'D' AND subtype = 'N' AND name = ? AND deleted = '0';",
				array($nodeUNI)
			);
			if (isset($list[0]['obj']))
			{
				$nodeid = $list[0]['obj'];
			}
		}

		$toAvatar = !empty($attributes["AVATARID"]);

		$node = new Node();
		$CAMERA_REGISTRATION = Template::boolVal($node->getAttribute($nodeid, "CAMERA_REGISTRATION"));
		if (!$toAvatar && !$CAMERA_REGISTRATION)
		{
			throw new InvalidArgumentException("ERROR: camera can not be registered on this node");
		}

		$otype = 'D';
		$subtype = 'C';

		$CAMERAMODEL = $attributes["CAMERAMODEL"];
		$MODELID = isset($attributes["MODELID"]) ? trim($attributes["MODELID"]) : "";

		//Get ASSOCIATE and check it for correct values
		if (isset($attributes["ASSOCIATE"]))
		{
			$res_assoc = $this->isASSOCIATEvalid('', $attributes["ASSOCIATE"]);
			if (isset($res_assoc))
			{
				$attributes["ASSOCIATE"] = $res_assoc;
			}
		}

		if (isset($attributes["IMAGESIZE_LIST"]) && !isset($attributes["IMAGESIZE"]))
		{
			$imagesizelist = Template::parseList($attributes["IMAGESIZE_LIST"]);

			// TODO: add correct checking of attribute template type
			if (isset($attributes["CAMERAMODEL"]) && ($attributes["CAMERAMODEL"] == "Vivotek" || $attributes["CAMERAMODEL"] == "panasonic"))
			{
				$imagesizelist = Template::parseList($imagesizelist[$attributes["CAMERA"]]);
			}

			foreach ($imagesizelist as $key => $value)
			{
				$attributes["IMAGESIZE"] = $key;
				break;
			}
		}

		// Only for iStream camers, in system must be only one phone with current HWID
		if (isset($attributes["HWID"]))
		{
			$HWID = $attributes["HWID"];
			$list = DB::select("
				SELECT
					_objs.obj
				FROM
					_obj_attr, _objs
				WHERE
					_objs.obj=_obj_attr.obj
					and _obj_attr.attr = 'HWID'
					and _objs.deleted = 0
					and _obj_attr.val = ?;",
				array($HWID));
			if (isset($list) && count($list) > 0)
			{
				return $list[0]["obj"];
			}
		}

		DB::query(
			"INSERT INTO _objs
				(node_id, node_ip,
				name, otype, subtype,
				deleted, protected, stime)
			VALUES
			(
			?, ?,
			now() at time zone 'UTC', ?, ?,
			0, 0, now() at time zone 'UTC');",
			array($nodeid, $nodeUNI, $otype, $subtype));

		$list = DB::select("select currval_seq_obj() as currval;");
		$newid = $list[0]['currval'];

		// sm logic
		$result = DB::select(
			"SELECT count(objid) FROM sm_space WHERE objid = ?;",
			array($newid)
		);
		$count = $result[0]["count"];
		if ($count == 0)
		{
			// we need add start point for calculating getStorageUsageTotals
			DB::query(
				"INSERT INTO
					sm_space(nodeid, objid, stime, space)
					VALUES(?, ?, now() at time zone 'UTC', 0);",
				array($nodeUNI, $newid)
			);
		}
		// end: sm logic

		// fill with defaults values from template
		$defaultAttributes = $this->getTemplate()->getDefaultAttributes($CAMERAMODEL, $MODELID);
		if (count($defaultAttributes) == 0)
		{
			throw new Exception("Problem with reading template .cfg files");
		}

		$attributes["TIME_ZONE"] = Identity::getAttribute("TIME_ZONE");
		$attributes["DEVID"] = $newid;
		$attributes["NODEID"] = $nodeid;

		// TODO: temporary solution for camera stream prioritization
		if (isset($defaultAttributes["AV_PRIORITY"]))
		{
			$defaultAttributes["AV_PRIORITY"] = $newid;
		}

		foreach ($attributes as $key => $value)
		{
			if ($value === null)
			{
				$value = "";
			}

			if (isset($defaultAttributes[$key]))
			{
				$defaultAttributes[$key] = $value;
			}
		}

		$mediaFormatList = Template::parseList ( $defaultAttributes['MEDIA_FORMAT_LIST'] );
		if(count($mediaFormatList) > 0 && !isset($mediaFormatList [ $defaultAttributes['MEDIA_FORMAT'] ] ) ){
			$defaultAttributes['MEDIA_FORMAT']=reset($mediaFormatList);
		}
		//We get non-standard default imagesize list for Bosch cameras. We need to process it properly.
		if (isset($defaultAttributes["CAMERAMODEL"]) && ($defaultAttributes["CAMERAMODEL"] == "Bosch"))
		{
			$imagesizelist = Template::parseList($defaultAttributes["IMAGESIZE_LIST"]);
			$imagesizelist = Template::parseList($imagesizelist[$defaultAttributes["MEDIA_FORMAT"]]);
			$defaultAttributes['IMAGESIZE'] = key($imagesizelist);
		}

		$this->createAttributes($newid, $defaultAttributes);

		$set = new Set();
		// add camera to default set and protect it
		$types = Obj::getTypes();
		$set->add($types[$this->otype][$this->subtype]["defaultsetobj"], $newid);
		if ($attributes["CAMERAMODEL"] == "iStream")
		{
			$set->add(Set::MOBILE_CAMERAS, $newid);
		}

		// audit camera creation
		Audit::addRecordVArg(26, $newid, $_SESSION[SESSION_USERID], "Camera", "added", Audit::attrValueEntity('MODEL', $CAMERAMODEL), Audit::attrValueEntity('IP', $defaultAttributes["DEVIP"]), Audit::attrValueEntity('MEDIAFORMAT', $defaultAttributes["MEDIA_FORMAT"]));

		return $newid;
	}

	/**
	 * check for attribute occurrence in template
	 * @param  string $attribute
	 * @param  string [$type]
	 * @param  string [$cameraModel]
	 * @return bool
	 */
	private function inTemplate($attribute, $type, $cameraModel = "")
	{
		$result = false;

		$attributes = array(
			"NAME" => "",
			"LOCATION" => "",
			"DEVIP" => "",
			"HTTP_PORT" => "",
			"USRNAME" => "",
			"PASSWD" => "",
			"CAMERA" => "",
			"CAMERA_LIST" => "",
			"DEVURL" => "",
			"CAMERAMODEL" => "",
			"MODELID" => "",
			"FIRMWARE" => "",
			"IMAGESIZE_LIST" => "",
			"MEDIA_FORMAT" => "",
			"MEDIA_FORMAT_LIST" => "",
			"RC_MAXBITRATE_LIST" => "",
			"AUDIO_LIST" => "",
			"AUDIO_FORMAT_LIST" => "",
			"POSITIONCTL" => "",
			"POSITIONCTL_LIST" => "",
			"PTZID" => "",
			"PTZID_LIST" => "",
			"FRAMERATE_LIST" => "",
			"RTP_OVER_TCP_LIST" => "",
			"RTP_OVER_TCP" => "",
			"INTRAFRAME" => "",
			"AE_DESCR" => "",
			"SOURCE"  => "",
			"SOURCE_BASE"  => "",
			"PTZ_HFOV_MIN" => "",
			"PTZ_HFOV_MAX" => "",
			"PTZ_FOCAL_LENGTH_MIN" => "",
			"PTZ_FOCAL_LENGTH_MAX" => "",
			"ONVIF_PROFILE" => "",
			"ONVIF_PROFILES" => "",
			"ONVIF_PROFILE_LIST" => "",
			"ONVIF_MEDIA_SERVICE" => "",
			"ONVIF_PTZ_SERVICE" => ""
		);

		if (isset($attributes[$attribute]))
		{
			$result= true;
		}

		return $result;
	}

	/**
	 * @param  array $modelAttributes
	 * @return array
	 */
	public function getTicketForModels(array $modelAttributes)
	{
		$tickets = array();

		foreach ($modelAttributes as $attributes)
		{
			$tickets[] = $this->getTicketForModel($attributes);
		}

		return $tickets;
	}

	/**
	 * create request for camera discovery
	 *
	 * @param array $attributes
	 * @param int|null $nodeid
	 * @throws Exception
	 * @return string|null
	 */
	public function getTicketForModel(array $attributes, $nodeid = null)
	{
		$ticket = null;

		if (isset($attributes["CAMERAMODEL"]) && $attributes["CAMERAMODEL"] != 'auto')
		{
			// add additional parameters for probe script
			if(!isset($attributes["probe_single_brand"]) ) $attributes["probe_single_brand"] = "1";
			if(!isset($attributes["force_probing"]) ) $attributes["force_probing"] = "1";
		}

		if (isset($attributes["DEVIP"]) && isset($attributes["HTTP_PORT"]))
		{
			$avatar = new Avatar();
			$isAvatar = !empty($nodeid) && $avatar->isAvatar($nodeid);

			$pathToQueueDir = $isAvatar ? APL_VAR_PATH . "/probe/avatar_queue" : APL_VAR_PATH . "/probe/queue";

			// format: <host>_<port>_<input_number>_<timestamp>
			$getTicketName = function($ip, $port, $pathToQueueDir, $isAvatar)
			{
				do
				{
					$someNumber = rand(0, 100000);
					$ticketName = "{$ip}_{$port}_{$someNumber}";
					if ($isAvatar)
					{
						$ticketName = "avatar_$ticketName";
					}

				} while (is_file("$pathToQueueDir/$ticketName"));

				return $ticketName;
			};

			$ticketName = $getTicketName($attributes["DEVIP"], $attributes["HTTP_PORT"], $pathToQueueDir, $isAvatar);

			$pathToTicket = "$pathToQueueDir/$ticketName";

			$parameters = "";
			foreach ($attributes as $attribute=>$value)
			{
				$parameters .= "$attribute=$value\n";
			}

			if (file_put_contents($pathToTicket, $parameters))
			{
				$ticket = $ticketName;
			}

			if ($isAvatar && isset($ticket))
			{
				$avatarNodeId = $avatar->getNodeId($nodeid);
				$node = new Node();
				$IP = $node->getAttribute($avatarNodeId, "IP");
				$avatarUNI = $avatar->getAttribute($nodeid, "UNI");

				// create request to avatar
				$command = "APL_VAR=/var/sarch APL_CONF=/var/sarch/conf APL=/opt/sarch " . APL_PATH . "/av/bin/camera_discovery -n $IP -o $nodeid -u $avatarUNI -t $ticketName 2>&1";
				$output = array();
				$returnVar = 0;
				exec($command, $output, $returnVar);
				if ($returnVar != 0)
				{
					throw new Exception("problems while creating request to avatar, command='$command' code=$returnVar " . print_r($output, true));
				}
			}
		}

		return $ticket;
	}

	/**
	 * @param  $tickets
	 * @return array
	 */
	public function getModelByTickets(array $tickets)
	{
		$models = array();

		foreach ($tickets as $ticket)
		{
			$models[] = $this->getModelByTicket($ticket);
		}

		return $models;
	}

	public function getModelByTicket($ticket)
	{
		$result = array();

		$pathToQueue = APL_VAR_PATH . "/probe/queue/" . $ticket;
		$pathToAvatarQueue = APL_VAR_PATH . "/probe/avatar_queue/" . $ticket;
		$pathToResult = APL_VAR_PATH . "/probe/stage/" . $ticket;
		$pathToLog = APL_VAR_PATH . "/probe/log/" . $ticket;

		if (is_file($pathToResult))
		{
			$parameters = array();
			$attributes = array();
			$log = array();
			$lines = file($pathToResult);
			foreach ($lines as $line)
			{
				list($key, $value) = explode('=', $line);
				$key = trim($key);
				$value = trim($value);
				if ($this->inTemplate($key, "camera") && $key != "CAMERA_LIST")
				{
					$attributes[$key] = $value;
				} else {
					$parameters[$key] = $value;
				}

				// TODO: modify this wrong logic of checking attribute for camera
				if ($key == "CAMERA_LIST")
				{
					$attributes[$key] = $value;
					$parameters[$key] = $value;
				}
			}
			if (is_file($pathToLog))
			{
				$lines = file($pathToLog);
				foreach ($lines as $line)
				{
					$log[] = $line;
				}
			}

			$exists = "";
			$camexists = array();
			$existList = array();
			if ($parameters["state"] == "OK")
			{
				if (isset($parameters["SNAPSHOT_LIST"]))
				{
					$parameters["SNAPSHOT_LIST"] = Template::parseList($parameters["SNAPSHOT_LIST"]);
					// list should contain only file name
					foreach ($parameters["SNAPSHOT_LIST"] as $obj => &$path)
					{
						$path = basename($path);
					}
				}

				if (isset($parameters["CAMERA_LIST"]))
					$parameters["CAMERA_LIST"] = Template::parseList($parameters["CAMERA_LIST"]);

				// check if camera already exists in the system
				$queryString =
					"SELECT _objs.obj, _objs.udid, _objs.name, attr2.val as camera
						FROM
							_objs, _obj_attr as attr1, _obj_attr as attr2
						WHERE
							_objs.deleted = 0
							AND _objs.subtype = 'C'
							AND _objs.otype = 'D'
							AND _objs.obj = attr1.obj
							AND _objs.obj = attr2.obj
							AND attr1.attr = 'DEVIP'
							AND attr1.val = ?
							AND attr2.attr = 'CAMERA';";
				$list = DB::select($queryString, array($attributes["DEVIP"]));

				if (count($list) > 0)
				{
					$n = 0;
					$cameraNameList = "";
					foreach ($list as $row)
					{
						$existList[$row['obj']] = array();
						$existList[$row['obj']]["attributes"] = array();
						$existList[$row['obj']]["attributes"]["DEVIP"] = $attributes["DEVIP"];
						$existList[$row['obj']]["attributes"]["CAMERA"] = $row["camera"];

						$camexists[] = $row['obj'];
						$cameraNameList .= "'[" . $row['udid'] . "] " . $row['name'] . "' ";
						$n++;
						if (count($list) > 0 && $n != count($list))
							$cameraNameList .= ", ";
					}
					$cameraNameList = trim($cameraNameList);
					$exists = sprintf(__("The camera you are configuring is already registered in the system as %s. Adding multiple registrations for the same camera will open multiple video streams from the camera and may lead to video performance degradation."), $cameraNameList);
				}

				$result["attributes"] = $attributes;
			}

			$result["existList"] = $existList;
			$result["exists"] = $exists;
			$result["camexists"] = $camexists;
			$result["parameters"] = $parameters;
			$result["log"] = $log;
		} else
		if (is_file($pathToQueue) || is_file($pathToAvatarQueue))
		{
			$result = array(
				"parameters" => array("state" => "inqueue"),
			);
		}

		return $result;
	}

	/**
	 * synchronous camera probing
	 * @param  array $attributes
	 * @param  bool $onlyOneModelSearch
	 * @return array
	 */
	public function getModelAttributes(array $attributes, $onlyOneModelSearch = false)
	{
		if ($onlyOneModelSearch)
		{
			// add additional parameters for probe script
			$attributes["probe_single_brand"] = "1";
			$attributes["force_probing"] = "1";
		}

		$ticket = $this->getTicketForModel($attributes);

		$response = $this->getModelByTicket($ticket);

		while (!isset($response["parameters"]) || ($response["parameters"]["state"] !== "OK" && $response["parameters"]["state"] !== "ERROR"))
		{
			sleep(1);
			$response = $this->getModelByTicket($ticket);
		}

		return $response;
	}

	/**
	 * @return array
	 */
	public static function getModelList()
	{
		$models = array();
		$cartrige_dir = APL_PATH . "/conf/etc/device/camera/";
		$files = scandir($cartrige_dir);
		if (is_array($files))
		{
			foreach($files as $file)
			{
				if ($file !== "."
					&& $file !== ".."
					&& $file !== "default"
					&& is_dir($cartrige_dir . $file)
					&& is_file("$cartrige_dir/$file/etc/descr.xml")
				)
				{
					$element = array();
					$element["name"] = $file;
					array_push($models, $element);
				}
			}
		}
		return $models;
	}

	/**
	 * @return array
	 */
	public function getOrderedModelList()
	{
		$models = array();
		$cartrige_dir = APL_PATH . "/conf/etc/device/camera/";
		$files = scandir($cartrige_dir);
		if (is_array($files))
		{
			foreach($files as $file)
			{
				if ($file != "."
					&& $file != ".."
					&& $file != "default"
					&& is_dir($cartrige_dir . $file)
					&& is_file("$cartrige_dir/$file/etc/cartridge.cfg")
				)
				{
					$pathToCartridge = "$cartrige_dir/$file/etc/cartridge.cfg";
					$lines = file($pathToCartridge);
					$models[$file] = str_replace('PROBEORDER=', '', $lines[0]);
				}
			}
		}
		return $models;
	}

	/**
	 * @param $obj
	 * @param $assobj
	 * @return bool
	 */
	private function canDeleteAssociate($obj, $assobj)
	{
		$result = true;
		$obj = (int)$obj;
		$assobj = (int)$assobj;

		$list = DB::select("
			SELECT _objs.obj FROM _objs, _obj_attr
				WHERE _objs.obj = ?
				AND _objs.obj = _obj_attr.obj
				AND _objs.otype = 'D' AND _objs.subtype = 'A'
				AND _obj_attr.attr = 'BELONGS2DEVID'
				AND _obj_attr.val = ?
				AND _objs.deleted = 0;", array($assobj, $obj));
		if (isset($list[0]['obj']))
		{
			$result = false;
		}
		return $result;
	}

	/**
	 * @param $obj
	 * @return array
	 */
	private function getUndeletedAssociateList($obj)
	{
		$obj = (int)$obj;

		$ASSOCIATE = parent::getAttribute($obj, "ASSOCIATE");
		$new_ass_arr = array();
		if (isset($ASSOCIATE))
		{
			$ass_arr = preg_split("/,/", $ASSOCIATE);
			foreach ($ass_arr as $objid)
			{
				$objid = trim($objid);
				if (!$this->canDeleteAssociate($obj, $objid))
					$new_ass_arr[] = $objid;
			}
		}
		return $new_ass_arr;
	}

	private function mergeUndeletedAssociate($assstr, $obj)
	{
		$und_arr = $this->getUndeletedAssociateList($obj);
		if (count($und_arr) > 0)
		{
			if (trim($assstr) == '')
				return implode(", ", $und_arr);
			$ass_arr = preg_split("/, /", $assstr);
			foreach ($und_arr as $undobjid)
			{
				$undobjid = trim($undobjid);
				$inList = false;
				foreach ($ass_arr as $objid)
				{
					$objid = trim($objid);
					if ($undobjid == $objid)
					{
						$inList = true;
						break;
					}
				}
				if (!$inList)
					$ass_arr[] = $undobjid;
			}
			$result = implode(", ", $ass_arr);
			return $result;
		}
		else
			return $assstr;
	}

	/**
	 * Check, is associate device list valid
	 *
	 * @param string $obj camera ID
	 * @param string $assstr input associate device list
	 * @return string output associate device list
	 * @throws InvalidArgumentException
	 */
	private function isASSOCIATEvalid($obj = '', $assstr)
	{
		if (trim($assstr) == '')
		{
			return "";
		}

		$ass_arr = preg_split("/,/", $assstr);
		$new_ass_arr = array();
		$devs_present = true;
		foreach ($ass_arr as $objid)
		{
			$objid = trim($objid);
			if ($obj == $objid && $obj !== '')
			{
				$devs_present = false;
				$objid = ">>" . $objid . "<<";
			}
			else if ((int)$objid > 0)
			{

				$list = DB::select("SELECT obj FROM _objs WHERE ((otype = 'D' and subtype <> 'N') OR (otype = 'X' and subtype <> 'G')) AND obj = ? AND deleted = 0;", array($objid));
				if (!isset($list[0]['obj']))
				{
					$devs_present = false;
					$objid = ">>" . $objid . "<<";
				}
			}
			else
			{
				$devs_present = false;
				$objid = ">>" . $objid . "<<";
			}
			$new_ass_arr[] = $objid;
		}

		if (!$devs_present)
		{
			$err_list = implode(", ", $new_ass_arr);
			$err_msg = "[$obj] " . $this->getName($obj) . "\n" .  __("Incorrect list of Associated devices.") . "\n" . $err_list;
			throw new InvalidArgumentException($err_msg);
		} else {
			$assstr = implode(", ", $new_ass_arr);
		}

		return $assstr;
	}

	/**
	 * Edit camera attributes with ability of change CAMERAMODEL.
	 *
	 * @param int   $obj camera ID
	 * @param array $attributes new attributes
	 * @return void
	 * @throws InvalidArgumentException
	 */
	private function setAttributesExt($obj, array $attributes)
	{
		$CAMERAMODEL = $attributes["CAMERAMODEL"];
		$MODELID = $attributes["MODELID"];

		$doNotChangeAttributeList = array(
			"UDID", "OBJID", "DEVIP", "HTTP_PORT", "USRNAME", "PASSWD", "NAME", "LOCATION", "ASSOCIATE", "AUDIO_DEVID", "CAMERAMODEL", "MODELID", "CAMERAFIRMWARE", "AVATARID", "NODEID"
		);

		$currentAttributes = $this->getAttributes($obj);
		$currentCAMERAMODEL =$currentAttributes["CAMERAMODEL"];
		$currentMODELID = $currentAttributes["MODELID"];

		// fill attributes with current values
		if ($currentCAMERAMODEL == $CAMERAMODEL && $currentMODELID == $MODELID)
		{
			foreach ($currentAttributes as $currentAttribute => $currentValue)
			{
				if (isset($attributes[$currentAttribute])
					&& !in_array($currentAttribute, $doNotChangeAttributeList)
				)
				{
					$attributes[$currentAttribute] = $currentValue;
				}
			}
		}

		$attributes["DEVID"] = $obj;

		$isProtected = $this->isProtected($obj);
		if (!isset($isProtected))
		{
			$err_msg = sprintf( __("Camera with obj = %s doesn't exists"), $obj);
			throw new InvalidArgumentException($err_msg);
		}

		if ($isProtected)
		{
			$err_msg =  __('Can not modify protected object.');
			throw new InvalidArgumentException($err_msg);
		}

		//Get ASSOCIATE and check it for correct values
		if (isset($attributes["ASSOCIATE"]))
		{
			$res_assoc = $this->isASSOCIATEvalid($obj, $attributes["ASSOCIATE"]);
			if (isset($res_assoc))
			{
				$attributes["ASSOCIATE"] = $this->mergeUndeletedAssociate($res_assoc, $obj);
			}
		}

		// get old AUDIO_DEVID and check it for correct values
		if (isset($currentAttributes["AUDIO_DEVID"]))
		{
			$attributes["AUDIO_DEVID"] = $currentAttributes["AUDIO_DEVID"];
		}

		$this->deleteAllAttributes($obj);

		// fill with defaults values from template
		$defaultAttributes = $this->getTemplate()->getDefaultAttributes($CAMERAMODEL, $MODELID);
		if (count($defaultAttributes) == 0)
		{
			throw new InvalidArgumentException(__('Problem with reading template .cfg files'));
		}

		// fill with input attributes
		foreach ($attributes as $attribute => $value)
		{
			if ($value === null)
			{
				$value = "";
			}

			if (isset($defaultAttributes[$attribute]))
			{
				$defaultAttributes[$attribute] = $value;
			}
		}

		$this->createAttributes($obj, $defaultAttributes);
	}

	/**
	 * get list of attributes which are disabled for schedule
	 *
	 * @param  int $obj
	 * @return array
	 */
	private function getDisabledAttributes($obj)
	{
		$disabledAttributes = array();

		$scheduleid = $this->getAttribute($obj, "SCHEDULEID");
		if (!empty($scheduleid))
		{
			$schedule = new Schedule();
			$defaultPostureId = $schedule->getDefaultPostureId($scheduleid);

			$posture = new Posture();
			$properties = $posture->getProperties($defaultPostureId);

			foreach ($properties as $key => $value)
			{
				$disabledAttributes = array_merge($disabledAttributes, $posture->getAttributesAssignedToProperty($key));
			}
		}

		return $disabledAttributes;
	}

	/**
	 * @param int   $obj camera ID
	 * @param array $attributes new attributes
	 * @throws InvalidArgumentException
	 * @throws Exception
	 */
	public function setAttributes($obj, array $attributes)
	{

		// make all not configurable attributes readonly
		// TODO: turn on next logic for ENCODER_SETTING_OVERRIDE
		/*
		if (
			$this->getAttribute($obj, "ENCODER_SETTING_OVERRIDE") == "no"
			&&
			(
				isset($attributes["ENCODER_SETTING_OVERRIDE"]) && $attributes["ENCODER_SETTING_OVERRIDE"] == "no"
				||
				!isset($attributes["ENCODER_SETTING_OVERRIDE"])
			)
		)
		{
			foreach ($attributes as $attribute => $value)
			{
				if ($this->isAttributeCanBeOverriden($attribute))
				{
					unset($attributes[$attribute]);
				}
			}
		}
		*/

		// check for attributes which are disabled for schedule
		$disabledAttributes = $this->getDisabledAttributes($obj);
		foreach ($attributes as $key => $value)
		{
			if (in_array($key, $disabledAttributes))
			{
				throw new Exception("Attribute '$key' is readonly for schedule");
			}
		}

		$isProtected = $this->isProtected($obj);
		if (!isset($isProtected))
		{
			$err_msg = sprintf(__("Camera with obj = %s doesn't exists."), $obj);
			throw new InvalidArgumentException($err_msg);
		}

		if ($isProtected)
		{
			$err_msg =  __('Can not modify protected object.');
			throw new InvalidArgumentException($err_msg);
		}

		$VAEList = Node::getVAEList();
		if (isset($attributes["SCHEDULEID"]))
		{
			$oldscheduleid = $this->getAttribute($obj, 'SCHEDULEID');
			if ($oldscheduleid && $oldscheduleid != '')
			{
				$this->updateSTime($oldscheduleid);
			}

			// if SCHEDULEID was changed, ANALYTICS_SCHEDULER must be changed too
			// ANALYTICS_SCHEDULER - has json format
			// we need to clear "postures" subattribute and set "state" subattribute to "nonactive" value

			foreach($VAEList as $VAE_NAME => $path){
				$VAE_NAME = strtoupper($VAE_NAME);
				if ($oldscheduleid != $attributes["SCHEDULEID"] && !isset($attributes["VAE_${VAE_NAME}_SCHEDULER"]))
				{
					$j_ANALYTICS_SCHEDULER = $this->getAttribute($obj, "VAE_${VAE_NAME}_SCHEDULER");
					$ANALYTICS_SCHEDULER = json_decode($j_ANALYTICS_SCHEDULER, true);

					if (is_array($ANALYTICS_SCHEDULER))
					{
						for ($i = 0; $i <= count($ANALYTICS_SCHEDULER); $i++)
						{
							if (isset($ANALYTICS_SCHEDULER[$i]["postures"]))
							{
								$ANALYTICS_SCHEDULER[$i]["postures"] = array();
							}
							if (isset($ANALYTICS_SCHEDULER[$i]["state"]) && $ANALYTICS_SCHEDULER[$i]["state"] == "when")
							{
								$ANALYTICS_SCHEDULER[$i]["state"] = "nonactive";
								$ANALYTICS_SCHEDULER[$i]["active"] = false;
							}
						}
						$ANALYTICS_SCHEDULER = json_encode($ANALYTICS_SCHEDULER);
						$attributes["VAE_${VAE_NAME}_SCHEDULER"] = $ANALYTICS_SCHEDULER;
					}
				}
			}

			if (!empty($attributes["SCHEDULEID"]))
			{
				$this->updateSTime($attributes["SCHEDULEID"]);
			}
		}

		/* if attr VAE_${VAE_NAME}_SCHEDULER was changed
		// we need to change scheduler's sTime too
		*/
		$oldscheduleid = $this->getAttribute($obj, 'SCHEDULEID');
		if (isset($attributes["SCHEDULEID"]))
		{
			$scheduleid = $attributes["SCHEDULEID"];
		} else {
			$scheduleid = $oldscheduleid;
		}

		if (!empty($scheduleid)){
			$doChangeStime = false;
			foreach($VAEList as $VAE_NAME => $path){
				if (isset($attributes[strtoupper ("VAE_${VAE_NAME}_SCHEDULER")])){
					$doChangeStime = true;
				}
			}
			if ($doChangeStime) $this->updateSTime($scheduleid);
		}

		$oldAttributes = parent::getAttributes($obj);

		// create VAE configuration for retriever
		$VAE_CONFIG = array("engines" => array());
		$list = array("ACTIVE", "MIN_FPS", "MAX_FPS");
		foreach ($VAEList as $vae => $path)
		{
			foreach ($list as $attribute)
			{
				$realAttribute = "VAE_" . strtoupper($vae) . "_" . $attribute;
				$value = isset($attributes[$realAttribute]) ? $attributes[$realAttribute] : $oldAttributes[$realAttribute];
				$VAE_CONFIG["engines"][$vae][$attribute] = $value;
			}
		}
		$attributes["VAE_CONFIG"] = json_encode($VAE_CONFIG);

		// fill UPDATE_TIME attribute
		$attributes["UPDATE_TIME"] = time();

		if (isset($attributes["CAMERAMODEL"]) && isset($attributes["MODELID"]))
		{
			$templateattrs = $this->getTemplate()->getTemplateAttributes($attributes["CAMERAMODEL"], $attributes["MODELID"], $obj);
			if (isset($templateattrs['AUDIO']))
			{
				$attributes['AUDIO'] = $templateattrs['AUDIO'];
			} else {
				$attributes['AUDIO'] = '';
			}

			$this->setAttributesExt($obj, $attributes);
		} else {
			//Get ASSOCIATE and check it for correct values
			if (isset($attributes["ASSOCIATE"]))
			{
				$res_assoc = $this->isASSOCIATEvalid($obj, $attributes["ASSOCIATE"]);
				if (isset($res_assoc))
				{
					$attributes["ASSOCIATE"] = $this->mergeUndeletedAssociate($res_assoc, $obj);
				}
			}

			$auditValues = array("Camera", "edited");
			foreach ($attributes as $key => &$value)
			{
				if ($value === null)
				{
					$value = "";
				}

				// Important to save json analytics data for camera!
				if (is_object($value) || is_array($value))
					$value = json_encode($value);

				array_push($auditValues, Audit::attrChangeEntity($key, $oldAttributes[$key], $value));
			}

			parent::setAttributes($obj, $attributes);

			// audit camera edit
			Audit::addRecord(26, $auditValues, $obj, $_SESSION[SESSION_USERID]);

			if (!empty($attributes["SCHEDULEID"]))
			{
				$schedule = new Schedule();
				$schedule->auditObjectsOnSchedule($attributes["SCHEDULEID"]);
			}

			$this->updateSTime($obj);
		}
	}

	/**
	 * delete camera
	 *
	 * @throws Exception
	 * @param  int $obj
	 * @param  bool $onlyMark - set deleted = 1 (by default) or delete from DB
	 * @return bool
	 */
	public function delete($obj, $onlyMark = true)
	{
		$isProtected = $this->isProtected($obj);
		$scheduleid = $this->getAttribute($obj, "SCHEDULEID");
		if (!isset($isProtected))
		{
			$err_msg = sprintf( __('There is no camera: %s.'), $obj);
			throw new Exception($err_msg);
		}

		if ($isProtected)
		{
			$err_msg = __('Can not delete protected object.');
			throw new Exception($err_msg);
		}

		// delete $obj from Assotiate list in other objects
		$i_obj = (int)$obj;
		$list = DB::select("
			SELECT
				obj, val FROM _obj_attr
			WHERE
				attr  = 'ASSOCIATE'
				and (val = ? or val like ? or val like ? or val like ?);", array("$i_obj", "$i_obj, %", "%, $i_obj, %", "%, $i_obj"));
		if ($list !== FALSE)
		{
			foreach ($list as $row)
			{
				$ass_arr = preg_split("/, /", $row["val"]);
				$new_ass_arr = array();
				foreach ($ass_arr as $ass)
				{
					if ($ass != $i_obj)
						$new_ass_arr[] = $ass;
				}
				$new_row = implode(", ", $new_ass_arr);

				parent::setAttributes($row["obj"], array("ASSOCIATE" => $new_row));
			}
		}

		parent::delete($obj, $onlyMark);

		// If camera was assigned to schedule, update schedule's stime
		if (!empty($scheduleid)) {
			$this->updateSTime($scheduleid);
		}

		// audit camera delete
		Audit::addRecordVArg(26, $obj, $_SESSION[SESSION_USERID], "Camera", "deleted");

		return true;
	}

	/**
	 * return attributes description from template
	 * @param  int|null $obj
	 * @param  array $attributes
	 * @return array
	 */
	public function getTemplateAttributes($obj = null, array $attributes = array())
	{
		$list = parent::getTemplateAttributes($obj, $attributes);

		// make all not configurable attributes readonly
		// TODO: turn on next logic for ENCODER_SETTING_OVERRIDE
		/*
		if (isset($list["attributes"]["ENCODER_SETTING_OVERRIDE"]) && $list["attributes"]["ENCODER_SETTING_OVERRIDE"]["VALUE"] == "no")
		{
			foreach ($list["attributes"] as $attribute => &$value)
			{
				if ($this->isAttributeCanBeOverriden($attribute))
				{
					if (preg_match("/^textfield/", $value["TYPE"]))
						$value["TYPE"] = "stat";
					else
					if ($value["TYPE"] == "dropmenu")
						$value["TYPE"] = "hidden";
				}
			}
		}
		*/

		// disable some attributes for schedule
		$disabledAttributes = $this->getDisabledAttributes($obj);
		foreach ($list["attributes"] as $attribute => &$value)
		{
			if (!in_array($attribute, $disabledAttributes))
				continue;

			$value["IS_DISABLED"] = "true";
			$value["BY_SCHEDULE"] = "true";

			if (preg_match("/^textfield/", $value["TYPE"]))
			{
				$value["TYPE"] = "stat";
			} else
			if ($value["TYPE"] == "dropmenu")
			{
				$newList = array();
				foreach ($value["LIST"] as $index => $row)
				{
					if ($row["value"] == $value["VALUE"])
					{
						$newList[] = $row;
					}
				}
				$value["LIST"] = $newList;
			}
		}

		return $list;
	}

	/**
	 * @param  string $attribute
	 * @return boolean
	 */
	private function isAttributeCanBeOverriden($attribute)
	{
		// TODO: use attributes list from Image category from template
		$configurableAttributesList = array(
			"ENCODER_SETTING_OVERRIDE", "STAT_IMAGESIZE", "IMAGESIZE", "IMAGESIZE_LIST", "VIEWSIZE", "STAT_FRAMERATE", "FRAMERATE", "ARCFRAMERATE", "CAMCOMPRESSION", "RC_MODE", "STAT_BITRATE", "RC_TARGETBITRATE", "RC_MAXBITRATE", "DEINTERLACE", "STAT_MEDIA_FORMAT", "MEDIA_FORMAT", "MEDIA_FORMAT_LIST", "VSTANDARD", "PIXEL_ASPECT_RATIO", "CHUNK_SIZE", "FRAMES_PER_WRITE", "PROTO", "HTTP_PORT", "RTSP_PORT"
		);

		$result = in_array($attribute, $configurableAttributesList);

		return $result;
	}

	/**
	 * get camera log
	 *
	 * @param  int $obj
	 * @return array
	 */
	public function getLog($obj)
	{
		$log = array();

		if (isset($obj))
		{
			// First get node IP
			$list = DB::select(
				"SELECT a.val AS ip FROM _obj_attr a, _objs o1, _objs o2
				WHERE o1.obj = ? AND o1.node_id=o2.obj AND a.obj=o2.obj AND a.attr = 'IP';",
				array($obj)
			);
			$ip = $list[0]['ip'];

			$re = "|^(\d{4}/\d{2}/\d{2})\s(\d{2}:\d{2}:\d{2})\s(\w+)\s-?\s?\[(\d+)\]\s+(.*)$|";
			$errlev = array('DEBUG', 'INFO', 'WARN', 'ERROR', 'FATAL');

			// Get PHP Long date format from Identity
			$dateFormat = Template::convertStrDateFormatToPHPDateFormat(Identity::getAttribute("NLS_LONG_DATE_FORMAT"));

			// Read camera log from AMQ-based daemon
			$arrEntries = array();
			/*
				Send broadcast query via ActiveMQ
			*/
			$AMQConf = Node::getAMQConf();
			$con = new Stomp($AMQConf["stomp.url"]);
			$con->setReadTimeout(5);
			$con->sync = false;
			$con->connect();
			$queue_id = uniqid(rand());
			$dest_req = "/topic/cameralog_req";
			$dest_resp = "/queue/cameralog/$queue_id";
			$con->subscribe($dest_resp);
			$result = $con->send($dest_req, "$obj;$queue_id");
			$resp = $con->readFrame();
			if ($resp !== null) {
				$con->ack($resp);
				$arrEntries = preg_split("/\n/", $resp->body);
			}
			$con->disconnect();
			unset($con);

			foreach ($arrEntries as $str)
			{
				$str = trim($str);
				if ($str == '') continue;

				$matches = array();
				if (preg_match($re, $str, $matches))
				{
					$str_date = $matches[1];
					//convert str_date to Identity date format
					$str_time = $matches[2];
					$ts = gmmktime(substr($str_time,0,2), substr($str_time,3,2), substr($str_time,6,2), substr($str_date,5,2), substr($str_date,8,2), substr($str_date,0,4));
					$str_date = date($dateFormat, mktime(0, 0, 0, substr($str_date,5,2), substr($str_date,8,2), substr($str_date,0,4)));

					$str_err = $matches[3];
					$str_note = $matches[5];

					if (array_search($str_err, $errlev) === false)
					{
						$str_err = __('unknown');
					}

					$log[] = array(
						'TS' => $ts,
						'DATE' => $str_date,
						'TIME' => $str_time,
						'ERRTYPE' => $str_err,
						'NOTE' => $str_note
					);
				} else {
					if (count($log))
					{
						$log[count($log) - 1]['NOTE'] .= "<br>$str";
					}
				}
			}
		}

		return array_reverse($log);
	}
}
