<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id:$
 * ------------------------------------------------------------------------------
 * This class represents logic for Avatar objects
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2010
 * ------------------------------------------------------------------------------
 */

class Avatar extends Device
{
	public function __construct()
	{
		parent::__construct("avatar", "V", "*");
	}

	/**
	 * create avatar object in the system and assign it to the node
	 *
	 * @param array $attributes list of attributes
	 * @param null $nodeId node id
	 * @throws InvalidArgumentException
	 * @return int
	 */
	public function create(array $attributes, $nodeId = null)
	{
		if (!isset($nodeId))
		{
			throw new InvalidArgumentException("nodeid not defined");
		}

		$obj = parent::create($attributes, $nodeId);

		// specify avatar to node
		DB::query("UPDATE _objs SET node_id = ? WHERE obj = ?;", array($nodeId, $obj));

		// TODO: add to trusted hosts

		// save summarize info about event policies to avatar
		$eventpolicy = new EventPolicy();
		$eventpolicy->savePolicyToAvatar($obj);

		return $obj;
	}

	/**
	 * delete avatar from system
	 *
	 * @param $obj
	 * @param  bool $onlyMark - set deleted = 1 (by default) or delete from DB
	 * @throws Exception
	 * @return bool
	 */
	public function delete($obj, $onlyMark = true)
	{
		$userid = $_SESSION[SESSION_USERID];

		$objectList = $this->getObjects($obj, $userid, null, "D", "C");
		$camera = new Camera();
		foreach ($objectList as $object)
		{
			$camera->delete($object["obj"]);
		}

		$objectList = $this->getObjects($obj, $userid, null, "D", "S");
		$sensor = new SensorSKM();
		foreach ($objectList as $object)
		{
			$sensor->delete($object["obj"]);
		}

		parent::delete($obj);
	}

	/**
	 * get list of objects from avtar
	 *
	 * @param int $obj
	 * @param int|null $userid
	 * @param int|null $roleid
	 * @param string|null $otype
	 * @param string|null $subtype
	 * @param bool $withAttributes
	 * @return array
	 */
	public function getObjects($obj, $userid, $roleid = null, $otype = null, $subtype = null, $withAttributes = false)
	{
		$list = DB::select(
			"SELECT
				_objs.obj, _objs.udid, _objs.name, _objs.description, _objs.location, _objs.otype, _objs.subtype, _objs.protected, objects.permission, objects.credentials
			FROM
				_objs, _obj_attr, getObjects(?, ?, ?, ?) as objects
			WHERE
				deleted = 0
				AND (cast(? as char) IS NULL OR _objs.otype = ?)
				AND (cast(? as char) IS NULL OR _objs.subtype = ?)
				AND _objs.obj = objects.obj
				AND _objs.obj = _obj_attr.obj
				AND _obj_attr.attr = 'AVATARID'
				AND _obj_attr.val = ?;",
			array($userid, $roleid, $otype, $subtype, $otype, $otype, $subtype, $subtype, $obj)
		);

		if ($withAttributes)
		{
			foreach ($list as &$row)
			{
				$row["attributes"] = $this->getAttributes($row["obj"]);
			}
		}

		return $list;
	}

	/**
	 * get avatar id
	 *
	 * @static
	 * @param  string $avatarUNI
	 * @return int|null
	 */
	public function getId($avatarUNI)
	{
		$avatarId = null;

		$list = DB::select("
			SELECT _objs.obj
				FROM _objs, _obj_attr
				WHERE
					_objs.obj = _obj_attr.obj
					AND _objs.otype = 'V'
					AND _objs.subtype = '*'
					AND deleted = 0
					AND _obj_attr.attr = 'UNI'
					AND _obj_attr.val = ?;",
			array($avatarUNI)
		);
		if (count($list) > 0)
		{
			$avatarId = $list[0]["obj"];
		}

		return $avatarId;
	}

	/**
	 * get UNI of node related to avatar
	 *
	 * @static
	 * @param  string $avatarUNI
	 * @return string|null
	 */
	public function getNodeUni($avatarUNI)
	{
		$nodeUNI = null;

		$list = DB::select("
			SELECT node.name as uni
				FROM _objs as node, _objs as avatar, _obj_attr as avatar_attribute
				WHERE
					node.obj = avatar.node_id

					AND avatar.otype = 'V'
					AND avatar.subtype = '*'

					AND node.otype = 'D'
					AND node.subtype = 'N'

					AND node.deleted = 0
					AND avatar.deleted = 0

					AND avatar.obj = avatar_attribute.obj
					AND avatar_attribute.attr = 'UNI'
					AND avatar_attribute.val = ?;",
			array($avatarUNI)
		);
		if (count($list) > 0)
		{
			$nodeUNI = $list[0]["uni"];
		}

		return $nodeUNI;
	}

	public function getNodeId($avatarID)
	{
		$nodeid = null;

		if ($this->isAvatar($avatarID))
		{
			$list = DB::select("SELECT node_id FROM _objs WHERE obj = ?;", array($avatarID));
			if (count($list) > 0)
			{
				$nodeid = $list[0]["node_id"];
			}
		}

		return $nodeid;
	}

	/**
	 * set avatat attributess
	 *
	 * @param int $obj
	 * @param  array $attributes
	 * @throws AuthException
	 */
	public function setAttributes($obj, array $attributes)
	{
		$commandAttributes = array("POWER", "CHANNEL", "ENGINE");
		foreach ($commandAttributes as $attribute)
		{
			if (isset($attributes[$attribute]))
			{
				$droneMobile = new DroneMobile($obj);
				$droneMobile->sendCommand($attribute, $attributes[$attribute]);
			}
		}

		$profileAttributes = array("PROFILE_START", "PROFILE_SET_LIST", "PROFILE_CMD");
		$intersect = array_intersect($profileAttributes, array_keys($attributes));
		if (count($intersect) > 0)
		{
			$user = new User();
			$userid = $_SESSION[SESSION_USERID];

			if (isset($attributes["PROFILE_START"])
				&& !$user->checkObjectCredentials($userid, $obj, "x"))
			{
				throw new AuthException("Not enough rights for changing Avatar profile");
			}

			if ((isset($attributes["PROFILE_SET_LIST"]) || isset($attributes["PROFILE_CMD"]))
				&& !$user->checkObjectCredentials($userid, $obj, "z")
				&& !$user->checkObjectCredentials($userid, $obj, "x")
			)
			{
				throw new AuthException("Not enough rights for changing Avatar profile");
			}
		}

		parent::setAttributes($obj, $attributes);
	}

	/**
	 * check if object is avatar
	 *
	 * @param $obj
	 * @return bool
	 */
	public function isAvatar($obj)
	{
		$result = false;

		$list = DB::select("
			SELECT otype, subtype
				FROM _objs
				WHERE
					deleted = 0
					AND otype = 'V'
					AND subtype = '*'
					AND obj = ?;", array($obj)
		);
		if (count($list) > 0)
		{
			$result = true;
		}

		return $result;
	}

	/**
	 * send attributes by sms to avatar
	 *
	 * @param int $obj
	 * @param array $attributes
	 */
	public function sendAttributesBySMS($obj, array $attributes)
	{
		$time = time();
		$path = APL_VAR_PATH . "/sms/out/$time.$obj";

		$attributes["OBJ"] = $obj;
		$attributes["TELEPHONE_NUMBER"] = $this->getAttribute($obj, "TELEPHONE_NUMBER");

		if (($file = fopen($path, 'w')) !== FALSE)
		{
			foreach ($attributes as $key => $value)
			{
				fwrite($file, "$key=$value\n");
			}
			fclose($file);
		}
	}

	/**
	 * @param int $obj
	 *
	 * @return string
	 */
	public function getStatus($obj)
	{
		//TODO: check logic

		/*
		// Avatar status (ONLINE/OFFLINE) will be calculated based on time difference between now and STAT_LAST_REPORT,
		// then compared against "AVATAR status reporting interval" plus 5 seconds.

		$status = "ONLINE";
		$STAT_LAST_REPORT = intval($this->getAttribute($obj, "STAT_LAST_REPORT"));
		if (time() - $STAT_LAST_REPORT > $STAT_LAST_REPORT + 5)
		{
			$status = "OFFLINE";
		}
		*/

		$statusDescription = array(
			"" => "",
			"ONLINE" => "",
			"OFFLINE" => __("OFFLINE: The application is not running (OFFLINE state). Please contact your System Administrator.")
		);

		$command = "VERSIONER_PERL_VERSION=" . $_SERVER["VERSIONER_PERL_VERSION"] . " APL=" . $_SERVER["APL"] . " PERL5LIB=" . $_SERVER["PERL5LIB"] . " " . APL_PATH . "/av/bin/avatar_status";
		$content = exec(escapeshellcmd($command));
		$avatarStatus = json_decode($content, true);

		if (!isset($avatarStatus[$obj]))
		{
			throw new InvalidArgumentException("Incorrect avatar");
		}

		$status = $avatarStatus[$obj]["status"];

		return array(
			"status" => $status,
			"description" => isset($statusDescription[$status]) ? $statusDescription[$status] : "",
			"reasons" => array()
		);
	}

	/*
	 * get list of avatar external commands listed in configuration file
	 *
	 * @static
	 * @return array
	 */
	public function getExternalCommands()
	{
		$list = array();
		$content = file_get_contents(APL_PATH . "/etc/avatar-external-commands.conf");
		$lines = preg_split("/\n/", $content);
		foreach ($lines as $line)
		{
			$line = trim($line);
			if ($line == ''
				|| $line[0] == '#'
				|| strpos($line, "=") === false)
			{
				continue;
			}
			$parts = preg_split("/=/", $line);
			$list[$parts[0]] = $parts[1];
		}

		return $list;
	}

	/**
	 * run external avatar command
	 * command specification should be in /opt/sarch/etc/avatar-external-commands.conf
	 *
	 * @param int $obj
	 * @param string $commandName
	 *
	 * @return array
	 */
	public function execExternalCommand($obj, $commandName)
	{
		$list = array(
			"exit_status" => 0,
			"message" => ""
		);
		$commands = $this->getExternalCommands();
		if (!isset($commands[$commandName]))
		{
			$list["exit_status"] = -1;
			$list["message"] = __("Command not found");
		} else {
			$last = exec("/usr/bin/timeout -k 1 5 " . escapeshellcmd($commands[$commandName] . " $obj") . " 2>&1", $out, $status);
			$list["exit_status"] = $status;
			if ($status == 124) // Killed with TERM, timeout expired
			{
				$list["message"] = __("Timeout expired");
			} else
			if ($status == 137) // Killed with KILL if TERM didn't stop script
			{
				$list["message"] = __("Timeout expired but script was still running. Killed by KILL signal");
			} else
			if ($status != 0)
			{
				$list["message"] = join("\n", $out);
			}
		}

		return $list;
	}
}
