<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: Audit.php 31894 2015-03-09 14:49:38Z atsybulnik $
 * ------------------------------------------------------------------------------
 * DB implementation of logger interface
 * This class provides simple logging functionality
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2010
 * ------------------------------------------------------------------------------
 */

class Audit
{
	public function __construct() {}

	/**
	 * Set of functions that generate predefned "blocks" for audit record details
	 *
	 * @param string $text
	 * @return array
	 */
	public static function textEntity($text) {
		return array('@T', $text);
	}

	public static function attrValueEntity($attr, $value) {
		return array('@AV', $attr, $value);
	}

	public static function attrChangeEntity($attr, $oldValue, $newValue) {
		return array('@AC', $attr, $oldValue, $newValue);
	}


	/**
	 * addRecord - insert record into audit table
	 * Example:
	 * <code>
	 * Audit::addRecord(2,
	 *    array('DIGEST_Param1',
	 *    Audit::attrValueEntity('ATTRNAME','VALUE'),
	 *    Audit::attrChangeEntity('ATTRNAME','OLDVALUE','NEWVALUE'),
	 *    'DIGEST_Param2',
	 *    Audit::textEntity('Some text')),
	 *    101, null);
	 * </code>
	 *
	 * @static
	 * @param int      $category audit event category - one of the value defined in auditcategory table
	 * @param array    $paramArray audit record parameters
	 * @param int|null $objid object id or event id (optional)
	 * @param int|null $userid user objid (optional)
	 * @return bool
	 */
	public static function addRecord($category, array $paramArray, $objid = null, $userid = null)
	{
		return self::_addRecordImpl($category, $objid, $userid, $paramArray, 0);
	}

	/**
	 * addRecordVArg - insert record into audit table (alternative version)
	 *
	 * Example:
	 * <code>
	 *        Audit::addRecordVArg(2, 101, null, 'DIGEST_Param1',
	 *                Audit::attrValueEntity('ATTRNAME','VALUE'),
	 *                Audit::attrChangeEntity('ATTRNAME','OLDVALUE','NEWVALUE'),
	 *                'DIGEST_Param2',
	 *                Audit::textEntity('Some text'));
	 * </code>
	 *
	 * @param int       $category audit event category - one of the value defined in auditcategory table
	 * @param int|null  $objid object id or event id (optional)
	 * @param int|null  $userid user objid (optional)
	 * @param ...  all additional arguments are treated as audit record parameters
	 * @return bool
	 */
	public static function addRecordVArg($category, $objid = null, $userid = null)
	{
		if (func_num_args() > 3)
		{
			return self::_addRecordImpl($category, $objid, $userid, func_get_args(), 3);
		} else {
			return self::_addRecordImpl($category, $objid, $userid, array(), 0);
		}
	}

	private static function _addRecordImpl($category, $objid, $userid, $paramArray, $paramStartIdx)
	{
		$result = true;

		//
		// basic verification
		//
		if (!isset($category)) {
			throw new Exception("Audit category is not set");
		}

		if (!isset($userid) && isset($_SESSION[SESSION_USERID])) $userid = $_SESSION[SESSION_USERID];

		//
		// assemble parameters string
		//
		$parameters = '';
		$paramCount = count($paramArray) - $paramStartIdx;
		if ($paramCount > 0) {
			$details = '';

			for($i = $paramStartIdx; $i < $paramStartIdx + $paramCount; $i++) {
				$val = $paramArray[$i];
				if (is_array($val)) {
					if (!empty($details)) $details .= ',';
					$details .= json_encode($val);
				} else {
					if (!empty($parameters)) $parameters .= ',';
					$parameters .= json_encode($val);
				}
			}

			if(!empty($details))
				$parameters = '[[' . $parameters .'],[' . $details . ']]';
			elseif (!empty($parameters))
				$parameters = '[[' . $parameters .']]';
			else
				$parameters = '[]';
		}

		//
		// insert record in DB
		//
		try	{
			DB::query(
				"INSERT INTO audit(category, userid, objid, parameters) VALUES (?, ?, ?, ?);",
				array($category, $userid, $objid, $parameters));
		} catch (Exception $e) {
			error_log('Failed to insert record into audit table: ' . $e->getMessage());
			$result = false;
		}

		return $result;
	}

	/**
	 * getRecords
	 *
	 * @param null      $from
	 * @param null      $to
	 * @param array     $categories
	 * @param int|null  $userid
	 * @param int|null  $objid
	 * @param null      $msgFilter
	 * @param int       $pagesize
	 * @param int       $pagenum first page's ID is 0
	 * @param bool      $countRows if true function returns total number of rows matching provided filter
	 * @param bool      $includeDigest if true function returns category/digest array for all categories used in
	 *                         audit records data set
	 * @throws Exception
	 * @return array array containing audit records data set ('data' nested array), category/digest pairs ('categories')
	 *          and total number of rows ('total')
	 */
	public static function getRecords(
			$from = null, $to = null, array $categories = array(), $userid = null, $objid = null, $msgFilter = null,
			$pagesize = 0, $pagenum = 0, $countRows = false, $includeDigest = false)
	{
		//
		// assemble where clause
		//
		$where = '';
		$params = array();

		if (isset($from)) {
			$where .= ' a.date >= to_timestamp(?)';
			$params[] = $from; 	// TODO
		}

		if (isset($to)) {
			if (!empty($where)) $where .= ' AND';
			$where .= ' a.date <= to_timestamp(?)';
			$params[] = $to; 	// TODO
		}

		if (count($categories) > 0) {
			// select distinc low level audit categories
			$condition = ' IN (';
			if (count($categories) > 1) {
				foreach($categories as $idx => $category) {
					if ($idx > 0) $condition .= ',';
					$condition .= $category;
				}
				$condition .= ')';
			} else {
				$condition = ' = ' . $categories[0];
			}

			$rSet = DB::select('SELECT DISTINCT category FROM auditcategory WHERE parent' . $condition .
			 ' OR parent IS NOT NULL AND category' . $condition);

			// prepare where clause for main statement
			if (count($rSet) < 1) {
				throw new Exception("Invalid category(ies) provided");
			} else {
				if (!empty($where)) $where .= ' AND';

 				if (count($rSet) == 1) {
					$where .= ' a.category = ?';
					$params[] = $rSet[0]['category'];
				} else {
					$where .= ' a.category IN (';
					foreach($rSet as $idx => $val) {
						if ($idx > 0) $where .= ',';
						$where .= $val['category'];
					}
					$where .= ')';
				}
			}
		}

		if (isset($userid)) {
			if (!empty($where)) $where .= ' AND';
			$where .= ' a.userid = ?';
			$params[] = $userid;
		}


		if (isset($objid)) {
			if (!empty($where)) $where .= ' AND';
			$where .= ' a.objid = ?';
			$params[] = $objid;
		}

		if (isset($msgFilter) && !empty($msgFilter)) {
			// search within locallized digest message
			$catSpecFilter = '';
			$catResultSet = DB::select("SELECT category, digest FROM auditcategory WHERE length(digest)>0");
			for($i = 0; $i < count($catResultSet); $i++) {
				if (stristr(__($catResultSet[$i]['digest']), $msgFilter)) {
					if (empty($catSpecFilter)) {
						$catSpecFilter = $catResultSet[$i]['category'];
					} else {
						$catSpecFilter .= ',' . $catResultSet[$i]['category'];
					}
				}
			}

			// put all together
			if (!empty($where)) $where .= ' AND';
			if (empty($catSpecFilter)) {
				$where .= " a.parameters ILIKE ?";
			} else {
				$where .= " (a.parameters ILIKE ? OR a.category IN (" . $catSpecFilter ."))";
			}
			$params[] = '%' .$msgFilter . '%';
		}


		if (!empty($where)) $where = ' WHERE' . $where;
		$resultSet = array();

		// count rows if requested
		$countAllRows = -1;
		if ($countRows)	{
			$stmt = "SELECT count(*) as total FROM audit a" . $where;
			$rset = DB::select($stmt, $params);
			$countAllRows = $rset[0]['total'];
			$resultSet['total'] = $countAllRows;
		}

		// get rows
		if ($countAllRows != 0) {
			$stmt = "SELECT TRUNC(EXTRACT(EPOCH FROM a.date)) as date, a.category, a.parameters,
							a.userid, u.name as uname,
   	    	                a.objid, o.name as objname, o.otype, o.subtype as osubtype, COALESCE(o.udid, '') as udid
            	       FROM audit a
					   LEFT JOIN _objs u ON u.obj=a.userid
					   LEFT JOIN _objs o ON o.obj=a.objid AND NOT a.category BETWEEN 16 AND 24
					" . $where . ' ORDER BY a.date DESC';


			if (!empty($pagesize)) {
				$stmt .= ' LIMIT ?';
				$params[] = $pagesize;
				if (!empty($pagenum)) {
					$stmt .= ' OFFSET ?';
					$params[] = $pagenum * $pagesize;
				}
			}

			$resultSet['data'] = DB::select($stmt, $params);

			// include message digest if requested
			if ($includeDigest && !empty($resultSet['data'])) {
				// get unique category list
				$categoriesList = array();
				foreach($resultSet['data'] as $idx => $row) {
					$categoriesList[$row['category']] = $row['category'];
				}
				// assemble SQL statement
				$inList = '';
				foreach($categoriesList as $idx => $row) {
					if (!empty($inList) > 0) $inList .= ',';
					$inList .= $idx;
				}
				$stmt = 'SELECT category, digest FROM auditcategory WHERE category IN(' . $inList. ') ORDER BY category';
				$resultSet['caterories'] = DB::select($stmt);
			}
		} else {
			$resultSet['data'] = array();
		}

		return $resultSet;
	}


	/*
	 * Provide list of available audit categories
	 */
	public static function getCategories()
	{
		$stmt = "SELECT category, parent, name, digest FROM auditcategory";
		$resultSet = array();
		$resultSet['categories'] = DB::select($stmt);
		return $resultSet;
	}
}
