<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id:$
 * ------------------------------------------------------------------------------
 * This class represents logic for Announce objects
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2014
 * ------------------------------------------------------------------------------
 */

class Announce
{
	/**
	 *
	 * @param int $userid
	 * @param int $eventid
	 * @param null $toRoleId
	 * @param null $toUserId
	 * @param string $note
	 * @throws DBException
	 * @throws StompException
	 */
	public function create($userid, $eventid, $toRoleId = null, $toUserId = null, $note = "")
	{
		if ($note != "")
		{
			$noteList = array();
			$list = DB::select("SELECT note FROM event WHERE eventid = ?;", array($eventid));
			if (count($list) > 0)
			{
				$noteField = $list[0]["note"];
				if (isset($noteField) && $noteField != "")
				{
					$noteList = json_decode($noteField, true);
				}
			}

			$user = new User();

			$noteList[] = array(
				"user" => $user->getName($userid),
				"timestamp" => time(),
				"note" => $note
			);

			DB::query(
				"UPDATE event SET note = ? WHERE eventid = ?;",
				array(json_encode($noteList), $eventid)
			);
		}

		$AMQConf = Node::getAMQConf();
		$con = new Stomp($AMQConf["stomp.url"]);
		$con->connect();
		$message = array(
			"error" => "",
			"announce" => array(
				"add" => array("eventid" => $eventid, "note" => $note)
			)
		);

		$userList = array();
		if (isset($toRoleId))
		{
			$role = new Role();
			$userList = $role->getUsers($toRoleId);
		} else {
			$userList[] = array("obj" => $toUserId);
		}
		foreach($userList as $user)
		{
			$announce_to_userid = $user["obj"];

			$list = DB::select("SELECT * FROM eventannounce WHERE eventid = ? AND announce_to_userid = ?;", array($eventid, $announce_to_userid));
			$result = 1;
			if (count($list) == 0)
			{
				$result = DB::query(
					"INSERT INTO eventannounce(eventid, announced_by_userid, announce_to_userid) VALUES(?, ?, ?);",
					array($eventid, $userid, $announce_to_userid)
				);
			}

			if ($result > 0)
			{
				$sessionManager = new SessionManager();
				$sessionList = $sessionManager->getList($announce_to_userid);

				foreach ($sessionList as $sessionId => $session)
				{
					$destination = "/topic/event.announce." . $sessionId;
					$result = $con->send($destination, json_encode($message), array('persistent' => 'true', 'slow_consumer_policy' => 'queue'));
				}
			}
		}
	}

	/**
	 *
	 * @param int $userid
	 * @param array $eventList
	 * @throws DBException
	 * @throws StompException
	 */
	public function markAsRead($userid, $eventList)
	{
		$AMQConf = Node::getAMQConf();
		$con = new Stomp($AMQConf["stomp.url"]);
		$con->connect();

		foreach ($eventList as $eventid)
		{
			$result = DB::query("DELETE FROM eventannounce WHERE eventid = ? AND announce_to_userid = ?;", array($eventid, $userid));

			if ($result > 0)
			{
				$sessionManager = new SessionManager();
				$sessionList = $sessionManager->getList($userid);

				foreach ($sessionList as $sessionId => $session)
				{
					$message = array(
						"error" => "",
						"announce" => array(
							"remove" => array("eventid" => $eventid)
						)
					);

					$destination = "/topic/event.announce." . $sessionId;
					$result = $con->send($destination, json_encode($message), array('persistent' => 'true', 'slow_consumer_policy' => 'queue'));
				}
			}
		}
	}

	/**
	 *
	 * @param int $userid
	 * @return array
	 */
	public function getList($userid)
	{
		$query = "SELECT e.eventid as id, e.objid as objId, e.eventsource as sourceId, e.eventtype, e.priority, e.workflow, e.state,
  e.message as msg, e.note, e.lifespan,
  extract(epoch from e.utc_when at time zone 'UTC') as utc_when, 
  extract(epoch from e.utc_from at time zone 'UTC') as utc_from, 
  extract(epoch from e.utc_to at time zone 'UTC') as utc_to,
  extract(epoch from e.updated_at at time zone 'UTC') as updated_at,
  (SELECT oa.val from _obj_attr oa WHERE oa.obj = q.announced_by_userid AND oa.attr = 'NAME') as announcedby,
  s.actions, 
  ARRAY(SELECT eventtag FROM eventstatetag p where p.eventstate=e.state) as tags, 
  ARRAY(SELECT o.obj||'~^~#~'||otype||'~^~#~'||subtype|| '~^~#~' || COALESCE(b.val , 'GMT')||'~^~#~'||COALESCE(name,'')
    FROM eventwitness w
      INNER JOIN _objs o ON o.obj=w.objid
      LEFT OUTER JOIN _obj_attr b ON b.obj=o.obj AND b.attr='TIME_ZONE'
    WHERE w.eventid=e.eventid) AS witnesses,
  (SELECT ARRAY(SELECT name||'='||value FROM eventproperty p WHERE p.eventid=e.eventid)) as properties,
	COALESCE(otz.val, 'GMT') as otzone, os.name as oname, os.otype as otype, os.subtype as subtype
FROM
  (SELECT e1.eventid, ea1.announced_by_userid
    FROM event e1 
    INNER JOIN eventannounce ea1 ON ea1.eventid = e1.eventid
    WHERE ea1.announce_to_userid = ?) q 
INNER JOIN event e ON e.eventid = q.eventid 
INNER JOIN eventstate s ON s.eventstate = e.state 
INNER JOIN _objs os ON os.obj=e.objid 
LEFT OUTER JOIN _obj_attr otz ON otz.obj=os.obj AND otz.attr='TIME_ZONE';";

		$list = DB::select(
			$query,
			array($userid)
		);
		
		// format list for eventlog
		for($i=0;$i<count($list);$i++){
			$list[$i] = Announce::formatEventForEventLogGUI($list[$i]);
		}

		return $list;
	}
	
	/**
	 * Get event detailed information
	 * @param int $eventid
	 * @return array
	 */
	public function getEventInfo($eventid)
	{
		$query = "SELECT e.eventid as id, e.objid as objId, e.eventsource as sourceId, e.eventtype, e.priority, e.workflow, e.state,
  e.message as msg, e.note, e.lifespan,
  extract(epoch from e.utc_when at time zone 'UTC') as utc_when, 
  extract(epoch from e.utc_from at time zone 'UTC') as utc_from, 
  extract(epoch from e.utc_to at time zone 'UTC') as utc_to,
  extract(epoch from e.updated_at at time zone 'UTC') as updated_at,
  (SELECT oa.val from _obj_attr oa WHERE oa.obj = ea2.announced_by_userid AND oa.attr = 'NAME') as announcedby,
  s.actions, 
  ARRAY(SELECT eventtag FROM eventstatetag p where p.eventstate=e.state) as tags, 
  ARRAY(SELECT o.obj||'~^~#~'||otype||'~^~#~'||subtype|| '~^~#~' || COALESCE(b.val , 'GMT')||'~^~#~'||COALESCE(name,'')
    FROM eventwitness w
      INNER JOIN _objs o ON o.obj=w.objid
      LEFT OUTER JOIN _obj_attr b ON b.obj=o.obj AND b.attr='TIME_ZONE'
    WHERE w.eventid=e.eventid) AS witnesses,
  (SELECT ARRAY(SELECT name||'='||value FROM eventproperty p WHERE p.eventid=e.eventid)) as properties,
	COALESCE(otz.val, 'GMT') as otzone, os.name as oname, os.otype as otype, os.subtype as subtype
FROM event e 
INNER JOIN eventstate s ON s.eventstate = e.state 
INNER JOIN _objs os ON os.obj=e.objid 
INNER JOIN eventannounce ea2 ON ea2.eventid = e.eventid
LEFT OUTER JOIN _obj_attr otz ON otz.obj=os.obj AND otz.attr='TIME_ZONE'
WHERE e.eventid = ?;";

		$event = DB::select(
			$query,
			array($eventid)
		);
		
		$event = Announce::formatEventForEventLogGUI($event[0]);

		return $event;
	}

/**
	 * Get event objid
	 * @param int $eventid
	 * @return int
	 */
	public function getEventObjId($eventid)
	{
		$query = "SELECT e.objid as objid FROM event e WHERE e.eventid = ?;";

		$event = DB::select(
			$query,
			array($eventid)
		);

		return $event[0]['objid'];
	}
	
	/**
	 * Format event data for GUI
	 * @param array $data
	 * @return array
	 */
	public static function formatEventForEventLogGUI($data)
	{
		// properties
		$tmpProps = explode(",", ltrim(rtrim($data['properties'], "}"),"{"));
		$newProps = array();
		for($j=0;$j<count($tmpProps);$j++){
			$curProp = explode("=",$tmpProps[$j]);
			$newProps[] = array("name"=> $curProp[0], "value" => $curProp[1]);
		}		
		$data['properties'] = $newProps;	
			
		// witnesses
		$tmpWitnesses = explode('","', ltrim(rtrim($data['witnesses'], '"}'),'{"'));
		$newWitnesses = array();
		for($j=0;$j<count($tmpWitnesses);$j++){
			$curWitness = explode("~^~#~",$tmpWitnesses[$j]);
			$newWitnesses[] = array("objid" => $curWitness[0]);
		}		
		$data['witnesses'] = $newWitnesses;
			
		// tags
		$data['tags'] = explode(",", ltrim(rtrim($data['tags'], "}"),"{"));
			
		// actions
		if($data['actions'] == null){
			$data['actions'] = array();
		}
			
		//utc
		$data['utc'] = (object) array(
			'when' => $data['utc_when'], 
			'from' => $data['utc_from'], 
			'to' => $data['utc_to'],
			'updated' => $data['updated_at']);
		unset($data['utc_when']);
		unset($data['utc_from']);
		unset($data['utc_to']);
		unset($data['updated_at']);
			
		// obj
		$data['obj'] = (object) array(
			'TimeDST' => $data['otzone'], 
			'name' => $data['oname'], 
			'type' => $data['otype'],
			'subtype' => $data['subtype'],
			'objid' => $data['objid']);
		unset($data['otzone']);
		unset($data['oname']);
		unset($data['otype']);
		unset($data['subtype']);
			
		$data['cachestate'] = "A";
		$data['objId'] = $data['objid'];
		$data['sourceId'] = $data['sourceid'];
		unset($data['sourceid']);
		
		return $data;
	}
	
	/**
	 * Count announced events for user
	 * @param int $userid
	 * @return array
	 */
	public function countEvents($userid)
	{
		$result = DB::select(
			"SELECT COUNT(event.eventid) as amount
				FROM eventannounce, event
				WHERE eventannounce.eventid = event.eventid AND announce_to_userid = ?;",
			array($userid)
		);
		
		return $result[0]['amount'];
	}
}
