<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: API_POST.php 33538 2016-01-27 00:39:59Z astarostin $
 *
 * This class represents HTTP API for setting data.
 *
 * Methods of this class must only for external purposes, parameters order of methods not garanted.
 * All methods must be called using Proxy class.
 *
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions LLC 2006
 * ------------------------------------------------------------------------------
 */

class API_POST extends API
{
	/**
	 * add new object to system
	 *
	 * @docBlock
	 * # Object management
	 * ## Add object to Stratus
	 * #### Request
	 * > POST /api/call/addObject
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"error":"","obj":106,"code":200}
	 *
	 * @static
	 * @param  string $type type of added object. Supported types are: camera
	 * @param  array $attributes JSONobject list of default values for new created object
	 * @param  int|null $setid
	 * @param  int|null $nodeid
	 * @throws InvalidArgumentException
	 * @throws Exception
	 * @return array
	 */
	public static function addObject($type, array $attributes, $setid = null, $nodeid = null)
	{
		$userId = $_SESSION[SESSION_USERID];
		$user = new User();
		if ($type == "user" && !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "u") && !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "U"))
		{
			throw new AuthException("Not enough rights");
		}
		if (in_array($type, array("role", "set")) && !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "U"))
		{
			throw new AuthException("Not enough rights");
		}
		if (in_array($type, array("camera", "sensor", "relay", "gateway")) && !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "F"))
		{
			throw new AuthException("Not enough rights");
		}

		$obj = null;

		$set = new Set();
		if (isset($setid) && !$set->isExists($setid))
		{
			throw new InvalidArgumentException(sprintf("set with id=%s not exists", $setid));
		}

		$object = Factory::getByType($type);

		if (isset($attributes['UDID']))
		{
			$attributes['UDID'] = trim(strtoupper($attributes['UDID']));
			if (!empty($attributes['UDID']))
			{
				if ($object->checkUniqueness(null, 'UDID', $attributes['UDID']))
				{
					throw new InvalidArgumentException(__("UDID attribute is not unique"));
				}
			}
		}

		DB::query("BEGIN;");
		switch ($type)
		{
			case "camera":
				$node = new Node();
				$nodeUNI = $node->getUNI($nodeid);

				// check if nodeid is Avatar id
				if (!isset($nodeUNI))
				{
					$avatar = new Avatar();

					if (!$avatar->isAvatar($nodeid))
					{
						throw new Exception(__("Incorrect 'nodeid' parameter"));
					}

					$avatarId = $nodeid;
					if (isset($avatarId))
					{
						$attributes["AVATARID"] = $avatarId;

						$avatarUNI = $avatar->getAttribute($avatarId, "UNI");
						$nodeUNI = $avatar->getNodeUni($avatarUNI);
						if (!isset($nodeUNI))
						{
							throw new InvalidArgumentException(__("Cannot get node UNI related to specified avatar UNI"));
						}
					}
				}

				$obj = $object->create($attributes, $nodeUNI);
			break;
			default:
				$obj = $object->create($attributes);
		}

		// add new created object to not protected set
		if (isset($obj) && isset($setid) && !$set->isProtected($setid))
		{
			$set->add($setid, $obj);
		}

		$result["obj"] = intval($obj);

		DB::query("COMMIT;");

		return $result;
	}

	/**
	 * remove camera from system
	 * this method only for support with current API
	 *
	 * @deprecated 2.6.3
	 * @static
	 *
	 * @param int $obj
	 *
	 * @return array
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 */
	public static function deleteCamera($obj)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		trigger_error('Function ' . __METHOD__ . ' is deprecated');

		DB::query("BEGIN;");
		$object = new Obj();
		$type = $object->getType($obj);

		if ($type == "camera")
		{
			$camera = new Camera();
			$camera->delete($obj);
		} else {
			throw new InvalidArgumentException(sprintf("unknown type=%s", $type));
		}
		DB::query("COMMIT;");

		ObjectManager::send();

		return array(
			"deprecated" => "true"
		);
	}

	/**
	 * delete object (objects) from system
	 *
	 * @docBlock
	 * # Object management
	 * ## Delete object from Stratus
	 * #### Request
	 * > POST /api/call/deleteObject
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"error":"","code":200}
	 *
	 * @static
	 *
	 * @param array $objList list of obj id's for delete
	 *
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 */
	public static function deleteObject(array $objList)
	{
		$userId = $_SESSION[SESSION_USERID];
		$user = new User();
		$object = new Obj();
		foreach ($objList as $obj)
		{
			$type = $object->getType($obj);
			if ($type == "user" && !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "u") && !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "U"))
			{
				throw new AuthException("Not enough rights");
			}
			if (in_array($type, array("role", "set")) && !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "U"))
			{
				throw new AuthException("Not enough rights");
			}
			if (in_array($type, array("camera", "sensor", "relay", "gateway")) && !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "F"))
			{
				throw new AuthException("Not enough rights");
			}
		}

		DB::query("BEGIN;");
		foreach ($objList as $obj)
		{
			$object = Factory::getByObj($obj);

			if (!isset($object))
				throw new InvalidArgumentException(sprintf("unknown object=%s", $obj));

			if (!$object->isProtected($obj))
			{
				$object->delete($obj);
			}
		}
		DB::query("COMMIT;");

		ObjectManager::send();
	}

	/**
	 * set attributes for object
	 *
	 * @docBlock
	 * # Object management
	 * ## Change object configuration
	 * #### Request
	 * > POST /api/call/setAttributes
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"error":"","code":200}
	 *
	 * @static
	 * @param  int $obj object id
	 * @param  array $attributes JSONobject list of attributes
	 * @throws InvalidArgumentException
	 * @throws Exception
	 * @return void
	 */
	public static function setAttributes($obj, array $attributes)
	{
		$userId = $_SESSION[SESSION_USERID];
		$user = new User();
		$object = new Obj();
		$type = $object->getType($obj);
		if ($type == "user" && !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "u") && !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "U"))
		{
			throw new AuthException("Not enough rights");
		}
		if (in_array($type, array("role", "set")) && !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "U"))
		{
			throw new AuthException("Not enough rights");
		}
		if (in_array($type, array("camera", "sensor", "relay", "gateway"))
			&& !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "F")
			&& !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "f")
			&& !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "R")
			&& !$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "E"))
		{
			throw new AuthException("Not enough rights");
		}

		$object = Factory::getByObj($obj);
		if (!isset($object))
			throw new InvalidArgumentException(sprintf("unknown object=%s", $obj));

		if (isset($attributes['UDID']) && trim($attributes['UDID'])!= '')
		{
			if ($object->checkUniqueness($obj, 'UDID', strtoupper($attributes['UDID']), true))
				throw new InvalidArgumentException(__("UDID attribute is not unique."));
		}

		DB::query("BEGIN;");
		$object->setAttributes($obj, $attributes);
		DB::query("COMMIT;");

		ObjectManager::send();
	}

	/**
	 * set attributes for identity
	 *
	 * @static
	 *
	 * @param array $attributes JSONobject
	 *
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 */
	public static function setIdentityAttributes(array $attributes)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		DB::query("BEGIN;");
		Identity::setAttributes($attributes);
		DB::query("COMMIT;");
	}

	/**
	 * delete blocks
	 *
	 * @static
	 *
	 * @param int $obj
	 * @param array $blockIdList
	 *
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 */
	public static function deleteBlocks($obj, array $blockIdList)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$object = Factory::getByObj($obj);
		if (!isset($object))
			throw new InvalidArgumentException(sprintf("unknown object=%s", $obj));

		DB::query("BEGIN;");
		foreach ($blockIdList as $blockId)
		{
			$object->deleteBlock($obj, $blockId);
		}
		DB::query("COMMIT;");
	}

	/**
	 * create or update (if exist) block
	 *
	 * @static
	 *
	 * @param int $obj
	 * @param int|null $blockId
	 * @param string $mime
	 * @param string $description
	 * @param bool $fromPDFToPNG
	 *
	 * @return array
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 */
	public static function setBlock($obj, $blockId = null, $mime = "", $description = "", $fromPDFToPNG = false)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$fromPDFToPNG = Template::boolVal($fromPDFToPNG);

		$error = $_FILES['block']['error'];
		$tmp_name = $_FILES['block']['tmp_name'];
		$name = $_FILES['block']['name'];
		if ($blockId == 'null')
		{
			$blockId = null;
		}

		if (!isset($mime) || $mime == "")
			$mime = $_FILES['block']['type'];

		$errorMessage = "";
		if (!empty($error))
		{
			switch ($error)
			{
				case UPLOAD_ERR_OK:
					$errorMessage = __('The uploaded file exceeds the upload_max_filesize directive in php.ini');
					break;
				case UPLOAD_ERR_INI_SIZE:
					$errorMessage = __('The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form');
					break;
				case UPLOAD_ERR_PARTIAL:
					$errorMessage = __('The uploaded file was only partially uploaded');
					break;
				case UPLOAD_ERR_NO_FILE:
					$errorMessage = __('No file was uploaded');
					break;
				case UPLOAD_ERR_NO_TMP_DIR:
					$errorMessage = __('Missing a temporary folder');
					break;
				case UPLOAD_ERR_CANT_WRITE:
					$errorMessage = __('Failed to write file to disk');
					break;
				case UPLOAD_ERR_EXTENSION:
					$errorMessage = __('File upload stopped by extension');
					break;
				case '999':
				default:
					$errorMessage = __('No error code available');
			}
		} else
		if (empty($tmp_name) || $tmp_name == 'none')
		{
			$errorMessage = __('No file was uploaded');
		}

		$fileToLoad = $tmp_name;
		if ($fromPDFToPNG)
		{
			$tmp_name_2 = $tmp_name . "_2";
			$fileToLoad = $tmp_name_2;
			$image = new Imagick();
			$image->setResolution(300, 300);
			$image->readImage($tmp_name . '[0]');
			$image->setImageFormat("png");
			$image->writeImage($tmp_name_2);
			$name = "$name.png";
			$mime = $image->getImageMimeType();
		}

		if ($errorMessage != "")
			throw new InvalidArgumentException($errorMessage);

		DB::query("BEGIN;");
		$object = Factory::getByObj($obj);
		if (!isset($object))
			throw new InvalidArgumentException(sprintf("unknown object=%s", $obj));

		$url = null;

		$value = file_get_contents($fileToLoad);

		$result = $object->setBlock($obj, $blockId, $value, $mime, $name, $description);
		if (isset($result))
			$url = $object->getBlockURL($obj, $result);
		DB::query("COMMIT;");

		// for security reason, we force to remove all uploaded files
		if (is_file($tmp_name))
		{
			unlink($tmp_name);
		}
		if (is_file($fileToLoad))
		{
			unlink($fileToLoad);
		}

		return array(
			"url" => $url,
			"blockId" => $result,
			"test" => $_FILES['block']
		);
	}

	/**
	 * add or remove objects from role, set
	 *
	 * @static
	 * @param  int $parentobj
	 * @param  array $addList list of objects obj for adding to role
	 * @param  array $removeList list of objects obj for removing from role
	 * @throws Exception
	 * @throws InvalidArgumentException
	 */
	public static function changeObjectList($parentobj, array $addList = array(), array $removeList = array())
	{
		if (!isset($addList) && !isset($removeList))
		{
			throw new InvalidArgumentException("addList or removeList should be defined");
		}

		$userId = $_SESSION[SESSION_USERID];
		$user = new User();
		if (!$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "u"))
		{
			throw new AuthException("Not enough rights");
		}

		DB::query("BEGIN;");
		$object = new Obj();
		$type = $object->getType($parentobj);
		switch ($type)
		{
			case "set":
				$set = new Set();

				$set->addList($parentobj, $addList, true);
				$set->removeList($parentobj, $removeList, true);
			break;
			case "role":
				$role = new Role();

				$role->addList($parentobj, $addList);
				$role->removeList($parentobj, $removeList);
			break;
		}
		DB::query("COMMIT;");
	}

	/**
	 * set permission on Set->Role link
	 * @static
	 *
	 * @param  int $roleid
	 * @param  int $setid
	 * @param  string $permission
	 *
	 * @throws AuthException
	 */
	public static function setPermission($roleid, $setid, $permission)
	{
		$userId = $_SESSION[SESSION_USERID];
		$user = new User();
		if (!$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "U"))
		{
			throw new AuthException("Not enough rights");
		}

		DB::query("BEGIN;");
		$role = new Role();
		$role->setPermission($roleid, $setid, $permission);
		DB::query("COMMIT;");
	}

	/**
	 * set special credentials on Set->Role link
	 * @static
	 *
	 * @param  int $roleid
	 * @param  int $setid
	 * @param  string $credentials
	 *
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 */
	public static function setSpecialCredentials($roleid, $setid, $credentials)
	{
		$userId = $_SESSION[SESSION_USERID];
		$user = new User();
		if (!$user->checkObjectCredentials($userId, Obj::CONTROL_PANEL, "U"))
		{
			throw new AuthException("Not enough rights");
		}

		if ($credentials != ""){
			$allowedCredentials = array("I", "u", "U", "f", "F", "R", "E");
			$aCredentials = str_split($credentials);

			for($i = 0; $i < count($aCredentials); $i++){
				if(!in_array($aCredentials[$i], $allowedCredentials)){
					throw new InvalidArgumentException(sprintf("unknown special credentials"));
				}
			}
		}

		DB::query("BEGIN;");
		$role = new Role();
		$role->setSpecialCredentials($roleid, $setid, $credentials);
		DB::query("COMMIT;");
	}

	/**
	 * internal SKM function, not for use from API
	 * create new list of depended objects (relay, sensor)
	 *
	 * @static
	 *
	 * @param  int $cameraid
	 * @param  array $attributes JSONobject
	 *
	 * @throws AuthException
	 * @throws Exception
	 */
	public static function createCameraAssociatedDevices($cameraid, array $attributes)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$parameters = API_GET::getCameraModel($attributes, true);

		$camera = new Camera();

		$a["getCameraModelByTicket"]=$parameters;
		$aedescr = $parameters["attributes"]["AE_DESCR"];

		$userid = $_SESSION[SESSION_USERID];
		$cattr = $parameters['attributes'];
		$list = DB::select("SELECT node_id from _objs where obj=?;",array($cameraid));
		$nodeid = $list[0]['node_id'];

		if ($aedescr && $aedescr !== ''){
			$set = new Set();
			$descr = json_decode($aedescr);
			$created = array();
			$user = new User();
			$sets =  $user->getSets($userid, null, $cameraid);

			DB::query("BEGIN;");
			foreach ($descr as $type=>$objtype){
				$a[$type] = $objtype;
				if (isset($objtype->PORT_NUMBERS)){
					if ($type == 'relay'){ $device = new Relay();}
					else { $device = new SensorSKM();}
					$portnumbers =  explode(',', $objtype->PORT_NUMBERS);
					for ($i=0; $i<count($portnumbers); $i++){
						$name = $type . ($i+1) . ' ' . $attributes['NAME'];
						$attr = array();
						$attr['NAME'] = $name;
						$attr['LOCATION'] = $cattr['LOCATION'];
						$attr['TYPE'] = 'D';
						$attr['ASSOCIATE'] = $cameraid;
						$attr['NODE_ID'] = $nodeid;
						if (isset($attributes['TIME_ZONE'])) $attr['TIME_ZONE'] = $attributes['TIME_ZONE'];
						if ($type == 'relay'){
							$attr['SUBTYPE'] = 'R';
							$attr['HW_ID'] = $portnumbers[$i];
							$attr['MODELID'] = "SKM";
						}
						else {
							$attr['SUBTYPE'] = 'S';
							$attr['HW_PORT'] = $portnumbers[$i];
							$attr['USRNAME'] = $cattr['USRNAME'];
							$attr['PASSWD'] = $cattr['PASSWD'];
							$attr['MODELID'] = "SKM";
						}

						$tmpcreated = $device->create($attr);
						if (isset($tmpcreated["obj"]))
							$tmpcreated = $tmpcreated["obj"];
						if (isset($tmpcreated))
						{
							$created[] = $tmpcreated;
							foreach ($sets as $oneset){
								 if (isset($oneset['obj']) && !$set->isProtected($oneset['obj'])){
									 $set->add($oneset['obj'], $tmpcreated);
								 }
							}
						}
					}
				}
			}
			if (count($created)>0){
				$createdlist = implode(',', $created);
				$camera->setAttributes($cameraid, array('AE_LIST' => $createdlist));
			}
			DB::query("COMMIT;");
		}

		ObjectManager::send();
	}

	/**
	 * internal SKM function, not for use from API
	 * invoke method of relay
	 *
	 * @static
	 * @throws Exception
	 * @param  int $obj
	 * @param  string $methodname
	 * @return array array("info" => )
	 */
	public static function invokeMethod($obj, $methodname)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$userid = $_SESSION[SESSION_USERID];
		$service = new ModifyObjects();

		$info = $service->invokeMethod($obj, $userid, $methodname);
		if (isset($info) && $info != ''){
			if (isset($info->faultstring))
				throw new Exception($info->faultstring);
			else
				throw new Exception(__('Error of testing method.'));
		}

		return array(
			"info" => $info
		);
	}

	/**
	 * Create request for creating certificate
	 *
	 * @static
	 * @param  array $requestParameters JSONobject
	 * @throws Exception
	 * @return array array("request" => )
	 */
	public static function createCertificateRequest(array $requestParameters)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$request = Certificate::createRequest($requestParameters);

		if (!isset($request) || $request == array())
			throw new Exception(__('Error getting request.'));

		return array(
			"request" => $request
		);
	}

	/**
	 * Install certificate by copy\paste certificate text
	 * @static
	 * @throws AuthException
	 * @throws Exception
	 * @param  string $certificate
	 * @return array array("result" => )
	 */
	public static function installCertificate($certificate)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$result = Certificate::install($certificate);

		if (!isset($result))
			throw new Exception(__('Error installing certificate.'));

		return array(
			"result" => $result
		);
	}

	/**
	 * Add audit record
	 *
	 * @static
	 * @param string $category
	 * @param string $paramsJSON
	 * @param int|null $objid
	 * @throws Exception
	 * @return array array("result" => )
	 */
	public static function addAuditRecord($category, $paramsJSON, $objid = null)
	{
		$userid = $_SESSION[SESSION_USERID];
		$paramArray = array();

		$paramsJSON = json_decode($paramsJSON);

		for($i = 0; $i < count($paramsJSON->digestparams); $i++){
			$paramArray[$i] = $paramsJSON->digestparams[$i];
		}

		foreach ((array)$paramsJSON->params as $key => $value) {
			array_push($paramArray, Audit::attrValueEntity($key,$value));
		}

		$result = Audit::addRecord($category, $paramArray, $objid, $userid);

		if (!isset($result))
			throw new Exception(__('Error adding audit record.'));

		return array(
			"result" => $result
		);
	}

	/**
	 *
	 * @static
	 *
	 * @param int $scheduleid
	 * @param array $oldobjlist
	 * @param array $objlist
	 *
	 * @return array
	 * @throws AuthException
	 */
	public static function saveObjectsOnSchedule($scheduleid, array $oldobjlist, array $objlist)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$schedule = new Schedule();
		DB::query("BEGIN;");
		$list = $schedule->saveObjectsOnSchedule($scheduleid, $oldobjlist, $objlist);
		DB::query("COMMIT;");
		return array(
			"list" => $list
		);
	}

	/**
	 *
	 * @static
	 *
	 * @param int $scheduleid
	 * @param int $postureid
	 * @param int $timecardid
	 *
	 * @throws AuthException
	 * @throws DBException
	 */
	public static function releasePosture($scheduleid, $postureid, $timecardid)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$schedule = new Schedule();
		DB::query("BEGIN;");
		$schedule->releasePosture($scheduleid, $postureid, $timecardid);
		DB::query("COMMIT;");
	}

	/**
	 * @static
	 *
	 * @param array $scheduleInfo JSONobject
	 *
	 * @return array
	 * @throws AuthException
	 * @throws DBException
	 */
	public static function setScheduleInfo(array $scheduleInfo)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$schedule = new Schedule();
		DB::query("BEGIN;");
		$list = $schedule->setScheduleInfo($scheduleInfo);
		DB::query("COMMIT;");

		return array(
			"list" => $list
		);
	}

	/**
	 *
	 * @static
	 *
	 * @param int $scheduleid
	 * @param $scheduleInfo
	 * @param $datefrom
	 * @param $dateto
	 *
	 * @return array
	 * @throws AuthException
	 */
	public static function convertScheduleInfoToEventSource($scheduleid, $scheduleInfo, $datefrom, $dateto)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$schedule = new Schedule();
		// convert $datefrom, $dateto from timestamp to DateTime
		$dtfrom = new DateTime();
		$dtfrom->setTimestamp($datefrom);
		$dtto = new DateTime();
		$dtto->setTimestamp($dateto);
		$scheduleInfo = json_decode($scheduleInfo);
		$list = $schedule->convertScheduleInfoToEventSource($scheduleid, $scheduleInfo, $dtfrom, $dtto);

		return array(
			"list" => $list
		);
	}

	/**
	 * send attributes by sms to avatar
	 *
	 * @static
	 *
	 * @param int $obj
	 * @param array $attributes JSONobject
	 *
	 * @throws AuthException
	 */
	public static function sendAttributesBySMS($obj, array $attributes)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$avatar = new Avatar();
		$avatar->sendAttributesBySMS($obj, $attributes);
	}

	/**
	 * remove session and send notifucation about it to user open GUI
	 *
	 * @static
	 *
	 * @param  string|null $sessionid
	 * @param  int|null $userid
	 *
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 * @throws Exception
	 */
	public static function stopSession($sessionid = null, $userid = null)
	{
		if (!isset($sessionid) && !isset($userid))
		{
			throw new InvalidArgumentException("'sessionid' or 'userid' param should be defined");
		}

		$userid = intval($userid);

		$sessionManager = new SessionManager();
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			$useridRelatedToSessionId = null;
			if (isset($sessionid))
			{
				$sessionContent = $sessionManager->getSessionContent($sessionid);
				$useridRelatedToSessionId = $sessionContent["userid"];
			} else
			if (isset($userid))
			{
				$useridRelatedToSessionId = $userid;
			}

			if ($useridRelatedToSessionId != $_SESSION[SESSION_USERID])
			{
				throw new AuthException("Not enough rights");
			}
		}

		$isStopped = false;

		if (isset($sessionid))
		{
			$isStopped = $sessionManager->stop($sessionid, "SESSION_TERMINATED");
		} else
		if (isset($userid))
		{
			$isStopped = $sessionManager->stopForUser($userid);
		}

		if (!$isStopped)
		{
			throw new Exception("Cannot stop this session");
		}
	}

	/**
	 * set new password for user
	 * only users from Admin role can set passwords
	 *
	 * @static
	 *
	 * @param int $userid obj id of user
	 * @param string $password new user password
	 *
	 * @throws AuthException
	 * @throws DBException
	 */
	public static function setPassword($userid, $password)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		DB::query("BEGIN;");
		$user = new User();
		$user->setPassword($userid, $password, true);
		DB::query("COMMIT;");
	}

	/**
	 * change password for user
	 * only users from Admin role can change passwords for all users
	 * other users can change only own passwords
	 *
	 * @static
	 *
	 * @param string $currentPassword current user password
	 * @param string $newPassword new user password
	 * @param int $userid obj id of user, by default equal to current session user
	 *
	 * @throws AuthException
	 * @throws DBException
	 * @throws Exception
	 */
	public static function changePassword($currentPassword, $newPassword, $userid = null)
	{
		$currentUserId = $_SESSION[SESSION_USERID];
		if (!isset($userid))
		{
			$userid = $currentUserId;
		}

		DB::query("BEGIN;");
		$user = new User();
		$isAdmin = $user->isAdmin($currentUserId);
		if ($userid != $currentUserId && !$isAdmin)
		{
			throw new AuthException("Not enough rights");
		}

		// user should be able to change password once every CHANGE_PASSWORD_INTERVAL hours
		if (!$user->isAbleToChangePassword($userid, $changePasswordInterval))
		{
			throw new Exception(sprintf(__("User can change account passwords once every %d hours"), $changePasswordInterval));
		}

		$user = new User();
		if (!$user->changePassword($userid, $currentPassword, $newPassword))
		{
			throw new Exception(__("Some problems while changing password"));
		}

		// after successful password change we can login to system
		if ($_SESSION[SESSION_STATUS] == SessionManager::SESSION_STATUS_PASSWORD_EXPIRED)
		{
			$_SESSION[SESSION_STATUS] = SessionManager::SESSION_STATUS_LOGGED_IN;
		}
		DB::query("COMMIT;");
	}

	/**
	 * change metadata.csv file
	 *
	 * @static
	 *
	 * @param string $metaname
	 * @param string $ttl
	 * @param string $color
	 * @param string $font
	 * @param string $style
	 * @param string $height
	 * @param string $align
	 * @param string $valign
	 *
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 * @throws Exception
	 */
	public static function setMetadataInfo($metaname, $ttl, $color, $font, $style, $height, $align, $valign)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$csvName = APL_PATH . "/mgears/etc/metadata.csv";
		$csvFile = file($csvName, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);

		if (!$csvFile)
		{
			throw new Exception("Can't read file");
		}

		$found = false;
		// get column number by header
		$deHeader = array();

		$header = preg_split('/,/', $csvFile[0]);
		for ($i = 0, $n = count($header); $i < $n; $i++)
		{
			$deHeader[$header[$i]] = $i;
		}

		// parse all data lines (second and further)
		for ($i = 1, $n = count($csvFile); $i < $n; $i++)
		{
			$cells = preg_split('/,/', $csvFile[$i]);
			if ($cells[$deHeader["Inner Name"]] != $metaname)
				continue; // skip 'cause doesn't match

			if ($ttl)
			{
				$cells[$deHeader["TTL"]] = $ttl;
			}

			$styles = array();
			foreach (preg_split('/;/', $cells[$deHeader["Style"]]) as $stl)
			{
				$kv = preg_split('/:/', $stl);
				$styles[$kv[0]] = $kv[1];
			}

			if ($color)
			{
				$styles["color"] = $color;
			} else {
				unset($styles["color"]);
			}

			if ($font)
			{
				$styles["font"] = $font;
			} else {
				unset($styles["font"]);
			}

			if ($style)
			{
				$styles["style"] = $style;
			} else {
				unset($styles["style"]);
			}

			if ($height)
			{
				$styles["height"] = $height;
			} else {
				unset($styles["height"]);
			}

			if ($align)
			{
				$styles["align"] = $align;
			} else {
				unset($styles["align"]);
			}

			if ($valign)
			{
				$styles["v-align"] = $valign;
			} else {
				unset($styles["v-align"]);
			}

			$astyle = array();
			foreach ($styles as $key => $value)
			{
				$astyle[] = "$key:$value";
			}
			$cells[$deHeader["Style"]] = join(";", $astyle);

			$csvFile[$i] = join(",", $cells);
			$found = true;

			break; // skip all to end
		}

		if (!$found)
			throw new InvalidArgumentException("Specified metadata name '" . $metaname . "' is not found");

		if (!file_put_contents($csvName, join("\n", $csvFile), LOCK_EX))
		{
			throw new Exception("Can't write file");
		}
	}

	/**
	 * submit action for eventlog
	 *
	 * @param string $name
	 * @param array $parameters JSONobject
	 * @param array $properties JSONobject
	 * @param array $actions
	 * @throws Exception
	 */
	public static function submitAction($name, array $parameters, array $properties, array $actions = array())
	{
		$node = new Node();
		$node->submitAction($name, $parameters, $properties, $actions);
	}

	/**
	 * @param int $eventid
	 * @param int $roleid
	 * @param string $note
	 * @throws AuthException
	 * @throws DBException
	 */
	public static function announceEventToRole($eventid, $roleid, $note = "")
	{
		$userid = $_SESSION[SESSION_USERID];

		// user can send announce to own roles only
		$canSend = false;
		$user = new User();
		$roleList = $user->getRoles($userid);
		foreach ($roleList as $role)
		{
			if ($role["obj"] == $roleid)
			{
				$canSend = true;
				break;
			}
		}

		if (!$canSend)
		{
			throw new AuthException("Not enough rights");
		}

		DB::query("BEGIN;");

		$announce = new Announce();
		$announce->create($userid, $eventid, $roleid, null, $note);

		DB::query("COMMIT;");
	}

	/**
	 * @param int $eventid
	 * @param array $useridList
	 * @param string $note
	 * @throws DBException
	 */
	public static function announceEventToUser($eventid, array $useridList, $note = "")
	{
		$fromUserid = $_SESSION[SESSION_USERID];

		if (empty($useridList))
		{
			return;
		}

		$announce = new Announce();
		$objectid = $announce->getEventObjId($eventid);

		// user can send announce to user in own roles only
		$usr = new User();
		$userRole = new Role();
		$roleList = $usr->getRoles($fromUserid);
		$toUserIdList = array();
		foreach ($roleList as $role)
		{
			$userList = $userRole->getUsers($role["obj"]);
			foreach ($userList as $user)
			{
				if (in_array($user["obj"], $useridList) && $usr->getObjectCredentials($user["obj"], $objectid) != '')
				{
					$toUserIdList[] = $user["obj"];
				}
			}
		}

		if (!empty($toUserIdList))
		{
			DB::query("BEGIN;");

			foreach ($toUserIdList as $userid)
			{
				$announce->create($fromUserid, $eventid, null, $userid, $note);
			}

			DB::query("COMMIT;");
		}
	}

	/**
	 * @param array $eventList
	 */
	public static function markAnnounceEventAsRead(array $eventList)
	{
		$userid = $_SESSION[SESSION_USERID];

		DB::query("BEGIN;");

		$announce = new Announce();
		$announce->markAsRead($userid, $eventList);

		DB::query("COMMIT;");
	}

	/**
	 * used by Avatar
	 *
	 * @param string $csr
	 *
	 * @return array
	 * @throws AuthException
	 * @throws Exception
	 */
	public static function registerOpenVPNClient($csr)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$node = new Node();
		if (!$node->isOpenVPNActivated())
		{
			throw new Exception(__("Stratus activated without OpenVPN support"));
		}

		$tmp_csr_name = substr(str_shuffle("0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"), 0, 12);
		$tmp_csr_path = APL_VAR_PATH . "/tmp/php/" . $tmp_csr_name;
		if (!$fh = fopen($tmp_csr_path . ".csr", "w"))
		{
			throw new Exception(__("Error saving CSR"));
		}
		if (!fwrite($fh, $csr))
		{
			throw new Exception(__("Error saving CSR"));
		}
		if (!fclose($fh))
		{
			throw new Exception(__("Error saving CSR"));
		}
		system(escapeshellcmd("sudo " . APL_PATH . "/base/bin/pki_helper sign $tmp_csr_path") . ">/dev/null 2>&1", $ret);
		if ($ret != 0)
		{
			throw new Exception(__("Error signing CSR"));
		}
		// pkitool should create {$tmp_csr_path}.crt file
		if (!is_file($tmp_csr_path . ".crt"))
		{
			throw new Exception(__("Error creating certificate"));
		}
		$client_cert = file_get_contents($tmp_csr_path . ".crt");
		exec(escapeshellcmd("sudo " . APL_PATH . "/base/bin/pki_helper print_ca") . " 2>/dev/null", $out);
		$ca_cert = join("\n", $out) . "\n";

		unlink($tmp_csr_path . ".csr");
		unlink($tmp_csr_path . ".crt");

		return array(
			"ca_cert" => $ca_cert,
			"client_cert" => $client_cert
		);
	}

	/**
	 * @param string $name
	 * @param string $description
	 * @param array $eventsource_workflow_parameters
	 * @param array $eventpolicystorage
	 *
	 * @throws AuthException
	 * @throws DBException
	 * @throws Exception
	 */
	public static function addEventPolicy($name, $description, array $eventsource_workflow_parameters, array $eventpolicystorage)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		DB::query("BEGIN;");

		$eventpolicy = new EventPolicy();
		$eventpolicy->addEventPolicy($name, $description, $eventsource_workflow_parameters, $eventpolicystorage);

		DB::query("COMMIT;");
	}

	/**
	 * @param int $id
	 * @param string $name
	 * @param string $description
	 * @param array $eventsource_workflow_parameters
	 * @param array $eventpolicystorage
	 *
	 * @throws AuthException
	 * @throws DBException
	 * @throws Exception
	 */
	public static function editEventPolicy($id, $name, $description, array $eventsource_workflow_parameters, array $eventpolicystorage)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		DB::query("BEGIN;");

		$eventpolicy = new EventPolicy();
		$eventpolicy->editEventPolicy($id, $name, $description, $eventsource_workflow_parameters, $eventpolicystorage);

		DB::query("COMMIT;");
	}

	/**
	 * @param int $id
	 *
	 * @throws AuthException
	 * @throws DBException
	 */
	public static function deleteEventPolicy($id)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		DB::query("BEGIN;");

		$eventpolicy = new EventPolicy();
		$eventpolicy->deleteEventPolicy($id);

		DB::query("COMMIT;");
	}

	/**
	 * accept "License Agreement" for current user
	 */
	public static function acceptLicense()
	{
		$userid = $_SESSION[SESSION_USERID];

		DB::query("BEGIN;");

		$user = new User();
		$user->acceptLicense($userid);

		DB::query("COMMIT;");
	}

	/**
	 * @param string $serialNumber
	 *
	 * @throws AuthException
	 * @throws DBException
	 * @throws Exception
	 */
	public static function activateSerialNumber($serialNumber)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		DB::query("BEGIN;");

		$user = new User();
		if (!$user->checkObjectCredentials($_SESSION[SESSION_USERID], Obj::CONTROL_PANEL, "s"))
		{
			// if (!ManageUser::hasRole(11)){
			// FileLog::info(__("Tried to activate Serial Number, but did not have credentials"));
			throw new AuthException(__("Not enough rights"));
		}

		$license = new License();
		$license->activateSerialNumber($serialNumber);

		DB::query("COMMIT;");
	}

	/**
	 * @param string $serialNumber
	 *
	 * @throws AuthException
	 * @throws DBException
	 * @throws Exception
	 */
	public static function createPCID($serialNumber)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		DB::query("BEGIN;");

		$user = new User();
		if (!$user->checkObjectCredentials($_SESSION[SESSION_USERID], Obj::CONTROL_PANEL, "s"))
		{
			throw new AuthException(__("Not enough rights"));
		}

		$license = new License();
		$license->createPCID($serialNumber);

		DB::query("COMMIT;");
	}

	/**
	 * @param int $obj
	 * @param string $commandName
	 *
	 * @return array
	 * @throws InvalidArgumentException
	 */
	public static function execAvatarExternalCommand($obj, $commandName)
	{
		$avatar = new Avatar();
		if (!$avatar->isAvatar($obj))
		{
			throw new InvalidArgumentException(__("Incorrect 'obj' parameter"));
		}

		return $avatar->execExternalCommand($obj, $commandName);
	}
}
