<?php defined('APL_PATH') or die('No direct script access.');
/**
 * @version $Id: API_GET.php 33561 2016-02-01 07:03:22Z sabaev $
 *
 * This class represents HTTP API for getting data.
 *
 * Methods of this class must only for external purposes, parameters order of methods not garanted.
 * All methods must be called using Proxy class.
 *
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions LLC 2006
 * ------------------------------------------------------------------------------
 */

class API_GET extends API
{
	/**
	 * get object attribute value
	 *
	 * Examples:
	 *
	 * /api/call/getAttribute?obj=101&attribute=NAME
	 *
	 * @static
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 * @param  int $obj objid
	 * @param  string $attribute
	 * @return array("value" => string)
	 */
	public static function getAttribute($obj, $attribute)
	{
		$object = Factory::getByObj($obj);
		if (!isset($object))
			throw new InvalidArgumentException(sprintf("unknown object=%s", $obj));

		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID])
		    && !Node::isLowCredentialAttribute($attribute))
		{
			throw new AuthException("Not enough rights");
		} else
		if ($attribute == "UNI")
		{
			$value = $object->getUNI($obj);
		} else {
			$value = $object->getAttribute($obj, $attribute);
		}

		if ($object->getType() == "camera"
		    && $attribute == "PASSWD")
		{
			$value = null;
		}

		if (!isset($value))
		{
			throw new InvalidArgumentException(sprintf("obj=%s or attribute=%s does not exists", $obj, $attribute));
		}

		return array("value" => $value);
	}

	/**
	 * get list of object attributes
	 *
	 * Examples:
	 *
	 * /api/call/getAttributes?obj=101
	 *
	 * @docBlock
	 * # Object management
	 * ## Get object configuration
	 * #### Request
	 * > GET /api/call/getAttributes
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"list":{"ARCHSTATE":"off","NAME":"some name"},"type":"camera","error":"","code":200}
	 *
	 * @static
	 * @throws InvalidArgumentException
	 * @param  int $obj object id
	 * @return array("list" => array, "type" => string);
	 */
	public static function getAttributes($obj)
	{
		$object = Factory::getByObj($obj);
		if (!isset($object))
			throw new InvalidArgumentException(sprintf("unknown object=%s", $obj));

		$list = $object->getAttributes($obj);

		$type = $object->getType();

		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			$list = Utils::array_intersect_keys($list, Node::getLowCredentialAttributes(), "STAT_");
		}

		if ($type == "camera")
		{
			unset($list["PASSWD"]);
		}

		return array(
			"list" => $list,
			"type" => $type
		);
	}

	/**
	 * get object block url for downloading
	 *
	 * Examples:
	 *
	 * /api/call/getBlockURL?obj=101&blockId=1
	 *
	 * @static
	 * @throws InvalidArgumentException
	 * @param  $obj
	 * @param  $blockId
	 * @return array("url" => string);
	 */
	public static function getBlockURL($obj, $blockId)
	{
		$object = Factory::getByObj($obj);
		if (!isset($object))
			throw new InvalidArgumentException(sprintf("unknown object=%s", $obj));

		$url = $object->getBlockURL($obj, $blockId);

		return array("url" => $url);
	}

	/**
	 * get list of object blocks info for downloading
	 *
	 * Examples:
	 *
	 * /api/call/getBlocksInfo?obj=101
	 *
	 * @static
	 * @throws InvalidArgumentException
	 * @param  $obj
	 * @return array("list" => array);
	 */
	public static function getBlocksInfo($obj)
	{
		$object = Factory::getByObj($obj);
		if (!isset($object))
			throw new InvalidArgumentException(sprintf("unknown object=%s", $obj));

		$list = $object->getBlocksInfo($obj);

		return array("list" => $list);
	}

	/**
	 * get identity attribute
	 *
	 * Examples:
	 *
	 * /api/call/getIdentityAttribute?attribute=URL
	 *
	 * @static
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 * @param  string $attribute
	 * @return array("value" => string);
	 */
	public static function getIdentityAttribute($attribute)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID])
		    && !Node::isLowCredentialAttribute($attribute))
		{
			throw new AuthException("Not enough rights");
		}

		$value = Identity::getAttribute($attribute);

		if (!isset($value))
		{
			throw new InvalidArgumentException(sprintf("attribute=%s does not exists", $attribute));
		}

		return array("value" => $value);
	}

	/**
	 * get list of identity attributes
	 *
	 * Examples:
	 *
	 * /api/call/getIdentityAttributes
	 *
	 * @static
	 * @return array("list" => array);
	 */
	public static function getIdentityAttributes()
	{
		$list = Identity::getAttributes();

		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			$list = Utils::array_intersect_keys($list, Node::getLowCredentialAttributes(), "STAT_");
		}

		return array("list" => $list);
	}

	/**
	 * get list of objects
	 *
	 * Examples:
	 *
	 * get full list of objects
	 * /api/call/getObjectList
	 *
	 * get list of objects with specified type
	 * /api/call/getObjectList?type=camera
	 *
	 * get list of objects from specified node
	 * /api/call/getObjectList?nodeid=100
	 *
	 * get list of objects with attributes and blocks, with specified type
	 * /api/call/getObjectList?type=camera&withAttributes=true&$withBlocks
	 *
	 * get list of objects with specified type and filter by objList
	 * /api/call/getObjectList?type=camera&objList=[101,102]
	 *
	 * get childs of parentobject
	 * /api/call/getObjectList?parentObj=101
	 *
	 * get all sets with specified object
	 * /api/call/getObjectList?type=set&isAll=true&childObj=110
	 *
	 * @docBlock
	 * # Object management
	 * ## Get object  list
	 * #### Request
	 * > GET /api/call/getObjectList
	 *
	 * examples:
	 *
	 * > Get list of roles for current user:
	 * > GET /api/call/getObjectList?type=role
	 *
	 * > Get list of sets in role:
	 * > GET /api/call/getObjectList?type=set&roleid=31
	 *
	 * > Get list of objects in set:
	 * > GET /api/call/getObjectList?setid=14
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"list":[{"obj":"102","udid":"DC102","name":"Axis 241S","description":"","location":"","otype":"D","subtype":"C","protected":"0","permission":"MM","credentials":"LVvApaeDMrCSPsmxLVvApaeDMrCSPsmx","attributes":{"DEVICETYPE":"CAMERA","DEVID":"102","OBJID":"102","STATUS":"ON"}},{"obj":"109","udid":"DC109","name":"Axis Q7404","description":"","location":"","otype":"D","subtype":"C","protected":"0","permission":"MM","credentials":"LVvApaeDMrCSPsmxLVvApaeDMrCSPsmx","attributes":{"DEVICETYPE":"CAMERA","DEVID":"102","OBJID":"102","STATUS":"ON"}}],"error":"","code":200}
	 *
	 * @static
	 *
	 * @param  int|null $parentObj filter by parent obj id
	 * @param  int|null $childObj
	 * @param  string $type type of objects in list. Supported types are: camera, audio, node, role, set, user
	 * @param  bool $withAttributes If "true", then object list will be returned with attributes list
	 * @param  bool $withBlocks return with blocks
	 * @param  bool $withCredentials return with credentials, only for set
	 * @param  array $objList filter by obj id
	 * @param  int|null $setid filter by set id
	 * @param  int|null $roleid filter by role id
	 * @param  int|null $userid filter by user id
	 * @param  int|null $nodeid filter by node id
	 * @param  bool $isAll return all objects with type user, role or set without filtering based on credentials management
	 * @param  array $filterAttributeList if withAttributes=true then return only attributes from list
	 * @param  string $filterAttribute Name of filtered attribute. May be used in pair with $filterMask
	 * @param  string $filterMask Mask of filtered attribute value. May be used in pair with $filterAttribute
	 *
	 * @return array("list" => array)
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 */
	public static function getObjectList($parentObj = null, $childObj = null, $type = null, $withAttributes = false, $withBlocks = false, $withCredentials = false, array $objList = array(), $setid = null, $roleid = null, $userid = null, $nodeid = null, $isAll = null, array $filterAttributeList = array(), $filterAttribute = null, $filterMask = null)
	{
		$withAttributes = Template::boolVal($withAttributes);
		$withBlocks = Template::boolVal($withBlocks);
		$userid = isset($userid) ? intval($userid) : $_SESSION[SESSION_USERID];
		$isAll = Template::boolVal($isAll);
		$parentObj = isset($parentObj) ? intval($parentObj) : null;
		$childObj = isset($childObj) ? intval($childObj) : null;
		$setid = isset($setid) ? intval($setid) : null;
		$roleid = isset($roleid) ? intval($roleid) : null;
		$nodeid = isset($nodeid) ? intval($nodeid) : null;

		if (
			isset($filterAttribute) && !isset($filterMask)
			||
			!isset($filterAttribute) && isset($filterMask)
			)
		{
			throw new InvalidArgumentException(sprintf("%s and %s should be specified in pair", "filterAttribute", "filterMask"));
		}

		$result = array();
		$list = array();

		if (isset($parentObj))
		{
			$parentObject = Factory::getByObj($parentObj);
			if (!isset($parentObject))
				throw new InvalidArgumentException(sprintf("unknown object=%s", $parentObj));

			$parentObjectType = $parentObject->getType($parentObj);
			switch ($parentObjectType)
			{
				case "schedule":
					if (isset($type) && $type == "posture")
					{
						$list = $parentObject->getPostures($parentObj, $withAttributes);
					}
					break;
				case "avatar":
					$list = $parentObject->getObjects($parentObj, $userid, $roleid, null, null, $withAttributes);
					break;
				default:
					$list = $parentObject->getObjects($parentObj, null, null, $withAttributes, $withBlocks);
			}
		} else
		if (isset($type))
		{
			switch ($type)
			{
				case "set":
					if ($isAll)
					{
						$node = new Node();
						$list = $node->getSets($childObj, $withAttributes);
					} else
					if (isset($roleid))
					{
						$role = new Role();
						$list = $role->getSets($roleid, $withAttributes);
					} else
					if (isset($childObj))
					{
						$user = new User();
						$list = $user->getSets($userid, null, $childObj, $withAttributes, false, $withCredentials);
					} else {
						$user = new User();
						$list = $user->getSets($userid, null, null, $withAttributes, false, $withCredentials);
					}
					break;
				case "role":
					if ($isAll)
					{
						$node = new Node();
						$list = $node->getRoles($withAttributes);
					} else
					if (isset($setid))
					{
						$set = new Set();
						$list = $set->getRoles($setid, $withAttributes);
					} else {
						$user = new User();
						$list = $user->getRoles($userid, $withAttributes);
					}
					break;
				case "user":
					if ($isAll)
					{
						$node = new Node();
						$list = $node->getUsers($withAttributes);
					} else {
						if (!isset($roleid))
						{
							throw new InvalidArgumentException("roleid should be specified");
						}

						$role = new Role();
						$list = $role->getUsers($roleid, $withAttributes);
					}

					// user from not Admin role, cannot get all users
					$user = new User();
					if (!$user->isAdmin($_SESSION[SESSION_USERID]))
					{
						$list = array_values(array_filter($list, function($row){
							return $row["obj"] == $_SESSION[SESSION_USERID];
						}));
					}
					break;
				case "node":
					$node = new Node();
					$list = $node->getListWithNumberOfCameras($withAttributes);
					break;
				case "avatar":
					$node = new Node();
					$list = $node->getObjects(null, "V", "*", $withAttributes);
					break;
				case "configuration":
					$configuration = new Configuration();
					if (!isset($childObj)) {
						$list = $configuration->getListByType($roleid, 'configuration');
					} else {
						$list = $configuration->getListByType($roleid, $childObj);
					}
					break;
				case "location":
					$location = new Location();
					$list = $location->getListByRole($roleid);
					break;
				case "view":
					$view = new View();
					$list = $view->getList();
					break;
				case "schedule":
					$node = new Node();
					$list = $node->getObjects(null, "T", "S", $withAttributes);
					break;
				case "posture":
					$node = new Node();
					$list = $node->getObjects(null, "T", "P", $withAttributes);
					break;
				case "timecard":
					$node = new Node();
					$list = $node->getObjects(null, "T", "T", $withAttributes);
					break;
				default:
					$typeToOtypeSubtype = Obj::getTypes(true);
					if (isset($typeToOtypeSubtype[$type]))
					{
						$user = new User();
						if (isset($setid))
						{
							$list = $user->getObjectsFromSet($userid, $roleid, $setid, $withAttributes, $withBlocks, $typeToOtypeSubtype[$type]["otype"], $typeToOtypeSubtype[$type]["subtype"], $filterAttribute, $filterMask);
						} else {
							$list = $user->getObjects($userid, $roleid, $withAttributes, $withBlocks, $typeToOtypeSubtype[$type]["otype"], $typeToOtypeSubtype[$type]["subtype"], $filterAttribute, $filterMask);
						}
					}
			}
		} else {
			if (isset($setid))
			{
				$user = new User();
				$list = $user->getObjectsFromSet($userid, $roleid, $setid, $withAttributes, $withBlocks, null, null, $filterAttribute, $filterMask);
			} else {
				$user = new User();
				$node = new Node();

				$objectsFromNode = array();
				if (isset($nodeid))
				{
					$objectsFromNode = $node->getObjects($nodeid);
				}

				$sets = $node->getSets();
				foreach ($sets as $set)
				{
					$objectsFromSetList = $user->getObjectsFromSet($userid, $roleid, $set["obj"], $withAttributes, $withBlocks, null, null, $filterAttribute, $filterMask);
					foreach ($objectsFromSetList as $objectFromSet)
					{
						$isUnique = true;

						// add to list only unique items
						foreach ($list as $element)
						{
							if ($element["obj"] == $objectFromSet["obj"])
							{
								$isUnique = false;
								break;
							}
						}

						$add = false;
						$onNode = false;

						// add to list only items from specified node
						if (isset($nodeid) && $isUnique)
						{
							foreach ($objectsFromNode as $objectFromNode)
							{
								if ($objectFromNode["obj"] == $objectFromSet["obj"])
								{
									$onNode = true;
									break;
								}
							}

							if ($onNode)
								$add = true;
						} else
						if (!isset($nodeid) && $isUnique)
						{
							$add = true;
						}

						if ($add)
						{
							$list[] = $objectFromSet;
						}
					}
				}
			}
		}

		$types = Obj::getTypes();
		foreach ($list as &$row)
		{
			if (!isset($row["udid"]) || (isset($row["udid"]) && $row["udid"] === null))
			{
				$row["udid"] = '';
			}

			$rowType = $type;
			if (isset($row["otype"]) && isset($row["subtype"]))
			{
				$row["type"] = $types[$row["otype"]][$row["subtype"]]["template"];
			}

			if ($withAttributes && $rowType == "camera")
			{
				unset($row["attributes"]["PASSWD"]);
			}
		}
		unset($row);

		// TODO: slow code, need optimizing
		if (count($objList) > 0)
		{
			$newList = array();
			foreach ($list as $row)
			{
				if (in_array($row["obj"], $objList))
				{
					array_push($newList, $row);
				}
			}
			$list = $newList;
		}

		// TODO: slow code, need optimizing
		if (count($filterAttributeList) > 0)
		{
			foreach ($list as &$row)
			{
				foreach ($row["attributes"] as $attribute => $value)
				{
					if (!in_array($attribute, $filterAttributeList))
					{
						unset($row["attributes"][$attribute]);
					}
				}
			}
			unset($row);
		}

		// TODO: slow code, need optimizing
		$user = new User();
		if ($withAttributes && !$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			// filter attributes
			foreach ($list as &$row)
			{
				$row["attributes"] = Utils::array_intersect_keys($row["attributes"], Node::getLowCredentialAttributes(), "STAT_");
			}
			unset($row);
		}

		$result["list"] = $list;

		return $result;
	}

	/**
	 * get objects marked as deleted
	 *
	 * @static
	 * @param  null $otype
	 * @param  string $subtype
	 * @throws AuthException
	 * @return array
	 */
	public static function getDeletedObjectList($otype = null, $subtype = '*')
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$list = array();
		if (isset($otype))
		{
			$queryString = "SELECT * FROM _objs WHERE otype = ? And subtype = ? AND deleted = 1;";
			$list = DB::select($queryString, array($otype, $subtype));
		}

		return array(
			"list" => $list
		);
	}

	/**
	 * get object template
	 *
	 * @deprecated 2.6.3
	 * @static
	 * @param $obj
	 *
	 * @return array
	 * @throws AuthException
	 * @throws Exception
	 */
	public static function getTemplate($obj)
	{
		trigger_error('Function ' . __METHOD__ . ' is deprecated');
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$template = null;

		$object = new Obj();

		$templateName = $object->getTemplateName($obj);

		if ($templateName == 'camera')
		{
			$camera = new Camera();
			$template = $camera->getTemplateAttributes($obj);
		} else
		if ($templateName == 'gui')
		{
			if ($obj == 53)
			{
				$template = Identity::getTemplateAttributes();
			}
		}

		if (!isset($template))
		{
			throw new Exception("Can't get template");
		}

		return array(
			"template" => $template,
			"deprecated" => "true"
		);
	}

	/**
	 * get templates list
	 *
	 * Examples:
	 *
	 * get template of specified type
	 * /api/call/getTemplates?type=camera
	 * /api/call/getTemplates?type=camera&attributes={"NAME":"new"}
	 *
	 * get objects templates from specified set
	 * /api/call/getTemplates?setid=99
	 *
	 * get objects from specified list
	 * /api/call/getTemplates?objList=[101,102]
	 *
	 * get objects from specified list and type
	 * /api/call/getTemplates?objList=[101,102]&type=camera
	 *
	 * get objects templates from specified node
	 * /api/call/getTemplates?nodeid=100
	 *
	 * @static
	 * @throws InvalidArgumentException
	 * @param  string $type
	 * @param  array $objList
	 * @param  int|null $setid
	 * @param  int|null $nodeid
	 * @param  array $attributes JSONobject
	 * @return array("template" => array)
	 */
	public static function getTemplates($type = null, array $objList = array(), $setid = null, $nodeid = null, array $attributes = array())
	{
		if (!isset($type) && !isset($objList) && !isset($setid) && !isset($nodeid))
		{
			throw new InvalidArgumentException(sprintf("%s, %s, %s or %s should be defined", "type", "objList", "setid", "nodeid"));
		}

		$templateList = null;

		if (isset($type) && count($objList) == 0 && !isset($setid) && !isset($nodeid))
		{
			$object = Factory::getByType($type);
			$templateList[null] = $object->getTemplateAttributes(null, $attributes);
			$templateList[null]["type"] = $type;
		} else {
			if (count($objList) == 0 && isset($setid))
			{
				$objList = array();
				$set = new Set();
				$list = $set->getObjects($setid);
				foreach ($list as $row)
				{
					$objList[] = $row["obj"];
				}
			} else if (count($objList) == 0 && isset($nodeid)){
				$objList = array();
				$node = new Node();
				$list = $node->getObjects($nodeid);
				foreach ($list as $row)
				{
					$objList[] = $row["obj"];
				}
			}

			foreach ($objList as $obj)
			{
				$object = new Obj();
				$objectType = $object->getTemplateName($obj);

				// filter objects by type
				if (!isset($objectType) || isset($type) && $objectType != $type)
					continue;

				switch ($objectType)
				{
					case "gui":
						if ($obj == 53)
						{
							$templateList[$obj] = Identity::getTemplateAttributes();
						}
					break;
					default:
						$object = Factory::getByType($objectType);

						$templateList[$obj] = $object->getTemplateAttributes($obj);
						$templateList[$obj]["type"] = $objectType;

						$templateList[$obj]["attributes"]["PASSWD"]["VALUE"] = "";
				}
			}
		}


		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			// filter attributes
			foreach ($templateList as $obj => &$template)
			{
				foreach ($template["categories"] as $index => &$category)
				{
					$category["attributes"] = array_values(array_filter($category["attributes"], function ($attribute) use (&$templateList, $obj)
					{
						$isLow = Node::isLowCredentialAttribute($attribute);
						if (!$isLow)
						{
							unset($templateList[$obj]["attributes"][$attribute]);
						}

						return $isLow;
					}));
				}
				unset($category);
			}
			unset($template);
		}

		return array(
			"template" => $templateList
		);
	}

	/**
	 * get list of system types
	 *
	 * Examples:
	 *
	 * get list of system types
	 * /api/call/getTypes
	 *
	 * get list of system types sorted by template
	 * /api/call/getTypes?byTemplate=true
	 *
	 * @static
	 * @param  bool $byTemplate
	 * @return array("types" => array)
	 */
	public static function getTypes($byTemplate = false)
	{
		$byTemplate = Template::boolVal($byTemplate);

		return array(
			"types" => Obj::getTypes($byTemplate)
		);
	}

	/**
	 * Get JSON representation of resource tree for current user and given role.
	 * If role isn't defined, default role or first role in list of user roles will be used
	 *
	 * @static
	 * @param  int|null $roleid id of role
	 * @return array resource tree: role, sets and devices
	 */
	public static function getResourceTree($roleid = null)
	{
		function fillObjects(&$set, &$objects, &$result) {
			$set['objects'] = array();

			// Process all objects in set
			$monitors = array();
			foreach ($objects as $object) {

				//Add object into objects array
				$result['objects'][$object['obj']] = $object;

				//Fill monitors
				if($object['otype'] == 'D' && $object['subtype'] =='W'){
					if(!count($monitors)) {
						$node = new Node();
						$getmonitors = $node->getObjects(null, "D", "V", true);
						foreach ($getmonitors as $monitor) {
							$monitors[$monitor['obj']] = $monitor;
						}
					}

					// get monitors from wall
					//[XML_CONF] => 0,0,;0,1,;1,0,;1,1,122;
					$cells = explode(";", $object['attributes']['XML_CONF']);
					foreach ($cells as $cell) {
						$cellinfo = explode(",", $cell);
						if (isset($cellinfo[2]) && isset($monitors[$cellinfo[2]])){
							$result['objects'][$cellinfo[2]] = $monitors[$cellinfo[2]];
						}
					}
				}

				// set may contain only cameras and external objects
				if (($object['otype'] != 'D' && $object['otype'] != 'X') ||
					($object['otype'] == 'D' && in_array($object['subtype'], array('A', 'J', 'V', 'W'))) ||
					($object['otype'] == 'X' && $object['subtype'] == 'G'))
					continue;

				// Add object into set
				array_push($set['objects'], $object['obj']);
			}
		}

		$userid = $_SESSION[SESSION_USERID];
		$result = array(
			'role' => array(),
		);
		$user = new User();

		// Fetch all roles
		$roles = $user->getRoles($userid);
		foreach ($roles as $role) {
			if ($role['protected'] == 1) {
				$role['name'] = __($role['name']);
			}
			$result['roles'][$role['obj']] = $role;
		}

		// If roleid is wrong or isn't specified, get default role
		if (!$roleid || !isset($result['roles'][$roleid])) {
			$roleid = $user->getAttribute($userid, 'DEFAULT_ROLE');
		}

		// Get first given role, if default isn't specified
		if (!isset($result['roles'][$roleid])) {
			$roleid = $roles[0]['obj'];
		}
		$result['role'] = $result['roles'][$roleid];

		$sets = $user->getSets($userid, $roleid);

		$node = new Node();
		$avatar = new Avatar();

		$avatars = $node->getObjects(null, "V", "*", false);
		$sets = array_merge($sets, $avatars);

		// Fetch all objects by each set
		foreach ($sets as $set) {
			if ($set['protected'] == 1) {
				$set['name'] = __($set['name']);
			}

			// This set is an avatar
			if (isset($set['otype']) && $set['otype'] == 'V' && $set['subtype'] == '*') {
				fillObjects($set, $avatar->getObjects($set['obj'], $userid, $roleid, null, null, true), $result);

				// remove empty avatar
				if(count($set['objects']) == 0){
					unset($result['objects'][$set['obj']]);
					continue;
				}
			} else {
				fillObjects($set, $user->getObjectsFromSet($userid, $roleid, $set['obj'], true), $result);
				// If set contain no objects, skip it
				if(!count($set['objects']))
					continue;
			}

			// Add set into sets array
			$result['sets'][$set['obj']] = $set;
		}

		// Add guis, to provide correct rights for admin
		$guis = $user->getGuis($userid, $roleid);
		foreach ($guis as $gui) {
			$result['objects'][$gui['obj']] = $gui;
		}

		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			// filter attributes
			foreach ($result['objects'] as $key => &$row)
			{
				$row["attributes"] = Utils::array_intersect_keys($row["attributes"], Node::getLowCredentialAttributes(), "STAT_");
			}
			unset($row);
		}

		// filter attributes
		foreach ($result['objects'] as $key => $row)
		{
			if ($row["otype"] == "D" && $row["subtype"] == "C")
			{
				unset($result['objects'][$key]["attributes"]["PASSWD"]);
			}
		}

		return $result;
	}

	/**
	 * start camera probing process (asynchronous) and get ticket for ir results
	 *
	 * Examples:
	 *
	 * get ticket for camera probing process
	 * /api/call/getTicketForCameraModel?attributes={"DEVIP":"192.168.0.123","HTTP_PORT":"80","USRNAME":"root","PASSWD":"pass"}
	 *
	 * @static
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 * @param  int|null $obj
	 * @param  array $attributes JSONobject
	 * @param  int|null $nodeid
	 * @return array("ticket" => string)
	 */
	public static function getTicketForCameraModel($obj = null, array $attributes = array(), $nodeid = null)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		if (isset($obj))
		{
			$camera = new Camera();
			$attributes = array_merge($camera->getAttributes($obj), $attributes);
		}

		if (!isset($attributes["DEVIP"]) || !isset($attributes["HTTP_PORT"]))
		{
			throw new InvalidArgumentException(sprintf("attribute %s and %s should be defined", "DEVIP", "HTTP_PORT"));
		}

		$camera = new Camera();
		$ticket = $camera->getTicketForModel($attributes, $nodeid);

		return array(
			"ticket" => $ticket
		);
	}

	/**
	 * get camera parameters by ticket
	 *
	 * Examples:
	 *
	 * get result of camera probing process from ticket
	 * /api/call/getCameraModelByTicket?ticket=192.168.0.123_80_4354353
	 *
	 * @static
	 *
	 * @param  string $ticket
	 *
	 * @return array
	 * @throws AuthException
	 */
	public static function getCameraModelByTicket($ticket)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$camera = new Camera();
		$parameters = $camera->getModelByTicket($ticket);

		return $parameters;
	}

	/**
	 * start camera probing process (synchronous) and get camera parameters
	 *
	 * Examples:
	 *
	 * get camera model with checking all camera model
	 * /api/call/getCameraModel?attributes={"DEVIP":"192.168.0.123","HTTP_PORT":"80","USRNAME":"root","PASSWD":"pass"}
	 *
	 * get camera model  with checking only specified camera model
	 * /api/call/getCameraModel?attributes={"DEVIP":"192.168.0.123","HTTP_PORT":"80","USRNAME":"root","PASSWD":"pass","CAMERAMODEL":"Axis"}
	 *
	 * @static
	 * @throws Exception
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 * @param  array $attributes JSONobject
	 * @param  bool $onlyOneModelSearch
	 * @return array
	 */
	public static function getCameraModel(array $attributes, $onlyOneModelSearch = false)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$onlyOneModelSearch = Template::boolVal($onlyOneModelSearch);

		if (!isset($attributes["DEVIP"]) || !isset($attributes["HTTP_PORT"]))
		{
			throw new InvalidArgumentException(sprintf("attribute %s and %s should be defined", "DEVIP", "HTTP_PORT"));
		}

		session_write_close();

		$result = array();

		$camera = new Camera();

		$response = $camera->getModelAttributes($attributes, $onlyOneModelSearch);

		$result["parameters"] = array();
		$result["parameters"]["state"] = $response["parameters"]["state"];
		$result["parameters"]["ticket"] = $response["parameters"]["id"];
		$result["parameters"]["CAMERA_LIST"] = $response["parameters"]["CAMERA_LIST"];
		$result["parameters"]["SNAPSHOT_LIST"] = $response["parameters"]["SNAPSHOT_LIST"];
		$result["log"] = $response["log"];
		$result["exists"] = $response["exists"];
		$result["camexists"] = $response["camexists"];

		if ($response["parameters"]["state"] == "OK")
		{
			$result["attributes"] = $response["attributes"];
		} else {
			$errorCode = array(
				"PDE-0001" => __("Connection to a specified HTTP port failed: camera unreachable"),
				"PDE-0002" => __("Unknown camera type"),
				"PDE-0003" => __("Timeout for task execution expired"),
				"PDE-0004" => __("Interrupted because of application shut down"),
				"PDE-0401" => __("Camera authentication failed"),
				"PDE-0404" => __("Cannot retrieve camera configuration"),
			);
			throw new Exception($errorCode[$response["parameters"]["error"]]);
		}

		return $result;
	}

	/**
	 * get list of system permission types
	 *
	 * Examples:
	 *
	 * /api/call/getPermissionTypeList
	 *
	 * @static
	 * @return array("type" => array)
	 */
	public static function getPermissionTypeList()
	{
		$list = DB::select("SELECT * FROM permission_type;");
		$result = array();
		foreach ($list as $permission_type)
		{
			$result[$permission_type["permission"]] = array();
			$result[$permission_type["permission"]]["credentials"] = $permission_type["credentials"];
			$result[$permission_type["permission"]]["description"] = $permission_type["description"];
		}
		return array(
			"type" => $result
		);
	}

	/**
	 * get time interval for Elog timeline
	 * return time interval of $size seconds, or with $count events to the left or right from the
	 * starting point
	 *
	 * Examples:
	 *
	 * /api/call/getEventsInterval?start=1289475916&interval=14400&direction=left&count=100&filter={"obj":[],"tag":0,"tagOnFlag":false}
	 *
	 * @static
	 * @param  int $start starting timestamp
	 * @param  int $interval max size of interval in seconds
	 * @param  bool|string $toFuture direction of interval from starting point
	 * @param  int $count max number of events in interval
	 * @param  array $filter JSONobject
	 * @return array("start" => int, "end" => int)
	 */
	public static function getEventsInterval($start, $interval, $toFuture = true, $count = 100, array $filter = array())
	{
		date_default_timezone_set("UTC");
		$toFuture = Template::boolVal($toFuture);
		if ($toFuture) {
			$order = "eventid asc";
			$to = $start + $interval;
		} else {
			$to = $start;
			$order = "eventid desc";
			$start = $start - $interval;
		}

		$query = array(
			"utc_from >= ?",
			"utc_from < ?"
		);

		if (is_array($filter['obj']) && count($filter['obj'])) {
			$query[] = "objid IN (" . join(',', $filter['obj']) . ")";
		}
		if (isset($filter['event_type']) && ($filter['event_type'] == 0 || $filter['event_type'] == 1)) {
			$query[] = "eventtype = " . $filter['event_type'];
		}
		if (isset($filter['tag']) && $filter['tag']) {

			if (!isset($filter['tagOnFlag'])) {
				$filter['tagOnFlag'] = true;
			}
			if ($filter['tagOnFlag'])
				$query[] = "state IN (SELECT eventstate FROM eventstatetag WHERE eventtag = " . $filter['tag'] . ")";
			else
				$query[] = "state NOT IN (SELECT eventstate FROM eventstatetag WHERE eventtag = " . $filter['tag'] . ")";
		}

		$list = DB::select(
			"SELECT utc_from FROM event
			 WHERE " . join(' AND ', $query) . "
			 ORDER BY $order
			 LIMIT ?",
			array(strftime("%Y-%m-%d %H:%M:%S", $start), strftime("%Y-%m-%d %H:%M:%S", $to), $count)
		);

		$end = $to;

		if (count($list) >= $count)
		{
			$utc_from = strtotime($list[$count - 1]['utc_from']);

			if ($toFuture) {
				$end = $utc_from;
			} else {
				$start = $utc_from;
			}
		}

		return array(
			"start" => $start,
			"end" => $end
		);
	}

	/**
	 * Get events from EventLog tables.
	 * returns a list of Events in backward chronological order.
	 * Primary use of this call is to supply data to CAVU mobile devices.
	 *
	 * Examples:
	 *
	 * /api/call/getEvents?obj=123&count=50
	 *
	 * @param int $count
	 * @param int $obj
	 * @param array $filter
	 * @return array
	 */
	public static function getEvents($count = 25, $obj = 0, array $filter = array())
	{
		$userid = $_SESSION[SESSION_USERID];

		$queryStr = "
		SELECT evt.*
			FROM (
				select e.eventid, e.objid, utc_when, e.utc_from, e.utc_to-e.utc_from as length, ep.name as priority_name, ep.colorcode as priority_color, e.message, e.note, es.name as source,
					array_to_string(ARRAY(SELECT eventtag FROM eventstatetag p where p.eventstate=state), ',') as taglist
					from event e
					join eventpriority ep on ep.eventpriority=e.priority
					join eventsource es on es.eventsource=e.eventsource
			) evt
		JOIN (
			select distinct obj_res
				from _links
				where link_type='D2S'
				and obj_cons in (
					select obj_res
						from _links where link_type='S2R'
						and obj_cons in (select obj_cons from _links where obj_res=" . pg_escape_string($userid) . ")
						and permission in (select permission from permission_type where position('v' in credentials) > 0)
				)
			) objs ON objs.obj_res=evt.objid";


		if ($obj > 0)
		{
			$queryStr .= " WHERE objs.obj_res=" . pg_escape_string($obj);
		}

		if (is_array($filter) && count($filter) > 0)
		{
			$queryStr .= $obj > 0 ? " AND " : " WHERE ";
			$queryStr .= "evt.eventid IN (" . join(',', $filter) . ")";
		}

		$queryStr .= " ORDER BY 1 desc LIMIT " . pg_escape_string($count);

		$list = DB::select($queryStr);

		return array(
			"list" => $list
		);
	}

	/**
	 * get number of events from time interval
	 *
	 * Examples:
	 *
	 * /api/call/getEventsStatistics?objList=[101,102]&beginTime=432534543&endTime=653463645=granularity=hour
	 *
	 * @static
	 * @param  array $objList
	 * @param  int $beginTime
	 * @param  int $endTime
	 * @param  string $granularity
	 * @return array("statistics" => array)
	 */
	public static function getEventsStatistics(array $objList, $beginTime, $endTime, $granularity)
	{
		$g_arr = array('day','hour','min'); // List of available granularity values
		$statistics = array();
		if (!$granularity || !in_array($granularity, $g_arr))
		{
			$granularity = 'hour';
		}
		date_default_timezone_set('UTC');
		$u_from = strftime("%Y-%m-%d %H:%M:%S", $beginTime);
		$u_to = strftime("%Y-%m-%d %H:%M:%S", $endTime);
		$obj_ph = join(',', array_fill(0, count($objList), '?'));

		$list = DB::select(
			"SELECT EXTRACT(epoch FROM date_trunc('$granularity', utc_from)) AS from_trunk,
			 COUNT(*) AS cnt FROM event
			 WHERE utc_from >= date_trunc('$granularity',timestamp without time zone '$u_from')
				AND utc_from < date_trunc('$granularity',timestamp without time zone '$u_to' + interval '1 $granularity')
				AND objid in ($obj_ph)
			 GROUP BY from_trunk
			 ORDER BY from_trunk", $objList
		);
		foreach ($list as $row)
		{
			$element = array(
				"time" => $row["from_trunk"],
				"number" => $row["cnt"]
			);
			$statistics[] = $element;
		}

		return array(
			"statistics" => $statistics
		);
	}

	/**
	 * check if event is exists
	 * @param int $eventid
	 * @return bool
	 */
	public static function isEventExists($eventid)
	{
		$list = DB::select("SELECT * FROM event WHERE eventid = ?;", array($eventid));

		$isExists = false;
		if (count($list) > 0 )
		{
			$isExists = true;
		}

		return array(
			"exists" => $isExists
		);
	}

	/**
	 * get offsets list from timezone list
	 *
	 * Examples:
	 *
	 * /api/call/getTimezoneOffsetList?timezoneList=["Europe/Kiev"]
	 * @deprecated
	 * @static
	 * @param  array $timezoneList
	 * @return array array("list" => array)
	 */
	public static function getTimezoneOffsetList(array $timezoneList)
	{
		$list = array();

		$utcDateTimeZone = new DateTimeZone("UTC");
		$utcDateTime = new DateTime("now", $utcDateTimeZone);

		foreach ($timezoneList as $timezone)
		{
			if (!isset($list[$timezone]))
			{
				$currentDateTimeZone = new DateTimeZone($timezone);
				if ($currentDateTimeZone)
				{
					$timeOffset = $currentDateTimeZone->getOffset($utcDateTime);
					$list[$timezone] = $timeOffset;
				} else
				{
					$list[$timezone] = null;
				}
			}
		}
		return array(
			"list" => $list
		);
	}

	/**
	 * get offsets list and date start of DST and date end of DST from timezone list
	 * If there was no daylight saving time in a given year, for given Time Zone
	 * then returned for given year and given Time Zone:
	 *   offsetDST = offset
	 *   startDST = endDST = 0
	 *
	 * Examples:
	 *
	 * /api/call/getTimezoneInformationList?timezoneList=["Europe/Kiev"]&yearList=[2010]
	 *
	 * @static
	 * @param  array $timezoneList
	 * @param  array|null $yearList
	 * @return array array(
	 *						 "list" => array(
	 *							 "Europe/Kiev" => array(
	 *								 "2010" => array(
	 *									 "offsetDST" => int
	 *									 "offset" => int
	 *									 "startDST" => int
	 *									 "endDST" => int
	 *								 )
	 *							 )
	 *						 )
	 *					)
	 */
	public static function getTimezoneInformationList(array $timezoneList, array $yearList = null)
	{
		$currentYear = gmdate('Y');
		if (!isset($yearList)) $yearList = array($currentYear);
		$list = array();

		$utcDateTimeZone = new DateTimeZone("UTC");
		$utcDateTime = new DateTime("now", $utcDateTimeZone);

		foreach ($timezoneList as $timezone)
		{
			if (!isset($list[$timezone]))
			{
				$currentDateTimeZone = new DateTimeZone($timezone);
				if ($currentDateTimeZone)
				{
					$timeOffset = $currentDateTimeZone->getOffset($utcDateTime);
					$transitions = $currentDateTimeZone->getTransitions();
					foreach($yearList as $year){
						if (!isset($list[$timezone][$year])){
							$startDST = 0;
							$endDST = 0;
							$timeOffsetSummer = $timeOffset;
							$timeOffsetWinter = $timeOffset;
							$currentYear = $year;

							foreach ($transitions as $thedate){
								if (date('Y', $thedate['ts']) ==  $currentYear){
									if ($thedate['isdst'] == 1) {
										$startDST = $thedate['ts'];
										$timeOffsetSummer = $thedate['offset'];
									}
									else {
										$endDST = $thedate['ts'];
										$timeOffsetWinter = $thedate['offset'];
									}
								}
							}

							$list[$timezone][$year] = array(
								'offsetDST' => $timeOffsetSummer,
								'offset'=> $timeOffsetWinter,
								'startDST' => $startDST,
								'endDST' => $endDST
							);
						}
					}
				}else{
					$list[$timezone] = null;
				}
			}
		}

		return array(
			"list" => $list
		);
	}

	/**
	 * Return existing of certificate and some info from installed certificate
	 * exists - return true, if file with certificate is exists and it valid
	 * info - some info from certificate
	 *
	 * Examples:
	 *
	 * /api/call/getCertificateRequestInfo
	 *
	 * @static
	 * @throws Exception
	 * @throws AuthException
	 * @return array array("exists" => bool, "info" => array)
	 *
	 */
	public static function getCertificateRequestInfo()
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$request = Certificate::getRequestInfo();
		if (!isset($request) || $request == '')
			throw new Exception(__('Error getting request.'));
		return array(
			'exists' => $request["exists"],
			'info' => $request["info"]
		);
	}

	/**
	 * Return existing of certificate request and text of request if it exists
	 * exists - return true, if file with request is exists
	 * info - text of request for send for Certificate Organization
	 *
	 * Examples:
	 *
	 * /api/call/getCertificateInfo
	 *
	 * @static
	 * @throws Exception
	 * @throws AuthException
	 * @return array array("exists" => bool, 'info' => array)
	 */
	public static function getCertificateInfo()
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$info = Certificate::getCertificateInfo();
		if (!isset($info))
			throw new Exception(__('Error getting request.'));
		return array(
			'exists' => $info["valid"],
			'info' => $info["info"]
		);
	}

	/**
	 * Return certificate info and certificate request info
	 *
	 * Examples:
	 *
	 * /api/call/getCertificateInfo
	 *
	 * @static
	 * @return array array(
	 * array(
	 * "requestexists" => bool,
	 * "requestinfo" => array,
	 * "certificatevalid" => bool,
	 * "certificateinfo" => array,
	 * )
	 * @throws AuthException
	 */
	public static function getCertificateFullInfo()
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$requestexists = false;
		$requestinfo = '';
		$certificatevalid = false;
		$certificateinfo = '';

		$request = Certificate::getRequestInfo();
		if (isset($request) && $request != ''){
 			$requestexists = $request["exists"];
			$requestinfo = $request["info"];
		}
		$info = Certificate::getCertificateInfo();
		if (isset($info) && $info != ''){
 			$certificatevalid = $info["valid"];
			$certificateinfo = $info["info"];
		}
		return array(
			'requestexists' => $requestexists,
			'requestinfo' => $requestinfo,
			'certificatevalid' => $certificatevalid,
			'certificateinfo' => $certificateinfo,
		);
	}

	/**
	 * get storage usage
	 *
	 * Examples:
	 *
	 * /api/call/getStorageUsage?objList=[101]&granularity=hour
	 *
	 * @docBlock
	 * # Storage
	 * ## Get storage usage
	 * #### Request
	 * > GET /api/call/getStorageUsage
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"list":{"106":{"storageUsage":"","unit":"MB\/hour"}},"error":"","code":200}
	 *
	 * @static
	 * @param  array $objList list of objects ids
	 * @param  string $granularity can be hour, day
	 * @throws InvalidArgumentException
	 * @return array array("101" => array("storageUsage" => "11", "unit" => "MB/hour"))
	 */
	public static function getStorageUsage(array $objList, $granularity)
	{
		$correspondence = array(
			"hour" => array(
				"unit" => "MB/hour",
				"attribute" => "STAT_VA_RATE_HOUR"
			),
			"day" => array(
				"unit" => "MB/day",
				"attribute" => "STAT_VA_RATE_DAY"
			)
		);

		if (!isset($correspondence[$granularity]))
			throw new InvalidArgumentException(sprintf("invalid granularity=%s parameter", $granularity));

		$list = array();

		foreach ($objList as $obj)
		{
			$object = Factory::getByObj($obj);
			if (!isset($object))
				throw new InvalidArgumentException(sprintf("unknown object=%s", $obj));

			if ($object->getType($obj) != "camera")
				throw new InvalidArgumentException(sprintf("object=%s not a camera", $obj));

			$list[$obj] = array(
				"storageUsage" => $object->getAttribute($obj, $correspondence[$granularity]["attribute"]),
				"unit" => $correspondence[$granularity]["unit"]
			);
		}

		return array(
			"list" => $list
		);
	}

	/**
	 * get total size and oldest date in media archive
	 *
	 * Examples:
	 *
	 * /api/call/getStorageUsageTotals?objList=[101]
	 *
	 * @docBlock
	 * # Storage
	 * ## Get total size and oldest date in media archive
	 * #### Request
	 * > GET /api/call/getStorageUsageTotals
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"list":{"106":{"totalStorageUsage":44793328,"oldestTime":1373241600}},"error":"","code":200}
	 *
	 * @static
	 * @param  array $objList list of objects ids
	 * @throws InvalidArgumentException
	 * @return array array("101" => array("totalStorageUsage" => "11", "oldestTime" => "1289475916"))
	 */
	public static function getStorageUsageTotals(array $objList)
	{
		$list = array();

		foreach ($objList as $obj)
		{
			$object = Factory::getByObj($obj);
			if (!isset($object))
				throw new InvalidArgumentException(sprintf("unknown object=%s", $obj));

			$result = DB::select("SELECT sum(space) FROM sm_space WHERE objid = ?;", array($obj));

			$space = 0;
			$stime = 0;
			if (count($result) > 0 && isset($result[0]["sum"]))
			{
				$space = $result[0]["sum"];

				// in some cases in DB we have negative value
				if ($space < 0)
				{
					$space = 0;
				}

				$result = DB::select("SELECT round(max(EXTRACT(EPOCH from stime))) as stime FROM sm_space WHERE objid = ?;", array($obj));
				if (count($result) > 0)
				{
					$stime = $result[0]["stime"];
				}

				$STAT_VL_RATE = $object->getAttribute($obj, "STAT_VL_RATE");
				$space += (time() - $stime) * $STAT_VL_RATE;
			}
			$result = DB::select(
				"SELECT day, sum(space)
					FROM
						sm_space_spread
					WHERE
						objid = ?
					GROUP BY objid, day;",
				array($obj)
			);

			$oldestTime = mktime(0, 0, 0);
			if (count($result) > 0 && isset($result[0]["sum"]))
			{
				foreach ($result as $row)
				{
					$date = $row["day"]; // 110121
					$day = $date % 100;
					$month = ($date / 100) % 100;
					$year = ($date / 10000) % 100;
					$timestamp = mktime(0, 0, 0, $month, $day, $year);
					if ($timestamp < $oldestTime)
					{
						$oldestTime = $timestamp;
					}
				}
			}

			$list[$obj] = array(
				"totalStorageUsage" => $space,
				"oldestTime" => $oldestTime
			);
		}

		return array(
			"list" => $list
		);
	}

	/**
	 * get space utilization day-by-day
	 *
	 * Examples:
	 *
	 * /api/call/getStorageUsageSpread?objList=[101]
	 *
	 * @docBlock
	 * # Storage
	 * ## Get space utilization day-by-day
	 * #### Request
	 * > GET /api/call/getStorageUsageSpread
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"list":{"106":[{"storageUsage":"26436","timestamp":1373241600},{"storageUsage":"13590072","timestamp":1373328000},{"storageUsage":"21695872","timestamp":1373414400},{"storageUsage":"9480948","timestamp":1373500800}]},"error":"","code":200}
	 *
	 * @static
	 * @param  array $objList list of objects ids
	 * @return array array("101" => array("time" => array(array("timestamp" => "1289475916", storageUsage => "10")))
	 */
	public static function getStorageUsageSpread(array $objList)
	{
		$list = array();

		foreach ($objList as $obj)
		{
			$result = DB::select(
				"SELECT day, sum(space)
					FROM
						sm_space_spread
					WHERE
						objid = ?
					GROUP BY objid, day
					ORDER BY objid, day;",
				array($obj)
			);

			$days = array();
			foreach ($result as $row)
			{
				$date = $row["day"]; // 110121
				$day = $date % 100;
				$month = ($date / 100) % 100;
				$year = ($date / 10000) % 100;
				$timestamp = mktime(0, 0, 0, $month, $day, $year);

				$space = $row["sum"];
				if ($space < 0)
				{
					$space = 0;
				}

				$days[] = array(
					"storageUsage" => $space,
					"timestamp" => $timestamp
				);
			}

			$list[$obj] = $days;
		}

		return array(
			"list" => $list
		);
	}

	/**
	 * @param int|null $nodeid
	 * @param string|null $nodename
	 *
	 * @return array
	 */
	public static function getNodeStorageUsageTotal($nodeid = null, $nodename = null)
	{
		$node = new Node();
		$objectsFromNode = array();
		if (isset($nodename) && $nodename !== null && $nodename !== '')
		{
			$nodeid = $node->getNodeId($nodename);
		}
		if (isset($nodeid) && $nodename !== null)
		{
			$objectsFromNode = $node->getObjects($nodeid);
		}
		$objectList = array();
		foreach ($objectsFromNode as $objectFromNode){
			if ($objectFromNode["otype"] == 'D' && ($objectFromNode["subtype"] == "C" || $objectFromNode["subtype"] == "A")){
				$objectList[] = $objectFromNode["obj"];
			}
		}
		$list = API_GET::getStorageUsageTotals($objectList);
		foreach ($objectsFromNode as $objectFromNode){
			if (isset($list['list'][$objectFromNode["obj"]])){
				if (isset($objectFromNode["udid"]) && $objectFromNode["udid"] !== null && $objectFromNode["udid"] != ''){
					$list['list'][$objectFromNode["obj"]]['udid'] = '['.$objectFromNode["udid"].']';
					$list['list'][$objectFromNode["obj"]]['name'] = $objectFromNode["name"];
					$list['list'][$objectFromNode["obj"]]['fullname'] = '['.$objectFromNode["udid"].'] ' . $objectFromNode["name"];
				}else{
					$list['list'][$objectFromNode["obj"]]['udid'] = '#'.$objectFromNode["obj"].'';
					$list['list'][$objectFromNode["obj"]]['name'] = $objectFromNode["name"];
					$list['list'][$objectFromNode["obj"]]['fullname'] = '#'.$objectFromNode["obj"].' ' . $objectFromNode["name"];
				}
			}
		}
		return $list;
	}

	/**
	 * get StoragePolicy info for selected device
	 *
	 * Examples:
	 *
	 * /api/call/getObjStoragePolicyInfo?obj=101
	 *
	 * @static
	 * @param  $obj object id
	 * @return array array(storagepolicy => "1", storagertntype => "1", preservedaysmin=>'2', preservedaysmax=>'5')
	 */
	public static function getObjStoragePolicyInfo($obj)
	{
		$list = array();
		$divice = new Camera();

		$STORAGE_POLICY = $divice->getAttribute($obj,"STORAGE_POLICY");

		$result = DB::select(
			"SELECT storagepolicy, storagertntype, preservedaysmin from storagepolicy where storagepolicy = ?;",
			array($STORAGE_POLICY)
		);
		$list['storagepolicy'] = $STORAGE_POLICY;
		foreach ($result as $row)
		{
			$list['storagertntype'] = $row["storagertntype"];
			$list['preservedaysmin'] = $row["preservedaysmin"];
			$list['preservedaysmax'] = 0;
			if (isset($row["preservedaysmax"]))
				$list['preservedaysmax'] = $row["preservedaysmax"];
		}
		return array(
			"list" => $list
		);
	}

	/**
	 * get system status
	 * only for internal purposes
	 *
	 * @docBlock
	 * # System
	 * ## Getting system status
	 * #### Request
	 * > GET /api/call/getSystemStatus
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"status":"CRITICAL","description":"CRITICAL: System in a CRITICAL state.  See details for more information.","reasons":["Video Archive loss detected 1 minutes ago"],"error":"","code":200}
	 * > status can be: ONLINE, OFFLINE, STARTING, STOPPING, BROKEN, CRITICAL, ALERT
	 * > description - short description about status
	 * > reasons - list of reasons about status
	 *
	 * @static
	 * @return array("status" => string, "description" => string, "reasons" => array())
	 */
	public static function getSystemStatus()
	{
		return Node::getSystemStatus();
	}

	/**
	 * get avatar status
	 * only for internal purposes
	 *
	 * @docBlock
	 * # System
	 * ## Getting avatar status
	 * from version 1.5
	 * #### Request
	 * > GET /api/call/getAvatarStatus?obj=101
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"status":"OFFLINE","description":"OFFLINE: The application is not running (OFFLINE state). Please contact your System Administrator.","error":"","code":200}
	 * > status can be: ONLINE, OFFLINE
	 * > description - short description about status
	 *
	 * @static
	 * @param int $obj
	 * @return array("status" => string, "description" => string)
	 */
	public static function getAvatarStatus($obj)
	{
		$avatar = new Avatar();
		$type = $avatar->getType($obj);
		if (!isset($type) || $type != "avatar")
		{
			throw new InvalidArgumentException(__("Incorrect obj"));
		}

		$status = $avatar->getStatus($obj);
		return array(
			"status" => $status["status"],
			"description" => $status["description"]
			//"reasons" => $status["reasons"]
		);
	}

	/**
	 * get audit records
	 *
	 * @static
	 * @param null $from
	 * @param null $to
	 * @param array $categories
	 * @param int|null $userid
	 * @param int|null $objid
	 * @param null $msgFilter
	 * @param int $pagesize
	 * @param int $pagenum
	 * @param bool $countRows
	 * @param bool $includeDigest
	 * @return array
	 */
	public static function getAuditRecords($from = null, $to = null, array $categories = array(),
			$userid = null, $objid = null, $msgFilter = null,
			$pagesize = 0, $pagenum = 0, $countRows = false, $includeDigest = false)
	{
		$list = Audit::getRecords($from, $to, $categories, $userid, $objid, $msgFilter,
			$pagesize, $pagenum, $countRows, $includeDigest);

		for($i = 0; $i < count($list['data']); $i++){
			$list['data'][$i]['parameters'] = json_decode($list['data'][$i]['parameters']);
		}

		return array(
			"list" => $list
		);
	}

	/**
	 * get audit categories
	 *
	 * @static
	 * @return array
	 */
	public static function getAuditCategories()
	{
		$list = Audit::getCategories();

		return array(
			"list" => $list
		);
	}

	public static function getEventPriorities()
	{
		return array(
			"list" => DB::select("Select * FROM eventpriority ORDER BY eventpriority")
		);
	}

	/**
	 * get parameters of qtCam cameras on node
	 * @static
	 *
	 * @param string $type
	 *
	 * @return array array("list" => array)
	 * @throws AuthException
	 * @throws Exception
	 */
	public static function getqtCamParametersList($type)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$node = new Node();
		return array("list" => $node->getqtCamParametersList($type));
	}

	/**
	 * get name of qtCam camera snapshot
	 * @static
	 *
	 * @param  string $id unique
	 *
	 * @return array array("SNAPSHOT" => string)
	 * @throws AuthException
	 * @throws Exception
	 */
	public static function getqtCamCameraPicture($id)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$node = new Node();
		return array("SNAPSHOT" => $node->getqtCamCameraPicture($id));
	}

	/**
	 * get server time (UTC)
	 *
	 * @docBlock
	 * # System
	 * ## Get server time
	 * #### Request
	 * > GET /api/call/getServerTime
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"servertime":1373554305,"error":"","code":200}
	 *
	 * @static
	 * @return array ("servertime" => UNIX_timestamp)
	 */
	public static function getServerTime()
	{
		return array("servertime" => time());
	}

	/**
	 * @static
	 *
	 * @param int|null $scheduleid
	 *
	 * @return array
	 * @throws AuthException
	 */
	public static function getObjectOnScheduleList($scheduleid = null)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$schedule = new Schedule();
		$list = $schedule->getObjectOnScheduleList($scheduleid);
		return array(
			"list" => $list
		);
	}

	/**
	 * @static
	 *
	 * @param int $scheduleid
	 *
	 * @return array
	 * @throws AuthException
	 */
	public static function getUnusedPosturesList($scheduleid)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$posture = new Posture();
		$list = $posture->getUnusedPosturesList($scheduleid);
		return array(
			"list" => $list
		);
	}

	/**
	 * @static
	 *
	 * @param int|null $scheduleid
	 *
	 * @return array
	 * @throws AuthException
	 */
	public static function getScheduleInfo($scheduleid = null)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$schedule = new Schedule();
		$list = $schedule->getScheduleInfo($scheduleid);
		return array(
			"list" => $list
		);
	}

	/**
	 * @static
	 *
	 * @param int $scheduleid
	 * @param $scheduleInfo
	 * @param $datefrom
	 * @param $dateto
	 *
	 * @return array
	 * @throws AuthException
	 */
	public static function getEventSource($scheduleid, $datefrom, $dateto)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$schedule = new Schedule();
		// convert $datefrom, $dateto from timestamp to DateTime
		$dtfrom = new DateTime();
		$dtfrom->setTimestamp($datefrom);
		$dtto = new DateTime();
		$dtto->setTimestamp($dateto);

		$scheduleInfo = self::getScheduleInfo($scheduleid);

		$list = $schedule->getEventSource($scheduleid, $scheduleInfo, $dtfrom, $dtto);

		return array(
			"list" => $list
		);
	}

	/**
	 * @static
	 * @return array
	 */
	public static function getAnalyticsSchema()
	{
		return array(
			"list" => str_replace(array("\n", "\t"), "", file_get_contents(APL_PATH . '/analytics/etc/analytics_schema.json'))
		);
	}

	/**
	 * get list of sessions on server
	 *
	 * @static
	 * @param  bool $isAll return all sessions without filtering based on credentials management
	 * @throws AuthException
	 * @return array
	 */
	public static function getSessionList($isAll = false)
	{
		$isAll = Template::boolVal($isAll);

		$user = new User();
		if ($isAll && !$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$userid = null;
		if (!$isAll)
		{
			$userid = $_SESSION[SESSION_USERID];
		}
		$sessionManager = new SessionManager();
		$sessions = $sessionManager->getList($userid);

		$list = array();
		foreach ($sessions as $sessionid => $session)
		{
			$list[$sessionid] = array(
				"age" => $session["age"],
				"content" => array(
					"status" => $session["content"]["status"],
					"user_agent" => Purifier::clean($session["content"]["user_agent"]),
					"user_ip" => $session["content"]["user_ip"],
					"userid" => $session["content"]["userid"],
					"username" => $session["content"]["username"]
				)
			);
		}

		return array(
			"list" => $list
		);
	}

	/**
	 * @docBlock
	 * # Object management
	 * ## Get video url authorization information
	 * #### Request
	 * > GET /api/call/authorizationManager
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"result":{"authid":"c99dbbab890856416159524013565b653023","url":"rtsp:\/\/192.168.0.117:8554\/xmedia?dev=102&objid=102","param":{"MEDIA_TRANSPORT":"tcp","DEVICE_TYPE":"C","TIME_ZONE":"UTC","FRAMERATE":"4","PIXEL_ASPECT_RATIO":"auto","POSITIONCTL":"Axis"}},"error":"","code":200}
	 *
	 * @param int $objid camera object id
	 * @param string $streamtype type of stream, can be 'live', 'archive', 'snapshot'
	 * @param string $player
	 * @param string|null $analytics
	 * @throws Exception
	 * @return array
	 */
	public static function authorizationManager($objid, $streamtype, $player = "MPLAYER", $analytics = null)
	{
		// Check session/required parameters
		$streamtypeList = array('live', 'archive', 'snapshot');
		if (!in_array($streamtype, $streamtypeList))
		{
			throw new Exception("Missing or invalid parameter 'streamtype'");
		}

		// Translate parameters
		$params = array(
			"PLAYER" => strtoupper($player),
			"ANALYTICS" => strtoupper($analytics)
		);
		foreach ($_GET as $param => $value)
		{
			if ($param == 'time_start' || $param == 'time_end')
			{
				$value = str_replace('_', ' ', $value);
			}
			$params[strtoupper($param)] = $value;
		}

		if (isset($params['STREAMNUM']) && $params['STREAMNUM'] == 0)
		{
			unset($params['STREAMLOCAL']);
		}

		// Apply some algorithms
		$videoHelper = new VideoHelper();
		$videoHelper->getDeviceInfo($params);
		if (!$videoHelper->checkMediaCredentials($params))
		{
			throw new AuthException("Not enough rights");
		}
		$videoHelper->getMediaUrl($params);
		$videoHelper->getAuthorizationId($params);

		// /api/call/authorizationManager?return=mediastreamauth&player=MPLAYER&objid=147&streamtype=live&streamnum=1

		$result = array(
			"authid" => $params['AUTHID'],
			"url" => $params['URL']
		);

		$outParams = array('MEDIA_TRANSPORT', 'DEVICE_TYPE', 'TIME_ZONE', 'FRAMERATE', 'PIXEL_ASPECT_RATIO', 'POSITIONCTL', 'MULTICAST_STREAM');
		foreach ($outParams as $i => $outParamName)
		{
			if (isset($params[$outParamName]))
			{
				$result["param"][$outParamName] = $params[$outParamName];
			}
		}

		return array(
			"result" => $result
		);
	}

	/**
	 * get camera log
	 *
	 * @param  int $obj
	 *
	 * @return array
	 * @throws AuthException
	 */
	public static function getCameraLog($obj)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$camera = new Camera();
		$list = $camera->getLog($obj);

		return array(
			"list" => $list
		);
	}

	/**
	 * get describe-info about postures and timecards on schedule
	 *
	 * @param int $scheduleid
	 * @param bool $useDefault
	 * @param bool $allPostures
	 * @param bool $actualOnly
	 *
	 * @return array
	 * @throws AuthException
	 */
	public static function getPosturesInfo($scheduleid, $useDefault = false, $allPostures = false, $actualOnly = false)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$schedule = new Schedule();
		$list = $schedule->getPosturesInfo($scheduleid, $useDefault, $allPostures, $actualOnly);

		return array(
			"list" => $list
		);
	}

	/**
	 *
	 * @docBlock
	 * # Storage
	 * ## Get storage coverage
	 * #### Request
	 * > GET /api/call/getStorageCoverage
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"list":{"119":{"1373029740":"100","1373029800":"100","1373029860":"100","1373029920":"100","1373029980":"100","1373030040":"100","1373030100":"100","1373030160":"100","1373030220":"100","1373030280":"100","1373030340":"100","1373030400":"100","1373030460":"100","1373030520":"63"}},"error":"","code":200}
	 *
	 * @param array $objList list of obj id's
	 * @param int $startTime Unix timestamp in seconds
	 * @param int $endTime Unix timestamp in seconds
	 * @param string $granularity min, hour, day
	 * @param int $streamNumber 1 - full image, 0 - digest
	 * @throws InvalidArgumentException
	 * @return array
	 */
	public static function getStorageCoverage(array $objList, $startTime, $endTime, $granularity, $streamNumber = 1)
	{
		$granularityList= array("min", "hour", "day");
		if (!in_array($granularity, $granularityList))
		{
			throw new InvalidArgumentException("granularity can ba only min, hour, day");
		}

		if (!is_numeric($startTime) || !is_numeric($endTime) || !is_numeric($streamNumber))
		{
			throw new InvalidArgumentException("startTime, endTime, streamnum should be a numbers");
		}

		$list = array();

		session_write_close();

		foreach ($objList as $obj)
		{
			$list[$obj] = Node::getStorageCoverage($obj, $startTime, $endTime, $granularity, $streamNumber);
		}

		return array(
			"list" => $list
		);
	}

	/**
	 * @param string $section
	 * @return array
	 * @throws InvalidArgumentException
	 */
	public static function getVersionControlInformation($section)
	{
		if (!in_array($section, array("info", "history")))
		{
			throw new InvalidArgumentException("unknown section");
		}

		return array(
			"section" => array(
				$section => Node::getInfo()
			)
		);
	}

	/**
	 * @deprecated
	 * @return array
	 * @throws AuthException
	 * @throws Exception
	 */
	public static function getAxisCameraList()
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$node = new Node();
		return array(
			"CAMERAS" => $node->getAxisCameraList()
		);
	}

	/**
	 * @deprecated
	 * @return array
	 * @throws AuthException
	 */
	public static function getCameraModelList()
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$camera = new Camera();
		return array(
			"list" => $camera->getModelList()
		);
	}

	/**
	 * Returns VCA license and usage info for each node
	 *
	 * @static
	 * @return array
	 * @throws AuthException
	 */
	public static function getVCADetails()
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$list = array();
		$node = new Node();
		$nodelist = $node->getListWithNumberOfCameras(true);

		foreach ($nodelist as $row) {
			$attr = $row["attributes"];

			$queryString = "SELECT COUNT(*) as total
					   FROM _objs o, _obj_attr a, _obj_attr at, _obj_attr attr
					   WHERE o.otype='D' AND o.subtype='C' AND o.deleted=0
						AND o.node_id=? AND a.obj=o.obj AND a.attr='VAE_ACTIVE' AND a.val='yes'
						AND at.obj=o.obj and at.attr='VAE_VCA_ACTIVE' and at.val='yes'
						AND attr.obj=o.obj and attr.attr='ARCHSTATE' and attr.val='on'";
			$vca_count = DB::select($queryString, array($row["obj"]));

			$list[] = array (
				"obj" => $row["obj"],
				"uni" => $row["uni"],
				"location" => $row["location"],
				"vca_version" => isset($attr["SSVA_VCA_VERSION"]) ?
							$attr["SSVA_VCA_VERSION"] : 'N/A',
				"vca_channels" => isset($attr["SSVA_VCA_CHANNELS"]) ?
							$attr["SSVA_VCA_CHANNELS"] : 'N/A',
				"vca_active" => $vca_count[0]["total"]
			);
		}

		return array("list" => $list);
	}

	/**
	 * Returns OV6 license and usage info
	 *
	 * @static
	 * @return array
	 * @throws AuthException
	 */
	public static function getOV6Details()
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$list = array();
		$node = new Node();
		$nodelist = $node->getListWithNumberOfCameras(true);

		$LICENSE_OV6_KEYPACK = json_decode(Identity::getAttribute("LICENSE_OV6_KEYPACK"), true);

		foreach ($nodelist as $row) {
			$cameras = [];
			if($LICENSE_OV6_KEYPACK[$row["uni"]]["features"][0]["name"] != "N/A"){
				foreach ($LICENSE_OV6_KEYPACK[$row["uni"]]["features"] as $key => $feature) {
					if(isset($feature["ch_ids"])){
						$cameras = array_merge($cameras, $feature["ch_ids"]);
					}
				}
			}

			$ov6_cameras = [];
			if(count($cameras) > 0){
				$queryString = "SELECT o.obj, o.name
					   FROM _objs o
					   WHERE o.otype='D' AND o.subtype='C' AND o.deleted=0
						AND o.obj IN (".implode(",", $cameras).")
						AND o.node_id=?";
				$ov6_cameras = DB::select($queryString, array($row["obj"]));
			}

			$list[] = array (
				"obj" => $row["obj"],
				"uni" => $row["uni"],
				"location" => $row["location"],
				"cameras" => $ov6_cameras
			);
		}

		return array(
			"list" => $list,
			"LICENSE_OV6_KEYPACK" => $LICENSE_OV6_KEYPACK
		);
	}

	/**
	 * get targets for last seconds
	 *
	 * @return array
	 */
	public static function getLastTargets()
	{
		$list = DB::select("
			SELECT
				id, time, objid, targetid, targettype, targetsource, targeteleventid, targetlatitude, targetlongitude, targetaltitude
			FROM
				gismo_data
			WHERE
				deleted = 0
				AND time > date_part('epoch', now()) * 1000 - 15000;");

		return array(
			"list" => $list
		);
	}

	/**
	 * @param string $targetId
	 * @return array
	 */
	public static function getTargetRelatedZone($targetId)
	{
		$targetWatcher = new TargetWatcher();

		return $targetWatcher->getTargetRelatedZone($targetId);
	}

	/**
	 * @param array $source
	 * @param array $destination
	 *
	 * @return array
	 * @throws AuthException
	 * @throws Exception
	 */
	public static function findHomography(array $source, array $destination)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$node = new Node();
		$homography = $node->findHomography($source, $destination);
		return array("homography" => $homography);
	}

	/**
	 * @param string $provider
	 *
	 * @return array
	 * @throws AuthException
	 */
	public static function getCloudStorageStatus($provider)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$status = "offline";

		switch ($provider)
		{
			case "S3":
			case "VCS":
				$attributes = Identity::getAttributes();
				$keyid = $attributes["CLOUD_KEY_ID"];
				$secret = $attributes["CLOUD_SECRET_KEY"];
				$date = date("r"); // Must include special header
				$buf = "GET\n\n\n$date\n/";
				$enc = base64_encode(hash_hmac("sha1", $buf, $secret, true));

				if ($keyid && $secret)
				{
					$host = $provider === "S3" ? "s3.amazonaws.com" : "storage-iad3a.cloud.verizon.com";
					$ch = curl_init();
					curl_setopt_array($ch, array(
						CURLOPT_URL => "https://$host",
						//CURLOPT_VERBOSE => true,
						CURLOPT_RETURNTRANSFER => true,
						CURLOPT_TIMEOUT => 10,
						CURLOPT_HTTPHEADER => array(
							"Date: $date",
							"Authorization: AWS $keyid:$enc"
						)
					));

					$data = curl_exec($ch);
					$code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
					curl_close($ch);

					if ($code === 200)
					{
						$status = "online";
					} else {
						$status = "offline";
					}
				}

				break;
		}

		return array(
			"status" => $status
		);
	}

	/**
	 * @return array
	 */
	public static function getAnnounceEventList()
	{
		$userid = $_SESSION[SESSION_USERID];

		$announce = new Announce();
		$list = $announce->getList($userid);

		return array(
			"list" => $list
		);
	}

	/**
	 * @return array
	 */
	public static function countAnnouncedEvents()
	{
		$userid = $_SESSION[SESSION_USERID];

		$announce = new Announce();
		$amount = $announce->countEvents($userid);

		return array(
			"amount" => $amount
		);
	}

	/**
	 * @return array
	 */
	public static function getEventInfo($eventId)
	{
		$announce = new Announce();
		$event = $announce->getEventInfo($eventId);

		return array(
			"event" => $event
		);
	}

	/**
	 * @return array
	 */
	public static function getAnnounceRoleUserList()
	{
		$userid = $_SESSION[SESSION_USERID];

		$list = array(
			"role" => array(),
			"user" => array()
		);

		$user = new User();
		$roleList = $user->getRoles($userid);
		foreach ($roleList as $userRole)
		{
			$list["role"][] = $userRole;

			$role = new Role();
			$userList = $role->getUsers($userRole["obj"]);
			foreach ($userList as $user)
			{
				$list["user"][] = $user;
			}
		}

		return array(
			"list" => $list
		);
	}

	/**
	 *
	 * @docBlock
	 * # Object management
	 * ## Get camera downtime
	 * #### Request
	 * > GET /api/call/getCameraDowntime
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"error": "","code":200,"list":{"101":{"downtime":250,"details":[{"start":1411060120,"end":1411060150,"status":3},{"start":1411060200,"end":1411060420,"status":2}]},"101" : {"downtime":12,"details":[{"start":1411061218,"end":1411061230,"status":1 }] } }}
	 *
	 * @param array $objList list of obj id's
	 * @param int $startTime Unix timestamp in seconds
	 * @param int $endTime Unix timestamp in seconds
	 * @param int|null $status Camera status to filter by (0 - on, 1 - off, 2 - off by schedule, 3 - broken)
	 * @param bool $details If "true" all downtime intervals for specified period will be returned, otherwise only total downtime duration
	 * @throws InvalidArgumentException
	 * @return array
	 */
	public static function getCameraDowntime(array $objList, $startTime, $endTime, $status = null, $details = true)
	{
		$details = Template::boolVal($details);
		$list = array();
		$phd_str = preg_replace("/\d+/", "?", join(",", $objList));
		$status_cond = "";
		$params = $objList;
		$params[] = $startTime;
		$params[] = $endTime;
		if (isset($status))
		{
			if ($status > 3 || $status < 0)
			{
				throw new InvalidArgumentException(sprintf("unknown status=%s", $status));
			}
			$status_cond = "AND status=?";
			$params[] = $status;
		}
		// Init resulting list with zero values for all cameras
		foreach ($objList as $obj) {
			$list[$obj] = array("downtime" => 0);
		}
		// Simply count total amount of time
		$result = DB::select("
			SELECT
				objid,
				sum(extract(epoch FROM (coalesce(utc_to,now() at time zone 'UTC') - utc_from)::interval)::int) as total
			FROM
				camera_downtime
			WHERE
				objid IN ($phd_str)
				AND extract('epoch' from coalesce(utc_to,now() at time zone 'UTC')) > ?
				AND extract('epoch' from utc_from) < ?
				$status_cond
			GROUP BY
				objid;",
			$params
		);
		foreach ($result as $row)
		{
			$list[$row["objid"]]["downtime"] = (int)$row["total"];
		}
		//Fetch all matching periods from table
		if ($details === true)
		{
			foreach ($list as $objid => &$dtime)
			{
				$dtime["details"] = array();
			}

			$result = DB::select("
				SELECT
					objid, extract(epoch FROM utc_from)::int as start,
					extract(epoch FROM coalesce(utc_to,now() at time zone 'UTC'))::int as end, status
				FROM
					camera_downtime
				WHERE
					objid IN ($phd_str)
					AND extract('epoch' from coalesce(utc_to,now() at time zone 'UTC')) > ?
					AND extract('epoch' from utc_from) < ?
					$status_cond
				",
				$params
			);
			foreach ($result as $row)
			{
				if (!isset($list[$row["objid"]]))
				{
					continue;
				}
				$list[$row["objid"]]["details"][] = array(
					"start" => (int)$row["start"],
					"end" => (int)$row["end"],
					"status" => (int)$row["status"]
				);
			}
		}

		return array(
			"list" => $list
		);
	}

	/**
	 *
	 * @docBlock
	 * # Object management
	 * ## Get camera downtime summarized by status with specified granularity
	 * #### Request
	 * > GET /api/call/getCameraDowntimeSummary
	 * #### Response
	 * > 200
	 * > Content-Type: application/json
	 * > {"list":{"101":{"1414882800":{"3":27}},"103":{"1414882800":{"3":47,"1":185}}},"error":"","code":200}
	 *
	 * @param array $objList list of obj id's
	 * @param int $startTime Unix timestamp in seconds
	 * @param int $endTime Unix timestamp in seconds
	 * @param string $granularity hour/day/month
	 * @throws InvalidArgumentException
	 * @return array
	 */
	public static function getCameraDowntimeSummary(array $objList, $startTime, $endTime, $granularity)
	{
		$g_arr = array('hour', 'day', 'month'); // List of available granularity values
		if (!is_numeric($startTime) || !is_numeric($endTime))
		{
			throw new InvalidArgumentException("startTime and endTime should be numeric");
		}
		if (!$granularity || !in_array($granularity, $g_arr))
		{
			throw new InvalidArgumentException(sprintf("granularity '%s' not supported", $granularity));
		}
		foreach ($objList as $obj)
		{
			if (!is_numeric($obj))
			{
				throw new InvalidArgumentException(sprintf("not a numeric value in objList: '%s'", $obj));
			}
		}

		$list = array();
		$objs_str = join(",", $objList);

		$params = array();
		$params[] = $startTime;
		$params[] = $endTime;
		$params[] = $granularity;

		$result = DB::select("
			SELECT
				objid, ts_from, duration, status
			FROM
				getCameraDowntimeSummary(E'\{$objs_str\}', ?, ?, ?)
			ORDER BY
				objid, ts_from",
			$params
		);

		foreach ($result as $row)
		{
			$objid = $row["objid"];
			$ts_start = $row["ts_from"];
			$duration = $row["duration"];
			$status = $row["status"];
			if (!isset($list[$objid]))
			{
				$list[$objid] = array();
			}
			if (!isset($list[$objid][$ts_start]))
			{
				$list[$objid][$ts_start] = array();
			}
			if (!isset($list[$objid][$ts_start][$status]))
			{
				$list[$objid][$ts_start][$status] = 0;
			}
			$list[$objid][$ts_start][$status] += $duration;
		}

		// For cameras with no downtime set empty hashes in resulting json
		foreach ($objList as $obj) {
			if (!isset($list[$obj])) {
				$list[$obj] = new stdClass;
			}
		}

		return array(
			"list" => $list
		);
	}

	/**
	 * test connection to LDAP server
	 *
	 * @throws AuthException
	 * @throws Exception
	 */
	public static function testLDAP()
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$identity = Identity::getAttributes();
		$LDAP_AUTHENTICATION = Template::boolVal($identity["LDAP_AUTHENTICATION"]);
		if (!$LDAP_AUTHENTICATION)
		{
			throw new Exception("LDAP authentication not activated");
		}

		$ldap = new LDAP();
		$ldap->auth($identity["LDAP_USER"], $identity["LDAP_PASSWORD"]);
		$groupList = $ldap->getGroups();
		$currentUserGroupList = $ldap->getGroups($user->getName($_SESSION[SESSION_USERID]));

		return array(
			"groupList" => $groupList,
			"currentUserGroupList" => $currentUserGroupList
		);
	}

	/**
	 * @param int $id
	 *
	 * @return array
	 * @throws AuthException
	 */
	public static function getEventSourceParametersList($id)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$eventpolicy = new EventPolicy();

		$list = array(
			'eventlifespan' => $eventpolicy->getEventPolicyStorageEventLifeSpan($id),
			'eventsourceparameters' => $eventpolicy->getEventSourceParametersList($id),
			'eventsourceparameters_list' => array(
				'priority' => $eventpolicy->getEventPriorityList(),
				'lifespan' => $eventpolicy->getEventLifeSpanList(),
				'eventtype' => $eventpolicy->getEventType()
			)
		);

		return $list;
	}

	/**
	 * @return array
	 * @throws AuthException
	 */
	public static function getEventPolicyList()
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$eventpolicy = new EventPolicy();
		return array(
			"list" => $eventpolicy->getEventPolicyList()
		);
	}

	/**
	 * @param int $eventpolicy_id
	 * @param int $eventsource_id
	 *
	 * @return array
	 * @throws AuthException
	 * @throws InvalidArgumentException
	 */
	public static function getEventSourceWorkflowList($eventpolicy_id, $eventsource_id)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		if (!isset($eventpolicy_id) || !isset($eventsource_id))
		{
			throw new InvalidArgumentException(__("eventpolicy_id and eventsource_id should be defined"));
		}

		$eventpolicy = new EventPolicy();

		$list = $eventpolicy->getEventSourceWorkflowList($eventsource_id);
		$workflowList = $eventpolicy->getWorkflowForEventPolicyContent($eventpolicy_id, $eventsource_id);
		if (count($workflowList) > 0)
		{
			array_push($list, $workflowList[0]);
		} else {
			array_push($list, "");
		}

		return array(
			"list" => $list
		);
	}

	/**
	 * @param int $id
	 *
	 * @return array
	 * @throws AuthException
	 */
	public static function getEventPolicy($id)
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$eventpolicy = new EventPolicy();

		return array(
			"list" => $eventpolicy->getEventPolicy($id)
		);
	}

	/**
	 * @return array
	 * @throws AuthException
	 * @throws Exception
	 */
	public static function hasConnectToLicenseServer()
	{
		$user = new User();
		if (!$user->isAdmin($_SESSION[SESSION_USERID]))
		{
			throw new AuthException("Not enough rights");
		}

		$license = new License();
		return array(
			"result" => $license->hasConnectToLicenseServer()
		);
	}

	/**
	 * @return array
	 * @throws AuthException
	 */
	public static function getLicenseAttributesList()
	{
		$user = new User();
		if (!$user->checkObjectCredentials($_SESSION[SESSION_USERID], Obj::CONTROL_PANEL, "s"))
		{
			// if (!ManageUser::hasRole(11)){
			// FileLog::info(__("Tried to get License Attributes List, but did not have credentials"));
			return array("error" => __("Not enough rights"));
		}

		$license = new License();

		return array(
			"attributes" => $license->getLicenseAttributesList()
		);
	}
	
	/**
	 * @return array
	 */
	public static function getContentDeliveryDetails(array $objs)
	{
		$user = new User();
		$userDevs = array();
		foreach($objs as $obj){
			if (!$user->checkObjectCredentials($_SESSION[SESSION_USERID], $obj, "A")){
				continue;
			}
			array_push($userDevs, $obj);
		}
		$result = array();
		if(count($userDevs) > 0){
			$query = "SELECT *,extract(epoch from tstamp at time zone 'UTC') as timestamp FROM content_delivery_history WHERE objid IN (".implode(',', $userDevs).");";
			$result = DB::select($query);
		}
			
		return array(
			"details" => $result
		);
	}
	
	/**
	 * @static
	 * @return array
	 */
	public static function getSMWriterConf()
	{
		$sm_writer = explode("\n", file_get_contents(APL_PATH . '/sm/etc/sm_writer.conf'));
		
		$params = [];
		foreach($sm_writer as $param){
			if($param[0] != "#" && !empty($param)){
				$param = explode("=", $param);
				$params[$param[0]] = $param[1];
			}
		}
		
		return array(
			"list" => $params
		);
	}
}
