
// ///////////////////////////////
// couple of functions providing sync time method for multiple PlayerGroup instances

var __playerGroups = [];
var playerGroupSyncInterval = 200; // how often sync function runs (msec)

function registerPlayerGroup(playerGroup)
{
	__playerGroups.push(playerGroup);
}

function __groupSync()
{
	for(var i = 0; i < __playerGroups.length; i++) {
		__playerGroups[i].onSync();
	}
	setTimeout(__groupSync, playerGroupSyncInterval);
}
__groupSync();

/**
 * Data structure used for storing sync info
 *
 * @param {number} TS
 * @param {number} RTS
 * @param {number} speed
 * @constructor
 */
function SyncData(TS, RTS, speed)
{
  this.RTS = RTS;      // RTS when image was updated with TS
  this.TS = TS;        // player TS as last sync
  this.speed = speed;  // speed computed at last sync
  this.reset = function() { this.TS = 0; this.speed = null; }
}

/**
 * Class PlayerInfo
 *
 * @param player
 * @constructor
 */
function PlayerData(player)
{
	this.player = player;
	this.selectTS = 0;
	this.syncStatus = 0; //0-no sync, 1-positioning, 2-positioning ready, 3-RTS sync, 4-RTS sync ready, 5 - TS sync, 6-TS ready
	this.syncTS = null;   //  TS sync point (depends on syncStatus 3 and 4)
	this.syncRTS= null;  // RTS sync point (depends on syncStatus 3 and 4)
	this.syncData = new SyncData(null, null, null);

	this.reset = function() {
		this.syncStatus = 0;
		this.syncTS = null;
		this.syncRTS = null;
		this.syncData.reset();
	}
}

/**
 * Class PlayerGroup
 *
 * @param {string} id
 * @constructor
 */
function PlayerGroup(id)
{
	// interface
	this.setContext = elpe_setContext;
	this.play = elpe_play;
	this.playOneFrame = elpe_playOneFrame;
	this.playOneFrameTS = elpe_playOneFrameTS;
	this.pause = elpe_pause;
	this.stop = elpe_stop;
	this.getCommand = function() { return (this.command.parent == null) ? this.command : this.command.parent; };
	this.getCurrentTS = elpe_getCurrentTS;

	this.setParameter = elpe_setParameter;
	this.getParameter = elpe_getParameter;
	this.isReversePlaybackEnabled = elpe_isReversePlaybackEnabled;
	this.setListener = function(listener) { this.listener = listener; };
	this.removeListener = function() { this.listener = null; };

	this.addPlayer = elpe_addPlayer;
	this.removePlayer = elpe_removePlayer;
	this.isPlayerInGroup = function(mplayer) { return (this.getPlayerPos(mplayer) != -1); };
	this.getGroupSize = function() { return this.players.length; };
	this.getNumOfAudioDevices = elpe_getNumOfAudioDevices;
	this.getPlayer = function(index) { return (index < this.players.length) ? this.players[index].player : null };
	this.reset = elpe_reset;
	this.setAudio = elpe_setAudio;
	this.isAudioOn = else_isAudioOn;

	// properties
	this.id = id;
	this.contextType = 0; // 0 - unknown, 1-static.img, 2-live, 3-archive
    this.archiveInterval = { startTS : null, endTS : null };
	this.status = 3;

	// implementation (protected members) /////////////////////////////////////////
	this.players = [];
	this.listener = null;
	this.syncEngine = null;
	this.command = (new Command()).stop();
	this.parameters	= [];
	this.primaryPlayerPos = null;
	this.isRPlaybackEnabled = false;

	this.setStatus = elpe_setStatus;
	this.getPlayerPos = elpe_getPlayerPos;
	this.setContextImpl = elpe__setContextImpl;
	this.addPlayer2List = elpe_addPlayer2List;
	this.getPrimaryPlayerPos = elpe_getPrimaryPlayerPos;
	this.getPlayerCatchUpCommand = elpe_getPlayerCatchUpCommand;
	this.checkRPlayback = elpe_checkRPlayback;
	this.log = function(func, message) {}; //elpe_log;

	// player callbacks
	this.onSync = elpe_onSync;
	this.onContextChange= elpe_playerOnContextChange;
	this.onNewImage = elpe_playerOnNewImage;
	this.onStatusChange = elpe_playerOnStatusChange;
	this.onBufferChange = elpe_playerOnBufferChange;
	this.onClick = elpe_playerOnClick;
	this.onSnapshotFinish = elpe_onSnapshotFinish;

	// init
	registerPlayerGroup(this);
}

//
// params: contextType [, ts_from, ts_to]
//
function elpe_setContext(contextType)
{
	if (contextType == 0 || contextType == 1 || contextType == 2 || contextType == 3 && elpe_setContext.arguments.length >= 3) {
		this.setContextImpl(contextType,
			(contextType == 3) ? elpe_setContext.arguments[1] : null,
			(contextType == 3) ? elpe_setContext.arguments[2] : null,
			null);
	}
	else {
		throw new Error("setContext() error: invalid context provided");
    }
}

//
// params: contextType [, ts_from, ts_to]
//
function elpe__setContextImpl(contextType, ts_from, ts_to, exceptPlayerIndex)
{
	//if (contextType == 3) {
	//	this.log('setContext', 'contextType=' + contextType + ' <' + dt_formatIntTs(ts_from, "YYYY-MM-DD", "hh:mm:SS.ss") +
	//	          '>..<' + dt_formatIntTs(ts_to, "YYYY-MM-DD", "hh:mm:SS.ss") + '>');
	//}
	//else {
	//	this.log('setContext', 'contextType=' + contextType);
	//}

	// stop all players
	if (this.contextType != 0) {
        this.stop();
	}

	// set sync engine
	if (this.contextType != contextType) {
		if (contextType == 2) { this.syncEngine = new LiveSyncImpl(this); }
		else if (contextType == 3) { this.syncEngine = new ArchiveSyncImpl(this); }
	    else { this.syncEngine = null; }
	}

	// set group context
	this.contextType = contextType;
	this.archiveInterval.startTS = (ts_from == null) ? null : Math.floor(ts_from/1000)*1000;
	this.archiveInterval.endTS = (ts_to == null) ? null : Math.ceil(ts_to/1000)*1000;

	// apply new context for all players
	for(var i = 0; i < this.players.length; i++) {
		if (i == exceptPlayerIndex) {
			continue;
		}
		this.players[i].player.setContext(
				this.players[i].player.devObjId,
				this.contextType, this.archiveInterval.startTS, this.archiveInterval.endTS);
	}

	// invoke callback
	if (this.listener != null) { this.listener.onContextChange(this); }
}

//
// addPlayer(mplayer[, devObjId])
//

function elpe_addPlayer(mplayer)
{
	var devObjId = (elpe_addPlayer.arguments.length >= 2)? elpe_addPlayer.arguments[1] : mplayer.devObjId;
	//this.log('addPlayer', 'id=' + mplayer.devObjId + ' devObjId=' + devObjId + ' contextType=' + mplayer.contextType);

	// check if given instance of player is already included into group
	var index = this.getPlayerPos(mplayer);
	if (index == -1) {
		// player is not in group
		if (mplayer.devObjId != devObjId) {
			mplayer.setContext(devObjId, mplayer.contextType, mplayer.archiveInterval.startTS, mplayer.archiveInterval.endTS);
		}

		// adding player to the group
		if (this.contextType == 0 && mplayer.contextType != 0) {
			// propagate player context to group
			this.setContextImpl(mplayer.contextType, mplayer.archiveInterval.startTS, mplayer.archiveInterval.endTS);
			if (mplayer.playerImplStatus != 3) {
				// player is doing something while group is stopped
				// join group with player
				var groupCommand = this.getPlayerCatchUpCommand(mplayer);
				this.addPlayer2List(mplayer);
				if (this.syncEngine != null) {
					this.syncEngine.submitCommand(groupCommand);
				}
			}
			else {
				this.addPlayer2List(mplayer);
			}
		}
		else {
			// set group context to player
			mplayer.setContext(devObjId, this.contextType, this.archiveInterval.startTS, this.archiveInterval.endTS);
			this.addPlayer2List(mplayer);
			if (this.syncEngine != null) {
				this.syncEngine.catchUpWithGroup(mplayer);
			}
		}

		this.checkRPlayback();
	}
	else {
		// player is already in group
		if (mplayer.devObjId != devObjId) {
			mplayer.setContext(devObjId, this.contextType, this.archiveInterval.startTS, this.archiveInterval.endTS);
			this.syncEngine.catchUpWithGroup(mplayer);
			this.checkRPlayback();
		}
	}
}

//
// addPlayer2List
//
function elpe_addPlayer2List(mplayer)
{
	this.players.push(new PlayerData(mplayer));
	mplayer.setListener(this);

	this.getPrimaryPlayerPos();
}

//
// removePlayer
//
function elpe_removePlayer(mplayer)
{
	//this.log('removePlayer', 'id=' + mplayer.id);

	// remove player from list
    for(var i = 0; i < this.players.length; i++) {
		if (this.players[i].player.id == mplayer.id) {
			this.players[i].player.removeListener();
			this.players.splice(i, 1);
			this.checkRPlayback();
		}
	}

	this.getPrimaryPlayerPos();
	if (this.players.length == 0) {
		this.stop();
	}
}

function elpe_getNumOfAudioDevices()
{
	var audioDeviceNum = 0;
	for(var i = 0; i < this.players.length; i++) {
		var p = this.players[i].player;
		if (p.getParameter('DEVICE_TYPE') == 'A') {
			audioDeviceNum ++;
		}
	}
	return audioDeviceNum;
}

//
// Get player position in array by player's obj reference
//
function elpe_getPlayerPos(player)
{
  for(var i = 0; i < this.players.length; i ++) {
    if (player == this.players[i].player) {
      return i;
    }
  }
  return -1;
}

//
// getPlayerCatchUpCommand
// returns command, group has to execute to sync with player
//
function elpe_getPlayerCatchUpCommand(player)
{
	// Decide what to do
	var gcommand = new Command;
	var pcommand = player.getCommand();
	if (pcommand.cmd == 2 && (pcommand.status == 0 || pcommand.status == 1)) {
		// play starting with current TS
		if (player.contextType !=2) player.pause();
		gcommand.play(pcommand.direction, pcommand.speed, player.getCurrentTS());
	}
	else if (pcommand.cmd == 2 && pcommand.status == 2) {
		// pause - playback completed
		gcommand.playOneFrameTS(player.getCurrentTS());
	}
	else if (pcommand.cmd == 3 && (pcommand.status == 0 || pcommand.status == 1)) {
		// pause - future TS
		if (pcommand.timestamp == null) {
			gcommand.playOneFrameTS(player.getCurrentTS());
		}
		else {
			gcommand.playOneFrameTS(pcommand.timestamp);
		}
	}
	else if (pcommand.cmd == 3 && pcommand.status == 2) {
		// pause - current TS
		gcommand.playOneFrameTS(player.getCurrentTS());
	}
	else if (pcommand.cmd == 0 && (pcommand.status == 0 || pcommand.status == 1 || pcommand.status == 2)) {
		// stop requested
		gcommand.stop();
	}
	else {
		// pause command or if player.getCurrentTS() is not set - stop
		if (player.getCurrentTS() == 0) {
			gcommand.stop();
		}
		else {
			gcommand.playOneFrameTS(player.getCurrentTS());
		}
	}

	return gcommand;
}

//
// Stop all players
//
function elpe_stop()
{
	//this.log("<b>PlayerGroup::stop</b>","currentTS=" + dt_formatIntTs(this.getCurrentTS(), "YYYY-MM-DD", "hh:mm:SS"));

	if (this.command.cmd == 0 && this.command.status == 2  || this.syncEngine == null) {
		return false; // ignore second stop
	}
	else {
		return this.syncEngine.submitCommand((new Command()).stop());
	}
}

//
// Pause - put all players in idle mode
//
function elpe_pause()
{
	//this.log("<b>PlayerGroup::pause</b>","currentTS=" + dt_formatIntTs(this.getCurrentTS(), "YYYY-MM-DD", "hh:mm:SS"));

	if (this.command.cmd == 1 && this.command.status == 2 || this.syncEngine == null) {
		return false; // ignore second pause
	}
	else {
		return this.syncEngine.submitCommand((new Command()).pause());
	}
}

//
// Play
//
function elpe_play()
{
	//this.log("<b>PlayerGroup::play</b>", '');

	if (this.contextType < 2 || this.players.length == 0 || this.syncEngine == null) {
    	return false;
	}
	var direction = (elpe_play.arguments.length >= 1) ? elpe_play.arguments[0] : 1;
	var speed = (elpe_play.arguments.length >= 2) ? elpe_play.arguments[1] : 1;
	var timestamp = (elpe_play.arguments.length >= 3) ? elpe_play.arguments[2] : null;

	return this.syncEngine.submitCommand((new Command()).play(direction, speed, timestamp));
}

//
// playOneFrame
//
function elpe_playOneFrame(direction)
{
	//this.log("<b>PlayerGroup::playOneFrame</b>", '');

	if (this.contextType < 2 || this.players.length == 0 ||
		direction < 0 && !this.isReversePlaybackEnabled() || this.syncEngine == null) {
    	return false;
	}
	return this.syncEngine.submitCommand((new Command()).playOneFrame(direction));
}

//
// playOneFrameTS
//
function elpe_playOneFrameTS(timestamp)
{
	//this.log("<b>PlayerGroup::playOneFrameTS</b>", 'timestamp=' + timestamp);

	if (this.contextType < 2 || this.players.length == 0 || timestamp == null || this.syncEngine == null) {
    	return false;
	}
	return this.syncEngine.submitCommand((new Command()).playOneFrameTS(timestamp));
}

//
// elpe_getCurrentTS
//
function elpe_getCurrentTS()
{
	if (this.contextType == 0 || this.status == 3 || this.primaryPlayerPos == null)
		return 0;
	else
		return this.players[this.primaryPlayerPos].player.getCurrentTS();
}

//
// setStatus
//
function elpe_setStatus(commandStatus, groupStatus, errorCode)
{
	var hasChanged = (this.status != groupStatus || this.command.status != commandStatus);
	this.command.status = commandStatus;
  	this.status = groupStatus;

	//this.log("setStatus", "status=" + groupStatus + ' command: ' + this.command.commandText() + '/' + this.command.statusText());
	if (hasChanged && this.listener != null) { this.listener.onStatusChange(this, this.command, this.status, errorCode); }
}

// Callback processing /////////////////////////////////////////////////////////////////////////////////////

//
// Method is called by timer. Required to perform various sync activities
//
function elpe_onSync()
{
	if (this.syncEngine != null) { this.syncEngine.onSync(); }
}

//
// Player context change callback
//
function elpe_playerOnContextChange(player)
{
	// TODO: raise exception if player context was changed outside of the group
}

//
// One of player has change its status
//
function elpe_playerOnStatusChange(player, command, status, errorCode)
{
	var playerPos = this.getPlayerPos(player);
	//this.log("onStatusChange", "[" + player.id + "] status=" + status + " sync.status=" + this.players[playerPos].syncStatus + " errcode=" + errorCode);

	if (this.syncEngine != null) { this.syncEngine.onPlayerStatusChange(player, command, status, errorCode); }

	if (status == 3 || this.contextType != 3) {
		this.checkRPlayback(false);
	}
	else {
		this.checkRPlayback();
	}
}

//
// OnNewImage callback
//
function elpe_playerOnNewImage(player, timestamp)
{
	if (this.primaryPlayerPos!= null && player == this.players[this.primaryPlayerPos].player) {
		if (this.listener != null) { this.listener.onNewImage(this, timestamp); }
	}
	//this.log("playerOnNewImage", "playerPos=" + playerPos + " ts=" + dt_formatIntTs(msecTimestamp, "YYYY-MM-DD", "hh:mm:SS.ss"));
}

//
// OnBufferChange callback
//
function elpe_playerOnBufferChange(player, startTS, endTS)
{
	//this.log("playerOnBufferChange", player.objId + "[" + startTS + "... "  + endTS + "]");
	if (this.primaryPlayerPos!= null && player == this.players[this.primaryPlayerPos].player) {
		if (this.listener != null) { this.listener.onBufferChange(this, startTS, endTS); }
	}
}

//
// OnClick callback
//
function elpe_playerOnClick(player, x, y, isActive)
{
	if (this.listener != null)
	{
		this.listener.onClick(player, x, y, isActive);
	}
}

//
// onSnapshotFinish callback
//
function elpe_onSnapshotFinish(player)
{
	if (this.listener != null) { this.listener.onSnapshotFinish(player); }
}

//
// setParameter
//
function elpe_setParameter(name, value)
{
	// add or update parameter
	var newParameter = true;
	for(var i = 0; i < this.parameters.length && newParameter; i++) {
		if (this.parameters[i].name == name) {
			this.parameters[i].value = value;
			newParameter = false;
		}
	}
	if (newParameter) {
		this.parameters.push(new MplayerPair(name, value));
	}
}

//
// getParameter
//
function elpe_getParameter(name)
{
	for(var i = 0; i < this.parameters.length; i++) {
		if (this.parameters[i].name == name) {
			return this.parameters[i].value;
		}
	}
	return (elpe_getParameter.arguments.length > 1) ? elpe_getParameter.arguments[1] : null;
}

//
// isReversePlaybackEnabled
//
function elpe_isReversePlaybackEnabled()
{
	return this.isRPlaybackEnabled;
}

//
//
function elpe_checkRPlayback() {
	var currentRPlaybackEnabled = true;

	if (elpe_checkRPlayback.arguments.length >= 1) {
		currentRPlaybackEnabled = elpe_checkRPlayback.arguments[0];
	}
	else {
		var audioDevices = 0;
		for(var i = 0; i < this.players.length; i++) {
			var p = this.players[i].player;
			if (p.getParameter('DEVICE_TYPE') == 'A') {
				audioDevices ++;
			}
			else if (!p.isReversePlaybackEnabled()) {
				// any "non-reversable" camera makes whole group "non-reversable"
				currentRPlaybackEnabled = false;
				break;
			}
		}

		// group is "non-reversable" if it contains audio devices only
		if (currentRPlaybackEnabled) {
			currentRPlaybackEnabled = (audioDevices != this.players.length);
		}
	}

	if (this.isRPlaybackEnabled != currentRPlaybackEnabled) {
		this.isRPlaybackEnabled = currentRPlaybackEnabled;
		if (this.listener != null) { this.listener.onRPlaybackChange(this, this.isRPlaybackEnabled); }
	}
}

//
// log stub
//
function elpe_log(func, message)
{
	var msg = "[" + func + "] " + message;
	if (this.players.length > 0) {
		this.players[0].player.playerImpl.log(msg + "\n");
	}
	else {
		alert(msg);
	}
}

function elpe_getPrimaryPlayerPos()
{
	var primaryPos = null;
	var rank = -1;

	var i, p, pcmd;

	// find primay player (depends on status of group)
	if (this.command.cmd == 2 && this.command.status == 1) {
		// if group is playing: primary = any playing player preferrably audio (1) or any player (2) preferrable current primary in each category
		for(i = 0; i < this.players.length; i++) {
			p = this.players[i].player;
			pcmd = p.getCommand();
			if (pcmd.cmd == 2 && pcmd.status == 1) {
				if (p.isAudioOn() && rank < 4) { primaryPos = i; rank = 4; }
				else if (rank < 3) { primaryPos = i; rank = (this.primaryPlayerPos == i) ? 3 : 2; }
			}
			else if (rank < 1) { primaryPos = i; rank = (this.primaryPlayerPos == i) ? 1 : 0; }
		}
	}
	else {
		// otherwise - player doing something (1), completed command successfully (2), or failed (3) preferrable current_primary in each category
		for(i = 0; i < this.players.length; i++) {
			p = this.players[i].player;
			pcmd = p.getCommand();
			if (pcmd.status == 3 && rank < 1) { primaryPos = i; rank = (this.primaryPlayerPos == i) ? 1 : 0; }
			else if (pcmd.status == 2 && rank < 3) { primaryPos = i; rank = (this.primaryPlayerPos == i) ? 3 : 2; }
			else if ((pcmd.status == 0 || pcmd.status == 1) && rank < 5) { primaryPos = i; rank = (this.primaryPlayerPos == i) ? 5 : 4; }
		}
	}

	// if primary player got changed
	if (this.primaryPlayerPos != primaryPos) {
		this.primaryPlayerPos = primaryPos;
		//this.log('getPrimaryPlayerPos', 'Primary changed to [' + ((primaryPos == null) ? 'null' : this.players[primaryPos].player.id) + ']');

		if (this.primaryPlayerPos != null) {
			p = this.players[this.primaryPlayerPos].player;
			pcmd = p.getCommand();

			// invoke onNewImage callback to inform about timestamp changes
			if (this.listener != null) {
				this.listener.onNewImage(this, p.getCurrentTS());
				//this.listener.onBufferChange(this, p.buffer.startTS, p.buffer.endTS);
			}

			// make sure that primary player runs at group speed (player speed can be modified by sync algorithm)
			if (pcmd.cmd == 2 && pcmd.status == 1 && this.command.cmd == 2 && this.command.status == 1 &&
				Math.abs(pcmd.speed - this.command.speed) > 0.05) {
				p.play(this.command.direction, this.command.speed);
			}
		}
	}

	return this.primaryPlayerPos;
}

function elpe_reset()
{
	var invokeListener = (elpe_reset.arguments.length > 0 || elpe_reset.arguments[0] == true);
	// remove player listeners, so they don't impact on group state
	for(var i = 0; i < this.players.length; i++) {
		this.players[i].player.removeListener();
	}
	this.contextType = 0;
    this.archiveInterval = { startTS : null, endTS : null };
	this.status = 3;
	this.syncEngine = null;
	this.command = (new Command()).stop();
	this.parameters	= [];
	this.primaryPlayerPos = null;
	this.isRPlaybackEnabled = false;
	this.players = [];

	if (invokeListener && this.listener != null) {
 		this.listener.onStatusChange(this, this.command, this.status, null);
		this.listener.onContextChange(this);
	}
	this.listener = null;
}

function elpe_setAudio(isOn, playerId)
{
	var mplayer = null;
	var mPlayerPos = null;

	// turn off all audio
	for(var i = 0; i < this.players.length; i++) {
		var p = this.players[i].player;
		this.players[i].player.setAudio(false);

		if (playerId == p.id) {
			mplayer = p;
			mPlayerPos = i;
		}
	}

	// turn on audio on given player and make it primary
	if (isOn && mplayer != null) {
		if (mplayer.setAudio(true)) {
			if (this.primaryPlayerPos != mPlayerPos) {
				this.getPrimaryPlayerPos();
			}
		}
	}
}

// Returns true if any of player in the group playbacks audio
function else_isAudioOn()
{
	for(var i = 0; i < this.players.length; i++) {
		if (this.players[i].player.isAudioOn()) {
			return true;
		}
	}
	return false;
}
