//
// Contains all registered video players
//
var __unrecoverableErrors = 'MCA-0000,MCA-0003,MCA-0012,';
var __directDisplayErrors = 'MCA-0000,MCA-0003,MCA-0012,MCA-0011,MPL-0010,MPL-0020,MPL-0021,MPL-0030,MPL-0040,MPL-0041,RTSP-404,RTSP-455';
var __videoPlayers = [];

var playerWrapper = new MediaPlayer();

window.onunload = __onPlayerPageUnload;
setTimeout(__playerMonitor, 6000);

//
// Add player to container
//
function registerVideoPlayer(player) {
	__videoPlayers[__videoPlayers.length] = player;
}

//
// Look for player by playerId
//
function getVideoPlayer(id) {
	var player = null;

	for(var i = 0; i < __videoPlayers.length; i++) {
		if (__videoPlayers[i].id == id) {
			player = __videoPlayers[i];
			break;
		}
	}

	return player;
}

//
// Check that all applets got initialized
//
function arePlayersInitialized()
{
	var bOK = true;
	for(var i = 0; i < __videoPlayers.length; i++) {
		if (__videoPlayers[i].playerImpl == null && __videoPlayers[i].playerImplId != null) {
			var tapplet = __videoPlayers[i].playerImpl = document.getElementById(__videoPlayers[i].playerImplId);
			if (tapplet == null) {
				bOK = false;
				break;
			}
			else {
				__videoPlayers[i].playerImpl = tapplet;
			}
		}
	}
	return bOK;
}

//
// This function looks for players, which lost connection, and tries to reconnect them
//
function __playerMonitor()
{
	setTimeout(__playerMonitor, 6000);

	var player = null;
	for(var i = 0; i < __videoPlayers.length; i++) {
		try {
			player = __videoPlayers[i];
			if (player.playerImpl != null &&
			    player.contextType == 2 &&			// live mode
			    player.isContextValid &&			// have all parameters to establish connections
			    player.playerImplStatus == 3 &&		// connection is lost
				player.command.cmd == 2 &&			// command play
				player.command.status == 3 &&		// command failed
				(player.connErrCode == null || player.connErrCode == '' ||
				 __unrecoverableErrors.indexOf(player.connErrCode + ',') == -1))
			{
				if (player.playerImpl.isConnecting() == 0) {	// player is not attempting to reconnect right now
					// reconnect: every 6 sec for the first min, every 30 sec next 4 min, every 60 sec after that
					player.reconnectCounter++;
					if (player.reconnectCounter <= 10 ||
						player.reconnectCounter <= 50 && player.reconnectCounter % 5 == 0 ||
						player.reconnectCounter > 50 && player.reconnectCounter % 10 == 0)
					{
						// try to start playback
						player.play(player.command.direction, player.command.speed);
					}
				}
			}
		} catch (error) {}
	}
}

//
// Notify all players, that page is about to be destroyed
//
function __onPlayerPageUnload()
{
	for(var i = 0; i < __videoPlayers.length; i++) {
		if (__videoPlayers[i].playerImplId != null) {
			var tapplet = __videoPlayers[i].playerImpl = document.getElementById(__videoPlayers[i].playerImplId);
			if (tapplet != null) {
				try {
					tapplet.beforeStop();
				}
				catch (err) {}
			}
		}
	}
}

/**
 * Player callback interface
 *
 * @constructor
 */
function IPlayerListener()
{
	this.onContextChange = function(iplayer) {};
	this.onStatusChange = function(iplayer, command, status, errorCode) {};
	this.onNewImage = function(iplayer, timestamp) {};
	this.onBufferChange = function(iplayer, startTS, endTS) {};
	this.onClick = function(iplayer, x, y, isActive) {};
	this.onRPlaybackChange = function(iplayer, isEnabled) {};
	this.onSnapshotFinish = function(iplayer) {};
}

/**
 * Player Command
 *
 * @param {Command} [parent]
 * @constructor
 */
function Command(parent)
{
	this.cmd = 0;	// 0 -stop, 1-pause, 2-play, 3-playOneFrame or playOneFrameTS
	this.stepmode = 0;
	this.direction = 0;
	this.speed = 0;
	this.timestamp = null;
	this.status = 0; // 0 - submitted, 1-executing, 2-completed, 3-failed

	this.parent = typeof parent != "undefined" ? parent : null; // parent composite (command)
	this.children = null;
	this.childIndex = 0;	// index of current executing child

	this.stop = function() {
		this.cmd = 0; this.stepmode = 0; this.direction = 0; this.speed = 0; this.timestamp = null; this.status = 0;
		return this; };
	this.pause = function() {
		this.cmd = 1; this.stepmode = 0; this.direction = 0; this.speed = 0; this.timestamp = null; this.status = 0;
		return this; };
	this.play = function(direction, speed, timestamp) {
		this.cmd = 2; this.stepmode = 0; this.direction = direction; this.speed = speed; this.timestamp = timestamp; this.status = 0;
		return this; };
	this.playOneFrame = function(direction, timestamp) {
		this.cmd = 3; this.stepmode = 1; this.direction = direction; this.speed = 0; this.timestamp = timestamp; this.status = 0;
		return this; };
	this.playOneFrameTS = function(timestamp) {
		this.cmd = 3; this.stepmode = 1; this.direction = 0; this.speed = 0; this.timestamp = timestamp; this.status = 0;
		return this; };
	this.getTopCommand = function() {
		if (this.parent == null) return this;
		else return this.parent.getTopCommand();
	};
	this.commandText = function() {
		var cmdText;
		if (this.cmd == 0) { cmdText = 'stop[0]'; }
		else if (this.cmd == 1) { cmdText = 'pause[1]'; }
		else if (this.cmd == 2) {
			cmdText = 'play[2](' + this.direction + ',' + this.speed;
			if (this.timestamp != null) { cmdText += ','+ dt_formatIntTs(this.timestamp, null, "hh:mm:SS.ss"); }
			cmdText += ')'
		}
		else if (this.cmd == 3) {
			cmdText = 'play1f[3](';
			if (this.timestamp != null) { cmdText += dt_formatIntTs(this.timestamp, null, "hh:mm:SS.ss"); }
			else { cmdText += this.direction; }
			cmdText += ')';
		}
		return cmdText;
	};
	this.statusText = function() {
		if (this.status == 0) { return 'submitted(0)'; }
		else if (this.status == 1) { return 'executing(1)'; }
		else if (this.status == 2) { return 'completed(2)'; }
		else if (this.status == 3) { return 'failed(3)'; }
		return '';
	};
}

/**
 * Class MPlayer
 *
 * @param {string} id
 * @param {string} playerImplId
 * @constructor
 */
function MPlayer(id, playerImplId)
{
	// public methods
	this.setContext = mplayerSetContext;
	this.pause = mplayerPause;
	this.stop = mplayerStop;
	this.play = mplayerPlay;
	this.playOneFrame = mplayerPlayOneFrame;
	this.playOneFrameTS = mplayerPlayOneFrameTS;
	this.getCurrentTS = mplayerGetCurrentTS;
	this.setParameter = mplayerSetParameter;
	this.getParameter = mplayerGetParameter;
	this.setSize = mplayerSetSize;
	this.showCenterSign = mplayerShowCenterSign;
	this.saveSnapshot = mplayerSaveSnapshot;
	this.displayMessage = mplayerDisplayMessage;
	this.turnOnJitterBuffer = mplayerTurnOnJitterBuffer;
	this.turnOffJitterBuffer = mplayerTurnOffJitterBuffer;
	this.isReversePlaybackEnabled = mplayerIsReversePlaybackEnabled;
	this.getCommand = function() { return this.command.getTopCommand(); };
	this.setAudio = function(isOn) {
		if (isOn) {
			this.audioOn = isOn;
			if (this.getParameter('AUDIOOBJID') != null || this.getParameter('DEVICE_TYPE') == 'A') {
				this.playerImpl.setParameter('mute', 'off');
			} else {
				return false;
			}
		} else {
			if (this.audioOn) {
				this.playerImpl.setParameter('mute', 'on');
			}
			this.audioOn = isOn;
		}
		return true;
	};
	this.isAudioOn = function() {
		return (this.audioOn && (this.getParameter('AUDIOOBJID') != null || this.getParameter('DEVICE_TYPE') == 'A'));
	};

	// properties
	this.id = id;   // player obj id (page scope)
	/**
	 * @type {number}
	 */
	this.devObjId = null;    // device object id
	this.contextType = 0; // empty
	this.archiveInterval = { startTS : null, endTS : null };
	this.reconnectCounter = 0; // number of reconnect intervals (works in pair with __playerMonitor)
	this.isMulticastSupported = true;

	// implementation
	this.isContextValid = false;
	this.playerImplId = playerImplId;
	this.playerImpl = document.getElementById(this.playerImplId);
	this.playerImplStatus = 3; // stop
	this.playerImplStatusPrev = 3; // stop
	this.__currentTS = 0;
	this.parameters	= [];
	this.command = (new Command()).stop();
	this.connErrCode = '';
	this.audioOn = false;
	this.defaultJitterBufferLength = null;

	this._mplayerSetUrl = _mplayerSetUrl;
	this._mplayerRequestUrl = _mplayerRequestUrl;
	this._mplayerParseUrlResponse  = _mplayerParseUrlResponse;
	this.submitCommand = mplayer_submitCommand;
	this.onCompositeStatusChange = mplayer_onCompositeStatusChange;
	this.setCommandStatus = mplayer_setCommandStatus;
	this.checkConnectionUrl = mplayer_checkConnectionUrl;
	this.log = mplayerLog;

	// callback listener
	this.listener = null;
	this.setListener = function(listener) { this.listener = listener;};
	this.removeListener = function() { this.listener = null;};

	// applet callbacks
	this._onImageClick = mplayer_onImageClick;
	this._onStatusChange = mplayer_onStatusChange;
	this._onNewImage = mplayer_onNewImage;
	this._onBufferChange = mplayer_onBufferChange;
	this._onSnapshotFinish = mplayer_onSnapshotFinish;

	// system settings
	this.minSpeed = 0.1;
	this.maxSpeed = 4;
	this.maxPositionLag = 500;	// mas lag between player current TS and required TS that allows player to to jume to required TS

	// init
	this.setCommandStatus(2); // player stopped
	registerVideoPlayer(this);
}

//
// log
//
function mplayerLog(func, message)
{
	this.playerImpl.log("[" + func + "] " + message + "\n");
}

/**
 * setContext
 *
 * @param {Number} devObjId
 * @param {Number} contextType
 * @param {Number} [startTS]
 * @param {Number} [endTS]
 */
function mplayerSetContext(devObjId, contextType, startTS, endTS)
{
	//this.log('[' + this.id + "] setContext", 'devObjId=' + devObjId + ' contextType=' + contextType);

	var contextChanged = false;
	if (this.devObjId != devObjId) {
		// got new device
		this.devObjId = devObjId;

		// reset device specific parameters
		contextChanged = true;
		this.setParameter('DEVICE_TYPE', null);
		this.setParameter('TIME_ZONE', null);
		this.setParameter('POSITIONCTL', 'none');
		this.setParameter('FRAMERATE', '0');
		this.setParameter('PIXEL_ASPECT_RATIO', 'auto');
		this.setParameter('ANALYTICS', null);
		this.setParameter('AUDIOOBJID', null);
	}

	startTS = typeof startTS != "undefined" ? Math.floor(startTS / 1000) * 1000 : null;
	endTS = typeof endTS != "undefined" ? Math.floor(endTS / 1000) * 1000 : null;

	if (this.contextType != contextType || contextType == 3 &&
		(this.archiveInterval.startTS != startTS  || this.archiveInterval.endTS != endTS)) {
		// got new context type or interval
		contextChanged = true;
		this.contextType = contextType;
		this.archiveInterval.startTS = startTS;
		this.archiveInterval.endTS = endTS;
	}

	// verify context
	this.isContextValid = (devObjId != null &&
			(contextType == 0 || contextType == 1 || contextType == 2 ||
			 contextType == 3 && this.archiveInterval.startTS != null && this.archiveInterval.endTS != null));

	if (contextChanged) {
		// close connection
        this.stop();
		this.__currentTS = 0;

		// clear screen
		this.playerImpl.sendCommand("clear");

		// reset context specific parameters
		this.setParameter('URL', null);
		this.setParameter('MEDIA_TRANSPORT', null);
		this.setParameter('MULTICAST_STREAM', null);
		this.setParameter('EVENTID', null);

		// invoke callback
		if (this.listener != null) { this.listener.onContextChange(this); }

		if (this.contextType == 1) {
			// TODO: display static image
		}
	}
}

function mplayerGetCurrentTS() {
	if (this.playerImpl == null)
		return 0;
	else if (this.__currentTS == 0)
		return parseInt(this.playerImpl.getCurrentTS(), 10);
	else
		return this.__currentTS;
}


function _mplayerSetUrl()
{
	//this.log("mplayerSetUrl", "Setting URL for player=" + this.id);
	if (this.contextType == 2 || this.contextType == 3)
	{
		var transport = this.getParameter('MEDIA_TRANSPORT');
		this.playerImpl.setParameter("streamOverTCP", (transport == "tcp") ? "on" : "off");
		this.playerImpl.setParameter("streamOverHTTP", (transport == "http") ? "on" : "off");

		var devType = this.getParameter('DEVICE_TYPE');
    	if (devType == "C") {
			this.playerImpl.setParameter("videoURL", this.getParameter('URL'));
		}
		else if (devType == "A") {
			this.playerImpl.setParameter("audioURL", this.getParameter('URL'));
		}
	}
}

/**
 * Tiny helper for storing player parameters
 *
 * @param {string} name
 * @param {string} value
 * @constructor
 */
function MplayerPair(name, value)
{
	this.name = name;
	this.value = value;
}

//
// setParameter
//
function mplayerSetParameter(name, value)
{
	// add or update parameter
	var newParameter = true;
	for(var i = 0; i < this.parameters.length && newParameter; i++) {
		if (this.parameters[i].name == name) {
			this.parameters[i].value = value;
			newParameter = false;
		}
	}
	if (newParameter) {
		this.parameters.push(new MplayerPair(name, value));
	}

	// verify if parameter needs to be sent to player
	if (name == 'FRAMERATE') {
		this.playerImpl.setParameter('frameRate', value);
	}
	else if (name == 'PIXEL_ASPECT_RATIO') {
		this.playerImpl.setParameter('pixelAspectRatio', value);
	}
	else if (name == 'BANDWIDTH') {
		this.playerImpl.setParameter('bandwidth', value);
	}
	else if (name == 'MAX_GUI_FRAMERATE') {
		this.playerImpl.setParameter('maxHWFrameRate', value);
	}
}

/**
 * getParameter(name [,default_value])
 *
 * @param {String} name
 * @param {*} [default_value]
 * @returns {*}
 */
function mplayerGetParameter(name, default_value)
{
	if (name == 'WIDTH') { return this.playerImpl.width; }
	else if (name == 'HEIGHT') { return this.playerImpl.height; }

	for(var i = 0; i < this.parameters.length; i++) {
		if (this.parameters[i].name == name) {
			return this.parameters[i].value;
		}
	}

	return typeof default_value != "undefined" ? default_value : null;
}

/**
 * displayMessage(message[,errorCode[,signId]])
 *
 * @param {String} message
 * @param {String} [errorCode]
 * @param {Number} [signId]
 */
function mplayerDisplayMessage(message, errorCode, signId)
{
	var errCode = typeof errorCode != "undefined" ? errorCode : null;
	var displaySign = typeof signId != "undefined" ? signId : -1;

	//this.log('displayMessage', 'MSG=' + message + ' ErrCode=' + errCode + ' Sign=' + displaySign);

	var msg = '';
	if (errCode == 'RTSP-X') {
		// parse RTSP and MediaPlayer messages
		msg = message.replace(/cannot handle DESCRIBE response: /gi, '');
		if (msg.indexOf('RTSP') == -1) {
			errCode = 'MPL-0001';
		} else {
			// retrive RTSP error id
			var arr = msg.match('RTSP\/[0-9].[0-9] ([0-9]*)');
			if (arr != null && arr.length > 1 && arr[1]!= null && arr[1] != '') {
				var rtspErrorCode = arr[1];
				errCode = 'RTSP-' + rtspErrorCode;

				// format message depends on error id
				if (errCode == 'RTSP-404') {
					msg = (this.contextType == 3) ? "NO VIDEO IN ARCHIVE" : "NO VIDEO";
					displaySign = 2;
				} else {
					msg = msg.substr(msg.indexOf(rtspErrorCode) + rtspErrorCode.length + 1);
				}
			} else {
				errCode = 'MPL-0002';
			}
		}
	} else {
		msg = message;
	}

	// internal/external errors
	if (errCode != null && __directDisplayErrors.indexOf(errCode) == -1) {
		msg = 'INTERNAL: [' + errCode + '] ' + msg;
	}
	if (errCode != null) {
		this.connErrCode = errCode;
	}
	this.playerImpl.printError(msg);
	if (displaySign != -1) {
		this.playerImpl.displaySign(displaySign);
	}
}

/**
 * play(direction, speed [, timestamp])
 *
 * @param {Number} [direction]
 * @param {Number} [speed]
 * @param {Number} [timestamp]
 * @returns {*}
 */
function mplayerPlay(direction, speed, timestamp)
{

    if (!this.isContextValid || this.contextType < 2) {
		return false;
	}

	direction = typeof direction != "undefined" ? direction : 1;
	speed = typeof speed != "undefined" ? speed : 1;
	timestamp = typeof timestamp != "undefined" ? timestamp : null;

//	this.log('[' + this.id + "] play", 'direction=' + direction + ' speed=' + speed + ' timestamp=' + timestamp + ' status=' + this.playerImplStatus);

	var command = new Command().play(direction, speed, timestamp);
	return this.submitCommand(command);
}

//
// playOneFrame
//
function mplayerPlayOneFrame(direction)
{
//	this.log('[' + this.id + "] playOneFrame", 'direction=' + direction + ' status=' + this.playerImplStatus);

	// check if possible to execut this command
	// player must be in PAUSE state or in STOP state for forward playback
	if (direction < 0 && this.playerImplStatus == 0 && this.isReversePlaybackEnabled() || direction >= 0) {
		return this.submitCommand((new Command).playOneFrame((direction < 0) ? -1 : 1));
	}
	else {
//		this.log('[' + this.id + "] playOneFrame", 'unable to execute command, direction=' + direction);
		return false;
	}
}

//
// playOneFrameTS
//
function mplayerPlayOneFrameTS(timestamp)
{
//	this.log('[' + this.id + "] playOneFrameTS", 'timestamp=' + timestamp + ' status=' + this.playerImplStatus);
	// check if possible to execute this command
    var canPlayOneFrameTS = false;
	if (this.isContextValid) {
		if (this.contextType == 3 && this.archiveInterval.startTS <= timestamp*1 && timestamp*1 <= this.archiveInterval.endTS) {
			canPlayOneFrameTS = true;
		} else if (this.contextType == 2 && this.playerImplStatus != 3) {
			var timestamps = this.playerImpl.getTimestamps();
			for(var i = 0; i < timestamps.length; i++)
			{
				timestamps[i] = parseInt(timestamps[i], 10);
			}
			var bufferStartTS = timestamps[1];
			canPlayOneFrameTS = (bufferStartTS <= timestamp);
		}
	}

	if (canPlayOneFrameTS) {
		return this.submitCommand((new Command()).playOneFrameTS(timestamp));
	} else {
		//this.log('[' + this.id + "] playOneFrameTS", 'unable to jump to timestamp=' + timestamp);
		//if (this.contextType == 3) {
		//	this.log('[' + this.id + "] playOneFrameTS", "debug: start=" + this.archiveInterval.startTS +
		//		   " end" + this.archiveInterval.endTS + ' isContextValid=' + this.isContextValid);
		//}
		return false;
	}
}

//
// pause
//
function mplayerPause()
{
//	this.log('[' + this.id + "] pause", "status=" + this.playerImplStatus);
	return this.submitCommand((new Command()).pause());
}

//
// stop
//
function mplayerStop()
{
//	this.log('[' + this.id + "] stop", "status=" + this.playerImplStatus);
	return this.submitCommand((new Command()).stop());
}

/**
 * submitCommand
 *
 * @param {Command} command
 * @returns {boolean}
 */
function mplayer_submitCommand(command)
{
	//this.log('[' + this.id + "] submitCommand", 'command='+ command.commandText() + ((command.parent == null) ? '' : ' parent=' + command.parent.commandText()) + ' p_status=' + this.playerImplStatus );

	var bOk	= false;
	var prevCommand = this.command;

	// STOP
	if (command.cmd == 0) {
		this.command = command;

		if (this.playerImplStatus == 3) {
			// mark this command as completed
			this.setCommandStatus(2);
		}

		if (prevCommand.cmd != 0 || prevCommand.status != 1 && prevCommand.status != 2) {
			// send pause
			this.playerImpl.sendCommand("stop");
			this._onStatusChange(3,0,'');
		}
		bOk = true;
	}
	// PAUSE
	else if (command.cmd == 1) {
		this.command = command;

		if (this.playerImplStatus == 0 && command.direction != -1 || this.playerImplStatus == 3) {
			// mark this command as completed
			this.setCommandStatus(2);
		}

		if (!(prevCommand.cmd == 0 || prevCommand.cmd == 1) || !(prevCommand.status == 1 || prevCommand.status == 2)) {
			// send pause
			this.playerImpl.sendCommand("pause");
			this._onStatusChange(0,0,'');
		}
		bOk = true;
	}
	// PLAY
	else if (command.cmd == 2) {
		if (this.playerImplStatus == 3) { // player does not have open connection
			if (command.timestamp != null && this.contextType == 3) {
				// complex command. send play1f command to open RTSP session
				command.children = new Array(2);
				command.children[0] = (new Command(command)).playOneFrame(1);	// open RTSP session
				command.children[1] = (new Command(command)).play(command.direction, command.speed, command.timestamp);	// play

				this.playerImpl.setParameter("skipFirstFrame", "on");
				command.childIndex = 0;
				bOk = this.submitCommand(command.children[command.childIndex]);
			}
			else {
				this.command = command;
				if (this.checkConnectionUrl()) {
					this.playerImpl.setParameter("stepMode", "0");
					this.playerImpl.setParameter("direction", this.command.direction);
					var tmpSpeed = "" + this.command.speed;
					this.playerImpl.setParameter("streamSpeed", tmpSpeed);
					this.playerImpl.sendCommand("play");
					bOk = true;

				}
			}
		}
		else {
			this.command = command;
			if (prevCommand.status == 1 && (prevCommand.cmd == 2 || prevCommand.cmd == 3) &&
 			    (this.playerImplStatus == 1 || this.playerImplStatus == 2)) {
				this.setCommandStatus(1);
			}

			var timestamps = this.playerImpl.getTimestamps();
			for(var i = 0; i < timestamps.length; i++)
			{
				timestamps[i] = parseInt(timestamps[i], 10);
			}
			var bufferStartTS = timestamps[1];
			var bufferEndTS = timestamps[2];

			if (this.command.timestamp != null && this.playerImplStatus == 1 &&
				command.timestamp >= bufferStartTS && command.timestamp < bufferEndTS) {
				// complex command. special case to handle jump from stream to buffer
				command.children = new Array(2);
				command.children[0] = (new Command(command)).pause();	// pause
				command.children[1] = (new Command(command)).play(command.direction, command.speed, command.timestamp);	// play

				command.childIndex = 0;
				bOk = this.submitCommand(command.children[command.childIndex]);
			}
			// check if command play is required
			else if (prevCommand.cmd == 2 && prevCommand.status == 1 && (this.playerImplStatus == 1 || this.playerImplStatus == 2)) {
				// change playback parameters
				if (prevCommand.speed != this.command.speed) {
					var tmpSpeed = "" + this.command.speed;
					this.playerImpl.setParameter("streamSpeed", tmpSpeed);
				}
				if (this.command.timestamp != null ) {
    				this.playerImpl.setParameter("jump2timestamp", "" + Math.round(this.command.timestamp/1000));
				}
				if (prevCommand.direction != this.command.direction) {
					this.playerImpl.setParameter("direction", this.command.direction);
					this.playerImpl.sendCommand("play");
				}
				bOk = true;
			}
			else {
				this.playerImpl.setParameter("stepMode", "0");
				this.playerImpl.setParameter("direction", this.command.direction);
				var tmpSpeed = "" + this.command.speed;
				this.playerImpl.setParameter("streamSpeed", tmpSpeed);
				if (this.command.timestamp != null ) {
    				this.playerImpl.setParameter("jump2timestamp", "" + Math.round(this.command.timestamp/1000));
				}
				this.playerImpl.sendCommand("play");
				bOk = true;
			}
		}
	}
	// PLAYONEFRAME
	else if (command.cmd == 3 && command.timestamp == null) {
		this.command = command;
		if (this.playerImplStatus == 3) { // player does not have open connection
			if (!this.checkConnectionUrl()) {
				return false;
			}
		}
		this.playerImpl.setParameter("direction", this.command.direction);
		this.playerImpl.setParameter("stepMode", "1");
		this.playerImpl.sendCommand("play");
		bOk = true;
	}
	// PLAYONEFRAMETS
	else if (command.cmd == 3 && command.timestamp != null) {
		if (this.playerImplStatus == 3) { // player does not have open connection
			if (this.contextType == 3 && command.timestamp == this.archiveInterval.startTS) {
				this.command = command; // ???
				// optimization: archive mode + start of interval => just open connection
				if (this.checkConnectionUrl()) {
					this.playerImpl.setParameter("stepMode", "1");
					this.playerImpl.sendCommand("play");
					bOk = true;
				}
			} else {
				// complex command. send play command to open RTSP session
				command.children = new Array(2);
				command.children[0] = (new Command(command)).playOneFrame(1);	// open RTSP session
				command.children[1] = (new Command(command)).playOneFrameTS(command.timestamp); // jump to timestamp

				this.playerImpl.setParameter("skipFirstFrame", "on");
				command.childIndex = 0;
				bOk = this.submitCommand(command.children[command.childIndex]);
			}
		}
		else if (this.playerImplStatus == 0) { // player in pause state
			this.command = command;
			var timestamps = this.playerImpl.getTimestamps();
			for(var i = 0; i < timestamps.length; i++)
			{
				timestamps[i] = parseInt(timestamps[i], 10);
			}
			var frameTS = timestamps[0];
			var bufferStartTS = timestamps[1];

			// optimization: if previous command was jump to the same TS
			// or currentTS and required TS are close to each other
			if (prevCommand.cmd == 3 && prevCommand.timestamp == command.timestamp ||
				Math.abs(frameTS - command.timestamp) < this.maxPositionLag)
			{
				// simulate jump
				this.setCommandStatus(1);
				this.setCommandStatus(2);
				bOk = true;
			}
			else if (this.playerImplStatusPrev == 1 && bufferStartTS != 0 && bufferStartTS > command.timestamp)
			{
				// fix for the DE693 (player failed to jump to the beginning of interval after if reached end of interval)
				var pauseCmd = (new Command(command)).pause();
				pauseCmd.direction = -1; // enforce pause command (send pause to the server even when player is in pause state

				command.children = new Array(2);
				command.children[0] = pauseCmd;	// stop playback
				command.children[1] = (new Command(command)).playOneFrameTS(command.timestamp); // jump to timestamp

				command.childIndex = 0;
				bOk = this.submitCommand(command.children[command.childIndex]);
			}
			else {
				this.playerImpl.setParameter("stepMode", "1");
				this.playerImpl.setParameter("jump2timestamp", "" + Math.round(command.timestamp/1000));
				this.playerImpl.sendCommand("play");
				bOk = true;
			}
		}
		else {
			// complex command. send pause command to put stream on hold before jumping
			command.children = new Array(2);
			command.children[0] = (new Command(command)).pause();	// stop playback
			command.children[1] = (new Command(command)).playOneFrameTS(command.timestamp); // jump to timestamp

			command.childIndex = 0;
			bOk = this.submitCommand(command.children[command.childIndex]);
		}
	}

	return bOk;
}

//
// setSize
//
function mplayerSetSize(width, height)
{
    this.playerImpl.setSize(width,  height);
	this.playerImpl.width = width;
	this.playerImpl.height = height;
}

//
// showCenterSign
//
function mplayerShowCenterSign(showSign)
{
    this.playerImpl.showCenterSign(showSign);
}

//
// Callbacks
//
function mplayer_onStatusChange(playerImplStatus,errorCode,errorMsg)
{
	var prevCommandStatus = this.command.status;
	this.playerImplStatusPrev = this.playerImplStatus;

	this.playerImplStatus = playerImplStatus;
	//this.log('[' + this.id + "] status:", this.playerImplStatusPrev + '->' +  this.playerImplStatus);
    //this.log('[' + this.id + "] status", "cmd=" + this.command.cmd + ' cmd.status=' +  this.command.status);
    //this.log('[' + this.id + "] status", "resultCode=" + this.playerImpl.getResultCode());
    //this.log('[' + this.id + "] status", "resultMsg=" + this.playerImpl.getResultMsg());
	
	if (errorCode == 0 && errorMsg.indexOf("RTSP/1.0 404") != -1)
	{
		errorCode = MediaPlayer.INTERNALERROR;
	}

	if (this.playerImplStatus == 0) { // pause
		if (this.command.cmd == 1) {
			this.command.status = 2;	// pause cmd -> completed
		}
		else if (this.command.cmd == 2 || this.command.cmd == 3) {
			if (errorCode == MediaPlayer.RTCPTIMEOUT) { // timeout
				// do not change command status - stream is expected to recover
				this.displayMessage("INTERRUPTED", "MPL-0010", 1);
				if (this.playerImplStatusPrev == 0 && this.contextType == 2) {
					// DE3341 - timeout after pause - reconnect
					this.command.status = 3;  // failed
			 		var player = this;
					setTimeout(
						function() {
							player.playerImpl.sendCommand("stop");
							player._onStatusChange(3,errorCode,errorMsg);
						},
						1 );
				}
			}
			else if ((errorCode == MediaPlayer.ENDOFSTREAM && this.contextType == 3) || (this.command.cmd == 3 && errorCode == 0)) {
				// end of archive stream || one frame completed
				this.command.status = 2;  // completed
			}
			else if (errorCode == MediaPlayer.MEDIAFORMATCHANGED) {
				this.command.status = 3;  // failed
				this.displayMessage("MEDIA FORMAT CHANGED: " + dt_formatIntTs(this.getCurrentTS(), "YYYY-MM-DD", "hh:mm:SS"), "MPL-0020", 1);
			}
			else if (errorCode == MediaPlayer.UNKNOWNSTREAMTYPE) {
				this.command.status = 2;  // completed
				this.displayMessage("ARCHIVE/LIVE EDGE", "MPL-0021");
			}
			else {
				//this.log('[' + this.id + "] onStatusChange", "Unexpected pause: " + errorCode + " " + this.playerImpl.getResultMsg());
				this.command.status = 3;  // failed

				var msg = errorMsg + '';
				var errCode = "MPL-0030";
				if (msg == null || msg == "") {
					msg = "INTERRUPTED";
				} else {
					errCode = 'RTSP-X'; // special handling for RTSP errors provided by displayMessage
				}
				this.displayMessage(msg, errCode, 1);
			}
		}
		else {
			// command/player status collision
		}
	}
	else if (this.playerImplStatus == 3) { // stop (connection closed)
		// new authorization is required
		this.connErrCode = '';
		this.setParameter('URL', null);

		if (this.command.cmd == 0 || this.command.cmd == 1) {
			this.command.status = 2;	// -> completed
		}
		else if ((this.command.cmd == 2 || this.command.cmd == 3) && (this.command.status == 0 || this.command.status == 1)) {
			this.command.status = 3;	// -> failed
			var errCode = '';
			var msg = errorMsg + '';

			// disable multicast support if conecet in mulitacst mode failed due to "RTSP" timeout
			if (this.playerImplStatusPrev == 3 && errorCode == MediaPlayer.RTCPTIMEOUT &&
				this.isMulticastSupported && this.getParameter('MULTICAST_STREAM') == 'TRUE')
			{
				this.isMulticastSupported = false;
				errCode = 'MPL-0041';
				msg = 'Switching to unicast stream...';
			}

			if (errorCode == 0 || msg == null || msg == '') {
				msg = (this.contextType == 3) ? "CONNECTION LOST" : "INTERRUPTED";
				errCode = 'MPL-0040';
			} else if (errCode == null || errCode == ''){
				errCode = 'RTSP-X'; // special handling for RTSP errors provided by displayMessage
			}
			var signInd = 1;	// paint sign (always)
			this.displayMessage(msg, errCode, signInd);
		}
	}
	else if (this.playerImplStatus == 1 || this.playerImplStatus == 2) { // play from server(1) or buffer(2)
		this.reconnectCounter = 0; // reset reconnect counter
		if (this.command.cmd == 0 || this.command.cmd == 1) {
			// status collision
		}
		else if (this.command.cmd == 2 || this.command.cmd == 3) {
			this.command.status = 1;	// -> playing
		}
	}
        else if (this.playerImplStatus == 5) { // buffering
            this.displayMessage("Buffering...");
        }

	// invoke callback if required
	if (prevCommandStatus != this.command.status || this.playerImplStatusPrev != this.playerImplStatus)
	{
//		this.log('[' + this.id + "] status:", this.playerImplStatusPrev + '->' +  this.playerImplStatus +
//		         ' command:' + this.command.commandText() + ' ' + prevCommandStatus + '->' + this.command.statusText() +
//		         ((this.command.parent == null) ? '' : (' parent='+ this.command.parent.commandText() + '/' + this.command.parent.childIndex)));

		if (this.command.parent != null) {
			if (this.onCompositeStatusChange() || this.playerImplStatusPrev != this.playerImplStatus) {
				if (this.listener != null) {
					this.listener.onStatusChange(this, this.command.parent, this.playerImplStatus, errorCode);
				}
			}
		}
		else {
			if (this.listener != null) {
				this.listener.onStatusChange(this, this.command, this.playerImplStatus, errorCode);
			}
		}
	}
}

function mplayer_onNewImage(timestamp, timestampFraction, beginSec, beginSecFraction, endSec, endSecFraction)
{
	var ts = timestamp * 1000 + timestampFraction * 1;
	//this.log('[' + this.id + "] onNewImage:", ts);
	if (this.__currentTS != ts && this.playerImplStatus != 3) {
		this.__currentTS = ts;
		// invoke callback
		if (this.listener != null) { this.listener.onNewImage(this, this.__currentTS); }
	}
	this._onBufferChange(beginSec, beginSecFraction, endSec, endSecFraction);
}

function mplayer_onBufferChange(beginSec, beginSecFraction, endSec, endSecFraction)
{
	// invoke callback
	// commented out (this callback is not used by GUI (as of 2.6.1-2.6.3))
	//if (this.listener != null) { this.listener.onBufferChange(this, beginSec * 1000 + beginSecFraction * 1, endSec * 1000 + endSecFraction * 1); }
}

function mplayer_onImageClick(x, y, isActive)
{
	if (this.listener != null)
	{
		this.listener.onClick(this, x, y, isActive);
	}
}

function mplayer_onSnapshotFinish()
{
	if (this.listener != null) {
		this.listener.onSnapshotFinish(this);
	}
}

//
// onCompositeStatusChange
//
function mplayer_onCompositeStatusChange()
{
//	this.log('[' + this.id + "] compositStatus", 'command='+ this.command.commandText() + ' parent='+ this.command.parent.commandText() +
//	         ' p_status=' + this.playerImplStatus);

	var parent = this.command.parent;
    var isStatusChanged = false;

	if (this.command.status == 1) { // executing
		isStatusChanged = (parent.status != 1);
		parent.status = 1;
	}
	if (this.command.status == 2) { // completed
		if (parent.children.length == parent.childIndex + 1) {
			// last child, execution completed
			isStatusChanged = (parent.status != 2);
			parent.status = 2;
		}
		else {
			isStatusChanged = (parent.status != 1);
			parent.status = 1;
			// submit next command for execution
			parent.childIndex ++;
			this.submitCommand(parent.children[parent.childIndex]);
		}
	}
	else if (this.command.status == 3) { // failed
		isStatusChanged = (parent.status != 3);
		parent.status = 3;
	}
	// INFO: parent command status change is handled by onStatusChange callback

	return isStatusChanged;
}

//
// onCommandStatusChange
// invokes by onStatusChange callback. Use this method to inform that current/new command get new status
//
function mplayer_setCommandStatus(status)
{
	this.command.status = status;

	if (this.command.parent != null) {
		if (this.onCompositeStatusChange()) {
			if (this.listener != null) {
				this.listener.onStatusChange(this, this.command.parent, this.playerImplStatus, 0);
			}
		}
	}
	else {
		if (this.listener != null) {
			this.listener.onStatusChange(this, this.command, this.playerImplStatus, 0);
		}
	}
}

//
// isReversePlaybackEnabled
//
function mplayerIsReversePlaybackEnabled()
{
	// if not connected
	if (this.playerImplStatus == 3) {
		return false;
	}
	else {
		// audio playback is not reversable
		return (this.getParameter('DEVICE_TYPE') != 'A');
	}
}

//
// checkConnectionUrl
//
function mplayer_checkConnectionUrl()
{
	if (this.playerImplStatus == 3) { // player does not have open connection
		this.displayMessage("Opening stream...");
		this.connErrCode = '';
		// get new URL if needed
		if (this.getParameter('URL') == null) {
			try {
				this._mplayerRequestUrl();
			}
			catch(error) {
				this.setCommandStatus(3); // failed
				var msg = "Failed to get media URL";
				var errCode = "MPL-0100";
				if (error.message != null && error.message != "") {
					msg = error.message;
					// retrive error code
					var regEx = /\x28([A-Z]{3}-[0-9]{3,})\x29/;
					var result = msg.match(regEx);
					if (result != null && result.length > 1 && result[1]!= null && result[1] != '') {
						errCode = result[1];
						var i = msg.indexOf(errCode);
        				if (i>=2 && msg.charAt(i-2) == ' ') i--;
						msg = msg.substr(0, i - 1);
					}
				}
				this.log('checkConnectionUrl', errCode + ': ' + msg);
				this.displayMessage(msg, errCode, 1);
				return false;
			}
		}
		this._mplayerSetUrl();
	}
	return true;
}

//
// Request media stream URL, authorisation and devece/stream parameters
//
function _mplayerRequestUrl() // throws Errors
{
	//this.log("mplayerSetUrl", "Requesting URL for player=" + this.id);
	if (!this.isContextValid) {
    	throw new Error("Player context is not set (don't know what to play)");
	}

	// Assemble request
	var data = {
		'return': 'mediastreamauth',
		player: 'MPLAYER',
		objid: this.devObjId
	};

	switch (this.contextType) {
		case 1: data.streamtype = 'snapshot';
				break;
		case 2: data.streamtype = 'live';
				break;
		case 3: data.streamtype = 'archive';
				data.time_start = dt_formatIntTs(this.archiveInterval.startTS, "YYYY-MM-DD", "_hh-mm-SS");
				data.time_end = dt_formatIntTs(this.archiveInterval.endTS, "YYYY-MM-DD", "_hh-mm-SS");
				break;
	}

	if (this.getParameter('ANALYTICS')) {
		data.analytics = this.getParameter('ANALYTICS');
	}
	if (this.getParameter('AUDIOOBJID')) {
		data.audioobjid = this.getParameter('AUDIOOBJID');
	}
	if (this.getParameter('EVENTID')) {
		data.eventid = this.getParameter('EVENTID');
	}
	if (!this.isMulticastSupported) {
		data.mcastsupported = false;
	}

	var param = "" + this.getParameter('NOAUDIT');
	if (param == '1' || param.toLowerCase() == 'true') {
		data.noaudit = true;
	}

	if (this.getParameter('SKIP_GAPS') != null) {
		data.skip_gaps = this.getParameter('SKIP_GAPS');
	}

	var streamnum = this.getParameter('STREAMNUM');
	data.streamnum = (streamnum !== null && streamnum !== undefined) ? this.getParameter('STREAMNUM') : 1;

	var self = this;

	// check local content on Avatar
	if(self.getParameter('AVATARID')){
		var api = new API(false);
		var allowLocalStreaming = false;

		api.getIdentityAttribute({attribute: "AVATAR_LOCAL_STREAMING"})
			.done(function(response){
				if(response.value == "yes"){
					allowLocalStreaming = true;
				}
			});

		// if local streaming allowed
		if(allowLocalStreaming){
			var streamlocal = readCookie("streamlocal." + self.devObjId);

			if(streamlocal != null){
				if(streamlocal == "yes"){
					data.streamlocal = true;
				}
			}
			else {
				// get avatar attrs
				api.getAttributes({obj: self.getParameter('AVATARID')})
					.done(function(response){
						$.ajax({
							url: "http://" + response.list['IP'] + ":8554/checkuni",
							data: {
								uni: response.list["UNI"]
							},
							dataType: 'json',
							async: false,
							success: function(response) {
								if(response.error != ""){
									setCookie("streamlocal." + self.devObjId, "", -1, "/");
								}
								else if(response.result == "yes"){
									data.streamlocal = true;
									setCookie("streamlocal." + self.devObjId, "yes", 60 * 60, "/");
								}
								else if(response.result == "no"){
									setCookie("streamlocal." + self.devObjId, "no", 60 * 60, "/");
								}
							}
						})
					});
			}
		}
	}

	if(data.hasOwnProperty('streamlocal')){
		playerWrapper.addText("STREAM_LOCAL_INFO", "Local streaming", 5, 35, 14, 0xFFFFDD00);
		setTimeout(function(){ playerWrapper.removePrimitive("STREAM_LOCAL_INFO"); }, 7000);
	}
	else if (this.contextType == 2 && this.getParameter('AVATARID') && this.getParameter('AV_DELIVERY') != 'LIVE' && this.getParameter('STREAMNUM') == 1 ) {
		this.displayMessage("LIVE STREAM from remote location is requested");
	}

	$.ajax({
		url: '/api/authmng.php',
		data: data,
		dataType: 'xml',
		async: false,
		success: function(data) {
			self._mplayerParseUrlResponse(data.documentElement);
		},
		error: function(xhr) {
			throw new Error('Media authorization error (HTTP-' + xhr.status + ")");
		}
	});
}

function _mplayerParseUrlResponse(xml) // throws Errors
{
	if (xml == null) {
		throw new Error('Media authorization error (MCA-0204)');
	}

	xml = $(xml);

	// check status
	if (xml.find('STATUS').attr('VALUE') != "OK")
		throw new Error(xml.find('STATUS').attr('MESSAGE') || 'Media authorization error (MCA-0206)');

	// parse authorization element
	var auth = xml.find('AUTHORIZATION');
	if (auth.length == 0)
    	throw new Error('Media authorization error (MCA-0207)');

	var url = decodeURIComponent(auth.attr("URL"));
	if (!url) {
		throw new Error('Media authorization error (MCA-0208)');
	}

	if (auth.attr('ID')) {
		url += "&authorizationid=" + auth.attr('ID');
	}
	this.setParameter('URL', url);

	var self = this;
	// parse parameters
	auth.find('PARAM').each(function() {
		var $this = $(this);
		self.setParameter($this.attr('NAME'), decodeURIComponent($this.attr('VALUE')));
	});
}

//
// saveSnapshot
//
function mplayerSaveSnapshot()
{
	if (this.contextType >= 2 && this.getCurrentTS() != 0 && this.getParameter('DEVICE_TYPE') == "C") {
		if (this.playerImplStatus == 1 || this.playerImplStatus == 2) {
			this.pause();
		}

        var fileName = this.getParameter("DEVICE_NAME") + " - " + (new Date(this.getCurrentTS())).toLocaleString().replace(/:/g, "-");
		this.playerImpl.setParameter("snapshotFileName", fileName + ".jpg");
		this.playerImpl.sendCommand("snapshot");
	}
}

/**
 *  turn on jitter buffer for player
 */
function mplayerTurnOnJitterBuffer()
{
	this.playerImpl.setJitterBufferLength(this.defaultJitterBufferLength);
}

/**
 *  turn off jitter buffer for player
 */
function mplayerTurnOffJitterBuffer()
{
	this.playerImpl.setJitterBufferLength(0);
}

/**
 * embedMPlayer
 *
 * @param {number} obj
 * @param {string} id
 * @param {string} codebase
 * @param {string} width
 * @param {string} height
 * @param {number} cashSizeMb
 * @param {string} callbackPrefix
 * @param {string} playerVersion
 * @param {boolean} isDebugging
 * @param imageCallbacks
 * @param {number} jitterBufferLength
 */
function embedMPlayer(obj, id, codebase, width, height, cashSizeMb, callbackPrefix, playerVersion, isDebugging, imageCallbacks, jitterBufferLength, PLAYER_BROKEN_STREAM_INDICATION, PLAYER_BROKEN_STREAM_INDICATION_DELAY)
{
	$.ajax({
		url: "/sdi/player/manifest.json",
		cache: false,
		dataType: "json"
	})
		.done(function(manifest){
			var platform = "";
			if (navigator.appVersion.indexOf("Win") != -1)
			{
				platform = "win32";
			} else
			if (navigator.appVersion.indexOf("Mac") != -1)
			{
				platform = "macos";
			}

			var mime = manifest.platform[platform] ? manifest.platform[platform].mime : "";
			// var mime = "application/x-vnmplayer";
			var html = '<object id="' + id + '" type="' + mime + '" width="100%" height="100%"></object>';

			$("body").append(html);

			var vnMPlayer = document.getElementById(id);

			if (vnMPlayer)
			{
				var settings = {
					logFile: "", // path to file for logging, Example logFile="/somedir/1.txt" then we will writing to "/somedir/%playerId%_1.txt"
					logLevel: ((isDebugging) ? 'FINE' : 'WARNING'), // log level can be INFO|WARN|DEBUG|ERROR
					cacheSize: cashSizeMb, // size of cache in MB
					backgroundColor: 0x2E2E2E, // background color, number in next format 0xRRGGBB
					playerId: id, // player unique identificator (DOM id)
					jitterBufferLength: jitterBufferLength || 0,
					PLAYER_BROKEN_STREAM_INDICATION: PLAYER_BROKEN_STREAM_INDICATION,
					PLAYER_BROKEN_STREAM_INDICATION_DELAY: PLAYER_BROKEN_STREAM_INDICATION_DELAY
				};

				var isShowText = true;
				var text = "";
				var check = function()
				{
					if (!window[callbackPrefix])
					{
						return;
					}

					// not live
					if (window[callbackPrefix].contextType != 2)
					{
						isShowText = false;
						playerWrapper.removePrimitive("STAT_SHOW");
					} else {
						isShowText = true;
						showText(text);
					}
				};

				var showText = function(text)
				{
					if (isShowText && text)
					{
						playerWrapper.addText("STAT_SHOW", text, 5, 35, 30, 0xFFFFDD00);
					}
				};

				playerWrapper
					.init("#" + id, {}, false, obj)
					.done(function(){
						playerWrapper.subscribe("playerready", window[callbackPrefix + '_onPlayerReady'], "");
						playerWrapper.subscribe("statuschange", window[callbackPrefix + '_onStatusChange'], "");
						if (imageCallbacks > 0)
						{
							playerWrapper.subscribe("newimage", window[callbackPrefix + '_onNewImage'], "");
							playerWrapper.subscribe("bufferchange", window[callbackPrefix + '_onBufferChange'], "");
						}
						playerWrapper.subscribe("imageclick", window[callbackPrefix + '_onImageClick'], "");
						playerWrapper.subscribe("snapshotfinish", window[callbackPrefix + '_onSnapshotFinish'], "");

						playerWrapper.subscribe("play", check, "player");
						playerWrapper.subscribe("stop", check, "player");

						var oldLat, oldLng, oldAlt;
						var eps = 0.0000000000001;
						playerWrapper.subscribe("positionchange", function(lat, lng, alt){
							if (Math.abs(lat - oldLat) <= eps
								&& Math.abs(lng - oldLng) <= eps
								&& Math.abs(alt - oldAlt) <= eps)
							{
								return;
							}

							oldLat = lat;
							oldLng = lng;
							oldAlt = alt;

							var altMode = "absolute";

							if (parent.mx)
							{
								var geiFrame = parent.mx.MATRIX2.getServer().getJSXByName('ge_iframe');
								if (geiFrame)
								{
									parent.mx.MATRIX2.getIFrame(geiFrame).cameraCheckPosition(obj, altMode, lat, lng, alt);
								}
							}
						}, "");
					});

				if (typeof vnMPlayer.init != "undefined")
				{
					vnMPlayer.init(settings);

					var api = new API();
					api.getAttribute({obj: obj, attribute: "STAT_SHOW"})
						.done(function(response){
							text = response.value;
							showText(text);
						});

					AMQ.subscribe("topic://obj_stat_change." + obj, function(messageList){
						messageList.forEach(function(message){
							try
							{
								var change = JSON.parse(message);
								text = change[obj].STAT_SHOW ? change[obj].STAT_SHOW : "";
								showText(text);
							}
							catch (e)
							{
								window.console.error(e.name + " " + e.message);
							}
						});
					});
				} else {
					$("#" + id).remove();

					var message = "<div>Video Player for your system is not supported, use Windows or Mac only</div>";

					var path = manifest.platform[platform] ? manifest.platform[platform].url : "";

					if (path != "")
					{
						message = '<div>Video Player is not installed or broken. To install it <a href="' + path + '">download</a>, install and restart your browser.</div>';
					}

					$("body").append(message);
				}
			}
		});
}
