/**
 * Archive Synchronizer implementation
 *
 * @param {PlayerGroup} playerGroup
 * @constructor
 */
function ArchiveSyncImpl(playerGroup)
{
	this.submitCommand = async_submitCommand;
	this.catchUpWithGroup = async_catchUpWithGroup;
	this.onPlayerStatusChange = async_onPlayerStatusChange;
	this.onSync = async_onSync;

	// implementation
	this.group = playerGroup;
	this.syncIntervalCounter = 0;  // 0..this.syncPeroid

	this._startPlaying = async_startPlaying;
	this._startSyncPlaying = async_startSyncPlaying;
	this._playerSetPosition = async_playerSetPosition;
	this._playerSingAlong = async_playerSingAlong;
	this._playerPosition = async_playerPosition;
	this._syncPlayers = async_syncPlayers;
	this._checkIfGroupStopped = async_checkIfGroupStopped;

	// const (system parameters)
	this.positioningDelay = 1000; // how long does it take to position player (milliseconds)
	this.syncPeroid = 10;     // how often players speed sync happenes (number of syncInterval)
	this.syncWindow = 1600; // interval, by the end of which all playing players should have same TS (msec)
	this.maxSpeed = 3;
	this.verySlowSpeed = 0.1;
	this.jumpMls = 800; //not more than syncWindow value
}


//
// submitCommand
//
function async_submitCommand(command)
{
	var bOk = false;

	//this.group.log("Arch::submitCommand", command.commandText());
	// command dispatcher
	if (command.cmd == 0) {
		// STOP

		this.group.command = command;
		this.group.setStatus(1, 0);

		for(var i = 0; i < this.group.players.length; i++) {
			if (this.group.players[i].player.contextType != 0) {
				this.group.players[i].player.stop();
			}
			this.group.players[i].reset();
		}
		this.group.setStatus(2, 3);
		bOk = true;
	}
	else if (command.cmd == 1) {
		// PAUSE

		this.group.command = command;
		this.group.setStatus(1, 0);

		for(var i = 0; i < this.group.players.length; i++) {
			if (this.group.players[i].player.contextType != 0) {
				this.group.players[i].player.pause();
			}
			this.group.players[i].reset();
		}
		this.group.setStatus(2, 0);
		bOk = true;
	}
	else if (command.cmd == 2) {
		// PLAY
		if (this.group.command.cmd == 2 && this.group.command.status == 1) { // active playback
			if (this.group.command.direction == command.direction && command.timestamp == null) {
				// change speed command
				this.group.command.speed = command.speed;
				for(var i = 0; i < this.group.players.length; i++) {
					var pcommand = this.group.players[i].player.command;
        	  		if (pcommand.cmd == 2 && pcommand.status == 1) {
						// reset syns status - it's not longer valid due to speed change
						this.group.players[i].syncData.reset();
						// set new speed
						this.group.players[i].player.play(command.direction, command.speed);
					}
				}
				bOk = true;
			}
			else {
				bOk = this._startPlaying(command);
			}
		}
		else {
			bOk = this._startPlaying(command);
		}
	}
	else if (command.cmd == 3 && command.timestamp == null) {
		// PLAYONEFRAME

		this.group.command = command;
		this.group.setStatus(1, 1);

		var failed = 0;
		var participated = 0;
		for(var i = 0; i < this.group.players.length; i++) {
			var p = this.group.players[i].player;

			// next frame is not applicable for audio
			if (p.getParameter('DEVICE_TYPE') == 'A') {
				continue;
			}
			else {
				participated ++;
				if (!p.playOneFrame(command.direction)) {
					failed ++;
				}
			}
		}

		if (participated == failed) {
			bOk = false;
			this.group.setStatus(2, 0);	// treat as completed command
		}
		else {
			bOk = true;
		}
	}
	else if (command.cmd == 3) {
		// PLAYONEFRAMETS

		this.group.command = command;
		this.group.setStatus(1, 1);

		var failed = 0;
		var completed = 0;
		for(var i = 0; i < this.group.players.length; i++) {
			switch (this._playerSetPosition(i, command.timestamp)) {
				case 0: failed++ ;
					this.group.players[i].player.pause();
					break;
				case 2: completed ++; break;
				default:
			}
		}

		if (failed == this.group.players.length) { // all failed
			this.group.setStatus(1, 0);
			bOk = false;
		}
		else if (completed = this.group.players.length) { // all in place
			this.group.setStatus(2, 0);
			bOk = true;
		}
	}

	return bOk;
}


//
// catchUpWithGroup
//
function async_catchUpWithGroup(player)
{
	//this.group.log('[' + this.group.id + ']catchUpWithGroup', 'playerId=' + player.id);

	var pos = this.group.getPlayerPos(player);
	if (pos == -1) { return; }

	var gcommand = this.group.command;
	if (gcommand.cmd == 2 && gcommand.status == 1) {
		if (this.group.status == 1 || this.group.status == 2) {	// playback
			//	join playback
			this._playerSingAlong(pos, null);
		}
		else if (this.group.status == 4) {	// sync
			// join synchronization
			this._playerSetPosition(pos, this.group.command.timestamp);
		}
	}
	else if (gcommand.cmd == 3 && gcommand.status == 1) {
		//playing one frame - position player to current TS
		this._playerPosition(pos);
	}
	else if (gcommand.cmd == 0 && gcommand.status != 3 || this.group.getCurrentTS() == 0)
	{
		//	stop
		player.stop();
	}
	else {
		// position player to current TS
		this._playerPosition(pos);
	}
}


//
// startPlaying
//
function async_startPlaying(command)
{
	//this.group.log("asi:startPlaying", 'beginning');

	if (command.timestamp == null) {
		var groupTS = this.group.getCurrentTS();
		command.timestamp = (groupTS == 0) ? this.group.archiveInterval.startTS : groupTS;
	}
	this.group.command = command;
	this.group.setStatus(1, 4);

	// position group - start sync
	var completed = 0;
	var failed = 0;
	for(var i = 0; i < this.group.players.length; i ++) {
		switch(this._playerSetPosition(i, command.timestamp)) {
			case 0: failed ++; break;	// sync n/a or failed
			case 2: completed ++; break;	// ready to play
		}
	}

	if (failed == this.group.players.length) {	// no players to start playback
		this.group.setStatus(3, 0);
		return false;
	}
	else if (completed == this.group.players.length) {
		return this._startSyncPlaying();
	}
}


//
// Add player, when ensemble is in "sync" (4) state
// @return  0 -sync failed, 1 - command submitted, 2 - ready to play
//
function async_playerSetPosition(playerPos, startTS)
{
	var playerData = this.group.players[playerPos];
	//this.group.log('asi:playerSetPosition', 'player=' + playerData.player.id +
	//			   ' timestamp=' + dt_formatIntTs(startTS, "YYYY-MM-DD", "hh:mm:SS.ss") +  "(" + startTS + ")");

	// adjust startTS (if it's too close to the end of interval)
	var framerate = parseInt(playerData.player.getParameter('FRAMERATE'));
	var gap = (framerate == null || isNaN(framerate) || framerate <= 0) ? 1 : 1.5 * 1000 / framerate;
	if (startTS > this.group.archiveInterval.endTS - gap) {
		startTS = this.group.archiveInterval.endTS - gap;
	}

	// position player to start TS
	var playerCurrentTS = playerData.player.getCurrentTS();
	if (playerData.player.playerImplStatus == 0 && playerCurrentTS != 0 &&
		Math.abs(playerCurrentTS - startTS) < playerData.player.maxPositionLag) {
		// player is in pause state and it's current TS is quite close to the TS we need
		playerData.syncStatus = 2;
	}
    else {
		playerData.syncStatus = 1;
		startTS = Math.floor(startTS/1000)*1000; // truncate startTS to 1 sec boundary

		if (!playerData.player.playOneFrameTS(startTS)) {
			// jump failed
			playerData.syncStatus = 0;
		}
		//this.group.log("asi:playerSetPosition", "player " + playerPos + " result=" + playerData.syncStatus +
		//			   " ts=" + dt_formatIntTs(startTS, "YYYY-MM-DD", "hh:mm:SS"));
	}

	return playerData.syncStatus;
}


//
// turn on synchronous playback assuming that all players are at the same timestamp in pause mode
//
function async_startSyncPlaying()
{
	//this.group.log('asi:startSyncPlaying', '');

   	// no sync start is requred - start playing
   	this.group.setStatus(1, 1);
	var started = 0;
   	for(var i = 0; i < this.group.players.length; i ++) {
		if (this.group.players[i].syncStatus == 2) {
        	//this.group.log('asi:startPlaying', 'playing ' + i +
			//			   ' at ' + dt_formatIntTs(this.group.players[i].player.getCurrentTS(), "YYYY-MM-DD", "hh:mm:SS"));
			this.group.players[i].syncStatus = 0;
			if (this.group.players[i].player.play(this.group.command.direction, this.group.command.speed)) {
				started++;
			}
		}
	}

	if (started == 0) {
		this.group.setStatus(3, 0); // playback failed
		return false;
	}
	return true;
}


//
// Player join in a song
//
function async_playerSingAlong(playerPos)
{
	var p = this.group.players[playerPos];

	//this.group.log('asi:playerSingAlong', 'player=' + p.player.id);

	p.syncStatus = 5; // TS sync
	var positioningDelay = (p.player.playerImplStatus == 3) ? this.positioningDelay * 2 : this.positioningDelay;

	// TODO: use more accurate TS sync (see also elpe_onSync())
	p.syncRTS = dt_getCurrentUTCMsecTs() + positioningDelay;
	// use declarative speed of ensemble (not very accurate)
	p.syncTS = this.group.getCurrentTS() + this.group.command.speed * this.group.command.direction * positioningDelay;

	//this.group.log('asi:playerSingAlong', 'currentRTS=' + dt_formatIntTs(dt_getCurrentUTCMsecTs(), "YYYY-MM-DD", "hh:mm:SS.ss") + 				' predictedRTS=' + dt_formatIntTs(p.syncRTS, "YYYY-MM-DD", "hh:mm:SS.ss"));
	//this.group.log('asi:playerSingAlong', 'currentTS=' + dt_formatIntTs(this.group.getCurrentTS(), "YYYY-MM-DD", "hh:mm:SS.ss") + ' predictedTS=' + dt_formatIntTs(p.syncTS, "YYYY-MM-DD", "hh:mm:SS.ss"));

	// position player to predicted TS
	if (!this._playerPosition(playerPos, p.syncTS)) {
		p.syncStatus = 0;
	}
}


//
// Position player to given TS or current TS (default)
// @param playerPos [, timestamp]
//
function async_playerPosition(playerPos)
{
	var ts = (async_playerPosition.arguments.length > 1) ? async_playerPosition.arguments[1] : this.group.getCurrentTS();
	var p = this.group.players[playerPos].player;
	var pCurrentTS = p.getCurrentTS();
	if (pCurrentTS != 0 && Math.abs(pCurrentTS - ts) < p.maxPositionLag)
	{
	    // if diff between current and new TS is less then p.maxPositionLag - do not reposition player
    	// player positioning precision is 1 sec
		return true;
	}
	else {
		return p.playOneFrameTS(ts);
	}
}


//  callbacks /////////////////////////////////////////////////////////////////////

//
// onPlayerStatusChange
//
function async_onPlayerStatusChange(player, command, status, errorCode)
{
	//this.group.log('asi:onPlayerStatusChange', '[' + player.id + '] command: ' +  command.commandText() + '/' + command.statusText());
	
	//if(parent.mx){ parent.appLogger.info('asi:onPlayerStatusChange ' + '[' + player.id + '] command: ' +  command.commandText() + '/' + command.statusText() + '. errorCode=' + errorCode); }

	if (this.group.command.cmd == 2 && this.group.command.status == 1) {
		// still in pause
		if (this.group.status == 0 || this.group.status == 3) {
			if (command.cmd == 2 && command.status == 1 && !(this.group.status == 1 || this.group.status ==2)) { // playback started
				this.group.setStatus(1, 1);
			}
		}
		// sync in progress
		else if (this.group.status == 4) {
			if (command.cmd == 3 && command.status == 2) { // positioned successfully
				this.group.players[this.group.getPlayerPos(player)].syncStatus = 2; // ready to play
			}
			else if (command.cmd == 3 && command.status == 3) { // positioning failed
				this.group.players[this.group.getPlayerPos(player)].syncStatus = 0; // no sync
				this._checkIfGroupStopped(command.status, errorCode);
				// onSync methods stops group if there is no other positioning/waiting players
			}
			else if (command.cmd == 2 && command.status == 1 && !(this.group.status == 1 || this.group.status ==2)) { // playback started
				this.group.setStatus(1, 1);
			}
		}
		// playback in progress
		else if (this.group.status == 1 || this.group.status == 2) {
			if (command.cmd == 2 && command.status == 1 && !(this.group.status == 1 || this.group.status == 2)) { // playback started
				this.group.setStatus(1, 1);
			}
			else if (command.cmd == 2 && (command.status == 2 || command.status == 3)) { // playback completed or failed
				// always reset sync data when player has stopped
				this.group.players[this.group.getPlayerPos(player)].syncData.reset();
				this._checkIfGroupStopped(command.status, errorCode);
			}
			else if (command.cmd == 3 && command.status == 2) { // catch up or sync completed
				var playerData = this.group.players[this.group.getPlayerPos(player)];
				if (playerData.syncStatus == 3 || playerData.syncStatus == 5) {
					// player completed positioning and is ready to play
					playerData.syncStatus++;
				}
			}
			else if (command.cmd == 3 && command.status == 3) { // catch up or sync failed
				this._checkIfGroupStopped(command.status, errorCode);
			}
		}
	}
	else if (this.group.command.cmd == 3 && this.group.command.status == 1) { // play1f
		if (command.cmd == 3 && command.status == 1) { // started
		}
		else if (command.cmd == 3 && (command.status == 2 || command.status == 3)) { // completed or failed
			// check if group command is completed
			var isInProgress = false;
			for(var i = 0; i < this.group.players.length; i ++) {
				if (this.group.players[i].player.getCommand().status == 1) {
					isInProgress = true; break;
				}
			}
			if (!isInProgress) {
				this.group.setStatus(2, 0);
			}
		}
	}
	// INFO: pause and stop are implemented as sync command
	//this.group.log('asi:onPlayerStatusChange', '[group] command: ' + this.group.command.commandText() + '/' + this.group.command.statusText() +
	//	' status=' + this.group.status);
}

//
// onSync
//
function async_onSync()
{
	if (this.group.command.cmd == 2 && this.group.command.status == 1) { // play
		if (this.group.status == 1 || this.group.status == 2) { // playback is in progress
			this.syncIntervalCounter++;
			if (this.syncIntervalCounter >= this.syncPeroid) {
				this.syncIntervalCounter = 0;
				// check that all players are in sync with sync player
				this._syncPlayers();
			}

			// look for players ready to join others
			var groupCurrentTS = -1;
			for(var i = 0; i < this.group.players.length; i ++) {
				var p = this.group.players[i];
				if (p.syncStatus == 4 || p.syncStatus == 6) {
					// check TS sync start condition
					if (groupCurrentTS == -1) groupCurrentTS = this.group.getCurrentTS();
					var pCurrentTS = p.player.getCurrentTS();
					var syncTS = (pCurrentTS == 0) ? p.syncTS : pCurrentTS;
					if (this.group.command.direction >= 0 && groupCurrentTS >= syncTS  ||
						this.group.command.direction <  0 && groupCurrentTS <= syncTS)
					{
						// use  sync player timestamp (TS) sync
						//this.group.log("onSync",  "TS sync start player" + i + " at " + p.syncTS + " " + pCurrentTS + " " + groupCurrentTS);
						p.syncStatus = 0;
						p.player.play(this.group.command.direction, this.group.command.speed);
					}

					// then check RTS sync criteria
					if (p.syncStatus != 0 && p.syncRTS >= dt_getCurrentUTCMsecTs() - playerGroupSyncInterval/2) {
						// use realtime timestamp (RTS) sync
						//this.group.log("onSync", "RTS sync start player" + i + " at " + p.syncTS + " " + pCurrentTS + " " + groupCurrentTS);
						p.syncStatus = 0;
						p.player.play(this.group.command.direction, this.group.command.speed);
					}
				}
			}
		}
		else if (this.group.status == 4) { // sync in progress
			// TODO: if sync start takes too long - stop it
			// check if all players are ready for playback
			var syncPlayerNum = 0;
			var pendingPlayerNum = 0;
			var lastSyncPos = null;
			for(var i = 0; i < this.group.players.length; i ++) {
				// check if we have any player in pending/ready state
				if (this.group.players[i].syncStatus == 1) { // positioning
					syncPlayerNum++;
					pendingPlayerNum++;
					lastSyncPos = i;
				}
				else if (this.group.players[i].syncStatus == 2) {
					syncPlayerNum++;
					lastSyncPos = i;
				}
	    	}

			// start players
			if (syncPlayerNum == 0) {	// no player participating in sync
				this.group.setStatus(3, 0);
			}
			else if (syncPlayerNum == 1 && this.group.players[lastSyncPos].player.getParameter('DEVICE_TYPE') == 'A') {
				// only audio player participate in sync - pause it (rule: audio goes always with video)
				this.group.players[lastSyncPos].player.pause();
				this.group.setStatus(2, 0);
			}
			else if (pendingPlayerNum == 0) {
				// all players are in ready state - start playing
				//this.group.log("elpe_onSync", "start all players");
				this._startSyncPlaying()
			}
		} // end sync in progress
	}
}


//
// Correct speed of slave (non-sync) players in order to have them reach futureSyncTS
// at futureSyncRTS time in this.syncWindow milliseconds
//
function async_syncPlayers()
{
	// find synchronizing player
	var syncPlayerPos = this.group.getPrimaryPlayerPos();
	if (syncPlayerPos == null) { return; }

	var syncPlayer = this.group.players[syncPlayerPos];
	var syncPlayerCurrentTS = syncPlayer.player.getCurrentTS();
	var syncCmd = syncPlayer.player.getCommand();

	if (syncCmd.cmd != 2 || syncCmd.status != 1) { return; }// if sync player is not set or is not playing - don't run sync
	//this.group.log('asi:syncPlayers', 'Sync started. syncPlayer=' +  this.group.players[syncPlayerPos].player.id);

	// calculate future sync point
	var baseTS = syncPlayer.syncData.TS;
	var baseRTS = syncPlayer.syncData.RTS;
	var currentRTS = dt_getCurrentUTCMsecTs();
	var futureSyncRTS = currentRTS + this.syncWindow;
	var futureSyncTS = null;
	var speed = syncCmd.speed;     	 // use declared speed as default

	if (syncPlayerCurrentTS != 0) {
		if (syncPlayer.syncData.TS == 0 || syncPlayer.syncData.RTS == null) {
			// no prev sync data
			baseTS = syncPlayerCurrentTS;
			baseRTS = currentRTS;
		}
		else if (baseTS != syncPlayerCurrentTS && baseRTS != currentRTS) {
			// calculate speed
			speed = (syncPlayerCurrentTS - baseTS)/(currentRTS - baseRTS);
			syncPlayer.syncData.speed = speed;
		}

		futureSyncTS = baseTS + Math.round(speed * (futureSyncRTS - baseRTS))*syncCmd.direction;
		// save current values
		syncPlayer.syncData.TS = syncPlayerCurrentTS;
		syncPlayer.syncData.RTS = currentRTS;

		//this.group.log('syncPlayers NEW DATA', '[' + syncPlayer.player.id + "] - sync: futureSyncTS =["+ futureSyncTS +"]"); //-
		//this.group.log('syncPlayers NEW DATA', '[' + syncPlayer.player.id + "] - sync: p.current-TS =[" + syncPlayerCurrentTS +"]");
	}

	if (futureSyncTS != null)
	{
		// synchronyze all playing players
		for(var i = 0; i < this.group.players.length; i++)
		{
			var p = this.group.players[i];
			var c = p.player.getCommand();
			var pCurrentTS = p.player.getCurrentTS();

			if (i != syncPlayerPos && c.cmd == 2 && c.status == 1 && pCurrentTS != 0)
			{
				// playback is in progress

				// if player passed future sync point
				if (futureSyncTS < pCurrentTS && this.group.command.direction > 0 ||
				    futureSyncTS > pCurrentTS && this.group.command.direction < 0)
				{
					p.player.play(this.group.command.direction, this.verySlowSpeed);
					//this.group.log('syncPlayers', '[' + p.player.id + "]-Z passing sync point on slow speed = " + this.verySlowSpeed);
				}
				else
				{
					// calculate speed
					var newSpeed = (futureSyncTS - pCurrentTS) / (futureSyncRTS - currentRTS);
					p.syncData.speed = null;

					var newSpeedAbs = Math.round(Math.abs(newSpeed)*100)/100;
					if (newSpeedAbs > 0 && newSpeedAbs <= p.player.maxSpeed)
					{
						//this.group.log('syncPlayers', '[' + p.player.id + "]-Z devObjId="+p.player.devObjId+" newspeed=" + newSpeedAbs);
						p.player.play(syncCmd.direction, newSpeedAbs);
					}
					else
					{
						// if player couldn't catch up future sync point with speed - do a jump
						// position player to predicted TS
						p.syncTS = futureSyncTS + Math.round(speed * this.jumpMls) * syncCmd.direction;
						p.syncRTS = p.syncTS + this.jumpMls;
						p.syncStatus = 5; // TS sync
						//this.group.log('syncPlayers JUMP', '[' + p.player.id + "]-Z max speed exceeded - jumped to "+new Date(futureSyncTS + Math.round(speed * this.jumpMls)*syncCmd.direction));
						if (!p.player.playOneFrameTS(p.syncTS)) { p.syncStatus = 0; }
					}
				}
				// save current values
				p.syncData.TS = pCurrentTS;
				p.syncData.RTS = currentRTS;
			} // end playing player condition
		} // end for
	}
	else {
		//this.group.log('syncPlayers', '[' + p.player.id + "]-Z", "futureSyncTS is undefined");
	}
}


// check if group should be stopped
//
function async_checkIfGroupStopped(commandStopStatus, errorCode)
{
	var isVideoInProgress = false;
	var isAudioInProgress = false;
	var audioIndex = null;

	for(var i = 0; i < this.group.players.length; i ++) {
		var cmd = this.group.players[i].player.getCommand();
		if (cmd.status == 1 || // in progress
			cmd.cmd == 3 && cmd.status == 2 && this.group.players[i].syncStatus != 0) // sync completed - waiting
		{
			if (this.group.players[i].player.getParameter('DEVICE_TYPE') == 'A') {
				isAudioInProgress = true;
				audioIndex = i;
			}
			else {
				isVideoInProgress = true;
				break;
			}
		}
	}

	if (isVideoInProgress) {
		this.group.getPrimaryPlayerPos();
		return false;
	}
	else {
		if (isAudioInProgress) {
			// if the only playing stream is audio - stop it
			this.group.players[audioIndex].player.pause();
		}
		this.group.setStatus(commandStopStatus, 0, errorCode);
		return true;
	}
}
