#!/usr/bin/perl -w
# $Id: devinfo.cgi 30490 2014-04-15 14:49:05Z astarostin $
# ------------------------------------------------------------------------------
#  Author: Andrey Fomenko, Jeremiah Morrill
#  Edited by: Serg Pososhenko
#  QA by:  Christopher C Gettings
#  Copyright: videoNEXT Network Solutions LLC
# ------------------------------------------------------------------------------
#  API call to access device configuration parameters.
#
#  Usage: make a HTTP GET or POST request to http://<URI>/api/cgi-bin/devinfo.cgi
#
#  Parameters:
#   devid=<xx>                                  optional for 'return'/mandatory for 'modify'
#   return={conf|geometry|runstate|edtpars|storagespace|archive|media_format}     optional
#   modify=[create|delete|update]               optional
#   <camParamX>=<value>                         optional [multiple] parameters for modify=update
#
#  For "return" parameter mnost values are self-explanatory, but "edtpars".
#  This particular parameter is for "editor" parameters, describing how
#  parameter should be presented by editor. It may be used only with "conf".
#
#  During parameters check "return" has higher precedence and will superceed
#  "update" if both are provided.
#
#  BE CAREFULL! "modify=update(...)" command accepts values with no checks on
#  parameters validness. So, if it is wrong value or any resource conflicts - it
#  will not alert same way as it is done in the GUI.
# ------------------------------------------------------------------------------

use strict;
use SKM::DB;
use CGI qw/escape unescape/;
use Time::Local;
use LWP::Simple;
use NextCAM::WEBSession;
use XML::Simple;
use Node::Conf;
#use Data::Dumper;


# Let's get all parameters from request

my $request_method = $ENV{REQUEST_METHOD};
my $form_info='';
if ($ENV{REQUEST_METHOD} eq "GET") {
      $form_info = $ENV{QUERY_STRING};
} else {
      my $size_of_form_information = $ENV{CONTENT_LENGTH};
      read (STDIN, $form_info, $size_of_form_information);
}

my %args;
foreach(split /&/,$form_info){ $args{uc($1)} = unescape($2) if /(\S+)=(.*)/ }
my $devid = $args{DEVID} || '';
filterMetaChars($devid) if defined $devid;
filterMetaChars($args{DEVICETYPE}) if defined $args{DEVICETYPE};
#if($args{RETURN})
#{
#    print STDERR (($args{SID})?"<<<==DEVINFO=====$args{RETURN}=====================$args{SID}===!\n\n":"<<<==DEVINFO================== no SID in req:$args{RETURN}\n");
#}
#else
#{
#    print STDERR (($args{SID})?"<<<==DEVINFO=====$args{MODIFY}=====================$args{SID}===!\n\n":"<<<==DEVINFO================== no SID in req:$args{MODIFY}\n");
#    use Data::Dumper;
#    print STDERR Dumper(%args);
#}

#==================================== Get session attributes =======================================================

my ($SID, $WebUserID, $WebUserName);
my $sinfo = WEBSessionAttr($args{SID}, 1, 1);
if ($sinfo) {
    $SID = $sinfo->{sid};
    $WebUserID = $sinfo->{userid};
    $WebUserName = $sinfo->{username};
}
$WebUserName = "anonimous($ENV{REMOTE_ADDR})" unless defined $WebUserName;

# ------------------------------------------------------------------------------

use Log::Log4perl "get_logger";
require "$ENV{APL}/common/bin/logger.audit";
my $log = get_logger('APACHE.AUDIT.DEVICE_MANAGEMENT');

print <<XML;
Cache-Control: no-store, no-cache, must-revalidate
Cache-Control: post-check=0, pre-check=0
Pragma: no-cache
Content-Type: text/xml

<?xml version="1.0"?>
XML

use NextCAM::Init;
my ($APL, $APL_CONF)=($ENV{APL},$ENV{APL_CONF});

my %vols;

foreach ( <$ENV{APL}/var/vm/*/.volume*> ) {
    next if not /(\/opt\/sarch\/var\/vm\/\S+\/)\.volume_label_(\d+)/;
    $vols{$1} = $2;

}

my %conf = $devid? GetCfgs(('DEVID'=>$devid)) : GetCfgs();
my ( $device, $updMsg, $updValues );

if($args{RETURN}) { ###########################################################
    # Check permissions
    unless($SID) {
	print "<RESULT>\n<STATUS VALUE=\"ERROR\" MESSAGE=\"Operation not allowed\"/>\n</RESULT>\n";
	exit 0;
    }
    processReturn();
} # if($args{RETURN})
elsif($args{MODIFY}) { ########################################################
    # Check permissions
    unless($SID) {
	print "<RESULT>\n<STATUS VALUE=\"ERROR\" MESSAGE=\"Operation not allowed\"/>\n</RESULT>\n";
	exit 0;
    }
    processModify();
} # if($args{MODIFY})
else {
    print "<RESULT>\n<STATUS VALUE=\"OK\" MESSAGE=\"No action specified\"/>\n</RESULT>\n";
}
exit 0;

# ---------------------------------------------------------- processReturn -----
sub processReturn {
    print "<!-- This XML provides device configuration/operation information. DEVID=$devid / RETURN=$args{RETURN} -->\n";

    return  returnArchiveHours($devid) if $args{RETURN}=~/^ARCHIVEHOURS$/i;
    return  returnCameraStatus() if $args{RETURN}=~/CAMERASTATUS/i;
    #print STDERR Dumper(\%conf);
    print "<RESULT>\n\t<STATUS VALUE=\"OK\"/>\n";
    if($args{RETURN}=~/EDTPARS/i) { # ============================= EDTPARS =====
        returnEdtPars();
    }
    foreach $device (sort keys %conf) {
		next if $device=~/^n\d+$/; # it should not appear in new 2.6 systems, but old ones may have n1/n2/n3...
        next if defined $conf{$device}{LOCATION} && $conf{$device}{LOCATION} eq '@garbage';
        next if not defined $conf{$device}{DEVICETYPE};
        print "    <DEVICE ID=\"$device\" DEVICETYPE=\"$conf{$device}{DEVICETYPE}\">\n";

        if ($args{RETURN}=~/RUNSTATE/i) { # ======================== RUNSTATE =====
            returnRunState($device);
        }
        if ($args{RETURN}=~/GEOMETRY/i) { # ======================== GEOMETRY =====
            returnGeometry($device);
        }
        if ($args{RETURN}=~/CONF/i) { # ================================ CONF =====
            returnConf($device);
        }
        if ($args{RETURN}=~/ARCHIVE/i) { # ================================ ARCHIVE =====
           returnArchive($device);
        }
        if ($args{RETURN}=~/^MEDIA_FORMAT$/i) { # ====================MEDIA_FORMAT =====
           my $start=(defined $args{START})?$args{START}:'';
           returnMediaFormat($device,$start);
        }
		if ($args{RETURN}=~/STORAGESPACE/i) { # ================================ STORAGESPACE =====
           returnArchive($device);
        }
        print "    </DEVICE>\n";
    }
    print "</RESULT>\n";
} # sub processReturn

sub returnCameraStatus {
    my $dbm;
    my $raStates = [];
    my $raAttr = [];
    my $setid = $args{SETID} || '0'; # 'All Devices' by default
    eval {
	$dbm = DBMaster({PrintError => 0,RaiseError=>1});
        $dbm->{FetchHashKeyName} = 'NAME_uc';

        my $uid = $WebUserID == -1 ? 21 : $WebUserID;
        $raStates = $dbm->selectall_arrayref("SELECT obj from getObjects(?,?,'D','C')",undef,$uid,$setid);
        if (ref($raStates) && @$raStates) {
    	    my @objs = map {$_->[0]} @$raStates;
    	    local $"=',';
    	    $raAttr = $dbm->selectall_arrayref(
    		"SELECT obj,val from _obj_attr where obj in (@objs) and attr='STATUS'"
    	    ) if @objs;
        }
    };
    print "<RESULT VALUE='ERROR'/>",return if $@;
    print "<RESULT VALUE='OK'>\n";
    foreach my $rec (@$raAttr) {
	my $status = $rec->[1] || 'UNKNOWN';
	print "<CAMERA OBJID='$rec->[0]' STATUS='$status'/>\n";
    }
    print "</RESULT>";
    eval { $dbm->disconnect if ($dbm) };
}

# --------------------------------------------------------- returnEdtPars ------
sub returnEdtPars {
    print "  <EDTPARS>\n";
    # read templates from /opt/sarch/conf/etc
    my @flds = ( 'PAGENAME', 'PARAMETER', 'DESCRIPTION', 'ITEMTYPE', 'DEFAULT', 'VALUELIST', 'REGEXP', 'ERRMSG' );
    for my $template ( 'CAMERA', 'MACRO', 'NEXTCAM', 'RELAY', 'SENSOR', 'SWITCH', 'TERM', 'JOYSTICK' , 'MONITOR', 'AUDIO') {
        print "    <$template>\n";
        open F, "$APL/conf/etc/${template}.cfg";
        while($_=<F>) {
            chomp;
            next if /^#/;
            my @a = split(/:::/);
            next if $#a < 3;
            my $s = '';
            for(my $i=0; $i <= $#a; $i++ ) {
                #$templ{$template}{$a[1]}{$flds[$i]} = $a[$i];
                next if not $a[$i];
                last if $i > 7;
                $s .= "$flds[$i]=\"".escape($a[$i]).'" ';
            }
            print "      <$a[1] $s/>\n";
        }
        close F;
        print "    </$template>\n";
    }
    print "  </EDTPARS>\n";
} # sub returnEdtPars

# -------------------------------------------------------- returnRunState ------
sub returnRunState {
    my ($device)=@_;
    my $status = 'ONLINE';
    if($conf{$device}{DEVICETYPE} eq 'CAMERA') {
        my $maxdelay = 5/$conf{$device}->{FRAMERATE};   # allow camera to drop 5 frames before considered OFFLINE
        $maxdelay = 1 if $maxdelay<1;
        $status='OFFLINE' if abs(((stat("$APL_CONF/live/$device.jpg"))[9])-time) > $maxdelay;
    }
    else {
        $status = 'API does not support this query for non-cameras yet';
    }
    print "        <RUNSTATE VALUE=\"$status\"/>\n";
} # sub returnRunState

# --------------------------------------------------------- returnGeometry -----
sub returnGeometry {
    my ($device)=@_;
    my $geo = 'undefined';
    if($conf{$device}{DEVICETYPE} eq 'CAMERA') {
        eval {
            open(F, "$APL_CONF/$device/geometry") or die;
            $geo = <F>;
            close F;
            chomp $geo;
        };
        $geo = 'undefined' if ! $geo;
    }
    print "        <GEOMETRY VALUE=\"$geo\"/>";
} # sub returnGeometry
# --------------------------------------------------------- returnMediaFormat ---
sub returnMediaFormat {
    my ($device,$start)=@_;		#device id and start time
    my $mf = 'undefined';
    $mf= $conf{$device}{MEDIA_FORMAT} if defined $conf{$device}{MEDIA_FORMAT};
    if($start) {	     # if start defined then check media format in archive
      my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=gmtime($start);
      $year-=100 if $year>100;
      my $idx=sprintf "$APL/store/$device/%02d%02d%02d/%02d.idx", $year,$mon+1,$mday,$hour;
      open(IDX, $idx);
      foreach(<IDX>) {
        my ($ts)=/^(\d+)/;
        if ($ts >= $start) { # we found the the chank
           $mf='mpeg4' if /\.m4v\|/;
           $mf='mjpg'  if /\.mjpg\|/;
           $mf='pcm'   if /\.pcm\|/;
           $mf='aac'   if /\.aac\|/;
           last;
        }
      }
      close IDX;
    }
    print "        <MEDIA_FORMAT VALUE=\"$mf\"/>";
} # sub returnMediaFormat

# ------------------------------------------------------------- returnConf -----
sub returnConf {
    my ($device)=@_;
    foreach my $param (sort keys %{$conf{$device}}) {
        my $val=escape($conf{$device}->{$param});
        $val=$ENV{SERVER_NAME} if $param eq 'WAN_IP' and $val eq '127.0.0.1';
        print "        <$param VALUE=\"".$val."\" />\n";
    }
} # sub returnConf

###FOR CACHE ROUTINES###
my $cacheHour;
my $cacheDay;
my $cacheSize;
my $cacheDeviceId;
my @cacheFiles;
my $cacheLastModify;
my $cacheHourDir;
my $cacheValid = 0;
########################


# ------------------------------------------------------------- returnArchive -----
sub returnArchive {

	my ($device)=@_;
	return if ($conf{$device}->{DEVICETYPE} ne 'CAMERA');
    my $path = $APL . "/store/" . $device;
	$cacheDeviceId = $device;
	outputArchiveDays($path, $device);

} #sub returnArchive :CHUNK_SIZE :STORAGE_FORMAT

# ------------------------------------------------------------- returnArchive -----
sub outputArchiveDays {
	 my ($path,$loc_dev) = @_;#$_[0];



	print "<ARCHIVE>\n";
	my (@dirs) = getDirectoryContents($path, "dir");
	my ($d, $m, $y);

	foreach my $item (@dirs)
	{
		if( ($d,$m,$y) = ( $item =~/(\d{2})(\d{2})(\d{2})/ ))
		{
			print "<DAY VALUE=\"" . directoryToDate( $item ) . "\" >\n";
			$cacheDay = $item;
			if( (defined($args{INTERVALDATE})) and ($args{INTERVALDATE}=~/./i ))
			{
				if( $args{INTERVALDATE} eq directoryToDate( $item ) )
				{
					outputArchiveIntervals( $path . "/" . $item ,$loc_dev );
				}
			}
			else
			{
				outputArchiveIntervals( $path . "/" . $item ,$loc_dev);
			}

			print "</DAY>\n"
  		}
	}
	print "</ARCHIVE>\n";
} # ----------------------------------------------------------- outputArchiveIntervals --------

# ----------------------------------------------------------- outputArchiveIntervals --------
sub outputArchiveIntervals {
	my ($path, $dev) = @_;

	my @segmentList = getFileList( $path );

	if(scalar(@segmentList) == 0)
	{
		return;
	}

	if ($args{RETURN}=~/STORAGESPACE/i)
	{
		return;
	}

	my $segmentLength = 0;
	my $tm1 = fileTimeToSeconds(filenameToTime($segmentList[1]));
	my $tm2 = fileTimeToSeconds(filenameToTime($segmentList[0]));

	$segmentLength = $conf{$dev}->{CHUNK_SIZE};# $tm1 - $tm2;
    my $result_string ='';
	$result_string = "<INTERVAL FROM=\"" . formatFileTime(filenameToTime($segmentList[0])) . "\"";
	for(my $i = 0; $i < scalar(@segmentList) - 1; $i++)
	{

		 my $timeLen = fileTimeToSeconds(filenameToTime($segmentList[$i+1])) - fileTimeToSeconds(filenameToTime($segmentList[$i]));
		 if( $timeLen > ($conf{$dev}->{CHUNK_SIZE}+1))#$segmentLength )
		 {
		    my $sec_time = fileTimeToSeconds(filenameToTime($segmentList[$i]))+$conf{$dev}->{CHUNK_SIZE};
		 	$result_string .= " TO=\"" . formatSecTime($sec_time) . "\"/>\n";
			$i++;
			$result_string .= "<INTERVAL FROM=\"" . formatFileTime(filenameToTime($segmentList[$i])) . "\"";

			#$segmentLength = fileTimeToSeconds(filenameToTime($segmentList[$i])) - fileTimeToSeconds(filenameToTime($segmentList[$i-1]));
		 }
	}
	my $sec_time = fileTimeToSeconds(filenameToTime($segmentList[scalar(@segmentList) - 1]))+$conf{$dev}->{CHUNK_SIZE}-1;
	$result_string .= " TO=\"" . formatSecTime($sec_time) . "\"/>\n";
	print $result_string;

} #sub outputArchiveIntervals

sub isCacheValid{
	my $deviceId = $_[0];
	my $day = $_[1];
	my $hour = $_[2];
	my $lastModify='';
	my @stats=();

	my $cacheFilename = "/opt/sarch/store/tmp/" . $deviceId . "@" . $day . "@" . $hour . ".cache";

	if(open (TEST, $cacheFilename))
	{
    	$lastModify = <TEST>;
    	close(TEST);

    	@stats = stat("/opt/sarch/store/" . $deviceId . "/" . $day . "/" . $hour);

    	if($lastModify != $stats[9])
    	{

    		return 0;
    	}



    	return 1;
    }
    else {return 0;}

}

sub writeCacheFile{

	my $cacheFilename = "/opt/sarch/store/tmp/" . $cacheDeviceId . "@" . $cacheDay . "@" . $cacheHour . ".cache";

	my @vals = stat($cacheHourDir);
	$cacheLastModify = $vals[9];


	# write to the data file
	open(DAT,">$cacheFilename") || die("Cannot Open File");
	print DAT $cacheLastModify . "\n";
	print DAT int($cacheSize) . "\n";
	foreach my $file (@cacheFiles)
	{
		print DAT $file . ",";
	}

	close(DAT);

}

sub filesFromCache
{
    my ($size) = $_[0];
    my $files ='';
    my @filelist=();
    my $cacheFilename = "/opt/sarch/store/tmp/" . $cacheDeviceId . "@" . $cacheDay . "@" . $cacheHour . ".cache";

    if (open (TEST, $cacheFilename))
    {
        my $lastModify = <TEST>;
        $$size = <TEST>;
        $files = <TEST>;
        close(TEST);
    }
    if(defined($files))
    {
         @filelist = split(',', $files);
         my @checked_list;
         for my $elem (@filelist)
         {
            push (@checked_list,$elem) if ($elem =~/^\d+\.(mjpg|m4v|pcm|aac)$/i);
         }
         @filelist=@checked_list;
    }



	return @filelist;

}

# ----------------------------------------------------------- filenameToTime --------
sub filenameToTime {
	my ($filename) = $_[0];

	my @pathSplit = split("/", $filename);
	my @fileSplit = split(/\./, $pathSplit[ scalar(@pathSplit) - 1 ] );
	my $segment = $fileSplit[ 0 ];
	return $segment;

} #sub filenameToTime

# ----------------------------------------------------------- getFileList --------
sub getFileList {
	my ($path) = @_;

	my @hours = getDirectoryContents($path, "dir");
	my @fileList = ();
	my @list;
	my $filePath;

	@hours = sort @hours;



	foreach my $hour (@hours)
	{
		$filePath = $path . "/" . $hour;
		my $volumeid = getVolumeId($filePath);
		my $size = 0;
		$cacheHour = $hour;

		if( isCacheValid($cacheDeviceId, $cacheDay, $cacheHour) == 1)
		{
			@list = filesFromCache(\$size);

		}
		else
		{
			@list = getDirectoryContents( $filePath, "file", \$size );
			$size = int($size / 1024);
			@list = sort @list;
		}

		if ($args{RETURN}=~/STORAGESPACE/i)
		{
			print '<HOUR VALUE="' . int($hour) .'" SIZEKB="' . $size . '" VOLUMEID="' . $volumeid . '"/>';
		}



		@cacheFiles = @list;
		$cacheHourDir = $filePath;
		$cacheSize = $size;

		if( isCacheValid($cacheDeviceId, $cacheDay, $cacheHour) == 0)
		{
			writeCacheFile();
		}

		foreach my $segment (@list)
		{
			push(@fileList, $filePath . "/" . $segment)
		}
	}

	return @fileList;
}

# ----------------------------------------------------------- getVolumeId --------
sub getVolumeId{
	my ($path) = @_;


	if(! -l $path) { # not link!
	    return 0; # SYSTEM volume
	}


	my $link_dst = readlink($path);

	foreach my $vol_path (keys %vols){
	    if( $link_dst=~/$vol_path/ ) {
	        return $vols{$vol_path};
	    }
	}

	return undef;
}

# ----------------------------------------------------------- getDirectoryContents --------
sub getDirectoryContents{
	my ($path) = $_[0];
	my ($filter) = $_[1];
	my ($size) = $_[2];
	my $entry;
	my $type;
	my @contents = ();


	opendir( DIR, $path )
        or die "Can't open $path: $!";

    while ( $entry = readdir( DIR ) ) {
        $type = ( -d "$path/$entry" ) ? "dir" : "file";
        next if($entry eq '.' || $entry eq '..');

			if($filter eq "")
			{
				push(@contents, $entry);
			}

			if($filter eq $type)
			{
				push(@contents, $entry ) if($filter ne "file");

				if($filter eq "file")
				{
					my ($y,$M,$d,$h,$m,$s);

					if( ($y,$M,$d,$h,$m,$s) = ( $entry =~/(\d{2})(\d{2})(\d{2})(\d{2})(\d{2})(\d{2})\.(mjpg.*|m4v.*|pcm.*|aac.*)$/ ) )
					{
					    push(@contents, $entry );
  						$$size += -s "$path/$entry";
   					}
				}
			}
    }

    closedir( DIR );
	@contents = sort @contents;
	return @contents;
} #sub getDirectoryContents

# ----------------------------------------------------------- directoryToDate --------
sub directoryToDate{
	my ($dirName) = @_;
	my $returnDate;

	my $year = substr($dirName, 0, 2);
	my $month = substr($dirName, 2, 2);
	my $day = substr($dirName, 4, 2);

	#$returnDate = $day . "-" . $month . "-" . $year;
	$returnDate = ($year + 2000) . "-" . $month . "-" . $day;
	return $returnDate;
} #sub directoryToDate

# ----------------------------------------------------------- formatFileTime --------
sub formatFileTime {
	my ($filetime) = @_;

	my $year = substr($filetime, 0, 2) ;
	my $month = substr($filetime, 2, 2) ;
	my $day = substr($filetime, 4, 2);
	my $hour = substr($filetime, 6, 2);
	my $min = substr($filetime, 8, 2);
	my $sec = substr($filetime, 10, 2);


	return $hour . ":" . $min . ":" . $sec;

} #sub formatFileTime

# ----------------------------------------------------------- fileTimeToSeconds --------
sub fileTimeToSeconds {
	my ($filetime) = @_;

	my $year = substr($filetime, 0, 2) * 1;
	my $month = substr($filetime, 2, 2) * 1;
	my $day = substr($filetime, 4, 2) * 1;
	my $hour = substr($filetime, 6, 2) * 1;
	my $min = substr($filetime, 8, 2) * 1;
	my $sec = substr($filetime, 10, 2) * 1;
	my $tm = timegm($sec, $min, $hour, $day, $month-1, $year);


	return $tm;

} #sub fileTimeToSeconds

sub formatSecTime {
   	my ($filetime) = @_;

    my($seconds, $minutes, $hours, $day_of_month, $month, $year, $wday, $yday, $isdst) = gmtime($filetime);
    my $ret = sprintf("%02d:%02d:%02d",$hours,$minutes ,$seconds);
    return $ret;
}

# ----------------------------------------------------------- filterMetaChars --------------
sub filterMetaChars {
    my $meta = "][<>\|&;`'\"*\$\?~\^(){}\n\r";
    $_[0] =~ s/([$meta])//g;
}

# ----------------------------------------------------------- rand_str ---------------------
sub rand_str {
    my $len = shift;
    my @chars = ("A".."Z","a".."z",0..9);
    return join("", @chars[ map { rand @chars } ( 1..$len ) ] );
}
# --------------------------------------------------------- processModify ------
sub processModify {
    if((uc($devid) ne 'NEW') && (!$devid || !$conf{$devid}{DEVID})) { # check for parameter/config presence
        print <<XML;
<RESULT>
    <STATUS VALUE="ERROR" MESSAGE="DEVID parameter was not provided or it is no such device registered. DEVID=$devid"/>
</RESULT>
XML

        exit;
    } # check for parameter/config presence

    print "<!-- This XML returns status of attempt to edit device configuration. -->\n"
    ."<RESULT>\n";

    if($args{MODIFY}=~/DELETE/i) { # ============================== DELETE =====
        $conf{$devid}->{LOCATION}='@garbage';
        $conf{$devid}->{MATRIXID}=0;
        print "    <STATUS VALUE=\"OK\" MESSAGE=\"Device marked for deletion\"/>\n";
        $log->warn("User:\"$WebUserName\" delete camera $devid");
    } # DELETE
    elsif($args{MODIFY}=~/FPSADVANCE/i) { # =================== FPSADVANCE =====
        # check/adjust paramneters
        if(defined($args{FPSLIVE})) {
            my $fps_live = $args{FPSLIVE};
            my $fps_arch = $args{FPSARCH}? $args{FPSARCH} : $args{FPSLIVE};
            my $fps_time = $args{TIMEOUT}? $args{TIMEOUT} : $conf{$devid}->{VAE_MOTION_MOTIONAFTER};
            if(open(F,">$APL_CONF/$devid/fps_advance")) {
                print F "$fps_live:$fps_arch:$fps_time";
                close F;
                `$APL/cam/bin/nph_sighup $devid`;       # send SIGHUP to all nph_mjpg & cam_mretr associated with devid
                print "    <STATUS VALUE=\"OK\" MESSAGE=\"Sent signal to camera. LIVE:$fps_live ARCH:$fps_arch TIME:$fps_time\"/>\n";
            }
            else {
                print "    <STATUS VALUE=\"ERROR\" MESSAGE=\"Error sending signal to camera\"/>\n";
            }
        }
        else {
            print "    <STATUS VALUE=\"ERROR\" MESSAGE=\"Request has no mandatory 'FPSLIVE' parameter\"/>\n";
        }
        print "</RESULT>\n";
        exit;
    }
    elsif($args{MODIFY}=~/UPDATE/i) { # =========================== UPDATE =====
        # CAREFULL! no checks for correctness of values!!!!
        $updMsg = '';
        $updValues = '';
        foreach my $par ( keys %args ) {
            next if 'DEVID, RETURN, MODIFY, SID' =~ /$par/;
            if( not defined $conf{$devid}->{$par} ) {
                $updMsg .= " $par";
                next;
            }
            $conf{$devid}->{$par} = $args{$par};
            $updValues .= " $par=$args{$par}";
        }
        if($updMsg) {
            print "    <STATUS VALUE=\"ERROR\" MESSAGE=\"Incorrect parameters:${updMsg}\"/>\n";
            print "</RESULT>\n";
            exit 0;
        }
        else {
            print "    <STATUS VALUE=\"OK\" MESSAGE=\"Device parameters updated:${updValues}\"/>\n";
            my $reportVal=$updValues;
            $reportVal =~ s/PASSWD=.+\sVAE_MOTION_ZONENOISELEVEL/PASSWD=XXX VAE_MOTION_ZONENOISELEVEL/i;
            $log->warn("User:\"$WebUserName\" updated device $devid as: $reportVal");
        }
    }
    elsif($args{MODIFY}=~/CREATE/i) { # ========================== CREATE =====
        my %devLtr = ('CAMERA'=>'', 'MACRO'=>'m', 'NEXTCAM'=>'n', 'RELAY'=>'r', 'SENSOR'=>'s', 'SWITCH'=>'x', 'TERM'=>'t');
        $devid = '';
        my $devtype=uc($args{DEVICETYPE});
        if(not "'CAMERA', 'MACRO', 'NEXTCAM', 'RELAY', 'SENSOR', 'SWITCH', 'TERM', 'JOYSTICK'" =~ /'$devtype'/ ) {
            print "<STATUS VALUE=\"ERROR\" MESSAGE=\"No correct DEVICETYPE parameter provided\"/>\n</RESULT>\n";
            exit
        }
        # create new $devid
        my $dbh=DBMaster({PrintError => 1}) or die("Error connecting to database: $DBI::errstr");
        my $hostId = $dbh->selectrow_array("select val from _obj_attr where attr='HOST_ID' and obj = 53") || 0;
        $dbh->disconnect;
        my $seqStart = $hostId * 100 + 1;

        for($devid=$seqStart; $devid < $seqStart+100; $devid++) {
            last if (! -e "$APL_CONF/$devLtr{$devtype}$devid/conf")
                and (! -e "$APL_CONF/$devLtr{$devtype}/conf/$devid.conf");
        }
        if($devid==$seqStart+100) {
            print "<STATUS VALUE=\"ERROR\" MESSAGE=\"Can not get new device ID\"/>\n</RESULT>\n";
            exit;
        }
        $devid = "$devLtr{$devtype}$devid";
        # now, let's predefine defaults
        open F, "$APL/conf/etc/".uc($args{DEVICETYPE}).".cfg";
        while($_=<F>) {
            # section:::param::::::MenuItem:::MenuItemType:::DefaultVal:::Val1,Val2,Val3:::regexp:::Error Message
            chomp;
            next if /^#/;
            my @a = split(/:::/);
            next if $#a < 3;
            $conf{$devid}->{$a[1]} = $a[4];
        }
        close F;
        # here we apply CGI parameters on the top of defualts:
        $updMsg = '';
        $updValues = '';
        foreach my $par ( keys %args ) {
            next if 'RETURN, MODIFY, DEVID, SID' =~ /$par/;
            if( not defined $conf{$devid}->{$par} ) {
                $updMsg .= " $par";
                next;
            }
            $conf{$devid}->{$par} = $args{$par};
            $updValues .= " $par=$args{$par}";
        }
        $conf{$devid}->{DEVICETYPE} = $devtype;
        $conf{$devid}->{DEVID} = $devid;
        if($updMsg) {
            print "    <STATUS VALUE=\"ERROR\" MESSAGE=\"Incorrect parameters:${updMsg}\"/>\n";
            print "</RESULT>\n";
            exit 0;
        }
        else {
            print "    <STATUS VALUE=\"OK\" MESSAGE=\"Device parameters are defaults with next overrides:${updValues}\"/>\n";
            print "    <DEVICE ID=\"$devid\"/>\n";
            $log->info("User:\"$WebUserName\" created device $devid");
        }
    }
    # now, let's save it and let db2conf and cam_patrol to take care of the rest
      my $dbm;
      my %Subtypes = (CAMERA=>'C',RELAY=>'R',SENSOR=>'S',AUDIO=>'A',JOYSTICK=>'J',MONITOR=>'V',WALL=>'W');
      eval {
        $dbm = DBMaster({PrintError => 0,RaiseError=>1,AutoCommit=>0});
        $dbm->{FetchHashKeyName} = 'NAME_uc';
        # Get Node ID
        my $uni_arr = $dbm->selectall_arrayref("SELECT obj FROM _objs WHERE name=? AND otype='D' AND subtype='N'", undef, UNI());
        die "Can not determine NODE_ID for UNI=".UNI() if @$uni_arr != 1;
	my $nodeid = $uni_arr->[0][0];
	# Determine OBJID, create/update/delete OBJECT as needed
	my $obj = $devid; # $objid=[asrvw]?devid Since 2.6.0
	$obj=~s/^[\D]//;
	my $OBJID;
	my $oid_arr = $dbm->selectall_arrayref('SELECT obj FROM _objs WHERE obj=? AND node_id=? and deleted=0',undef,$obj,$nodeid);
	if(@$oid_arr>1) {
	    die "Non-unique data in '_objs' table for device:$devid, node:$nodeid\n";
	}
	elsif(@$oid_arr) {
	    # ----- device already exists in database: update/ressurect or update/delete dependent on "location"=="@garbage" ----
	    $OBJID = $oid_arr->[0][0];
	#    $dbm->do("UPDATE _objs SET description=?, location=?, deleted=?, stime=now() at time zone 'UTC' WHERE obj=?",
	#	      undef,
	#	      $conf{$devid}{NAME}, $conf{$devid}{LOCATION}, $conf{$devid}{LOCATION} eq '@garbage' ? 1 : 0, $OBJID);
	}
	else {
	    # ---- device does not exist in database and we will have to create one ----
	    # node_id,node_ip,otype,subtype,name,description,location
	    my $tmp_name = rand_str(22);
	    $dbm->do("INSERT INTO _objs (node_id,node_ip,otype,subtype,name,description,location,stime)
		      VALUES (?,?,?,?,?,?,?,now() at time zone 'UTC')",
		      undef,
		      $nodeid, UNI, UNI, 'D', $Subtypes{$conf{$devid}{DEVICETYPE}}, $tmp_name, "$conf{$devid}{NAME}", "$conf{$devid}{LOCATION}");
	    # important: recover OBJID (automagically assigned by DB sequence)
	    my $seq_id = $dbm->selectall_arrayref("SELECT currval('seq_obj')");
	    $OBJID = $seq_id->[0][0];
	    # new device will have to be added into "All Devices" set (obj=0)
	    $dbm->do("INSERT INTO _links (obj_res,obj_cons,link_type,protected) VALUES (?, 0,'D2S',1)", undef, $OBJID);
	    # !!! Set DEVID=OBJID, so update 'name' field in _objs
	    my $devChr = ($devid =~ /^\d/) ? '' : substr($devid, 0, 1);
	    my $dev = $devChr . $OBJID;
	    #$dbm->do("UPDATE _objs set name=? where obj=?", undef, $dev, $OBJID);
	    $conf{$devid}{DEVID} = $dev;
	}

	$conf{$devid}{OBJID} = $OBJID;

	# update OBJECT's ATTRIBUTES
	$dbm->do("DELETE FROM _obj_attr WHERE obj=? and attr not in ('MON','tourinfo','STATUS')", undef, $OBJID);
	foreach my $attr ( keys %{$conf{$devid}} ) {
            next if $attr=~/^(MON|tourinfo|STATUS)$/;
	    $dbm->do('INSERT INTO _obj_attr (obj,attr,val) VALUES (?,?,?)', undef, $OBJID, $attr, $conf{$devid}{$attr});
	}
	$dbm->do("UPDATE _objs set rtime=null where obj=?",undef,$OBJID);

	$dbm->commit;
      };
      if($@) {
        my $errMsg = $@;
        eval { $dbm->rollback };
        $log->error("Error storing device params to DB (DEV=$devid): $errMsg");
    	print "    <STATUS VALUE=\"ERROR\" MESSAGE=\"Cannot store parameters to DB\"/>\n";
        print "</RESULT>\n";
        $dbm->disconnect if $dbm;
        exit 0;
      }
      $dbm->disconnect if $dbm;


    #--------------------------------------- DE1419 workaround TBD!! Replace!
    if($ENV{APL_MOD} eq 'CIRRUS' and $conf{$devid}->{OBJID}>0) {
        my $dbh=DBMaster({PrintError => 1});
        if($dbh) {
           foreach my $key (grep {/^(VAE_MOTION_CELLNOISE|VAE_MOTION_PIXELNOISE|VAE_MOTION_ZONEMAP|VAE_MOTION_ZONENOISELEVEL)$/} keys %{$conf{$devid}}) {
             $dbh->do("update _obj_attr set val='$conf{$devid}->{$key}' where ATTR='$key' and obj='$conf{$devid}->{OBJID}'");
           }
           $dbh->disconnect;
        }
    }
    #-----------------------------------------------------------------------
    print "</RESULT>\n";
} # sub processModify



#--------------------------------------------------------------------------------
#  returnArchiveHours return only not-empty hours for each device
#  this function has compatible output with returnArchive except it does not
#  spend resources for geting actual intervals
#
#--------------------------------------------------------------------------------
sub returnArchiveHours {
 my $store="$ENV{APL}/store";
 my $dev=shift;
 my @find;      # result from find
 my %result;
 my $find =($dev)? "find ./$dev -follow -type d -mindepth 2 -maxdepth 2 ! -empty"
                 : "find .      -follow -type d -mindepth 3 -maxdepth 3 ! -empty";
 eval { # ----------------------------- error handling in ON ----------
   chdir  $store || die ("Store $store is not defined\n");
# ------------------------------------- define a sourser --------------
   open(FIND, "$find 2>/dev/null|") || die ("Cannot find $store\n") ;
   @find=<FIND>;
   close FIND;
 };
 if ($@) {                     # ERROR catch
   $_="$@"; chomp;
   %result=('STATUS','ERROR','MESSAGE',$_);
 } elsif (not @find) {
   %result=('STATUS','OK','MESSAGE',"NO HOURS ARE FOUND");
 }else  {
# ------------------------------------- build result structure --------
  my ($today_sec,$today_min,$today_hour,$tday,$tmon,$tyear)=gmtime(time);
  my $today=sprintf("%04d-%02d-%02d",$tyear+1900,$tmon+1,$tday);
  my ($devid,$day,$hour,$devtype); $hour=-1;
  my @devices;              # device collection
  my @day;                  # day collection for current device
  my @interval;             # interval collection for current day
  foreach (sort @find) {
    next if not m|\./(a?\d+)/(\d{2})(\d{2})(\d{2})/(\d{2})(.\d{4})?$|;
    my ($c_devid,$yy,$mm,$dd,$c_hour)=($1,$2,$3,$4,$5);
    my $c_day=sprintf("%04d-%02d-%02d",$yy+2000,$mm,$dd); # ex: 2007-08-31
    ($devid,$day)=($c_devid,$c_day) if ! $devid;# assign current for first iteration
    if($c_devid ne $devid) {    # new device in a list
        my $devtype=($devid=~/^\d+$/)?'CAMERA':'AUDIO';  # camera if numbers only
        push @day,{'VALUE',$day,'INTERVAL',[@interval]};
        my %header=('ID',$devid, 'DEVICETYPE','CAMERA','ARCHIVE',{'DAY',[@day]});
        push  @devices, \%header;
        @day=();@interval=();
        $devid=$c_devid;
        $day=$c_day;
        $hour=-1;
    }
    if($day ne $c_day) {
      push @day,{'VALUE',$day,'INTERVAL',[@interval]};
      @interval=();
      $day=$c_day;
      $hour=-1;
    }
    my $end_interval=($today eq $c_day and $c_hour==$today_hour) # is this a today current current hour?
       ? sprintf("%02d:%02d:%02d",$c_hour,$today_min,$today_sec) # special case for current hour
       :"$c_hour:59:59";				         # for any hour except today current
    push @interval,{'FROM',"$c_hour:00:00",'TO',$end_interval} if $hour!=$c_hour;
    $hour=$c_hour;
  }
  if($devid) {       # push last element
     push @day,{'VALUE',$day,'INTERVAL',[@interval]};
     my %header=('ID',$devid, 'DEVICETYPE','CAMERA','ARCHIVE',{'DAY',[@day]});
     push  @devices, \%header;
  }
  %result=('STATUS','OK','DEVICE',[@devices]);
 }
# ------------------------------------- output XML result  ----------
  #print Dumper(\%result);
 print XMLout(\%result,rootname => 'RESULT');
} # end of returnArchiveHours


# END_OF_FILE
