#!/usr/bin/perl
# -----------------------------------------------------------------------------
# $Id$
#	Advantor asn daemon. Communicates via serial port with Advanror C&C  using Bosch protocol
# -----------------------------------------------------------------------------
#  Author: Yaroslav Sinyatkin (yas@videonext.net)
#  Edited by:
#  QA by:  
#  Copyright: videoNEXT Network Solutions, Inc
# -----------------------------------------------------------------------------
use strict;
use lib "$ENV{APL}/acc/bin";
use bytes; # workaround for 'Wide character in syswrite & print'
use warnings;
use POSIX;
use Fcntl;
use IO::Select;
use IO::Handle;
use SKM::DB;
use NextCAM::OMClient;
use NextCAM::Init "GetAsrv";
use LWP; # HTTP requests
use Time::Local;
use XML::Simple;
use AccElogClient;
use VMXSimpleClient;
#use Data::Dumper;

use Device::SerialPort;


use constant STATE_UNKNOWN      => 0; # 
use constant STATE_CONNECTED    => 1; # connected but not streaming yet
use constant STATE_STREAMING    => 2; # streaming OK
use constant STATE_OFFLINE      => 3; # disconnected  

#public data
our $TIMEOUT_SO = 1;
our $BUFFER_SIZE = 1024;
our $CHECK_DELAY = 1; # period between camera checks
our $CHECK_VMX_DELAY = 30; # period between vMX monitors status checks

my $DEBUG = shift || 0;
my $APL=$ENV{APL} || '';                                 # directory structure root
my $APL_VAR=$ENV{APL_VAR} || '';                         
my $APL_CONF=$ENV{APL_CONF} || '';                       # conf-files are here - will be depreciated!!!
my $LOG_CFG_FILENAME = "$APL/common/etc/logger.camstat"; # logger settings config file
my $LOG_CFG_REFRESH_INTERVAL = 60;                       # logger config refresh interval
my $ASRV_CONF = "$APL_CONF/asrv.conf";                   # asrv conf file
my $SWITCHER_SIGN = "\nTC8800 > ";                       # sign to add at the end of each responce
my $ERR_OUT_OF_RANGE = "ERROR: argument(s) out of range";
my $dbh;
my $BACKUP_FILE = "$APL_VAR/tmp/advantord.backup";
my $data = "";
my $ASN_VAR = "$APL_VAR/advantor_asn";
my $ASN_CFG = "$ASN_VAR/asn.5.cfg";
my $HOME_PRESET = '1';

# Configuration params
my $PROPAGATE2ELOG  = 0;
my $PROPAGATE2VMX   = 1;
my $SENDHOMEONPOPUP = 0;

use Log::Log4perl "get_logger";
Log::Log4perl::init_and_watch($LOG_CFG_FILENAME, $LOG_CFG_REFRESH_INTERVAL);
my $log = get_logger('ADVANTORD');
# ==============================================================================
sub log_debug { $log->debug(@_); print "@_\n" if $DEBUG; }
sub log_die   { $log->fatal(@_); die @_;}
# ==============================================================================

&daemonize() unless $DEBUG;

my $sighup=0;
$SIG{HUP} = sub {$sighup++;};
# ==============================================================================
#open LOG, ">./advantord.log";
#my $old = select LOG;
#$|=1;
#select $old;
# ==============================================================================

# private data
#my $_select = new IO::Select;
#$_select->add( \*STDOUT ); #put in something that will never be ready +to read :)

my %cam_state = (); # cameras' states
my %cam_watch = (); # whatched cameras
my %cam_devid = (); # devids
my %boxconf   = ();
my %vmxmap    = ();
my %vmx_status= (); # vMX monitor statuses. If status=0, monitor should be resplit
my $asncfg    = {};
my $timer 	  = time + $CHECK_DELAY;
my $timer_vmx     = time + $CHECK_VMX_DELAY;

my $om_client = new OMClient() || log_die('cannot init OM client');

my $elog_client = new AccElogClient() || log_die('cannot init ELog client');

my $vmx_client  = new VMXSimpleClient || log_die('cannot init vMX client');

# ==============================================================================
init(); # connect to DB & read cameras configs

my $self = create();

# ptz(1, 1); # debug test ptz on startup
#print  "test popup & exit\n";
#print "popup res=".$elog_client->send_event(105)."\n";
#exit;

while (1) {
    if($sighup) {
	$sighup = 0;
	$log->info("SIGHUP received.");
	init();
	$self = create();
    }
    log_debug("reading from port..".time);
    read_sda ($self);
    sleep (1);
    
    if ($timer <= time) {
        check_cams();
        $timer += $CHECK_DELAY;
    }
    
    if ($timer_vmx <= time) {
	check_vmx();
	$timer_vmx += $CHECK_VMX_DELAY;
    }
}

# ==============================================================================
sub check_cams
{
    my $schedule_notify = 0;
    log_debug ("checking cams");

    foreach my $id (keys %cam_state) 
    {
      next if $cam_watch{$id} == 0;
      $log->debug( "checking camera ID=$id");
      
      my $oldstate = $cam_state{$id};
      my $newstate = get_cam_state($id);
#$log->info ("oldstate=$oldstate, new state=$newstate");

      if ( defined ($newstate) 
		#&& ($newstate eq STATE_STREAMING || $oldstate eq STATE_STREAMING) 
		&&  $newstate ne $oldstate ) 
	  {
	    $log->info( "state changed ID=$id, $cam_state{$id} -> $newstate");
		$schedule_notify = 1;
      }
      $cam_state{$id} = $newstate;
    }
	# send video-status once for all changed cams
	if ( $schedule_notify ) {
    	notify_state_off ();
	}
}

sub notify_state_off
{
    $log->info ( "notify about state changed!");
    
	my $resp = "video-status\r";
	$resp .= get_status ();
    $resp .= "\r".$SWITCHER_SIGN;
    write_response ($self, $resp);
}

sub create {
    $log->info( "init serial port");

    my $self = {};

    $self->{dev} = $boxconf{'ACC_PORT'} || '/dev/ttyS0';
	my $bd = $boxconf{'ACC_BAUDRATE'} || 9600;
	if 		($bd eq 50)  {$self->{baud} = B50;}
	elsif 	($bd eq 75)  {$self->{baud} = B75;}
	elsif 	($bd eq 110) {$self->{baud} = B110;}
	elsif 	($bd eq 134)  {$self->{baud} = B134;}
	elsif 	($bd eq 150)  {$self->{baud} = B150;}
	elsif 	($bd eq 200)  {$self->{baud} = B200;}
	elsif 	($bd eq 300)  {$self->{baud} = B300;}
	elsif 	($bd eq 600)  {$self->{baud} = B600;}
	elsif 	($bd eq 1200)  {$self->{baud} = B1200;}
	elsif 	($bd eq 1800)  {$self->{baud} = B1800;}
	elsif 	($bd eq 2400)  {$self->{baud} = B2400;}
	elsif 	($bd eq 4800)  {$self->{baud} = B4800;}
	elsif 	($bd eq 9600)  {$self->{baud} = B9600;}
	elsif 	($bd eq 19200)  {$self->{baud} = B19200;}
	elsif 	($bd eq 38400)  {$self->{baud} = B38400;}

	my $iflag = 0;
	my $stop_bits = $boxconf{'ACC_STOPBITS'} || '1';# set CSTOPB if 2
	my $data_bits = $boxconf{'ACC_DATABITS'} || '8';
	my $cflag 	  = CLOCAL | CREAD;
	$cflag		 &= CSTOPB if $stop_bits eq '2'; 
	if ($data_bits eq '7' ) { $cflag		 |= CS7; }
	elsif ($data_bits eq '8' ) { $cflag		 |= CS8; }

	my $parity = $boxconf{'ACC_PARITY'} || 'none';
	if ( $parity eq 'none' ) { $iflag |= IGNPAR; }
	elsif ($parity eq 'odd') { $cflag |= PARENB|PARODD; }
	elsif ($parity eq 'mark') { $iflag |= PARMRK; } 

	my $handshake = $boxconf{'ACC_HANDSHAKE'} || 'none';
	#if ($handshake eq 'on') { $cflag |= CRTSCTS; }

###### New stuff using Device::SerialPort ######

	my $port = Device::SerialPort->new($self->{dev});# | log_debug("Failed to get $self->{dev}: $!");
	$port->databits($data_bits); 
	$port->baudrate($bd);
	$port->stopbits($stop_bits);
	$port->parity($parity);
	$port->handshake($handshake);
	
	$port->write_settings;
	
	$self->{port} = $port;

################################################


    $log->info( "init serial port finished: $self->{dev}:$bd,$data_bits,$parity,$stop_bits $handshake");

    return $self;
}

sub read_sda {


    my $self = shift or return undef;
    my $resp = "";
    my $cmderr = "";
	my $bytes = 0;

	my $part = '';
	while (not $data =~ /\r/) {
		($bytes, $part) = $self->{port}->read($BUFFER_SIZE);

	$log->debug("## Read 1 byte length!") if $bytes == 1; # ?
    	return if (! defined ($bytes) || $bytes <= 1 );       # ?
		$data .= $part;
	    $log->info ( "part IN:[$part], whole data=[$data], data has \\r:".($data =~ /\r/ ? 'yes' : 'no'));
	}

    chomp ($data);
    $log->info ( "IN:\t\t ($bytes): [$data]");
	
   	my @parts = split (/\r/, $data);
    foreach my $cmd (@parts) 
	{
		chomp($cmd);
		next unless $cmd;

        $log->info ( "process cmd:\t\t [$cmd]");

        $resp = "";
        $cmderr = "";

        if ($cmd =~ /^LCM\s(\d+)\s(\d+)$/) { # put camera id=X to monitor
            $cmderr = popup_camera ($1, $2);
            ptz($1, $HOME_PRESET) if $SENDHOMEONPOPUP;
			$cmd = "LCM $1 $2";
        } 
        elsif ($cmd =~ /^prepos\s(\d+)\s(\d+)$/) { # Move PTZ camera id=X to a preset position Y
            $cmderr = ptz ($1, $2);
			$cmd = "prepos $1 $2";
        }
        elsif ($cmd =~ /^LCMP\s(\d+)\s(\d+)(?:\s(\d+))?$/) { # popup + move to preset
    	    $cmderr = popup_camera($1, $2);
	    $cmderr.= ptz($1, $3);
    		if (defined $3) {
    			$cmd = "LCMP $1 $2 $3";
    		}
    		else {
    			$cmd = "LCMP $1 $2";
    		}
        } 
        elsif ($cmd =~ /^video-status$/) {   #report cameras status
            $cmd = 'video-status'.get_status ();
        }
        elsif ($cmd =~ /^INSTALL-CAM\s(\d+)\s(\d+)$/) {  # set watching on|off
            $cmderr = install_cam ($1, $2);
			$cmd = "INSTALL-CAM $1 $2";
        } 
        elsif ($cmd =~ /^UNINSTALL-ALL$/) {  # unset watching for all cameras
            $cmderr = uninstall_all();
			$cmd = 'UNINSTALL-ALL';
        }
        elsif ($cmd =~ /^ver$/) {    # ping request
			$cmd = 'ver';
            #$log->info ( "ping");
            # responce as 'ver'+$SWITCHER_SIGN;
        }
    	else {
            $log->info ( "unsupported input ignored->[$cmd]");
			$data = '';
    		last;
    	}

        # make response
        $resp = $cmd."\r";

        if (! $cmderr eq '') {
            $resp .= "\n$cmderr\n\r";
        } 
        $resp .= $SWITCHER_SIGN;
        write_response ($self, $resp);
        #write_response ($self, $SWITCHER_SIGN);
	} # for each part
	

    $log->info ( "data before strip: [$data]");
	$data =~ s/.*\r(.*)/$1/g;
    $log->info ( "data after strip: [$data]");
}

# from Bosch protocol: page 54, video-status command
#  Each byte contains status information for four cameras in the following format:
# I3 V3 I2 V2 I1 V1 I0 V0
# V0 is bit 0 (the least significant bit); I3 is bit 7 (the most significant). V0 will be 1 if 
# the first camera represented in this byte has video present. I0 will be 1 if system 
# monitoring of the camera input has been enabled. V0 is valid regardless of the value 
# of I0. Subsequent data bytes will each contain data for 4 cameras in the same fashion 
# until status information is received for all inputs.
sub get_status
{
    my $cambytes = ''; 
    my $chsum = 0;  
	my $tmp = 0;

	my $bitpairs_cnt = 1;
	my $byte = 0;
	for (my $id=1; $id < 256*4; ++$id)
	{
		if (exists $cam_watch{$id}) {
		    $tmp = ($cam_watch{$id} ? 2 : 0) + ($cam_state{$id} == STATE_STREAMING ? 1 : 0);
			for (my $i=0;$i<$bitpairs_cnt-1; ++$i) { $tmp <<= 2; }
    		$byte += $tmp;
    		$log->debug("cam_watch[$id] exist, tmp=$tmp, cam_watch=".$cam_watch{$id}.", cam_state=".$cam_state{$id});
#    		$log->info ("!!!byte for $id=$byte (watch=$cam_watch{$id}, state=$cam_state{$id})");
		}
  		
#    		$log->info ("byte for $id=$byte (watch=$cam_watch{$id}, state=$cam_state{$id})");
        if ($bitpairs_cnt % 4 == 0) {
	        $cambytes .= chr ( $byte );
#$log->info ( "cambytes($id)=--->".chr($byte) );
	        $chsum += $byte;
			$byte = 0;
			$bitpairs_cnt = 0;
		}
		#else { $byte <<= 2; }
		$bitpairs_cnt += 1; 
	}

    $chsum = $chsum % 256; # The checksum byte is the low order byte of the sum of the contents of the 128 bytes.
    
    return chr(21).$cambytes.chr ($chsum);
}

# set monitoring on
sub install_cam 
{
    my $camid = shift;
    my $isOn  = shift;

    $log->info("install_cam $camid");

    if (exists $cam_watch{$camid}) {
        $cam_watch{$camid} = $isOn;
        $log->info ("watch $camid=$isOn");
		backup();
    }
    else {
        return $ERR_OUT_OF_RANGE;
    }
    return '';
}

#
sub popup_camera
{
    my $camid = shift;
    my $monid = shift;

    log_debug("popup_camera $camid $monid");
    
    my $clear = ! exists $cam_watch{$camid};
    my $map = $vmxmap{$monid};
    my $err = '';
    
    if ($PROPAGATE2VMX) {
	if ($map && ref($map) eq 'ARRAY') {
    	    my %camids = (); # Those cameras will be moved to home position
    	    foreach my $cell (@$map) {
    		my $vmxid = $cell->[0];
    		# Find out what camera is running in the target cell
    		my $hcells = $vmx_client->get_monitor_state($vmxid);
    		if ($hcells and ref($hcells) eq 'HASH') {
    		    my $cs = $hcells->{"Cell $cell->[1].$cell->[2]"};
    		    $camids{$cs->{srcOBJID}-100} = 1 if $cs && $cs->{srcOBJID}>=100;
    		}
    		else {
    		    $log->error("Cannot retrieve layout for ".
    			"vMX monitor id=$vmxid: $vmx_client->{err}");
    		}
    		    
    		if ($clear) { # CLEAR MONITOR
    		    $log->info("clear cell: $monid, @$cell");
		    $vmx_client->clear_cell(@$cell);
		    if ($vmx_client->{err}) {
			$log->error("Error clearing monitor $monid: $vmx_client->{err}");
			$err = 'vMX cell control error';
		    }
		}
		else { # POPUP CAMERA
		    my $cam_id = $camid+100;
    		    $log->info("Assign stream: monid=$monid -> @$cell");
		    $vmx_client->assign_stream($cam_id, @$cell);
		    if ($vmx_client->{err}) {
			$log->error("Error assigning camera $cam_id to ".
			    "monitor $monid: $vmx_client->{err}");
			$err = 'vMX stream assign error';
		    }
		}
	    }
		
	    # Move cameras that were in cleared/reassigned cells to their home positions
	    $log->info("Move cameras [@{[keys %camids]}] to their home positions");
	    delete $camids{$camid}; # If camera is reassigned, do not move it
	    ptz($_, $HOME_PRESET) foreach keys %camids;
	}
	else {
	    $log->error("popup_camera: no cell associated with monid=$monid");
	}	    
    }
    
    if ($PROPAGATE2ELOG) {
	if ($clear) {
	    $err = $ERR_OUT_OF_RANGE if not $PROPAGATE2VMX;
	}
	else {
	    $elog_client->send_event($camid+100);
	}
    }
    
    return $err;
}

# PTZ camera $1 to preset position $2
sub ptz
{
    my ($id, $preset) = @_;
    $preset = $HOME_PRESET unless defined $preset;

    log_debug("ptz_camera $id to $preset");

    if ( ! exists $cam_state{$id}) 
    {
	   log_debug("ptz_camera $id - no such ID in camera list");
       return $ERR_OUT_OF_RANGE;
    }

	# call PTZ service (write to special local port)
	# ....
	# Direction_C = "<PTZ_Command>&mode=preset&goto=$preset</PTZ_Command>";
	# http://url/ptz/cgi-bin/send_message.pl?data=" + Direction_C.replace(/\&/g, "%26");	

	my $url = "http://localhost/ptz/cgi-bin/send_message.pl?data=<PTZ_Command>dev=$cam_devid{$id}%26mode=preset%26goto=$preset</PTZ_Command>";	
	my $http_agent  = LWP::UserAgent->new;
	my $request 	= HTTP::Request->new(GET => $url);

	my $resp = undef;
    eval {
      local $SIG{ALRM} = sub{ die "request TIMEOUT $!" }; 
	  alarm 2;
	  log_debug("requesting URL->[$url]");

      $resp	= $http_agent->request($request);
    };
    alarm 0;

    if($@ || ! $resp->is_success || $resp->code ne 200) {
	  log_debug("PTZ call failed. URL=[$url], response=[".$resp->content."]");
      return $resp->content;
    }
    else {
		log_debug ('PTZ call OK');
	}
    return '';
}

# init
sub init 
{
    $log->info( "Initialization started");

    mkdir $ASN_VAR unless -d $ASN_VAR;
    
	my $path = "$APL/var/conf/";
    opendir  DH, $path or die "failed to open dir";

    while (my $name = readdir DH) {
        next unless $name =~/^\d+$/;
        if ( ! open (CONF, "<".$path.'/'.$name.'/conf') ) {
			$log->error( "Can't open conf inside $path/$name! ");
			next;
		}
        my %asrv = ();
        while (<CONF>) {
           chomp($_);
           my ($param,$value) = split(/=/,$_);
           $asrv{uc($param)}=$value;
        }
        close (CONF);
	next if (! $asrv{DEVICETYPE} eq 'CAMERA');
        my $objid = $asrv{OBJID} || next;#die "OBJID attribute not found in $name/conf";
        $log->info ("config read for objid=$asrv{OBJID}, type=$asrv{DEVICETYPE}");   
	next if ( $objid =~ /\D+/); # ("OBJID NOT a number in $path/$name/conf - skipping");
	$objid = 0 + $objid -100;  # reduce OBJID by 100 (cam id - 100) to enable whole camera range for AC&C protocol
        $cam_watch{$objid} = 0;
        $cam_state{$objid} = STATE_UNKNOWN;
        $cam_devid{$objid} = $asrv{DEVID};
        $log->info ("config: DEVID=$name -> OBJID=$objid(-100), initial state=$cam_state{$objid}");   
    }
    # read box params
    $log->info( "reading Box params");
    %boxconf = GetAsrv();
    foreach my $id (keys %boxconf) 
    {
        $log->info( "Box param $id=$boxconf{$id}");
    }

    # init node cameras
    $dbh = DBMaster({PrintError => 1});
    my $dbref = $dbh->selectall_arrayref(
	"select obj, name from _objs where otype='D' and subtype='C' and deleted=0"
    );

    if(!$dbref) {
      $log->error( "init(): ERROR getting slave cameras. $DBI::errstr");
	  exit;
    }
     my ($obj,$devid);
     foreach (@{$dbref}) {
        ($obj,$devid) = @{$_};
    	if ( ! exists $cam_watch{$obj-100} ) {
    		$obj = 0 + $obj - 100;  # reduce OBJID by 100 (cam id - 100) to enable whole camera range for AC&C protocol
            $cam_watch{$obj} = 0;
            $cam_state{$obj} = STATE_UNKNOWN;
            $cam_devid{$obj} = $devid;
            $log->info ("SLAVE CAMERA: DEVID=$devid -> OBJID=".($obj+100)."($obj), initial state=$cam_state{$obj}");   
    	}
     }
     
     read_asn_cfg();
     
     init_vmx();


	restore(); # restore watched IDs from backup
}

# ask state in OM
sub get_cam_state 
{
    my $camobjid 	= shift;
	my $state 		= STATE_UNKNOWN;#undef;
	my $err			= '';

	my $om_client = new OMClient();
	if (!$om_client) {
		$log->error('cannot init OM client');
		return;
	}
	
	# make call to OM service via MBus
	$camobjid += 100; # real ID is greater
	log_debug ( "calling getObjects($camobjid)" );

	my @res   = $om_client->getObjects($camobjid, \$err);

	#$log->info ( Data::Dumper->Dump([@res], ["getObjects($camobjid)"]) );
	log_debug ('get_objects result='.(@res ? @res : 'undef').", err?=".(defined $err ? $err : 'no err')."");
	
	if (length $err || ! @res || ! defined ($res[0])) {
		log_debug("err=[$err], res[0] def?->".(defined ($res[0]) ? 'yes' : 'no'));
		return undef;
	}

	my @r = @{$res[0]};
	log_debug ( Data::Dumper->Dump([@r], ["getObjects($camobjid)"]) );
	if (defined ($r[0]->{properties}{'health'}) ) {
		$state = $r[0]->{properties}{'health'};
		$log->info ("getObjects($camobjid)=$state");
	} else {
		$log->error ("get cam state FAILED: 'health' property for OBJID=$camobjid not found in responce");
	}
	log_debug ("return state=".(defined $state ? $state : '')." for $camobjid");

	return $state;
}


#
sub uninstall_all
{
    #$log->info ( "uninstall all");
    foreach my $key (keys %cam_watch) {
        $cam_watch{$key} = 0;
    }
	backup();
    return '';
}


#
sub write_response
{
    my $self = shift;
    my $text = shift;
    return if ($text eq '');

    log_debug ("OUT:\t\t(".(length ($text)).")->[$text]...");
    my $bytes = 1;
    
    $self->{port}->write($text) || log_debug("Failed to write :$!");

    log_debug (defined ($bytes) ? "OK" : "failed");
    #print ('can read='.($_select->can_read( $TIMEOUT_SO ) ? '1' : '0'));
}

sub toHex
{
	my @hex = qw{A B C D E F};
    my $ret = '';
    my $num = shift;
    if ($num == 0) { $ret = '0'; }
    elsif ($num < 10) {
        $ret = chr($num);
    } elsif ($num < 16) { $ret = $hex[$num-10]; }
	else {
		$ret = sprintf("%x", $num);
    }
	#log_debug("return toHex $num=$ret");
   	return $ret;
}

sub ReconnectDB {
    $log->debug( 'ReconnectDB');
  eval {
    local $SIG{ALRM} = sub{ log_die ("TIMEOUT $!") }; 
	alarm 1;
   $dbh = DBI->connect('dbi:Pg:dbname=apl','','',{PrintError => 1});
    $log->debug( 'DB connected');
  };
  alarm 0;
}

#
# Make current process a daemon
#
sub daemonize {
    chdir '/'                  or die "Can't chdir to /: $!";
    #open STDIN, '/dev/null'    or die "Can't read /dev/null: $!";
    #open STDOUT, '>>/dev/null' or die "Can't write to /dev/null: $!";
    #open STDERR, '>>/dev/null' or die "Can't write to /dev/null: $!";
    #defined(my $pid = fork)    or die "Can't fork: $!";
    #exit if $pid;
    #setsid                     or die "Can't start a new session: $!";
    #umask 0002;
}

sub backup()
{
	$log->info("do backup to $BACKUP_FILE");
	if (! open BF, ">$BACKUP_FILE") {
		$log->error("backup failed: $!");
		return;
	}
    print BF "# Advantor daemon backup file. Contains list of Camera IDs to watch disconnections for. Last updated ".timegm(gmtime)."\n";
    my $line= '';
    foreach my $key (keys %cam_watch) {
		$line .= "$key," if $cam_watch{$key} ne 0;
    }
    print BF $line;
	close BF;

	$log->info("backup content: [$line]");
}

sub restore()
{
	$log->info("restore from $BACKUP_FILE");

	if (! open BF, $BACKUP_FILE) {
		$log->error("open $BACKUP_FILE filed: $!");
		return;
	}
	while ( <BF>) {
	    chomp($_);
	    next if (/^#./);
		$log->info("got backuped list [$_]");
    	my @ids = split(/,/);
        foreach my $key (@ids) {
			next if ! $key =~ /\d+/;
            $cam_watch{$key} = 1;
			$cam_state{$key} = 0;
			$log->info("restored watch on OBJID=[$key]");
        }
	}
	close BF;
}

sub read_asn_cfg {
    $asncfg = eval { XMLin($ASN_CFG,ForceArray=>1,KeyAttr=>['name','id']) };
    if ($@) {
	$log->error("Cannot read asn configuration: $@");
	$asncfg = {};
	return;
    }
    $PROPAGATE2ELOG  = $asncfg->{parameter}{propagate2elog}{content} =~ /true/i ? 1 : 0;
    $PROPAGATE2VMX   = $asncfg->{parameter}{propagate2vmx}{content}  =~ /true/i ? 1 : 0;
    $SENDHOMEONPOPUP = $asncfg->{parameter}{sendHomeOnPopup}{content}=~ /true/i ? 1 : 0;
}

sub init_vmx {
    %vmxmap = ();
    %vmx_status = ();
    unless ($asncfg->{vmxconfig}) {	
	$log->warn("No vMX configuration section found in $ASN_CFG");
    }
    else {
	# Wait for vMX daemon to start
	my $wait = 5;
	while (not $vmx_client->get_monitor_list and --$wait >= 0) {
	    $log->info("Waiting for vMX daemon to start...");
	    sleep 5;
	}
	
	foreach my $vmxid (keys %{$asncfg->{vmxconfig}[0]{vmx}}) {
	    my $vmx = $asncfg->{vmxconfig}[0]{vmx}{$vmxid};
	    my $rows = parse_attr($vmx->{rows},'num');
	    my $cols = parse_attr($vmx->{cols},'num');
	    if (!$rows || !$cols) {
		$log->error("Bad dimensions in config for vMX id=$vmxid");
		next;
	    }
	    my $cell = $vmx->{cell};
	    if ($cell && ref($cell) eq 'ARRAY') {
		foreach my $c (@$cell) {
		    my $row = parse_attr($c->{row},'num');
		    my $col = parse_attr($c->{col},'num');
		    if (!$row || !$col || $row>$rows || $col>$cols) {
			$log->warn("Invalid row or col number in config for vMX id=$vmxid");
			next;
		    }
		    my $monid = parse_attr($c->{monitor_id},'num');
		    my @monid_list = parse_attr($c->{monitor_id_list},'num_list');
		    push(@monid_list, $monid) if defined $monid;
		    foreach $monid (@monid_list) {
			next unless defined $monid;
			$vmxmap{$monid} = [] unless $vmxmap{$monid};
			push( @{$vmxmap{$monid}}, [$vmxid,$row-1,$col-1] );
		    }
		}
	    }
	    else {
		$log->warn("No cell mapping defined for vMX monitor id=$vmxid");
	    }
	    
	    # Assign initial layout to vMX monitor
	    $vmx_status{$vmxid} = [0,$rows,$cols];
	    init_vmx_monitor($vmxid);
	}
    }
}

sub init_vmx_monitor {
    my $vmxid = shift;
    my $status = $vmx_status{$vmxid};
    if (! $status) {
	$log->warn("init_monitor() called for MONID=$vmxid which is not in cache");
	return;
    }
    
    my ($rows,$cols) = ($status->[1], $status->[2]);
    $vmx_client->split_monitor($vmxid, $rows, $cols);
    if ($vmx_client->{err}) {
	$log->error("Failed to split vMX monitor id=$vmxid: $vmx_client->{err}");
	$status->[0] = 0;
    }
    else {
	$log->info("vMX monitor id=$vmxid split successfully: $rows rows, $cols cols");
	$status->[0] = 1;
    }
    return $status->[0];
}

sub check_vmx {
    foreach my $vmxid (keys %vmx_status) {
	my ($status,$row,$col) = @{ $vmx_status{$vmxid} };
	next if $status;
	init_vmx_monitor($vmxid);
    }
}

sub parse_attr {
    my ($attr, $type) = @_;
    return unless defined $attr;
    no warnings 'numeric';
    $attr =~ s/^\s+//;
    $attr =~ s/\s+$//;
    if ($type eq 'num') {
	return 0 if $attr eq '0';
	return abs(int($attr)) || undef;
    }
    elsif ($type eq 'num_list') {
	return map {parse_attr($_,'num')} split(/,/,$attr);
    }
}
