"use strict";

/**
 * @class analytics Incapsulates all logic related with editing analytics configuration
 * This functional closely connected with analytics schema (analytics/etc/analytics_schema.json)
 */
var analytics = {

	vaeName: "vca",
	loadAttempts: 0,
	defaultObjSize: 6553, // line size of newly created zone and line
	maxSize: 60000, // max value that x,y coordinates can have
	objid: null,  // id of camera being edited
	props: null,
	cameraAttributes: {}, // Attributes of current camera
	vcaAllowed: false, // VCA tracker on/off
	ovAllowed: false,  // OV tracker on/off
	ovcAllowed: false,  // OV counting tracker on/off
	/* License features for vca and ov tracker */
	features: {
		vca: {},
		ov: {}
	},
	data: {},
	isinit: false,

	/**
	 *
	 * @param {Object} props array of init properties: objid and trackers on/off
	 */
	init: function(props) {

		this.objid = props['objid'];
		this.vcaAllowed = props['vca'] === 'yes';
		this.ovAllowed = props['ov'] === 'yes';
		this.ovcAllowed = props['ovc'] === 'yes';
		this.props = props;
		var self = this;

		$.when(
			$.ajax({url:'/api/call.php', data:{'function': 'getAnalyticsSchema'}, dataType: 'json'}),
			$.ajax({url:'/api/call.php', data:{'function': 'getAttributes', 'obj': this.objid}, dataType:'json', cache: false}),
			this.playerLoaded()
		).done(function(res_schema, res_attributes) {
			try {
				self.schema = $.parseJSON(res_schema[0].list).properties;

				self.cameraAttributes = res_attributes[0].list;
				self.data = $.parseJSON(self.cameraAttributes.VAE_VCA_CONFIG).root;
			} catch (e) {

				if (self.cameraAttributes.AVATARID || !self.cameraAttributes.VAE_VCA_CONFIG) {
					Log.info('Trying to set default VCA configuration');
					$.ajax({
						type: "GET",
						async: false,
						dataType: 'json',
						url: '/sdi/analytics/etc/OVDefaultConfiguration.json',
						success: function(json) {
							self.data = json.root;
							$.ajax({
								type: "POST",
								async: true,
								url: '/api/call.php',
								data: {
									'function': 'setAttributes',
									'obj': self.objid,
									'attributes': JSON.stringify({'VAE_VCA_CONFIG': {root: self.data}})
								}
							});
						}
					});
				} else {
					self.retryInit(props);
				}
			}

			if (self.data.vca_features) {
				self.features.vca = analytics.parseFeatures(analytics.data.vca_features);
			}
			if (self.data.ov_features) {
				self.features.ov  = analytics.parseFeatures(analytics.data.ov_features);
			}

			if (!self.isinit){
				// Build convenient rules list
				self.buildRules();
				self.initPlayer();
				self.initUI();
				self.scheduler.init(self.player.configurator, self.cameraAttributes, analytics.vaeName);

				self.load();
				self.updateZonesList();
				self.addEventHandlers();
				self.checkEncoder();
				self.isinit = true;
			}
		})
		.fail(function() {
			this.retryInit(props);
		});
	},

	checkEncoder: function() {
		if (this.cameraAttributes.ENCODER_SETTING_OVERRIDE === 'no') {
			alert('Currently, the option "encoder programming" is disabled, so all changes made to the configuration will not be applied. To apply changes after save, you must restart the encoder by clicking "Reset request" button in general tab');
		}
	},

	retryInit: function(props) {
		this.loadAttempts += 1;

		if (this.loadAttempts > 10) {
			Log.error('Failed to load analytics configuration!');
			return;
		}
		Log.warning('Failed to load analytics configuration, retry in 2 sec');
		setTimeout(function() {analytics.init(props);}, 2000);
	},

	/**
	 * @returns {Object} jQuery deferred object, it resolves when player plugin fully loaded
 	 */
	playerLoaded: function(d) {
		if (!d) {
			d = new $.Deferred();
		}

		var iframe_player = $('#analyticsPlayer');

		if (!iframe_player.size() || !$('#analyticsPlayer')[0].contentWindow || !$('#analyticsPlayer')[0].contentWindow.document.getElementById('_applet_player_livePlayer')) {
			setTimeout(function() {analytics.playerLoaded(d)}, 100);
		} else {
			d.resolve();
		}

		return d.promise();
	},

	/**
	 * Wait until player loads, then init zone configurator and calibrators
	 */
	initPlayer: function() {
		this.player = null;
		this.player = $('#analyticsPlayer')[0].contentWindow.document.getElementById('_applet_player_livePlayer');
		this.playerJS = $('#analyticsPlayer')[0].contentWindow.window._mplayer_livePlayer;

		if(this.vcaAllowed && this.ovAllowed){
			this.player.initCalibratorVCA();
			this.player.initCalibratorOV();
			this.player.initConfigurator();
			//this.player.configurator.activate();
		}
		else if(this.vcaAllowed){
			this.player.initCalibratorVCA();
			this.player.initConfigurator();
		}
		else if(this.ovAllowed || this.ovcAllowed){
			this.player.initCalibratorOV();
			this.player.initConfigurator();
		}

		// Add callbacks to player
		this.player.configurator.onzonechange = function() {
			analytics.player.configurator.onzoneactivate.apply();
		};

		this.player.configurator.onzoneactivate = function() {
			var objectIndex = analytics.player.configurator.activeObjectIndex;

			// Repaint zones list if nesessary
			if ($('#zones_list option').size() != analytics.player.configurator.objects.length + 1) {
				analytics.updateZonesList();
			}

			$('#zones_list').val(objectIndex);
			if (objectIndex == -1) {
				$('#zone_editor').hide();
			} else {
				$('#zone_editor').show();
				analytics.initDialog(objectIndex);
			}
		};

		// Build array of rules and their parameters, send it to configurator, so configurator is used as storage
		// for all zones and parameters
		var props = {
			trk_objs: [],
			include: 'include',
			active: true,
			state: 'active',
			postures: []
		};
		for (var i in this.rules) {
			props[i] = false;
			if (this.rules[i].properties) {
				for (var j in this.rules[i].properties) {
					props[j] = this.rules[i].properties[j]['default'];
				}
			}
		}
		this.player.configurator.addObjectProperties(props);

		if(this.ovAllowed || this.ovcAllowed){
			this.player.calibrator_ov.samples = this.data.ov_calib_set;
		}
	},

	/**
	 * Add callbacks to all needed UI events
	 */
	addEventHandlers: function() {

		$("#tabs_menu").tabs("#tabs_content > div", {
			onClick: function(event, tabIndex) {
				if (!analytics.player)
					return;

				// Switching player contexts depending on active tab
				switch (this.getTabs()[tabIndex].id) {
					case 'tab_zones': analytics.player.configurator.activate();
						break;
					case 'tab_calibration_vca': analytics.calibrator_vca.activate();
						break;
					case 'tab_calibration_ov': analytics.player.calibrator_ov.activate();
						break;
					default: analytics.player.player.activate();
				}
			}
		});
		$("#tabs_menu").data('tabs').click('zones');
		$('#add_zone').click(function() {analytics.addObject('zone');});
		$('#add_line').click(function() {analytics.addObject('line');});
		$('#add_multiline').click(function() {analytics.addObject('multiline');});
		$('#remove_object').click(function() {analytics.removeObject()});
		$('#button_apply').click(function() {analytics.save()});

		$('#button_addSample').click(function() {analytics.player.calibrator_ov.addSample()});
		$('#button_removeSample').click(function() {
			analytics.player.calibrator_ov.removeActiveSample()
		});

		// Zone rules activate/deactivate
		$('#zone_rules input:checkbox').change(function() {
			if (this.checked) {
				$(this).parent().siblings('ul').show();
			} else {
				$(this).parent().siblings('ul').hide();
			}
		});

		// Zone type: detect / nondetect
		$('#zone_editor select[name="type"]').change(function() {
			if (this.value === 'nondetection') {
				$('#zone_object_filter, #zone_rules').hide();
			} else {
				$('#zone_object_filter, #zone_rules').show();
			}
		});

		// Toggle abobj and rmobj rules depending on "Abandoned/removed object detection" option
		$('#pane_channel input:checkbox[name="enableabobj"]').change(function() {
			if (this.checked) {
				$('#rule_abobj, #rule_rmobj').show();
			} else {
				$('#rule_abobj, #rule_rmobj').hide();
			}
		}).change();

		// Toggle counting line rules depending on "Counting line" option
		$('#pane_channel input:checkbox[name="enablecntline"]').change(function() {
			if (this.checked && this.dataObj && this.dataObj.style === 'line') {

				$('#rule_linecounter_a, #rule_linecounter_b').show();
			} else {
				$('#rule_linecounter_a, #rule_linecounter_b').hide();
			}
		}).change();

		$('#zones_list').change(function() {
			if ($(this).val()) {
				analytics.player.configurator.activeObjectIndex = $(this).val();
			}
		}).change();

		$('#trk_objs').change(function() {
			if ($(this).data('value')) {
				analytics.dataObj.trk_objs = $(this).data('value');
			}
		});
	},

	/**
	 * Parse features list from analytics schema and represent it in convenient JS hash (so we can easily check whether some feature exists or not)
	 * @param {Array} features array of features given by analytics schema
	 * @returns {Object} Hash containing features as keys
	 */
	parseFeatures: function(features) {
		var i;
		this.rules = {};
		var result = {};

		// Build convenient features list
		for (i = 0; i < features.length; i++) {
			result[features[i]] = 1;

			// Exceptions, one feature adds several rules
			if (features[i] === 'abobj') {
				result['rmobj'] = 1;
			}
			if (features[i] === 'linecounter') {
				result['linecounter_a'] = 1;
				result['linecounter_b'] = 1;
			}
		}
		return result;
	},

	/**
	 * Init UI accordingly to available trackers, features and analytics schema
	 */
	initUI: function() {
		$('#form_channel_settings').append(this.formBuilder.getFormHtml(this.schema.channel_set.properties));

		if (this.vcaAllowed) {
			$('#form_vca_channel').append(this.formBuilder.getFormHtml(this.schema.vca_channel_set.properties));

			$('#pane_channel select[name="meaunits"]').parent().parent().remove();

			if (this.features.vca.tamper) {
				$('#form_tamper').append(this.formBuilder.getFormHtml(this.schema.vca_tamper_set.properties));
			} else {
				$('#tab_tamper, #pane_tamper').remove();
			}

			this.classificator.init();

		} else {
			$('#form_vca_channel').remove();
			$('#tab_tamper, #pane_tamper, #zone_object_filter, #detection_type').remove();
			$('#pane_classification, #tab_classification').remove();
		}

		if (this.ovAllowed || this.ovcAllowed) {
			this.formBuilder.buildOVForm('#form_ov_channel', this.schema.ov_channel_set.properties, this.schema.ov_categories);

			//UI changes connected with OV tracker features: rename "zones" to "rules", etc.
			$('#tab_zones').html('Rules');
			$('#zone_editor_legend').html('Rule editor');
			$('#add_zone').html('Add area');
			$('#zones_list_legend').html('Rules list');

			$('#tab_channel').html('View settings');
			$('#zone_rules').prepend('<li class="rules_header">Behaviours</li>');
			$('#rule_shape_direction, #rule_minimum, #rule_maximum, #rule_size_change').first().before('<li class="rules_header">Filters</li>');
		} else {
			$('#form_ov_channel, #add_multiline').remove();
		}

		// tracking: people, surveillance
		if (!this.features.vca.trackingengine && !this.features.ov.trackingengine) {
			$('#pane_channel select[name="trackmode"] option[value="surv"]').remove();
		}
		if (!this.features.vca.peopletracking && !this.features.ov.peopletracking) {
			$('#pane_channel select[name="trackmode"] option[value="people"]').remove();
		}

		if (!this.vcaAllowed || !this.features.vca.calibration) {
			$('#tab_calibration_vca, #pane_calibration_vca').remove();
		} else {
			this.calibrator_vca.init(this.schema.vca_calib_set);
		}

		if ((!this.ovAllowed && !this.ovcAllowed) || !this.features.ov.calibration) {
			$('#tab_calibration_ov, #pane_calibration_ov').remove();
		} else {
			this.calibrator_ov.init(this.schema.ov_calib_set);
		}

		$('#tab_eagle_i, #pane_eagle_i').remove();

		$('#pane_channel :range, #pane_tamper :range')
			.rangeinput({
				css: {input:  'range-input', slider: 'range-slider', progress: 'range-progress', handle: 'range-handle'},
				change: function(event,value){
							this.getInput().change();
				}
			});

		$('#form_ov_channel select[multiple]').jaoselect();
	},

	/**
	 * Add HTML list with event rules to zone editor depending on schema and available features
	 */
	buildRules: function() {

		// Get all rules description from schema
		var rules = this.schema.event_set.items.properties.rule_type['enum'];
		for (var i=0; i < rules.length; i++) {
			// get item name in some freaky way because of inconvenient data structure
			for (var itemName in rules[i]) {}

			// Remove rules that license doesn't contain
			if (!(this.vcaAllowed && this.features.vca[itemName]) && !((this.ovAllowed || this.ovcAllowed) && this.features.ov[itemName]))
				continue;

			this.rules[itemName] = rules[i][itemName];

			// Add rule prefix to each rule property to correct building of html form
			// (and to provide easy datalinking with player) ex.: speed filter: speed_upperbound, speed_lowerbound
			if (this.rules[itemName].properties) {
				var temp = {};

				for (var j in this.rules[itemName].properties) {
					temp[itemName + '_' + j] = this.rules[itemName].properties[j];
				}
				this.rules[itemName].properties = temp;
			}

			$('#zone_rules').append(this.formBuilder.getRuleHtml(itemName, this.rules[itemName]));
		}

		$('#zone_rules :range[min][max]').rangeinput({
			css: {input:  'range-input', slider: 'range-slider', progress: 'range-progress', handle: 'range-handle'},
			change: function(event,value){
				this.getInput().change();
			}
		});
		var deadlock = false;
		$('#zone_rules :range[min][max]').change(
			function(event, value) {
				if(deadlock)
					return;
				deadlock = true;
				$(this).closest('.input_wrapper').find('.range-input').change();
				deadlock = false;
			});
	},

	/**
	 * Update contents of zones selector
	 */
	updateZonesList: function() {
		var objects = this.player.configurator.objects;
		var html;

		html = '<option value="-1">' + '-- Select zone --' + '</option>';
		if (this.ovAllowed) {
			html = '<option value="-1">' + '-- Select rule --' + '</option>';
		}

		for (var i = 0; i < objects.length; i++) {
			html += '<option value="' + i + '">' + objects[i].name + '</option>';
		}
		$('#zones_list').html(html);
	},

	/**
	 * Fill zone editor fields with data of currently selected zone in configurator,
	 * provide datalinking between them
	 * @param objectIndex index of selected zone
	 */
	initDialog: function(objectIndex) {
		var dialog = $('#zone_editor');

		if (this.dataObj) {
			try {
				dialog.unlink(this.dataObj);
			} catch(e) {}
		}

		this.dataObj = this.player.configurator.objects[objectIndex];
		dialog.link(this.dataObj, {
			'jaoselect_trk_objs': null,  // exclude classificator selector checkboxes
			'jaoselect_postures': null
		});

		// Dialog view depending on object style: polygon, line, multiline
		$('#zone_rules, #zone_rules > li, #zone_object_filter').show();
		$('#detection_type').show();
		if (this.dataObj.style === 'polygon') {
			dialog.find('#rule_linecounter_a, #rule_linecounter_b, #rule_cross, #rule_cross2, #rule_order').hide();

			if ($('#rule_presence').size()) {
				var html = $('#rule_presence').html().replace('Cross (any direction)', 'Inside');
				$('#rule_presence').html(html);
			}

			if (!this.ovcAllowed) {
				$('#rule_counting').hide();
			}
		}
		if (this.dataObj.style === 'line' || this.dataObj.style === 'multiline') {
			dialog.find('#detection_type').hide();
			if (!$('#pane_channel input:checkbox[name="enablecntline"]').prop("checked")) {
				$('#rule_linecounter_a, #rule_linecounter_b').hide();
			}
			$('#rule_enter, #rule_exit, #rule_appear, #rule_disappear, #rule_stop, #rule_dwell, #rule_loiters, #rule_abobj, #rule_rmobj').hide();

			if (this.dataObj.style === 'line') {
				$('#rule_cross2, #rule_order').hide();
			}

			if (this.dataObj.style === 'multiline') {
				$('#rule_cross').hide();
			}

			if (!this.ovcAllowed) {
				$('#rule_counting').hide();
			}

			if (this.ovAllowed) {
				$('#rule_presence').hide();
			} else {
				if ($('#rule_presence').size()) {
					// Change "inside rule" on "cross any direction" for VCA
					html = $('#rule_presence').html().replace('Inside', 'Cross (any direction)');
					$('#rule_presence').html(html);
				}
			}
		}

		// Set correct values of form fields
		for (var i in this.dataObj) {

			if ((typeof this.dataObj[i] === 'function' || $.isPlainObject(this.dataObj[i]) || typeof this.dataObj[i] === 'undefined') && i != 'trk_objs')
				continue;

			var field = dialog.find('[name="' + i + '"]');
			if (field.hasClass('text')) {
				field.val('');
			}

			$(this.dataObj).triggerHandler( "changeField!", [i, this.dataObj[i]] );
			if (field.hasClass('range-input')) {
				field.data("rangeinput").setValue(this.dataObj[i]);
			}
			field.change();
		}
	},

	/**
	 * Add object in configurator in random place
	 * @param {String} type Type of object to be added, [zone|line]
	 */
	addObject: function(type) {
		var x = Math.random() * (this.maxSize - this.defaultObjSize),
		    y = Math.random() * (this.maxSize - this.defaultObjSize);

		switch (type) {
			case 'zone':
				this.player.configurator.addZone([x, y, x+this.defaultObjSize, y, x+this.defaultObjSize, y+this.defaultObjSize, x, y+this.defaultObjSize]);
			break;
			case 'line':
				this.player.configurator.addPath([x, y, x+this.defaultObjSize, y+this.defaultObjSize]);
			break;
			case 'multiline':
				var x2 = Math.random() * (this.maxSize - this.defaultObjSize),
				    y2 = Math.random() * (this.maxSize - this.defaultObjSize);
				this.player.configurator.addMultiline([x, y, x+this.defaultObjSize, y+this.defaultObjSize], [x2, y2, x2+this.defaultObjSize, y2+this.defaultObjSize]);
		}
		this.player.configurator.activeObjectIndex = this.player.configurator.objects.length-1;
	},

	/**
	 * Remove currently selected object from configurator
	 */
	removeObject: function() {
		if (this.player.configurator.activeObjectIndex >= 0) {
			this.player.configurator.removeObject(this.player.configurator.activeObjectIndex);
			this.player.configurator.onzoneactivate.apply();
		}
	},

	handleOVRestart: function() {
		if (!this.ovAllowed)
			return;

		Log.info('Please wait until retriever restarts and OV analytics update configuration');
		var self = this;

		$('#tabs_content').mask({text: 'Please wait until retriever restarts and changes will update'});

		if (!this.playerJS){
			this.playerJS = $('#analyticsPlayer')[0].contentWindow.window._mplayer_livePlayer;
		}
		this.playerJS.listener.onStatusChange = function(iplayer, command, status/*, errorCode*/) {
			if (status == 1 || status == 2) {
				self.playerJS.listener.onStatusChange = function() {};

				$.getJSON('/api/call.php', {'function': 'getAttribute', 'obj': self.objid, 'attribute': 'VAE_VCA_CONFIG', '_': Math.random()},
				function(res_data) {
					self.cameraAttributes.VAE_VCA_CONFIG = res_data.value;
					self.data = $.parseJSON(self.cameraAttributes.VAE_VCA_CONFIG).root;
					self.scheduler.updateCamAttributes(self.cameraAttributes);
					self.load();
					$('#tabs_content').unmask();
				});
			}
		}

	},

	/**
	 * Close analytics iframe and return to admin GUI (works only if page was opened from analytics tab in admin GUI)
	 */
	close: function() {
	},

	/**
	 * Serialize all configuration data into json and save it to server
	 */
	save: function() {

		var self = this;

		delete this.data.zone_set;
		delete this.data.event_set;

		// Try..catch block to catch user input errors
		try {
			$('input.error').removeClass('error');

			$.extend(this.data, this.serializeZones());

			this.data.channel_set = this.formBuilder.serializeForm('#form_channel_settings', this.schema.channel_set.properties);

			// Saving VCA data
			if (this.vcaAllowed) {
				this.data.vca_channel_set = this.formBuilder.serializeForm('#form_vca_channel', this.schema.vca_channel_set.properties);
				this.data.vca_cls_set     = this.classificator.serialize();

				// VCA Calibration set
				if (this.features.vca.calibration) {
					this.data.vca_calib_set = this.calibrator_vca.serialize();
				}

				// VCA Tamper set
				if (this.features.vca.tamper) {
					this.data.vca_tamper_set = this.formBuilder.serializeForm('#pane_tamper', this.schema.vca_tamper_set.properties);
				}
			}

			// Saving OV data
			if (this.ovAllowed) {
				this.data.ov_channel_set = this.formBuilder.serializeForm('#form_ov_channel', this.schema.ov_channel_set.properties);

				// OV Calibration set
				if (this.features.ov.calibration) {
					this.data.ov_calib_set = this.player.calibrator_ov.samples;
				}
			}

			// Saving OVC data
			if (this.ovcAllowed) {
				this.data.ov_channel_set = this.formBuilder.serializeForm('#form_ov_channel', this.schema.ov_channel_set.properties);

				// OV Calibration set
				if (this.features.ov.calibration) {
					this.data.ov_calib_set = this.player.calibrator_ov.samples;
				}
			}

			$.when(
				$.ajax({
					type: "POST",
					async: true,
					url: '/api/call.php',
					data: {
						'function': 'setAttributes',
						'obj': this.objid,
						'attributes': JSON.stringify({'VAE_VCA_CONFIG': {root: this.data}})
					}
				}),
				this.scheduler.save()
			).done(function() {
				Log.info('Configuration saved succesfully!');
				self.handleOVRestart();
			});

			this.handleOVRestart();

		} catch(e) {
			if (e.name != 'inputError')
				throw e;

			if (typeof e.prepare === 'function') {
				e.prepare();
			}

			// Get index of pane which contains error element
			var pane = $(e.field).parents('#panes > div');
			var paneIndex = $('#panes > div').index(pane);

			// Open pane, focus index field and add error class to it
			$('#tabs_menu > div').eq(paneIndex).click();
			$(e.field).addClass('error').focus();
			Log.error(e.message);
		}
	},

	/**
	 * Serialize zones and events data from configurator into json
	 * @returns {Object} Object with saved data
	 */
	serializeZones: function() {

		var zones = [];
		var events = [];
		var rulesCount = 0;

		for (var i = 0; i < this.player.configurator.objects.length; i++) {
			var obj = this.player.configurator.objects[i];
			zones.push({
				id: i,
				name: obj.name,
				type: obj.type,
				style: obj.style,
				color: obj.color,
				point_set: obj.point_set,
				include: obj.include || 'include',
				trk_objs: this.classificator.cleanTrkObjects(obj.trk_objs)
			});

			// Ignore all rules if nondetect zone
			if (obj.type === 'nondetection' && this.vcaAllowed)
				continue;

			for (var name in this.rules) {

				// If this rule disabled in zone, ignore it
				if (!obj[name])
					continue;

				var event = {
					id: rulesCount++,
					name: name,
					zone_id: i,
					trk_objs: this.classificator.getTrkObjects(obj.trk_objs, obj.include),
					rule_type: {}
				};
				event.rule_type[name] = {};
				if (this.rules[name].properties) {
					for (var prop in this.rules[name].properties) {
						if (this.rules[name].properties[prop].type === 'integer')
							obj[prop] = parseInt(obj[prop], 10);
						if (this.rules[name].properties[prop].type === 'number')
							obj[prop] = parseFloat(obj[prop]);

						// Set corrent names of rule properties, delete rule name from their names
						event.rule_type[name][prop.slice(name.length + 1)] = obj[prop];
					}
				}
				if ($.isEmptyObject(event.rule_type[name])) {
					event.rule_type[name] = null;
				}

				events.push(event);
			}
		}

		return {zone_set: zones, event_set: events};
	},

	/**
	 * Parse previously stored zones and events data and push it to configurator
	 * @param {Object} zone_set Zones data
	 * @param {Object} event_set Events data
	 */
	unserializeZones: function(zone_set, event_set) {
		var objects = [];

		for (var i=0; i < zone_set.length; i++) {
			var zone = zone_set[i];

			if (!zone.include) {
				zone.include = 'include';
			}
			if (!zone.trk_objs) {
				zone.trk_objs = [];
			}
			objects.push({
				id: i,
				name: zone.name,
				type: zone.type,
				style: zone.style,
				color: zone.color,
				point_set: zone.point_set,
				include: zone.include,
				trk_objs: zone.trk_objs
			});
		}

		for (i=0; i < event_set.length; i++) {
			var event = event_set[i];
			if (!objects[event.zone_id]) {
				continue;
			}
			zone = objects[event.zone_id];

			zone[event.name] = true;
			if (event.rule_type[event.name]) {
				for (var propName in event.rule_type[event.name]) {
					zone[event.name + '_' + propName] = event.rule_type[event.name][propName];
				}
			}
		}
		this.player.configurator.objects = objects;
	},

	/**
	 * Parse previously saved data and fill with data all input fields, configurator and calibrators
	 */
	load: function() {
		if (!this.data)
			return;

		// General channel settings
		this.formBuilder.unserializeForm('#form_channel_settings', this.data.channel_set);

		// VCA data loading
		if (this.vcaAllowed) {
			this.formBuilder.unserializeForm('#form_vca_channel', this.data.vca_channel_set);

			if (this.data.vca_cls_set) {
				this.classificator.unserialize(this.data.vca_cls_set);
			}

			// VCA Calibration set
			if (this.features.vca.calibration) {
				this.calibrator_vca.unserialize(this.data.vca_calib_set);
			}

			// VCA Tamper set
			if (this.features.vca.tamper) {
				this.formBuilder.unserializeForm('#pane_tamper', this.data.vca_tamper_set);
			}
		}

		if (this.ovAllowed || this.ovcAllowed) {
			this.formBuilder.unserializeForm('#form_ov_channel', this.data.ov_channel_set);

			// OV Calibration set
			if (this.features.ov.calibration) {
				//this.calibrator_ov.unserialize(this.data.ov_calib_set);
			}
		}

		// Zones and events
		if (this.data.zone_set && this.data.event_set) {
			this.unserializeZones(this.data.zone_set, this.data.event_set);
		}

		this.scheduler.unserialize();
	}
};
