define("vaxocr_configurator", ["mithril", "shared", "i18n", "jquery", "msemediaplayer", "vistimeline", "aod", "coverage", "jcanvas", "api", "log"], function () {
	"use strict";

	let {__} = require("i18n");
	let MSEMediaPlayer = require("msemediaplayer");
	let {VisTimeLine} = require("vistimeline");
	let {Frame} = require("shared");
	let AODAPI = require("aod").AODAPI;
	let AODWait = require("aod").AODWait;
	let {getCoverage} = require("coverage");

	//CONFIG
	let parameters = window.location.href.match(/\/sdi\/vae\/([0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12})\/([^_]+)\/index\.html\?conf=([^&]*)(&type=(.*))?$/);
	let objid = parameters[1];
	let name = decodeURIComponent(parameters[3]);
	const styles = {
		'main': ['/sdi/global/global.css', '/sdi/global/editor.css', '/sdi/player/mediaplayer.css', '/sdi/lib/js/3rdparty/vis/css/vis.min.css', '/sdi/timeline/vistimeline.css', 'style.css'],
		'simple': ['simplestyle.css', '/sdi/player/mediaplayercontrols.css', '/sdi/player/mediaplayer.css', '/sdi/lib/js/3rdparty/vis/css/vis.min.css', '/sdi/timeline/vistimeline.css']
	};
	let configsArray = [];
	// configurator type main|simple
	const type = parameters[5] ? decodeURIComponent(parameters[5]) : 'main';
	let headElem = document.getElementsByTagName('head')[0];
	for(let i=0; i<styles[type].length; i++){
		let link  = document.createElement('link');
		link.rel = 'stylesheet';
		link.type = 'text/css';
		link.href = styles[type][i];
		headElem.appendChild(link);
	}
	if(type === 'simple'){
		let url = window.parent.location.protocol+'//'+window.parent.location.host+window.parent.location.pathname;
		let styleElem = $('head', window.parent.document).find('link[rel="stylesheet"]').clone();
		$(styleElem).attr('href', url + $(styleElem).attr('href')).appendTo('head');
		$('head', window.parent.document).find('style').clone().appendTo('head');
	}
	let canvas = {
		e: false,
		w: 640,
		h: 360,
		aspectW: 1,
		aspectH: 1
	};
	let states = {
		player: {
			load: false,
			e: null,
			firstLoad: true
		},
		config: {
			load: true,
			vae_engines: {
				VAE_ENGINES: {}
			},
			isActiveConfig: false,
			isActiveEngine: false,
			firstLoadComplete: false
		},
		current_tab: "alpr_base",
		tabs: {
			showZones: false
		}
	};
	let tabList = [
		{name: "alpr_base", text: __("Base"), isActive: true},
		{name: "alpr_regions", text: __("Countries and Regions / US States")},
		{name: "alpr_zones", text: __("Zones")}//,
		//{name: "alpr_instructions", text: __("Instructions")}
	];
	let colorArray = [16711680,65280,255,16776960,16711935,65535,16753920];
	let colorLabels = [__("Red"), __("Green"), __("Blue"), __("Yellow"), __("Fuchsia"), __("Aqua"), __("Orange")];
	let roi = {
		min: 0,
		max: 65535
	};

	let regions = {
		countries: {
			name: __("Countries"),
			values: ["Afghanistan","Albania","Algeria","American Samoa","Andorra","Angola","Anguilla","Antarctica","Antigua and Barbuda","Argentina","Armenia","Aruba","Australia","Austria","Azerbaijan","Bahamas","Bahrain","Bangladesh","Barbados","Belarus","Belgium","Belize","Benin","Bermuda","Bhutan","Bolivia","Bosnia","Botswana","Brazil","Brunei","Bulgaria","Burkina Faso","Burundi","Cambodia","Cameroon","Canada","Cape Verde","Cayman Islands","Central African Republic","Chad","Chile","China","Christmas Island","Cocos Islands","Colombia","Comoros","Congo","Cook Islands","Costa Rica","Croatia","Cuba","Cyprus","Czech Republic","Denmark","Djibouti","Dominica","Dominican Republic","East Timor","Ecuador","Egypt","El Salvador","Equatorial Guinea","Eritrea","Estonia","Ethiopia","Falkland Islands","Faroe Islands","Fiji","Finland","France","French Guiana","French Polynesia","French Southern Territories","Gabon","Gambia","Georgia","Germany","Ghana","Gibraltar","Great Britain","Greece","Greenland","Grenada","Guadeloupe","Guam","Guatemala","Guinea","Guinea Bissau","Guyana","Haiti","Holy See","Honduras","Hong Kong","Hungary","Iceland","India","Indonesia","Iran","Iraq","Ireland","Israel","Italy","Ivory Coast","Jamaica","Japan","Jordan","Kazakhstan","Kenya","Kiribati","Kosovo","Kuwait","Kyrgyzstan","Lao","Latvia","Lebanon","Lesotho","Liberia","Libya","Liechtenstein","Lithuania","Luxembourg","Macau","Macedonia","Madagascar","Malawi","Malaysia","Maldives","Mali","Malta","Marshall Islands","Martinique","Mauritania","Mauritius","Mayotte","Mexico","Micronesia","Moldova","Monaco","Mongolia","Montenegro","Montserrat","Morocco","Mozambique","Myanmar","Namibia","Nauru","Nepal","Netherlands","Netherlands Antilles","New Caledonia","New Zealand","Nicaragua","Niger","Nigeria","Niue","North Korea","Northern Mariana Islands","Norway","Oman","Pakistan","Palau","Palestinia","Panama","Papua New Guinea","Paraguay","Peru","Philippines","Pitcairn Island","Poland","Portugal","Puerto Rico","Qatar","Reunion Island","Romania","Russia","Rwanda","Saint Kitts and Nevis","Saint Lucia","Saint Vincent and the Grenadines","Samoa","San Marino","Sao Tome and Principe","Saudi Arabia","Senegal","Serbia","Seychelles","Sierra Leone","Singapore","Slovakia","Slovenia","Solomon Islands","Somalia","South Africa","South Korea","South Sudan","Spain","Sri Lanka","Sudan","Suriname","Swaziland","Sweden","Switzerland","Syria","Taiwan","Tajikistan","Tanzania","Thailand","Tibet","Timor Leste","Togo","Tokelau","Tonga","Trinidad and Tobago","Tunisia","Turkey","Turkmenistan","Turks and Caicos Islands","Tuvalu","Ugandax","Ukraine","United Arab Emirates","United Kingdom","United States","Uruguay","Uzbekistan","Vanuatu","Vatican","Venezuela","Vietnam","Virgin Islands British","Virgin Islands USA","Wallis and Futuna Islands","Western Sahara","Yemen","Zambia","Zimbabwe","European Trailer","USADOT","ADR"]
		},
		"United States": {
			name: __("United States"),
			values: ["Alabama","Alaska","Arizona","Arkansas","California","Colorado","Connecticut","Delaware","District Columbia","Florida","Georgia","Hawaii","Idaho","Illinois","Indiana","Iowa","Kansas","Kentucky","Louisiana","Maine","Maryland","Massachusetts","Michigan","Minnesota","Mississippi","Missouri","Montana","Nebraska","Nevada","New Hampshire","New Jersey","New Mexico","New York","North Carolina","North Dakota","Ohio","Oklahoma","Oregon","Pennsylvania","Puerto Rico","Rhode Island","South Carolina","South Dakota","Tennessee","Texas","Utah","Vermont","Virginia","Washington","West Virginia","Wisconsin","Wyoming"]
		}
	};

	let validationValues = {
		// COMMON
		plate_cut_out_position: {
			tooltip: false,
			tooltipText: "",
			label: __("License Plate cut-out superimposed"),
			values: ['left', 'right', 'none'],
			labels: [__('Left'), __('Right'), __('None')],
			simpleLabels: [__('Left'), __('Right'), __('None')],
			type: "select",
			vType: "string",
			simpleLabel: __("License Plate cut-out superimposed:"),
			simpleType: 'select',
			size: 1,
			defaultValue: "right",
			actualValue: "right",
			disabled: false,
			name: "plate_cut_out_position",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		time_between_events: {
			tooltip: true,
			tooltipText: __("Time between events in seconds"),
			simpleTooltip: true,
			simpleTooltipText: __("Time between events in seconds"),
			label: __("Time between events, sec"),
			min: 0,
			max: 5,
			step: 0.1,
			defaultValue: 0.5,
			actualValue: 0.5,
			type: "slider",
			vType: "float",
			disabled: false,
			name: "time_between_events",
			simpleLabel: __("Time between events, sec:"),
			simpleType: 'slider',
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		scans_per_event: {
			tooltip: true,
			tooltipText: __("Number of scans plate recognizer will make per event"),
			simpleTooltip: true,
			simpleTooltipText: __("Number of scans plate recognizer will make per event"),
			label: __("Scans per event"),
			min: 1,
			max: 10,
			step: 1,
			defaultValue: 2,
			actualValue: 2,
			type: "slider",
			vType: "int",
			disabled: false,
			name: "scans_per_event",
			simpleLabel: __("Scans per event:"),
			simpleType: 'slider',
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		time_between_scans: {
			tooltip: true,
			tooltipText: __("Time between scans in seconds"),
			simpleTooltip: true,
			simpleTooltipText: __("Time between scans in seconds"),
			label: __("Time between scans, sec"),
			min: 0,
			max: 1,
			step: 0.05,
			defaultValue: 0.1,
			actualValue: 0.1,
			type: "slider",
			vType: "float",
			disabled: false,
			name: "time_between_scans",
			simpleLabel: __("Time between scans, sec:"),
			simpleType: 'slider',
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		// MD-related
		md: {
			tooltip: false,
			tooltipText: "",
			label: __('Detect Motion'),
			values: [true, false],
			type: "checkbox",
			vType: "bool",
			defaultValue: false,
			actualValue: false,
			disabled: false,
			name: "md",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		min_object_width: {
			tooltip: true,
			tooltipText: __("Detection will ignore vehicles that are too large. This is a good efficiency technique to use if the vehicles are going to be a fixed distance away from the camera (e.g., you will never see vehicles that fill up the entire image)"),
			simpleTooltip: true,
			simpleTooltipText: __("Detection will ignore vehicles that are too large. This is a good efficiency technique to use if the vehicles are going to be a fixed distance away from the camera (e.g., you will never see vehicles that fill up the entire image)"),
			label: __("Min Vehicles Width (pixels)"),
			min: 10,
			step: 1,
			defaultValue: 110,
			actualValue: 110,
			max: canvas.w,
			type: "slider",
			vType: "int",
			disabled: false,
			name: "min_object_width",
			simpleLabel: __("Min Vehicles Width in px:"),
			simpleType: 'textButton',
			simpleButtonLabel: __("Set"),
			simpleDisabled: true,
			simpleSelected: false,
			simpleButtonAction: function(setFalse){
				if(setFalse){
					this.simpleSelected = false;
					let params = {
						visible: this.simpleSelected
					};
					canvas.e.setLayer('MaxObjectRect', params);
				}else{
					validationValues.min_object_height.simpleButtonAction(true);
					validationValues.simple_zone.simpleButtonAction(true);
					this.simpleSelected = !this.simpleSelected;
					let params = {
						visible: this.simpleSelected,
						width: Math.floor(this.actualValue*canvas.aspectW),
						height: Math.floor(validationValues.min_object_height.actualValue*canvas.aspectH)
					};
					canvas.e.setLayer('MaxObjectRect', params).drawLayers();
				}
			},
			setActualValue: function(v, draw){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
					if(!draw) {
						canvas.e.setLayer('MaxObjectRect', {
							width: Math.floor(newVal*canvas.aspectW)
						}).drawLayers();
					}
				}
			}
		},
		min_object_height: {
			tooltip: true,
			tooltipText: __("Detection will ignore vehicles that are too large. This is a good efficiency technique to use if the vehicles are going to be a fixed distance away from the camera (e.g., you will never see vehicles that fill up the entire image)"),
			simpleTooltip: true,
			simpleTooltipText: __("Detection will ignore vehicles that are too large. This is a good efficiency technique to use if the vehicles are going to be a fixed distance away from the camera (e.g., you will never see vehicles that fill up the entire image)"),
			label: __("Min Vehicles Height (pixels)"),
			min: 10,
			step: 1,
			defaultValue: 80,
			actualValue: 80,
			max: canvas.h,
			type: "slider",
			vType: "int",
			disabled: false,
			name: "min_object_height",
			simpleLabel: __("Min Vehicles Height in px:"),
			simpleType: 'textButton',
			simpleButtonLabel: __("Set"),
			simpleDisabled: true,
			simpleSelected: false,
			simpleButtonAction: function(setFalse){
				if(setFalse){
					this.simpleSelected = false;
					let params = {
						visible: this.simpleSelected
					};
					canvas.e.setLayer('MaxObjectRect', params);
				}else{
					validationValues.min_object_width.simpleButtonAction(true);
					validationValues.simple_zone.simpleButtonAction(true);
					this.simpleSelected = !this.simpleSelected;
					let params = {
						visible: this.simpleSelected,
						width: Math.floor(validationValues.min_object_width.actualValue*canvas.aspectW),
						height: Math.floor(this.actualValue*canvas.aspectH)
					};
					canvas.e.setLayer('MaxObjectRect', params).drawLayers();
				}
			},
			setActualValue: function(v, draw){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
					if(!draw) {
						canvas.e.setLayer('MaxObjectRect', {
							height: Math.floor(newVal*canvas.aspectH)
						}).drawLayers();
					}
				}
			}
		},
		// VaxOCR-specific parameters
		min_chars_height: {
			tooltip: true,
			tooltipText: __("Minimum characters height in pixels"),
			simpleTooltip: true,
			simpleTooltipText: __("Minimum characters height in pixels"),
			label: __("Minimum characters height"),
			min: 14,
			max: 70,
			step: 1,
			defaultValue: 14,
			actualValue: 14,
			simpleLabel: __("Minimum characters height:"),
			simpleType: 'slider',
			type: "slider",
			vType: "int",
			disabled: false,
			name: "min_chars_height",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		max_chars_height: {
			tooltip: true,
			tooltipText: __("Maximum characters height in pixels"),
			simpleTooltip: true,
			simpleTooltipText: __("Maximum characters height in pixels"),
			label: __("Maximum characters height"),
			min: 14,
			max: 70,
			step: 1,
			defaultValue: 30,
			actualValue: 30,
			simpleLabel: __("Maximum characters height:"),
			simpleType: 'slider',
			type: "slider",
			vType: "int",
			disabled: false,
			name: "max_chars_height",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		min_global_confidence: {
			tooltip: true,
			tooltipText: __("Plate read minimum global confidence to consider"),
			simpleTooltip: true,
			simpleTooltipText: __("Plate read minimum global confidence to consider"),
			label: __("Confidence Threshold %"),
			min: 10,
			max: 100,
			step: 1,
			defaultValue: 80,
			actualValue: 80,
			simpleLabel: __("Confidence Threshold %:"),
			simpleType: 'slider',
			type: "slider",
			vType: "int",
			disabled: false,
			name: "min_global_confidence",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		min_char_confidence: {
			tooltip: true,
			tooltipText: __("Minimum confidence per character read"),
			simpleTooltip: true,
			simpleTooltipText: __("Minimum confidence per character read"),
			label: __("Minimum character confidence %"),
			min: 0,
			max: 100,
			step: 1,
			defaultValue: 70,
			actualValue: 70,
			simpleLabel: __("Character Confidence Threshold %:"),
			simpleType: 'slider',
			type: "slider",
			vType: "int",
			disabled: false,
			name: "min_char_confidence",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		grammar_strict: {
			tooltip: true,
			tooltipText: __("Plate read must match any country defined grammar"),
			label: __('Grammar Strict'),
			values: [true, false],
			type: "checkbox",
			vType: "bool",
			defaultValue: false,
			actualValue: false,
			disabled: false,
			name: "grammar_strict",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		min_num_plate_chars: {
			tooltip: true,
			tooltipText: __("Minimum number of characters the plate read shall have"),
			simpleTooltip: true,
			simpleTooltipText: __("Minimum number of characters the plate read shall have"),
			label: __("Minimum characters in plate"),
			min: 5,
			max: 12,
			step: 1,
			defaultValue: 5,
			actualValue: 5,
			simpleLabel: __("Minimum characters in plate:"),
			simpleType: 'slider',
			type: "slider",
			vType: "int",
			disabled: false,
			name: "min_num_plate_chars",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		max_num_plate_chars: {
			tooltip: true,
			tooltipText: __("Maximum umber of characters the plate read shall have"),
			simpleTooltip: true,
			simpleTooltipText: __("Maximum umber of characters the plate read shall have"),
			label: __("Maximum characters in plate"),
			min: 5,
			max: 12,
			step: 1,
			defaultValue: 11,
			actualValue: 11,
			simpleLabel: __("Maximum characters in plate:"),
			simpleType: 'slider',
			type: "slider",
			vType: "int",
			disabled: false,
			name: "max_num_plate_chars",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		max_slop_angle: {
			tooltip: true,
			tooltipText: __("Plate maximum slop angle"),
			simpleTooltip: true,
			simpleTooltipText: __("Plate maximum slop angle"),
			label: __("Plate maximum slop angle"),
			min: 0,
			max: 40,
			step: 1,
			defaultValue: 30,
			actualValue: 30,
			simpleLabel: __("Plate maximum slop angle:"),
			simpleType: 'slider',
			type: "slider",
			vType: "int",
			disabled: false,
			name: "max_slop_angle",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		background_mode: {
			tooltip: true,
			tooltipText: __("Plate foreground/background contrast"),
			label: __('Background Mode'),
			values: [1, 2, 3],
			labels: [__('Dark on light'), __('Light on dark'), __('Both')],
			simpleLabels: [__('Dark on light'), __('Light on dark'), __('Both')],
			type: "select",
			vType: "int",
			simpleLabel: __('Background Mode:'),
			simpleType: 'select',
			defaultValue: 1,
			actualValue: 1,
			disabled: false,
			name: "background_mode",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		ocr_complexity: {
			tooltip: true,
			tooltipText: __("OCR algorithm complexity"),
			label: __('OCR Complexity'),
			values: [1, 2, 3],
			labels: [__('Low'), __('Normal'), __('High')],
			simpleLabels: [__('Low'), __('Normal'), __('High')],
			type: "select",
			vType: "int",
			simpleLabel: __('OCR Complexity:'),
			simpleType: 'select',
			defaultValue: 2,
			actualValue: 2,
			disabled: false,
			name: "ocr_complexity",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		find_plate_depth: {
			tooltip: true,
			tooltipText: __("Plates finder algorithm depth"),
			label: __('Plates Depth'),
			values: [1, 2, 3],
			labels: [__('Low'), __('Normal'), __('High')],
			simpleLabels: [__('Low'), __('Normal'), __('High')],
			type: "select",
			vType: "int",
			simpleLabel: __('Plates Depth:'),
			simpleType: 'select',
			defaultValue: 1,
			actualValue: 1,
			disabled: false,
			name: "find_plate_depth",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		detect_multiline_plate: {
			tooltip: true,
			tooltipText: __("The OCR will read or discard double line plates"),
			label: __('Detect Multiline Plates'),
			values: [true, false],
			type: "checkbox",
			vType: "bool",
			defaultValue: true,
			actualValue: true,
			disabled: false,
			name: "detect_multiline_plate",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		enable_multicountry_nn: {
			tooltip: true,
			tooltipText: __("Enable/Disable the use of different neural networks when the intialization includes multiple countries"),
			label: __('Multicountry Neural Networks'),
			values: [true, false],
			type: "checkbox",
			vType: "bool",
			defaultValue: false,
			actualValue: false,
			disabled: false,
			name: "enable_multicountry_nn",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		num_threads: {
			tooltip: true,
			tooltipText: __("Number of threads to use"),
			label: __("Number of threads"),
			min: 1,
			max: 16,
			step: 1,
			defaultValue: 1,
			actualValue: 1,
			type: "slider",
			vType: "int",
			disabled: false,
			name: "num_threads",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		// VaxMMC-specific parameters
		mmc_type: {
			tooltip: true,
			tooltipText: __("Predict vehicle make, model, color and possibly class. Set parameter to one of ('vehicle make-model-color', 'vehicle classification', 'both') if you sure that activated license contain such feature(s) to get vehicle make, model, color and possibly class or set it to 'disable' to completely disable it"),
			label: __('Vehicle MMC and classification:'),
			disabled_tooltip: true,
			disabled_tooltipText: __("Predict vehicle make, model, color and possibly class. Set parameter to one of ('vehicle make-model-color', 'vehicle classification', 'both') if you sure that activated license contain such feature(s) to get vehicle make, model, color and possibly class or set it to 'disable' to completely disable it"),
			values: [0, 1, 2, 3],
			labels: [__('Disabled'), __('Vehicle make-model-color'), __('Vehicle classification'), __('Both MMC and classification')],
			simpleLabels: [__('Disabled'), __('Vehicle make-model-color'), __('Vehicle classification'), __('Both MMC and classification')],
			type: "select",
			vType: "int",
			simpleLabel: __('Vehicle MMC and classification:'),
			simpleType: 'select',
			defaultValue: 0,
			actualValue: 0,
			disabled: false,
			name: "mmc_type",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		mmc_quality: {
			tooltip: true,
			tooltipText: __("Vehicle MMC and Classification quality"),
			label: __('MMC Quality'),
			values: [1, 2, 3],
			labels: [__('Normal'), __('High'), __('Maximum')],
			simpleLabels: [__('Normal'), __('High'), __('Maximum')],
			type: "select",
			vType: "int",
			simpleLabel: __('MMC and Classification quality:'),
			simpleType: 'select',
			defaultValue: 1,
			actualValue: 1,
			disabled: false,
			name: "mmc_quality",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		mmc_min_global_confidence: {
			tooltip: true,
			tooltipText: __("MMC and Classification minimum global confidence required"),
			simpleTooltip: true,
			simpleTooltipText: __("MMC and Classification minimum global confidence required"),
			label: __("MMC Confidence Threshold %"),
			min: 20,
			max: 100,
			step: 1,
			defaultValue: 20,
			actualValue: 20,
			simpleLabel: __("MMC Confidence Threshold %:"),
			simpleType: 'slider',
			type: "slider",
			vType: "int",
			disabled: false,
			name: "mmc_min_global_confidence",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
				}
			}
		},
		//ZONES
		simple_zone: {
			defaultValue: false,
			actualValue: false,
			simpleLabel: __("Enable Detection Zone:"),
			simpleType: 'checkboxButton',
			simpleButtonLabel: __("Show"),
			simpleDisabled: true,
			simpleSelected: false,
			simpleButtonAction: function(setFalse){
				if(setFalse){
					this.simpleSelected = false;
					let params = {
						visible: this.simpleSelected
					};
					canvas.e.setLayerGroup('zones', params);
				}else{
					validationValues.min_object_height.simpleButtonAction(true);
					validationValues.min_object_width.simpleButtonAction(true);
					this.simpleSelected = !this.simpleSelected;
					let params = {
						visible: this.simpleSelected
					};
					let id = Object.keys(validationValues.zones)[0];
					canvas.e.setLayerGroup('zone'+id, params).drawLayers();
				}
				this.simpleButtonLabel = this.simpleSelected ? __("Hide") : __("Show");
			},
			values: [true, false],
			type: "checkbox",
			vType: "boolean",
			disabled: false,
			name: "simple_zone",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if(newVal !== this.actualValue) {
					this.actualValue = newVal;
					if(newVal){
						let id = Object.keys(validationValues.zones)[0];
						if(!id){
							addZone();
						}else{
							validationValues.zonesItems.setActualValue(id);
						}
						this.simpleDisabled = false;
						this.simpleButtonAction();
					}else{
						delete validationValues.zones[validationValues.zonesItems.actualValue];
						canvas.e.removeLayerGroup('zone'+validationValues.zonesItems.actualValue);
						validationValues.zonesItems.setActualValue("-1");
						this.simpleDisabled = true;
						this.simpleButtonAction(true);
					}
				}
			}
		},
		zonesButtonSet: {
			type: "button_set",
			items: [
				{
					onclick: function(){
						addZone();
					},
					disabled: false,
					label: __("Add Zone"),
					disable: function(){
					}
				},
				{
					onclick: function(){
						delete validationValues.zones[validationValues.zonesItems.actualValue];
						canvas.e.removeLayerGroup('zone'+validationValues.zonesItems.actualValue);
						validationValues.zonesItems.setActualValue("-1");
						validationValues.zonesButtonSet.items[2].disable();
					},
					disabled: true,
					label: __("Remove"),
					disable: function(){
						this.disabled = validationValues.zonesItems.actualValue === -1;
					}
				},
				{
					onclick: function(){
						Object.keys(validationValues.zones).map(function(v){
							delete validationValues.zones[v];
						});
						canvas.e.removeLayerGroup('zones');
						validationValues.zonesItems.setActualValue("-1");
						validationValues.zonesButtonSet.items[2].disable();
					},
					disabled: true,
					label: __("Remove All"),
					disable: function(){
						this.disabled = !(Object.keys(validationValues.zones).length > 0);
					}
				}
			]
		},
		zonesItems: {
			tooltip: false,
			tooltipText: "",
			type: "selectList",
			list: "zones",
			size: 1,
			vType: "int",
			firstValue: -1,
			firstLabel: __("--Select Zone--"),
			defaultValue: -1,
			actualValue: -1,
			disabled: false,
			name: "zones_items",
			setActualValue: function(v){
				let newVal = validateValue(this, v);
				if (newVal !== this.actualValue) {
					this.actualValue = newVal;
					validationValues.zonesButtonSet.items[1].disable();
					canvas.e.setLayerGroup('zones', {
						opacity: 0.3
					}).setLayerGroup('zoneCircles', {
						radius: 3
					});
					if (newVal !== -1) {
						let lastLayer = canvas.e.getLayer(-1);
						let layerGroup = canvas.e.getLayerGroup('zoneCircles' + newVal);
						let j = 0;
						layerGroup.map(function (l) {
							canvas.e.moveLayer(l.name, lastLayer.index - j);
							j++;
						});
						canvas.e.moveLayer("zonePolygon" + newVal, lastLayer.index - j)
							.setLayerGroup('zone' + newVal, {
								opacity: 0.7
							}).setLayerGroup('zoneCircles' + newVal, {
								radius: 5
							});
					}
					canvas.e.drawLayers();
				}
			}
		},
		zones: {},
	};

	let regionKeys = Object.keys(regions);
	for(let i=0; i<regionKeys.length; i++){
		validationValues[regionKeys[i]] = {
			tooltip: false,
			tooltipText: "",
			simpleTooltip: false,
			simpleTooltipText: "",
			label: regions[regionKeys[i]].name,
			values: regions[regionKeys[i]].values,
			labels: regions[regionKeys[i]].values,
			type: "multi-select",
			vType: "string",
			size: 10,
			simpleSize: 4,
			defaultValue: [],
			actualValue: [],
			disabled: false,
			display: i === 0,
			name: regionKeys[i],
			simpleLabel: regions[regionKeys[i]].name + ':',
			simpleLabels: regions[regionKeys[i]].values,
			simpleType: 'multi-select',
			setActualValue: function(v){
				let self = validationValues[regionKeys[i]];
				if(self.values.indexOf(v) === -1){
					return;
				}
				let index = self.actualValue.indexOf(v);
				if(index === -1){
					self.actualValue.push(v);
					if(typeof regions[v] !== "undefined")
						validationValues[v].display = true;
				}else{
					self.actualValue.splice(index, 1);
					if(typeof regions[v] !== "undefined") {
						validationValues[v].actualValue = [];
						validationValues[v].display = false;
					}
				}
				m.redraw();
			}
		}
	}

	let pointOnLine = function(point, l1, l2, lineThickness){
		let d = lineThickness*2;
		let P = {
			x: Math.round(point[0]),
			y: Math.round(point[1])
		};
		let A = {
			x: Math.round(l1[0]),
			y: Math.round(l1[1])
		};
		let B = {
			x: Math.round(l2[0]),
			y: Math.round(l2[1])
		};
		if(A.x === B.x && A.y === B.y){
			return false;
		}
		if(P.x > (A.x - d) && P.x < (A.x + d) && P.y > (A.y - d) && P.y < (A.y + d)){
			return false;
		}
		if(P.x > (B.x - d) && P.x < (B.x + d) && P.y > (B.y - d) && P.y < (B.y + d)){
			return false;
		}
		if((P.x >= Math.max(A.x-d, B.x-d) && P.y >= Math.max(A.y-d, B.y-d)) || (P.x <= Math.min(A.x+d, B.x+d) && P.y <= Math.min(A.y+d, B.y+d))){
			return false;
		}

		let delta = Math.abs((B.y-A.y)*P.x-(B.x-A.x)*P.y+B.x*A.y-B.y*A.x)/Math.sqrt(Math.pow(B.y-A.y, 2)+Math.pow(B.x-A.x, 2));

		return delta <= d;
	};

	let createZoneCircle = function(index, i, x, y, visible, hexColor){
		return	{
			type: "arc",
			name: "zoneCircle"+index+"-"+i,
			layer: true,
			draggable: true,
			groups: ['zone'+index, 'zoneCircles'+index, 'zones', 'zoneCircles'],
			dragGroups: ['zone'+index],
			drag: function(self){
				if(self.x < 0){
					self.x = 0;
				}else if(self.x > canvas.w){
					self.x = canvas.w;
				}
				if(self.y < 0){
					self.y = 0;
				}else if(self.y > canvas.h){
					self.y = canvas.h;
				}
				let poly = canvas.e.getLayer("zonePolygon"+index);
				poly["x"+self.data.index] = self.x;
				poly["y"+self.data.index] = self.y;
			},
			dragstop: function(){
				changeZoneBoundaries(index)
			},
			mousedown: function(self){
				self.dragGroups = [];
				validationValues.zonesItems.setActualValue("" + index);
				m.redraw();
			},
			mouseup: function(self){
				self.dragGroups = ['zone'+index];
			},
			data: {
				index: i+1
			},
			strokeStyle: hexColor,
			strokeWidth: 2,
			fillStyle: hexColor,
			radius: visible ? 5 : 3,
			visible: visible,
			opacity: visible ? 0.7 : 0.3,
			x: x,
			y: y
		};
	};

	let addZone = function(index, params){
		const lineThickness = 3;
		let colorIndex = 0;
		let newZone = true;
		if(typeof index === "undefined") {
			params = {};
			params.point_set = [
				{
					x: 95,
					y: 85
				},
				{
					x: 95,
					y: 150
				},
				{
					x: 195,
					y: 150
				},
				{
					x: 195,
					y: 85
				}
			];
			let arr = Object.keys(validationValues.zones);
			for (index = 0; index < arr.length; index++) {
				if (arr.indexOf("" + index) === -1) {
					break;
				}
				colorIndex++;
				if(colorIndex % colorArray.length === 0){
					colorIndex = 0;
				}
			}
		}else{
			for(let i=0; i<params.point_set.length; i++){
				params.point_set[i].x = Math.round((canvas.w / roi.max) * params.point_set[i].x);
				if(params.point_set[i].x > canvas.w){
					params.point_set[i].x = canvas.w;
				}else if(params.point_set[i].x < 0){
					params.point_set[i].x = 0;
				}
				params.point_set[i].y = Math.round((canvas.h / roi.max) * params.point_set[i].y);
				if(params.point_set[i].y > canvas.h){
					params.point_set[i].y = canvas.h;
				}else if(params.point_set[i].y < 0){
					params.point_set[i].y = 0;
				}
			}
			newZone = false;
			colorIndex = colorArray.indexOf(params.color);
			if(colorIndex === -1){
				colorIndex = 0;
			}
		}
		let hexColor = decToHexColor(colorArray[colorIndex]);
		let polygon = {
			type: "line",
			fillStyle: hexColor,
			strokeStyle: hexColor,
			strokeWidth: 2,
			opacity: newZone ? 0.7 : 0.3,
			layer: true,
			draggable: true,
			bringToFront: false,
			visible: newZone,
			groups: ['zone'+index, 'zonePolygon', 'zones'],
			dragGroups: ['zone'+index],
			name: "zonePolygon"+index,
			closed: true,
			data: {
				points: params.point_set.length
			},
			dblclick: function(self) {
				let point = [self.eventX, self.eventY];
				let line = 0;
				for(let i = 1; i <= self.data.points; i++){
					let l1 = [self["x"+i], self["y"+i]];
					let l2 = [];
					if(i<self.data.points){
						l2 = [self["x"+(i+1)], self["y"+(i+1)]];
					}else{
						l2 = [self["x1"], self["y1"]];
					}
					if(pointOnLine(point, l1, l2, lineThickness)){
						line = i;
						break;
					}
				}
				if(line){
					for(let i = self.data.points; i > line; i--){
						let circle = canvas.e.getLayer("zoneCircle"+index+"-"+(i-1));
						self["x"+(i+1)] = self["x"+i];
						self["y"+(i+1)] = self["y"+i];
						circle.data.index += 1;
						canvas.e.setLayer("zoneCircle"+index+"-"+(i-1),{
							name: "zoneCircle"+index+"-"+i
						});
					}
					self["x"+(line+1)] = self.eventX;
					self["y"+(line+1)] = self.eventY;
					self.data.points = self.data.points+1;
					let circle = createZoneCircle(index, line, self.eventX, self.eventY, true, self.strokeStyle);
					canvas.e.addLayer(circle).drawLayers();
				}
			},
			mousedown: function() {
				validationValues.zonesItems.setActualValue("" + index);
				m.redraw();
			},
			drag: function(self){
				let circles = canvas.e.getLayerGroup('zoneCircles'+index);
				if(self["x"+self.data.minXIndex] + self.x < 0){
					self.x -= self["x"+self.data.minXIndex] + self.x;
					self.dx = 0;
					for(let i = 0; i < self.data.points; i++){
						circles[i].x = circles[i].data.minX;
					}
				}else if(self["x"+self.data.maxXIndex] + self.x > canvas.w){
					self.x = canvas.w - self["x"+self.data.maxXIndex];
					self.dx = 0;
					for(let i = 0; i < self.data.points; i++){
						circles[i].x = circles[i].data.maxX;
					}
				}
				if(self["y"+self.data.minYIndex] + self.y < 0){
					self.y -= self["y"+self.data.minYIndex] + self.y;
					self.dy = 0;
					for(let i = 0; i < self.data.points; i++){
						circles[i].y = circles[i].data.minY;
					}
				}else if(self["y"+self.data.maxYIndex] + self.y > canvas.h){
					self.y = canvas.h - self["y"+self.data.maxYIndex];
					self.dy = 0;
					for(let i = 0; i < self.data.points; i++){
						circles[i].y = circles[i].data.maxY;
					}
				}
			},
			dragstop: function(self){
				for(let i = 1; i <= self.data.points; i++){
					self["x"+i] += self.x;
					self["y"+i] += self.y;
				}
				self.x = 0;
				self.y = 0;
				self.dx = 0;
				self.dy = 0;
			}
		};
		for(let i = 0; i < params.point_set.length; i++){
			polygon["x"+(i+1)] = params.point_set[i].x;
			polygon["y"+(i+1)] = params.point_set[i].y;
			let circle = createZoneCircle(index, i, params.point_set[i].x, params.point_set[i].y, newZone, hexColor);
			canvas.e.addLayer(circle);
		}
		canvas.e.addLayer(polygon).drawLayers();
		changeZoneBoundaries(index);

		validationValues.zones[index] = {
			name: {
				tooltip: false,
				label: __("Name"),
				tooltipText: "",
				type: "input_text",
				maxlength: 30,
				vType: "string",
				defaultValue: Gettext.strargs(__("Zone %1"), [index]),
				actualValue: Gettext.strargs(__("Zone %1"), [index]),
				disabled: false,
				name: "zones_item_name",
				setActualValue: function(v){
					let newVal = validateValue(this, v);
					if(newVal !== this.actualValue) {
						this.actualValue = newVal;
					}
				}
			},
			color: {
				tooltip: false,
				label: __("Color"),
				tooltipText: "",
				values: colorArray,
				labels: colorLabels,
				type: "selectColor",
				size: 1,
				vType: "int",
				defaultValue: 16711680,
				actualValue: colorArray[colorIndex],
				disabled: false,
				name: "zone_color",
				setActualValue: function(v){
					let newVal = validateValue(this, v);
					if(newVal !== this.actualValue) {
						this.actualValue = newVal;
						let hexColor = decToHexColor(newVal);
						canvas.e.setLayerGroup('zone' + index, {
							strokeStyle: hexColor,
							fillStyle: hexColor
						}).drawLayers();
					}
				}
			}
		};
		if(newZone) {
			validationValues.zonesItems.setActualValue("" + index);
			validationValues.zonesButtonSet.items[2].disable();
		}
	};

	let validateValue = function(params, newVal){
		if (params.type === "select" || params.type === "checkbox" || params.type === "selectColor" || params.type === "selectList") {
			let arr = [];
			if(typeof newVal === "string"){
				if(params.vType === "bool"){
					newVal = newVal === "true";
				}else if(params.vType === "int"){
					newVal = parseInt(newVal);
				}
			}
			if (params.type === "selectList"){
				arr = Object.keys(validationValues[params.list]);
				arr = arr.map(function(v){return parseInt(v)});
				arr.push(-1);
			}else{
				arr = params.values.slice();
			}
			if (arr.indexOf(newVal) === -1) {
				newVal = params.defaultValue;
			}
		} else if (params.type === "slider") {
			if(newVal !== "") {
				let dot = false;
				if(typeof newVal === "string") {
					if (params.vType === "int") {
						newVal = parseInt(newVal);
					} else if (params.vType === "float") {
						if (newVal.charAt(newVal.length - 1) === ".") {
							newVal = newVal.replace(/\.+/g, ".");
							if (newVal === ".") {
								newVal = "";
							}
							if (newVal.split(".").length > 2) {
								newVal = parseFloat(newVal);
							} else {
								dot = true;
							}
						} else {
							let valArr = newVal.split(".");
							if(valArr.length > 1){
								newVal = valArr[0] + "." + valArr[1].slice(0,2);
							}
							newVal = parseFloat(newVal);
						}
					}
				}
				if(!dot) {
					if (!isNaN(newVal)) {
						if (newVal < params.min) {
							newVal = params.min;
						} else if (newVal > params.max) {
							newVal = params.max;
						}
					} else {
						newVal = params.defaultValue;
					}
				}
			}
		}
		return newVal;
	};

	let changeZoneBoundaries = function(index){
		let poly = canvas.e.getLayer("zonePolygon"+index);
		let circles = canvas.e.getLayerGroup('zoneCircles'+index);

		let min = [poly.x1, poly.y1];
		let max = [poly.x1, poly.y1];
		poly.data.minXIndex = 1;
		poly.data.minYIndex = 1;
		poly.data.maxXIndex = 1;
		poly.data.maxYIndex = 1;
		for (let i = 2; i <= poly.data.points; i++){
			if(poly["x"+i]<min[0]){
				min[0] = poly["x"+i];
				poly.data.minXIndex = i;
			}
			if(poly["y"+i]<min[1]){
				min[1] = poly["y"+i];
				poly.data.minYIndex = i;
			}
			if(poly["x"+i]>max[0]){
				max[0] = poly["x"+i];
				poly.data.maxXIndex = i;
			}
			if(poly["y"+i]>max[1]){
				max[1] = poly["y"+i];
				poly.data.maxYIndex = i;
			}
		}
		for(let i = 0; i < poly.data.points; i++){
			circles[i].data.minX = circles[i].x - min[0];
			circles[i].data.minY = circles[i].y - min[1];
			circles[i].data.maxX = circles[i].x + (canvas.w - max[0]);
			circles[i].data.maxY = circles[i].y + (canvas.h - max[1]);
		}
	};

	let changeCanvasHeight = function(heightRatio){
		let layers = canvas.e.getLayers();
		for(let i = 0; i < layers.length; i++){
			let layer = layers[i];
			if(layer.type === "line"){
				for(let j = 1; j <= layer.data.points; j++){
					layer["y"+j] = Math.round(layer["y"+j]*heightRatio);
				}
			}else{
				layer.y = Math.round(layer.y*heightRatio);
			}
		}
		let Index = Object.keys(validationValues.zones);
		for(let i = 0; i < Index.length; i++){
			changeZoneBoundaries(Index[i]);
		}
		canvas.e.drawLayers();
	};

	let calculateDimensions = function(newWidth, newHeight){
		validationValues.min_object_height.max = newHeight;
		validationValues.min_object_width.max = newWidth;
		let oldWidth = canvas.w;
		let ratio = newWidth / newHeight;
		newHeight = Math.round(oldWidth / ratio);
		if (canvas.h !== newHeight) {
			$(".player").height(newHeight).find(".mediaplayer").height(newHeight);
			let heightRatio = newHeight / canvas.h;
			canvas.h = newHeight;
			changeCanvasHeight(heightRatio);
		}
		canvas.aspectW = canvas.w / newWidth;
		canvas.aspectH = canvas.h / newHeight;
		validationValues.min_object_height.setActualValue(validationValues.min_object_height.actualValue);
		validationValues.min_object_width.setActualValue(validationValues.min_object_width.actualValue);
		m.redraw();
	};

	const Player = {
		obj: objid,
		player: null,
		timeline: null,
		isPlay: false,
		isArchive: false,
		avatar_coverage: {0: [0, 0], 1: [0, 0]},
		aodWait: new AODWait(),
		onupdate: function(){
			if(canvas.e) {
				canvas.e.drawLayers();
			}
		},
		oncreate: function (vnode) {
			vnode.state.timeline = new VisTimeLine({
				node: vnode.dom.querySelector(".timeline"),
				isDebug: false
			});
			let timeline = vnode.state.timeline;

			states.player.e = new MSEMediaPlayer({node: vnode.dom.querySelector(".video")});
			vnode.state.player = states.player.e;
			const player = vnode.state.player;

			Promise.all([
				timeline.init(),
				player.init(null, {streamid: MSEMediaPlayer.STREAM_FULL, obj: vnode.state.obj, isMetaDataDraw: states.config.isActiveEngine && states.config.isActiveConfig, showControlsOnPause: false})
			])
				.then(function () {
					timeline.setParameters({
						selection: {
							fixed: false,
							visible: false
						},
						scale: {granularity: "min"}
					});
					timeline.createThumb(vnode.state.obj);

					timeline.onGetData = function (line, beginTime, endTime, granularity) {
						let maxMinutes = screen.width / 36;
						beginTime -= maxMinutes * 60 * 1000;
						endTime += maxMinutes * 60 * 1000;

						granularity = "chunk";
						getCoverage(vnode.state.obj, Math.round(beginTime / 1000), Math.round(endTime / 1000), granularity)
							.then(function ({list, avatar_coverage}) {
								timeline.setData(line, list);
								vnode.state.avatar_coverage = avatar_coverage;
							});
					};

					timeline.onTimeChange = function (time, changedByUser) {
						if (changedByUser) {
							vnode.state.player.setTime(vnode.state.obj, time);
							vnode.state.isArchive = true;

							m.redraw();
						}
					};

					player.subscribe("play", function (param) {
						vnode.state.isPlay = true;

						vnode.state.aodWait.clear();
						if(states.player.firstLoad) {
							calculateDimensions(param.width, param.height);
							states.player.load = false;
							states.player.firstLoad = false;
						}
					});

					player.subscribe("pause", function (code) {
						vnode.state.isPlay = false;
					});

					player.subscribe("stop", function (code) {
						vnode.state.isPlay = false;

						let transportType = player.getTransportType();
						if (vnode.state.isArchive
							&&
							transportType === MSEMediaPlayer.TRANSPORT_WEBSOCKET
							&&
							(code === MSEMediaPlayer.E_VIDEO_UNAVAILABLE
							|| code === MSEMediaPlayer.E_END_OF_ARCHIVE)) {
							player.printErrorMessage(__("Request archive from avatar"));

							let start = timeline.getTime();
							let streamid = MSEMediaPlayer.STREAM_FULL;
							vnode.state.aodWait.add(vnode.attrs.obj, start, undefined, streamid)
								.then((request) => {
									timeline.updateData();
									if (request.status === AODAPI.STATUS_COMPLETED) {
										player.play(vnode.attrs.obj, start);
									} else
									if (![AODAPI.STATUS_CANCELED, AODAPI.STATUS_FAILED].includes(request.status)) {
										vnode.state.aodWait.once(`request.${request.requestid}`, () => {
											timeline.updateData();
											player.play(vnode.attrs.obj, start);
										});
									}
								})
								.catch(function (e) {
									Log.error(e.message);
									player.printErrorMessage(e.message);
								});
						}
					});

					player.subscribe("frame", function (timestamp, width, height) {
						if (!timestamp) return;
						timestamp = parseInt(timestamp);

						timeline.setTime(timestamp, false)
							.then(function () {
								vnode.state.timestamp = timestamp;
								m.redraw();
							})
							.catch(function () {
								// maybe we set time not in user selection (if defined)
								player.pause();
							});
					});

					player.subscribe("live", function () {
						vnode.state.isArchive = false;
						m.redraw();
					});
					player.subscribe("archive", function () {
						vnode.state.isArchive = true;
						m.redraw();
					});

					timeline.setTime(Date.now(), true);
				});
		},
		view: function (vnode) {
			let timestampString = "";
			if (vnode.state.timestamp) {
				let date = new Date();
				date.setTime(vnode.state.timestamp);
				timestampString = String(date.getHours()).padStart(2, "0") + ":" + String(date.getMinutes()).padStart(2, "0") + ":" + String(date.getSeconds()).padStart(2, "0") + "." + String(date.getMilliseconds()).padStart(3, "0");
			}

			let transportType = vnode.state.player ? vnode.state.player.getTransportType() : "";
			return m("#player_wrapper", {
				class: (
					(
						type === 'simple' &&
						(
							validationValues.min_object_width.simpleSelected ||
							validationValues.min_object_height.simpleSelected ||
							validationValues.simple_zone.simpleSelected
						)
					)
					? 'no-hover'
					: 'hover'
				),
				style: {
					height: type === 'simple' ? canvas.h+"px" : ""
				}
			}, [
				m(".player[data-obj=" + vnode.state.obj + "]", [
					m(".video"),
					m("canvas", {id: "canv", width: canvas.w, height: canvas.h})
				]),
				m(".controls", [
					m(".timeline"),
					m(".player-controls", [
						m(".left", [
							!vnode.state.isPlay && m("i.fas.fa-play", {
								onclick: function () {
									vnode.state.player.play();
								}
							}),
							vnode.state.isPlay && m("i.fas.fa-pause", {
								onclick: function () {
									vnode.state.player.pause();
								}
							}),
							m(".timestamp", timestampString),
							m("button.player-controls-button.live-badge", {
								disabled: !vnode.state.isArchive,
								onclick: function () {
									vnode.state.player.play(vnode.state.obj);
								}
							}, __("LIVE")),
						]),
						m(".right", [
							transportType && (transportType === MSEMediaPlayer.TRANSPORT_WEBSOCKET ? m("i.fas.fa-cloud") : m("i.fas.fa-home")),
							!vnode.state.isPlay && m("i.fas.fa-cloud-upload-alt", {
								onclick: function () {
									let start = vnode.state.timeline.getTime();
									let streamid = MSEMediaPlayer.STREAM_FULL;
									vnode.state.aodWait.add(vnode.state.obj, start, 5 * 60, streamid)
										.then((request) => {
											vnode.state.timeline.updateData();
										});
								}
							})
						])
					])
				])
			]);
		}
	};

	let generateTooltip = function(row){
		if(row.disabled && row.disabled_tooltip){
			return m("td", [
				m("div", {"class": "tooltip"}, [
					row.label,
					m("span", {"class": "tooltiptext"}, m.trust(row.disabled_tooltipText))
				])
			]);
		}else if(row.tooltip){
			return m("td", [
				m("div", {"class": "tooltip"}, [
					row.label,
					m("span", {"class": "tooltiptext"}, m.trust(row.tooltipText))
				])
			]);
		}
		return m("td", row.label);
	};

	let generateSimpleTooltip = function(row){
		if(row.simpleTooltip){
			return m("div", {"class": "simple_tooltip"}, [
				m('label', {'for': row.name+"-input"}, row.simpleLabel),
				m("span", {"class": "simple_tooltiptext"}, m.trust(row.simpleTooltipText))
			]);
		}
		return m('label', {'for': row.name+"-input"}, row.simpleLabel);
	};

	let decToHexColor = function(dec){
		return "#"+dec.toString(16).toUpperCase().padStart(6,0);
	};

	let generateTableRow = function(row){
		switch (row.type) {
			case "text" :
				return m("tr",[
					m("td", {colspan: row.colspan, style: {"text-align": "center"}}, row.defaultValue)
				]);
			case "input_text" :
				return m("tr",[
					generateTooltip(row),
					m("td", [
						m("input", {
							type: "text",
							name: row.name,
							id: row.name+"-input",
							value: row.actualValue,
							disabled: row.disabled || states.config.load || states.player.load,
							maxlength: row.maxlength,
							oninput: m.withAttr("value", function(v){
								row.setActualValue(v);
							})
						})
					])
				]);
			case "checkbox" :
				return m("tr",[
					generateTooltip(row),
					m("td", [
						m("input", {
							type: "checkbox",
							name: row.name,
							id: row.name+"-input",
							checked: row.actualValue,
							disabled: row.disabled || states.config.load || states.player.load,
							onchange: m.withAttr("checked", function(v){
								row.setActualValue(v);
							})
						})
					])
				]);
			case "button_set": {
				return m("tr", [
					m("td", {colspan: 2, "class": "alpr_button_set"}, row.items
							.map(function(button){
								return m("button", {
									onclick: button.onclick,
									disabled: button.disabled || states.config.load || states.player.load
								}, button.label);
							})
					)
				])
			}
			case "selectList":
				let options = [
					m("option", {value: row.firstValue}, row.firstLabel)
				];
				Object.keys(validationValues[row.list]).map(function(v){
					options.push(m("option", {value: v, selected: v == row.actualValue}, validationValues[row.list][v].name.actualValue));
				});
				return m("tr",[
					m("td",{colspan: 2, style: {"text-align": "center"}}, [
						m("select", {
								id: row.name+"-input",
								size: row.size,
								name: row.name,
								style: {
									width: "50%"
								},
								disabled: row.disabled || states.config.load || states.player.load,
								onchange: m.withAttr("value", function(v){
									row.setActualValue(v);
								})
							}, options
						)
					])
				]);
			case "select" :
				return m("tr",[
					generateTooltip(row),
					m("td", [
						m("select", {
								id: row.name+"-input",
								size: row.size,
								name: row.name,
								value: row.actualValue,
								disabled: row.disabled || states.config.load || states.player.load,
								onchange: m.withAttr("value", function(v){
									row.setActualValue(v);
								})
							},
							row.values.map(function(v,i){
								return m("option", {value: v}, row.labels[i]);
							})
						)
					])
				]);
			case "multi-select" :
				return m("tr.multi-select-row"+(row.display ? '' : '.hidden'),[
					m("td", {colspan: 2}, [
						m("div", [
							m("strong", m.trust(row.label))
						]),
						m("select", {
								id: row.name+"-input",
								size: row.values.length > row.size ? row.size : row.values.length,
								name: row.name,
								multiple: true,
								disabled: row.disabled || states.config.load || states.player.load
							},
							row.values.map(function(v,i){
								if(row.actualValue.includes(v)){
									return m("option", {
										value: v,
										selected: 'selected',
										onclick: () => {
											row.setActualValue(v);
										}
									}, row.values[i]);
								}else {
									return m("option", {
										value: v,
										onclick: () => {
											row.setActualValue(v);
										}
									}, row.values[i]);
								}
							})
						)
					])
				]);
			case "selectColor" :
				return m("tr",[
					generateTooltip(row),
					m("td", [
						m("select", {
								id: row.name+"-input",
								size: row.size,
								name: row.name,
								value: row.actualValue,
								disabled: row.disabled || states.config.load || states.player.load,
								style: {color: decToHexColor(row.actualValue)},
								onchange: m.withAttr("value", function(v){
									row.setActualValue(v);
								})
							},
							row.values.map(function(v,i){
								return m("option", {value: v, style:{color: decToHexColor(v)}}, row.labels[i]);
							})
						)
					])
				]);
			case "slider" :
				let val = row.min;
				if(row.actualValue !=="") {
					val = typeof row.actualValue == "string" ? row.vType === "int" ? parseInt(row.actualValue) : parseFloat(row.actualValue) : row.actualValue;
				}
				return m("tr",[
					generateTooltip(row),
					m("td", [
						m("div", {"class": "slider"}, [
							m("input", {
								type: "text",
								"class": "slider-input",
								id: row.name+"-input",
								name: row.name,
								value: row.actualValue,
								disabled: row.disabled || states.config.load || states.player.load,
								oninput: m.withAttr("value", function(v){
									row.setActualValue(v);
								}),
								onchange: m.withAttr("value", function(v){
									if(v === "") {
										row.setActualValue(row.min);
									}else{
										row.setActualValue(row.vType === "int" ? parseInt(v) : parseFloat(v));
									}
								})
							}),
							m("input", {
								type: "range",
								min: row.min,
								max: row.max,
								step: row.step,
								value: val,
								id: row.name+"-slider",
								disabled: row.disabled || states.config.load || states.player.load,
								oninput: m.withAttr("value", function(v){
									row.setActualValue(v);
								})
							})
						])
					])
				]);
			default :
				return false;
		}
	};

	let generateSimpleRow = function(row){
		switch (row.simpleType) {
			case "checkbox":
				return m('.configuration_row', {'class': 'row_checkbox'}, [
					generateSimpleTooltip(row),
					m('div',{'class': 'ui checkbox'}, [
						m("input", {
							type: "checkbox",
							name: row.name,
							id: row.name+"-input",
							checked: row.actualValue,
							onchange: m.withAttr("checked", function(v){
								row.setActualValue(v);
							})
						}),
						m('label')
					]),
					m('span')
				]);
			case "text":
				return m('.configuration_row', {'class': 'row_textinput'}, [
					generateSimpleTooltip(row),
					m("input", {
						'class': 'ui input '+(!!row.simpleDisabled ? 'disabled': ''),
						type: "text",
						name: row.name,
						id: row.name+"-input",
						value: row.actualValue,
						oninput: m.withAttr("value", function(v){
							row.setActualValue(v);
						}),
						onchange: m.withAttr("value", function(v){
							row.setActualValue(row.vType === "int" ? parseInt(v) : parseFloat(v));
						})
					}),
					m('span')
				]);
			case "textButton":
				return m('.configuration_row', {'class': 'row_textinputbutton'}, [
					generateSimpleTooltip(row),
					m("input", {
						'class': 'ui input '+(!!row.simpleDisabled ? 'disabled': ''),
						type: "text",
						name: row.name,
						id: row.name+"-input",
						disabled: !!row.simpleDisabled,
						value: row.actualValue,
						oninput: m.withAttr("value", function(v){
							row.setActualValue(v);
						}),
						onchange: m.withAttr("value", function(v){
							row.setActualValue(row.vType === "int" ? parseInt(v) : parseFloat(v));
						})
					}),
					m('button', {
						'class': 'ui button '+(!!row.simpleSelected ? 'positive' : ''),
						onclick: function(){
							row.simpleButtonAction();
						}
					}, row.simpleButtonLabel)
				]);
			case "checkboxButton":
				return m('.configuration_row', {'class': 'row_checkboxbutton'}, [
					generateSimpleTooltip(row),
					m('div',{'class': 'ui checkbox'}, [
						m("input", {
							type: "checkbox",
							name: row.name,
							id: row.name+"-input",
							checked: row.actualValue,
							onchange: () => {
								row.setActualValue(!row.actualValue);
							}
						}),
						m('label')
					]),
					m('button', {
						'class': 'ui button '+(!!row.simpleSelected ? 'positive' : ''),
						'disabled': row.simpleDisabled,
						onclick: function(){
							row.simpleButtonAction();
						}
					}, row.simpleButtonLabel)
				]);
			case "select": {
				return m('.configuration_row', {'class': 'row_select'}, [
					generateSimpleTooltip(row),
					m("select", {
							id: row.name + "-input",
							name: row.name,
							value: row.actualValue,
							disabled: row.simpleDisabled,
							onchange: m.withAttr("value", function (v) {
								row.setActualValue(v);
							})
						},
						row.values.map(function (v, i) {
							return m("option", {value: v}, row.simpleLabels[i]);
						})
					)
				]);
			}
			case "multi-select": {
				return m('.configuration_row'+(row.display ? '' : '.hidden'), {'class': 'row_multi_select'}, [
					generateSimpleTooltip(row),
					m("select", {
							id: row.name + "-input",
							size: row.values.length > row.simpleSize ? row.simpleSize : row.values.length,
							name: row.name,
							value: row.actualValue,
							disabled: row.simpleDisabled,
							multiple: true,
							tabIndex: -1
						},
						row.values.map(function(v,i){
							if(row.actualValue.includes(v)){
								return m("option.selected", {
									value: v,
									selected: 'selected',
									tabIndex: -1,
									onclick: () => {
										row.setActualValue(v);
									}
								}, row.simpleLabels[i]);
							}else {
								return m("option", {
									value: v,
									tabIndex: -1,
									onclick: () => {
										row.setActualValue(v);
									}
								}, row.simpleLabels[i]);
							}
						})
					)
				]);
			}
			case "slider" : {
				let val = row.min;
				if (row.actualValue !== "") {
					val = typeof row.actualValue == "string" ? (row.vType === "int" ? parseInt(row.actualValue) : parseFloat(row.actualValue)) : row.actualValue;
				}
				return m('.configuration_row', {'class': 'row_slider'}, [
					generateSimpleTooltip(row),
					m("input.ui.input.disabled", {
						type: "text",
						"class": "slider-input",
						id: row.name + "-input",
						name: row.name,
						value: row.actualValue,
						disabled: true
					}),
					m("input", {
						type: "range",
						min: row.min,
						max: row.max,
						step: row.step,
						value: val,
						id: row.name + "-slider",
						disabled: row.simpleDisabled,
						oninput: m.withAttr("value", function (v) {
							row.setActualValue(v);
						})
					})
				]);
			}

			default :
				return false;
		}
	};

	let Base = {
		pageElements: [
			{
				header: "",
				items: [
					// Common parameters
					validationValues.time_between_events,
					validationValues.scans_per_event,
					validationValues.time_between_scans,
					validationValues.plate_cut_out_position,

					// VaxOCR-specific parameters
					validationValues.min_chars_height,
					validationValues.max_chars_height,
					validationValues.min_global_confidence,
					validationValues.min_char_confidence,
					validationValues.grammar_strict,
					validationValues.min_num_plate_chars,
					validationValues.max_num_plate_chars,
					validationValues.max_slop_angle,
					validationValues.background_mode,
					validationValues.ocr_complexity,
					validationValues.find_plate_depth,
					validationValues.detect_multiline_plate,
					validationValues.enable_multicountry_nn,
					validationValues.num_threads,

					// VaxMMC-specific parameters
					validationValues.mmc_type,
					validationValues.mmc_quality,
					validationValues.mmc_min_global_confidence,

					// MD-related
					validationValues.md,
					validationValues.min_object_width,
					validationValues.min_object_height
				]
			}
		],
		view: function (vnode) {
			return [
				m("div", {"class": "alpr_content", id: "alpr_base"},[
					m("div", {"class": "clear"}),
					m("div", {"class": "alpr_tabs_form tabs_form"}, [
						m("div", {style: {"text-align": "center", "margin": "10px 0 0 0"}}, __("The green zone depicts the expected maximum vehicle size. You can drag it to any part of video to compare to the real objects.")),
						m("div", vnode.state.pageElements.map(function(table){
								return m("table",[
									m("tr", [
										m("th", {colspan: 2}, table.header)
									]),
									m("tbody", table.items.map(function(row){
											return generateTableRow(row);
										})
									)
								])
							})
						)
					])
				])
			]
		}
	};

	let Regions = {
		view: function () {
			return [
				m("div", {"class": "alpr_content", id: "alpr_regions", style: {display: "none"}},[
					m("div", {"class": "clear"}),
					m("div", {"class": "alpr_tabs_form tabs_form"}, [
						m("table",[
							m("tbody", regionKeys.map(function(region){
								return generateTableRow(validationValues[region])
							}))
						])
					]),
					m("div", {"class": "alpr_tabs_form tabs_form"}, generateZone())
				])
			]
		}
	};

	let generateZone = function(){
		if(validationValues.zonesItems.actualValue === -1){
			return [];
		}
		let zone = validationValues.zones[validationValues.zonesItems.actualValue];
		return [
			m("table",[
				m("tr", [
					m("th", {colspan: 2}, __("Zone Detailed Information"))
				]),
				m("tbody",[
					generateTableRow(zone.name),
					generateTableRow(zone.color)
				])
			])
		];
	};

	let Zones = {
		view: function () {
			return [
				m("div", {"class": "alpr_content", id: "alpr_zones", style: {display: "none"}},[
					m("div", {"class": "clear"}),
					m("div", {"class": "alpr_tabs_form tabs_form"}, [
						m("table",[
							m("tr", [
								m("th", {colspan: 2}, __("Define active VaxOCR detection zone[s] here"))
							]),
							m("tbody", [
								m("tr", [
									m("td", {colspan: 2, style: {"text-align": "center"}}, __("By default, everything is included, and by defining zones you can reduce CPU load"))
								]),
								m("tr", [
									m("td", {colspan: 2, style: {"text-align": "center"}}, __("Double click on zone edge to add node"))
								]),
								generateTableRow(validationValues.zonesItems),
								generateTableRow(validationValues.zonesButtonSet)
							])
						])
					]),
					m("div", {"class": "alpr_tabs_form tabs_form"}, generateZone())
				])
			]
		}
	};

	let Instructions = {
		view: function () {
			return [
				m("div", {"class": "alpr_content", id: "alpr_instructions", style: {display: "none"}},[
					m("div", {"class": "clear"}),
					m("div", {"class": "alpr_tabs_form tabs_form"}, [
						m("p", __("Automatic Number Plate Recognition, or ANPR, is a great asset to law enforcement, parking management, highway monitoring, gated community surveillance, toll management and other uses.  However, for professionals and hobbyists alike, there are several issues with ANPR camera setup that impedes its accuracy. If your license plate recognition accuracy rate is at 68 percent rather than 98 percent, then there is hope for you to increase the effectiveness of your system.")),
						m("h3", __("What is ANPR?")),
						m("p", __("ANPR is automatic number plate recognition, also known as Automatic License Plate Recognition (ALPR) in some countries. It is a system that is capable of reading vehicle number plates from images captured.")),
						m("p", __("As this industry expands, it has become important to note that there are factors that impact its success. When ANPR accuracy rates are suboptimal, it is common to blame the underlying ANPR engine.")),
						m("h3", __("3 Important Factors Impacting ANPR Success")),
						m("p", __("There are three fundamental components that make ANPR successful:")),
						m("ol", [
							m("li", __("Camera")),
							m("li", __("Camera setup")),
							m("li", __("ANPR engine"))
						]),
						m("p", __("Of these three, we see that camera setup holds the most potential for improvement for businesses and consumers to significantly bolster their ANPR results.")),
						m("p", __("We draw this conclusion because we know that our Plate Recognizer ANPR engine has been thoroughly tested and optimized to handle blurry, low-res, and dark images as well as other aspects of the real-world environment.  Also, counter to claims from camera manufacturers, we see that a modest, mid-range IP camera performs just as well as a high-end, expensive camera.  We have observed how proper camera setup along with a strong ANPR engine can be an effective solution to detect and decode vehicle license plates.")),
						m("p", __("This article offers specific tips and suggestions on how to best set up your camera to maximize your ANPR results. We tried our best to prioritize the list, with the highest potential contributors on top.")),
						m("h3", __("Camera Zoom for Best ANPR")),
						m("p", __("For beginners handling license plate recognition, there is a common misconception that a wide-angle shot is the best.  Cameras will seek to aim wide across parking lots rather than towards entrance or exits, which allow them to truly pick up plates and get a good read.")),
						m("p", __("This is a simple fix if you know how to fix it. Adjust the camera positioning and width of the focus so that the license plate is picked up in the shot, and you’ll instantly have greater accuracy. Rather than have a wide-angle view of the street with all the homes around it, it is better to focus on just the street itself.")),
						m("p", __("In the photo below, you can find the image with the red box showing the plate. If the camera was zoomed in closer to the gate, the focus would improve. Make sure you are not over-zooming because you’ll need at least 50 pixels of width in order to get a good read on the plate.")),
						m("img", {src: "img/Camera-Setup-Best-ANPR-Zoom.jpg", alt: "Camera Setup Best ANPR Zoom"}),
						m("div.img-tooltip", __("Zooming into the vehicle yields stronger ANPR accuracy as compared to taking a wide-angle view of the area")),
						m("h3", __("Camera Distance for Better ANPR")),
						m("p", __("The maximum advisable distance between the camera and the vehicle is 35 meters.  Actually, whenever possible, it is preferred to minimize that distance.  Why?  Because minimizing the distance between the camera and vehicle helps ensure that the camera can easily focus without the need to zoom in to the target vehicle.  This helps reduce image blurriness.")),
						m("img", {src: "img/Camera-Setup-Best-ANPR-Distance.jpg", alt: "Camera Setup Best ANPR Distance"}),
						m("div.img-tooltip", __("Distance between camera and vehicle should be minimized as much as possible and definitely under 35 meters.")),
						m("h3", __("Camera Angle to Improve ANPR")),
						m("p", __("While Plate Recognizer ANPR has been tuned to support a wide variety of license plate angles, it’s always ideal to have the camera set up appropriately. In terms of angle, the setup of the ANPR camera can be positioned in two ways, slope as well as vertically and horizontally.  For both cases, it is advisable to have a maximum of 45 degrees for a proper read of the license plate.")),
						m("img", {src: "img/Camera-Setup-Best-ANPR-Vertical-Angle.jpg", alt: "Camera Setup Best ANPR Vertical Angle"}),
						m("div.img-tooltip", __("Vertical angle of camera on vehicle license plate should be under 45 degrees.")),
						m("img", {src: "img/Camera-Setup-Best-ANPR-Horizontal-Angle.jpg", alt: "Camera Setup Best ANPR Horizontal Angle"}),
						m("div.img-tooltip", __("Horizontal angle of camera on vehicle license plate should be under 45 degrees.")),
						m("h3", __("Minimum Resolution for Best ANPR")),
						m("p", __("While Plate Recognizer works with any IP camera, we recommend that the license plate itself must have at least 50-100 pixels in width.  Otherwise, the ANPR engine may not be able to effectively read the license plate.")),
						m("p", __("This means that depending on your use-case, you may need a camera with a certain number of megapixels (MPs).  Here are some of our guidelines or suggestions based on the ANPR use case:")),
						m("ol", [
							m("li", __("2 MP camera is suitable for parking management and toll ANPR projects.")),
							m("li", __("4 MP camera is good for logistics or vehicle repair-type projects.")),
							m("li", __("8 MP camera is needed for highway or street monitoring."))
						]),
						m("p", __("The above are just rough guidelines.  If you’re unsure, then just grab a few frames from your camera or take a few photos with your mobile phone with settings based on camera megapixels.  Thereafter, run our ANPR engine through some of those images to see if the plates are correctly detected.")),
						m("h3", __("Camera Frames May Impact ANPR")),
						m("p", __("The camera frame (and thus your ANPR camera setup) is largely based on the vehicle’s speed. If you’re looking to capture license plates that are still versus license plates that are moving (also called “free flow”) then the specifications for altering your camera will be different.")),
						m("p", __("You will need to calculate the net difference in speed. If your camera is in a fixed position, then the net speed is the speed of the vehicle. If the camera is maintained in your vehicle (e.g. police car), and then you’re driving the same direction of the target vehicle, then the net speed is the difference in your speed and the target vehicle’s speed. Keeping this in mind prevents you from having any unusable images because you didn’t take the time to equip your camera for the proper camera speed.")),
						m("p", __("For example, if the vehicle is traveling at 10 miles per hour (mph), you can get good ANPR results with just 10 to 15 frames per second.  With the vehicle at 30 mph, grabbing images at 15 to 25 frames per second should suffice.  At 60 mph, you’ll need 30 to 40 frames per second.  The suggestions here may differ greatly depending on the camera quality and zoom level, so it is always best to test and refine.")),
						m("p", __("Moreover, the faster the vehicle is traveling, the more images you’d want to send over to the Plate Recognizer engine.  This way, our ANPR engine has more opportunities to evaluate and decode the license plate as the vehicle passes a certain point.  At 10 mph, 1 or 2 images would be sufficient, starting at 0 and then 0.5 second of when motion was first detected.  At 30 mph, you may want to send 3-5 images to the ANPR engine, at intervals of 0.2 to 0.4 seconds.  At 60 mph, it’s best to send 5-10 images, at intervals of 0.1 to 0.2 seconds.")),
						m("img", {src: "img/Camera-Setup-Best-ANPR-fps.jpg", alt: "Camera Setup Best ANPR fps"}),
						m("div.img-tooltip", __("Good summary of how vehicle speed impacts FPS, images sent and also time between images.")),
						m("h3", __("Camera Lighting for Better ANPR")),
						m("p", __("While Plate Recognizer has greatly improved its algorithms to support dark images, it is nevertheless a good idea to ensure that the camera is set for the right lighting conditions.  You can adjust the camera shutter speed accordingly so as to not to over-expose or under-expose the image.  For example, if it is bright outside, adjust the shutter speed to capture at 1/5000 of a second.  At night time, set the shutter speed to 0.75 to 1 second.")),
						m("p", __("If the camera does not auto-adjust the shutter speed based on the outside lighting conditions, then you can try to have a light fixture directed towards the entering vehicle.  This way, there will always be adequate lighting on the vehicle.")),
						m("img", {src: "img/Camera-Setup-Best-ANPR-Ligthing.jpg", alt: "Camera Setup Best ANPR Ligthing"}),
						m("div.img-tooltip", __("Ensure adequate lighting to avoid over-exposed or under-exposed images.")),
						m("h3", __("Additional Tips to Improve ANPR")),
						m("p", __("While there’s plenty that you should do, there’s also plenty that you should not do. Let’s face it. Cameras come with a lot of settings, and it’s easy to switch on a bunch of settings that you either don’t know what they are or don’t need and forget that you’ve activated. Here are some to keep in mind:")),
						m("p", __("Automatic gain control (AGC), digital noise reduction (DNR), autofocus, and back light compensation (BLC) are all features you want to keep disabled while enabling ANPR camera setup. Once again, this is because it will give you the best chance of grabbing that license plate number from a moving vehicle using ANPR.")),
						m("p", __("AGC creates issues because the gain itself prompts digital noise and lower recognition in the image. It’s often much simpler just to leave the feature off. DNR is best left alone because it is performed by removing pixels based on comparing two frames. Although this might seem harmless, it’s often not because it can easy remove pixels that could be helpful to you in the future. Next, you can pass on autofocus because adjusting the sharpness often reduces the recognition quality in the image itself. Finally, the BLC can cause issues with the image because it often occurs when a light source enters a frame. When the pixels do not have enough time to properly adjust, the camera will not be able to capture a good image.")),
						m("p", __("And, while this may seem obvious, we see that the best ANPR contains images that are in landscape rather than in portrait view.  This makes intuitive sense, since the license plate itself is more landscape than portrait.  And, just like watching TV, we (and thus our ANPR engine) is used to seeing the world in a portrait format.")),
						m("h3", __("What is the Fast Mode for Plate Recognizer?")),
						m("p", __("We have created a Regular Mode and a Fast Mode to further optimize the software. The Fast Mode can be used in situations where the vehicle size is at least 10% of the image size.  For example, Fast Mode can be applied to these real-world situations:")),
						m("ol", [
							m("li", __("A vehicle enters a parking lot and the camera takes a close-up photo of one vehicle at a time.")),
							m("li", __("A specific vehicle is detected by your highway monitoring video analytics software.")),
						]),
						m("h3", __("Fast Mode:")),
						m("img", {src: "img/Plate-Recognizer-Fast-ALPR-Fast-Mode.jpg", alt: "Plate Recognizer Fast ALPR Fast Mode"}),
						m("div.img-tooltip", __("Images above are suitable for Fast Mode.")),
						m("h3", __("Regular Mode:")),
						m("img", {src: "img/Plate-Recognizer-Fast-ALPR-Regular.jpg", alt: "Plate Recognizer Fast ALPR Regular"}),
						m("div.img-tooltip", __("Images above are suitable for Regular Mode.")),
						m("h3", __("Is there an Easy Way to Determine Fast Mode for ALPR?")),
						m("p", __("Sometimes, the 10% parameter may not be as easy to comprehend visually.  So we created this graphic below to help you get a feel for whether you should use Fast or Regular mode for license plate recognition.")),
						m("p", __("Imagine that the image below is the full image of your camera frame.  Basically, if your vehicle image is proportionally larger than our Hero Car below, then you should be OK to use Fast Mode.  If smaller, then use Regular Mode.")),
						m("img", {src: "img/Plate-Recognizer-Fast-ALPR-Determine-Mode.jpg", alt: "Plate Recognizer Fast ALPR Determine Mode"}),
						m("p", __("As always, we recommend that you try frist in Fast Mode and if things don’t work out, you can resort back to Regular Mode.")),
						m("h3", __("How Does a Fast ALPR Engine Benefit You?")),
						m("p", __("There are a number of important benefits to having a fast ALPR system:")),
						m("ol", [
							m("li", __("Fast ALPR means you don’t have to invest as much on your hardware to process the same number of vehicle images.")),
							m("li", __("Fast ANPR lets you process more camera feeds using the same hardware as before.  This leads to significant savings in hardware costs.")),
							m("li", __("Fast ALPR allows for constant monitoring (24 x 7 x 365) so that you do not miss any vehicles in applications such as highway monitoring."))
						]),
					])
				])
			]
		}
	};

	let deserialize = function(config){
		let c = config.configuration;
		if (c === undefined){
			return false;
		}
		if(type === 'main') {
			validationValues.mmc_type.setActualValue(c.mmc_type);
		}

		// Common parameters
		validationValues.plate_cut_out_position.setActualValue(c.plate_cut_out_position);
		validationValues.scans_per_event.setActualValue(c.scans_per_event);
		validationValues.time_between_events.setActualValue(c.time_between_events);
		validationValues.time_between_scans.setActualValue(c.time_between_scans);
		if(typeof c.image_height === "number" && typeof c.image_width === "number"){
			calculateDimensions(c.image_width, c.image_height);
		}

		// VaxOCR-specific parameters
		validationValues.min_chars_height.setActualValue(c.min_chars_height);
		validationValues.max_chars_height.setActualValue(c.max_chars_height);
		validationValues.min_global_confidence.setActualValue(c.min_global_confidence);
		validationValues.min_char_confidence.setActualValue(c.min_char_confidence);
		validationValues.grammar_strict.setActualValue(c.grammar_strict);
		validationValues.min_num_plate_chars.setActualValue(c.min_num_plate_chars);
		validationValues.max_num_plate_chars.setActualValue(c.max_num_plate_chars);
		validationValues.max_slop_angle.setActualValue(c.max_slop_angle);
		validationValues.background_mode.setActualValue(c.background_mode);
		validationValues.ocr_complexity.setActualValue(c.ocr_complexity);
		validationValues.find_plate_depth.setActualValue(c.find_plate_depth);
		validationValues.detect_multiline_plate.setActualValue(c.detect_multiline_plate);
		validationValues.enable_multicountry_nn.setActualValue(c.enable_multicountry_nn);
		validationValues.num_threads.setActualValue(c.num_threads);

		// VaxMMC-specific parameters
		validationValues.mmc_quality.setActualValue(c.mmc_quality);
		validationValues.mmc_min_global_confidence.setActualValue(c.mmc_min_global_confidence);

		// MD-related
		validationValues.md.setActualValue(c.md);
		validationValues.min_object_width.setActualValue(c.min_object_width);
		validationValues.min_object_height.setActualValue(c.min_object_height);

		validationValues.zonesButtonSet.items[2].onclick();
		if(Array.isArray(c.roi) && c.roi.length>0) {
			let zs = false;
			if(Array.isArray(c.zone_set) && c.roi.length === c.zone_set.length){
				zs = true;
			}
			for(let i = 0; i < c.roi.length; i++){
				let id = i;
				if(Array.isArray(c.roi[i]) && c.roi[i].length % 2 === 0){
					let params = {
						color: 65280,
						point_set: []
					};
					for(let j = 0; j < c.roi[i].length; j += 2){
						params.point_set.push({x: c.roi[i][j], y: c.roi[i][j+1]});
					}
					if(zs){
						params.color = c.zone_set[i].color;
						id = c.zone_set[i].id;
					}
					addZone(id, params);
					if(zs){
						validationValues.zones[id].name.setActualValue(c.zone_set[i].name);
					}
				}
				if(type === 'simple'){
					validationValues.simple_zone.actualValue = true;
					validationValues.simple_zone.simpleDisabled = false;
					validationValues.zonesItems.setActualValue(id);
					if(validationValues.simple_zone.simpleSelected){
						canvas.e.setLayerGroup('zones', {
							visible: true
						});
					}
					break;
				}
			}
		}else{
			validationValues.simple_zone.actualValue = false;
			validationValues.simple_zone.simpleSelected = false;
		}
		validationValues.zonesButtonSet.items[2].disable();
		if(Array.isArray(c.countries)){
			for(let i=0; i<c.countries.length; i++) {
				regionKeys.forEach(k => {
					if(k !== 'countries')
						return;
					if(validationValues[k].values.includes(c.countries[i])){
						validationValues[k].setActualValue(c.countries[i]);
						return;
					}
				});
			}
		}
		if(Array.isArray(c.states)){
			for(let i=0; i<c.states.length; i++){
				regionKeys.forEach(k => {
					if(k === 'countries')
						return;
					if(validationValues[k].values.includes(c.states[i])){
						validationValues[k].setActualValue(c.states[i]);
						return;
					}
				});
			}
		}
	};

	let prepareConfig = function() {
		let res = {
			name: configsArray[0].name,
			description: configsArray[0].description,
			configuration: {
				// Common parameters
				"plate_cut_out_position": validationValues.plate_cut_out_position.actualValue,
				"image_width": validationValues.min_object_width.max,
				"image_height": validationValues.min_object_height.max,
				"scans_per_event": validationValues.scans_per_event.actualValue,
				"time_between_events": validationValues.time_between_events.actualValue,
				"time_between_scans": validationValues.time_between_scans.actualValue,
				"zone_set": [],
				// VaxOCR-specific parameters
				"countries": [],
				"states": [],
				"min_chars_height": validationValues.min_chars_height.actualValue,
				"max_chars_height": validationValues.max_chars_height.actualValue,
				"min_global_confidence": validationValues.min_global_confidence.actualValue,
				"min_char_confidence": validationValues.min_global_confidence.actualValue,
				"grammar_strict": validationValues.grammar_strict.actualValue,
				"min_num_plate_chars": validationValues.min_num_plate_chars.actualValue,
				"max_num_plate_chars": validationValues.max_num_plate_chars.actualValue,
				"max_slop_angle": validationValues.max_slop_angle.actualValue,
				"background_mode": validationValues.background_mode.actualValue,
				"ocr_complexity": validationValues.ocr_complexity.actualValue,
				"find_plate_depth": validationValues.find_plate_depth.actualValue,
				"detect_multiline_plate": validationValues.detect_multiline_plate.actualValue,
				"enable_multicountry_nn": validationValues.enable_multicountry_nn.actualValue,
				"num_threads": validationValues.num_threads.actualValue,
				// VaxMMC-specific parameters
				"mmc_type": validationValues.mmc_type.actualValue,
				"mmc_quality": validationValues.mmc_quality.actualValue,
				"mmc_min_global_confidence": validationValues.mmc_min_global_confidence.actualValue,
				// MD-related
				"md": validationValues.md.actualValue,
				"min_object_width": validationValues.min_object_width.actualValue,
				"min_object_height": validationValues.min_object_height.actualValue,
				"roi": []
			}
		};
		regionKeys.forEach(k => {
			let region = validationValues[k];
			if(k === 'countries')
				res.configuration.countries = res.configuration.countries.concat(region.actualValue);
			else
				res.configuration.states = res.configuration.states.concat(region.actualValue);
		});
		//for(let i=0; i<regionKeys.length; i++){
		//	let region = validationValues[regionKeys[i]];
		//	res.configuration.countries = res.configuration.countries.concat(region.actualValue);
		//}
		Object.keys(validationValues.zones).map(function(v) {
			let zoneObj = {
				"id": parseInt(v),
				"name": validationValues.zones[v].name.actualValue,
				"color": validationValues.zones[v].color.actualValue
			};
			let zoneLayer = canvas.e.getLayer('zonePolygon'+v);
			let point_set = [];
			for(let i=1; i<=zoneLayer.data.points; i++){
				let x = zoneLayer["x"+i]+zoneLayer.x;
				let y = zoneLayer["y"+i]+zoneLayer.y;
				if(x < roi.min){
					x = roi.min;
				}else if(x > canvas.w){
					x = roi.max;
				}else{
					x = Math.round((roi.max / canvas.w) * x);
				}
				if(y < roi.min){
					y = roi.min;
				}else if(y > canvas.h){
					y = roi.max;
				}else{
					y = Math.round((roi.max / canvas.h) * y);
				}
				point_set.push(x);
				point_set.push(y);
			}
			res.configuration.zone_set.push(zoneObj);
			res.configuration.roi.push(point_set);
		});

		return res;
	};

	let addRedBorder = function(){
		canvas.e.drawRect({
			layer: true,
			strokeStyle: "#F00",
			strokeWidth: 10,
			width: canvas.w,
			height: canvas.h,
			draggable: false,
			visible: true,
			fromCenter: true,
			x: Math.round(canvas.w/2),
			y: Math.round(canvas.h/2),
			name: "confApply"
		});
		setTimeout(function(){canvas.e.removeLayer("confApply"); m.redraw();}, 1000);
	};

	let generateConfigMessage = function(){
		if(states.config.firstLoadComplete) {
			if (states.config.isActiveEngine) {
				return states.config.isActiveConfig === false ?
					m("#non-active-config", [
						__("You are editing configuration which is not active. Live metadata will not be shown."),
						m("br"),
						m("span.span-link", {
							onclick: function () {
								if (!states.config.isActiveConfig) {
									states.config.isActiveConfig = true;
									states.config.vae_engines["VAE_ENGINES"].vaxocr.configuration = name;
									let api = new API();
									api.setAttributes({
										obj: objid,
										attributes: states.config.vae_engines
									}).then(function (res) {
										if (res.code == 200) {
											states.player.e.showMetaData();
											Log.info("Configuration successfully activated");
											addRedBorder();
										} else {
											states.config.isActiveConfig = false;
											Log.error(__("ERROR 61010: Can not activate configuration, try repeating operation after a minute"));
											m.redraw();
										}
									}).fail(function () {
										states.config.isActiveConfig = false;
										Log.error(__("ERROR 61010: Can not activate configuration, try repeating operation after a minute"));
										m.redraw();
									});
								}
							}
						}, __("Make configuration active"))
					])
					: false;
			} else {
				return m("#non-active-config", [
					__("License Plate Recognizer is not active. Live metadata will not be shown."),
					m("br"),
					m("span.span-link", {
						onclick: function () {
							$('.vae-back-button', window.parent.document)[0].click();
						}
					}, __("Go to analytics configuration screen"))
				]);
			}
		}else{
			return false;
		}
	};

	let saveConfig = function(res, push){
		m.request({
			method: "PUT",
			url: '/api/vae/config/'+objid+'/vaxocr/'+name,
			data: res
		}).then(function() {
			if(push){
				configsArray.push(res);
			}
			addRedBorder();
			Log.info(__("Configuration successfully applied"), "info");
			states.config.load = false;
		}).catch(function(e) {
			Log.error(e.message ? e.message : __("Can not save configuration. Please try again later"));
			states.config.load = false;
		})
	};

	let pointOnEdge = function(oX, oY, w, h, x, y){
		let thickness = 4;
		let	edge = false;
		let left = Math.floor(oX - w/2);
		let right = Math.ceil(oX + w/2);
		let top = Math.floor(oY - h/2);
		let bottom = Math.ceil(oY + h/2);
		let direction = '';
		let cursors = {
			left: 'ew-resize',
			right: 'ew-resize',
			top: 'ns-resize',
			bottom: 'ns-resize',
			leftTop: 'nwse-resize',
			rightBottom: 'nwse-resize',
			rightTop: 'nesw-resize',
			leftBottom: 'nesw-resize'
		};
		if(x >= left && x <= left + thickness){
			edge = true;
			direction = 'left';
		}else if(x <= right && x >= right - thickness){
			edge = true;
			direction = 'right';
		}
		if(y >= top && y <= top + thickness){
			edge = true;
			if(direction === 'left'){
				direction = 'leftTop';
			}else if(direction === 'right'){
				direction = 'rightTop';
			}else{
				direction = 'top';
			}
		}else if(y <= bottom && y >= bottom - thickness){
			edge = true;
			if(direction === 'left'){
				direction = 'leftBottom';
			}else if(direction === 'right'){
				direction = 'rightBottom';
			}else{
				direction = 'bottom';
			}
		}
		return {
			edge: edge,
			cursor: cursors[direction]
		};
	};

	let onCanvasCreate = function(){
		canvas.e = $("#canv");
		canvas.e.drawRect({
			layer: true,
			strokeStyle: "#000",
			fillStyle: "#0F0",
			opacity: 0.3,
			width: validationValues.min_object_width.actualValue,
			height: validationValues.min_object_height.actualValue,
			draggable: true,
			visible: type === 'main',
			fromCenter: true,
			x: Math.round(canvas.w/2),
			y: Math.round(canvas.h/2),
			name: "MaxObjectRect",
			data: {
				move: false,
				expand: false,
				cursor: false,
				x: 0,
				y: 0,
				oX: 0,
				oY: 0
			},
			mouseover: function(self){
				if(!self.data.expand && !self.data.move){
					let {edge, cursor} = pointOnEdge(self.x, self.y, self.width, self.height, self.eventX, self.eventY);
					if(edge){
						canvas.e.css('cursor', cursor);
					}else{
						canvas.e.css('cursor', 'grab');
					}
				}
			},
			mousemove: function(self){
				let {edge, cursor} = pointOnEdge(self.x, self.y, self.width, self.height, self.eventX, self.eventY);
				if(!self.data.expand){
					if(self.data.move){
						canvas.e.css('cursor', 'grabbing');
					}else {
						if (edge) {
							canvas.e.css('cursor', cursor);
						} else {
							canvas.e.css('cursor', 'grab');
						}
					}
				}else{
					canvas.e.css('cursor', self.data.cursor);
				}
			},
			mousedown: function(self){
				let {edge, cursor} = pointOnEdge(self.x, self.y, self.width, self.height, self.eventX, self.eventY);
				self.data.expand = edge;
				if(self.data.expand){
					self.data.x = self.eventX;
					self.data.y = self.eventY;
					self.data.oX = self.x;
					self.data.oY = self.y;
					self.data.cursor = cursor;
				}else{
					self.data.move = true;
					canvas.e.css('cursor', 'grabbing');
				}
			},
			mouseup: function(self){
				if(self.data.move){
					canvas.e.css('cursor', 'grab');
				}
				self.data.expand = false;
				self.data.move = false;
				self.data.cursor = false;
			},
			dragstop: function(self){
				if(self.data.move){
					canvas.e.css('cursor', 'grab');
				}
				self.data.expand = false;
				self.data.move = false;
				self.data.cursor = false;
			},
			dragcancel: function(self){
				self.data.expand = false;
				self.data.move = false;
				self.data.cursor = false;
			},
			updateDragX: function (layer, x) {
				if(layer.data.expand){
					return layer.data.oX;
				}
				if(Math.floor(x - layer.width/2) <= 0){
					return Math.floor(layer.width/2);
				}
				if(Math.floor(x + layer.width/2) >= canvas.w){
					return Math.floor(canvas.w - layer.width/2);
				}
				return x;
			},
			updateDragY: function (layer, y) {
				if(layer.data.expand){
					return layer.data.oY;
				}
				if(Math.floor(y - layer.height/2) <= 0){
					return Math.floor(layer.height/2);
				}
				if(Math.floor(y + layer.height/2) > canvas.h){
					return Math.floor(canvas.h - layer.height/2);
				}
				return y;
			},
			drag: function(self){
				if(self.data.expand) {
					self.x = self.data.oX;
					self.y = self.data.oY;
					let normalize = (param, aspect, min, max, a, oA, canvasMax) => {
						let realParam = Math.floor(param/aspect);
						if(realParam < min){
							realParam = min;
							param = Math.floor(realParam*aspect);
						}else if(realParam > max){
							realParam = max;
							param = Math.floor(realParam*aspect);
						}
						if(Math.ceil(a + param/2) > canvasMax){
							param = (canvasMax - oA)*2;
							realParam = Math.floor(param/aspect);
						}else if(Math.ceil(a - param/2) <= 0){
							param = oA*2;
							realParam = Math.floor(param/aspect);
						}
						return {param, realParam}
					};
					if(self.data.cursor === 'ns-resize' || self.data.cursor === 'nwse-resize' || self.data.cursor === 'nesw-resize'){
						let dHeight = (self.data.y < self.y ? self.data.y - self.eventY : self.eventY - self.data.y) * 2;
						if(dHeight !== 0){
							let {param, realParam} = normalize(
								self.height+dHeight,
								canvas.aspectH,
								validationValues.min_object_height.min,
								validationValues.min_object_height.max,
								self.y,
								self.data.oY,
								canvas.h
							);
							validationValues.min_object_height.setActualValue(realParam, true);
							self.height = param;
						}
					}
					if(self.data.cursor === 'ew-resize' || self.data.cursor === 'nwse-resize' || self.data.cursor === 'nesw-resize'){
						let dWidth = (self.data.x < self.x ? self.data.x - self.eventX : self.eventX - self.data.x) * 2;
						if(dWidth !== 0){
							let {param, realParam} = normalize(
								self.width+dWidth,
								canvas.aspectW,
								validationValues.min_object_width.min,
								validationValues.min_object_width.max,
								self.x,
								self.data.oX,
								canvas.w
							);
							validationValues.min_object_width.setActualValue(realParam, true);
							self.width = param;
						}
					}
					self.data.x = self.eventX;
					self.data.y = self.eventY;
					m.redraw();
				}
			}
		});
	};

	let Page = {
		oncreate: function(){
			onCanvasCreate();
			$("body").on("click", ".tab_inactive", function () {
				if(!states.config.load && !states.player.load) {
					$(".vae-configurator .current_tab")
						.removeClass("current_tab")
						.addClass("tab_inactive");
					$(this)
						.removeClass("tab_inactive")
						.addClass("current_tab");
					let newTab = $(this).attr("data-name");
					if (newTab === "alpr_zones") {
						states.tabs.showZones = true;
						canvas.e.setLayerGroup('zones', {
							visible: true
						}).drawLayers();
					} else {
						states.tabs.showZones = false;
						canvas.e.setLayerGroup('zones', {
							visible: false
						}).drawLayers();
					}
					if (newTab === "alpr_base") {
						canvas.e.setLayer('MaxObjectRect', {
							visible: true
						}).drawLayers();
					} else {
						canvas.e.setLayer('MaxObjectRect', {
							visible: false
						}).drawLayers();
					}
					states.current_tab = newTab;
					$(".alpr_content").hide();
					$("#" + $(this).attr("data-name")).show();
				}
			});
		},
		oninit: function() {
			let loading = window.parent.document.querySelector(".loading_iframe");
			if(loading){
				loading.style.display = "none";
			}
			m.request({
				method: "GET",
				url: '/api/vae/config/'+objid+'/vaxocr/'+name
			}).then(function(items) {
				configsArray.push(items.config);
				let api = new API();
				api.getAttributes({
					obj: objid
				}).then(function(attr){
					states.config.vae_engines["VAE_ENGINES"] = JSON.parse(attr.list["VAE_ENGINES"]);
					if(states.config.vae_engines["VAE_ENGINES"].vaxocr) {
						states.config.isActiveEngine = states.config.vae_engines["VAE_ENGINES"].vaxocr.active;
						states.config.isActiveConfig = states.config.vae_engines["VAE_ENGINES"].vaxocr.configuration === name;
					}
					let f = JSON.parse(attr.list["FEATURES"]);
					if(f["vae-vaxocr-class"]){
						validationValues.mmc_type.disabled = false;
					}
					deserialize(configsArray[0]);
					states.config.load = false;
					states.config.firstLoadComplete = true;
					m.redraw();
					if (window.parent.onVAELoad) {
						window.parent.onVAELoad(true);
					}
				}).fail(function(){
					Log.error(__("Can not load configuration. Please try reloading the page"));
					if (window.parent.onVAELoad) {
						window.parent.onVAELoad(false);
					}
				});
			}).catch(function(e) {
				Log.error(e.message);
				if (window.parent.onVAELoad) {
					window.parent.onVAELoad(false);
				}
			});
		},
		view: function () {
			return [
				m(Frame, {
					"class": "vae-configurator editor_container",
					header: Gettext.strargs(__("Triggered ALPR Configuration - %1"), [configsArray[0] && configsArray[0].name ? configsArray[0].name : ""]),
					tabList: tabList
				}, [
					m("div", {style: {position: "relative"}}, [
						m("#messages", {style: {position: "absolute", top: "5px", right: "5px"}})
					]),
					generateConfigMessage(),
					m(Player),
					m("div", {"class": "tabs_content"}, [
						m(Base),
						m(Regions),
						m(Zones)//,
						//m(Instructions)
					]),
					m("div", {"class": "tabs_manage_buttons_wrapper"},[
						m("div", {"class": "tabs_manage_buttons"}, [
							m("div",{style: {"float": "right", "margin": "0 15px"}}, [
								m("input", {
									disabled: states.config.load || configsArray.length < 2,
									type:"button",
									id:"saveObject",
									value:__("Default"),
									onclick: function(){
										states.config.load = true;
										validationValues.zonesButtonSet.items[2].onclick();
										let res = configsArray[0];
										configsArray = [configsArray[0]];
										deserialize(res);
										if(states.tabs.showZones) {
											canvas.e.setLayerGroup('zones', {
												visible: true
											}).drawLayers();
										}
										saveConfig(res);
									}
								}),
								m("input", {
									disabled: states.config.load || configsArray.length < 2,
									type:"button",
									id:"saveObject",
									value:__("Undo"),
									onclick: function(){
										states.config.load = true;
										validationValues.zonesButtonSet.items[2].onclick();
										configsArray.pop();
										let res = configsArray[configsArray.length-1];
										deserialize(res);
										if(states.tabs.showZones) {
											canvas.e.setLayerGroup('zones', {
												visible: true
											}).drawLayers();
										}
										saveConfig(res);
									}
								}),
								m("input", {
									disabled: states.config.load || states.player.load,
									type: "button",
									id: "saveObject",
									value: __("Apply"),
									onclick: function(){
										states.config.load = true;
										let res = prepareConfig();
										saveConfig(res, true);
									}
								})
							])
						])
					])
				])
			]
		}
	};

	let SimpleConfigurationSegment = {
		show: false,
		oninit: function(vnode){
			vnode.state.show = !!vnode.attrs.show;
		},
		showHide: function(){
			this.show = !this.show;
		},
		view: function(vnode){
			return m('.configuration_segment_wrap', [
				m('.configuration_segment_header', {
					onclick: () => {vnode.state.showHide()}
				}, [
					vnode.attrs.header,
					m(".arrow", vnode.state.show ? m("i.fas.fa-angle-up") : m("i.fas.fa-angle-right"))
				]),
				vnode.state.show && m('.configuration_segment_body', vnode.attrs.pageElements
					.map(function(row){
						return generateSimpleRow(row);
					}))
			])
		}
	};

	let vaxocrControls = {
		'save': () => {
			let res = prepareConfig();
			return m.request({
				method: "PUT",
				url: '/api/vae/config/'+objid+'/vaxocr/'+name,
				data: res
			});
		},
		'revert': function() {
			let res = configsArray[0];
			deserialize(res);
			return this.save().then(function(){
				configsArray = [configsArray[0]];
				m.redraw();
			});
		},
		'setAttributes': (attr) => {
			return new Promise((resolve, reject) => {
				try {
					let keys = Object.keys(attr);
					for(let i=0; i<keys.length; i++){
						if(validationValues[keys[i]]) {
							validationValues[keys[i]].setActualValue(attr[keys[i]]);
						}
					}
					m.redraw();
					resolve(true);
				} catch (e) {
					reject(e);
				}
			});
		},
		'getAttribute': (attr) => {
			return validationValues[attr].actualValue;
		}
	};

	let SimplePage = {
		oncreate: () => {
			onCanvasCreate();
		},
		oninit: () => {
			m.request({
				method: "GET",
				url: '/api/vae/config/'+objid+'/vaxocr/'+name
			}).then(function(items) {
				configsArray.push(items.config);
				deserialize(configsArray[0]);
				states.config.load = false;
				states.config.firstLoadComplete = true;
				window.vaxocrControls = vaxocrControls;
				m.redraw();
			}).catch(function(e) {
				window.vaxocrControls = false;
				console.log(e.message);
			});
		},
		view: function (vnode) {
			return [
				m('.configurator_wrap', [
					m('div', [
						m(Player),
						validationValues.simple_zone.simpleSelected && m('.canvas_hint', __('Use mouse to adjust zone position, size and shape. To move, place the mouse cursor over the zone, press and hold down the left mouse button, then move the mouse while still holding down the left mouse button. Let go of the mouse button once the zone is in the desired location. To change size, click & drag polygon points. Double-click on zone edge to add points to polygon.'))
					]),
					m('.configuration_wrap', [
						m(SimpleConfigurationSegment, {
							header: __('Common configuration'),
							pageElements: [
								validationValues.time_between_events,
								validationValues.scans_per_event,
								validationValues.time_between_scans,
								validationValues.md,
								validationValues.min_object_width,
								validationValues.min_object_height,
								validationValues.plate_cut_out_position,
								validationValues.simple_zone
							],
							show: true
						}),
						m(SimpleConfigurationSegment, {
							header: __('Countries and Regions / US States'),
							pageElements: regionKeys.map(function(region){
								return validationValues[region]
							}),
							show: false
						}),
						m(SimpleConfigurationSegment, {
							header: __('Advanced configuration'),
							pageElements: [								
								// VaxOCR-specific parameters
								validationValues.min_chars_height,
								validationValues.max_chars_height,
								validationValues.min_global_confidence,
								validationValues.min_char_confidence,
								validationValues.grammar_strict,
								validationValues.min_num_plate_chars,
								validationValues.max_num_plate_chars,
								validationValues.max_slop_angle,
								validationValues.background_mode,
								validationValues.ocr_complexity,
								validationValues.find_plate_depth,
								validationValues.detect_multiline_plate,
								validationValues.enable_multicountry_nn,
								validationValues.num_threads,
								
								// VaxMMC-specific parameters
								validationValues.mmc_type,
								validationValues.mmc_quality,
								validationValues.mmc_min_global_confidence
							],
							show: false
						})
					])
				])
			];
		}
	};

	m.mount(document.body, type === 'main' ? Page : SimplePage);
});
