﻿using System;
using System.Runtime.InteropServices;
using Newtonsoft.Json;
using System.Collections.Generic;

namespace VehicleClassifierNet
{
    public enum ClassificationType
    {
        VEHICLE_COLOR = 1,
        VEHICLE_MAKE = 2,
        VEHICLE_MAKEMODEL = 3
    }


    public class VehicleClassifier : IDisposable
    {

        private const int REALLY_BIG_PIXEL_WIDTH = 999999999;

        private string config_file;
        private string runtime_dir;
        private IntPtr native_instance;
        private bool is_initialized = false;

        private List<ClassificationType> loaded_classifiers = new List<ClassificationType>(3);

        /// <summary>
        /// VehicleClassifier library constructor. The classification types that you wish to use must each 
        /// be initialized separately with a call to LoadClassifier()
        /// </summary>
        /// <param name="config_file">The path to the openalpr.conf file.  Leave it blank to use the file in the current directory.</param>
        /// <param name="runtime_dir">The path to the runtime_data directory.  Leave it blank to use the runtime_data in the current directory</param>
        public VehicleClassifier(string config_file = "", string runtime_dir = "")
        {
            this.config_file = config_file;
            this.runtime_dir = runtime_dir;
            // Deinitialize the library

            this.is_initialized = false;
            try
            {
                // Destroy the native obj
                native_instance = vehicleclassifier_init(config_file, runtime_dir);
                this.is_initialized = true;
            }
            catch (System.DllNotFoundException)
            {
                Console.WriteLine("Could not find/load native library (libvehicleclassifier.dll)");
            }
            
        }


        ~VehicleClassifier()
        {
            Dispose();
        }

        /// <summary>
        /// Release the memory associated with the VehicleClassifier when you are done using it
        /// </summary>
        public void Dispose()
        {
            Dispose(true);
        }

        protected virtual void Dispose(bool disposing)
        {

            // Deinitialize the library
            if (this.is_initialized)
            {
                try
                {
                    // Destroy the native obj
                    vehicleclassifier_cleanup(native_instance);
                }
                catch (System.DllNotFoundException)
                {
                    // Ignore.  The library couldn't possibly be loaded anyway
                }
            }
            this.is_initialized = false;
        }
        /// <summary>
        /// Load the classifier specified by ClassificationType into memory.
        /// Initialization may take several seconds to load
        /// </summary>
        /// <returns>True if successful, false otherwise</returns>
        public bool LoadClassifier(ClassificationType classificationType)
        {
            // Initialize the library

            if (loaded_classifiers.Contains(classificationType))
            {
                // Idempotent behavior
                return true;
            }

            this.is_initialized = true;

            try
            {
                bool success = vehicleclassifier_load_classifier(native_instance, (int) classificationType ) != 0;
                
                if (success)
                    loaded_classifiers.Add(classificationType);
            }
            catch (System.DllNotFoundException)
            {
                Console.WriteLine("Could not find/load native library (libvehicleclassifier.dll)");
                return false;
            }

            return IsLoaded(classificationType);


        }

        /// <summary>
        /// Verifies that the Vehicle Classifier library has been initialized
        /// </summary>
        /// <param name="classificationType">The type of recognition (make, color, make/model)</param>
        /// <returns>True if the library is loaded and ready to recognize, false otherwise.</returns>
        public bool IsLoaded(ClassificationType classificationType)
        {
            if (!this.is_initialized)
                return false;

            try
            {
                return vehicleclassifier_is_loaded(this.native_instance, (int)classificationType) != 0;
            }
            catch (System.DllNotFoundException)
            {
                return false;
            }
        }

        // Checks if the instance can receive commands (e.g., is initialized)
        // If not, it throws an exception
        private void check_initialization(ClassificationType classification_type)
        {
            if (!IsLoaded(classification_type))
                throw new InvalidOperationException("OpenALPR Library cannot execute this request since it has not been initialized.");
        }


        /// <summary>
        /// Sets the maximum number of results for the Vehicle Classifier to return with each classification
        /// </summary>
        /// <param name="top_n">An integer describing the maximum number of results to return</param>
        public void setTopN(int top_n)
        {

            vehicleclassifier_set_topn(this.native_instance, top_n);
        }


        /// <summary>
        /// Recognizes an encoded image (e.g., JPG, PNG, etc) from a file on disk
        /// </summary>
        /// <param name="classificationType">The type of recognition (make, color, make/model)</param>
        /// <param name="filepath">The path to the image file</param>
        /// <returns>A list of candidate classifications ordered by confidence</returns>
        /// <exception cref="InvalidOperationException">Thrown when the library has not been initialized for this classification type</exception>
        public List<Models.Candidate> Classify(ClassificationType classification_type, string filepath)
        {
            return Classify(classification_type, System.IO.File.ReadAllBytes(filepath));
        }

        /// <summary>
        /// Recognizes an encoded image (e.g., JPG, PNG, etc) provided as an array of bytes
        /// </summary>
        /// <param name="classificationType">The type of recognition (make, color, make/model)</param>
        /// <param name="image_bytes">The raw bytes that compose the image</param>
        /// <returns>A list of candidate classifications ordered by confidence</returns>
        /// <exception cref="InvalidOperationException">Thrown when the library has not been initialized for this classification type</exception>
        public List<Models.Candidate> Classify(ClassificationType classification_type, byte[] image_bytes)
        {
            check_initialization(classification_type);

            NativeROI roi;
            roi.x = 0;
            roi.y = 0;
            roi.width = REALLY_BIG_PIXEL_WIDTH;
            roi.height = REALLY_BIG_PIXEL_WIDTH;

            IntPtr unmanagedArray = Marshal.AllocHGlobal(image_bytes.Length);
            Marshal.Copy(image_bytes, 0, unmanagedArray, image_bytes.Length);

            IntPtr resp_ptr = vehicleclassifier_recognize_encodedimage(this.native_instance, (int) classification_type, unmanagedArray, image_bytes.Length, roi);

            Marshal.FreeHGlobal(unmanagedArray);

            // Commented out test JSON string
            //string json = @"{""version"":2,""data_type"":""alpr_results"",""epoch_time"":1476716853320,""img_width"":600,""img_height"":600,""processing_time_ms"":116.557533,""regions_of_interest"":[{""x"":0,""y"":0,""width"":600,""height"":600}],""results"":[{""plate"":""627WWI"",""confidence"":94.338623,""matches_template"":1,""plate_index"":0,""region"":""wa"",""region_confidence"":82,""processing_time_ms"":50.445648,""requested_topn"":10,""coordinates"":[{""x"":242,""y"":360},{""x"":358,""y"":362},{""x"":359,""y"":412},{""x"":241,""y"":408}],""candidates"":[{""plate"":""627WWI"",""confidence"":94.338623,""matches_template"":1},{""plate"":""627WKI"",""confidence"":80.588486,""matches_template"":1},{""plate"":""627WI"",""confidence"":79.943542,""matches_template"":0},{""plate"":""627WVI"",""confidence"":79.348930,""matches_template"":1},{""plate"":""627WRI"",""confidence"":79.196785,""matches_template"":1},{""plate"":""627WNI"",""confidence"":79.165802,""matches_template"":1}]}]}";

            string json = Marshal.PtrToStringAnsi(resp_ptr);
            List<Models.Candidate> response = JsonConvert.DeserializeObject<List<Models.Candidate>>(json);
            vehicleclassifier_free_response_string(resp_ptr);

            return response;
        }


        /// <summary>
        /// Recognizes an image from a .NET Bitmap object
        /// </summary>
        /// <param name="classificationType">The type of recognition (make, color, make/model)</param>
        /// <param name="bitmap">The .NET Bitmap object to recognize</param>
        /// <returns>A list of candidate classifications ordered by confidence</returns>
        /// <exception cref="InvalidOperationException">Thrown when the library has not been initialized for this classification type</exception>
        public List<Models.Candidate> Classify(ClassificationType classification_type, System.Drawing.Image image)
        {
            check_initialization(classification_type);

            System.Drawing.Bitmap clone = new System.Drawing.Bitmap(image.Width, image.Height, System.Drawing.Imaging.PixelFormat.Format32bppRgb);
            using (System.Drawing.Graphics gr = System.Drawing.Graphics.FromImage(clone))
            {
                gr.DrawImage(image, new System.Drawing.Rectangle(0, 0, clone.Width, clone.Height));
            }

            System.Drawing.Imaging.BitmapData locked_bmp = clone.LockBits(new System.Drawing.Rectangle(0, 0, clone.Width, clone.Height),
                 System.Drawing.Imaging.ImageLockMode.ReadWrite, clone.PixelFormat);
            byte[] raw_bytes = new byte[locked_bmp.Stride * locked_bmp.Height];
            System.Runtime.InteropServices.Marshal.Copy(locked_bmp.Scan0, raw_bytes, 0, raw_bytes.Length);
            clone.UnlockBits(locked_bmp);

            int bytes_per_pixel = System.Drawing.Image.GetPixelFormatSize(clone.PixelFormat) / 8;

            NativeROI roi;
            roi.x = 0;
            roi.y = 0;
            roi.width = image.Width;
            roi.height = image.Height;

            IntPtr unmanagedArray = Marshal.AllocHGlobal(raw_bytes.Length);
            Marshal.Copy(raw_bytes, 0, unmanagedArray, raw_bytes.Length);

            IntPtr resp_ptr = vehicleclassifier_recognize_rawimage(this.native_instance, (int) classification_type, unmanagedArray, bytes_per_pixel, image.Width, image.Height, roi);

            Marshal.FreeHGlobal(unmanagedArray);

            string json = Marshal.PtrToStringAnsi(resp_ptr);
            List<Models.Candidate> response = JsonConvert.DeserializeObject<List<Models.Candidate>>(json);
            vehicleclassifier_free_response_string(resp_ptr);

            return response;
        }


        // Enumerate the native methods.  Handle the plumbing internally

        [StructLayout(LayoutKind.Sequential)]
        private struct NativeROI
        {
            public int x;
            public int y;
            public int width;
            public int height;
        }

        [DllImport("libvehicleclassifier.dll")]
        private static extern IntPtr vehicleclassifier_init([MarshalAs(UnmanagedType.LPStr)] string configFile, [MarshalAs(UnmanagedType.LPStr)] string runtimeDir);

        [DllImport("libvehicleclassifier.dll")]
        private static extern int vehicleclassifier_cleanup(IntPtr instance);

        [DllImport("libvehicleclassifier.dll")]
        private static extern int vehicleclassifier_load_classifier(IntPtr instance, int classification_type);

        [DllImport("libvehicleclassifier.dll")]
        private static extern int vehicleclassifier_is_loaded(IntPtr instance, int classification_type);
        
        [DllImport("libvehicleclassifier.dll")]
        private static extern IntPtr vehicleclassifier_recognize_rawimage(IntPtr instance, int classification_type, IntPtr pixelData, int bytesPerPixel, int imgWidth, int imgHeight, NativeROI roi);

        [DllImport("libvehicleclassifier.dll")]
        private static extern IntPtr vehicleclassifier_recognize_encodedimage(IntPtr instance, int classification_type, IntPtr bytes, long length, NativeROI roi);

        [DllImport("libvehicleclassifier.dll")]
        private static extern void vehicleclassifier_set_topn(IntPtr instance, int topN);

        [DllImport("libvehicleclassifier.dll")]
        private static extern void vehicleclassifier_free_response_string(IntPtr response);
    }
}
